/*

  SmartClient Ajax RIA system
  Version SNAPSHOT_v15.0d_2026-02-19/LGPL Deployment (2026-02-19)

  Copyright 2000 and beyond Isomorphic Software, Inc. All rights reserved.
  "SmartClient" is a trademark of Isomorphic Software, Inc.

  LICENSE NOTICE
     INSTALLATION OR USE OF THIS SOFTWARE INDICATES YOUR ACCEPTANCE OF
     ISOMORPHIC SOFTWARE LICENSE TERMS. If you have received this file
     without an accompanying Isomorphic Software license file, please
     contact licensing@isomorphic.com for details. Unauthorized copying and
     use of this software is a violation of international copyright law.

  DEVELOPMENT ONLY - DO NOT DEPLOY
     This software is provided for evaluation, training, and development
     purposes only. It may include supplementary components that are not
     licensed for deployment. The separate DEPLOY package for this release
     contains SmartClient components that are licensed for deployment.

  PROPRIETARY & PROTECTED MATERIAL
     This software contains proprietary materials that are protected by
     contract and intellectual property law. You are expressly prohibited
     from attempting to reverse engineer this software or modify this
     software for human readability.

  CONTACT ISOMORPHIC
     For more information regarding license rights and restrictions, or to
     report possible license violations, please contact Isomorphic Software
     by email (licensing@isomorphic.com) or web (www.isomorphic.com).

*/
//> @class FileDropZone
// The FileDropZone class provides a straightforward way to upload files from a user's
// desktop using HTML5 file drop capabilities.
// <P>
// A FileDropZone does not itself upload files - it provides the UI for file collection
// and progress indication. Upload is typically handled by a surrounding form or
// application code.
// <P>
// Users can add files by dragging them onto the drop zone, or by clicking the drop zone
// to open a standard file browser dialog (when +link{canAddFilesOnClick} is true).
// <P>
// When used with a form, uploads include per-file progress indication and error handling.
// <P>
// For use within a DynamicForm, see +link{FileUploadItem}.
// <P>
// FileDropZone extends +link{DropZone} (and therefore StatefulCanvas), so CSS styling uses
// standard state suffixes: Over (during drag), Disabled, and custom states Populated
// (files present) and Processing (upload in progress).
//
// @inheritsFrom DropZone
// @treeLocation Client Reference/Foundation
// @visibility external
//<



isc.ClassFactory.defineClass("FileDropZone", "DropZone");

isc.FileDropZone.addClassProperties({

    //> @classAttr FileDropZone.emptyDropAreaMessage (String : "Drop files here" : IR)
    // Default message shown when no files have been added.
    // @group i18nMessages
    // @visibility external
    //<
    emptyDropAreaMessage: "Drop files here",

    //> @classAttr FileDropZone.clickToAddMessage (String : "or click to browse" : IR)
    // Message shown below +link{emptyDropAreaMessage} when +link{canAddFilesOnClick} is true.
    // @group i18nMessages
    // @visibility external
    //<
    clickToAddMessage: "or click to browse",

    //> @classAttr FileDropZone.dragOverReplaceMessage (String : "Drop to replace files" : IR)
    // Message shown during dragOver when the zone is populated and
    // +link{replaceFilesOnDrop} is true.
    // @group i18nMessages
    // @visibility external
    //<
    dragOverReplaceMessage: "Drop to replace files",

    //> @classAttr FileDropZone.dragOverAddMessage (String : "Drop to add files" : IR)
    // Message shown during dragOver when the zone is populated and
    // +link{replaceFilesOnDrop} is false.
    // @group i18nMessages
    // @visibility external
    //<
    dragOverAddMessage: "Drop to add files",

    //> @classAttr FileDropZone.multipleFilesErrorMessage (String : "Please select a single file to upload" : IR)
    // Error message shown when multiple files are dropped but +link{multiple} is false.
    // @group i18nMessages
    // @visibility external
    //<
    multipleFilesErrorMessage: "Please select a single file to upload",

    //> @classAttr FileDropZone.maxFilesErrorMessage (String : "Cannot add more than ${maxFiles} files" : IR)
    // Error message shown when +link{maxFiles} is exceeded. Dynamic string supporting
    // ${maxFiles} substitution.
    // @group i18nMessages
    // @visibility external
    //<
    maxFilesErrorMessage: "Cannot add more than ${maxFiles} files",

    //> @classAttr FileDropZone.maxSizeErrorMessage (String : "The specified file(s) are too large" : IR)
    // Error message shown when +link{maxSize} is exceeded.
    // @group i18nMessages
    // @visibility external
    //<
    maxSizeErrorMessage: "The specified file(s) are too large",

    //> @classAttr FileDropZone.minSizeErrorMessage (String : "The specified file(s) are too small" : IR)
    // Error message shown when +link{minSize} is not met.
    // @group i18nMessages
    // @visibility external
    //<
    minSizeErrorMessage: "The specified file(s) are too small",

    //> @classAttr FileDropZone.maxFileSizeErrorMessage (String : "${file.name} exceeds the maximum file size" : IR)
    // Error message shown when +link{maxFileSize} is exceeded. Dynamic string supporting
    // ${file} substitution.
    // @group i18nMessages
    // @visibility external
    //<
    maxFileSizeErrorMessage: "${file.name} exceeds the maximum file size",

    //> @classAttr FileDropZone.duplicateFileNameMessage (String : "${file.name} has already been added" : IR)
    // Error message shown when a duplicate file is detected and +link{replaceFilesOnDrop}
    // is false. Dynamic string supporting ${file} substitution.
    // @group i18nMessages
    // @visibility external
    //<
    duplicateFileNameMessage: "${file.name} has already been added",

    //> @classAttr FileDropZone.invalidFileTypeMessage (String : "${file.name} is not an accepted file type" : IR)
    // Error message shown when +link{acceptedFileTypes} is violated. Dynamic string
    // supporting ${file} substitution.
    // @group i18nMessages
    // @visibility external
    //<
    invalidFileTypeMessage: "${file.name} is not an accepted file type",

    //> @classAttr FileDropZone.processingMessage (String : "Uploading..." : IR)
    // Message shown during upload/processing.
    // @group i18nMessages
    // @visibility external
    //<
    processingMessage: "Uploading...",

    //> @classAttr FileDropZone.cancelButtonTitle (String : "Cancel" : IR)
    // Title for the cancel button.
    // @group i18nMessages
    // @visibility external
    //<
    cancelButtonTitle: "Cancel",

    //> @classAttr FileDropZone.retryButtonTitle (String : "Retry" : IR)
    // Title for the retry button shown on failed uploads.
    // @group i18nMessages
    // @visibility external
    //<
    retryButtonTitle: "Retry"
});


isc.FileDropZone.addProperties({

    // FileDropZone only accepts file drops, not content drops
    canDropContent: false,

    //> @attr fileDropZone.multiple (Boolean : false : IR)
    // Does this FileDropZone support multiple files?
    // @visibility external
    //<
    multiple: false,

    //> @attr fileDropZone.maxFiles (Integer : null : IR)
    // Maximum number of files allowed when +link{multiple} is true.
    // @visibility external
    //<

    //> @attr fileDropZone.maxSize (Integer : null : IR)
    // Maximum total size in bytes for all files combined.
    // @visibility external
    //<

    //> @attr fileDropZone.minSize (Integer : null : IR)
    // Minimum total size in bytes for all files combined.
    // @visibility external
    //<

    //> @attr fileDropZone.maxFileSize (Integer : null : IR)
    // Maximum size in bytes for any individual file.
    // @visibility external
    //<

    //> @attr fileDropZone.replaceFilesOnDrop (Boolean : true : IR)
    // If true, dropping new files replaces existing files. If false, new files are
    // added to the existing list.
    // @visibility external
    //<
    replaceFilesOnDrop: true,

    //> @attr fileDropZone.acceptedFileTypes (Array of String : null : IR)
    // Array of accepted MIME types (e.g., ["image/*", "application/pdf"]).
    // If null, all file types are accepted.
    // @visibility external
    //<

    //> @attr fileDropZone.canAddFilesOnClick (Boolean : true : IR)
    // If true, clicking the drop zone opens a file browser dialog.
    // @visibility external
    //<
    canAddFilesOnClick: true,

    // Style properties
    //> @attr fileDropZone.baseStyle (CSSStyleName : "fileDropZone" : IR)
    // Base CSS class for the drop zone. Supports state suffixes: Over, Populated,
    // Processing, Disabled.
    // @visibility external
    //<
    baseStyle: "fileDropZone",

    //> @attr fileDropZone.align (Alignment : "center" : IR)
    // Horizontal alignment for drop zone content.
    // @visibility external
    //<
    align: "center",

    //> @attr fileDropZone.valign (VerticalAlignment : "center" : IR)
    // Vertical alignment for drop zone content.
    // @visibility external
    //<

    // File display settings
    //> @attr fileDropZone.showFileThumbnails (Boolean : true : IR)
    // If true, display thumbnails/icons for added files. If false, show only file names.
    // @visibility external
    //<
    showFileThumbnails: true,

    //> @attr fileDropZone.showImagePreviews (Boolean : true : IR)
    // When +link{showFileThumbnails} is true, should actual image previews be generated
    // for image files? If false, image files will show a generic image icon instead.
    // Image previews are generated using the FileReader API.
    // @visibility external
    //<
    showImagePreviews: true,

    //> @attr fileDropZone.thumbnailWidth (Integer : 64 : IR)
    // Width in pixels for file thumbnails/icons.
    // @visibility external
    //<
    thumbnailWidth: 64,

    //> @attr fileDropZone.thumbnailHeight (Integer : 64 : IR)
    // Height in pixels for file thumbnails/icons.
    // @visibility external
    //<
    thumbnailHeight: 64,

    // File type icons
    //> @attr fileDropZone.defaultFileIcon (SCImgURL : "[SKINIMG]FileDropZone/file.png" : IR)
    // Default icon for files when no specific type icon is available.
    // @visibility external
    //<
    defaultFileIcon: "[SKINIMG]FileDropZone/file.png",

    //> @attr fileDropZone.imageFileIcon (SCImgURL : "[SKINIMG]FileDropZone/image.png" : IR)
    // Icon for image files (when showImagePreviews is false).
    // @visibility external
    //<
    imageFileIcon: "[SKINIMG]FileDropZone/image.png",

    //> @attr fileDropZone.pdfFileIcon (SCImgURL : "[SKINIMG]FileDropZone/pdf.png" : IR)
    // Icon for PDF files.
    // @visibility external
    //<
    pdfFileIcon: "[SKINIMG]FileDropZone/pdf.png",

    //> @attr fileDropZone.documentFileIcon (SCImgURL : "[SKINIMG]FileDropZone/document.png" : IR)
    // Icon for document files (Word, text, etc.).
    // @visibility external
    //<
    documentFileIcon: "[SKINIMG]FileDropZone/document.png",

    //> @attr fileDropZone.spreadsheetFileIcon (SCImgURL : "[SKINIMG]FileDropZone/spreadsheet.png" : IR)
    // Icon for spreadsheet files (Excel, CSV, etc.).
    // @visibility external
    //<
    spreadsheetFileIcon: "[SKINIMG]FileDropZone/spreadsheet.png",

    //> @attr fileDropZone.audioFileIcon (SCImgURL : "[SKINIMG]FileDropZone/audio.png" : IR)
    // Icon for audio files.
    // @visibility external
    //<
    audioFileIcon: "[SKINIMG]FileDropZone/audio.png",

    //> @attr fileDropZone.videoFileIcon (SCImgURL : "[SKINIMG]FileDropZone/video.png" : IR)
    // Icon for video files.
    // @visibility external
    //<
    videoFileIcon: "[SKINIMG]FileDropZone/video.png",

    //> @attr fileDropZone.archiveFileIcon (SCImgURL : "[SKINIMG]FileDropZone/archive.png" : IR)
    // Icon for archive files (ZIP, TAR, etc.).
    // @visibility external
    //<
    archiveFileIcon: "[SKINIMG]FileDropZone/archive.png",

    //> @attr fileDropZone.codeFileIcon (SCImgURL : "[SKINIMG]FileDropZone/code.png" : IR)
    // Icon for code/script files.
    // @visibility external
    //<
    codeFileIcon: "[SKINIMG]FileDropZone/code.png",

    //> @attr fileDropZone.removeIcon (SCImgURL : "[SKINIMG]FileDropZone/remove.png" : IR)
    // Icon for the remove button on file tiles.
    // @visibility external
    //<
    removeIcon: "[SKINIMG]FileDropZone/remove.png",

    //> @attr fileDropZone.showCancelButton (Boolean : true : IR)
    // Whether to show a cancel button during processing that allows the user
    // to abort the upload.
    // @visibility external
    //<
    showCancelButton: true,

    // Internal state
    files: null,
    _processing: false,
    _percentDone: null,

    // Default dimensions
    width: 300,
    height: 150,

    // AutoChild: Container for processing UI elements (label, progressbar, button)
    // This VLayout is shown over the FileDropZone content during upload processing.
    processingContainerConstructor: "VLayout",
    processingContainerDefaults: {
        width: "100%",
        height: "100%",
        align: "center",
        defaultLayoutAlign: "center",
        membersMargin: 8,
        padding: 10,
        visibility: "hidden"
    },

    //> @attr fileDropZone.progressLabel (AutoChild Label : null : IR)
    // AutoChild Label showing the processing message during uploads.
    // @visibility external
    //<
    progressLabelConstructor: "Label",
    progressLabelDefaults: {
        width: "100%",
        height: 20,
        align: "center",
        wrap: false
    },

    //> @attr fileDropZone.progressBar (AutoChild Progressbar : null : IR)
    // AutoChild Progressbar showing upload progress.
    // @visibility external
    //<
    progressBarConstructor: "Progressbar",
    progressBarDefaults: {
        width: "80%",
        height: 20,
        showTitle: false,
        percentDone: 0
    },

    //> @attr fileDropZone.progressText (AutoChild Label : null : IR)
    // AutoChild Label showing the percentage complete text.
    // @visibility external
    //<
    progressTextConstructor: "Label",
    progressTextDefaults: {
        width: 50,
        height: 20,
        align: "center",
        wrap: false
    },

    //> @attr fileDropZone.cancelButton (AutoChild Button : null : IR)
    // AutoChild Button allowing users to cancel an in-progress upload.
    // Only shown if +link{showCancelButton} is true.
    // @visibility external
    //<
    cancelButtonConstructor: "Button",
    cancelButtonDefaults: {
        autoFit: true,
        click : function () {
            this.creator.cancelProcessing();
            return false;
        }
    }
});


isc.FileDropZone.addMethods({

    initWidget : function () {
        this.Super("initWidget", arguments);

        // Initialize files array
        this.files = [];

        // Create the hidden file input for click-to-add
        if (this.canAddFilesOnClick) {
            this._createFileInput();
        }

        // Set initial content
        this._updateDisplay();
    },

    // Create hidden file input element for click-to-add functionality
    _createFileInput : function () {
        var input = document.createElement("input");
        input.type = "file";
        input.style.display = "none";
        input.multiple = this.multiple;

        // Set accept attribute based on acceptedFileTypes
        if (this.acceptedFileTypes && this.acceptedFileTypes.length > 0) {
            input.accept = this.acceptedFileTypes.join(",");
        }

        var _this = this;
        input.onchange = function() {
            if (input.files && input.files.length > 0) {
                _this._handleFilesSelected(Array.prototype.slice.call(input.files));
            }
            // Reset so same file can be selected again
            input.value = "";
        };

        this._fileInput = input;
        document.body.appendChild(input);
    },

    // Handle click to open file browser
    click : function () {
        if (this.canAddFilesOnClick && !this._processing && this._fileInput) {
            // Don't open file browser if click was on a remove button - the remove button
            // has its own onclick handler and we don't want to trigger the browse dialog
            // after removing a file
            var target = isc.EH.getTarget();
            var nativeTarget = isc.EH.lastEvent ? isc.EH.lastEvent.nativeTarget : null;
            if (nativeTarget) {
                // Check if click target or any ancestor is the remove button
                var element = nativeTarget;
                while (element && element !== this.getHandle()) {
                    if (element.className &&
                        element.className.indexOf("fileDropZoneRemoveButton") >= 0) {
                        return;
                    }
                    element = element.parentNode;
                }
            }
            this._fileInput.click();
        }
        return this.Super("click", arguments);
    },

    // Handle files dropped onto the zone
    drop : function () {
        var files = isc.EH.getNativeDragData();
        if (!files || files.length == 0) return false;

        // Convert FileList to Array
        files = Array.prototype.slice.call(files);

        this._handleFilesSelected(files);
        return false;
    },

    // Process files from drop or file input
    _handleFilesSelected : function (files) {
        // Validation: multiple
        if (!this.multiple && files.length > 1) {
            this.showDropError(this._getMessage("multipleFilesErrorMessage"));
            return;
        }

        // Validation: maxFiles
        var totalFileCount = files.length;
        if (!this.replaceFilesOnDrop && this.files) {
            totalFileCount += this.files.length;
        }
        if (this.maxFiles && totalFileCount > this.maxFiles) {
            this.showDropError(this._evalDynamicString(
                this._getMessage("maxFilesErrorMessage"), {maxFiles: this.maxFiles}));
            return;
        }

        // Validation: acceptedFileTypes
        if (this.acceptedFileTypes) {
            for (var i = 0; i < files.length; i++) {
                if (!this._isAcceptedFileType(files[i])) {
                    this.showDropError(this._evalDynamicString(
                        this._getMessage("invalidFileTypeMessage"), {file: files[i]}));
                    return;
                }
            }
        }

        // Validation: maxFileSize (per-file)
        if (this.maxFileSize) {
            for (var i = 0; i < files.length; i++) {
                if (files[i].size > this.maxFileSize) {
                    this.showDropError(this._evalDynamicString(
                        this._getMessage("maxFileSizeErrorMessage"), {file: files[i]}));
                    return;
                }
            }
        }

        // Calculate total size
        var totalSize = 0;
        for (var i = 0; i < files.length; i++) {
            totalSize += files[i].size;
        }

        // Handle replace vs add mode
        if (!this.replaceFilesOnDrop && this.files && this.files.length > 0) {
            // Check for duplicates
            for (var i = 0; i < files.length; i++) {
                for (var j = 0; j < this.files.length; j++) {
                    if (files[i].name == this.files[j].name) {
                        this.showDropError(this._evalDynamicString(
                            this._getMessage("duplicateFileNameMessage"), {file: files[i]}));
                        return;
                    }
                }
            }
            // Combine files
            files = this.files.concat(files);
            totalSize = 0;
            for (var i = 0; i < files.length; i++) {
                totalSize += files[i].size;
            }
        }

        // Validation: maxSize (total)
        if (this.maxSize && totalSize > this.maxSize) {
            this.showDropError(this._getMessage("maxSizeErrorMessage"));
            return;
        }

        // Validation: minSize (total)
        if (this.minSize && totalSize < this.minSize) {
            this.showDropError(this._getMessage("minSizeErrorMessage"));
            return;
        }

        // Store files and update UI
        this.files = files;
        this._updateDisplay();
        this.filesAdded(files);
    },

    // _isAcceptedFileType is inherited from DropZone

    // Get message from instance or class properties
    _getMessage : function (messageName) {
        return this[messageName] || isc.FileDropZone[messageName];
    },

    // Evaluate dynamic string with variable substitution
    _evalDynamicString : function (template, vars) {
        if (!template) return "";
        return template.replace(/\$\{([^}]+)\}/g, function(match, expr) {
            try {
                // Simple property access
                var parts = expr.split(".");
                var value = vars;
                for (var i = 0; i < parts.length; i++) {
                    if (value == null) return match;
                    value = value[parts[i]];
                }
                return value != null ? value : match;
            } catch (e) {
                return match;
            }
        });
    },

    // Get the appropriate icon for a file based on its MIME type
    _getFileIcon : function (file) {
        var type = file.type || "";

        if (type.indexOf("image/") == 0) {
            return this.imageFileIcon;
        } else if (type == "application/pdf") {
            return this.pdfFileIcon;
        } else if (type.indexOf("audio/") == 0) {
            return this.audioFileIcon;
        } else if (type.indexOf("video/") == 0) {
            return this.videoFileIcon;
        } else if (type.match(/spreadsheet|excel|csv/i) ||
                   type == "application/vnd.ms-excel" ||
                   type == "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet") {
            return this.spreadsheetFileIcon;
        } else if (type.match(/document|word|text/i) ||
                   type == "application/msword" ||
                   type == "application/vnd.openxmlformats-officedocument.wordprocessingml.document" ||
                   type.indexOf("text/") == 0) {
            return this.documentFileIcon;
        } else if (type.match(/zip|tar|gzip|compress|archive/i)) {
            return this.archiveFileIcon;
        } else if (type.match(/javascript|json|xml|html|css|script/i)) {
            return this.codeFileIcon;
        }

        return this.defaultFileIcon;
    },

    // Format file size for display
    _formatFileSize : function (bytes) {
        if (bytes == null) return "";
        if (bytes < 1024) return bytes + " B";
        if (bytes < 1024 * 1024) return Math.round(bytes / 1024) + " KB";
        if (bytes < 1024 * 1024 * 1024) {
            return Math.round(bytes / (1024 * 1024) * 10) / 10 + " MB";
        }
        return Math.round(bytes / (1024 * 1024 * 1024) * 10) / 10 + " GB";
    },

    // Update the display based on current state
    _updateDisplay : function () {
        if (this._processing) {
            // Show processing UI using autoChildren
            this._showProcessingUI();
        } else {
            // Hide processing UI if it exists
            this._hideProcessingUI();

            // Show content-based display (empty state or file list)
            var contents;
            if (this.files && this.files.length > 0) {
                contents = this._getFileListHTML();
            } else {
                contents = this._getEmptyHTML();
            }
            this.setContents(contents);
        }
        this._updateStyleName();
    },

    // Create and show the processing UI autoChildren
    _showProcessingUI : function () {
        // Create processing container if it doesn't exist
        if (!this.processingContainer) {
            this.processingContainer = this.addAutoChild("processingContainer");
            this.addChild(this.processingContainer);

            // Create processing label
            this.progressLabel = this.addAutoChild("progressLabel", {
                contents: this._getMessage("processingMessage")
            });
            this.processingContainer.addMember(this.progressLabel);

            // Create progress bar
            this.progressBar = this.addAutoChild("progressBar");
            this.processingContainer.addMember(this.progressBar);

            // Create progress text
            this.progressText = this.addAutoChild("progressText", {
                contents: "0%"
            });
            this.processingContainer.addMember(this.progressText);

            // Create cancel button if enabled
            if (this.showCancelButton) {
                this.cancelButton = this.addAutoChild("cancelButton", {
                    title: this._getMessage("cancelButtonTitle")
                });
                this.processingContainer.addMember(this.cancelButton);
            }
        }

        // Clear the contents and show the processing container
        this.setContents("");
        this.processingContainer.show();
    },

    // Hide the processing UI autoChildren
    _hideProcessingUI : function () {
        if (this.processingContainer) {
            this.processingContainer.hide();
        }
    },

    // Generate HTML for empty state
    _getEmptyHTML : function () {
        var html = "<div class='fileDropZoneContent'>";
        html += "<div class='fileDropZoneMessage'>" +
                this._getMessage("emptyDropAreaMessage") + "</div>";
        if (this.canAddFilesOnClick) {
            html += "<div class='fileDropZoneClickMessage'>" +
                    this._getMessage("clickToAddMessage") + "</div>";
        }
        html += "</div>";
        return html;
    },

    // Generate HTML for file list display
    _getFileListHTML : function () {
        var html = "<div class='fileDropZoneFileList'>";

        for (var i = 0; i < this.files.length; i++) {
            var file = this.files[i];
            html += this._getFileTileHTML(file, i);
        }

        html += "</div>";
        return html;
    },

    // Generate HTML for a single file tile
    _getFileTileHTML : function (file, index) {
        var html = "<div class='fileDropZoneFileTile' data-file-index='" + index + "'>";

        if (this.showFileThumbnails) {
            // Resolve icon URL - handles [SKINIMG] and other SmartClient URL prefixes
            var iconUrl = isc.Page.getURL(this._getFileIcon(file));
            var isImage = file.type && file.type.indexOf("image/") == 0;

            // For images, we'll load preview asynchronously if enabled
            var imgId = this.getID() + "_thumb_" + index;
            if (isImage && this.showImagePreviews) {
                html += "<img id='" + imgId + "' class='fileDropZoneThumbnail' " +
                        "src='" + iconUrl + "' " +
                        "style='width:" + this.thumbnailWidth + "px;height:" +
                        this.thumbnailHeight + "px;object-fit:contain;'/>";
                // Schedule async preview load
                this._loadImagePreview(file, imgId);
            } else {
                html += "<img class='fileDropZoneThumbnail' src='" + iconUrl + "' " +
                        "style='width:" + this.thumbnailWidth + "px;height:" +
                        this.thumbnailHeight + "px;'/>";
            }
        }

        html += "<div class='fileDropZoneFileName' title='" + file.name + "'>" +
                file.name + "</div>";
        html += "<div class='fileDropZoneFileSize'>" +
                this._formatFileSize(file.size) + "</div>";

        // Remove button - resolve the icon URL for [SKINIMG] prefix
        var removeIconUrl = isc.Page.getURL(this.removeIcon);
        html += "<div class='fileDropZoneRemoveButton' " +
                "onclick='if(window." + this.getID() + ")window." + this.getID() +
                ".removeFile(" + index + ");event.stopPropagation();'>" +
                "<img src='" + removeIconUrl + "' width='16' height='16'/></div>";

        html += "</div>";
        return html;
    },

    // Load image preview asynchronously
    _loadImagePreview : function (file, imgId) {
        if (!window.FileReader) return;

        var reader = new FileReader();
        reader.onload = function(e) {
            var img = document.getElementById(imgId);
            if (img) {
                img.src = e.target.result;
            }
        };
        reader.readAsDataURL(file);
    },

    // Override getCustomState to return FileDropZone-specific state suffixes.
    // StatefulCanvas handles Over (via showRollOver) and Disabled (via showDisabled)
    // automatically. We add Populated and Processing as custom states.
    getCustomState : function () {
        if (this._processing) {
            return "Processing";
        } else if (this.files && this.files.length > 0) {
            return "Populated";
        }
        return null;
    },

    // Trigger style update when internal state changes
    _updateStyleName : function () {
        this.stateChanged();
    },

    // Handle drag over - StatefulCanvas handles the Over state via showRollOver
    dropOver : function () {
        this._dragOver = true;
        // Set the rollOver state which triggers "Over" suffix via showRollOver
        this.setState(isc.StatefulCanvas.STATE_OVER);
        return this.Super("dropOver", arguments);
    },

    // Handle drag out - reset to normal state
    dropOut : function () {
        this._dragOver = false;
        this.setState(isc.StatefulCanvas.STATE_UP);
        return this.Super("dropOut", arguments);
    },

    //> @method fileDropZone.getFiles()
    // Retrieves the files that have been added to this fileDropZone.
    // @return (Array of File) JavaScript File objects added to this fileDropZone
    // @visibility external
    //<
    getFiles : function () {
        return this.files ? this.files.slice() : [];
    },

    //> @method fileDropZone.setFiles()
    // Programmatically populate a fileDropZone with files.
    // @param files (File | Array of File) JavaScript File object(s) to attach
    // @visibility external
    //<
    setFiles : function (files) {
        if (files == null) {
            this.clearFiles();
            return;
        }

        if (!isc.isAn.Array(files)) {
            files = [files];
        }

        this.files = files;
        this._updateDisplay();

        // Notify listeners that files were added - this is essential for FileUploadItem
        // to update the form's value so _hasInlineBinaryData() returns true
        if (this.filesAdded) this.filesAdded(files);
    },

    //> @method fileDropZone.clearFiles()
    // Clear all files from this fileDropZone.
    // @visibility external
    //<
    clearFiles : function () {
        var oldFiles = this.files;
        this.files = [];
        this._updateDisplay();

        if (oldFiles && oldFiles.length > 0) {
            this.filesRemoved(oldFiles);
        }
    },

    //> @method fileDropZone.removeFile()
    // Remove a specific file from this fileDropZone.
    // @param file (File | Integer) The file to remove, or its index
    // @visibility external
    //<
    removeFile : function (file) {
        if (this.files == null || this.files.length == 0) return;

        var index;
        if (isc.isA.Number(file)) {
            index = file;
        } else {
            index = this.files.indexOf(file);
        }

        if (index >= 0 && index < this.files.length) {
            var removed = this.files.splice(index, 1);
            this._updateDisplay();
            this.filesRemoved(removed);
        }
    },

    //> @method fileDropZone.getSize()
    // Get the total size in bytes of all selected files.
    // @return (Integer) Total size in bytes, or 0 if no files
    // @visibility external
    //<
    getSize : function () {
        if (this.files == null || this.files.length == 0) return 0;

        var total = 0;
        for (var i = 0; i < this.files.length; i++) {
            total += this.files[i].size || 0;
        }
        return total;
    },

    //> @method fileDropZone.filesAdded()
    // Notification fired when files are successfully added (via drag or click).
    // @param files (Array of File) Files that were added
    // @visibility external
    //<
    filesAdded : function (files) {
        // Override point
    },

    //> @method fileDropZone.filesRemoved()
    // Notification fired when files are removed.
    // @param files (Array of File) Files that were removed
    // @visibility external
    //<
    filesRemoved : function (files) {
        // Override point
    },

    //> @method fileDropZone.showDropError()
    // Display an error when a drop fails validation.
    // Default implementation calls isc.warn(). Override for custom handling.
    // @param errorMessage (String) Error message to display
    // @visibility external
    //<
    showDropError : function (errorMessage) {
        isc.warn(errorMessage);
    },

    //> @method fileDropZone.startProcessing()
    // Show UI indicating processing has started. Masks the component and shows
    // progressIndicator.
    // @return (Boolean) false if no files selected, true otherwise
    // @visibility external
    //<
    startProcessing : function () {
        if (this.files == null || this.files.length == 0) {
            return false;
        }

        this._processing = true;
        this._percentDone = 0;
        this._updateDisplay();
        return true;
    },

    //> @method fileDropZone.setProcessingProgress()
    // Update progress indication during upload/processing.
    // @param percentDone (Float) Progress percentage (0-100)
    // @param processed (Integer) Bytes processed so far
    // @param total (Integer) Total bytes being processed
    // @return (Boolean) false if no files selected, true otherwise
    // @visibility external
    //<
    setProcessingProgress : function (percentDone, processed, total) {
        if (this.files == null || this.files.length == 0) {
            return false;
        }

        this._percentDone = percentDone;
        this._processed = processed;
        this._total = total;

        // Update progress bar widget
        if (this.progressBar) {
            this.progressBar.setPercentDone(percentDone);
        }

        // Update progress text widget
        if (this.progressText) {
            this.progressText.setContents(Math.round(percentDone) + "%");
        }

        return true;
    },

    //> @method fileDropZone.setFileProgress()
    // Update progress indication for a specific file during concurrent uploads.
    // @param file (File) The file being uploaded
    // @param percentDone (Float) Progress percentage (0-100) for this file
    // @param processed (Integer) Bytes processed so far for this file
    // @param total (Integer) Total bytes for this file
    // @visibility external
    //<
    setFileProgress : function (file, percentDone, processed, total) {
        // This could update per-file progress indicators in the UI
        // For now, just store the state
        if (!this._fileProgress) this._fileProgress = {};

        var fileId = file.name + "_" + file.size;
        this._fileProgress[fileId] = {
            percentDone: percentDone,
            processed: processed,
            total: total
        };
    },

    //> @method fileDropZone.getProcessingPercentDone()
    // Get current progress percentage.
    // @return (Integer) Current percentage, or null if not processing
    // @visibility external
    //<
    getProcessingPercentDone : function () {
        if (!this._processing) return null;
        return this._percentDone;
    },

    //> @method fileDropZone.cancelProcessing()
    // Cancel an in-progress upload/processing operation.
    // <P>
    // This will abort the active XHR request (if any), hide the processing UI,
    // and fire the +link{processingCancelled} notification.
    // @visibility external
    //<
    cancelProcessing : function () {
        this._processing = false;
        this._percentDone = null;
        this._updateDisplay();
        this.processingCancelled();
    },

    //> @method fileDropZone.processingCancelled()
    // Notification fired when processing is cancelled via +link{cancelProcessing()}.
    // <P>
    // This is a notification method intended for override. The FileDropZone itself does not
    // perform any network abort logic - it only manages local UI state (hiding progress,
    // preserving files). When used as the canvas of a +link{FileUploadItem}, the item's
    // default configuration overrides this method to propagate the cancel to the containing
    // +link{DynamicForm}, which handles aborting any in-flight XHR upload request via
    // +link{RPCManager.cancelQueue()}.
    // <P>
    // For standalone FileDropZone usage (outside of FileUploadItem), implement this method
    // to abort any custom upload logic you have initiated.
    // @visibility external
    //<
    processingCancelled : function () {
        // Override point - see JSDoc above for integration notes
    },

    //> @method fileDropZone.endProcessing()
    // Hide processing UI. Called when upload/processing completes.
    // @visibility external
    //<
    endProcessing : function () {
        this._processing = false;
        this._percentDone = null;
        this._updateDisplay();
    },

    //> @method fileDropZone.fileUploadComplete()
    // Notification fired when an individual file upload completes (in concurrent mode).
    // @param file (File) The file that completed
    // @param success (Boolean) Whether the upload succeeded
    // @param response (DSResponse) The server response for this file
    // @visibility external
    //<
    fileUploadComplete : function (file, success, response) {
        // Override point
    },

    //> @method fileDropZone.fileUploadFailed()
    // Notification fired when an individual file upload fails (in concurrent mode).
    // @param file (File) The file that failed
    // @param error (String) Error message describing the failure
    // @visibility external
    //<
    fileUploadFailed : function (file, error) {
        // Override point
    },

    // Cleanup
    destroy : function () {
        // Remove the hidden file input
        if (this._fileInput && this._fileInput.parentNode) {
            this._fileInput.parentNode.removeChild(this._fileInput);
        }
        this._fileInput = null;

        // AutoChildren are automatically destroyed by the framework when their
        // parent is destroyed, so no explicit cleanup is needed for:
        // processingContainer, progressLabel, progressBar, progressText, cancelButton

        this.Super("destroy", arguments);
    }

});
