/*

  SmartClient Ajax RIA system
  Version SNAPSHOT_v15.0d_2026-02-19/LGPL Deployment (2026-02-19)

  Copyright 2000 and beyond Isomorphic Software, Inc. All rights reserved.
  "SmartClient" is a trademark of Isomorphic Software, Inc.

  LICENSE NOTICE
     INSTALLATION OR USE OF THIS SOFTWARE INDICATES YOUR ACCEPTANCE OF
     ISOMORPHIC SOFTWARE LICENSE TERMS. If you have received this file
     without an accompanying Isomorphic Software license file, please
     contact licensing@isomorphic.com for details. Unauthorized copying and
     use of this software is a violation of international copyright law.

  DEVELOPMENT ONLY - DO NOT DEPLOY
     This software is provided for evaluation, training, and development
     purposes only. It may include supplementary components that are not
     licensed for deployment. The separate DEPLOY package for this release
     contains SmartClient components that are licensed for deployment.

  PROPRIETARY & PROTECTED MATERIAL
     This software contains proprietary materials that are protected by
     contract and intellectual property law. You are expressly prohibited
     from attempting to reverse engineer this software or modify this
     software for human readability.

  CONTACT ISOMORPHIC
     For more information regarding license rights and restrictions, or to
     report possible license violations, please contact Isomorphic Software
     by email (licensing@isomorphic.com) or web (www.isomorphic.com).

*/
//
//	Comm.serialize() methods for the comm package
//

// XXX this package must not be dependant on the Comm package, because serialization is a useful
// feature completely apart from Comm.  Unfortunately, the methods are currently expected to be on
// the Comm class, so if the Comm class doesn't exist we need to define it.
if (!isc.Comm) isc.ClassFactory.defineClass("Comm");

isc.Comm.addClassProperties( {
    //>	@classAttr	Comm._simpleIdentifierRE (Regex : /^[\$_a-zA-Z][\$\w]*$/ : IR)
    //			Regular expression to match legal identifier names so we can 
    //			 avoid unnecessary quotes when serializing.
    //		@group	serialization
    //<
    _simpleIdentifierRE : 	/^[\$_a-zA-Z][\$\w]*$/,

    //>	@classAttr	Comm.BACKREF_PREFIX (String : "$$BACKREF$$:" : IR)
    //			Prefix for back-references in serialized object references.
    //		@group	serialization
    //<
    BACKREF_PREFIX : "$$BACKREF$$:",	

    indent : "    "

    
});


isc.Comm.addClassMethods({

//>	@classMethod	Comm.serialize()
// Serialize an object of any type into a string, in a form that
// can be simply eval'd to rebuild the original object.
//
//		@group	serialization
//		
//		@param	object		(Any)		object to be serialized
//		@param	[indent]	(boolean)	should output be formatted with line breaks and indenting
//                                      for readability? If unspecified, indenting occurs if
//                                      +link{JSONEncoder.prettyPrint} is true.
//		@return				(String)	serialized form of the object
//<

serialize : function (object, indent) {
    var props = { strictQuoting:false, dateFormat:"logicalDateConstructor"};

    // if indent was explicitly specified, respect it
    if (indent != null) props.prettyPrint = indent;
    return isc.JSON.encode(object, props);
}

});	// END isc.addMethods(isc.Comm, {})

//> @class JSON
// Utilities for working with JSON data.
// <P>
// <h3>Native JSON.stringify() vs isc.JSON.encode()</h3>
// <P>
// For maximum performance when serializing large datasets, consider using the native
// <code>JSON.stringify()</code> when your data meets certain criteria. Native serialization
// is approximately <b>6-10x faster</b> than <code>isc.JSON.encode()</code>, but lacks several
// features that SmartClient provides:
// <P>
// <b>Use native <code>JSON.stringify()</code> when ALL of these conditions are met:</b>
// <ul>
// <li>Data contains <b>no circular references</b> - native throws an error on cycles rather
//     than handling them gracefully. If your data might have cycles (e.g., parent/child
//     back-references in tree structures), you must use <code>isc.JSON.encode()</code>.</li>
// <li>Data contains <b>no Date values</b>, OR you accept JSON's default date handling.
//     Native <code>JSON.stringify()</code> converts Dates to ISO 8601 strings
//     (e.g., "2024-01-15T14:30:00.000Z"). This loses:
//     <ul>
//     <li><b>Logical Date distinction</b>: SmartClient's logical dates (date-only, no time
//         component) are serialized identically to datetimes by native JSON</li>
//     <li><b>Logical Time distinction</b>: SmartClient's logical times (time-only, no date
//         component) become full datetime strings</li>
//     <li><b>Timezone handling</b>: Native always outputs UTC; SmartClient can preserve
//         local timezone context for logical dates/times</li>
//     </ul>
//     Use +link{JSONEncoder.dateFormat} with values like "logicalDateConstructor" to preserve
//     these distinctions when round-tripping data.</li>
// <li>Data contains <b>no SmartClient class instances</b> - native JSON cannot serialize
//     SmartClient widgets or other framework objects meaningfully</li>
// <li>You do <b>not need pretty-printed output</b> - native JSON.stringify's optional
//     formatting is less flexible than +link{JSONEncoder.prettyPrint}</li>
// <li>You do <b>not need custom serialization</b> via object <code>_serialize()</code> methods</li>
// </ul>
// <P>
// <b>Use <code>isc.JSON.encode()</code> when ANY of these apply:</b>
// <ul>
// <li>Data might contain circular references (automatically detected and handled)</li>
// <li>You need to preserve SmartClient's logical date/time/datetime distinctions</li>
// <li>You're serializing SmartClient components or class instances</li>
// <li>You need back-reference paths for reconstructing object graphs</li>
// <li>You want configurable date formats (+link{JSONDateFormat})</li>
// <li>You need readable, pretty-printed output</li>
// </ul>
//
// @treeLocation Client Reference/Data Binding
// @visibility external
//<
isc.ClassFactory.defineClass("JSON", null, null, true);
isc.JSON.addClassProperties({
//> @classMethod JSON.encode()
// Serialize an object as a JSON-like string by creating a +link{JSONEncoder} and calling
// +link{JSONEncoder.encode()}. The resulting string, which by default is <b>not</b> strict
// JSON, may be decoded back to objects via +link{JSON.decode()}.
// <P>
// There is an important caveat regarding Date values: for Date values to be decoded properly
// by decode(), it is required to use +link{JSONEncoder.dateFormat, dateFormat}
// "dateConstructor" or "logicalDateConstructor"; this causes Date values to be represented
// in the encoded string as JavaScript <code>new Date(...)</code> or
// +link{DateUtil.createLogicalDate()} calls, respectively.
// <P>
// <b>Strict JSON output</b>: to ensure output conforms to strict JSON (no unquoted keys or
// embedded JavaScript), pass settings like:
// <pre>
// isc.JSON.encode(data, {
//     strictQuoting: true,          // quote all property names
//     dateFormat: "xmlSchema",      // or "logicalDateString" for use with +link{JSON.decodeSafeWithDates()}
//     serializeInstances: "skip",   // skip instances of SmartClasses
//     showDebugOutput: false        // avoid placeholders that are not valid JSON
// });
// </pre>
// When strict JSON is desired, avoid dateFormats "dateConstructor" and "logicalDateConstructor",
// which embed JavaScript in the string output and therefore do not produce strict JSON.
//
// @param object (Any) object to serialize
// @param [settings] (JSONEncoder Properties) optional settings for encoding
// @return (String) object encoded as a JSON String
// @visibility external
//<
encode : function (object, settings) {
    return isc.JSONEncoder.create(settings).encode(object);    
},

//> @classMethod JSON.decode()
// De-serialize an object from JSON.  Currently, this is simply a JavaScript eval() and should
// be used for trusted data only.
//
// @param jsonString (String) JSON data to be de-serialized
// @return (Object) object derived from JSON String
// @visibility external
//<
decode : function (jsonString) {
    //!OBFUSCATEOK
    // CSP check: if eval() is blocked, fall back to decodeSafe() directly
    if (!isc.Browser.allowsNewFunction) {
        // decodeSafe() uses native JSON.parse - works for strict JSON but not pseudo-JSON
        return this.decodeSafe(jsonString);
    }
    // Add parens to the JSON to avoid
    // an issue where eval() gets confused and believes it is dealing with a block
    return eval("(" + jsonString + ")");
},

//> @classMethod JSON.decodeStrict()
// De-serialize an object from JSON, attempting native JSON.parse() first for performance,
// then falling back to eval() if the input contains SmartClient pseudo-JSON extensions
// (unquoted keys, single quotes, trailing commas, etc.).
// <P>
// This method is optimized for cases where the JSON is typically strict (conforming to
// the JSON spec) but may occasionally contain pseudo-JSON extensions. The native JSON.parse()
// fast path provides significant performance improvements in V8 and other modern engines.
// <P>
// For data that is always strict JSON, use +link{JSON.decodeSafe()} instead.
//
// @param jsonString (String) JSON or pseudo-JSON data to be de-serialized
// @return (Object) object derived from JSON String
// @visibility external
//<
decodeStrict : function (jsonString) {
    
    try {
        return window.JSON.parse(jsonString);
    } catch (e) {
        // Parse failed - likely pseudo-JSON with SmartClient extensions
        // Fall back to eval() for compatibility
        //!OBFUSCATEOK
        try {
            return eval("(" + jsonString + ")");
        } catch (evalError) {
            this.logWarn("Error parsing JSON (tried both JSON.parse and eval): " +
                evalError.toString() + ", JSON text:\n" + jsonString);
            return null;
        }
    }
},

//> @classMethod JSON.decodeSafe()
// Decodes strict JSON using native browser JSON parsing APIs. This API will not work with
// pseudo-JSON that must be parsed as JavaScript (such as that produced by encoding with
// +link{JSONEncoder.dateFormat} "dateConstructor" or "logicalDateConstructor").
// <P>
// This API is called "safe" because using a JavaScript eval() to decode JSON is potentially 
// unsafe if the data being decoded is untrusted. For example, if users are able to save data 
// as JSON for other uses to see (such as sharing +link{listGrid.viewState}) and there is no 
// validation of the saved data to ensure safety and some users are untrusted, then saved JSON 
// could be used similarly to an XSS attack, allowing one user to execute JavaScript code in 
// another user's browser.
// <P>
// Note that, because JSON has no way of representing dates, serializing a structure that contains 
// dates and then deserializing with decodeSafe() necessarily results in any Dates becoming strings.  
// Use +link{JSONEncoder.dateFormat} "logicalDateString" in combination with +link{JSON.decodeSafeWithDates()}
// to round-trip date, time and datetime values accurately.
// 
// @param jsonString (String) JSON data to be de-serialized
// @param jsonReviver (Function) optional setting
// @return (Object) object derived from JSON String
// @visibility external
//<
decodeSafe : function (jsonString, jsonReviver, dontCatchErrors) {
    if (!dontCatchErrors) {
        try {
            return window.JSON.parse(jsonString, jsonReviver); 
        } catch(err) {
            this.logWarn("Invalid JSON data. "+err);
            return null;
        } 
    // Let upstream code handle catching parsing errors if requested
    } else {
        return window.JSON.parse(jsonString, jsonReviver); 
    }
},

//> @classMethod JSON.decodeSafeWithDates()
// Uses +link{JSON.decodeSafe()} to decode strict JSON using native browser JSON parsing APIs, 
// with settings to correctly process formatted date values
// created with +link{JSONEncoder.dateFormat} "logicalDateString".
// 
// @param jsonString (String) JSON data to be de-serialized
// @visibility external
//<
decodeSafeWithDates : function (jsonString, dontCatchErrors) {
    return this.decodeSafe(jsonString, this.jsonDateReviver, dontCatchErrors);
},

// When using strict JSON (and therefore native JSON.parse()), we can't put
// "date literals" (or calls to new Date(...)) into our response, so we need
// a custom format for date, time and datetime, and use a reviver function to 
// create Date objects from these strings.
// Use the following format:
// $$DATE$$:YYYY-MM-DD - logical date
// $$DATESTAMP$$:ms - datetime represented as date-stamp [date.getTime()]
// $$DATE$$:YYYY-MM-DD hh:mm:ss - datetime
// $$TIME$$:hh:mm:ss - time
// Note - these are legacy formats that we used to use on the server which have
// resurrected to support strict JSON.
_$datePrefix:"$$DATE$$:",
_$timePrefix:"$$TIME$$:",
_$datestampPrefix:"$$DATESTAMP$$:",
jsonDateReviver : function (key, value) {

    if (isc.isA.String(value)) {
        if (value.startsWith(isc.JSON._$datestampPrefix)) {
            value = parseInt(value.substring(14));
            value = new Date(value);
        } else if (value.startsWith(isc.JSON._$datePrefix)) {
            value = value.substring(9);
            if (value.indexOf(":") == -1) {
                // logical date
                var split = value.split("-");
                value = isc.DateUtil.parseServerDate(
                            parseInt(split[0]), 
                            parseInt(split[1])-1, 
                            parseInt(split[2]));
            } else {
                
                var split = value.split(" ");
                var dateSplit = split[0].split("-"),
                    timeSplit = split[1].split(":");
                var dot = timeSplit[2].indexOf(".");
                var ms;
                if (dot != -1) {
                    ms = timeSplit[2].substring(dot+1);
                    timeSplit[2] = timeSplit[2].substring(0, dot);
                }
                var date = new Date(28800000);
                date.setUTCFullYear(dateSplit[0]);
                date.setUTCMonth(dateSplit[1]-1);
                date.setUTCDate(dateSplit[2]);
                date.setUTCHours(timeSplit[0]);
                date.setUTCMinutes(timeSplit[1]);
                date.setUTCSeconds(timeSplit[2]);
                if (ms) {
                    date.setUTCMilliseconds(ms);
                }
                    
                value = date;
            }
        } else if (value.startsWith(isc.JSON._$timePrefix)) {
            // Expected format prefix + HMS:
            // EG: "$$TIME$$11:22:0"
            var split = value.substring(9).split(":");
            value = isc.DateUtil.parseServerTime(
                        parseInt(split[0]), parseInt(split[1]), parseInt(split[2]));
        }
    }
    return value;
}

});

//> @class JSONEncoder
// Class for encoding objects as JSON strings.  
// @treeLocation Client Reference/Data Binding
// @visibility external
//<
isc.ClassFactory.defineClass("JSONEncoder");
isc.JSONEncoder.addClassProperties({

//>	@classMethod	JSONEncoder._serialize_remember()	(A)
//			Remember an object that has just been serialized, so we don't
//			 attempt to serialize it again (and thus get into an endless loop).
//		@group	serialization
//
//		@param	objRefs	(Map)	    Map from objects to their serialization paths
//		@param	object	(Any)		object to serialize
//		@param	path	(String)	global variable path to this object
//<
// Uses Map for O(1) lookup. The "replace" parameter updates the path for an already-tracked
// object (used when the canonical path changes during serialization).
_serialize_remember : function (objRefs, object, path, replace) {
    if (replace) {
        
    }
    // Map.set() handles both insert and update
    objRefs.set(object, path);
},

// If this object is a Tree node, automatically clean off properties that the Tree adds to the
// node that should not be saved.

_serialize_cleanNode : function (object) {
    var treeId = object["_isc_tree"];
    if (treeId != null) {
        var theTree = window[treeId];
        if (theTree && theTree.parentProperty && object[theTree.parentProperty]) {
            object = theTree.getCleanNodeData(object);
        }
    }
    return object;
},

// Have we already output a particular object in this serialize pass? If so, return the path
// to that object. Uses Map.get() for O(1) lookup.
_serialize_alreadyReferenced : function (objRefs, object) {
    var path = objRefs.get(object);
    return path !== undefined ? path : null;
},

// Add a new identifier to an object path for circular reference tracking. The path format
// matches JavaScript property access syntax: .prop for simple identifiers, ["prop"] for
// complex ones, [0] for array indices.
_serialize_addToPath : function (objPath, newIdentifier) {
    // Fast path for numbers (array indices) - use typeof for speed
    if (typeof newIdentifier == "number") {
        return objPath + "[" + newIdentifier + "]";
    }
    // For string identifiers, check if quoting is needed
    if (isc.Comm._simpleIdentifierRE.test(newIdentifier)) {
        return objPath + "." + newIdentifier;
    }
    return objPath + '["' + newIdentifier + '"]';
}        
});

isc.JSONEncoder.addProperties({
_indent: isc.Comm.indent,
// Pre-computed indentation and separator strings by depth level, built lazily.
// Avoids repeated string concatenation in deep structures (97K+ nodes).
_indentCache: null,
_sepCache: null,
_getIndent : function (depth) {
    if (!this._indentCache) {
        // Build cache for depths 0-20 (covers most real-world cases)
        this._indentCache = [""];
        this._sepCache = [",\n"];
        var indent = "";
        for (var i = 1; i <= 20; i++) {
            indent += this._indent;
            this._indentCache[i] = indent;
            this._sepCache[i] = ",\n" + indent;
        }
    }
    return depth < this._indentCache.length ? this._indentCache[depth]
                                            : this._indentCache[this._indentCache.length - 1];
},
_getSep : function (depth) {
    if (!this._sepCache) this._getIndent(0);  // Initialize caches
    return depth < this._sepCache.length ? this._sepCache[depth]
                                         : this._sepCache[this._sepCache.length - 1];
},

//> @method JSONEncoder.encode()
// Serialize an object as a JSON string.
// <P>
// Automatically handles circular references - see +link{JSONEncoder.circularReferenceMode}.
// <smartgwt>
// <P>
// Because GWT does not support Java reflection, JSON encoding cannot discover the properties
// of an arbitrary Java POJO.  The following objects are supported:
// <ul>
// <li> any primitive type (String, Date, Number, Boolean)
// <li> any Map or Collection in any level of nesting
// <li> DataClass (Record's superclass) and RecordList
// <li> any widget (see +link{JSONEncoder.serializeInstances})
// <li> JavaScriptObject
// <li> an Array containing any of the above
// </ul>
// </smartgwt>
// <P>
// Note that using the String produced by this API with +link{JSON.decode()} <b>will not
// successfully preserve dates</b>.  Use +link{JSONEncoder.dateFormat} "dateConstructor" or
// "logicalDateConstructor" to have dates round-trip properly.
//
// @param object (Any) object to serialize
// @return (String) object encoded as a JSON String
// @visibility external
//<
encode : function (object) {
    // Use Map for cycle detection: keys are ancestor objects, values are indices into
    // _pathStack. Only ancestors tracked, so memory is O(depth) not O(nodes).
    this._ancestors = new Map();
    // Lazy path building: store path segments in a stack, only build full path string
    // when a cycle is actually detected (rare). This avoids string concatenation for
    // every object in the common case where no cycles exist.
    this._pathStack = [];
    this._pathMode = this.circularReferenceMode == "path";
    // For pretty printing, pass depth instead of prefix to avoid string concat at each level
    var retVal = this._serialize(object, this.prettyPrint ? 0 : null, null);
    this._ancestors = null;
    this._pathStack = null;
    return retVal
},

// Build full path string from path stack when a cycle is detected. Only called in the
// rare case when circularReferenceMode is "path" and an actual cycle exists.
_buildPathFromStack : function (startIndex) {
    var stack = this._pathStack,
        parts = [];
    for (var i = startIndex; i < stack.length; i++) {
        var segment = stack[i];
        if (segment.key == null) {
            // Root object with ID
            if (segment.id) parts.push(segment.id);
        } else if (typeof segment.key == "number") {
            parts.push("[" + segment.key + "]");
        } else if (isc.Comm._simpleIdentifierRE.test(segment.key)) {
            parts.push("." + segment.key);
        } else {
            parts.push('["' + segment.key + '"]');
        }
    }
    return parts.join("");
},

//> @type JSONDateFormat
// Format for encoding dates in JSON.  Note you can override +link{JSONEncoder.encodeDate()}
// for a custom format.
//
// @value "xmlSchema" dates are encoded as a String in <a target=_blank
//        href="http://www.w3.org/TR/xmlschema-2/#dateTime">XML Schema date format</a> in UTC,
//        for example, "2005-08-02" for logical date fields or "2005-08-01T21:35:48.350"
//        for <code>datetime</code> fields. See +link{group:dateFormatAndStorage,Date format and
//        storage} for more information.<br>
//        <b>Note.</b>If JSON containing xmlSchema-formatted date values is passed to 
//        +link{JSON.decodeSafe()} or +link{JSON.decodeSafeWithDates()}, these formatted date values 
//        will not be converted to actual date objects in the generated JavaScript object. 
//        Use "logicalDateString" instead.
// @value "dateConstructor" dates are encoded as raw JavaScript code for creating a Date object,
//        that is:
//        <pre>
//        new Date(1238792738633)
//        </pre>
//        This is not strictly valid JSON, but if eval()d, will result in an identical date object,
//        regardless of timezone.  However, it does not preserve the distinction between
//        logical dates vs full datetime values - use "logicalDateConstructor" mode for that.<br>
//        <b>Note.</b>This format does not work with +link{JSON.decodeSafe()}. 
//        If you need to use +link{JSON.decodeSafe()} and/or +link{JSON.decodeSafeWithDates()},
//        you will need to use "logicalDateString" instead.
// @value "logicalDateConstructor" serializes Date instances in a way that preserves the
//        distinction between logical dates, logical times, and full datetime values, as
//        explained +link{group:dateFormatAndStorage,here}.  Like 'dateConstructor' mode, this
//        does not produce strictly valid JSON, and instead embeds JavaScript calls.  
//        <p>
//        In addition, unlike 'dateConstructor' mode, using eval() to reconstruct the original
//        JavaScript objects will only work in the presence of SmartClient, and not just in a
//        generic JavaScript interpreter.<br>
//        <b>Note.</b>This format does not work with +link{JSON.decodeSafe()}. 
//        If you need to use +link{JSON.decodeSafe()} and/or +link{JSON.decodeSafeWithDates()},
//        you will need to use "logicalDateString" instead.
// @value "logicalDateString" Dates are encoded as strings in a format that 
//        +link{JSON.decodeSafeWithDates()} will recognize. This allows developers to round-trip 
//        date, time and datetime values to and from strict JSON.
//
// @visibility external
//<



//> @type JSONInstanceSerializationMode
// Controls the output of the JSONEncoder when instances of SmartClient classes (eg a ListGrid)
// are included in the data to be serialized.
//
// @value "long" instances will be shown as a specially formatted JSON listing the most
//               relevant properties of the instance. Result is not expected to
//               decode()/eval() successfully if instances are included.
// @value "short" instances will be shown in a shorter format via a call to <smartclient>
//                +link{isc.echoLeaf()}</smartclient><smartgwt>
//                {@link com.smartgwt.client.util.SC#echoLeaf SC.echoLeaf()}</smartgwt>.
//                Result is not expected to decode()/eval() successfully if instances are
//                included.
// @value "skip" no output will be shown for instances (as though they were not present in the
//               data).  Result should decode()/eval() successfully (depending on other
//               settings)
//
// @visibility external
//<


//> @attr JSONEncoder.serializeInstances (JSONInstanceSerializationMode : "long" : IR)
// Controls the output of the JSONEncoder when instances of SmartClient classes (eg a ListGrid)
// are included in the data to be serialized.  See +link{JSONInstanceSerializationMode}.
// <P>
// Note that the JSONEncoder does not support a format that will recreate the instance if
// passed to decode() or eval().
//
// @visibility external
//<
serializeInstances: "long",

//> @attr JSONEncoder.skipInternalProperties (Boolean : false : IR)
// If true, don't show SmartClient internal properties when encoding an object.
// @visibility external
//<

//> @attr JSONEncoder.skipNullValues (Boolean : false : IR)
// If true, don't include properties with null values when encoding an object.
// @visibility external
//<

//> @attr JSONEncoder.showDebugOutput (Boolean : false : IR)
// If objects that cannot be serialized to JSON are encountered during serialization, show a
// placeholder rather than just omitting them. 
// <P>
// The resulting String will not be valid JSON and so cannot be decoded/eval()'d
// @visibility external
//<


//> @attr JSONEncoder.dateFormat (JSONDateFormat : "xmlSchema" : IR)
// Format for encoding JavaScript Date values in JSON.  See +link{type:JSONDateFormat} for
// valid options, or override +link{JSONEncoder.encodeDate()} to do something custom.
// @visibility external
//<
dateFormat: "xmlSchema",

//> @method JSONEncoder.encodeDate()
// Encode a JavaScript Date value.
// <P>
// By default, follows the +link{JSONEncoder.dateFormat} setting.  <smartclient>Override to do
// custom encoding.</smartclient><smartgwt>To override the date format, all Dates should be
// converted to Strings beforehand.</smartgwt>
// 
// @param theDate (Date) JavaScript date object to be serialized
// @return (String) value to be included in result.  <b>If this value is intended to appear
//                  as a String it should include quotes (")</b>
//
// @visibility external
//<
encodeDate : function (date) {
    // If we were handed a date from some other window without our extensions on it, 
    // duplicate it.
    if (!date.toSchemaDate) {
        var newDate = new Date(date.getTime());
        // Unlikely to be set for a date picked up from another frame, but respect logical
        // date/time flags if present.
        newDate.logicalDate = this.logicalDate;
        newDate.logicalTime = this.logicalTime;
        date = newDate;
    }
    if (this.dateFormat == "dateConstructor") {
        return date._serialize();
    } else if (this.dateFormat == "logicalDateConstructor") {
        // SC-dependent - uses createLogicalDate/createLogicalTime
        if (date.logicalTime) {
            return "isc.Time.createLogicalTime(" + date.getHours() + ", " + 
                date.getMinutes() + ", " + date.getSeconds() + ", " + date.getMilliseconds() +
                ")";
        } else if (date.logicalDate) {
            return "isc.DateUtil.createLogicalDate(" + date.getFullYear() + ", " + 
                date.getMonth() + ", " + date.getDate() + ")";
        } else {
            return date._serialize();
        }
    } else if (this.dateFormat == "logicalDateString") {
        if (date.logicalTime) {
            return '"' +  isc.JSON._$timePrefix + date.getHours() + ":" + date.getMinutes() + ":" + date.getSeconds() + '"';
        } else if (date.logicalDate) {
            // $$DATE$$ uses 1-based months, so we need to add 1 to the 0-based month.
            return '"' + isc.JSON._$datePrefix + date.getFullYear() + "-" + (date.getMonth() + 1) + "-" + date.getDate() + '"';
        } else {
            return '"' + isc.JSON._$datestampPrefix + date.getTime() + '"';
        }
    } else { // quotes for xml schema
        return '"' + date.toSchemaDate(null, this.trimMilliseconds) + '"';    
    }
},

//> @attr JSONEncoder.escapeNonPrintable (Boolean : true : IRW)
// By default, obscure non-printable characters such as DC3 (Device Control 3, U+0013 hexadecimal)
// will be escaped according to JSON standards. ECMA-404 / The JSON Data Interchange Format
// requires the quotation mark (U+0022), reverse solidus (U+005C), and control characters (U+0000
// through U+001F) to be escaped.
// <p>
// These characters are very rarely used in JSON data in web applications.  If you know that
// your application does not use such characters in JSON data, there can be a performance
// advantage to setting <code>escapeNonPrintable</code> to false in order to disable the
// logic for escaping these characters.  This is a detectable difference only when dealing
// with very large JSON structures on older browsers that do not provide native support (for
// example, Internet Explorer 8).
// @visibility external
//<
escapeNonPrintable: true,

//> @attr JSONEncoder.strictQuoting (Boolean : true : IR)
// Whether all property names should be quoted, or only those property names that are not valid
// identifiers or are JavaScript reserved words (such as "true").
// <P>
// Encoding only where required produces slightly shorter, more readable output which is still
// compatible with JavaScript's eval():
// <pre>
// {
//     someProp : "someValue",
//     "true" : "otherValue",
//     otherProp : "otherValue"
// }
// </pre>
// .. but is not understood by many server-side JSON parser implementations.
// @visibility external
//<  
strictQuoting: true,

//> @type JSONCircularReferenceMode
// What the +link{JSONEncoder} should do when it encounters a circular reference in an object
// structure.
// @value "omit" circular references in Arrays will be represented as a null entry, and objects
//               will have a property with a null value
// @value "marker" leave a string marker, the +link{jsonEncoder.circularReferenceMarker},
//                 wherever a circular reference is found
// @value "path" leave a string marker <i>followed by</i> the path to the first occurrence of
//               the circular reference from the top of the object tree that was serialized.
//               This potentially allows the original object graph to be reconstructed.
// @visibility external
//<

//> @attr JSONEncoder.circularReferenceMode (JSONCircularReferenceMode : "path" : IR)
// What the JSONEncoder should do if it encounters a circular reference.
//
// @visibility external
//<
circularReferenceMode: "path",

//> @attr JSONEncoder.circularReferenceMarker (String : "$$BACKREF$$" : IR)
// The string marker used to represent circular references.  See +link{circularReferenceMode}.
//
// @visibility external
//<
circularReferenceMarker: "$$BACKREF$$",

//> @attr JSONEncoder.prettyPrint (Boolean : true : IR)
// Whether to add indentation to the returned JSON string.  This makes the returned JSON much
// easier to read but adds size.  Note that when delivering JSON responses compressed, the size
// difference between prettyPrinted JSON and normal JSON is negligible.
// @visibility external
//<
prettyPrint: true,

//>	@method	JSONEncoder._serialize()	(A)
//		Internal routine that actually does the serialization.
//		@group	serialization
//
//		@param	object	(Any)		object to serialize
//		@param	prefix	(String)	string to put before each line of serialization output
//		@param	context (Object)	context that tracks previously encountered objects and
//                                  settings
//
//		@return	(String)			serialized object as a string
//<
_serialize : function (object, depth, pathKey) {
    // Fast path for null/undefined - most common early exit
    if (object == null) return null;

    // Use isc.isA.* for primitive type checks. These handle cross-frame boxed primitives
    // (new otherFrame.String("text")) and legacy GWT strings (with .Class property).
    // When neither isc._crossFrameCompat nor isc._legacyGWTWorkaround is set, a faster
    // variant using typeof is installed below.
    if (isc.isA.String(object)) {
        if (this.escapeNonPrintable) {
            return String.asJSONString(object);
        } else {
            // In Safari a cross-frame scripting bug means asSource may not be available
            return object.asSource != null ? object.asSource() : String.asSource(object);
        }
    }
    if (isc.isA.Boolean(object)) return object;
    if (isc.isA.Number(object)) return object;
    if (isc.isA.SpecialNumber(object)) return object;
    if (typeof object == "function") return null;

    // Cache ancestors Map in local variable to reduce property access in hot path
    var ancestors = this._ancestors;

    // Cycle detection: check if object is already in our ancestor chain. Do this early
    // before any other object-type checks since it's needed for all non-primitive types.
    if (ancestors.has(object)) {
        var mode = this.circularReferenceMode;
        if (mode == "marker") {
            return "'" + this.circularReferenceMarker + "'";
        } else if (mode == "path") {
            // Lazy path building: only construct the full path string now that we've
            // actually found a cycle. This avoids string concatenation for every object
            // in the common no-cycle case.
            var cycleStartIndex = ancestors.get(object);
            var path = this._buildPathFromStack(cycleStartIndex);
            return "'" + this.circularReferenceMarker + ":" + path + "'";
        }
        return null;
    }

    // Check for arrays early since they're very common in nested structures and don't need
    // the Date/Instance/Class checks. This avoids unnecessary isc.isA.* calls per array.
    // Note: arrays can never be the window object, so no window check needed here.
    if (Array.isArray(object)) {
        var stackIndex = this._pathStack.length;
        this._pathStack.push({key: pathKey});
        ancestors.set(object, stackIndex);
        var result = this._serializeArray(object, null, null, depth);
        ancestors.delete(object);
        this._pathStack.pop();
        return result;
    }

    // Handle SGWT Java objects - rare case
    var isSGWT = isc.Browser.isSGWT;
    if (isSGWT && window.SmartGWT.isNativeJavaObject(object)) {
        if (window.SmartGWT.warnOnSerializeError) {
            window.SmartGWT.throwUnconvertibleObjectException(
                object, window.SmartGWT.serializeErrorMessage
            );
        }
        return String.asSource(object + "");
    }

    // Inline type checks to avoid function call overhead (significant for 97K+ objects).
    // Date: instanceof handles same-frame dates; cross-frame needs constructor comparison.
    // The cross-frame path is rare, so we only pay that cost when instanceof fails.
    if (object instanceof Date) return this.encodeDate(object);
    if (!isSGWT && typeof object == "object" && ("" + object.constructor) == ("" + Date)) {
        return this.encodeDate(object);
    }

    // Inline Instance/Class checks: just property existence tests, avoiding function call.
    // Instance: object._scPrototype != null; Class: object._isClassObject == true
    if (object._scPrototype != null || object._isClassObject === true) {
        if (this.serializeInstances == "skip") return null;
        else if (this.serializeInstances == "short") return isc.echoLeaf(object);
        // else "long".. fall through to logic below to have properties output
    }

    if (object == window) {
        this.logWarn("Serializer encountered the window object at path: " +
            (this._pathMode ? this._buildPathFromStack(0) : ""));
        return null;
    }

    // Push path segment onto stack (lazy path building - no string concat here)
    var stackIndex = this._pathStack.length;
    if (pathKey == null) {
        // Root object - check for ID
        var id = (object.getID && typeof object.getID == "function") ? object.getID() : null;
        this._pathStack.push({key: null, id: id});
    } else {
        this._pathStack.push({key: pathKey});
    }
    ancestors.set(object, stackIndex);

    var result;

    // if there is a serialize method associated with this object, call that
    // Custom _serialize expects string prefix, so convert depth to prefix string.
    // Inline typeof check (isc.isA.Function is just typeof == "function")
    if (typeof object._serialize == "function") {
        var prefix = depth != null ? this._getIndent(depth) : null;
        // Build path string for custom serializers that need it
        var objPath = this._pathMode ? this._buildPathFromStack(0) : "";
        result = object._serialize(prefix, null, objPath, prefix);
    }
    else {
        // Regular object serialization
        var data;
        if (object.getSerializeableFields) {
            
            data = object.getSerializeableFields([], []);
        } else {
            data = object;
        }
        result = this._serializeObject(data, null, null, depth);
    }

    // Remove from ancestors after processing children (critical for correct cycle detection)
    ancestors.delete(object);
    this._pathStack.pop();

    return result;
},

//>	@method	JSONEncoder._serializeArray()	(A)
//			Internal routine to serialize an array.
//		@group	serialization
//
//		@param	object	(Any)		object o serialize
//		@param	objPath	(String)	global variable path to this object, for serializing object references
//		@param	objRefs	(Array of Object[])	array of objects that have been serialized already so
//									 we don't get into endless loops		
//		@param	prefix	(String)	string to put before each line of serialization output 
//
//		@return	(String)			serialized object as a string
//<
_serializeArray : function (object, unused1, unused2, depth) {
    var len = object.length;
    if (len == 0) return "[]";

    // Use array join for efficient string building - avoids trailing comma removal
    var parts = new Array(len),
        childDepth = depth != null ? depth + 1 : null;

    for (var i = 0; i < len; i++) {
        // Pass array index as path key for lazy path building
        var serializedValue = this._serialize(object[i], childDepth, i);
        parts[i] = serializedValue + "";
    }

    if (depth != null) {
        // Pretty print using cached indentation strings. Inline separator for speed.
        var indent = this._getIndent(depth),
            childIndent = this._getIndent(childDepth),
            sepCache = this._sepCache,
            sep = childDepth < sepCache.length ? sepCache[childDepth] : sepCache[sepCache.length-1];
        return "[\n" + childIndent + parts.join(sep) + "\n" + indent + "]";
    }
    return "[" + parts.join(",") + "]";
},

//>	@method	JSONEncoder._serializeObject()	(A)
//			Internal routine to serialize an object.
//		@group	serialization
//
//		@param	object	(Any)		object o serialize
//		@param	prefix	(String)	string to put before each line of serialization output
//		@param	objRefs	(Array of Object[])	array of objects that have been serialized already so
//									 we don't get into endless loops		
//		@param	objPath	(String)	global variable path to this object, for serializing object references
//
//		@return	(String)			serialized object as a string
//<
_serializeObject : function (object, unused1, unused2, depth) {
    object = isc.JSONEncoder._serialize_cleanNode(object);

    try {
        
        for (var key in object) break;
    } catch (e) {
        if (this.showDebugOutput) {
            if (isc.isAn.XMLNode(object)) return isc.echoLeaf(object);
            var message;
            if (e.message) {
                message = (e.message.asSource != null ? e.message.asSource()
                                                      : String.asSource(e.message));
                return "{ cantEchoObject: " + message + "}";
            } else {
                return "{ cantEchoObject: 'unspecified error' }";
            }
        } else return null;
    }

    // Collect key:value pairs in an array for efficient joining (avoids trailing comma
    // removal which required expensive substring operations)
    var parts = [],
        childDepth = depth != null ? depth + 1 : null,
        strictQuoting = this.strictQuoting,
        simpleIdentifierRE = isc.Comm._simpleIdentifierRE,
        skipInternalProperties = this.skipInternalProperties,
        skipNullValues = this.skipNullValues,
        serializeInstances = this.serializeInstances,
        showDebugOutput = this.showDebugOutput,
        isSGWT = isc.Browser.isSGWT;

    for (var key in object) {
        if (key == null) continue;

        // Skip internal properties if configured
        if (skipInternalProperties) {
            var firstChar = key.charAt(0);
            if (firstChar == "_" || firstChar == "$") continue;
        }

        var value = object[key];

        // Skip functions - use typeof for speed
        if (typeof value == "function") continue;

        // Check for Java objects in SmartGWT environment
        var isJavaObj = isSGWT ? window.SmartGWT.isNativeJavaObject(value) : false;

        
        // Inline Instance check: isc.isAn.Instance = object._scPrototype != null
        if (key != isc.gwtRef && !isJavaObj && value != null && value._scPrototype != null &&
            serializeInstances == "skip") continue;

        // Build the key string with proper quoting
        var keyStr = key.toString(),
            isSimpleKey = simpleIdentifierRE.test(keyStr);

        if (strictQuoting || !isSimpleKey) {
            if (keyStr.indexOf('"') != -1) {
                keyStr = '"' + this.convertToEncodedQuotes(keyStr) + '"';
            } else {
                keyStr = '"' + keyStr + '"';
            }
        }

        var serializedValue;
        if (key == isc.gwtRef) {
            if (!showDebugOutput) continue;
            
            serializedValue = String.asSource("{GWT Java Obj}");
        } else if (key == isc.gwtModule) {
            if (!showDebugOutput) continue;
            serializedValue = String.asSource("{GWT Module}");
        } else if (isJavaObj) {
            serializedValue = (value == null ? null : String.asSource(value + ""));
        } else {
            // Pass key for lazy path building (no string concat, just the key)
            serializedValue = this._serialize(value, childDepth, key);
        }

        // Skip null values if configured
        if (skipNullValues && serializedValue == null) continue;

        // Append "null" string for null values to ensure proper JSON
        parts.push(keyStr + ":" + serializedValue);
    }

    if (parts.length == 0) return "{}";

    if (depth != null) {
        // Pretty print using cached indentation strings (avoids string concat per level)
        var indent = this._getIndent(depth),
            childIndent = this._getIndent(childDepth),
            sepCache = this._sepCache,
            sep = childDepth < sepCache.length ? sepCache[childDepth] : sepCache[sepCache.length-1];
        return "{\n" + childIndent + parts.join(sep) + "\n" + indent + "}";
    }
    return "{" + parts.join(",") + "}";
},

// Converts a string so that embedded ' and " characters are converted to the HTML encodings
// &apos; and &quot;  Only used if a key string contains both ' and " (otherwise, we just 
// quote it using the symbol that isn't contained in the key name)
convertToEncodedQuotes : function (string) {
    return string.replace(String._doubleQuoteRegex, "&quot;").
                  replace(String._singleQuoteRegex, "&apos;");
},
convertFromEncodedQuotes : function (string) {
    return string.replace(new RegExp("&quot;", "g"), '"').
                  replace(new RegExp("&apos;", "g"), "'");
}
});



if (!isc.Browser.isSGWT && !isc._crossFrameCompat && !isc._legacyGWTWorkaround) {
    // Fast path: No special compat needed - use typeof for maximum performance.
    // Cross-frame boxed primitives (new otherFrame.String()) will serialize as objects,
    // but this is acceptable since the user has not enabled compat flags.
    isc.JSONEncoder.addMethods({
            _serialize : function (object, depth, pathKey) {
                if (object == null) return null;

                var typeofObj = typeof object;
                if (typeofObj == "string") {
                    if (this.escapeNonPrintable) {
                        return String.asJSONString(object);
                    } else {
                        return object.asSource != null ? object.asSource() : String.asSource(object);
                    }
                }
                if (typeofObj == "boolean") return object;
                if (typeofObj == "number") {
                    return isc.isA.SpecialNumber(object) ? object : object;
                }
                if (typeofObj == "function") return null;

                var ancestors = this._ancestors;

                if (ancestors.has(object)) {
                    var mode = this.circularReferenceMode;
                    if (mode == "marker") {
                        return "'" + this.circularReferenceMarker + "'";
                    } else if (mode == "path") {
                        var cycleStartIndex = ancestors.get(object);
                        var path = this._buildPathFromStack(cycleStartIndex);
                        return "'" + this.circularReferenceMarker + ":" + path + "'";
                    }
                    return null;
                }

                if (Array.isArray(object)) {
                    var stackIndex = this._pathStack.length;
                    this._pathStack.push({key: pathKey});
                    ancestors.set(object, stackIndex);
                    var result = this._serializeArray(object, null, null, depth);
                    ancestors.delete(object);
                    this._pathStack.pop();
                    return result;
                }

                // Date check: instanceof for same-frame, constructor toString for cross-frame
                if (object instanceof Date) return this.encodeDate(object);
                if (typeof object == "object" && ("" + object.constructor) == ("" + Date)) {
                    return this.encodeDate(object);
                }

                if (object._scPrototype != null || object._isClassObject === true) {
                    if (this.serializeInstances == "skip") return null;
                    else if (this.serializeInstances == "short") return isc.echoLeaf(object);
                }

                if (object == window) {
                    this.logWarn("Serializer encountered the window object at path: " +
                        (this._pathMode ? this._buildPathFromStack(0) : ""));
                    return null;
                }

                var stackIndex = this._pathStack.length;
                if (pathKey == null) {
                    var id = (object.getID && typeof object.getID == "function") ? object.getID() : null;
                    this._pathStack.push({key: null, id: id});
                } else {
                    this._pathStack.push({key: pathKey});
                }
                ancestors.set(object, stackIndex);

                var result;
                if (typeof object._serialize == "function") {
                    var prefix = depth != null ? this._getIndent(depth) : null;
                    var objPath = this._pathMode ? this._buildPathFromStack(0) : "";
                    result = object._serialize(prefix, null, objPath, prefix);
                }
                else {
                    var data;
                    if (object.getSerializeableFields) {
                        data = object.getSerializeableFields([], []);
                    } else {
                        data = object;
                    }
                    result = this._serializeObject(data, null, null, depth);
                }

                ancestors.delete(object);
                this._pathStack.pop();

                return result;
            },

            _serializeObject : function (object, unused1, unused2, depth) {
                object = isc.JSONEncoder._serialize_cleanNode(object);

                try {
                    for (var key in object) break;
                } catch (e) {
                    if (this.showDebugOutput) {
                        if (isc.isAn.XMLNode(object)) return isc.echoLeaf(object);
                        var message;
                        if (e.message) {
                            message = (e.message.asSource != null ? e.message.asSource()
                                                                  : String.asSource(e.message));
                            return "{ cantEchoObject: " + message + "}";
                        } else {
                            return "{ cantEchoObject: 'unspecified error' }";
                        }
                    } else return null;
                }

                var parts = [],
                    childDepth = depth != null ? depth + 1 : null,
                    strictQuoting = this.strictQuoting,
                    simpleIdentifierRE = isc.Comm._simpleIdentifierRE,
                    skipInternalProperties = this.skipInternalProperties,
                    skipNullValues = this.skipNullValues,
                    serializeInstances = this.serializeInstances,
                    showDebugOutput = this.showDebugOutput;

                for (var key in object) {
                    if (key == null) continue;

                    if (skipInternalProperties) {
                        var firstChar = key.charAt(0);
                        if (firstChar == "_" || firstChar == "$") continue;
                    }

                    var value = object[key];

                    if (typeof value == "function") continue;

                    if (key != isc.gwtRef && value != null && value._scPrototype != null &&
                        serializeInstances == "skip") continue;

                    var keyStr = key.toString(),
                        isSimpleKey = simpleIdentifierRE.test(keyStr);

                    if (strictQuoting || !isSimpleKey) {
                        if (keyStr.indexOf('"') != -1) {
                            keyStr = '"' + this.convertToEncodedQuotes(keyStr) + '"';
                        } else {
                            keyStr = '"' + keyStr + '"';
                        }
                    }

                    var serializedValue;
                    if (key == isc.gwtRef) {
                        if (!showDebugOutput) continue;
                        serializedValue = String.asSource("{GWT Java Obj}");
                    } else if (key == isc.gwtModule) {
                        if (!showDebugOutput) continue;
                        serializedValue = String.asSource("{GWT Module}");
                    } else {
                        serializedValue = this._serialize(value, childDepth, key);
                    }

                    if (skipNullValues && serializedValue == null) continue;

                    parts.push(keyStr + ":" + serializedValue);
                }

                if (parts.length == 0) return "{}";

                if (depth != null) {
                    var indent = this._getIndent(depth),
                        childIndent = this._getIndent(childDepth),
                        sepCache = this._sepCache,
                        sep = childDepth < sepCache.length ? sepCache[childDepth] : sepCache[sepCache.length-1];
                    return "{\n" + childIndent + parts.join(sep) + "\n" + indent + "}";
                }
                return "{" + parts.join(",") + "}";
            }
        });
}
