
package com.smartgwt.client.docs;

/**
 * <h3>Error Handling Overview</h3>
 * {@link com.smartgwt.client.rpc.RPCResponse} and {@link com.smartgwt.client.data.DSResponse}
 * objects have an integer status field that the 
 *  RPCManager inspects when the response is received from the server. If the value of this 
 *  field is less than zero, the request is considered to have failed.  Otherwise it is 
 *  considered to have succeeded.  This value is settable via the setStatus() method call 
 *  on the server-side DSResponse and RPCResponse objects.
 *  <p>
 *  Errors in a Smart GWT application fall into two main categories:<ul>
 *  <li>Validation errors, which arise as a result of rules in the application's business logic
 *  being broken.  These are part of the normal operation of the system.  A response with 
 * validation errors has a status of {@link
 * com.smartgwt.client.rpc.RPCResponse#STATUS_VALIDATION_ERROR STATUS_VALIDATION_ERROR}</li>
 *  <li>Unrecoverable errors, which are errors with the system itself.  These are not part of 
 *  the normal operation of the system</li>
 *  </ul>
 * <b>Validation errors</b> occur when an {@link com.smartgwt.client.data.DataSource#addData add},
 * or {@link com.smartgwt.client.data.DataSource#updateData update} operation contains values that
 * do not conform
 * to {@link com.smartgwt.client.data.DataSourceField#getValidators specified validation rules}.
 * When a user is
 * {@link com.smartgwt.client.widgets.form.DynamicForm#saveData saving} or {@link
 * com.smartgwt.client.widgets.form.DynamicForm#validate validating} edits
 *  in a databound component such as a {@link com.smartgwt.client.widgets.form.DynamicForm} 
 * or {@link com.smartgwt.client.widgets.grid.ListGrid#getCanEdit canEdit:true ListGrid},
 * validation errors are handled by
 *  redrawing the component to display those errors to the user.
 *  <P>
 *  How the user sees those errors is completely configurable -
 * for example, see the DynamicForm properties {@link
 * com.smartgwt.client.widgets.form.DynamicForm#getShowErrorIcons showErrorIcons},
 *  {@link com.smartgwt.client.widgets.form.DynamicForm#getShowErrorText showErrorText}, 
 * {@link com.smartgwt.client.widgets.form.DynamicForm#getShowInlineErrors showInlineErrors}, and
 * indeed most DynamicForm properties
 *  that contain the word "Error" - but the default in most skins is to highlight the field 
 *  with some kind of error icon, and provide the actual error text message in a floating box 
 *  when the user hovers over the field.
 *  <P>
 * Note that the centralized {@link com.smartgwt.client.rpc.HandleErrorCallback#handleError
 * HandleErrorCallback.handleError()} method (see below)
 *  will not be invoked for validation errors that occurred while editing data in a component.
 *  <P>
 *  Validation errors can also occur when application code directly invokes dataSource APIs to 
 * save data instead of calling {@link com.smartgwt.client.widgets.form.DynamicForm#saveData
 * saveData()} on an edit component.
 * (See {@link com.smartgwt.client.data.DataSource#addData DataSource.addData()}, {@link
 * com.smartgwt.client.data.DataSource#updateData DataSource.updateData()}). In this case, since
 *  there is no component in which validation errors can be displayed, 
 * {@link com.smartgwt.client.rpc.HandleErrorCallback#handleError centralized error handling}
 * <b>will</b> be invoked by default.
 *  <p>
 *  In addition to validation errors, RPCRequests and DSRequests can fail due to errors with
 *  the system itself. For example:<ul>
 *  <li>A network transport problem</li>
 *  <li>A server-side crash</li>
 *  <li>An update failed because a transaction was rolled back</li>
 *  </ul>
 *  Errors like this can either be handled centrally, or you can choose to handle them in your
 * regular callback code. The {@link com.smartgwt.client.rpc.RPCRequest#getWillHandleError
 * RPCRequest.willHandleError} attribute determines whether or
 *  not central error handling will be invoked.
 *  <p>
 *  <b>Centralized Error Handling</b><br>
 *  If the status field for a request shows a failure, and <code>willHandleError</code> has
 * not been set to true for, {@link com.smartgwt.client.rpc.HandleErrorCallback#handleError
 * HandleErrorCallback.handleError()} will be invoked, 
 *  (unless the error was a validation error occurring while editing data in a 
 *  DataBoundComponent, as discussed above).
 *  <P>
 *  By default, <code>RPCManager.handleError()</code> logs a warning and shows a dialog 
 * with the contents of the response's {@link com.smartgwt.client.rpc.RPCResponse#getData data}
 * field (which is assumed 
 *  to contain a meaningful description of the error that occurred).  If you specified a 
 *  callback in your request, it will <b>not</b> be called.
 *  <P>
 *  This default behavior means that any Smart GWT application has a basic handling mechanism for 
 *  unrecoverable errors, without any code to write.
 *  <p>
 *  You can customize centralized error handling at two levels:<ul>
 *  <li>
 *  
 *  Use 
 * {@link
 * com.smartgwt.client.rpc.RPCManager#setHandleErrorCallback(com.smartgwt.client.rpc.HandleErrorCallback)}
 *  to provide your own error handling logic (note
 *  that customization takes place at the static class level, not per-instance)</li>
 *  <li>
 *  
 *  Use 
 * {@link
 * com.smartgwt.client.rpc.RPCManager#setHandleTransportErrorCallback(com.smartgwt.client.rpc.HandleTransportErrorCallback)}.
 *  This logic is called earlier than
 *  handleError, and it is called even when you are using custom error handling (discussed
 *  below).  It is intended to allow your code to inspect the failed response early in the 
 *  handling flow, to see if it is really unrecoverable.  For example, a failure might have 
 *  happened because of a temporary network problem, so resubmitting the request may be a valid
 *  thing to do to try to resolve the error.  Note, as with handleError, this is a static 
 *  class-level customization</li>
 *  </ul>
 *  <p>
 *  <b>Custom Error Handling</b><br>
 *  As an alternative to handling errors centrally, you can handle them in your regular callback
 * methods.  To do this, specify {@link com.smartgwt.client.rpc.RPCRequest#getWillHandleError
 * willHandleError} as a 
 * request property.  When you do this, {@link
 * com.smartgwt.client.rpc.HandleErrorCallback#handleError HandleErrorCallback.handleError()} is
 * bypassed, 
 *  and your callback is invoked as normal. If you want to invoke the default 
 * error handling logic in your callback, you can use {@link
 * com.smartgwt.client.rpc.RPCManager#runDefaultErrorHandling
 * RPCManager.runDefaultErrorHandling()}
 *  <P>
 *  <i>Note: if <code>handleTransportError()</code> was configured it <b>will</b> be called
 *  for error codes indicating a transport error even if <code>willHandleError</code> is true.</i>
 *  <P>
 *  For <code>willHandleError:true</code> requests, your callback code can determine that 
 *  it is in error state by inspecting the status property on the response. Any value 
 *  less than zero indicates an error. See the documented status codes such as
 * {@link com.smartgwt.client.rpc.RPCResponse#STATUS_FAILURE STATUS_FAILURE} for more information
 * on specific error codes that
 *  may be returned by the Smart GWT server.
 *  <P>
 *  Note that if validation failed for a save request on an edit component, 
 *  setting <code>willHandleError</code> to true will cause your callback to be invoked, 
 *  but the normal behavior of populating the field errors onto the form 
 *  and redrawing it <b>also</b> takes place.
 *  <p>
 *  Note, when you are handling errors in user callbacks, an error status other than 
 * {@link com.smartgwt.client.rpc.RPCResponse#STATUS_VALIDATION_ERROR STATUS_VALIDATION_ERROR}
 * typically indicates some sort of serious, 
 *  unrecoverable error.  Therefore, ensure that your error handling 
 *  code does not assume that the response will be properly formed or contain particular 
 *  elements.
 *  <p>
 *  You can specify <code>willHandleError</code> (or any other DSRequest/RPCRequest property)
 *  on a component request by providing the DSRequest Properties parameter.  For example, on
 *  a {@link com.smartgwt.client.widgets.grid.ListGrid#fetchData ListGrid.fetchData()}:
 *  
 *  <pre>
 *      DSRequest properties = new DSRequest();
 *      properties.setWillHandleError(true);
 *      listGrid.fetchData(new Criteria(), new DSCallback() {
 *          public void execute(DSResponse response, Object rawData, DSRequest request) {
 *              if (response.getStatus() < 0) {
 *                  // Error handling here
 *              } else {
 *                  // Normal processing here
 *              }
 *          }
 *      }, <b>properties</b>);
 *  </pre>
 *  
 *  <b>Error Status Codes</b><br>
 *  The error status codes used by the framework are documented as class variables of the 
 * {@link com.smartgwt.client.rpc.RPCResponse RPCResponse class}.  Status codes in the range -1 to
 * -100 are 
 *  reserved for use by the framework; if you wish to introduce new custom error statuses for 
 *  your own use, avoid this range.
 *  <p>
 *  <b>Errors indicating login is required</b><br>
 *  Some of the framework error statuses indicate that login is required, typically because a 
 *  user's HTTP session has timed out.  The best way to handle this situation is to use the 
 * built-in {@link com.smartgwt.client.docs.Relogin re-login flow} to automatically prompt users
 * to log back
 *  in as required, and then resubmit the requests that triggered the login required response.
 *  <p>
 *  <b>Errors during a file download</b>
 *  <P>
 * If the server responds to a {@link com.smartgwt.client.rpc.RPCRequest#getDownloadResult file
 * download request}
 *  with an error status code, standard error handling will be invoked.
 *  <P>
 *  This includes:
 *  <ul>
 * <li> export operations such as {@link com.smartgwt.client.widgets.grid.ListGrid#exportData
 * exportData()},
 *       {@link com.smartgwt.client.widgets.grid.ListGrid#exportClientData exportClientData()}, or
 *       {@link com.smartgwt.client.rpc.RPCManager#exportContent exportContent()}.
 * <li> downloading of binary field values via {@link
 * com.smartgwt.client.data.DataSource#downloadFile DataSource.downloadFile()}
 * <li> custom download operations where the {@link
 * com.smartgwt.client.rpc.RPCRequest#getDownloadResult RPCRequest.downloadResult} flag is set
 *  </ul>
 *  However if a server encounters an error while
 * {@link com.smartgwt.client.data.DSRequest#getStreamResults streaming a response} to the browser
 * this will not
 * trigger {@link com.smartgwt.client.rpc.HandleErrorCallback#handleError
 * HandleErrorCallback.handleError()} and the centralized error handling pathway.  
 *  <p>
 * Therefore if you have an error condition that could arise in the middle of a file download,
 * best
 *  practice is to:
 *  <ul>
 *  <li> <i>pre-validate the request</i>: do an ordinary, non-download request to check all
 *       common error conditions, before the request that actually initiates a download.
 *       This can avoid problems like a user who tries to download after their session has
 *       timed out, or tries to download a file that another user has deleted
 *  <li> <i>return a valid file containing a user-friendly error message</i>: for example,
 *       if the download is for an Excel spreadsheet but the database was unexpectedly
 *       unavailable, return a valid spreadsheet containing just the error message.
 *  </ul>
 *  <p>
 *  <b>Unrecoverable server error handling</b><br>
 *  Unrecoverable server <code>Exception</code> will be written to HTTP response as a warning
 *  containing the exception message. Depending on framework setting
 * <code>servlet.sendStackTraceToClient</code> (boolean) exception stacktrace can also be
 * included.
 * @see com.smartgwt.client.data.DataSource#getSimpleErrors
 * @see com.smartgwt.client.rpc.HandleErrorCallback#handleError
 * @see com.smartgwt.client.rpc.RPCManager#runDefaultErrorHandling
 * @see com.smartgwt.client.rpc.HandleTransportErrorCallback#handleTransportError
 * @see com.smartgwt.client.data.events.ErrorEvent
 * @see com.smartgwt.client.widgets.form.fields.FormItem#clearErrors
 * @see com.smartgwt.client.widgets.form.fields.FormItem#setErrors
 * @see com.smartgwt.client.widgets.form.fields.FormItem#hasErrors
 * @see com.smartgwt.client.data.DSResponse#getStatus
 * @see com.smartgwt.client.data.DSResponse#getQueueStatus
 * @see com.smartgwt.client.data.DSResponse#getErrors
 * @see com.smartgwt.client.data.DSRequest#getCallback
 * @see com.smartgwt.client.rpc.RPCRequest#getCallback
 * @see com.smartgwt.client.rpc.RPCRequest#getWillHandleError
 */
public interface ErrorHandling {
}
