/*
 * Isomorphic SmartGWT web presentation layer
 * Copyright 2000 and beyond Isomorphic Software, Inc.
 *
 * OWNERSHIP NOTICE
 * Isomorphic Software owns and reserves all rights not expressly granted in this source code,
 * including all intellectual property rights to the structure, sequence, and format of this code
 * and to all designs, interfaces, algorithms, schema, protocols, and inventions expressed herein.
 *
 *  If you have any questions, please email <sourcecode@isomorphic.com>.
 *
 *  This entire comment must accompany any portion of Isomorphic Software source code that is
 *  copied or moved from this file.
 */
package com.smartgwt.sample.showcase.client.tour;

import java.util.HashMap;
import java.util.Map;

import com.smartgwt.client.rpc.LoadScreenCallback;
import com.smartgwt.client.rpc.RPCManager;
import com.smartgwt.client.widgets.Canvas;
import com.smartgwt.client.widgets.Label;
import com.smartgwt.client.widgets.IButton;
import com.smartgwt.client.widgets.Window;
import com.smartgwt.client.widgets.events.ClickEvent;
import com.smartgwt.client.widgets.events.ClickHandler;
import com.smartgwt.sample.showcase.client.PanelFactory;
import com.smartgwt.sample.showcase.client.ShowcasePanel;
import com.smartgwt.sample.showcase.client.SourceEntity;
import com.smartgwt.client.widgets.form.DynamicForm;
import com.smartgwt.client.widgets.form.SearchForm;
import com.smartgwt.client.widgets.form.fields.ComboBoxItem;
import com.smartgwt.client.widgets.form.fields.FormItemFunctionContext;
import com.smartgwt.client.widgets.grid.ListGrid;
import com.smartgwt.client.widgets.grid.ListGridRecord;
import com.smartgwt.client.widgets.grid.events.CellContextClickEvent;
import com.smartgwt.client.widgets.grid.events.CellContextClickHandler;
import com.smartgwt.client.widgets.grid.events.CellSavedEvent;
import com.smartgwt.client.widgets.grid.events.CellSavedHandler;
import com.smartgwt.client.widgets.grid.events.RecordClickEvent;
import com.smartgwt.client.widgets.grid.events.RecordClickHandler;
import com.smartgwt.client.widgets.menu.Menu;
import com.smartgwt.client.widgets.menu.events.ItemClickEvent;
import com.smartgwt.client.widgets.menu.events.ItemClickHandler;
import com.smartgwt.client.widgets.tab.TabSet;
import com.smartgwt.client.widgets.tab.events.TabSelectedEvent;
import com.smartgwt.client.widgets.tab.events.TabSelectedHandler;
import com.smartgwt.client.widgets.tree.TreeGrid;
import com.smartgwt.client.widgets.tree.events.NodeClickEvent;
import com.smartgwt.client.widgets.tree.events.NodeClickHandler;
import com.smartgwt.client.widgets.viewer.DetailViewer;
import com.smartgwt.client.widgets.viewer.DetailViewerRecord;
import com.smartgwt.client.data.Criteria;
import com.smartgwt.client.data.Record;
import com.smartgwt.client.data.DSCallback;
import com.smartgwt.client.data.DSResponse;
import com.smartgwt.client.data.DSRequest;
import com.smartgwt.client.data.DataSource;
import com.smartgwt.client.util.JSOHelper;
import com.smartgwt.client.util.tour.Tour;
import com.smartgwt.client.util.workflow.Process;  
import com.smartgwt.client.callbacks.ProcessCallback;

public class CompleteApplicationTutorial extends ShowcasePanel {

    private static final String DESCRIPTION =
            "The tutorial system makes it extremely easy to build interactive tutorials for your product. "+
            "In most cases, the entire tutorial can be declared in XML, with no coding required." + 
            "<P>" + 
            "Each step is defined with some instruction text, a target and an action to take on that target. "+
            "The tutorial system handles everything else:" + 
            "<ul>" + 
            "<li>creating an instruction window, outlining the target, and creating an arrow pointing at the target, "+
            	"all automatically placed</li>" + 
            "<li>blocking interactivity everywhere except the target</li>" + 
            "<li>reminding the user of the correct target or action if they do something wrong (like enter the wrong text)</li>" + 
            "<li>detecting when the user has completed the intended action, and proceeding to the next step </li>" + 
            "</ul>" + 
            "Even complex interactions like drag & drop can be declared as tutorial steps, without writing code!.";

    public static class Factory implements PanelFactory {

        private String id;

        public ShowcasePanel create() {
            CompleteApplicationTutorial panel = new CompleteApplicationTutorial();
            id = panel.getID();
            return panel;
        }

        public String getID() {
            return id;
        }

        public String getDescription() {
            return DESCRIPTION;
        }
    }

    @Override
    protected boolean isTopIntro() {
        return false;
    }

    private ListGrid itemList = null;
    private TabSet itemDetailTabs = null;
    private DetailViewer itemViewer = null;
    private DynamicForm editForm = null;
    private TreeGrid categoryTree = null;
    private Label editorMessage = null;
    private SearchForm findForm = null;
    private Menu itemListMenu = null;
    private IButton findButton = null;
    private Process tutorial = null;

    private void launchTutorial() {
        
        Tour.loadTour("tutorial", new ProcessCallback() {  
            @Override  
            public void execute(Process process) {  
                /*process.addFinishedHandler(new FinishedHandler() {
                    @Override
                    public void onFinished(ProcessFinishedEvent event) {
                        SC.say("You've completed the tour - we'll leave the application up for you to play with.<P>" + 
                                "If you want to re-run the tour, quit and re-launch the application from the 'Show Example' button");
                    }
                });*/
                tutorial = process;
            }  
        });

        final Window tutorialLaunchWindow = new Window();
        tutorialLaunchWindow.setTitle("Complete Application Tutorial");
        tutorialLaunchWindow.setAutoCenter(true);
        tutorialLaunchWindow.setAutoSize(true);
        Map<Object, Object> bodyDefaults = new HashMap<Object, Object>();
        bodyDefaults.put("defaultLayoutAlign", "center");
        bodyDefaults.put("layoutMargin", 10);
        tutorialLaunchWindow.setBodyDefaults(bodyDefaults);
    	
        final Label label = new Label();
        label.setWrap(false);
        label.setContents("Click 'Start Tutorial' to learn how to use this application!");
    	
        final IButton tutorialButton = new IButton();
        tutorialButton.setTitle("Start Tutorial");
        tutorialButton.disable();
        tutorialButton.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                tutorial.start();
                tutorialLaunchWindow.hide();
            }
        });    	
        tutorialLaunchWindow.addItem(label);
        tutorialLaunchWindow.addItem(tutorialButton);
    	
        DataSource supplyItem = DataSource.get("supplyItem");
        Record record = new Record();
        record.setAttribute("itemID", 1996);
        record.setAttribute("itemName", "Sugar White 1KG");
        record.setAttribute("unitCost", 2.45);
        record.setAttribute("inStock", true);
        record.setAttribute("units", "Pkt");
        record.setAttribute("SKU", "85201400");
        record.setAttribute("category", "Canteen and Washroom Products");
    	
        supplyItem.updateData(record, new DSCallback() {
            @Override
            public void execute(DSResponse dsResponse, Object data, DSRequest dsRequest) {
                tutorialButton.enable();
            }
        });

        tutorialLaunchWindow.show();
    }

    public Canvas getViewPanel() {
        final Canvas layout = new Canvas();
        layout.setWidth100();
        RPCManager.loadScreen("completeApplicationTutorial", new LoadScreenCallback() {
            @Override
            public void execute() {
                findButton = (IButton) Canvas.getById("findButton");
                findButton.addClickHandler(new com.smartgwt.client.widgets.events.ClickHandler() {
                    @Override
                    public void onClick(com.smartgwt.client.widgets.events.ClickEvent event) {
                        findItems(null);
                    }
                });
                itemList = (ListGrid) Canvas.getById("itemList"); 
                itemList.addCellContextClickHandler(new CellContextClickHandler() {
                    @Override
                    public void onCellContextClick(CellContextClickEvent event) {
                        event.cancel();
                        itemListMenu.showContextMenu();
                    }
                });
                itemList.addRecordClickHandler(new RecordClickHandler() {
                    @Override
                    public void onRecordClick(RecordClickEvent event) {
                        updateDetails();
                    }
                });
                itemList.addCellSavedHandler(new CellSavedHandler() {
                    @Override
                    public void onCellSaved(CellSavedEvent event) {
                        updateDetails();
                    }
                });
                itemDetailTabs = (TabSet) Canvas.getById("itemDetailTabs");
                itemDetailTabs.addTabSelectedHandler(new TabSelectedHandler() {
                    @Override
                    public void onTabSelected(TabSelectedEvent event) {
                        updateDetails();
                    }
                });
                itemViewer = (DetailViewer) Canvas.getById("itemViewer");
                editForm = (DynamicForm) Canvas.getById("editForm");
                categoryTree = (TreeGrid) Canvas.getById("categoryTree");
                categoryTree.addNodeClickHandler(new NodeClickHandler() {
                    @Override
                    public void onNodeClick(NodeClickEvent event) {
                        findItems(event.getNode().getAttribute("categoryName"));
                    }
                });
                editorMessage = (Label) Canvas.getById("editorMessage");
                findForm = (SearchForm) Canvas.getById("findForm");
                ComboBoxItem cbi = (ComboBoxItem)findForm.getItem("itemName");
                cbi.setPickListFilterCriteriaFunction(new com.smartgwt.client.widgets.form.fields.FormItemCriteriaFunction() {
                    @Override
                    public Criteria getCriteria(FormItemFunctionContext itemContext) {
                        Criteria criteria = new Criteria();
                        if (findForm.getValue("findInCategory") != null && categoryTree.getSelectedRecords().length == 1) {
                            criteria.addCriteria("category", categoryTree.getSelectedRecord().getAttribute("categoryName"));
                        }
                        return criteria;
                    }
                });
                
                itemListMenu = (Menu) Canvas.getById("itemListMenu");
                itemListMenu.addItemClickHandler(new ItemClickHandler() {
                    @Override
                    public void onItemClick(ItemClickEvent event) {
                        String menu = event.getItem().getAttribute("name");
                        if (menu.equals("newItem")) {
                            itemList.deselectAllRecords();
                            itemDetailTabs.selectTab(1);
                            updateDetails();
                        } else if (menu.equals("showDetails")) {
                            itemDetailTabs.selectTab(0);
                            updateDetails();
                        } else if (menu.equals("editItem")) {
                            itemDetailTabs.selectTab(1);
                            updateDetails();
                        } else if (menu.equals("deleteItem")) {
                            if (itemList.getSelectedRecord() != null) {
                                itemList.removeSelectedData();
                                clearDetails();
                            }
                        }
                    }
                });

                layout.addChild(this.getScreen());
                
                categoryTree.fetchData();

                launchTutorial();
            }
        }, new String[] {RPCManager.ALL_GLOBALS});

        return layout;
    }

    public void findItems(String categoryName) {
        Map findValues = null;
        
        if (findForm.getValue("findInCategory") != null && categoryTree.getSelectedRecords().length == 1) {
            // use tree category and form values
            if (categoryName == null) categoryName = categoryTree.getSelectedRecord().getAttribute("categoryName");
            
            findValues = findForm.getValues();
            findValues.put("category", categoryName);
            
        } else if (categoryName == null) {
            // use form values only
            findValues = findForm.getValues();
            
        } else {
            // use tree category only
            findValues = new HashMap();
            findValues.put("category", categoryName);
        }
        
        itemList.filterData(new Criteria(JSOHelper.convertMapToJavascriptObject(findValues)));
        
        clearDetails();
    }
    
    public void updateDetails() {
        ListGridRecord record = itemList.getSelectedRecord();
        if (record == null) {
            clearDetails();
        } else if (itemDetailTabs.getSelectedTabNumber() == 0) {
            // View tab: show selected record
            itemViewer.setData(new ListGridRecord[] { record });
        } else {
            // Edit tab: edit selected record
            itemDetailTabs.updateTab(itemDetailTabs.getTab("editTab"), editForm);
            editForm.editRecord(record);
        }
    }
    
    public void clearDetails() {
        int selectedTab = itemDetailTabs.getSelectedTabNumber();
        if (selectedTab == 0) {
            // View tab: show empty message
            itemViewer.setData(new DetailViewerRecord[0]);
        } else if (selectedTab == 1) {
            // Edit tab: show new record editor, or empty message
            if (categoryTree.getSelectedRecord() != null) {
                itemDetailTabs.updateTab(itemDetailTabs.getTab("editTab"), editForm);
                ListGridRecord rec = new ListGridRecord();
                rec.setAttribute("category", categoryTree.getSelectedRecord().getAttribute("categoryName"));
                editForm.editNewRecord(rec);
            } else {
                itemDetailTabs.updateTab(itemDetailTabs.getTab("editTab"), editorMessage);
            }
        }
    }

    @Override
    public String getIntro() {
        return DESCRIPTION;
    }

    @Override
    protected boolean isRightIntro() {
        return true;
    }
    
    @Override
    protected boolean isFullscreenMode() {
        return true;
    }

    @Override
    public SourceEntity[] getSourceUrls() {
        return new SourceEntity[]{
            new SourceEntity("completeApplicationTutorial.ui.xml", XML, "source/completeApplicationTutorial.ui.xml.html", true),
            //new SourceEntity("tutorial.proc.xml", XML, "source/tutorial.proc.xml.html", true),
            new SourceEntity("supplyCategory.ds.xml", XML, "source/supplyCategory.ds.xml.html", true),
            new SourceEntity("supplyItem.ds.xml", XML, "source/supplyItem.ds.xml.html", true)
        };
    }
}
