/**
* An ordinary JavaScript as obtained by "new Object()" or via 
*  ObjectLiteral syntax.
*  
*  Methods that return Objects or take Objects as parameters make use of the ability of a
*  JavaScript Object to contain an arbitrary set of named properties, without requiring
*  declaration in advance. This capability makes it possible to use a JavaScript Object much
*  like a HashMap in Java or .NET, but without the need to call get() or set() to create and
*  retrieve properties.
*  
*  For example if you created an Object using ObjectLiteral syntax
*  like so:
*  
*   var request = {
*     actionURL : "/foo.do",
*     showPrompt:false
*   };
*  
*  You could then access it's properties like so:
*  
*   var myActionURL = request.actionURL;
*   var myShowPrompt = request.showPrompt;
*  
*  .. and you could assign new values to those properties like so:
*  
*   request.actionURL = "newActionURL";
*   request.showPrompt = newShowPromptSetting;
*  
*  Note that while JavaScript allows you to get and set properties in this way on any Object,
*  SmartClient components require that if a setter or getter exists, it must be called, or no
*  action will occur. For example, if you had a ListGrid and you wanted to change the
*  ListGrid.showHeader property:
*  
*    myListGrid.setShowHeader(false); // correct
*    myListGrid.showHeader = false; // incorrect (nothing happens)
*  
*  All documented attributes have flags (eg IRW) that indicate when direct
*  property access is allowed or not.
*/
interface Object {

}
interface ObjectConstructor {

}/**
* Generic extensions to JavaScript Arrays. You can call these on any Array.
*  
*  JavaScript's native Array is retrofitted to support the List API.
*/
interface Array<T> {


/**
*  Return the item at a particular position
* 
* @param {number} position of the element to get
*/
get(pos:number): object;

/**
*  Return the number of items in this list
* 
* 
*/
getLength(): number;

/**
*  Return whether or not this array is empty
* 
* 
*/
isEmpty(): boolean;

/**
*  Return the first item in this list
* 
* 
*/
first(): any;

/**
*  Return the last item in this list
* 
* 
*/
last(): any;

/**
*  Return the position in the list of the first instance of the specified object.
*  
*  If pos is specified, starts looking after that position.
*  
*  Returns -1 if not found.
* 
* 
* @param {any} object to look for
* @param {number=} earliest index to consider
* @param {number=} last index to consider
*/
indexOf(obj:any, pos?:number, endPos?:number): number;

/**
*  Return the position in the list of the last instance of the specified object.
*  
*  If pos is specified, starts looking before that position.
*  
*  Returns -1 if not found.
* 
* 
* @param {any} object to look for
* @param {number=} last index to consider
* @param {number=} earliest index to consider
*/
lastIndexOf(obj:any, pos?:number, endPos?:number): number;

/**
*  Return if this list contains the specified object.
*  
*  If pos is specified, starts looking after that position.
* 
* 
* @param {any} item to look for
* @param {number=} optional position in the list to look after
*/
contains(obj:any, pos?:number): boolean;

/**
*  Return whether this list contains all the item in the specified list.
* 
* 
* @param {List} items to look for
*/
containsAll(list:isc.List): boolean;

/**
*  Return the list of items that are in both this list and the passed-in list(s).
* 
* 
* @param {any} lists to intersect with
*/
intersect(lists:any): isc.List;

/**
*  Return whether this list is equal to another list.
*  
*  Two lists are equal only if they have the same length and all contained items are in the same
*  order and are also equal.
* 
* 
* @param {List} list to check for equality
*/
equals(list:isc.List): boolean;

/**
*  Return the items at a list of specified positions.
* 
* 
* @param {Array<Partial<number>>} array of positions
*/
getItems(itemList:Array<Partial<number>>): Array<any>;

/**
*  Return the items between position start and end, non-inclusive at the end.
* 
* 
* @param {number} start position
* @param {number} end position
*/
getRange(start:number, end:number): Array<any>;

/**
*  Return an Array that is a shallow copy of the list, that is, containing the same items.
* 
* 
*/
duplicate(): Array<any>;

/**
*  Change the array element at a particular position.
*  
*  set() can be used to expand the length of the list.
* 
* 
* @param {number} position in the list to change
* @param {object} new value for that position
*/
set(pos:number, obj:object): object;

/**
*  Add a single item to this array at a specific position in the list, sliding other items over
*  to fit.
* 
* 
* @param {object} object to add
* @param {number} position in the list to add at
*/
addAt(obj:object, pos:number): object;

/**
*  Remove the item at the specified position, rearranging all subsequent items to fill the gap
* 
* 
* @param {number} position to remove
*/
removeAt(pos:number): any;

/**
*  Add an object to this list, at the end
* 
* 
* @param {any} object to add
*/
add(object:any): any;

/**
*  Add a list of items to this array.
*  
*  Note: you can specify that a subset range be added by passing start and end indices
* 
* 
* @param {Array<any>} list of items to add
* @param {number=} optional start index in list
* @param {number=} optional end index in list (non-inclusive)
*/
addList(list:Array<any>, listStartRow?:number, listEndRow?:number): isc.List;

/**
*  Set the length of this list.
*  
*  If the length of the list is shortened, any elements past the new length of the list are removed.
*  If the length is increased, all positions past the old length have the value
*  undefined.
* 
* 
* @param {number} new length
*/
setLength(length:number): void;

/**
*  Add list of items list to this array at item pos. All items after array[pos] will slide down to
*  fit new items.
* 
* 
* @param {Array<any>} new array of items
* @param {number} position in this list to put the new items
*/
addListAt(list:Array<any>, pos:number): Array<any>;

/**
*  Remove first instance of the passed object from this array, sliding other items around to
*  fill gaps.
* 
* 
* @param {any} item to remove
*/
remove(obj:any): boolean;

/**
*  Remove all instances of objects in the specified list from this list, sliding the remaining
*  objects around to fill gaps.
* 
* 
* @param {Array<any>} list of items to remove
*/
removeList(list:Array<any>): isc.List;

/**
*  Method called when this array changes in some way. Observe the method to react to changes in
*  this list.
*  
*  Note: dataChanged() will only fire when items are added, removed or rearranged. If a list
*  contains objects, dataChanged() will not fire if changes are made to objects within the list
*  without changing their position within the list. If an observer of dataChanged() needs to react
*  to such a change, you can manually fire dataChanged() by simply calling it.
*  
*  Note: may be called multiple times as the result of a multi-item add or remove, etc.
* 
* 
*/
dataChanged(): void;

/**
*  Return a new Array where the value of item i is the value of "property" of item i in this
*  array. If an item doesn't have that property or is null, return item will be null.
* 
* 
* @param {string} name of the property to look for
*/
getProperty(property:string): Array<any>;

/**
*  Get a map of the form { item[idField] -&gt; item[displayField] }, for all 
*  items in the list. Note that if more than one item has the same idProperty, 
*  the value for the later item in the list will clobber the value for the earlier item.
* 
* 
* @param {string} Property to use as ID (data value) in the valueMap
* @param {string} Property to use a display value in the valueMap
*/
getValueMap(idField:string, displayField:string): object;

/**
*  Calls a function for each member of an array, passing in the member, its index and
*  the array itself as arguments. Returns a new array containing the
*  resulting values.
*  
*  This behavior is part of the
*  ECMA-262 specification.
*  
*  Backwards compatibility note: Historically SmartClient provided a version of
*  array.map() which differed from the native behavior in a couple of ways:
*  
*  
* - If passed a string as the function argument, it would invoke a same-named method on
*   each member of the array. This is now deprecated in favor of 
*   calling Array.callMethod directly
*  
* - If additional arguments other than the function were passed to this
*   method, when the function was invoked for each member, these additional arguments
*   would be passed in when the function was invoked. This is also deprecated as it
*   conflicts with the default native implementation
*  
* 
* @param {Function} function to execute for each item
*/
map(method:Function): Array<any>;

/**
*  Calls a method on each member of an array and returns a new array containing the
*  resulting values.
*  
*  The method argument should be the String name of a
*  method present on each item, which will be invoked. If any item is null
*  or lacks the named method, null will be returned for that item.
*  
*  Examples:
*   // line up widgets at 20 pixels from page edge
*   [widget1, widget2].callMethod("setPageLeft", 20);
* 
*   // find furthest right widget
*   [widget1, widget2].callMethod("getPageRight").max();
*  
*  
* 
* @param {string} Name of method to execute for on item
* @param {any=} arguments to pass to the method                      invoked on each item
*/
callMethod(method:string, ...args:any[]): Array<any>;

/**
* 	Set item[property] = value for each item in this array.
* 
* @param {string} name of the property to set
* @param {any} value to set to
*/
setProperty(property:string, value:any): void;

/**
*  Delete property in each item in this array.
* 
* @param {string} name of the property to clear
*/
clearProperty(property:string): boolean;

/**
*  Return a list of each unique item in this list exactly once.
*  
*  Returns in the same order they were found in the list.
*  
*  Usage example:
*  &nbsp;&nbsp;&nbsp;&nbsp;uniqueList = myArray.getProperty("foo").getUniqueItems();
* 
* 
*/
getUniqueItems(): Array<any>;

/**
*  Find the index of the first Object where property == value in the object.
*  
*  Pass an Object instead to match multiple properties.
*  
*  Note: for string values, matches are case sensitive.
* 
* 
* @param {string | object | AdvancedCriteria} property to match, or if an Object is passed, set of                    properties and values to match
* @param {any=} value to compare against (if propertyName is a string)
*/
findIndex(propertyName:string | object | isc.AdvancedCriteria, value?:any): isc.Integer;

/**
*  Like Array.findIndex, but inspects a range from startIndex to endIndex.
*  
*  
*  For convenience, findNextIndex() may also be called with a function (called the predicate
*  function) for the propertyName parameter. In this usage pattern, the predicate
*  function is invoked for each value of the list until the predicate returns a true value.
*  The predicate function is passed three parameters: the current value, the current index, and
*  the list. The value of this when the predicate function is called is the
*  value parameter. For example:
*  var currentUserRecord = recordList.findNextIndex(0, function (record, i, recordList) {
*   if (record.username == currentUsername && !record.accountDisabled) {
*     return true;
*   }
* });
*  
* 
* 
* @param {Integer} first index to consider.
* @param {string | Function | object | AdvancedCriteria} property to match; or, if a function is passed, the predicate function to call; or, if an object is passed, set of properties and values to match.
* @param {any=} value to compare against (if propertyName is a string) or the value of this when the predicate function is invoked (if propertyName is a function)
* @param {Integer=} last index to consider (inclusive).
*/
findNextIndex(startIndex:isc.Integer, propertyName:string | Function | object | isc.AdvancedCriteria, value?:any, endIndex?:isc.Integer): isc.Integer;

/**
*  Like Array.findIndex, but returns the object itself instead of its index.
* 
* 
* @param {string | object | AdvancedCriteria} property to match, or if an Object is passed, set of                    properties and values to match
* @param {any=} value to compare against (if propertyName is a string)
*/
find(propertyName:string | object | isc.AdvancedCriteria, value?:any): object;

/**
*  Determine whether this array contains any members where the property passed in matches the value
*  passed in.
* 
* 
* @param {string} property to look for   	(object)	key:value pairs to look for
* @param {any=} value to compare against (if property is a string)
*/
containsProperty(property:string, value?:any): boolean;

/**
*  Find all objects where property == value in the object.
*  
*  Pass an Object as the propertyName argument to match multiple properties.
* 
* 
* @param {string | object | AdvancedCriteria} property to match, or if an Object is passed, set of                    properties and values to match
* @param {any=} value to compare against (if propertyName is a string)
*/
findAll(propertyName:string | object | isc.AdvancedCriteria, value?:any): Array<any>;

/**
*  Slide element at position start to position destination, moving all the other elements to cover
*  the gap.
* 
* 
* @param {number} start position
* @param {number} destination position for the value at start
*/
slide(start:number, destination:number): void;

/**
*  Slide a range of elements from start to end to position destination, moving all the other
*  elements to cover the gap.
* 
* 
* @param {number} start position
* @param {number} end position (exclusive, like substring() and slice())
* @param {number} destination position for the range
*/
slideRange(start:number, end:number, destination:number): void;

/**
*  Make an index for the items in this Array by a particular property of each item.
*  
*  Returns an Object with keys for each distinct listItem[property] value. Each key will point
*  to an array of items that share that property value. The sub-array will be in the same order
*  that they are in this list.
* 
* 
* @param {string} names of the property to index by
* @param {boolean} [false] if true, we always make an array for every index. if false, we make an Array only        when more than one item has the same value for the index property
*/
makeIndex(property:string, alwaysMakeArray:boolean): object;

/**
*  Sort a list of objects by a given property of each item.
*  
*  The optional normalizer, if passed as a function, is called for each item in the List, and
*  should return whatever value should be used for sorting, which does not have to agree with
*  the property value. By passing a normalizer function you can achieve any kind of sorting
*  you'd like, including sorting by multiple properties.
*  
*  NOTE: string sort is case INsensitive by default
* 
* 
* @param {string} name of the property to sort by
* @param {boolean} true == sort ascending, false == sort descending
* @param {Function | ValueMap=} May be specified as a function, with signature        normalize(item, propertyName, context), where item is        a pointer to the item in the array, propertyName is the        property by which the array is being sorted, and context is the        arbitrary context passed into this method. Normalizer function should return        the value normalized for sorting.        May also be specified as a ValueMap which maps property values to sortable values.
* @param {any=} Callers may pass an arbitrary context into the sort method, which             will then be made available to the normalizer function
*/
sortByProperty(property:string, up:boolean, normalizer?:Function | isc.ValueMap, context?:any): isc.List;

/**
*  Sort this Array by a list of SortSpecifiers. 
* 
* @param {Array<Partial<SortSpecifier>>} the list of SortSpecifiers to sort by
*/
setSort(sortSpecifiers:Array<Partial<isc.SortSpecifier>>): Array<any>;

/**
*  Turn sorting off for this array, indicating that the current sort
*  order should be preserved. Return true if this is supported in this List.
* 
*  Some implementations may not support this -- they should return false
*  to indicate to the caller that sort order must be maintained (eg: in
*  the case where sort order is derived from the server, etc).
* 
* 
*/
unsort(): boolean;

/**
* 
*   Returns the largest number in the array, skipping non-numeric values.
* 
*  	If the start and/or end are given, searches the specified subset of the list.
* 
* 
* @param {number=} optional start index (default is 0)
* @param {number=} optional end index (default is list.length)
*/
max(start?:number, end?:number): number;

/**
*      
*   Returns the smallest number in the array, skipping non-numeric values.
* 
*  	If the start and/or end are given, searches the specified subset of the list.
* 
* 
* @param {number=} optional start index (default is 0)
* @param {number=} optional end index (default is list.length)
*/
min(start?:number, end?:number): number;

/**
*   Returns the sum of the numbers in the array, skipping non-numeric values.
* 
*  	If the start and/or end are given, uses only the specified subset of the list.
* 
* 
* @param {number=} optional start index (default is 0)
* @param {number=} optional end index (default is list.length)
*/
sum(start?:number, end?:number): number;

/**
*  Returns true if all values between the start and end indices are true.
* 
* 
* @param {number=} optional start index (default is 0)
* @param {number=} optional end index (default is list.length)
*/
and(start?:number, end?:number): boolean;

/**
*  Returns true if at least one value between the start and end indices is true.
* 
* 
* @param {number=} optional start index (default is 0)
* @param {number=} optional end index (default is list.length)
*/
or(start?:number, end?:number): boolean;




}
interface ArrayConstructor {




/**
*  Is the object passed in a loading marker value? For use with Lists that manage remote
*  datasets, to indicate that a record has not yet been retrieved from the server. A typical
*  use case might be to check if a row has been loaded in a ListGrid - for example:
*  
*  
*  if (Array.isLoading(myList.getRecord(0))) isc.warn("Please wait for the data to load.");
*  
* 
* @param {any} data to test.
*/
isLoading(value:any): void;

/**
*  Compare two values for an ascending order sort, using locale-sensitive comparison.
* 
* @param {any} first value to compare
* @param {any} second value to compare
*/
compareAscending(a:any, b:any): number;

/**
*  Compare two values for a descending order sort, using locale-sensitive comparison.
* 
* @param {any} first value to compare
* @param {any} second value to compare
*/
compareDescending(first:any, second:any): number;




}/**
* Boolean object. Attributes, parameters, or return values declared as Boolean
*  may be null. Contrast with boolean.
*/
interface Boolean {





}
interface BooleanConstructor {







}/**
* Extra methods added to the Number object, available on all number variables. Attributes,
*  parameters, or return values declared as Number may be null.
*/
interface Number {


/**
* 
*  Return this number as a string padded out to digits length.
* 
* 
* @param {number=} [2] Number of digits to pad to. (Default is 2)
* @deprecated * Moved to a static method on NumberUtil to avoid the possibility of collision        with other libraries on the native Number object

*/
stringify(digits?:number): string;

/**
*  Returns true if the number parameter falls between the 'first' and 'second' paramters.
* 
* 
* @param {number} Number object to be evaluated
* @param {number=} Number at the lower boundary
* @param {number=} Number at the upper boundary
* @param {number=} Whether or not the numbers at either end of the boundary should be included in the comparison
*/
isBetween(number:number, first?:number, second?:number, inclusive?:number): boolean;

/**
*  Return this number as a currency-formatted string.
* 
* 
* @param {string=} Currency symbol, can be set to an empty string.                 If unset "$" will be used.
* @param {string=} Decimal separator symbol. If unset "." will be used.
* @param {boolean=} Should decimal portion be padded out to two digits? True                by default.
* @param {boolean=} Should currency symbol come at the end of the string?                   If unspecified, currency symbol will be shown at the                    beginning of the string.
* @deprecated * Moved to a static method on NumberUtil to avoid the possibility of collision        with other libraries on the native Number object

*/
toCurrencyString(currencyChar?:string, decimalChar?:string, padDecimal?:boolean, currencyCharLast?:boolean): string;




}
interface NumberConstructor {







}/**
* Generic extensions to JavaScript Strings. You can call these on any String.
*/
interface String {


/**
*  	Returns true if this string contains the specified substring.
* 
* @param {string} string to look for
*/
contains(substring:string): boolean;

/**
*  Returns true if this string starts with another string, or if the other string
*  occurs at the given position within this string.
* 
* 
* @param {string} other string to check
* @param {Integer=} optional position in this string. Defaults to 0.
*/
startsWith(substring:string, position?:isc.Integer): boolean;

/**
*  Returns true if this string ends with another string, or if the other string
*  occurs in this string beginning at position - substring.length.
* 
* 
* @param {string} other string to check
* @param {Integer=} optional position in this string. Defaults to the length of this string.
*/
endsWith(substring:string, position?:isc.Integer): boolean;




}
interface StringConstructor {


/**
*  Tests whether the given string is a valid JavaScript identifier.
* 
* 
* @param {string} the string to test.
*/
isValidID(string:string): boolean;






}
declare namespace isc {
interface Map<T> { [key: string]: T }
/**
* 
* 	Method available on the isc object to show a modal prompt to the user.
*  This method will display the message using the Dialog.Prompt singleton object.
*  Note: if this prompt is to be shown to the user during some slow JavaScript logic, we
*  advise calling this method, then using Class.delayCall or Timer.setTimeout
*  to kick off the slow logic in a separate thread. This ensures that the prompt is showing
*  before the lengthy execution begins.
*  Use "&#36;{loadingImage}" to include Canvas.loadingImageSrc.
* 
* 
* 
* @param {string} message to display
* @param {Partial<Dialog>=} additional properties for the Dialog, applied before                    the Dialog is shown
*/
function showPrompt(message:string, properties?:Partial<Dialog>):void;
/**
* 
* 	Clear the modal prompt being shown to the user.
* 
* 
*/
function clearPrompt():void;
/**
* 
* 	Method available on the isc object to show a temporary modal prompt to the user.
*  This method will display the message using the Dialog.Prompt singleton object, then hide it
*  using a fade animation effect.
*  Note: if this prompt is to be shown to the user during some slow JavaScript logic, we
*  advise calling this method, then using Class.delayCall or Timer.setTimeout
*  to kick off the slow logic in a separate thread. This ensures that the prompt is showing
*  before the lengthy execution begins.
*  
*  The prompt may be cleared before the duration has elapsed via a call to isc.clearPrompt
*  and any callback specified will still be fired even if the prompt is dismissed early.
* 
* 
* @param {string} message to display
* @param {number} how long the message should appear for in milliseconds before  fading from view.
* @param {Callback=} When the prompt is hidden, callback will be fired.
* @param {Partial<Dialog>=} additional properties for the Dialog, applied before                    the Dialog is shown
*/
function showFadingPrompt(message:string, duration:number, callback?:Callback, properties?:Partial<Dialog>):void;
/**
*  Show a modal dialog with a message, icon, and "OK" button. See Dialog.warnIcon.
*  
*  The callback will receive boolean true for an OK button click, or null if the Dialog is
*  dismissed via the close button.
* 
* 
* @param {string} message to display
* @param {Callback=} Optional Callback to fire when the user                   dismisses the dialog. This has the single parameter                    'value', indicating the value returned by the Warn                    dialog from 'okClick()' etc.
* @param {Partial<Dialog>=} additional properties for the Dialog.                   To set Dialog.buttons for                    the Dialog, set properties.buttons to                    an array of buttons     	eg:	{ buttons : [Dialog.OK, Dialog.CANCEL] }
*/
function warn(message:string, callback?:Callback, properties?:Partial<Dialog>):void;
/**
*  Show a modal dialog with a message, icon, and "OK" button. Intended for notifications which
*  are not really warnings (default icon is less severe). See Dialog.sayIcon.
*  
*  The callback will receive boolean true for an OK button click, or null if the Dialog is
*  dismissed via the close button.
* 
* 
* @param {string} message to display
* @param {Callback=} Optional Callback to fire when the user                   dismisses the dialog. This has the single parameter                    'value', indicating the value returned by the Warn                    dialog from 'okClick()' etc.
* @param {Partial<Dialog>=} additional properties for the Dialog.                   To set Dialog.buttons for                    the Dialog, set properties.buttons to an array of                    buttons     	eg:	{ buttons : [Dialog.OK, Dialog.CANCEL] }
*/
function say(message:string, callback?:Callback, properties?:Partial<Dialog>):void;
/**
*  Show a modal dialog with a message, icon, and "Yes" and "No" buttons. See Dialog.askIcon.
*  
*  The callback will receive boolean true for a Yes button click, boolean false for a No button
*  click, or null if the Dialog is dismissed via the close button.
* 
* 
* @param {string} message to display
* @param {Callback=} Callback to fire when the                   user clicks a button to dismiss the dialog.                    This has the single parameter 'value', indicating the                    value returned by the Warn dialog from 'okClick()' etc.
* @param {Partial<Dialog>=} additional properties for the Dialog.                   To set Dialog.buttons for                    the Dialog, set properties.buttons to an array                    of buttons     	eg:	{ buttons : [Dialog.OK, Dialog.CANCEL] }
*/
function ask(message:string, callback?:Callback, properties?:Partial<Dialog>):void;
/**
*  Show a modal dialog with a message, icon, and "OK" and "Cancel" buttons. See Dialog.confirmIcon.
*  
*  The callback will receive boolean true for an OK button click, or null for a Cancel click or
*  if the Dialog is dismissed via the close button.
*  
*  Note: this does not override the native window.confirm() method.
* 
* 
* @param {string} message to display
* @param {Callback=} Callback to fire when the                   user clicks a button to dismiss the dialog.                    This has the single parameter 'value', indicating the                    value returned by the Warn dialog from 'okClick()' etc.
* @param {Partial<Dialog>=} additional properties for the Dialog.                   To set Dialog.buttons for                    the Dialog, set properties.buttons to an array of                    buttons     	eg:	{ buttons : [Dialog.OK, Dialog.CANCEL] }
*/
function confirm(message:string, callback?:Callback, properties?:Partial<Dialog>):void;
/**
*  Show a modal dialog with a text entry box, asking the user to enter a value.
*  
*  As with other convenience methods that show Dialogs, such as isc.warn,
*  the dialog is shown and the function immediately returns. When the user responds, the
*  provided callback is called.
*  
*  If the user clicks OK, the value typed in is passed to the callback (including the empty
*  string ("") if nothing was entered. If the user clicks cancel, the value passed to the
*  callback is null.
*  
*  A default value for the text field can be passed via properties.defaultValue.
*  
*  Keyboard focus is automatically placed in the text entry field, and hitting the enter key is
*  the equivalent of pressing OK.
* 
* 
* @param {string} message to display
* @param {Callback=} Callback to fire when the                   user clicks a button to dismiss the dialog.                    This has the single parameter 'value', indicating the                    user entry, or null if cancel was pressed or the window                    closed
* @param {Partial<Dialog>=} additional properties for the Dialog.                   To set Dialog.buttons for                    the Dialog, set properties.buttons to an array of                    buttons     	eg:	{ buttons : [Dialog.OK, Dialog.CANCEL] }
*/
function askForValue(message:string, callback?:Callback, properties?:Partial<Dialog>):void;
/**
*  If a dialog triggered via isc.say, isc.ask,
*  isc.warn, isc.confirm or
*  isc.askForValue is currently visible, it will be dismissed. The
*  callback passed to the relevant method will never fire.
*  
*  Note this is a rarely used API with very few valid use cases. As an example, perhaps some kind of
*  periodic (non-user triggered) event would cause an entire area of the UI to be removed (such as a tab)
*  and the system wants to ensure that no modal dialogs are currently showing from that part of the UI.
*  In this case, while dismissCurrentDialog could be used to ensure the part of the UI being
*  removed didn't leave behind a modal dialog.
*  
*  To clear a modal prompt shown by isc.showPrompt, use isc.clearPrompt instead.
* 
* 
*/
function dismissCurrentDialog():void;
/**
*  Handle a complete login interaction with a typical login dialog asking for username and
*  password credentials using the LoginDialog class.
*  
*  As with other convenience methods that show Dialogs, such as isc.warn,
*  the dialog is shown and the function immediately returns. When the user responds, the
*  provided callback function is called.
*  
*  If the user clicks the "Log in" button, the credentials entered by the user are passed to
*  the provided "loginFunc" as an Object with properties "username" and "password" (NOTE: both
*  property names are all lowercase), as the variable "credentials". For example:
*  { username: "barney", password: "rUbbL3" }
*  
*  The "loginFunc" should then attempt to log in by whatever means is necessary. The second
*  parameter to the loginFunc, "dialogCallback", is a function, which must be called whether
*  login succeeds or fails with a true/false value indicating whether login succeeded.
*  
*  If the login dialog is dismissable (settable as properties.dismissable, default false) and
*  the user dismisses it, the loginFunc will be fired with null for the credentials.
*  
*  The following code shows typical usage. This code assumes you have created a global
*  function sendCredentials() that send credentials to some authentication system and fires a
*  callback function with the result:
*  
*  isc.showLoginDialog(function (credentials, dialogCallback) {
*    if (credentials == null) return; // dismissed
* 
*    // send credentials
*    sendCredentials(credentials, function (loginSucceeded) {
*      // report success or failure
*      dialogCallback(loginSucceeded);
*    })
*  })
*  
*  The login dialog has several built-in behaviors:
*  
*  
* - keyboard focus is automatically placed in the username field
*  
* - hitting enter in the username field proceeds to the password field
*  
* - hitting enter in the password field submits (fires the provided callback)
*  
*  In addition to normal properties supported by Dialog/Window, the following special
*  properties can be passed:
*  
*  
* - username: initial value for the username field
*  
* - password: initial value for the password field
*  
* - usernameTitle: title for the username field
*  
* - passwordTitle: title for the password field
*  
* - errorMessage: default error message on login failure
*  
* - loginButtonTitle: title for the login button
*  
* - dismissable: whether the dialog can be dismissed, default false
*  
* - errorStyle: CSS style for the error message, if shown
*  
*  See below for links to the default values for these properties.
* 
* 
* @param {Callback} Function to call to attempt login. Receives parameters                   "credentials" and "dialogCallback", described above
* @param {Partial<LoginDialog>=} additional properties for the Dialog
*/
function showLoginDialog(loginFunc:Callback, properties?:Partial<LoginDialog>):void;
/**
*  Same as Log.logWarn.
* 
* 
* @param {string} message to log
* @param {string=} category to log in, defaults to "Log"
*/
function logWarn(message:string, category?:string):void;
/**
*  Same as Log.echo.
* 
* 
* @param {any} object to echo
*/
function echo(value:any):string;
/**
*  Same as Log.echoAll.
* 
* 
* @param {any} object to echo
*/
function echoAll(value:any):string;
/**
*  Same as Log.echoLeaf.
* 
* 
* @param {any} object to echo
*/
function echoLeaf(value:any):string;
/**
*  Logs the echoed object (using isc.echo) as a warning, prefixed with an
*  optional message.
* 
* 
* @param {any} object to echo
* @param {string} message to log
*/
function logEcho(value:any, message:string):void;
/**
*  Logs the echoed object (using isc.echoAll) as a warning, prefixed with an
*  optional message.
* 
* 
* @param {any} object to echo
* @param {string} message to log
*/
function logEchoAll(value:any, message:string):void;
/**
* 
*  Add all properties and methods from any number of objects to a destination object, 
*  overwriting properties in the destination object.
*  
*  Common uses of addProperties include creating a shallow copy of an object:
* 
*    isc.addProperties({}, someObject);
* 
*  Combining settings in order of precedence:
* 
*    isc.addProperties({}, defaults, overrides, skinOverrides);
* 
*  
*  
*  NOTES:
*  
* - Do not use addProperties to add defaults to an ISC class.
*  Use Class.addProperties, as in: 
*  MyClassName.addProperties().
*  
* - You may find it more convenient to use the instance method Class.addProperties,
*  as in: myClassInstance.addProperties(), but there's no functional
*  difference from using this method.
*  
* 
* 
* @param {object} object to add properties to
* @param {object=} objects to obtain properties from. Properties of all      arguments other than destination are applied in turn.
*/
function addProperties(destination:object, ...args:any[]):object;
/**
* 
* 	Return all keys (property names) of a given object
* 
* 
* @param {object} object to get properties from
*/
function getKeys(object:object):Array<any>;
/**
*  Return the first property name in a given Object, according to for..in iteration order.
* 
* 
* @param {object} Object to get properties from
*/
function firstKey(object:object):string;
/**
* 
* 	Return all values of a given object
* 
* 
* @param {object} object to get properties from
*/
function getValues(object:object):Array<any>;
/**
*  Given a simple javascript object, return that object sorted by keys, such that when iterating
*  through the properties of the object, they will show up in sorted order.
*  Usage example - may be used to sort a FormItem.valueMap defined
*  as an object.
* 
* @param {object} Object to sort
* @param {Function=} Comparator function to use when sorting the objects keys
*/
function sortObject(object:object, comparator?:Function):object;
/**
*  Given a simple javascript object, return that object sorted by properties, such that when 
*  iterating through the properties of the object, values will show up in sorted order.
*  Usage example - may be used to sort a FormItem.valueMap defined
*  as an object by display value.
* 
* @param {object} Object to sort
* @param {Function=} Comparator function to use when sorting the object properties
*/
function sortObjectByProperties(object:object, comparator?:Function):object;
/**
* 
*  Copy any properties that do not already have a value in destination. Null and zero values
*  are not overwritten, but 'undef' values will be.
* 
* 
* @param {object} Object to which properties will be added.
* @param {object} Object from which properties will be added.
*/
function addDefaults(destination:object, source:object):object;
/**
* 
* 	Is some property specified on the object passed in? This will return true if 
*  object[propertyName] has ever been set to any value, and not deleted.
*  May return true even if object[propertyName] === undefined if the property 
*  is present on the object and has been explicitly set to undefined.
* 
* 
* @param {object} Object to test
* @param {string} Which property is being tested for?
*/
function propertyDefined(object:object, propertyName:string):boolean;
/**
*  Shortcut for isc.ClassFactory.defineClass().
* 
* @param {string} Name for the new class.
* @param {Class=} Optional SuperClass Class object or name
*/
function defineClass(className:string, superClass?:Class):Class;
/**
*  Shortcut for isc.ClassFactory.overwriteClass().
* 
*/
function overwriteClass():void;
/**
*  Shorthand for new Date().getTime();, this returns a timeStamp - a large number
*  which is incremented by 1 every millisecond. Can be used to generate unique identifiers,
*  or perform timing tasks.
* 
* 
*/
function timeStamp():Integer;
/**
* 
* 	Method available on the isc object to open the Developer Console.
* 
* 
*/
function showConsole():void;
/**
*  Given a key and an object of key:value pairs, return the value that corresponds to
*  that key.
*  
*  If the key is not found, defaultValue will be returned if provided, otherwise the
*  key will be returned.
* 
* 
* @param {string | number} key to look for
* @param {object} object of key:value pairs
* @param {any=} default value to return if key not found
*/
function getValueForKey(key:string | number, valueMap:object, defaultValue?:any):any;
/**
*  Given a value and an object of key:value pairs, return a key that corresponds
*  to that value.
*  
*  If the key is not found, defaultKey will be returned if provided, otherwise the
*  value will be returned.
* 
* 
* @param {string | number} value to look for
* @param {object} object of key:value pairs
* @param {any=} default key to return if value not found
*/
function getKeyForValue(value:string | number, valueMap:object, defaultKey?:any):any;
/**
*  Given a key:value map, return a new map as value:key.
*  
*  If the same value appears more than once, the key will correspond to the last instance of that
*  value.
* 
* 
* @param {object} object of key:value pairs
*/
function makeReverseMap(valueMap:object):object;
/**
*  Method to set up a static Class.observe on some target object. 
*  This allows developers to perform some action every time a particular method is invoked 
*  on a target object.
*  
*  This method returns a unique observation identifier string. To cancel the observation, 
*  pass this identifier to Page.ignore.
*  
*  If multiple observations are set up for the same target object and method, the 
*  notification actions will be fired in the order in which they were registered.
*  
*  This method is available as isc.Page.observe() or just isc.observe()
*  
*  Note [potential memory leak]: If the target object is a simple JavaScript object
*  (not an instance of a SmartClient class), developers should always call 
*  Page.ignore to stop observing the object if an observation is no longer necessary.
*  
*  This ensures that if the object is subsequently allowed to go out of scope by application 
*  code, the Page level observation system will not retain a reference to it (so the browser
*  can reclaim the allocated memory).
*  While cleaning up observations that are no longer required is always good practice, this
*  memory leak concern is not an issue if the target object is an instance of a SmartClient
*  class. In that case the observation is automatically released when the 
*  target is Class.destroy.
* 
* 
* @param {object} Object to observe. This may be any JavaScript object with the specified  target method, including native arrays, and instances of SmartClient classes such as   Canvas.
* @param {string} Name of the method to observe. Every time this method is invoked  on the target object the specified action will fire   (after the default implementation completes).
* @param {Function | string} Action to take when the observed method is invoked.  If action is a string to execute, certain keywords are available for context:    
- observed is the target object being observed (on which the method was invoked).  
- returnVal is the return value from the observed method (if there is one)  
- For functions defined with explicit parameters, these will also be available as keywords within    the action string    If action is a function, the arguments for the original method will also be   passed to this action function as arguments. If developers need to access the target   object being observed from the action function they may use native javascript techniques  such as   javascript closure  to do so. The return value from the observed method is not available to the action function.
*/
function observe(object:object, methodName:string, action:Function | string):string;
/**
*  Clear an observation set up by Page.observe.
*  
*  This method is available as isc.Page.ignore() or just isc.ignore()
* 
* 
* @param {string} ID returned from Page.observe call we want to clear
*/
function ignore(observerID:string):void;
/**
*  Set the global default setting for Canvas.autoDraw.
*  
*  After calling isc.setAutoDraw(), any newly created Canvas which is not given an
*  explicit setting for Canvas.autoDraw will follow the new default setting.
*  
*  autoDraw:false is the recommended default setting for most applications since it ensures
*  that extra draws will not occur when developers inadvertently omit the autoDraw:false
*  setting on child components.
* 
* 
* @param {boolean=} whether autoDraw should be enabled or disabled. Defaults to true.
*/
function setAutoDraw(enable?:boolean):void;
/**
*  Create a deep clone of an object that can be edited without affecting the original
*  
*  All mutable types, including Objects, Arrays and Dates, are copied. All immutable types
*  (Number, String, etc) are just preserved by reference.
*  
*  Only JavaScript built-in types may be cloned. SmartClient UI widgets do not support
*  cloning but must be created explicitly via Class.create.
*  Note that you also can't duplicate a live canvas by passing into create() as an
*  argument. If you need to create multiple components with similar configuration, some common
*  patterns inclulde:
*  
* - Create a new SmartClient class with the desired default configuration, and
*    create instances of this class as needed.
*  
* - For components created by some specific instance, the AutoChild system may be used.
*    Developers can specify a standard configuration in 
*    autoChildNameDefaults and 
*    autoChildNameProperties, and use Class.createAutoChild
*    to create a number of standard auto child components.
*  
* - A less formal approach might be to have a simple getter type method which
*    created and returned a new component each time it was called, passing in a standard
*    configuration block.
*  
*  Does not handle looping references (will infinite loop).
* 
* 
* @param {object} object to clone
*/
function clone(object:object):object;
/**
*  Creates a shallow copy of the passed-in Object or Array of Objects, that is, copies all
*  properties of an Object to a new Object, so that the clone now has exactly the same property
*  values as the original Object.
*  
*  If shallowClone() is passed an immutable type such as String and Number, it is returned
*  unchanged. Dates are copied via new Date(originalDate.getTime()).
*  
*  Note that if an Array is passed, all members of the Array will be cloned. For a copy of an
*  Array that contains exactly the same members (not copies), use Array.duplicate().
*  
*  Only an Array directly passed to shallowClone() is copied. Arrays contained
*  within Arrays will not be copied.
* 
* 
* @param {object | Array<any> | object} object to be cloned
*/
function shallowClone(object:object | Array<any> | object):object | Array<Partial<object>>;
/**
*  Enables full screen reader mode. Must be called before any components are created. See
*  accessibility.
* 
* @param {boolean} new setting
*/
function setScreenReaderMode(newState:boolean):void;
/**
*  Displays a new message that's automatically dismissed after a configurable amount of time,
*  as an alternative to isc.confirm dialogs that can lower end user
*  productivity.
*  
*  This method is simply a shorthand way to call Notify.addMessage. For further study,
*  see the Notify class overview, and the class methods Notify.dismissMessage and Notify.configureMessages.
* 
* @param {HTMLString} message to be displayed
* @param {Array<Partial<NotifyAction>>=} actions (if any) for this message
* @param {NotifyType=} category of message
* @param {NotifySettings=} display and behavior settings for this message,                        overriding any stored settings for this                        NotifyType
*/
function notify(contents:HTMLString, actions?:Array<Partial<NotifyAction>>, notifyType?:NotifyType, settings?:NotifySettings):MessageID;
/**
*  Returns the (offset) left-coordinate of an icon within its containing widget.
* 
* @param {object} icon definition
*/
function getIconLeft(icon:object):number;
/**
*  Returns the (offset) top-coordinate of an icon within its containing widget.
* 
* @param {object} icon definition
*/
function getIconTop(icon:object):number;
/**
*  Returns the size / position of an icon with respect to the widget rendering out the
*  form item as an array of coordinates.
* 
* @param {object} icon definition for the icon you want to determine the                position of (defaults to first icon in this.icons).
*/
function getIconRect(icon:object):Array<any>;
/**
*  Returns the size / position of an icon on the page as an array of coordinates.
* 
* @param {object} icon definition for the icon you want to determine the                position of (defaults to first icon in this.icons).
*/
function getIconPageRect(icon:object):Array<any>;
/**
*  Sets the StatefulCanvas.radioGroup identifier for this canvas's 
*  mutually exclusive selection group.
* 
*/
function setRadioGroup():void;
/**
*  A notification Callback fired by the TabIndexManager to allow 
*  application code to react to the numeric tab-index of some registered target being
*  modified.
* 
*/
type TabIndexUpdatedCallback = object;
/**
*  A Callback fired by the TabIndexManager when application code or user
*  action attempts to synthetically shift focus to some registered target. 
*  See TabIndexManager.shiftFocus.
*  
*  A typical implementation will shift focus to some native element associated
*  with the registered target, or if this is not currently possible, return false.
* 
* 
*/
type ShiftFocusCallback = object;
/**
*  Callback executed when a PrintCanvas.setHTML is rendered into
*  a printCanvas.
* 
* 
*/
type PrintCanvasCallback = object;
/**
*  Callback required for the property DataSourceField.getFieldValue.
* 
* 
*/
type GetFieldValueCallback = object;
/**
*  A Callback fired when DataSource.hasFile completes.
* 
* 
*/
type HasFileCallback = object;
/**
*  A Callback fired when Sound.load completes.
*  
* 
*/
type CanPlayCallback = object;
/**
*  A Callback fired when Sound.play completes.
*  
* 
*/
type PlaybackCompleteCallback = object;
/**
*  A Callback fired when DataSource.hasFileVersion completes.
* 
* 
*/
type HasFileVersionCallback = object;
/**
*  Callback fired when DataSource.getFile completes.
* 
* 
*/
type GetFileCallback = object;
/**
*  Callback fired when DataSource.getFileVersion completes.
* 
* 
*/
type GetFileVersionCallback = object;
/**
*  Generic callback interface.
* 
* 
*/
type Function = object;
/**
*  Generic callback interface.
* 
* 
*/
type ClientOnlyDataSourceCallback = object;
/**
*  A Callback to evaluate when a screen is loaded via RPCManager.loadScreen.
* 
* 
*/
type LoadScreenCallback = object;
/**
*  A Callback to evaluate when an RPCRequest completes.
*  
*  Parameters passed to this callback are:
*  
*  
* - rpcResponse: an RPCResponse encapsulating the server response to your
*    request
*  
* - data: just the "data" property from the RPCResponse, for convenience
*  
* - rpcRequest: the RPCRequest that was sent. You can use
*    RPCRequest.clientContext to track state during the server turnaround.
*  
*  For example, to take the data returned by the server and display it in a previously created
*  ListGrid with the ID "myGrid":
*  
*    isc.RPCManager.send("getData", "myGrid.setData(data)");
*  
*  Or
*  
*    isc.RPCManager.send("getData", function (rpcResponse, data, rpcRequest) { 
*                     myGrid.setData(data)
*    });
*  
* 
* 
*/
type RPCCallback = object;
/**
*  Callback to fire when a queue of requests sent via 
*  {@link com.smartgwt.client.rpc.RPCManager#sendQueue(RPCQueueCallback)} returns.
*  Note that the Array of RPCResponses passed to this callback
*  will actually be DSResponse objects for any requests that were actually 
*  DSRequests. 
*  
* 
* 
*/
type RPCQueueCallback = object;
/**
*  Callback for RPCManager.exportImage.
* 
*/
type ExportImageCallback = object;
/**
*  Callback fired with the PaletteNode obtained asynchronously.
* 
*/
type PaletteNodeCallback = object;
/**
*  Callback for DrawPane.getDataURL.
* 
*/
type DataURLCallback = object;
/**
*  A Callback to evaluate when form validation completes.
*  
*  The available parameters are:
* 
*/
type ValidationStatusCallback = object;
/**
*  Callback required for the property Validator.condition and
*  ValidatorDefinition.condition.
* 
*/
type ValidatorConditionCallback = object;
/**
*  Callback required for the property ValidatorDefinition.action.
* 
* 
*/
type ValidatorActionCallback = object;
/**
*  Callback for DateRangeDialog.askForRange.
* 
*/
type DateRangeCallback = object;
/**
*  A Callback called when the move completes.
* 
* 
*/
type AnimationCallback = object;
/**
*  Callback to execute after the section has been shown.
* 
*/
type ShowSectionCallback = object;
/**
*  Callback to execute after the section has been expanded.
* 
*/
type ExpandSectionCallback = object;
/**
*  Callback to execute after the section has been hidden.
* 
*/
type HideSectionCallback = object;
/**
*  Callback to execute after the section has been collapsed.
* 
*/
type CollapseSectionCallback = object;
/**
*  Callback executed when a message is sent to a channel that you have
*  Messaging.subscribe to.
* 
*/
type MessagingCallback = object;
/**
*  A Callback to evaluate when an {Process.loadProcess} method completes.
*  
*  Loaded process passed as a parameter to this callback are:
* 
* 
*/
type ProcessCallback = object;
/**
* This class is a synonym for Canvas that can be used to make intent clearer.
*  It is used by some development tools for that purpose.
*/
interface AbsoluteContainer extends Canvas {




}
/**
* Creates a new AbsoluteContainer
* 
* @param typeCheckedProperties {Partial<AbsoluteContainer>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {AbsoluteContainer} The new instance
*/
class AbsoluteContainer{
static create(typeCheckedProperties? : Partial<AbsoluteContainer>, uncheckedProperties?: Object): AbsoluteContainer;







}

/**
* EditProxy that handles Canvas objects when editMode is enabled.
*/
interface AbsoluteContainerEditProxy extends EditProxy {




}
/**
* Creates a new AbsoluteContainerEditProxy
* 
* @param typeCheckedProperties {Partial<AbsoluteContainerEditProxy>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {AbsoluteContainerEditProxy} The new instance
*/
class AbsoluteContainerEditProxy{
static create(typeCheckedProperties? : Partial<AbsoluteContainerEditProxy>, uncheckedProperties?: Object): AbsoluteContainerEditProxy;







}

/**
* This class is a DynamicForm with the default DynamicForm.itemLayout
*  property set to "absolute".
*/
interface AbsoluteForm extends DynamicForm {




}
/**
* Creates a new AbsoluteForm
* 
* @param typeCheckedProperties {Partial<AbsoluteForm>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {AbsoluteForm} The new instance
*/
class AbsoluteForm{
static create(typeCheckedProperties? : Partial<AbsoluteForm>, uncheckedProperties?: Object): AbsoluteForm;







}

/**
* ISC Abstraction for ActiveX controls
*/
interface ActiveXControl extends BrowserPlugin {
/**
* Sets the 'id' attribute on the object. If a name is not provided it will be
*  auto-generated. Note that in general you don't need to set this. If you have a reference to
*  your ISC ActiveX control object you can simply call
*  ActiveXControl.getPluginHandle to get a handle to the element.
* @type {string}
* @default null
*/
id?: string;

/**
* A map of key/value pairs to pass to the Active X control as parameters.
* @type {object}
* @default null
*/
params?: object;

/**
* Set this to the uuid of your Active X control - ISC will then generate the appropriate
*  classID entry for you.
* @type {string}
* @default null
*/
uuid?: string;

/**
* This sets the value of the classID property on the object. This is meant to give you
*  complete control over the generated HTML. In practice it may be more handy to set the uuid
*  property on this object and let the classID be generated from that.
* @type {string}
* @default null
*/
classID?: string;

/**
* Specifies the URL from which to load the ActiveX control.
* @type {URL}
* @default null
*/
codeBase?: URL;



/**
*  Returns the ID for this ISC ActiveX control object.
*  If the id property was specified for the object, that will be used, otherwise 
*  the ID will be auto-generated.
* 
*/
getPluginID(): string;

/**
*  Returns a handle to the element for this ISC ActiveX control object.
* 
*/
getPluginHandle(): DOMElement;



}
/**
* Creates a new ActiveXControl
* 
* @param typeCheckedProperties {Partial<ActiveXControl>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ActiveXControl} The new instance
*/
class ActiveXControl{
static create(typeCheckedProperties? : Partial<ActiveXControl>, uncheckedProperties?: Object): ActiveXControl;







}

/**
* A menu that can either show its menu options inline, or show them via a drop-down,
*  depending on available space in the surrounding Layout or ToolStrip.
*  
*  See Canvas.canAdaptWidth for background on adaptive layout.
*/
interface AdaptiveMenu extends Layout {
/**
* Whether the AdaptiveMenu should show some menu items inline as soon as there is enough space,
*  or should strictly switch between showing
* @type {boolean}
* @default true
*/
partialInlining?: boolean;

/**
* Placement of inlined items relative to the main MenuButton. Default is to place items
*  above the menu if the parent is a Layout with Layout.orientation,
*  otherwise to the left of the menuButton (or right if the Page.isRTL.
*  
*  A setting of "center" is invalid and will cause a warning and be ignored
* @type {Alignment | VerticalAlignment}
* @default null
*/
inlinePlacement?: Alignment | VerticalAlignment;

/**
* Default setting for MenuItem.showIconOnlyInline. Individual items can set
*  showIconOnlyInline to override this setting.
* @type {boolean}
* @default false
*/
showIconOnlyInline?: boolean;

/**
* MenuItems to be show either inline or as a drop-down Menu.
*  
*  When shown inline, items are rendered as different AutoChild according to the
*  settings on the MenuItem:
*  
*  
* - normal MenuItems render as the AdaptiveMenu.inlineMenuItem, a ToolStripButton AutoChild
*  
* - MenuItems that have submenus render as the AdaptiveMenu.inlineSubmenuItem, a
*    MenuButton AutoChild
*  
* - MenuItems with MenuItem.showIconOnlyInline set render as the
*    AdaptiveMenu.inlineImgButton, a ToolStripButton AutoChild
*  
* - MenuItems where MenuItem.embeddedComponent has been specified will have the
*    embedded component displayed directly instead (no AutoChild involvement here). If the
*    control should have different appearance when inlined vs embedded in the menu, one way
*    to achieve this is to detect whether the parent is a Menu when it is drawn.
*  
* @type {Array<Partial<MenuItem>>}
* @default null
*/
items?: Array<Partial<MenuItem>>;

/**
* Instance of the normal (non-Adaptive) Menu class used to show items that do not fit inline.
* @type {Menu}
* @default null
*/
menu?: Menu;

/**
* MenuButton used as a drop-down control for showing any items of the menu that are not
*  displayed inline.
* @type {MenuButton}
* @default null
*/
menuButton?: MenuButton;

/**
* Title used for the MenuButton.
* @type {HTMLString}
* @default null
*/
menuButtonTitle?: HTMLString;

/**
* Icon used for the MenuButton. Default of null means to use the default for the
*  MenuButton class.
* @type {SCImgURL}
* @default null
*/
menuButtonIcon?: SCImgURL;

/**
* MultiAutoChild used to create inline menu items.
*  
*  The MenuItem.icon and MenuItem.title will be rendered via Button.icon and
*  Button.title respectively; other MenuItem appearance-related properties do not
*  apply.
* @type {ToolStripButton}
* @default null
*/
inlineMenuItem?: ToolStripButton;

/**
* MultiAutoChild used to create inline menu items for menu items that have a submenu.
*  
*  The MenuItem.icon and MenuItem.title will be rendered via
*  IconButton.icon and Button.title respectively; other MenuItem
*  appearance-related properties do not apply.
* @type {IconMenuButton}
* @default null
*/
inlineSubmenuItem?: IconMenuButton;

/**
* Whether ToolStripSeparator should be shown for inline menu items. 
*  True by default for horizontal Layout.orientation, false for vertical.
* @type {boolean}
* @default null
*/
showInlineSeparators?: boolean;

/**
* ToolStripButton to display when AdaptiveMenu.showIconOnlyInline is set for one MenuItem
* @type {ImgButton}
* @default null
*/
inlineImgButton?: ImgButton;



/**
* 
* 
* @param {Array<Partial<MenuItem>> | MenuItem} array of menuItems to replace current menuItems
*/
setItems(items:Array<Partial<MenuItem>> | MenuItem): void;

/**
* 
* 
* @param {boolean} */
setPartialInlining(partialInlining:boolean): void;



}
/**
* Creates a new AdaptiveMenu
* 
* @param typeCheckedProperties {Partial<AdaptiveMenu>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {AdaptiveMenu} The new instance
*/
class AdaptiveMenu{
static create(typeCheckedProperties? : Partial<AdaptiveMenu>, uncheckedProperties?: Object): AdaptiveMenu;







}

/**
* A widget for editing a single, advanced HiliteRule for use by 
*  DataBoundComponent. Where a simple hilite provides 
*  configuration of a single criterion and either foreground or background color for 
*  application to a single field, an advanced hilite can specify more complex criteria which can 
*  both test and affect multiple fields and allow both background and foreground colors to 
*  be specified in a single rule. 
*  
*  Important Note: this class should not be used directly - it is exposed purely for
*  i18nMessages
*/
interface AdvancedHiliteEditor extends VStack {
/**
* AutoChild FilterBuilder for configuring the criteria for this Hilite.
*  
*  This component is an AutoChild and as such may be customized via 
*  advancedHiliteEditor.filterBuilderProperties.
* @type {FilterBuilder}
* @default null
*/
filterBuilder?: FilterBuilder;

/**
* The title for the Filter group.
* @type {string}
* @default "Filter"
*/
filterGroupTitle?: string;

/**
* Specifies a list of icons that can be used in hilites.
*  
*  hiliteIcons should be specified as an Array of SCImgURL.
*  When present, HiliteRule
*  will offer the user a drop down for picking one of these icons.
*  
*  If the user picks an icon, the created hiliting rule will have Hilite.icon set to 
*  the chosen icon. ListGridField.hiliteIconPosition controls where the icon will 
*  appear for that field -- the default is that it appears in front of the normal cell content.
* @type {Array<Partial<SCImgURL>>}
* @default null
*/
hiliteIcons?: Array<Partial<SCImgURL>>;

/**
* AutoChild DynamicForm for configuring the details of this Hilite.
*  
*  This component is an AutoChild and as such may be customized via 
*  advancedHiliteEditor.hiliteFormProperties.
* @type {DynamicForm}
* @default null
*/
hiliteForm?: DynamicForm;

/**
* The title for the Appearance group.
* @type {string}
* @default "Appearance"
*/
appearanceGroupTitle?: string;

/**
* The title for the Target Field(s) picker.
* @type {string}
* @default "Target Field(s)"
*/
targetFieldsItemTitle?: string;

/**
* AutoChild ImgButton that accepts this Hilite and fires the 
*  AdvancedHiliteEditor.callback.
*  
*  This component is an AutoChild and as such may be customized via 
*  advancedHiliteEditor.saveButtonProperties.
* @type {StatefulCanvas}
* @default null
*/
saveButton?: StatefulCanvas;

/**
* The title text for the AdvancedHiliteEditor.saveButton.
* @type {string}
* @default "Save"
*/
saveButtonTitle?: string;

/**
* AutoChild ImgButton that cancels this AdvancedHiliteEditor without saving 
*  any changes, firing the AdvancedHiliteEditor.callback with a null
*  parameter.
*  
*  This component is an AutoChild and as such may be customized via 
*  advancedHiliteEditor.cancelButtonProperties.
* @type {StatefulCanvas}
* @default null
*/
cancelButton?: StatefulCanvas;

/**
* The title text for the AdvancedHiliteEditor.cancelButton.
* @type {string}
* @default "Cancel"
*/
cancelButtonTitle?: string;

/**
* The message to show when the user clicks "Save" without entering any criteria.
* @type {string}
* @default "Enter at least one rule, a color or icon, and a target field, or press 'Cancel' to abandon changes."
*/
invalidHilitePrompt?: string;

/**
* The title text shown in the header bar of this editor's dialog.
* @type {string}
* @default "Advanced Highlight Editor"
*/
title?: string;

/**
* The callback to fire when the AdvancedHiliteEditor.saveButton is clicked.
* @type {Callback}
* @default null
*/
callback?: Callback;



/**
*  Save changes and fire the AdvancedHiliteEditor.callback.
* 
* 
*/
saveHilite(): void;

/**
*  Discard changes and fire the AdvancedHiliteEditor.callback with a null 
*  parameter.
* 
* 
*/
cancelEditing(): void;



}
/**
* Creates a new AdvancedHiliteEditor
* 
* @param typeCheckedProperties {Partial<AdvancedHiliteEditor>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {AdvancedHiliteEditor} The new instance
*/
class AdvancedHiliteEditor{
static create(typeCheckedProperties? : Partial<AdvancedHiliteEditor>, uncheckedProperties?: Object): AdvancedHiliteEditor;







}

/**
* Ask the user to input a value.
*/
interface AskForValueTask extends UserConfirmationGateway {
/**
* Default value.
* @type {string}
* @default null
*/
defaultValue?: string;





}
/**
* Creates a new AskForValueTask
* 
* @param typeCheckedProperties {Partial<AskForValueTask>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {AskForValueTask} The new instance
*/
class AskForValueTask{
static create(typeCheckedProperties? : Partial<AskForValueTask>, uncheckedProperties?: Object): AskForValueTask;







}

/**
* Synonym for the Authentication class
*/
interface Auth extends Authentication {




}
/**
* Creates a new Auth
* 
* @param typeCheckedProperties {Partial<Auth>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Auth} The new instance
*/
class Auth{
static create(typeCheckedProperties? : Partial<Auth>, uncheckedProperties?: Object): Auth;







}

/**
* The Authentication or Auth class represents a convenient, standard place to keep 
*  information about the currently logged in user and their assigned user roles.
*  
*  The intended usage is that a server authentication system would require the user to log in, 
*  then provide data about the currently logged in user via Authentication.setCurrentUser 
*  and Authentication.setRoles. This data is then available in the 
*  Canvas.ruleScope so that components can use it to enable or disable or
*  hide themselves, via properties such as FormItem.readOnlyWhen.
*  
*  The format for user records is not explicitly defined or restricted by the Authentication 
*  subsystem but we recommend using the format described by Authentication.getUserSchema.
*  Having a standardized user record allows application designers to rely on a 
*  well-known set of field names at design time, and then at deployment time when a 
*  particular authentication system is chosen, the deployer can simply fill in the 
*  standardized user record from the data that the chosen authentication system returns. 
*  This also allows authentication systems to be swapped out in the future without 
*  the need to change application code.
*  
*  The DataSource returned by Authentication.getUserSchema is used solely for visual 
*  tools to help with application authoring.
*  It is not intended to be used directly to store and retrieve user data, and while we
*  recommend this format it is not a requirement that user records conform to it.
*  
*  There are no security implications to calling setRoles() or
*  other APIs on the Authentication class. The provided data affects only 
*  client-side components. All actual security enforcement must be done server-side - 
*  see the QuickStart Guide, especially the sections on Declarative Security, 
*  to understand how role-based authorization can be used on the server.
*/
interface Authentication extends Class {




}
/**
* Creates a new Authentication
* 
* @param typeCheckedProperties {Partial<Authentication>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Authentication} The new instance
*/
class Authentication{
static create(typeCheckedProperties? : Partial<Authentication>, uncheckedProperties?: Object): Authentication;
/**
* URL to open for logging the current user out.
*  
*  This is a dynamic string - text within &#36;{...} are dynamic variables and will 
*  be evaluated as JS code when the message is displayed.
*  
*  The dynamic variables available are the fields in the Authentication.getCurrentUser record.
* @type {string}
* @default null
*/
static logOutURL?: string;

/**
* URL to open for reseting the current user's password.
*  
*  This is a dynamic string - text within &#36;{...} are dynamic variables and will 
*  be evaluated as JS code when the message is displayed.
*  
*  The dynamic variables available are the fields in the Authentication.getCurrentUser record.
* @type {string}
* @default null
*/
static resetPasswordURL?: string;





/**
*  Set up the current user. This method makes the user record available in the 
*  Canvas.ruleScope as "auth.currentUser".
* 
* 
* @param {Record} Record with attributes detailing the current user
*/
static setCurrentUser(user:Record): void;

/**
*  Returns the current user specified by Authentication.setCurrentUser. 
*  
*  This method returns the user record currently available in the 
*  Canvas.ruleScope as "auth.currentUser".
* 
* 
*/
static getCurrentUser(): Record;

/**
*  Returns a DataSource describing the standard schema for user data.
*  
*  The schema contains the following fields:
*  
*  Field NameType
*  "userId""text"
*  "email""text"
*  "firstName""text"
*  "lastName""text"
*  "title""text"
*  "phone""phoneNumber"
*  "superUser""boolean"
*  
* 
* 
*/
static getUserSchema(): DataSource;

/**
*  Set the user roles for the current user. Roles may be retrieved via Authentication.getRoles.
*  
*  Calling setRoles() makes the specified set of user roles available in the Canvas.ruleScope 
*  as a top-level property "userRoles", so that it can be used in criteria such as 
*  Canvas.visibleWhen or FormItem.readOnlyWhen. 
*  
*  Note that if this current user has been Authentication.setSuperUser,
*  Authentication.getRoles will return the full set of available roles rather than the set of
*  roles specified here.
*  
* 
* @param {Array<Partial<string>>} set of roles which apply to the current user
*/
static setRoles(roles:Array<Partial<string>>): void;

/**
*  Returns the current set of user roles. For Authentication.setSuperUser
*  this will be the intersection of any roles specified by 
*  Authentication.setRoles and the full set of Authentication.setAvailableRoles
*  - otherwise it will be the set of roles specified by Authentication.setRoles.
*  
*  Current set of user roles are available in the Canvas.ruleScope 
*  as a top-level property "userRoles", so that it can be used in criteria such as 
*  Canvas.visibleWhen or FormItem.readOnlyWhen. 
* 
* 
*/
static getRoles(): Array<Partial<string>>;

/**
*  Specify the full set of available user roles.
*  
*  Note that if the current user has been marked as a 
*  Authentication.isSuperUser, Authentication.getRoles will return
*  the full set of available roles.
* 
* 
* @param {Array<Partial<string>>} full set of possible user roles.
*/
static setAvailableRoles(roles:Array<Partial<string>>): void;

/**
*  Returns the full set of available user roles specified by Authentication.setAvailableRoles.
* 
*/
static getAvailableRoles(): Array<Partial<string>>;

/**
*  Mark the current user as a super-user. This causes Authentication.getRoles to return
*  the full set of Authentication.getAvailableRoles if specified
* 
* @param {boolean} New super user status
*/
static setSuperUser(isSuperUser:boolean): void;

/**
*  Has the current user been marked as a super-user via Authentication.setSuperUser?
* 
* @param {boolean} New super user status
*/
static isSuperUser(isSuperUser:boolean): void;




}

/**
* A button that automatically sizes to the length of its title. Implemented via the 
*  StatefulCanvas.autoFit property.
* @deprecated * As of Isomorphic SmartClient version 5.5, autoFit behavior can be achieved using the Button class instead by setting the property Button.autoFit to true.

*/
interface AutoFitButton extends Button {




}
/**
* Creates a new AutoFitButton
* 
* @param typeCheckedProperties {Partial<AutoFitButton>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {AutoFitButton} The new instance
*/
class AutoFitButton{
static create(typeCheckedProperties? : Partial<AutoFitButton>, uncheckedProperties?: Object): AutoFitButton;







}

/**
* Class for editable multi-line text areas (uses HTML &lt;TEXTAREA&gt; object)
*  automatically expands to accomodate its content
*/
interface AutoFitTextAreaItem extends TextAreaItem {
/**
* If specified, the autoFitTextArea will not grow taller than this height
* @type {Integer}
* @default null
*/
maxHeight?: Integer;





}
/**
* Creates a new AutoFitTextAreaItem
* 
* @param typeCheckedProperties {Partial<AutoFitTextAreaItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {AutoFitTextAreaItem} The new instance
*/
class AutoFitTextAreaItem{
static create(typeCheckedProperties? : Partial<AutoFitTextAreaItem>, uncheckedProperties?: Object): AutoFitTextAreaItem;







}

/**
* Standalone class providing a general interface for integration with Automated Testing Tools
*  
*  
*  For automated testing tools we need a way to create string identifiers for DOM elements such that 
*  when a page is reloaded, we can retrieve a functionally equivalent DOM element. We call these
*  AutoTestLocator.
*  
*  This allows automated testing tools to set up or record user generated events on DOM elements
*  then play them back on page reload and have our components react correctly.
*  
*  The primary APIs for the AutoTest subsystem are AutoTest.getLocator and 
*  AutoTest.getElement.
*  
*  Implementation considerations:
*  
*  
* - Some components react to the structure of DOM elements embedded within them - for example
*   GridRenderer cells have meaning to the grid. So in some cases we need to identify elements
*   within a component, while in others we can just return a pointer to a handle (A simple
*   canvas click handler doesn't care about what native DOM element within the handle received
*   the click).
* 
*  
* - When a DOM element is contained by a component, it is not sufficient to store the component
*   ID. Most SmartClient components are auto-generated by their parents, and rather than 
*   attempting to store a specific component identifier we should instead store the
*   "logical function" of the component.
*   For example a listGrid header button may have a different auto-generated ID across page
*   reloads due to various timing-related issues (which can change the order of of widget
*   creation), loading a new skin, or otherwise trivial changes to an application.
*   Rather than storing the header button ID therefore, we want to store this as
*   a string meaning "The header button representing field X within this list grid".
* 
*  
* - fallback strategies: In some cases a component or DOM element can be identified in 
*   several ways. For example a cell in a ListGrid can be identified by simple row and
*   column index, but also by fieldName and record primary key value. In these cases we
*   attempt to record information for multiple locator strategies and then when parsing
*   stored values we can provide APIs to govern which strategy is preferred. See the
*   LocatorStrategy documentation for more on this.
*  
*  
*  In order to address these concerns the AutoTest locator pattern is similar to an
*  XPath type structure, containing a root component identifier, followed by 
*  details of sub-components and then potentially details used to identify an element within
*  the components handle in the DOM.
*  
*  The actual implementation covers a large number of common cases, including (but not limited to)
*  the following. Note that for cases where an element is being identified from a pool of
*  possible candidates, such as the Canvas.children array, we usually will use
*  LocatorStrategy rather than simply relying on index:
*  
* - Root level components identified by explicit ID
*    
* - Any AutoChild
*    
* - Standard component parts such as scrollbars, edges, shadows, etc
*    
* - Section stack items and headers
*    
* - Window items
*    
* - ListGrid headers and cells
*    
* - TreeGrid headers and cells, including interactive open icon, checkbox icons
*    
* - DynamicForm form items, including details of elements within those items
*  
*  
*/
interface AutoTest extends Class {




}
/**
* Creates a new AutoTest
* 
* @param typeCheckedProperties {Partial<AutoTest>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {AutoTest} The new instance
*/
class AutoTest{
static create(typeCheckedProperties? : Partial<AutoTest>, uncheckedProperties?: Object): AutoTest;
/**
* Controls whether certain AutoTest APIs wait for network operations to complete
*  before returning true. When value is true, AutoTest.isElementClickable
*  will return false until all network operations have completed.
* @type {boolean}
* @default false
*/
static implicitNetworkWait?: boolean;

/**
* Sets the implicit root canvas available in scLocators starting "//testRoot[]".
*  Setting this property may enable one to use the same script to test identical
*  widget hierarchies that are rooted under different base widgets.
* @type {Canvas}
* @default null
*/
static testRoot?: Canvas;





/**
*  Returns the AutoTestLocator associated with some DOM element in a SmartClient
*  application page. If coords, representing the page position, is passed in, the locator
*  may be generated with a specific trailing "target area" identifer that will map back to
*  the appropriate, potentially different, physical coordinates, even if the widget is
*  moved. The coords argument will only have an effect in cases where the mouse position
*  over the target could potentially change behavior.
* 
* @param {DOMElement} DOM element within in the page. If null the locator for the last mouse event target will be generated
* @param {boolean=} If this parameter is passed in, check whether the target element responds to native browser events directly rather than going through  the SmartClient widget/event handling model. If we detect this case, return null rather  than a live locator. This allows us to differentiate between (for example) an event on  a Canvas handle, and an event occurring directly on a simple  &lt;a href=...&gt; tag written inside a Canvas handle.
* @param {Array<any>=} X, Y page position
*/
static getLocator(DOMElement:DOMElement, checkForNativeHandling?:boolean, coords?:Array<any>): AutoTestLocator;

/**
*  Returns the QualityIndicatedLocator associated with some DOM element in a
*  SmartClient application page. If coords, representing the page position, is passed in,
*  the locator
*  may be generated with a specific trailing "target area" identifer that will map back to
*  the appropriate, potentially different, physical coordinates, even if the widget is
*  moved. The coords argument will only have an effect in cases where the mouse position
*  over the target could potentially change behavior.
* 
* @param {DOMElement} DOM element within in the page. If null the locator for the last mouse event target will be generated
* @param {boolean=} If this parameter is passed in, check whether the target element responds to native browser events directly rather than going through  the SmartClient widget/event handling model. If we detect this case, return null rather  than a live locator. This allows us to differentiate between (for example) an event on  a Canvas handle, and an event occurring directly on a simple  &lt;a href=...&gt; tag written inside a Canvas handle.
* @param {Array<any>=} X, Y page position
*/
static getLocatorWithIndicators(DOMElement:DOMElement, checkForNativeHandling?:boolean, coords?:Array<any>): QualityIndicatedLocator;

/**
*  Given an element in the DOM, returns the canvas containing this element, or null if
*  the element is not contained in any canvas handle.
* 
* @param {DOMElement} DOM element within in the page
*/
static locateCanvasFromDOMElement(element:DOMElement): Canvas;

/**
* @param {AutoTestLocator} Locator String previously returned by    AutoTest.getLocator
*/
static getElement(locator:AutoTestLocator): DOMElement;

/**
*  Given an AutoTestLocator, return the live SmartClient object it refers to, if any.
* 
* @param {AutoTestLocator} Locator String previously returned by    AutoTest.getLocator
*/
static getObject(locator:AutoTestLocator): Canvas | FormItem | SectionStackSection;

/**
*  Given an AutoTestLocator that refers to a live SmartClient object or a logical
*  subcomponent of that object, return the associated meaningful JS value, if any.
*  
*  For example:
*  
*    
* - For a locator to a ListGrid/CubeGrid cell, return the cell's value
*    
* - For a locator to a FormItem, return the FormItem's value
*    
* - For a locator to a ListGrid field header, return the checkbox/sorting state
*    
* - For a locator to a Calendar EventCanvas header or body, return the text
*  
* 
* @param {AutoTestLocator} Locator String previously returned by    AutoTest.getLocator
*/
static getValue(locator:AutoTestLocator): object;

/**
*  Returns the page-level coordinates corresponding to the supplied locator. Note: The
*  physical position might change due to app redesign, but these coordinates would still
*  reflect the same logical part of the DOM element for components where event position
*  matters.
* 
* @param {AutoTestLocator} Locator String previously returned by    AutoTest.getLocator
*/
static getPageCoords(locator:AutoTestLocator): Array<any>;

/**
*  Sets the implicit root canvas available in scLocators starting "//testRoot[]".
*  Setting this property may enable one to use the same script to test identical
*  widget hierarchies that are rooted under different base widgets.
* 
* @param {Canvas} the implicit root
*/
static setTestRoot(canvas:Canvas): void;

/**
*  Returns whether the canvas associated with the given DOM element
*  is in a consistent state with no pending operations. Returns null
*  if the argument is not valid or isn't associated with an element
*  representing a valid canvas.
*  Otherwise, returns true or false according as the conditions below are all satisfied:
*  
*    
* - page has finished loading
*    
* - canvas is drawn
*    
* - canvas isn't dirty
*    
* - canvas has no queued overflow operations
*    
* - canvas is not animating
*  
* 
* @param {Canvas | DOMElement | AutoTestLocator}  DOM element to test (element obtained from canvas or SmartClient locator if provided) 
*/
static isCanvasDone(element:Canvas | DOMElement | AutoTestLocator): boolean;

/**
*  Returns whether the TileLayout associated with the given DOM element
*  is in a consistent state with no pending operations. Returns null if the
*  argument is not valid or isn't associated with an element representing
*  a valid TileLayout.
*  Otherwise, returns true or false according as the conditions below are all satisfied:
*  
*    
* - page has finished loading
*    
* - the TileLayout (as a canvas) satisfies AutoTest.isCanvasDone
*    
* - the TileLayout is not currently animating any layout operations
*  
* 
* @param {Canvas | DOMElement | AutoTestLocator}  DOM element to test (element obtained from canvas or SmartClient locator if provided) 
*/
static isTileLayoutDone(element:Canvas | DOMElement | AutoTestLocator): boolean;

/**
*  Returns whether the TileGrid associated with the given DOM element
*  is in a consistent state with no pending operations. Returns null if the
*  argument is not valid or isn't associated with an element representing
*  a valid TileGrid.
*  Otherwise, returns true or false according as the conditions below are all satisfied:
*  
*    
* - page has finished loading
*    
* - the TileGrid (as a tileLayout) satisfies AutoTest.isTileLayoutDone
*    
* - the TileGrid has no pending layout animation operations queued
*  
* 
* @param {Canvas | DOMElement | AutoTestLocator}  DOM element to test (element obtained from canvas or SmartClient locator if provided) 
*/
static isTileGridDone(element:Canvas | DOMElement | AutoTestLocator): boolean;

/**
*  Returns whether the DynamicForm associated with the given DOM element
*  is in a consistent state with no pending operations. Returns null
*  if the argument is not valid or isn't associated with an element
*  representing a valid
*  DynamicForm. Otherwise, returns true or false according as the conditions
*  below are all satisfied:
*  
*    
* - page has finished loading
*    
* - form has no pending delayed "set values" or "set values focus" operations
*    
* - all contained items satisfy AutoTest.isItemDone
*  
* 
* @param {Canvas | DOMElement | AutoTestLocator}  DOM element to test (element obtained from canvas or SmartClient locator if provided) 
*/
static isFormDone(element:Canvas | DOMElement | AutoTestLocator): boolean;

/**
*  Returns whether the FormItem associated with the given DOM element
*  is in a consistent state with no pending operations. Returns null
*  if the argument is not valid or isn't associated with an element
*  representing a valid
*  FormItem. Otherwise, returns true or false according as the conditions
*  below are all satisfied:
*  
*    
* - page has finished loading
*    
* - if the container widget of the item isn't the parent DynamicForm, then the
*      container widget must satisfy AutoTest.isCanvasDone (or AutoTest.isGridDone,
*      etc., as appropriate)
*    
* - the item cannot have any fetches in progress for missing display/value field
*      values
*    
* - picklists (SelectItem or ComboBoxItem) cannot have any pending row
*      fetches
*    
* - any contained FormItems must satisfy isItemDone() themselves
*    
* - any contained Canvas must satisfy AutoTest.isCanvasDone (or 
*      AutoTest.isGridDone, etc., as appropriate)
*  
* 
* @param {FormItem | DOMElement | AutoTestLocator}  DOM element to test (element obtained from canvas or SmartClient locator if provided) 
*/
static isItemDone(element:FormItem | DOMElement | AutoTestLocator): boolean;

/**
*  Returns whether the ListGrid associated with or contained by the given
*  DOM element
*  is in a consistent state with no pending operations. Returns null if the 
*  argument is not valid or isn't associated with an element inside a valid ListGrid.
*  
*  Otherwise, returns true or false according as the conditions below are all satisfied:
*  
*    
* - page has finished loading
*    
* - no pending scroll operations
*    
* - no pending filter editor operations
*    
* - no unsaved edits to the grid records
*    
* - no asynchronous regrouping operations are in progress
*    
* - no outstanding fetch/filter operations are present for the ResultSet
*    
* - no outstanding sort operations are present that will update the ListGrid
*  
* 
* @param {Canvas | DOMElement | AutoTestLocator}  DOM element to test (element obtained from canvas or SmartClient locator if provided) 
*/
static isGridDone(element:Canvas | DOMElement | AutoTestLocator): boolean;

/**
*  Returns whether the DOM element
*  is ready to be clicked on by a Selenium test. Returns null if the 
*  argument is not valid or isn't associated with an element representing
*  a valid canvas or form item.
*  Otherwise, returns true or false according as the conditions below are all satisfied:
*  
*    
* - page has finished loading
*    
* - no network operations are outstanding (configurable, 
*      see AutoTest.implicitNetworkWait)
*    
* - canvas is visible, enabled, and not masked,
*    
* - canvas satisfies AutoTest.isCanvasDone
*    
* - if canvas is a TileGrid, it satisfies AutoTest.isTileGridDone
*    
* - if canvas is a TileLayout, it satisfies AutoTest.isTileLayoutDone
*    
* - if canvas is a ListGrid or body of a ListGrid, it satisfies AutoTest.isGridDone
*    
* - if canvas is a DynamicForm, it satisfies AutoTest.isFormDone
*  
*  Note that for a form item in a DynamicForm, the DynamicForm must satisfy the third
*  condition above, while the container widget of the element must satisfy the remaining
*  conditions.
*  
* 
* @param {Canvas | FormItem | DOMElement | AutoTestLocator}  DOM element to test (element obtained from canvas, form item, or SmartClient locator if provided) 
*/
static isElementClickable(element:Canvas | FormItem | DOMElement | AutoTestLocator): boolean;

/**
*  Given a DOM element, returns whether the associated SmartClient Canvas or FormItem is
*  ready to receive keyPress events from a Selenium test. Returns null if the locator is
*  not valid or doesn't represent a valid Canvas or FormItem. Otherwise, returns true or
*  false according as the conditions below are all satisfied:
*  
*    
* - page has finished loading
*    
* - if a TextItem, FileItem, or TextAreaItem, 
*      it has native focus,
*    
* - the element satisfies AutoTest.isElementClickable
*  
* 
* @param {Canvas | FormItem | DOMElement | AutoTestLocator} DOM element to test   (element obtained from canvas, form item, or SmartClient locator if provided)
*/
static isElementReadyForKeyPresses(element:Canvas | FormItem | DOMElement | AutoTestLocator): boolean;

/**
*  Returns whether the loaded page is in a consistent state with no pending operations.
*  Specifically, returns true or false according as the conditions below are all satisfied:
*  
*    
* - page has finished loading
*    
* - all ListGrids (as defined by isc.isA.ListGrid) satisfy AutoTest.isGridDone
*    
* - all TileGrids that are drawn satisfy AutoTest.isTileGridDone
*    
* - all TileLayouts that are drawn satisfy AutoTest.isTileLayoutDone
*    
* - all DynamicForms that are drawn satisfy AutoTest.isFormDone
*    
* - all Canvii that are drawn satisfy AutoTest.isCanvasDone
*    
* - no network operations are outstanding (configurable, 
*      see AutoTest.implicitNetworkWait)
*    
* - there are no pending Canvas redraws (if includeRedraws parameter is true)
*  
*  Note: In a situation where messaging is being used to periodically refresh components,
*  or if the application contains a label updated every second to show the current time,
*  it's possible that this call might always return false if includeRedraws is true.
* 
* @param {boolean=} whether to check for pending Canvas redraws
* @param {boolean=} whether to allow unsaved edits for ListGrids
*/
static isSystemDone(includeRedraws?:boolean, allowEdits?:boolean): boolean;




}

/**
* Base class for Canvas and DrawItem.
*/
interface BaseWidget extends Class {




}
/**
* Creates a new BaseWidget
* 
* @param typeCheckedProperties {Partial<BaseWidget>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {BaseWidget} The new instance
*/
class BaseWidget{
static create(typeCheckedProperties? : Partial<BaseWidget>, uncheckedProperties?: Object): BaseWidget;







}

/**
* The BatchUploader handles the upload, validation, review and saving of a dataset
*  expressed in CSV or other upload formats.
*  
*  NOTE: BatchUploader is only available with SmartClient Power or better.
*  
*  By default, a BatchUploader consists of a single FileItem form field.
*  This form field will upload a file using the special "batchUpload" built-in DataSource.
*  The uploaded file data will be parsed and validated using the
*  BatchUploader.uploadDataSource,
*  then streamed back to the browser, along with any errors, for display in a ListGrid. 
*  
*  The BatchUploader.previewShown notification method will be fired when the uploaded records
*  are being displayed in this grid.
*  
*  The user can then correct any errors and submit the final dataset, which will be
*  added to the DataSource via a series of "add" DSRequests, all submitted as a single
*  HTTP request via RPCManager.startQueue.
*  Developers may specify a custom "add" operation to use on the target 
*  BatchUploader.uploadDataSource via the BatchUploader.uploadOperation 
*  property.
*  
*  Additional form fields can be added to the form that uploads the data file via 
*  BatchUploader.uploadFormFields. Values entered into these fields
*  are not included in the "add" DSRequests used to store the uploaded records. Instead, they
*  are stored as HttpSession attributes with the names corresponding to the 
*  names of the specified uploadFormFields (optionally with a 
*  BatchUploader.uploadFieldPrefix applied, in case
*  this is necessary to avoid name collisions in the session). This allows any custom logic
*  for the "add" operation to access these additional fields via httpSession.getAttribute().
*  If BatchUploader.uploadFormFields are not provided method 
*  httpSession.getAttribute() will not be called.
*  
*  Because all records are saved in a single HTTP request, a similar strategy of 
*  storing data as servletRequest or session attributes allows reuse of objects required to
*  perform the "add" operations (such as field values common to all added records,
*  or a SQL connection or transaction manager). 
*  
*  If DataSourceField.uploadFieldName is set on any of the 
*  BatchUploader.uploadDataSource's fields, the BatchUploader will use
*  that name to map the uploaded file's content.
*  
*  Note, that for BatchUploader.dataFormat header line is optional. If 
*  first non-empty line in the uploaded file has no matching field names, it is assumed that 
*  there's no header row, and all rows (including the first one) are treated as data rows.
*  
*  Imported data can be transformed during import, see DataSourceField.importStrategy for details.
*  
*  A couple of server-side techniques are interesting in conjunction with the BatchUploader.
*  One is to set the DataSource.serverConstructor property to point at your own class
*  that inherits from com.isomorphic.datasource.BasicDataSource. The most interesting
*  reason for doing this is to override the validate method and provide complete
*  custom validation - for example, checking relations to other tables.
*  
*  Another technique is to handle the initial SmartClient call in your own servlet, by setting the 
*  BatchUploader.dataURL property. You then handle the add requests
*  with a combination of your own code and SmartClient server API calls. This is a 
*  good way to add special pre- and post-processing to the normal server-side flow.
*  
*  Note: The special "batchUpload" DataSource, which should reside in the shared/ds 
*  folder of your application's webroot (see 
*  iscInstall)
*  .
*  is not part of your application's data flow, and it has nothing to do with the 
*  BatchUploader.uploadDataSource you use to actually persist the validated
*  and error-corrected data: it is simply a means to uploading the raw data in the first place.
*  Normally, you should simply ignore its presence and treat it as an internal detail of the 
*  SmartClient framework.
*  
*  However, there are circumstances in which you may wish to change it to achieve specific aims.
*  For example, you may wish to override the Java class it invokes, in order to insert your own
*  security or other validation logic into the initial upload flow. This is entirely in keeping
*  with the design, but we regard it as an out-of-the-ordinary use-case: normal usage is simply 
*  to ignore the presence of the batchUpload DataSource.
*  
*  BatchUploader is a VStack, that simply stacks members on the vertical axis without
*  trying to manage their height. If you need to control heights, you can set 
*  Layout.vPolicy to "fill"
*/
interface BatchUploader extends VStack {
/**
* Form used to specify file to upload, and any additional fields required.
* @type {DynamicForm}
* @default null
*/
uploadForm?: DynamicForm;

/**
* DataSource used to save uploaded records. Should have an operation of type "add".
*  
*  Be careful to note that this is the DataSource representing your data as it will be 
*  persisted to your server. It is completely different from the special "batchUpload"
*  DataSource which is used purely as a medium to upload the raw data to the server in
*  the first place.
* @type {DataSource}
* @default null
*/
uploadDataSource?: DataSource;

/**
* Optional DSRequest.operationId for the "add" operation used to add new
*  records to the BatchUploader.uploadDataSource.
* @type {string}
* @default null
*/
uploadOperation?: string;

/**
* Optional fields for the uploadForm.
* @type {Array<Partial<Partial<FormItem>>>}
* @default null
*/
uploadFormFields?: Array<Partial<Partial<FormItem>>>;

/**
* Title to display next to the FileItem field where the user enters a filename to upload
* @type {string}
* @default "Upload File"
*/
uploadFileLabel?: string;

/**
* Error message to show when the uploading process detects a file with no data.
* @type {string}
* @default "The provided file is blank. Please, provide a file with data."
*/
errorMessageFileIsBlank?: string;

/**
* Error message to show when the uploaded file has rows other than the first row that could not be parsed.
*  
*  This is a dynamic string - text within &#36;{...} will be evaluated as JS code
*  when the message is displayed.
*  
*  The following variables are available to be used in this message:
*  
*  
* - goodRowCount: Total rows that were parsed correctly.
*  
* - totalRows: Total rows to be parsed in the uploaded file.
*  
* - firstBadRow: First row that could not be parsed.
*  
* - badRowCount: Total rows that could not be parsed.
*  
* @type {string}
* @default "Some rows could not be parsed; the grid below shows ${goodRowCount} of ${totalRows} data rows. Row number ${firstBadRow} was the first row to fail to be parsed."
*/
errorMessageRowsNotParsed?: string;

/**
* Error message to show when the uploading process detects an unterminated quote string in the first line.
* @type {string}
* @default "Unterminated quote string - problem found in the first line at character position: ${errorOffset}."
*/
errorMessageUnterminatedQuote?: string;

/**
* Error message to show when the uploading process detects a missing delimiter or end of line after quoted value in the first line.
* @type {string}
* @default "Delimiter or end of line expected after quoted value - problem found in the first line at character position: ${errorOffset}."
*/
errorMessageDelimiterOrEndOfLine?: string;

/**
* Error message to show when the uploading process detects an invalid inputType.
* @type {string}
* @default "Invalid inputType value was set!"
*/
errorMessageInputType?: string;

/**
* Error message to show when the uploading process detects an Excel File, which is not supported.
* @type {string}
* @default "You uploaded an Excel file. Please save your data as a csv file and re-upload"
*/
errorMessageExcelFileDetected?: string;

/**
* Error message to show when the uploading process is unable to detect the delimiter.
* @type {string}
* @default "We were unable to guess the delimiter"
*/
errorMessageUndeterminedDelimiter?: string;

/**
* Message to display after data has been committed, when 
*  BatchUploader.showCommitConfirmation is true.
* @type {string}
* @default "Records added"
*/
commitConfirmationMessage?: string;

/**
* Message to display when the user clicks "Commit" but there is nothing we can commit
*  because every row in the grid has errors
* @type {string}
* @default "All records have errors; nothing to commit"
*/
allRecordsInErrorMessage?: string;

/**
* Message to display after data has been committed, when 
*  BatchUploader.showCommitConfirmation is true.
* @type {string}
* @default "Valid records added; some records remain in error"
*/
partialCommitConfirmationMessage?: string;

/**
* Message to display if at least one update was rolled back due to errors in another row.
*  See the DataSource.autoJoinTransactions for details of 
*  SmartClient's automatic transactional updates feature
* @type {string}
* @default "One or more updates were rolled-back due to errors on other rows"
*/
updatesRolledBackMessage?: string;

/**
* Whether to show the BatchUploader.commitConfirmationMessage after
*  data is successfully committed.
* @type {boolean}
* @default true
*/
showCommitConfirmation?: boolean;

/**
* Title for the BatchUploader.uploadButton.
* @type {string}
* @default "Upload"
*/
uploadButtonTitle?: string;

/**
* Button that triggers the upload.
* @type {IButton}
* @default null
*/
uploadButton?: IButton;

/**
* Grid which will show a preview of data to be uploaded, with errors flagged
* @type {ListGrid}
* @default null
*/
grid?: ListGrid;

/**
* Fields to apply to BatchUploader.grid. These will override the field definitions
*  in the BatchUploader.uploadDataSource on a field by field basis, as described under
*  DataBoundComponent.fields.
* @type {Array<Partial<ListGridField>>}
* @default null
*/
gridFields?: Array<Partial<ListGridField>>;

/**
* Button that commits changes once the user is happy with the data.
* @type {IButton}
* @default null
*/
commitButton?: IButton;

/**
* Title for the BatchUploader.commitButton.
* @type {string}
* @default "Commit"
*/
commitButtonTitle?: string;

/**
* Button that cancels the uncommitted upload.
* @type {IButton}
* @default null
*/
cancelButton?: IButton;

/**
* Title for the BatchUploader.cancelButton.
* @type {string}
* @default "Cancel"
*/
cancelButtonTitle?: string;

/**
* Button that scrolls grid to the previous error.
* @type {IButton}
* @default null
*/
previousButton?: IButton;

/**
* Title for the BatchUploader.previousButton.
* @type {string}
* @default "Previous error"
*/
previousButtonTitle?: string;

/**
* Button that scrolls grid to the next error.
* @type {IButton}
* @default null
*/
nextButton?: IButton;

/**
* Title for the BatchUploader.nextButton.
* @type {string}
* @default "Next error"
*/
nextButtonTitle?: string;

/**
* Object containing properties to send with every "add" request this batchUploader sends.
* @type {DSRequest}
* @default null
*/
requestProperties?: DSRequest;

/**
* If set, the batchUploader will copy this value to the queue of "add" requests it sends 
*  to the server to actually populate the data. You can use this facility to route the 
*  queue to your own server-side logic, for example to add pre- or post-processing.
* @type {string}
* @default null
*/
dataURL?: string;

/**
* String to prepend to the names of the additional fields specified in BatchUploader.uploadFormFields 
*  before they are stored in the HttpSession on the server. This property provides a basic
*  namespace facility, allowing you to avoid name collisions with existing session attributes.
*  
*  Example usage: if you have an additional field called "someDate" and you set 
*  uploadFieldPrefix to "myFields_", your additionalFormField will be available as an
*  HttpSession attribute called "myFields_someDate"
* @type {string}
* @default null
*/
uploadFieldPrefix?: string;

/**
* Specifies what action to take if the user attempts to commit a partially validated set
*  of data (ie, one that still contains some errors).
* @type {PartialCommitOption}
* @default "prompt"
*/
partialCommit?: PartialCommitOption;

/**
* If BatchUploader.partialCommit is set to "prompt", the text to display to the user in the 
*  confirmation dialog. By default, this text is "There are errors in your data so it 
*  cannot all be saved. If you proceed, you will lose the records with errors. Click 'OK'
*  to proceed anyway, or 'Cancel' to return to your data"
* @type {string}
* @default "See below"
*/
partialCommitPrompt?: string;

/**
* If BatchUploader.partialCommit is set to "prevent", the text to display to the user if they try 
*  to commit a dataset containing errors. By default, this text is "There are errors in your 
*  data. Please correct all errors before clicking Commit"
* @type {string}
* @default "See below"
*/
partialCommitError?: string;

/**
* If set, indicates that a warning dialog should be shown when Cancel is clicked, asking 
*  the user to confirm that this is really what they want to do. The actual warning message
*  is specified with BatchUploader.cancelConfirmMessage
* @type {boolean}
* @default true
*/
warnOnCancel?: boolean;

/**
* Confirmation message to show if the user clicks the "Cancel" button and BatchUploader.warnOnCancel
*  is true. Defaults to "You will lose any work you have done on this data. Proceed anyway?"
* @type {string}
* @default "See below"
*/
cancelConfirmMessage?: string;

/**
* Format to assume for user-provided data. Use ImportFormat "auto" for
*  auto-detection.
* @type {ImportFormat}
* @default "csv"
*/
dataFormat?: ImportFormat;

/**
* The delimiter to use when importing character-delimited files. The default is
*  comma (CSV).
* @deprecated * in favor of BatchUploader.uploadDelimiter

* @type {string}
* @default ","
*/
defaultDelimiter?: string;

/**
* The delimiter to use when importing character-delimited files. The default is
*  comma (CSV).
* @type {string}
* @default ","
*/
uploadDelimiter?: string;

/**
* The default character used to quote strings.
* @deprecated * in favor of BatchUploader.uploadQuoteString

* @type {string}
* @default "\""
*/
defaultQuoteString?: string;

/**
* The character used to quote strings. The default is double quotes.
* @type {string}
* @default "\""
*/
uploadQuoteString?: string;

/**
* On server-side, BatchUploader uses DataImport to import uploaded
*  files, specifically APIs taking java.io.Reader parameter.
*  BatchUploader.uploadEncoding setting defines the encoding, which will
*  be used to create a java.io.Reader instance to read data from uploaded
*  files. The default is "UTF-8".
* @type {string}
* @default "UTF-8"
*/
uploadEncoding?: string;

/**
* Controls if server-side API BatchUpload.parseUploadData(...) will convert imported boolean
*  fields values to actual Booleans or will leave them as Strings. The default is true.
*  
*  Default behavior would interpret boolean values by following rules:
*  
*  
* - Accept all caps permutations of "true"/"false", "yes"/"no", and "null"
*  
* - "T" == true, "F" == false
*  
* - "Y" == true, "N" == false
*  
* - "0" is false
*  
* - empty string is null
*  
* - everything else is true
*  
* @type {boolean}
* @default true
*/
autoInterpretBooleans?: boolean;

/**
* If columns were present in the imported data that were discarded because they could not be
*  matched to any DataSource fields, whether these should be displayed to the user, using the
*  BatchUploader.discardedColumnsMessage shown within the BatchUploader.uploadStatusMessages component.
* @type {boolean}
* @default true
*/
displayDiscardedColumns?: boolean;

/**
* Message displayed when columns in the imported file were discarded and
*  BatchUploader.displayDiscardedColumns is true. Within this message, ${discardedColumns} can be
*  used to show a comma separated list of the column names that were discarded (example:
*  "price, saleDate, total").
*  
*  Default message is: "The following columns in your uploaded file were ignored because they
*  did not match any of the expected column names: ${discardedColumns}"
* @type {string}
* @default "..."
*/
discardedColumnsMessage?: string;

/**
* Container for informational messages that are shown when a user attempts an upload.
*  Appears above the BatchUploader.grid.
* @type {HTMLFlow}
* @default null
*/
uploadStatusMessages?: HTMLFlow;

/**
* FileItem for selecting the file to upload.
* @type {FileItem}
* @default null
*/
uploadFileItem?: FileItem;



/**
*  Notification method fired when the BatchUploader.grid is populated with
*  a new set of data for the user to preview before commit.
*  
*  This notification occurs after the user has uploaded a new data file, the data
*  has been processed on the server, and the preview grid populated with the data
*  ready for committing. Developers may use this notification to examine or modify
*  the data set to be uploaded. The ListGrid.data object will be populated with
*  the array of uploaded records, and standard grid APIs such as 
*  ListGrid.getRowErrors, ListGrid.setEditValue, etc may be used 
*  to interact with this data.
*  
*  Note that developers wishing to manipulate the uploaded data can also do this
*  on the server side when user hits the commit button to submit the data to the
*  BatchUploader.uploadDataSource. This can be achieved by setting the 
*  BatchUploader.uploadOperation to call a custom 
*  DataSource.operationBindings on the target dataSource.
* 
* 
* @param {ListGrid} The BatchUploader.grid autoChild.
*/
previewShown(grid:ListGrid): void;

/**
*  Notification method fired when the BatchUploader.commitButton is clicked.
*  
*  This notification occurs before actually committing data to the server. It allows to
*  make changes to the data after user edits, but before it will be sent to server.
*  
*  Read also BatchUploader.previewShown docs for details how to change data
*  in grid and for possibility to interrupt upload process on the server as well.
* 
* 
* @param {ListGrid} The BatchUploader.grid autoChild.
*/
beforeCommit(grid:ListGrid): void;



}
/**
* Creates a new BatchUploader
* 
* @param typeCheckedProperties {Partial<BatchUploader>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {BatchUploader} The new instance
*/
class BatchUploader{
static create(typeCheckedProperties? : Partial<BatchUploader>, uncheckedProperties?: Object): BatchUploader;







}

/**
* FormItem intended for inserting blurbs of instructional HTML into DynamicForms.
*  
*  Set the defaultValue of this item to the HTML you want to embed in the form.
*/
interface BlurbItem extends FormItem {
/**
* Default class used to construct the EditProxy for this component
*  when the component is Canvas.setEditMode.
* @type {SCClassName}
* @default "TextItemEditProxy"
*/
editProxyConstructor?: SCClassName;

/**
* Blurb items show no title by default.
* @type {boolean}
* @default false
*/
showTitle?: boolean;

/**
* By default, texts span all remaining columns
* @type {Integer | string}
* @default "*"
*/
colSpan?: Integer | string;

/**
* Base css style for this item.
* @type {CSSStyleName}
* @default "staticTextItem"
*/
textBoxStyle?: CSSStyleName;

/**
* Should the user be able to select the text in this item?
* @type {boolean}
* @default true
*/
canSelectText?: boolean;

/**
* If true, item contents can wrap. If false, all the contents should appear on a single line.
* @type {boolean}
* @default null
*/
wrap?: boolean;

/**
* If true, text that exceeds the specified size of the form item will be clipped.
* @type {boolean}
* @default false
*/
clipValue?: boolean;





}
/**
* Creates a new BlurbItem
* 
* @param typeCheckedProperties {Partial<BlurbItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {BlurbItem} The new instance
*/
class BlurbItem{
static create(typeCheckedProperties? : Partial<BlurbItem>, uncheckedProperties?: Object): BlurbItem;







}

/**
* Boolean form item, implemented with customizable checkbox images
*/
interface BooleanItem extends FormItem {




}
/**
* Creates a new BooleanItem
* 
* @param typeCheckedProperties {Partial<BooleanItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {BooleanItem} The new instance
*/
class BooleanItem{
static create(typeCheckedProperties? : Partial<BooleanItem>, uncheckedProperties?: Object): BooleanItem;







}

/**
* The Browser class contains various class attributes that indicate basic properties
*  of the browser and whether certain features are enabled.
*/
interface Browser extends Class {




}
/**
* Creates a new Browser
* 
* @param typeCheckedProperties {Partial<Browser>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Browser} The new instance
*/
class Browser{
static create(typeCheckedProperties? : Partial<Browser>, uncheckedProperties?: Object): Browser;
/**
* Does the browser support both mouse and touch input?
* @type {boolean}
* @default "varies"
*/
static supportsDualInput?: boolean;

/**
* Is the application running on a touch device (e.g. iPhone, iPad, Android device, etc.)?
*  
*  SmartClient's auto-detected value for isTouch can be overridden via
*  Browser.setIsTouch.
* @type {boolean}
* @default "auto-detected based on device"
*/
static isTouch?: boolean;

/**
* Is the application running on a tablet device (e.g. iPad, Nexus 7)?
*  
*  SmartClient can correctly determine whether the device is a tablet in most cases. On any
*  uncommon device for which this variable is incorrect, you can define the isc_isTablet
*  global with the correct value, and SmartClient will use isc_isTablet for
*  Browser.isTablet instead of its own detection logic. Alternatively, you can use
*  Browser.setIsTablet to change this global variable before any components are
*  created.
*  
*  The value of this variable is only meaningful on touch devices.
* @type {boolean}
* @default "auto-detected based on device"
*/
static isTablet?: boolean;

/**
* Is the application running on a handset-sized device, with a typical screen width of around
*  3-4 inches?
*  
*  This typically implies that the application will be working with only 300-400 pixels.
* @type {boolean}
* @default "auto-detected based on device"
*/
static isHandset?: boolean;

/**
* Is the application running in a desktop browser? This is true if Browser.isTablet
*  and Browser.isHandset are both false.
* @type {boolean}
* @default "auto-detected based on device"
*/
static isDesktop?: boolean;

/**
* Whether SmartClient supports the current browser.
*  
*  Note that this flag will only be available on browsers that at least support basic
*  JavaScript.
* @type {boolean}
* @default "auto-detected based on browser"
*/
static isSupported?: boolean;

/**
* Whether the current browser supports CSS3 and whether SmartClient is configured to use
*  CSS3 features (via the setting of window.isc_css3Mode).
*  
*  If isc_css3Mode is "on" then useCSS3 is set to true. If isc_css3Mode is set to
*  "supported", "partialSupport", or is unset, then useCSS3 is set to true only if the browser
*  is a WebKit-based browser, Firefox, IE 9 in standards mode, or IE 10+. If isc_css3Mode is set
*  to "off" then useCSS3 is set to false.
* @type {boolean}
* @default "see below"
*/
static useCSS3?: boolean;

/**
* Controls how agressive components based on the GridRenderer are with respect to
*  redraws and data fetches. Modern browsers can generally handle much more frequent redraws
*  and most server configurations can handle fetching more data more frequently in order to
*  reduce the lag the end user perceives when scrolling through databound grids. Starting with
*  SmartClient 11.0/SmartGWT 6.0, this more aggressive redraw and fetch behavior us the
*  default, but can be reverted to the old behavior if desired - see below.
*  
*  This flag controls the defaults for several other properties (value on left is default for
*  high performance mode, value on right is default when this mode is disabled.
*  
*  
* - ListGrid.dataFetchDelay 1 -> 300
*  
* - ListGrid.drawAheadRatio 2.0 -> 1.3
*  
* - ListGrid.quickDrawAheadRatio 2.0 -> 1.3
*  
* - ListGrid.scrollRedrawDelay 0 -> 75
*  
* - ListGrid.scrollWheelRedrawDelay 0 -> 250
*  
* - ListGrid.touchScrollRedrawDelay 0 -> 300
*  
*  Note: since TreeGrid is a subclass of ListGrid, the above settings
*  also apply to TreeGrids.
*  
*  
* - GridRenderer.drawAheadRatio 2.0 -> 1.3
*  
* - GridRenderer.quickDrawAheadRatio 2.0 -> 1.3
*  
* - GridRenderer.scrollRedrawDelay 0 -> 75
*  
* - GridRenderer.touchScrollRedrawDelay 0 -> 300
*  
*  
*  By default, for all browsers except Android-based Chrome, this flag is set to true, but can
*  be explicitly disabled by setting isc_useHighPerformanceGridTimings=false in a
*  script block before loading SmartClient modules. Turning off high performance timings
*  effectively enables the original SmartClient/SmartGWT behavior prior to the SmartClient
*  11.0/SmartGWT 6.0 release.
* @type {boolean}
* @default "see below"
*/
static useHighPerformanceGridTimings?: boolean;





/**
*  Setter for Browser.isTouch to allow this global variable to be changed at runtime.
*  This advanced method is provided to override SmartClient's auto-detection logic, since the
*  framework can only detect touch devices that existed at the time the platform was released.
*  Any change to Browser.isTouch must be made before any component is created;
*  it is an application error to attempt to change isTouch after
*  components have been created.
*  
*  Note that setting Browser.isTouch might affect the values of
*  Browser.isDesktop, Browser.isTablet, and/or Browser.isHandset.
* 
* 
* @param {boolean} new setting for Browser.isTablet.
*/
static setIsTouch(isTouch:boolean): void;

/**
*  Setter for Browser.isTablet to allow this global variable to be changed at runtime.
*  This advanced method is provided to override SmartClient's detection of devices, since the
*  framework can only detect devices that existed at the time the platform was released. Any
*  changes to Browser.isDesktop, Browser.isHandset, or Browser.isTablet
*  must be made before any component is created;
*  it is an application error to attempt to change isDesktop,
*  isHandset, or isTablet after components have been created.
*  
*  Note that setting Browser.isTablet might affect the values of
*  Browser.isDesktop and Browser.isHandset.
* 
* 
* @param {boolean} new setting for Browser.isTablet.
*/
static setIsTablet(isTablet:boolean): void;

/**
*  Setter for Browser.isHandset to allow this global variable to be changed at runtime.
*  This advanced method is provided to override SmartClient's detection of devices, since the
*  framework can only detect devices that existed at the time the platform was released. Any
*  changes to Browser.isDesktop, Browser.isHandset, or Browser.isTablet
*  must be made before any component is created;
*  it is an application error to attempt to change isDesktop,
*  isHandset, or isTablet after components have been created.
*  
*  Note that setting Browser.isHandset might affect the values of
*  Browser.isDesktop and Browser.isTablet.
* 
* 
* @param {boolean} new setting for Browser.isHandset.
*/
static setIsHandset(isHandset:boolean): void;

/**
*  Setter for Browser.isDesktop to allow this global variable to be changed at runtime.
*  This advanced method is provided to override SmartClient's detection of devices, since the
*  framework can only detect devices that existed at the time the platform was released. Any
*  changes to Browser.isDesktop, Browser.isHandset, or Browser.isTablet
*  must be made before any component is created;
*  it is an application error to attempt to change isDesktop,
*  isHandset, or isTablet after components have been created.
*  
*  Note that setting Browser.isDesktop might affect the values of
*  Browser.isHandset and Browser.isTablet.
* 
* 
* @param {boolean} new setting for Browser.isDesktop.
*/
static setIsDesktop(isDesktop:boolean): void;




}

/**
* Container for a Browser Plugin.
*/
interface BrowserPlugin extends Canvas {




}
/**
* Creates a new BrowserPlugin
* 
* @param typeCheckedProperties {Partial<BrowserPlugin>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {BrowserPlugin} The new instance
*/
class BrowserPlugin{
static create(typeCheckedProperties? : Partial<BrowserPlugin>, uncheckedProperties?: Object): BrowserPlugin;







}

/**
* The Button widget class implements interactive, style-based button widgets.
*/
interface Button extends StatefulCanvas {
/**
* The title HTML to display in this button.
* @type {HTMLString}
* @default "Untitled Button"
*/
title?: HTMLString;

/**
* If set to true, if the StatefulCanvas.title of this button contains the
*  specified Canvas.accessKey, when the title is displayed to the user
*  it will be modified to include HTML to underline the accessKey.
*  Note that this property may cause titles that include HTML (rather than simple strings)
*  to be inappropriately modified, so should be disabled if your title string includes
*  HTML characters.
* @type {boolean}
* @default null
*/
hiliteAccessKey?: boolean;

/**
* If true and the title is clipped, then a hover containing the full title of this button
*  is enabled.
* @type {boolean}
* @default false
*/
showClippedTitleOnHover?: boolean;

/**
* Optional icon to be shown with the button title text. 
*  
*  Specify as the partial URL to an image, relative to the imgDir of this component.
*  A sprited image can be specified using the SCSpriteConfig format.
*  
*  Note that the string "blank" is a valid setting for this attribute and will always 
*  result in the system blank image, with no state suffixes applied. Typically, this 
*  might be used when an iconStyle is also specified and the iconStyle renders the icon via 
*  a stateful background-image or other CSS approach.
* @type {SCImgURL}
* @default null
*/
icon?: SCImgURL;

/**
* Size in pixels of the icon image.
*  
*  The StatefulCanvas.iconWidth and StatefulCanvas.iconHeight
*  properties can be used to configure width and height separately.
*  
*  Note: When configuring the properties of a StatefulCanvas (or derivative)
*  AutoChild, it is best to set the iconWidth and iconHeight
*  to the same value rather than setting an iconSize. This is because certain
*  skins or customizations thereto might set the iconWidth and iconHeight,
*  making the customization of the AutoChild's iconSize ineffective.
* @type {Integer}
* @default "16"
*/
iconSize?: Integer;

/**
* Width in pixels of the icon image.
*  
*  If unset, defaults to StatefulCanvas.iconSize.
* @type {Integer}
* @default null
*/
iconWidth?: Integer;

/**
* Height in pixels of the icon image.
*  
*  If unset, defaults to StatefulCanvas.iconSize.
* @type {Integer}
* @default null
*/
iconHeight?: Integer;

/**
* Base CSS style applied to the icon image. If set, as the StatefulCanvas changes
*  StatefulCanvas.state and/or is StatefulCanvas.selected,
*  suffixes will be appended to iconStyle to form the className set on the
*  image element.
*  
*  The following table lists out the standard set of suffixes which may be appended:
*  
*  CSS Class AppliedDescription
*  iconStyleDefault CSS style
*  iconStyle+Selected
*    Applied when StatefulCanvas.selected and StatefulCanvas.showSelectedIcon
*    are true.
*  iconStyle+Focused
*    Applied when the component has keyboard focus, if
*    StatefulCanvas.showFocusedIcon is true, and
*    StatefulCanvas.showFocusedAsOver is not true.
*  iconStyle+Over
*    Applied when StatefulCanvas.showRollOverIcon is set to true and either
*    the user rolls over the component or StatefulCanvas.showFocusedAsOver is true
*    and the component has keyboard focus.
*  iconStyle+Down
*    Applied when the user presses the mouse button on the component if
*      StatefulCanvas.showDownIcon is set to true
*  iconStyle+Disabled
*    Applied when the component is Canvas.disabled
*     if StatefulCanvas.showDisabledIcon is true.
*  Combined styles
*  iconStyle+SelectedFocused
*    Combined Selected and focused styling
*  iconStyle+SelectedOver
*    Combined Selected and rollOver styling
*  iconStyle+FocusedOver
*    Combined Focused and rollOver styling
*  iconStyle+SelectedFocusedOver
*    Combined Selected, Focused and rollOver styling
*  iconStyle+SelectedDown
*    Combined Selected and mouse-down styling
*  iconStyle+FocusedDown
*    Combined Focused and mouse-down styling
*  iconStyle+SelectedFocusedDown
*    Combined Selected, Focused and mouse-down styling
*  iconStyle+SelectedDisabled
*    Combined Selected and Disabled styling
*  
*  
*  In addition, if StatefulCanvas.showRTLIcon is true, then in RTL mode, a final
*  "RTL" suffix will be appended.
* @type {CSSStyleName}
* @default null
*/
iconStyle?: CSSStyleName;

/**
* If this button is showing an icon should it appear to the left or right of the title?
*  valid options are "left" and "right".
* @type {string}
* @default "left"
*/
iconOrientation?: string;

/**
* If this button is showing an icon should it be right or left aligned?
* @type {string}
* @default null
*/
iconAlign?: string;

/**
* Pixels between icon and title text.
* @type {Integer}
* @default "6"
*/
iconSpacing?: Integer;

/**
* If using an icon for this button, whether to switch the icon image if the button becomes
*  disabled.
* @type {boolean}
* @default true
*/
showDisabledIcon?: boolean;

/**
* If using an icon for this button, whether to switch the icon image on mouse rollover.
* @type {boolean}
* @default false
*/
showRollOverIcon?: boolean;

/**
* Specifies the cursor to display when the mouse pointer is over the icon image.
* @type {Cursor}
* @default null
*/
iconCursor?: Cursor;

/**
* Specifies the cursor to display when the mouse pointer is over the icon image and this
*  StatefulCanvas is Canvas.disabled.
*  
*  If not set and the mouse pointer is over the icon image, Button.iconCursor
*  will be used.
* @type {Cursor}
* @default null
*/
disabledIconCursor?: Cursor;

/**
* If using an icon for this button, whether to switch the icon image when the button
*  receives focus.
*  
*  If StatefulCanvas.showFocusedAsOver is true, the "Over" icon will be
*  displayed when the canvas has focus, otherwise a separate "Focused" icon
*  will be displayed
* @type {boolean}
* @default false
*/
showFocusedIcon?: boolean;

/**
* If using an icon for this button, whether to switch the icon image when the mouse goes
*  down on the button.
* @type {boolean}
* @default false
*/
showDownIcon?: boolean;

/**
* If using an icon for this button, whether to switch the icon image when the button
*  becomes selected.
* @type {boolean}
* @default false
*/
showSelectedIcon?: boolean;

/**
* If true, ignore the specified size of this widget and always size just large
*  enough to accommodate the title. If setWidth() is explicitly called on an
*  autoFit:true button, autoFit will be reset to false.
*  
*  Note that for StretchImgButton instances, autoFit will occur horizontally only, as 
*  unpredictable vertical sizing is likely to distort the media. If you do want vertical 
*  auto-fit, this can be achieved by simply setting a small height, and having 
*  overflow:"visible"
* @type {boolean}
* @default null
*/
autoFit?: boolean;

/**
* Size for this component's horizontal dimension. See Canvas.width for more
*  details.
*  
*  Note that if StatefulCanvas.autoFit is set, this property will be ignored so that the widget
*  is always sized just large enough to accommodate the title.
* @type {number | string}
* @default null
*/
width?: number | string;

/**
* Size for this component's vertical dimension. See Canvas.height for more
*  details.
*  
*  Note that if StatefulCanvas.autoFit is set on non-StretchImgButton instances, this
*  property will be ignored so that the widget is always sized just large enough to
*  accommodate the title.
* @type {number | string}
* @default null
*/
height?: number | string;

/**
* Base CSS style className applied to the component. 
*  
*  As the component changes StatefulCanvas.state and/or is selected, 
*  suffixes will be added to the base style. In some cases more than one suffix will 
*  be appended to reflect a combined state ("Selected" + "Disabled", for example).
*  
*  See StatefulCanvas.getStateSuffix for a description of the default set
*  of suffixes which may be applied to the baseStyle
*  
*  Rotated Titles
*  
*  The Framework doesn't have built-in support for rotating button titles in a fashion
*  similar to FacetChart.rotateLabels. However, you can manually configure
*  a button to render with a rotated title by applying custom CSS via this property.
*  
*  For example, given a button with a height of 120 and a width of 48, if you
*  copied the existing buttonXXX style declarations from skin_styles.css as new,
*  rotatedTitleButtonXXX declarations, and then added the lines:
*  
*    -ms-transform:   translate(-38px,0px) rotate(270deg);
*    -webkit-transform: translate(-38px,0px) rotate(270deg);
*    transform:     translate(-38px,0px) rotate(270deg);
*    overflow: hidden;
*    text-overflow: ellipsis;
*    width:116px;
*  in the declaration section beginning:
*  
*  .rotatedTitleButton,
*  .rotatedTitleButtonSelected,
*  .rotatedTitleButtonSelectedOver,
*  .rotatedTitleButtonSelectedDown,
*  .rotatedTitleButtonSelectedDisabled,
*  .rotatedTitleButtonOver,
*  .rotatedTitleButtonDown,
*  .rotatedTitleButtonDisabled {
*  then applying that style to the button with Canvas.overflow: "clip_h"
*  would yield a vertically-rendered title with overflow via ellipsis as expected, and also
*  wrap with Button.wrap.
*  
*  Note that:
*  
* - The explicit width applied via CSS is needed because rotated
*  elements don't inherit dimensions in their new orientation from the DOM - 
*  the transform/rotation occurs independently of layout.
*  
* - The translation transform required along the x-axis is roughly
*  (width - height) / 2, but may need slight offsetting for optimal centering.
*  
* - We've explicitly avoided describing an approach based on CSS "writing-mode", since
*  support is incomplete and bugs are present in popular browsers such as Firefox and
*  Safari that would prevent it from being used without Framework assistance.
* @type {CSSStyleName}
* @default "button"
*/
baseStyle?: CSSStyleName;

/**
* if defined, iconOnlyBaseStyle is used as the base CSS style className,
*  instead of Button.baseStyle, if Button.canAdaptWidth is set and the 
*  Button.adaptWidthShowIconOnly.
* @type {CSSStyleName}
* @default null
*/
iconOnlyBaseStyle?: CSSStyleName;

/**
* Whether this component is selected. For some components, selection affects appearance.
* @type {boolean}
* @default false
*/
selected?: boolean;

/**
* String identifier for this canvas's mutually exclusive selection group.
* @type {string}
* @default null
*/
radioGroup?: string;

/**
* Behavior on state changes -- BUTTON, RADIO or CHECKBOX
* @type {SelectionType}
* @default "button"
*/
actionType?: SelectionType;

/**
* Current "state" of this widget. The state setting is automatically updated as the 
*  user interacts with the component (see StatefulCanvas.showRollOver, 
*  StatefulCanvas.showDown, StatefulCanvas.showDisabled).
*  
*  StatefulCanvases will have a different appearance based
*  on their current state. 
*  By default this is handled by changing the css className applied to
*  the StatefulCanvas - see StatefulCanvas.baseStyle and 
*  StatefulCanvas.getStateSuffix for a description of how this is done.
*  
*  For Img or StretchImg based subclasses of StatefulCanvas, the 
*  appearance may also be updated by changing the src of the rendered image. See
*  Img.src and StretchImgButton.src for a description of how the URL 
*  is modified to reflect the state of the widget in this case.
* @type {State}
* @default ""
*/
state?: State;

/**
* Should we visibly change state when disabled?
* @type {boolean}
* @default true
*/
showDisabled?: boolean;

/**
* Should we visibly change state when the mouse goes down in this object?
* @type {boolean}
* @default false
*/
showDown?: boolean;

/**
* Should we visibly change state when the canvas receives focus? If
*  StatefulCanvas.showFocusedAsOver is true, then "over"
*  will be used to indicate focus. Otherwise a separate "focused" state
*  will be used.
* @type {boolean}
* @default false
*/
showFocused?: boolean;

/**
* Should we visibly change state when the mouse goes over this object?
* @type {boolean}
* @default false
*/
showRollOver?: boolean;

/**
* Horizontal alignment of this component's title.
* @type {Alignment}
* @default "center"
*/
align?: Alignment;

/**
* Vertical alignment of this component's title.
* @type {VerticalAlignment}
* @default "center"
*/
valign?: VerticalAlignment;

/**
* A boolean indicating whether the button's title should word-wrap, if necessary.
* @type {boolean}
* @default false
*/
wrap?: boolean;

/**
* This flag enables Canvas.canAdaptWidth for the button.
*  
*  If enabled the button will support rendering in a 'collapsed' view if there isn't enough
*  space in a layout to render it at normal size. There are a couple of ways this can be achieved.
*  
*  
* - If Button.adaptWidthShowIconOnly is true and this button shows an icon, the title
*    will be hidden if there isn't enough space to render it, allowing it to shrink to either
*    the rendered icon width, or any specified Canvas.minWidth, whichever is larger.
*  
* - Otherwise, if the button has a specified Canvas.minWidth, and 
*    Button.autoFit is true, autoFit will be temporarily disabled, if there isn't enough
*    room, allowing the title to be clipped
*  
*  In either case the title will show on hover unless an explicit hover has been
*  specified such as by overriding Button.titleHoverHTML.
* @type {boolean}
* @default false
*/
canAdaptWidth?: boolean;

/**
* Clip the contents of the button if necessary.
* @type {Overflow}
* @default "hidden"
*/
overflow?: Overflow;

/**
* If Button.canAdaptWidth is true, and this button has a specified Button.icon, should
*  the title be hidden, allowing the button to shrink down to just show the icon when there isn't
*  enough horizontal space in a layout to show the default sized button?
* @type {boolean}
* @default true
*/
adaptWidthShowIconOnly?: boolean;



/**
*  Return the title - HTML drawn inside the component.
*  
*  Default is to simply return this.title.
* 
*/
getTitle(): HTMLString;

/**
*  Setter for the StatefulCanvas.title.
* 
* @param {HTMLString} the new title HTML.
*/
setTitle(newTitle:HTMLString): void;

/**
*  Changes the orientation of the icon relative to the text of the button.
* 
* 
* @param {string} The new orientation of the icon relative to the text of the button.
*/
setIconOrientation(orientation:string): void;

/**
*  Change the icon being shown next to the title text.
* 
* @param {SCImgURL} URL of new icon
*/
setIcon(icon:SCImgURL): void;

/**
*  Setter method for the StatefulCanvas.autoFit property. Pass in true or false to turn
*  autoFit on or off. When autoFit is set to false, canvas will be resized to
*  it's previously specified size.
* 
* @param {boolean} New autoFit setting.
*/
setAutoFit(autoFit:boolean): void;

/**
*  Sets the base CSS style. As the component changes state and/or is selected, suffixes will be
*  added to the base style.
* 
* @param {CSSStyleName} new base style
*/
setBaseStyle(style:CSSStyleName): void;

/**
*  Select this object.
* 
*/
select(): void;

/**
*  Select this object.
* 
*/
deselect(): void;

/**
*  Find out if this object is selected
* 
*/
isSelected(): boolean;

/**
*  Select this object.
* 
*/
setSelected(): void;

/**
*  Add this widget to the specified mutually exclusive selection group with the ID
*  passed in.
*  Selecting this widget will then deselect any other StatefulCanvases with the same
*  radioGroup ID.
*  StatefulCanvases can belong to only one radioGroup, so this method will remove from 
*  any other radiogroup of which this button is already a member.
* 
* @param {string} - ID of the radiogroup to which this widget should be added
*/
addToRadioGroup(groupID:string): void;

/**
*  Remove this widget from the specified mutually exclusive selection group with the ID
*  passed in.
*  No-op's if this widget is not a member of the groupID passed in.
*  If no groupID is passed in, defaults to removing from whatever radioGroup this widget
*  is a member of.
* 
* @param {string=} - optional radio group ID (to ensure the widget is removed                    from the appropriate group.
*/
removeFromRadioGroup(groupID?:string): void;

/**
*  Update the 'actionType' for this canvas (radio / checkbox / button)
*  If the canvas is currently selected, and the passed in actionType is 'button'
*  this method will deselect the canvas.
* 
*/
setActionType(): void;

/**
*  Return the 'actionType' for this canvas (radio / checkbox / button)
* 
*/
getActionType(): void;

/**
*  Sets the StatefulCanvas.state of this object, changing its appearance.
*  Note: newState cannot be
*  "Disabled"
*  
*  if StatefulCanvas.showDisabled is false.
* 
* 
* @param {State} the new state.
*/
setState(newState:State): void;

/**
*  Enable or disable this object
* 
* @param {boolean} true if this widget is to be disabled
*/
setDisabled(disabled:boolean): void;

/**
*  Return the state of this StatefulCanvas
* 
*/
getState(): State;

/**
*  This property contains the default 'action' for the Button to fire when activated.
* 
*/
action(): void;

/**
*  Is the title of this button clipped?
* 
*/
titleClipped(): boolean;

/**
*  Returns the HTML that is displayed by the default Button.titleHover
*  handler. Return null or an empty string to cancel the hover.
*  
* 
* @param {HTMLString} the HTML that would have been displayed by default
*/
titleHoverHTML(defaultHTML:HTMLString): HTMLString;

/**
*  Set whether the title of this button should be allowed to wrap if too long for the button's
*  specified width.
* 
* 
* @param {boolean} whether to wrap the title
*/
setWrap(newWrap:boolean): void;

/**
*  Sets the (horizontal) alignment of this buttons content.
* 
*/
setAlign(): void;

/**
*  Sets the vertical alignment of this buttons content.
* 
*/
setVAlign(): void;

/**
*  If this button is showing an Button.icon, a separate click handler for the
*  icon may be defined as this.iconClick.
*  Returning false will suppress the standard button click handling code.
* 
*/
iconClick(): boolean;

/**
*  Optional stringMethod to fire when the user hovers over this button and the title is
*  clipped. If Button.showClippedTitleOnHover is true, the default behavior is to
*  show a hover canvas containing the HTML returned by Button.titleHoverHTML.
*  Return false to suppress this default behavior.
* 
*/
titleHover(): boolean;



}
/**
* Creates a new Button
* 
* @param typeCheckedProperties {Partial<Button>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Button} The new instance
*/
class Button{
static create(typeCheckedProperties? : Partial<Button>, uncheckedProperties?: Object): Button;







}

/**
* FormItem for adding a Button to a form.
*/
interface ButtonItem extends CanvasItem {
/**
* Default class used to construct the EditProxy for this component
*  when the component is Canvas.setEditMode.
* @type {SCClassName}
* @default "ButtonItemEditProxy"
*/
editProxyConstructor?: SCClassName;

/**
* By default buttonItems are sized to match their content (see ButtonItem.autoFit).
*  Specifying an explicit size for the button will disable this behavior.
* @type {number}
* @default null
*/
height?: number;

/**
* Optional baseStyle will be applied to the button.
* @type {CSSStyleName}
* @default null
*/
baseStyle?: CSSStyleName;

/**
* Optional icon image to display on the button for this item. See Button.icon.
* @type {SCImgURL}
* @default null
*/
icon?: SCImgURL;

/**
* Buttons do not show a title by default.
* @type {boolean}
* @default false
*/
showTitle?: boolean;

/**
* These items are in a row by themselves by default
* @type {boolean}
* @default true
*/
startRow?: boolean;

/**
* These items are in a row by themselves by default
* @type {boolean}
* @default true
*/
endRow?: boolean;

/**
* This item is an autoChild generated Canvas displayed by
*  the ButtonItem and is an instance of Button by defaut, cuztomizeable 
*  via the ButtonItem.buttonConstructor attribute.
* @type {Canvas}
* @default null
*/
button?: Canvas;

/**
* Constructor class for the button.
* @type {Class}
* @default "Button"
*/
buttonConstructor?: Class;

/**
* Should the button auto fit to its title. Maps to Button.autoFit attribute.
*  Note that if an explicit width or height is specified for this item, it will be respected,
*  disabling autoFit behavior
* @type {boolean}
* @default true
*/
autoFit?: boolean;

/**
* Custom Properties to apply to our button item.
* @type {object}
* @default null
*/
buttonProperties?: object;

/**
* If this item is FormItem.getCanEdit, how should this item be displayed
*  to the user? If set, overrides the form-level DynamicForm.readOnlyDisplay default.
* @type {ReadOnlyDisplayAppearance}
* @default "disabled"
*/
readOnlyDisplay?: ReadOnlyDisplayAppearance;

/**
* Criteria to be evaluated to determine whether this item should be enabled. This
*  property is incompatible with FormItem.readOnlyWhen and any setting you provide for
*  the latter will be ignored if this property is set.
*  
*  Criteria are evaluated against the ${isc.DocUtils.linkForRef('method:DynamicForm.getValues','form\'s current values')} as
*  well as the current Canvas.ruleScope. Criteria are re-evaluated
*  every time form values or the rule context changes, whether by end user action or by
*  programmatic calls.
*  
*  A basic criteria uses textMatchStyle:"exact". When specified in
*  componentXML this property allows
*  xmlCriteriaShorthand for defining criteria.
*  
*  Note: A ButtonItem using enableWhen must have a FormItem.name defined.
*  CanvasItem.shouldSaveValue can be set to false to prevent the field from storing
*  its value into the form's values.
* @type {AdvancedCriteria}
* @default null
*/
enableWhen?: AdvancedCriteria;

/**
* This property governs whether StatefulCanvas.showFocusedAsOver is true
*  on the automatically created Button for this item.
* @type {boolean}
* @default null
*/
showFocusedAsOver?: boolean;

/**
* The (horizontal) alignment of this button's title.
* @type {Alignment}
* @default "center"
*/
buttonTitleAlign?: Alignment;



/**
*  Called when a ButtonItem is clicked on.
* 
* 
* @param {DynamicForm} the managing DynamicForm instance
* @param {FormItem} the form item itself (also available as "this")
*/
click(form:DynamicForm, item:FormItem): boolean;

/**
*  Set the title.
* 
* @param {string} new title
*/
setTitle(newTitle:string): void;

/**
*  Sets the (horizontal) alignment of this button's title.
* 
* @param {Alignment} new title alignment
*/
setButtonTitleAlign(alignment:Alignment): void;

/**
*  Sets showFocusedAsOver.
* 
* @param {boolean} */
setShowFocusedAsOver(showFocusedAsOver:boolean): void;



}
/**
* Creates a new ButtonItem
* 
* @param typeCheckedProperties {Partial<ButtonItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ButtonItem} The new instance
*/
class ButtonItem{
static create(typeCheckedProperties? : Partial<ButtonItem>, uncheckedProperties?: Object): ButtonItem;







}

/**
* EditProxy that handles ButtonItem when editMode is enabled.
*/
interface ButtonItemEditProxy extends FormItemEditProxy {


/**
*  Returns the text based on the current component state to be edited inline.
*  Called by the EditProxy.inlineEditForm to obtain the starting edit value.
*  
*  Returns the component's title or name.
* 
* 
*/
getInlineEditText(): void;

/**
*  Save the new value into the component's state. Called by the
*  EditProxy.inlineEditForm to commit the change.
*  
*  Updates the component's defaultValue.
* 
* 
* @param {string} the new component defaultValue
*/
setInlineEditText(newValue:string): void;



}
/**
* Creates a new ButtonItemEditProxy
* 
* @param typeCheckedProperties {Partial<ButtonItemEditProxy>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ButtonItemEditProxy} The new instance
*/
class ButtonItemEditProxy{
static create(typeCheckedProperties? : Partial<ButtonItemEditProxy>, uncheckedProperties?: Object): ButtonItemEditProxy;







}

/**
* The Calendar component provides several different ways for a user to view and
*  edit a set of events. Note that the loadingOptionalModules 
*  must be loaded to make use of the Calendar class.
*  
*  CalendarEvents
*  
*  Events are represented as ordinary JavaScript Objects (see CalendarEvent). 
*  The Calendar expects to be able to read and write a basic set of properties 
*  on events: name, startDate, endDate, description, etc, which can be stored 
*  under configurable property names (see eg Calendar.startDateField).
*  
*  Much like a ListGrid manages it's ListGridRecords, the Calendar can 
*  either be passed an ordinary Array of CalendarEvents or can fetch data from a 
*  DataSource. When this is the case, if the DataSource 
*  does not contain fields with the configured property names, an attempt is made to 
*  auto-detect likely-looking fields from those that are present. To see logs indicating that 
*  this has happened, switch default logging preferences to INFO level in the Developer Console.
*  
*  If the calendar is bound to a DataSource, event changes by user action or by 
*  calling methods will be saved to the DataSource.
*  
*  Navigation
*  
*  The calendar supports a number of views by default: Calendar.dayView,
*  Calendar.weekView, Calendar.monthView and 
*  Calendar.timelineView. The user can navigate using back and forward 
*  buttons or via an attached Calendar.dateChooser.
*  
*  Event Manipulation
*  
*  Events can be created by clicking directly onto one of the views, or via the 
*  Calendar.addEventButton button. In the day, week and timeline views, the user may 
*  click and drag to create an event of a specific duration.
*  
*  Creating an event via click or click and drag pops up the
*  Calendar.eventDialog, which provides a simple form for 
*  quick event entry (for normal events, only the description is required by default - for 
*  events that are shown in a Calendar.lanes, that field is also required).
*  
*  A separate editor called the Calendar.eventEditor provides 
*  an interface for editing all possible properties of an event, including custom 
*  properties. The EventEditor is used whenever a pre-existing event is being 
*  edited, and can also be invoked
*  by the user wherever the simpler EventDialog appears.
*  
*  Events can also be programmatically Calendar.addCalendarEvent,
*  Calendar.removeEvent, or Calendar.updateCalendarEvent.
*/
interface Calendar extends Canvas, DataBoundComponent {
/**
* The date for which events are displayed in the day, week, and month tabs of 
*  the calendar. Default is today.
* @type {Date}
* @default 'Today'
*/
chosenDate?: Date;

/**
* The numeric day (0-6, Sunday-Saturday) which the calendar should consider as the first 
*  day of the week in multi-day views, and in the popup Calendar.dateChooser.
*  
*  If unset, the default is taken from the current locale.
* @type {number}
* @default null
*/
firstDayOfWeek?: number;

/**
* The base name for the CSS class applied to the grid cells of the day and week views
*  of the calendar. This style will have "Dark", "Over", "Selected", or "Disabled"
*  appended to it according to the state of the cell.
*  
*  See cellStyleSuffixes for details on how stateful suffixes are combined with the
*  base style to generate stateful cell styles.
* @type {CSSStyleName}
* @default "calendar"
*/
baseStyle?: CSSStyleName;

/**
* The base name for the CSS class applied to the day headers of the month view.
*  This style will have "Dark", "Over", "Selected", or "Disabled"
*  appended to it according to the state of the cell.
*  
*  See cellStyleSuffixes for details on how stateful suffixes are combined with the
*  base style to generate stateful cell styles.
* @type {CSSStyleName}
* @default "calMonthDayHeader"
*/
dayHeaderBaseStyle?: CSSStyleName;

/**
* The base name for the CSS class applied to the day body of the month view
*  of the calendar. This style will have "Dark", "Over", "Selected", or "Disabled"
*  appended to it according to the state of the cell.
*  
*  See cellStyleSuffixes for details on how stateful suffixes are combined with the
*  base style to generate stateful cell styles.
* @type {CSSStyleName}
* @default "calMonthDayBody"
*/
dayBodyBaseStyle?: CSSStyleName;

/**
* The base name for the CSS class applied to the day headers of the month view.
*  This style will have "Dark", "Over", "Selected", or "Disabled"
*  appended to it according to the state of the cell.
*  
*  See cellStyleSuffixes for details on how stateful suffixes are combined with the
*  base style to generate stateful cell styles.
* @type {CSSStyleName}
* @default "calMonthDayHeader"
*/
otherDayHeaderBaseStyle?: CSSStyleName;

/**
* The base name for the CSS class applied to the day body of the month view
*  of the calendar. This style will have "Dark", "Over", "Selected", or "Disabled"
*  appended to it according to the state of the cell.
*  
*  See cellStyleSuffixes for details on how stateful suffixes are combined with the
*  base style to generate stateful cell styles.
* @type {CSSStyleName}
* @default "calMonthOtherDayBody"
*/
otherDayBodyBaseStyle?: CSSStyleName;

/**
* The CSS style applied to both the header and body of days from other months in the
*  Calendar.monthView, when Calendar.showOtherDays is false.
* @type {CSSStyleName}
* @default "calMonthOtherDayBlank"
*/
otherDayBlankStyle?: CSSStyleName;

/**
* In the Calendar.monthView when Calendar.showDayHeaders is true, this is the minimum
*  height applied to a day cell and its header combined.
*  
*  If showDayHeaders is false, this attribute has no effect - the minimum height 
*  of day cells is either an equal share of the available height, or the rendered height of the
*  cell's HTML content, whichever is greater. If the latter, a vertical scrollbar is shown.
* @type {Integer}
* @default "80"
*/
minimumDayHeight?: Integer;

/**
* Whether to show the Day view.
* @type {boolean}
* @default true
*/
showDayView?: boolean;

/**
* Whether to show the Week view.
* @type {boolean}
* @default true
*/
showWeekView?: boolean;

/**
* Whether to show the Month view.
* @type {boolean}
* @default true
*/
showMonthView?: boolean;

/**
* The base name for the CSS class applied to a cell that is selected via a mouse drag.
* @type {CSSStyleName}
* @default "calendarCellSelected"
*/
selectedCellStyle?: CSSStyleName;

/**
* The base name for the CSS class applied to event windows within calendars.
*  This style will have "Header", "HeaderLabel", and "Body" appended to it, according to 
*  which part of the event window is being styled. For example, to style the header, define
*  a CSS class called 'eventWindowHeader'.
* @deprecated * in favor of Calendar.eventStyleName

* @type {CSSStyleName}
* @default null
*/
eventWindowStyle?: CSSStyleName;

/**
* The base name for the CSS class applied to Calendar.eventCanvas when they're
*  rendered in calendar views.
*  This style will have "Header" and "Body" appended to it, according to 
*  which part of the event window is being styled. For example, to style the header, define
*  a CSS class called 'eventWindowHeader'.
* @type {CSSStyleName}
* @default "eventWindow"
*/
eventStyleName?: CSSStyleName;

/**
* The base name for the CSS class applied to the links rendered by Calendar.getDayBodyHTML.
*  
*  These links are rendered as plain HTML links using A elements, and the CSS style in the 
*  provided skins references the pseudo-classes :link, :visited, :active, :hover. 
*  
*  Even though it goes against the general policy of not exposing the HTML structures SC writes out
*  and not relying on them for styling, applying style to these particular selectors is acceptable,
*  as we're unlikely to use any other kind of HTML structure than a link.
* @type {CSSStyleName}
* @default "calMonthEventLink"
*/
calMonthEventLinkStyle?: CSSStyleName;

/**
* If Calendar.showWorkday is set, this is the style used for cells that are within the workday, 
*  as defined by Calendar.workdayStart and Calendar.workdayEnd, or by a date-specific range 
*  provided in Calendar.getWorkdayStart and Calendar.getWorkdayEnd implementations.
* @type {CSSStyleName}
* @default "calendarWorkday"
*/
workdayBaseStyle?: CSSStyleName;

/**
* When using Calendar.showWorkday:true, workdayStart and workdayEnd
*  specify the time of day when the workday starts and ends, specified as a
*  String acceptable to Time.parseInput.
*  
*  Both start and end time must fall on a 30 minute increment (eg 9:30, but not 9:45).
*  
*  The hours of the workday can be customized for particular dates by providing implementations
*  of Calendar.getWorkdayStart and Calendar.getWorkdayEnd.
* @type {Time}
* @default "9:00am"
*/
workdayStart?: Time;

/**
* When using Calendar.showWorkday:true, workdayStart and workdayEnd
*  specify the time of day when the workday starts and ends, specified as a
*  String acceptable to Time.parseInput.
*  
*  Both start and end time must fall on a 30 minute increment (eg 9:30, but not 9:45).
*  
*  The hours of the workday can be customized for particular dates by providing implementations
*  of Calendar.getWorkdayStart and Calendar.getWorkdayEnd.
* @type {Time}
* @default "5:00pm"
*/
workdayEnd?: Time;

/**
* When set to true, this setting enables various features related to cells that fall within 
*  the workday (as defined by Calendar.workdayStart and Calendar.workdayEnd) in vertical calendar
*  views (Calendar.dayView and Calendar.weekView). Workday cells can be 
*  Calendar.workdayBaseStyle and Calendar.sizeToWorkday,
*  and users can be prevented from scrolling the calendar beyond the 
*  Calendar.limitToWorkday.
*  
*  The hours of the workday can be customized for particular dates by providing implementations
*  of Calendar.getWorkdayStart and Calendar.getWorkdayEnd.
* @type {boolean}
* @default false
*/
showWorkday?: boolean;

/**
* When Calendar.showWorkday is true, applies the Calendar.workdayBaseStyle 
*  to cells that fall within the workday (as defined by Calendar.workdayStart and Calendar.workdayEnd),
*  in both the Calendar.dayView and Calendar.weekView.
*  
*  The hours of the workday can be customized for particular dates by providing implementations
*  of Calendar.getWorkdayStart and Calendar.getWorkdayEnd.
* @type {boolean}
* @default true
*/
styleWorkday?: boolean;

/**
* Array of days that are considered workdays when Calendar.showWorkday is true.
*  Has no effect if Calendar.dateIsWorkday is implemented.
* @type {Array<any>}
* @default "[1,2,3,4,5]"
*/
workdays?: Array<any>;

/**
* If set, and Calendar.showWorkday is true, automatically scrolls the 
*  Calendar.dayView and Calendar.weekView views to the start of the 
*  Calendar.workdayStart when the calendar is first displayed and whenever the user 
*  changes to a different day or week.
* @type {boolean}
* @default false
*/
scrollToWorkday?: boolean;

/**
* When Calendar.showWorkday is true, attempt to resize rows in the 
*  day and week views so that the Calendar.workdayStart fill the visible 
*  viewport-height, and the whole workday is visible without scrolling. If the Calendar is 
*  resized, the row-size is recalculated to keep the workday hours visible.
*  
*  Note that row-heights will not shrink below the Calendar.minRowHeight, so the entire workday 
*  may not be visible without scrolling if the workday is long or the viewport-height is 
*  insufficient.
* @type {boolean}
* @default false
*/
sizeToWorkday?: boolean;

/**
* When Calendar.showWorkday is true, this attribute prevents the user from 
*  scrolling vertical views beyond the specified workday Calendar.workdayStart and 
*  Calendar.workdayEnd hours.
* @type {boolean}
* @default false
*/
limitToWorkday?: boolean;

/**
* The number of minutes per row in Calendar.dayView and 
*  Calendar.weekView views. The default of 30 minutes shows two rows per hour.
*  Note that this value must divide into 60.
* @type {Integer}
* @default "30"
*/
minutesPerRow?: Integer;

/**
* A minute value that indicates which rows should show times in vertical views, like 
*  Calendar.dayView and Calendar.weekView. The default of 60 minutes 
*  shows titles on the first row of each hour. The value provided must be a multiple of 
*  Calendar.minutesPerRow and be no larger than 60.
* @type {Integer}
* @default "60"
*/
rowTitleFrequency?: Integer;

/**
* The name of the name field on a CalendarEvent.
* @type {string}
* @default "name"
*/
nameField?: string;

/**
* The name of the description field on a CalendarEvent.
* @type {string}
* @default "description"
*/
descriptionField?: string;

/**
* The name of the start date field on a CalendarEvent.
* @type {string}
* @default "startDate"
*/
startDateField?: string;

/**
* The name of the end date field on a CalendarEvent.
* @type {string}
* @default "endDate"
*/
endDateField?: string;

/**
* The name of the CalendarEvent.duration field on a CalendarEvent.
* @type {string}
* @default "duration"
*/
durationField?: string;

/**
* The name of the CalendarEvent.durationUnit field on a 
*  CalendarEvent.
* @type {string}
* @default "durationUnit"
*/
durationUnitField?: string;

/**
* The name of the field which will determine the Calendar.lanes in which this 
*  event will be displayed in Timelines and in the Calendar.dayView, if 
*  Calendar.showDayLanes is true.
* @type {string}
* @default "lane"
*/
laneNameField?: string;

/**
* When set to true, hides any Calendar.lanes that doesn't have any active events
*  in the current dataset.
* @type {boolean}
* @default null
*/
hideUnusedLanes?: boolean;

/**
* When set to true, causes Calendar.lanes to be sub-divided according to their
*  set of Lane.sublanes.
* @type {boolean}
* @default null
*/
useSublanes?: boolean;

/**
* The name of the field which will determine the Lane.sublanes in which this 
*  event will be displayed, within its parent Lane, in Timelines and in the 
*  Calendar.dayView, if Calendar.showDayLanes is true.
* @type {string}
* @default "sublane"
*/
sublaneNameField?: string;

/**
* The name of the leading date field for each event. When this attribute and 
*  Calendar.trailingDateField are present in the data, a line extends out from the event showing
*  the extent of the leading and trailing dates - useful for visualizing a pipeline of events 
*  where some can be moved a certain amount without affecting others.
* @type {string}
* @default "leadingDate"
*/
leadingDateField?: string;

/**
* The name of the trailing date field for each event. When this attribute and 
*  Calendar.leadingDateField are present in the data, a line extends out from the event showing 
*  the extent of the leading and trailing dates - useful for visualizing a pipeline of events 
*  where some can be moved a certain amount without affecting others.
* @type {string}
* @default "trailingDate"
*/
trailingDateField?: string;

/**
* When set to false, hides the frozen Label-Column in vertical CalendarViews.
*  
*  Always false when Calendar.showColumnLayouts is true.
* @type {boolean}
* @default true
*/
showLabelColumn?: boolean;

/**
* The name of the field used to override Calendar.eventWindowStyle for an individual
*  CalendarEvent. See CalendarEvent.eventWindowStyle.
* @deprecated * in favor of Calendar.eventStyleNameField

* @type {string}
* @default "eventWindowStyle"
*/
eventWindowStyleField?: string;

/**
* The name of the field used to override Calendar.eventStyleName for an individual
*  CalendarEvent.styleName.
* @type {string}
* @default "styleName"
*/
eventStyleNameField?: string;

/**
* Name of the field on each CalendarEvent that determines whether it can be edited in
*  the Calendar.eventEditor. Note that an event with canEdit 
*  set to true can also have Calendar.canDragEventField or
*  Calendar.canResizeEventField set to false, 
*  which would still allow editing, but not via drag operations.
* @type {string}
* @default "canEdit"
*/
canEditField?: string;

/**
* Name of the field on each CalendarEvent that determines whether that event can be 
*  moved between lanes.
* @type {string}
* @default "canEditLane"
*/
canEditLaneField?: string;

/**
* Name of the field on each CalendarEvent that determines whether that event can be 
*  moved between individual Lane.sublanes in a Lane.
* @type {string}
* @default "canEditSublane"
*/
canEditSublaneField?: string;

/**
* Name of the field on each CalendarEvent that determines whether an event shows a
*  remove button.
* @type {string}
* @default "canRemove"
*/
canRemoveField?: string;

/**
* Name of the field on each CalendarEvent that determines whether an EventCanvas
*  can be moved or resized by dragging with the mouse. Note that 
*  Calendar.canEditEvents must be true for dragging to be allowed.
* @type {string}
* @default "canDrag"
*/
canDragEventField?: string;

/**
* Name of the field on each CalendarEvent that determines whether an event can be 
*  resized by dragging.
* @type {string}
* @default "canResize"
*/
canResizeEventField?: string;

/**
* When set to true, allows events to be managed by duration, as well as by end date. Values
*  can be set for CalendarEvent.duration and 
*  CalendarEvent.durationUnit on each event, and are then maintained,
*  instead of the end date, when alterations are made to the event via editors or dragging 
*  with the mouse.
* @type {boolean}
* @default null
*/
allowDurationEvents?: boolean;

/**
* The pixel space to leave between events and the edges of the Calendar.lanes or 
*  Lane.sublanes they appear in. Only applicable to 
*  Calendar.timelineView and to Calendar.dayView showing
*  Calendar.showDayLanes.
* @type {Integer}
* @default "0"
*/
laneEventPadding?: Integer;

/**
* The number of pixels to leave to the right of events so overlapping events can still be 
*  added using the mouse.
* @type {Integer}
* @default "10"
*/
eventDragGap?: Integer;

/**
* Augments the width of week event windows slightly to avoid duplicate adjacent borders
*  between events.
* @type {boolean}
* @default false
*/
weekEventBorderOverlap?: boolean;

/**
* Configures the levels of HeaderLevel shown above the event area, and 
*  their time units.
*  
*  Header levels are provided from the top down, so the first header level should be the largest 
*  time unit and the last one the smallest. The smallest is then used for the actual 
*  field-headers.
* @type {Array<Partial<HeaderLevel>>}
* @default null
*/
headerLevels?: Array<Partial<HeaderLevel>>;

/**
* The number of minutes that determines the positions to which events will snap when rendered,
*  and when moved or resized with the mouse.
*  
*  If unset (the default), all views will snap to each cell boundary; 30 minutes in a default
*  vertical view, or one Calendar.timelineGranularity in a default Timeline.
*  
*  If set to zero, views will snap to one of a set of known "sensible" defaults: for a default 
*  vertical, this will be 5 minutes. For timelines, the eventSnapGap is automatic depending on
*  the current Calendar.timelineGranularity. If Calendar.timelineUnitsPerColumn 
*  is greater than 1, the snapGap is set to one unit of the current granularity. So, a 
*  cell-resolution of 15 minutes would snap to every minute, assuming there are at least 15 
*  pixels per column. Otherwise, the snapGap is either 15 minutes, 1 hour, one day or one 
*  month, depending on granularity.
*  
*  If any other value is specified, it is used where possible.
*  
*  If the specified or calculated value is less than the time covered by a single pixel in the 
*  current view, then it can't be represented. In this case, it is rounded up to the lowest of 
*  a set of "sensible" time-spans that can be represented: one of 
*  [1, 5, 10, 15, 20, 30, 60, 120, 240, 360, 480, 720, 1440].
*  
*  For example - a Timeline showing "day" columns cannot support an eventSnapGap of 1 minute,
*  unless each column is at least 1440 pixels wide - if the columns were only 150px wide, then 
*  each pixel would represent around 9.6 minutes, which would result in unpleasant and unexpected 
*  time-offsets when dragging events. So, the calculated eventSnapGap will be rounded 
*  up to the nearest "sensible" time-span - in this case, 10 minutes. If the columns were only 
*  60px wide, it would be 30 minutes.
* @type {Integer}
* @default null
*/
eventSnapGap?: Integer;

/**
* Determines whether the quick event dialog is displayed when a time is clicked. If this is
*  false, the full event editor is displayed.
* @type {boolean}
* @default true
*/
showQuickEventDialog?: boolean;

/**
* The set of fields for the Calendar.eventEditor.
*  
*  The default set of fields are: 
*  
*   {name: "startHours", title: "From",   editorType: "SelectItem", type: "integer", width: 60},
*   {name: "startMinutes", showTitle: false, editorType: "SelectItem", type: "integer", width: 60},
*   {name: "startAMPM", showTitle: false, type: "select", width: 60},
*   {name: "invalidDate", type: "blurb", colSpan: 4, visible: false}
*   {name: "endHours", title: "To",    editorType: "SelectItem", type: "integer", width: 60},
*   {name: "endMinutes", showTitle: false, editorType: "SelectItem", type: "integer", width: 60},
*   {name: "endAMPM", showTitle: false, type: "select", width: 60},
*   {name: "name", title: "Name", type: "text", colSpan: 4},
*   {name: "description", title: "Description", type: "textArea", colSpan: 4, height: 50}
*  
*  See the Customized Binding example below for more information on altering default datasource 
*  fields within forms.
* @type {Array<Partial<FormItem>>}
* @default "see below"
*/
eventEditorFields?: Array<Partial<FormItem>>;

/**
* The set of fields for the Calendar.eventDialog. 
*  
*  The default set of fields are: 
*  
*   {name: "name", title: "Event Name", type: nameType, width: 250 },
*   {name: "save", title: "Save Event", editorType: "SubmitItem", endRow: false},
*   {name: "details", title: "Edit Details", type: "button", startRow: false}
*  
*  See the Customized Binding example below for more information on altering default datasource 
*  fields within forms.
* @type {Array<Partial<FormItem>>}
* @default "see below"
*/
eventDialogFields?: Array<Partial<FormItem>>;

/**
* If true, users can create new events.
* @type {boolean}
* @default true
*/
canCreateEvents?: boolean;

/**
* If true, users can edit existing events.
* @type {boolean}
* @default true
*/
canEditEvents?: boolean;

/**
* If true, users can delete existing events. Defaults to Calendar.canEditEvents.
* @deprecated * in favor of Calendar.canRemoveEvents

* @type {boolean}
* @default null
*/
canDeleteEvents?: boolean;

/**
* If true, users can remove existing events. Defaults to Calendar.canEditEvents.
* @type {boolean}
* @default true
*/
canRemoveEvents?: boolean;

/**
* A boolean value controlling whether users can drag-reposition events. By default, this is 
*  false for Touch devices, where drag gestures scroll the view, and true otherwise.
*  
*  Only has an effect when Calendar.canEditEvents is true.
* @type {boolean}
* @default null
*/
canDragEvents?: boolean;

/**
* Can CalendarEvent be resized by dragging appropriate edges of the 
*  EventCanvas.verticalResize? Only has an effect when both
*  Calendar.canEditEvents and Calendar.canDragEvents
*  are true. Set this attribute to false to disallow drag-resizing.
*  
*  Always false when Calendar.showColumnLayouts is true.
* @type {boolean}
* @default true
*/
canResizeEvents?: boolean;

/**
* Determines whether the Calendar.dateChooser is displayed.
* @type {boolean}
* @default true
*/
showDateChooser?: boolean;

/**
* If true, weekend days appear in a disabled style and events cannot be created on weekends.
*  Which days are considered weekends is controlled by Calendar.weekendDays.
* @type {boolean}
* @default true
*/
disableWeekends?: boolean;

/**
* An array of integer day-numbers that should be considered to be weekend days by this
*  Calendar instance. If unset, defaults to the set of days indicated 
*  DateUtil.weekendDays.
* @type {Array<Partial<Integer>>}
* @default null
*/
weekendDays?: Array<Partial<Integer>>;

/**
* Suppresses the display of weekend days in the Calendar.weekView, 
*  Calendar.monthView and Calendar.timelineView views, and 
*  disallows the creation of events on weekends. Which days are considered weekends is 
*  controlled by Calendar.weekendDays.
* @type {boolean}
* @default true
*/
showWeekends?: boolean;

/**
* If true, the default, show a header cell for each day cell in the 
*  Calendar.monthView, with both cells having a minimum combined height of 
*  Calendar.minimumDayHeight. If false, the header cells will not be shown, and the value 
*  of Calendar.minimumDayHeight is ignored. This causes the available vertical space in month
*  views to be shared equally between day cells, such that no vertical scrollbar is required
*  unless the HTML in the cells renders them taller than will fit.
* @type {boolean}
* @default true
*/
showDayHeaders?: boolean;

/**
* If set to true, in the Calendar.monthView, days that fall in an adjacent month are
*  still shown with a header and body area, and are interactive. Otherwise days from other 
*  months are rendered in the Calendar.otherDayBlankStyle and are non-interactive.
* @type {boolean}
* @default true
*/
showOtherDays?: boolean;

/**
* When true, shows the current Calendar.chosenDate in a selected style
*  in the Calendar.monthView Has no effect in other views.
* @type {boolean}
* @default true
*/
selectChosenDate?: boolean;

/**
* If set to true, enables the auto-arrangement of events that share time in the calendar. The
*  default is true.
* @type {boolean}
* @default true
*/
eventAutoArrange?: boolean;

/**
* If set to true, clicking an event will bring it to the front of the zorder.
* @type {boolean}
* @default null
*/
bringEventsToFront?: boolean;

/**
* When Calendar.eventAutoArrange is true, setting eventOverlap to true causes events that 
*  share timeslots to overlap each other by a percentage of their width, specified by 
*  Calendar.eventOverlapPercent. The default is true for Calendars and false for 
*  Timeline.
* @type {boolean}
* @default true
*/
eventOverlap?: boolean;

/**
* The size of the overlap, presented as a percentage of the width of events sharing timeslots.
* @type {number}
* @default "10"
*/
eventOverlapPercent?: number;

/**
* When set to true, events that start at the same time will not overlap each other to prevent 
*  events having their close button hidden.
* @type {boolean}
* @default false
*/
eventOverlapIdenticalStartTimes?: boolean;

/**
* A boolean value controlling whether the Calendar shows tabs for available calendar views.
*  By default, this is true for handsets and false otherwise.
* @type {boolean}
* @default false
*/
minimalUI?: boolean;

/**
* A boolean value controlling whether new events of varying length can be created by dragging
*  the cursor. By default, this is false for Touch devices and true otherwise.
* @type {boolean}
* @default null
*/
canDragCreateEvents?: boolean;

/**
* TabSet for managing calendar views when multiple views are available (eg,
*  Calendar.dayView and Calendar.monthView).
* @type {TabSet}
* @default null
*/
mainView?: TabSet;

/**
* CalendarView used to display events that pertain to a given day.
* @type {CalendarView}
* @default null
*/
dayView?: CalendarView;

/**
* CalendarView used to display events that pertain to a given week.
* @type {CalendarView}
* @default null
*/
weekView?: CalendarView;

/**
* CalendarView used to display events that pertain to a given month.
* @type {CalendarView}
* @default null
*/
monthView?: CalendarView;

/**
* DateChooser used to select the date for which events will be displayed.
* @type {DateChooser}
* @default null
*/
dateChooser?: DateChooser;

/**
* When showing a Calendar.timelineView, or a Calendar.dayView when 
*  Calendar.showDayLanes is true, whether to make lane boundaries more obvious by showing
*  alternate lanes in a different color.
* @type {boolean}
* @default null
*/
alternateLaneStyles?: boolean;

/**
* When set to true, causes Calendar.timelineView to highlight the Lane under the 
*  mouse with the "Over" style.
* @type {boolean}
* @default null
*/
showLaneRollOver?: boolean;

/**
* A List of CalendarEvent objects, specifying the data to be used to populate the
*  calendar. 
*  
*  This property will typically not be explicitly specified for databound Calendars, where
*  the data is returned from the server via databound component methods such as
*  Calendar.fetchData. In this case the data objects will be set to a 
*  ResultSet rather than a simple array.
* @type {Array<Partial<CalendarEvent>>}
* @default null
*/
data?: Array<Partial<CalendarEvent>>;

/**
* The DataSource that this component should bind to for default fields and for performing
*  DSRequest.
*  
*  Can be specified as either a DataSource instance or the String ID of a DataSource.
* @type {DataSource | GlobalId}
* @default null
*/
dataSource?: DataSource | GlobalId;

/**
* If true, when this component is first drawn, automatically call this.fetchData().
*  Criteria for this fetch may be picked up from Calendar.initialCriteria, and textMatchStyle may
*  be specified via ListGrid.autoFetchTextMatchStyle.
*  
*  NOTE: if autoFetchData is set, calling
*  ListGrid.fetchData before draw will cause two requests to be issued, one from the manual
*  call to fetchData() and one from the autoFetchData setting. The second request will use
*  only Calendar.initialCriteria and not any other criteria or settings from the first request.
*  Generally, turn off autoFetchData if you are going to manually call ListGrid.fetchData at any time.
* @type {boolean}
* @default false
*/
autoFetchData?: boolean;

/**
* If Calendar.autoFetchData is true, this attribute allows the developer to
*  specify a textMatchStyle for the initial ListGrid.fetchData call.
* @type {TextMatchStyle}
* @default null
*/
autoFetchTextMatchStyle?: TextMatchStyle;

/**
* Criteria to be used when Calendar.autoFetchData is set.
*  
*  This property supports dynamicCriteria - use Criterion.valuePath
*  to refer to values in the Canvas.ruleScope.
* @type {Criteria}
* @default null
*/
initialCriteria?: Criteria;

/**
* Criteria that are never shown to or edited by the user and are cumulative with any criteria
*  provided via DataBoundComponent.initialCriteria and related methods
* @type {Criteria}
* @default null
*/
implicitCriteria?: Criteria;

/**
* Whether to show fields marked detail:true when a DataBoundComponent is 
*  given a DataSource but no component.fields.
*  
*  The detail property is used on DataSource fields to mark fields that 
*  shouldn't appear by default in a view that tries to show many records in a small space.
* @type {boolean}
* @default true
*/
showDetailFields?: boolean;

/**
* How to fetch and manage records retrieve from the server. See FetchMode.
*  
*  This setting only applies to the ResultSet automatically created by calling
*  ListGrid.fetchData. If a pre-existing ResultSet is passed to setData() instead, it's
*  existing setting for ResultSet.fetchMode applies.
* @type {FetchMode}
* @default "paged"
*/
dataFetchMode?: FetchMode;

/**
* If set to true, show the Calendar.timelineView.
* @type {boolean}
* @default false
*/
showTimelineView?: boolean;

/**
* CalendarView used to display events in lanes in a horizontal Timeline view.
* @type {CalendarView}
* @default null
*/
timelineView?: CalendarView;

/**
* When set to true, the default, each EventCanvas is rendered as it 
*  appears in the viewport. If set to false, all events are rendered up-front, whenever the 
*  current range changes.
*  
*  Has no effect when Calendar.showColumnLayouts is true.
* @type {boolean}
* @default true
*/
renderEventsOnDemand?: boolean;

/**
* The granularity in which the Calendar.timelineView will display events.
*  Possible values are those available in the built-in TimeUnit type.
* @type {TimeUnit}
* @default "day"
*/
timelineGranularity?: TimeUnit;

/**
* How many units of Calendar.timelineGranularity each cell represents.
* @type {Integer}
* @default "1"
*/
timelineUnitsPerColumn?: Integer;

/**
* Can Timeline events be stretched by their left and right edges?
* @deprecated * in favor of Calendar.canResizeEvents;

* @type {boolean}
* @default false
*/
canResizeTimelineEvents?: boolean;

/**
* Can events be moved between lanes? If so, the event can be dragged to a different 
*  Calendar.lanes, and the event Calendar.eventDialog and 
*  Calendar.eventEditor allow a lane to be selected with a drop-down chooser.
*  
*  In either case, the event's Calendar.laneNameField is updated automatically.
*  
*  If set to false, cross-lane dragging is disallowed and drop-down Lane-choosers are disabled
*  when editing existng events. When creating Calendar.canCreateEvents, 
*  the Lane-chooser remains enabled so an initial Lane can be selected.
*  
*  This setting can be overridden on each CalendarEvent.canEditLane.
* @type {boolean}
* @default null
*/
canEditLane?: boolean;

/**
* Can events be moved between sublanes?
*  
*  If so, the event can be dragged to a different Lane.sublanes within the same
*  parent Lane and, when it's editor is shown, an additional drop-down widget is provided 
*  allowing the sublane to be altered.
*  
*  If the sublane is locked, but the Calendar.canEditLane isn't, an update
*  to the event's Calendar.laneNameField will be allowed, assuming that the
*  new Lane has an existing sublane with the same name.
*  
*  In either case, the event's Calendar.sublaneNameField is updated 
*  automatically.
*  
*  This setting can be overridden on each CalendarEvent.canEditSublane.
* @type {boolean}
* @default null
*/
canEditSublane?: boolean;

/**
* If true, lanes can be reordered by dragging their Calendar.laneFields
*  with the mouse.
* @type {boolean}
* @default null
*/
canReorderLanes?: boolean;

/**
* The start date of the calendar Timeline. Has no effect in 
*  other views. If not specified, defaults to a timeline starting from the beginning 
*  of the current Calendar.timelineGranularity and spanning
*  Calendar.defaultTimelineColumnSpan columns of that granularity. 
*  
*  To set different start and Calendar.endDate dates after initial draw,
*  see Calendar.setTimelineRange.
*  
*  Note that the value you provide may be automatically altered if showing 
*  Calendar.headerLevels, to fit to header boundaries.
* @type {Date}
* @default null
*/
startDate?: Date;

/**
* The number of columns of the Calendar.timelineGranularity to
*  give the timeline by default if no Calendar.endDate is provided. The 
*  default is 20.
* @type {number}
* @default "20"
*/
defaultTimelineColumnSpan?: number;

/**
* When using the Next and Previous arrows to scroll a Timeline, this is the number of columns 
*  of the Calendar.timelineGranularity to scroll by. With the
*  default value of null, the Timeline will scroll by its current length.
* @type {number}
* @default null
*/
columnsPerPage?: number;

/**
* The end date of the calendar timeline view. Has no effect in other views.
*  
*  To set different Calendar.startDate and end dates after initial draw,
*  see Calendar.setTimelineRange.
*  
*  Note that the value you provide may be automatically altered if showing 
*  Calendar.headerLevels, to fit to header boundaries.
* @type {Date}
* @default null
*/
endDate?: Date;

/**
* The text to appear before the week number in the title of TimeUnit 
*  HeaderLevels when this calendar is showing a timeline.
* @type {HTMLString}
* @default "Week"
*/
weekPrefix?: HTMLString;

/**
* Indicates the type of controls to use in event-windows. Valid values are those in the 
*  DateEditingStyle type.
*  
*  If unset, the editing style will be set to the field-type on the DataSource, if there is one.
*  If there's no DataSource, it will be set to "date" if the 
*  Calendar.timelineGranularity is "day" or larger and "time" if granularity 
*  is "minute" or smaller, otherwise "datetime".
* @type {DateEditingStyle}
* @default null
*/
dateEditingStyle?: DateEditingStyle;

/**
* If true, allows the lanes in a Timeline to be grouped by providing a value for 
*  Calendar.laneGroupByField. The fields available for grouping on 
*  are those defined as Calendar.laneFields. Since these are definitions
*  for ListGridField, you can choose to ListGridField.showIf
*  the field in the timeline, but still have it available for grouping.
* @type {boolean}
* @default null
*/
canGroupLanes?: boolean;

/**
* For timelines with Calendar.canGroupLanes set to true, this is a 
*  field name or array of field names on which to group the lanes in a timeline.
* @type {string | Array<Partial<string>>}
* @default null
*/
laneGroupByField?: string | Array<Partial<string>>;

/**
* Describes the default state of lane groups in timelines when 
*  Calendar.groupLanesBy is called.
* 
*  Possible values are:
*  
*  
* - "all": open all groups
*  
* - "first": open the first group
*  
* - "none": start with all groups closed
*  
* - Array of values that should be opened
*  
* @type {GroupStartOpen | Array<any>}
* @default "first"
*/
laneGroupStartOpen?: GroupStartOpen | Array<any>;

/**
* An array of Lane definitions that represent the rows of the Calendar.timelineView, or
*  the columns of the Calendar.dayView if Calendar.showDayLanes is true.
* @type {Array<Partial<Lane>>}
* @default null
*/
lanes?: Array<Partial<Lane>>;

/**
* Field definitions for the frozen area of the Calendar.timelineView, which shows data about the
*  timeline Calendar.lanes. Each field shows one attribute of the objects provided as
*  Calendar.lanes.
*  
*  When Calendar.canGroupLanes is enabled, only fields that are specified
*  as lane fields can be used as group fields.
* @type {Array<Partial<ListGridField>>}
* @default null
*/
laneFields?: Array<Partial<ListGridField>>;

/**
* If set to true, the Calendar.dayView uses Calendar.lanes to render multiple
*  vertical "lanes" within the day, very much like a vertical Timeline.
*  
*  Day lanes are useful for showing events for various entities on the same day - agendas for 
*  various staff members, for example, or delivery schedules for a fleet of trucks.
*  
*  Each day lane is self-contained, showing in a column with a header and individual events 
*  are placed in CalendarEvent.lane, respecting padding and 
*  overlapping. If Calendar.canEditEvents is true, events can be drag-moved or drag-resized 
*  from their top and bottom edges, within the containing lane. To allow events to be dragged 
*  from one lane into another, see Calendar.canEditLane.
* @type {boolean}
* @default null
*/
showDayLanes?: boolean;

/**
* When showing Calendar.showDayLanes in the Calendar.dayView, this attribute sets 
*  the minimum width of each column or field.
* @type {Integer}
* @default null
*/
minLaneWidth?: Integer;

/**
* A set of SortSpecifier for customizing the render order of events
*  that overlap.
*  
*  In Timeline, this dictates the vertical rendering order of 
*  overlapped events in each Lane.
*  
*  In Calendar.dayView and Calendar.weekView views, it dictates the
*  horizontal rendering order of overlapped events in each column or Lane.
*  
*  By default, events that share space in a Lane or column are rendered from top to bottom,
*  or left to right according to their Calendar.startDateField - the earliest in a 
*  given lane appears top-most in that lane, or left-most in its column.
*  
*  Providing overlapSortSpecifiers allows for the events to be ordered by one or
*  more of the fields stored on the events, or in the underlying DataSource,
*  if the Calendar is databound.
* @type {Array<Partial<SortSpecifier>>}
* @default null
*/
overlapSortSpecifiers?: Array<Partial<SortSpecifier>>;

/**
* The background color for cells that represent today in all CalendarViews.
* @type {string}
* @default null
*/
todayBackgroundColor?: string;

/**
* When rendering the Calendar.eventCanvas for an event, whether to show the 
*  EventCanvas.showBody, typically containing brief details of the event - 
*  Calendar.getEventBodyHTML, 
*  Calendar.descriptionField.
*  
*  The default is true - if set to false, the event's EventCanvas.showHeader 
*  will fill the canvas.
* @type {boolean}
* @default true
*/
showEventDescriptions?: boolean;

/**
* When rendering the Calendar.eventCanvas for an event, whether to show the 
*  EventCanvas.showHeader, typically containing suitable title text - 
*  Calendar.getEventHeaderHTML, the event's Calendar.nameField.
*  
*  The default is true - if set to false, the event's EventCanvas.showBody 
*  will fill the canvas.
* @type {boolean}
* @default true
*/
showEventHeaders?: boolean;

/**
* When rendering the Calendar.eventCanvas for an event, whether to allow the 
*  content of the EventCanvas.showHeader to wrap to multiple lines.
*  
*  The default is true - if set to false, the header area is 
*  Calendar.eventHeaderHeight, unless Calendar.showEventDescriptions is
*  false, in which case the header area fills the canvas.
* @type {boolean}
* @default true
*/
eventHeaderWrap?: boolean;

/**
* When Calendar.eventHeaderWrap is false and 
*  Calendar.showEventDescriptions is true, this is the fixed 
*  height for the EventCanvas.showHeader in event canvases.
* @type {Integer}
* @default "14"
*/
eventHeaderHeight?: Integer;

/**
* If true, events will be sized to the grid, even if they start and/or end at times
*  between grid cells.
* @type {boolean}
* @default true
*/
sizeEventsToGrid?: boolean;

/**
* The title for the Calendar.dayView.
* @type {string}
* @default "Day"
*/
dayViewTitle?: string;

/**
* The title for the Calendar.weekView.
* @type {string}
* @default "Week"
*/
weekViewTitle?: string;

/**
* The title for the Calendar.monthView.
* @type {string}
* @default "Month"
*/
monthViewTitle?: string;

/**
* The title for the Calendar.timelineView.
* @type {string}
* @default "Timeline"
*/
timelineViewTitle?: string;

/**
* The title for the Calendar.nameField in the quick
*  Calendar.eventDialog and the detailed 
*  Calendar.eventEditor.
* @type {HTMLString}
* @default "Event Name"
*/
eventNameFieldTitle?: HTMLString;

/**
* The title for the Calendar.startDateField in the quick
*  Calendar.eventDialog and the detailed 
*  Calendar.eventEditor.
* @type {HTMLString}
* @default "From"
*/
eventStartDateFieldTitle?: HTMLString;

/**
* The title for the Calendar.endDateField in the quick
*  Calendar.eventDialog and the detailed 
*  Calendar.eventEditor.
* @type {HTMLString}
* @default "To"
*/
eventEndDateFieldTitle?: HTMLString;

/**
* The title for the Calendar.descriptionField field in the quick 
*  Calendar.eventDialog and the detailed 
*  Calendar.eventEditor.
* @type {HTMLString}
* @default "Description"
*/
eventDescriptionFieldTitle?: HTMLString;

/**
* The title for the Calendar.laneNameField in the quick 
*  Calendar.eventDialog and the detailed 
*  Calendar.eventEditor.
* @type {HTMLString}
* @default "Lane"
*/
eventLaneFieldTitle?: HTMLString;

/**
* The title for the Calendar.sublaneNameField in the quick 
*  Calendar.eventDialog and the detailed 
*  Calendar.eventEditor.
* @type {HTMLString}
* @default "Sublane"
*/
eventSublaneFieldTitle?: HTMLString;

/**
* The title for the Calendar.durationField in the quick
*  Calendar.eventDialog and the detailed 
*  Calendar.eventEditor.
* @type {HTMLString}
* @default "Duration"
*/
eventDurationFieldTitle?: HTMLString;

/**
* The title for the Calendar.durationUnitField in the quick
*  Calendar.eventDialog and the detailed 
*  Calendar.eventEditor.
* @type {HTMLString}
* @default "&amp;nbsp;"
*/
eventDurationUnitFieldTitle?: HTMLString;

/**
* The title for the Calendar.saveButton in the 
*  Calendar.eventDialog and the 
*  Calendar.eventEditor.
* @type {HTMLString}
* @default "Save Event"
*/
saveButtonTitle?: HTMLString;

/**
* The title for the edit button in the quick Calendar.eventDialog.
* @type {HTMLString}
* @default "Edit Details"
*/
detailsButtonTitle?: HTMLString;

/**
* The title for the Calendar.removeButton in the
*  Calendar.eventEditor.
* @type {HTMLString}
* @default "Remove Event"
*/
removeButtonTitle?: HTMLString;

/**
* The title for the Calendar.cancelButton in the
*  Calendar.eventEditor.
* @type {HTMLString}
* @default "Cancel"
*/
cancelButtonTitle?: HTMLString;

/**
* The title of the Calendar.monthButton, used for showing and hiding the
*  Calendar.monthView on Handsets.
*  
*  This is a dynamic string - text within &#36;{...} are dynamic variables and will
*  be evaluated as JS code when the message is displayed.
*  
*  Only one dynamic variable, monthName, is available and represents the name of the month 
*  containing the currently selected date.
*  
*  The default value is a left-facing arrow followed by the Month-name of the selected date.
*  
*  When the month view is already visible, the title for the month button is set according to
*  the value of Calendar.backButtonTitle.
* @type {HTMLString}
* @default "&amp;lt; ${monthName}"
*/
monthButtonTitle?: HTMLString;

/**
* The title of the link shown in a cell of a Calendar.monthView when there
*  are too many events to be displayed at once.
*  
*  This is a dynamic string - text within &#36;{...} are dynamic variables and will 
*  be evaluated as JS code when the message is displayed.
*  
*  Only one dynamic variable, eventCount, is available and represents the number of events that
*  are not currently displayed and that will appear in the menu displayed when the More Events 
*  link is clicked.
*  
*  The default value is a string like "+ 3 more...".
* @type {HTMLString}
* @default "+ ${eventCount} more..."
*/
monthMoreEventsLinkTitle?: HTMLString;

/**
* The title of the Calendar.monthButton on Handsets when the 
*  Calendar.monthView is the current visible view. 
*  
*  When the month view is not the current visible view, the title for the month button is set 
*  according to the value of Calendar.monthButtonTitle.
* @type {HTMLString}
* @default "Back"
*/
backButtonTitle?: HTMLString;

/**
* The text to be displayed when a user hovers over the Calendar.previousButton
*  toolbar button.
* @type {string}
* @default "Previous"
*/
previousButtonHoverText?: string;

/**
* The text to be displayed when a user hovers over the Calendar.nextButton 
*  toolbar button
* @type {string}
* @default "Next"
*/
nextButtonHoverText?: string;

/**
* The text to be displayed when a user hovers over the Calendar.addEventButton
*  toolbar button
* @type {string}
* @default "Add an event"
*/
addEventButtonHoverText?: string;

/**
* The text to be displayed when a user hovers over the Calendar.datePickerButton
*  toolbar button
* @type {string}
* @default "Choose a date"
*/
datePickerHoverText?: string;

/**
* The message to display in the Calendar.eventEditor when the 'To' date is greater than
*  the 'From' date and a save is attempted.
* @type {string}
* @default "From must be before To"
*/
invalidDateMessage?: string;

/**
* An AutoChild of type Window that displays a quick event entry form in a 
*  popup window.
* @type {Window}
* @default null
*/
eventDialog?: Window;

/**
* An AutoChild of type Window that displays the full 
*  Calendar.eventEditor
* @type {Window}
* @default null
*/
eventEditorLayout?: Window;

/**
* An AutoChild of type DynamicForm which displays CalendarEvent. 
*  This form is created within the Calendar.eventEditorLayout
* @type {DynamicForm}
* @default null
*/
eventEditor?: DynamicForm;

/**
* An AutoChild of type HLayout which houses the 
*  Calendar.saveButton, Calendar.removeButton
*  and Calendar.cancelButton buttons in the 
*  Calendar.eventEditor.
* @type {HLayout}
* @default null
*/
eventEditorButtonLayout?: HLayout;

/**
* An AutoChild of type IButton, used to save an event from the 
*  Calendar.eventEditor.
* @type {IButton}
* @default null
*/
saveButton?: IButton;

/**
* An AutoChild of type IButton, used to permanently remove an event from the 
*  Calendar.eventEditor.
* @type {IButton}
* @default null
*/
removeButton?: IButton;

/**
* An AutoChild of type IButton, used to cancel editing of an event and close the 
*  Calendar.eventEditor.
* @type {IButton}
* @default null
*/
cancelButton?: IButton;

/**
* Set to false to hide the Calendar.addEventButton button.
* @type {boolean}
* @default null
*/
showAddEventButton?: boolean;

/**
* An ImgButton that appears in a Calendar's week/day/month views and offers an 
*  alternative way to create a new CalendarEvent.
* @type {ImgButton}
* @default null
*/
addEventButton?: ImgButton;

/**
* Set to false to hide the Calendar.datePickerButton that allows selecting a new base date for 
*  this Calendar.
* @type {boolean}
* @default null
*/
showDatePickerButton?: boolean;

/**
* An ImgButton that appears above the various views of the
*  calendar and offers alternative access to a DateChooser to pick the current day.
* @type {ImgButton}
* @default null
*/
datePickerButton?: ImgButton;

/**
* If false the controls bar at the top of the calendar will not be displayed - this means 
*  that the Calendar.controlsBar will be hidden, so the autoChildren (Calendar.previousButton, 
*  Calendar.dateLabel, Calendar.nextButton, Calendar.addEventButton, and Calendar.datePickerButton) 
*  will not be created or shown.
* @type {boolean}
* @default true
*/
showControlsBar?: boolean;

/**
* An HLayout shown above the Calendar views and displaying a set of
*  controls for interacting with the current view - namely, the Calendar.nextButton,
*  Calendar.previousButton and Calendar.addEventButton buttons, 
*  the Calendar.dateLabel and the Calendar.datePickerButton icon.
* @type {HLayout}
* @default null
*/
controlsBar?: HLayout;

/**
* Set to false to prevent the Calendar.monthButton button from displaying on Handset 
*  devices.
* @type {boolean}
* @default null
*/
showMonthButton?: boolean;

/**
* A NavigationButton that appears to the left of other navigation controls in the 
*  Calendar.controlsBar on Handset devices.
*  
*  Used to show and hide the Calendar.monthView on devices with limited space.
* @type {NavigationButton}
* @default null
*/
monthButton?: NavigationButton;

/**
* Set to false to hide the Calendar.previousButton button.
* @type {boolean}
* @default null
*/
showPreviousButton?: boolean;

/**
* An ImgButton that appears above the week/day/month views of the
*  calendar and allows the user to move the calendar backwards in time.
* @type {ImgButton}
* @default null
*/
previousButton?: ImgButton;

/**
* Set to false to hide the Calendar.nextButton button.
* @type {boolean}
* @default null
*/
showNextButton?: boolean;

/**
* An ImgButton that appears above the week/day/month views of the
*  calendar and allows the user to move the calendar forwards in time.
* @type {ImgButton}
* @default null
*/
nextButton?: ImgButton;

/**
* The AutoChild Label used to display the current date or range above the 
*  selected calendar view.
* @type {Label}
* @default null
*/
dateLabel?: Label;

/**
* The height of time-slots in the calendar.
* @type {number}
* @default "20"
*/
rowHeight?: number;

/**
* The minimum height of time-rows in vertical calendar views. Rows will not shrink below 
*  this height when Calendar.sizeToWorkday is true, meaning that a Calendar with a long workday
*  may not be able to fit all workday rows in the viewport at once, and scrolling may be
*  necessary.
*  
*  To prevent users from scrolling beyond the workday hours, see Calendar.limitToWorkday.
* @type {number}
* @default "20"
*/
minRowHeight?: number;

/**
* The name of the view that should be visible initially by default.
* @type {ViewName}
* @default null
*/
currentViewName?: ViewName;

/**
* Dictates whether times throughout the widget are formatted and edited as 24-hour values. If
*  unset, defaults to the Time.use24HourTime. If set, and no
*  Calendar.timeFormatter is installed, causes the 
*  Calendar to choose an appropriate builtin formatter.
* @type {boolean}
* @default null
*/
twentyFourHourTime?: boolean;

/**
* Date formatter for displaying events.
*  Default is to use the system-wide default short date format, configured via
*  DateUtil.setShortDisplayFormat. Specify any valid DateDisplayFormat.
* @type {DateDisplayFormat}
* @default null
*/
dateFormatter?: DateDisplayFormat;

/**
* Display format to use for the time portion of events' date information. By default, times
*  are displayed in the global format, including the influence of the global 
*  Time.use24HourTime option, which is true by default.
*  P>
*  Note that this display setting does not affect the way in which time values are edited in the 
*  Calendar.eventEditor - see Calendar.twentyFourHourTime for more
*  information.
* @type {TimeDisplayFormat}
* @default null
*/
timeFormatter?: TimeDisplayFormat;

/**
* By default, EventCanvases show their content in hovers. If you set this attribute to false,
*  hovers will only be shown if the content of the event-canvas is visually clipped.
*  
*  Note - if you have custom hover-content/handling, you should leave this property set to 
*  true.
* @type {boolean}
* @default true
*/
alwaysShowEventHovers?: boolean;

/**
* In Calendar.indicators are showing, this attribute affects where in 
*  the z-order their canvases will be rendered: either in front of, or behind normal calendar
*  events.
* @type {boolean}
* @default true
*/
showIndicatorsInFront?: boolean;

/**
* When Calendar.showViewHovers is true, dictates whether to display
*  hover prompts when the mouse rolls over the normal cells in the body of CalendarViews.
*  
*  The content of the hover is determined by a call to 
*  Calendar.getCellHoverHTML, which can be overridden to return custom results; by 
*  default, it returns the cell's date as a string.
* @type {boolean}
* @default false
*/
showCellHovers?: boolean;

/**
* When Calendar.showViewHovers is true, dictates whether to display
*  hover prompts when the mouse rolls over the Calendar.headerLevels in
*  a CalendarView.
*  
*  The content of the hover is determined by a call to 
*  Calendar.getHeaderHoverHTML, which can be overridden to return custom results;
* @type {boolean}
* @default false
*/
showHeaderHovers?: boolean;

/**
* When set to true, the default value, causes the Calendar to show customizable hovers when 
*  the mouse moves over various areas of a CalendarView.
*  
*  See Calendar.showEventHovers, 
*  Calendar.showZoneHovers,
*  Calendar.showHeaderHovers,
*  Calendar.showCellHovers,
*  Calendar.showLaneFieldHovers,
*  Calendar.showDragHovers for further configuration options.
* @type {boolean}
* @default true
*/
showViewHovers?: boolean;

/**
* When Calendar.showViewHovers is true, dictates whether to display
*  hover prompts when the mouse moves over an EventCanvas in a 
*  calendarView.
*  
*  The content of the hover is determined by a call to 
*  Calendar.getCellHoverHTML, which can be overridden to return custom results.
* @type {boolean}
* @default true
*/
showEventHovers?: boolean;

/**
* When Calendar.showViewHovers is true, dictates whether to display
*  hover prompts when the mouse moves over a Calendar.zones in a calendarView.
*  
*  When Calendar.showCellHovers is true, this attribute is ignored and
*  zone hovers are not displayed.
*  
*  The content of the hover is determined by a call to 
*  Calendar.getZoneHoverHTML, which can be overridden to return custom results.
* @type {boolean}
* @default true
*/
showZoneHovers?: boolean;

/**
* When Calendar.showViewHovers is true, dictates whether to display
*  hover prompts when the mouse moves over the cells in a 
*  Calendar.laneFields.
*  
*  The content of the hover is determined by a call to 
*  Calendar.getCellHoverHTML, which can be overridden to return custom results. Note 
*  that getCellHoverHTML() is also called when the mouse moves over cells if
*  Calendar.showCellHovers is true - when called for a laneField, no
*  "date" parameter is passed to that method.
* @type {boolean}
* @default false
*/
showLaneFieldHovers?: boolean;

/**
* When Calendar.showViewHovers is true, dictates whether to display
*  hover prompts when an event is being dragged with the mouse.
*  
*  The content of the hover is determined by a call to 
*  Calendar.getDragHoverHTML, which can be overridden to return custom results; by 
*  default, it returns the date range of the drag canvas as a string.
* @type {boolean}
* @default false
*/
showDragHovers?: boolean;

/**
* Whether Calendar.eventCanvas should show a custom widget as content, 
*  rather than the default EventCanvas.showHeader and 
*  EventCanvas.showBody HTML.
* @type {boolean}
* @default false
*/
showEventCanvasComponents?: boolean;

/**
* Multi-AutoChild component, created as a space-filling member in individual 
*  EventCanvas, when Calendar.showEventCanvasComponents is
*  true.
*  
* @type {Canvas}
* @default null
*/
eventCanvasComponent?: Canvas;

/**
* Context menu displayed when an EventCanvas is right-clicked, or
*  when the rollover Calendar.eventCanvasContextButton is clicked. The 
*  context button, and the menu itself, will only be displayed if 
*  Calendar.getEventCanvasMenuItems returns 
*  an array of appropriate items for the event.
* @type {Menu}
* @default null
*/
eventCanvasContextMenu?: Menu;

/**
* By default, the Calendar.eventCanvasCloseButton and the
*  Calendar.eventCanvasHResizer and 
*  Calendar.eventCanvasVResizer resizer widgets 
*  for event canvases are shown only when the mouse is over a given event. Set this attribute
*  to false to have event canvases show these widgets permanently.
* @type {boolean}
* @default true
*/
useEventCanvasRolloverControls?: boolean;

/**
* When set to true, makes individual EventCanvas selectable.
*  Events may be selected via a single click, as well as being
*  included in the page's tab order. The current selected event is shown in a special 
*  style and pressing TAB or Shift-TAB will move the selection first among the events 
*  in the same lane, and then among those in the next or previous lane.
*  
*  Pressing Enter while an editable event is selected 
*  will show either the event- Calendar.eventDialog or 
*  Calendar.eventEditor. Pressing Delete will remove the event.
*  
*  Note that when this property is false, single clicking the event canvas for an
*  editable event will bring up an editing interface for that event. 
*  When true this is no longer the case - a user can double click to bring up the editing
*  interface instead (a single click will simply select the event canvas).
* @type {boolean}
* @default null
*/
canSelectEvents?: boolean;

/**
* The selection associated with the Calendar.
* @type {Selection}
* @default null
*/
selectionManager?: Selection;

/**
* The "gripper" widget that snaps to the top of an event canvas and allows an
*  event to be dragged with the mouse.
* @type {Img}
* @default null
*/
eventCanvasGripper?: Img;

/**
* Icon used as the default eveng gripper icon.
* @type {SCImgURL}
* @default "[SKIN]/Calendar/gripper.png"
*/
eventCanvasGripperIcon?: SCImgURL;

/**
* @type {Label}
* @default null
*/
eventCanvasLabel?: Label;

/**
* HLayout that snaps to the top-right of an event canvas on rollover and contains the
*  Calendar.eventCanvasCloseButton and/or 
*  Calendar.eventCanvasContextButton buttons.
* @type {HLayout}
* @default null
*/
eventCanvasButtonLayout?: HLayout;

/**
* The close button that snaps to the top-right of an event canvas on rollover and allows an
*  event to be removed from a CalendarView.
* @type {ImgButton}
* @default null
*/
eventCanvasCloseButton?: ImgButton;

/**
* The context button that snaps to the top-right of an event canvas on rollover and shows a
*  custom Calendar.getEventCanvasMenuItems when clicked.
* @type {ImgButton}
* @default null
*/
eventCanvasContextButton?: ImgButton;

/**
* The resizer image that snaps to the bottom of event canvases in Calendar.dayView
*  and Calendar.weekView views, allowing them to be resized vertically by dragging 
*  with the mouse.
* @type {Img}
* @default null
*/
eventCanvasVResizer?: Img;

/**
* The resizer image that snaps to the left and right edges of an editable event canvas in a
*  Timeline, allowing it to be resized horizontally by dragging with the mouse.
* @type {Img}
* @default null
*/
eventCanvasHResizer?: Img;

/**
* Set to true to render any defined Calendar.zones into 
*  Calendar.timelineView.
* @type {boolean}
* @default null
*/
showZones?: boolean;

/**
* An array of CalendarEvent instances representing pre-defined periods of time to be 
*  highlighted in Calendar.timelineView. Each zone renders out a
*  ZoneCanvas, a special, non-interactive subclass of 
*  EventCanvas, which spans all lanes and draws behind any normal, interactive 
*  events in the zorder.
*  
*  The default Calendar.zoneStyleName for these components renders them 
*  semi-transparent and with a bottom-aligned title label.
* @type {Array<Partial<CalendarEvent>>}
* @default null
*/
zones?: Array<Partial<CalendarEvent>>;

/**
* CSS style to apply to the Calendar.zoneCanvas created for each
*  specified Calendar.zones.
* @type {CSSStyleName}
* @default "zoneCanvas"
*/
zoneStyleName?: CSSStyleName;

/**
* AutoChild component created for each Calendar.zones entry.
* @type {ZoneCanvas}
* @default null
*/
zoneCanvas?: ZoneCanvas;

/**
* The vertical alignment of the header-text in each Calendar.zones.
* @type {VerticalAlignment}
* @default "bottom"
*/
zoneTitleOrientation?: VerticalAlignment;

/**
* Set to true to render any defined Calendar.indicators into 
*  Calendar.timelineView.
* @type {boolean}
* @default null
*/
showIndicators?: boolean;

/**
* An array of CalendarEvent instances representing instants in time, to be 
*  highlighted in Calendar.timelineView. Each indicator renders out as 
*  an IndicatorCanvas, a special, non-interactive subclass of 
*  EventCanvas, which spans all lanes and draws behind any normal, interactive 
*  events in the zorder, but in front of any Calendar.zones. The default 
*  Calendar.indicatorStyleName for these components renders them as thin vertical
*  lines that span all lanes and have a hover but no title.
* @type {Array<Partial<CalendarEvent>>}
* @default null
*/
indicators?: Array<Partial<CalendarEvent>>;

/**
* CSS style to apply to the Calendar.indicatorCanvas created for each
*  specified Calendar.indicators.
* @type {CSSStyleName}
* @default "indicatorCanvas"
*/
indicatorStyleName?: CSSStyleName;

/**
* AutoChild component created for each Calendar.indicators entry.
* @type {IndicatorCanvas}
* @default null
*/
indicatorCanvas?: IndicatorCanvas;

/**
* To display events in day and week views, the Calendar creates instance of EventWindow
*  for each event. Use the AutoChild system to customize these windows.
* @deprecated * in favor of Calendar.eventCanvas

* @type {EventWindow}
* @default null
*/
eventWindow?: EventWindow;

/**
* To display events in Calendar.dayView, Calendar.weekView and
*  Calendar.timelineView views, the Calendar creates instances of 
*  EventCanvas for each event. Use the AutoChild system to customize 
*  these canvases.
* @type {EventCanvas}
* @default null
*/
eventCanvas?: EventCanvas;

/**
* When true, causes Calendar.columnLayout to be added to each column in 
*  vertical views. In this mode, eventCanvases are stacked in these layouts, filling width 
*  and auto-sizing vertically to content, rather than being placed, sized and overlapped 
*  according to their times.
*  
*  Because times are ignored in this mode, various behaviors are switched off automatically;
*  for example, the Calendar.showLabelColumn is hidden and event-canvases 
*  cannot be Calendar.canResizeEvents or rendered
*  Calendar.renderEventsOnDemand.
* @type {boolean}
* @default false
*/
showColumnLayouts?: boolean;

/**
* When Calendar.showColumnLayouts is true, the layouts added to each column to stack 
*  events.
* @type {VLayout}
* @default null
*/
columnLayout?: VLayout;

/**
* AutoChild Menu, shown when a user clicks the 
*  Calendar.monthMoreEventsLinkTitle link in a cell of the 
*  Calendar.monthView. Items in this menu represent additional events, 
*  not already displayed in the cell, and clicking them fires the 
*  Calendar.eventClick notification.
* @type {Menu}
* @default null
*/
monthMoreEventsMenu?: Menu;



/**
*  Scrolls Calendar Calendar.dayView or Calendar.weekView views to the
*  time represented by the time parameter. This string parameter is expected to be an 
*  arbitrary logical time value in any parsable time format - no date portion is expected, but 
*  time formats like "13:31" or "1:20am" are supported.
*  
*  Has no effect in Calendar.timelineView, where an arbitrary time-value is
*  inapplicable to any range or resolution greater than a day.
* 
* @param {string} any parsable time-string
*/
scrollToTime(time:string): void;

/**
*  Move the viewport of a CalendarView to the start of it's scrollable range.
* 
* @param {CalendarView=} the view to affect, the current view if not specified
*/
scrollToStart(view?:CalendarView): void;

/**
*  Move the viewport of a CalendarView to the end of its scrollable range.
* 
* @param {CalendarView=} the view to affect, the current view if not specified
*/
scrollToEnd(view?:CalendarView): void;

/**
*  Resets the current visible range of a calendar view so that it shows the date on which the
*  passed event occurs.
* 
* @param {CalendarEvent} the event to move the calendar view to
* @deprecated * in favor of Calendar.scrollToEvent.

*/
moveToEvent(event:CalendarEvent): void;

/**
*  Scrolls the Calendar.getCurrentViewName so the passed event is 
*  visible. If the event is outside of the view's current date-range, the default behavior is
*  to automatically reload the view with a date-range starting at the event's 
*  Calendar.startDateField and then scroll to the event vertically as 
*  necessary. Pass false as the canReload param to prevent that default behavior.
* 
* @param {CalendarEvent} the event to move the calendar view to
* @param {boolean=} set to false to prevent a view from automatically reloading                with a new range if the passed event is not in its current                scrollable range
*/
scrollToEvent(event:CalendarEvent, canReload?:boolean): void;

/**
*  Setter for updating Calendar.hideUnusedLanes after creation.
* 
* @param {boolean} whether to hide unused lanes
*/
setHideUnusedLanes(hideUnusedEvents:boolean): void;

/**
*  For Timelines, configures the levels of HeaderLevel shown above the 
*  event area, and their time units, after initialization.
* 
* @param {Array<Partial<HeaderLevel>>} the array of HeaderLevels to set
*/
setHeaderLevels(headerLevels:Array<Partial<HeaderLevel>>): void;

/**
*  Returns true if the passed date should be considered disabled. Disabled dates don't allow 
*  events to be created by clicking on them, and drag operations that would start or end on
*  such dates are also disallowed.
*  
*  The default implementation returns false only for dates that fall on a 
*  DateUtil.getWeekendDays.
* 
* @param {Date} a Date instance
* @param {CalendarView=} the view the date appears in
*/
shouldDisableDate(date:Date, view?:CalendarView): boolean;

/**
*  Indicates whether the passed date should be visible in the passed CalendarView.
*  
*  The default implementation returns true, unless the date falls on a 
*  DateUtil.getWeekendDays and Calendar.showWeekends is 
*  false.
* 
* @param {Date} a Date instance
* @param {CalendarView=} the view the date appears in
*/
shouldShowDate(date:Date, view?:CalendarView): boolean;

/**
*  Indicates whether the passed Calendar.lanes should be visible in the passed 
*  CalendarView.
*  
*  The default implementation returns true, unless the lane has no events and 
*  Calendar.hideUnusedLanes is true.
* 
* @param {Lane | string} the lane object or name
* @param {CalendarView=} the view the lane appears in
*/
shouldShowLane(lane:Lane | string, view?:CalendarView): boolean;

/**
*  Indicates whether the passed CalendarEvent should be visible in the 
*  passed CalendarView.
*  
*  The default implementation returns true - note that this method only runs for events that are 
*  known to be in the accessible range and is a mechanism for extended custom filtering.
* 
* @param {CalendarEvent} the event to check
* @param {CalendarView=} the view the event will be rendered in
*/
shouldShowEvent(event:CalendarEvent, view?:CalendarView): boolean;

/**
*  Returns the start of the working day on the passed date. By default, this method returns
*  the value of Calendar.workdayStart.
* 
* @param {Date} a Date instance
* @param {string=} the name of the relevant lane - only passed for dayView with               showDayLanes: true
*/
getWorkdayStart(date:Date, laneName?:string): string;

/**
*  Returns the end of the working day on the passed date. By default, this method returns
*  the value of Calendar.workdayEnd.
* 
* @param {Date} a Date instance
* @param {string=} the name of the relevant lane - only passed for dayView with               showDayLanes: true
*/
getWorkdayEnd(date:Date, laneName?:string): string;

/**
*  Returns the first visible date in the passed, or currently selected, calendar view.
* 
* @param {CalendarView=} the view to get the startDate for, or current view if
*/
getVisibleStartDate(view?:CalendarView): Date;

/**
*  Returns the last visible date in the passed, or currently selected, calendar view.
* 
* @param {CalendarView=} the view to get the endDate for, or current view if null
*/
getVisibleEndDate(view?:CalendarView): Date;

/**
*  Returns the start of the selected week or month depending on the current calendar view. 
*  For the month view, and for the week view when not showing weekends, this will often be a 
*  different date than that returned by Calendar.getVisibleStartDate.
* 
* @param {CalendarView=} the view to get the periodStartDate for, or current view if null
*/
getPeriodStartDate(view?:CalendarView): Date;

/**
*  Returns the end of the period selected in the passed, or current, calendar view. 
*  For the Calendar.monthView, and for the 
*  Calendar.weekView when not showing weekends, this will often be a 
*  different date than that returned by Calendar.getVisibleEndDate.
* 
* @param {CalendarView=} the view to get the periodEndDate for, or current view if null
*/
getPeriodEndDate(view?:CalendarView): Date;

/**
*  Retrieves data from the DataSource that matches the specified criteria.
*  
*  When fetchData() is first called, if data has not already been provided via
*  ListGrid.setData, this method will create a ResultSet, which will be
*  configured based on component settings such as DataBoundComponent.fetchOperation
*  and DataBoundComponent.dataPageSize, as well as the general purpose
*  ListGrid.dataProperties. The created ResultSet will automatically send a DSRequest
*  to retrieve data from ListGrid.dataSource, and from then on will 
*  automatically manage paging through large datasets, as well as performing filtering and
*  sorting operations inside the browser when possible - see the ResultSet docs for
*  details.
*  
*  NOTE: do not use both DataBoundComponent.autoFetchData and a
*  call to fetchData() - this may result in two DSRequests to fetch data. Use
*  either DataBoundComponent.autoFetchData and Criteria or a manual call to fetchData()
*  passing criteria.
*  
*  Whether a ResultSet was automatically created or provided via ListGrid.setData, subsequent
*  calls to fetchData() will simply call ResultSet.setCriteria.
*  
*  Changes to criteria may or may not result in a DSRequest to the server due to
*  ResultSet.useClientFiltering. You can call
*  DataBoundComponent.willFetchData to determine if new criteria will result in a
*  server fetch.
*  
*  If you need to force data to be re-fetched, you can call
*  ListGrid.invalidateCache and new data will automatically be fetched
*  from the server using the current criteria and sort direction. NOTE: when using
*  invalidateCache() there is no need to also call fetchData()
*  and in fact this could produce unexpected results.
*  
*  This method takes an optional callback parameter (set to a DSCallback) to fire when
*  the fetch completes. Note that this callback will not fire if no server fetch is performed.
*  In this case the data is updated synchronously, so as soon as this method completes you
*  can interact with the new data. If necessary, you can use
*  DataBoundComponent.willFetchData to determine whether or not a server
*  fetch will occur when fetchData() is called with new criteria.
*  
*  In addition to the callback parameter for this method, developers can use 
*  ListGrid.dataArrived to be notified every time data is loaded.
*  
*  By default, this method assumes a TextMatchStyle of "exact"; that can be overridden
*  by supplying a different value in the requestProperties parameter. 
*  See DataBoundComponent.willFetchData;
*  
*  Changing the request properties
*  
*  Changes to TextMatchStyle made via requestProperties will be honored in
*  combination with the fetch criteria, possibly invalidating cache and triggering a server
*  request if needed, as documented for method:willFetchData. In contrast, changes to
*  DSRequest.operationId in the request properties will cause
*  the ResultSet or ResultTree to be rebuilt, always refetching from the server.
*  However, changes to other request properties after the initial fetch won't be detected, and
*  no fetch will get triggered based on that new request context.
*  
*  To pick up such changes, we recommend that you call Calendar.setData (passing an
*  empty array to ensure the data model is cleared), and then call this method to fetch again.
*  If you try to do it by calling method:invalidateCache, you may see duplicate fetches if you
*  haven't already updated the data context by calling this method with the new request
*  properties, and fail to do so before the component is Canvas.redraw.
* 
* 
* @param {Criteria=} Search criteria. If a DynamicForm is passed                     in as this argument instead of a raw criteria                      object, will be derived by calling                      DynamicForm.getValuesAsCriteria
* @param {DSCallback=} callback to invoke when a fetch is complete. Fires                     only if server contact was required
* @param {DSRequest=} additional properties to set on the DSRequest                     that will be issued
*/
fetchData(criteria?:Criteria, callback?:DSCallback, requestProperties?:DSRequest): void;

/**
*  Retrieves data that matches the provided criteria and displays the matching data in this
*  component.
*  
*  This method behaves exactly like ListGrid.fetchData except that
*  DSRequest.textMatchStyle is automatically set to "substring" so that String-valued
*  fields are matched by case-insensitive substring comparison.
* 
* 
* @param {Criteria=} Search criteria.            If a DynamicForm is passed in as this argument            instead of a raw criteria object, will be derived by calling            DynamicForm.getValuesAsCriteria
* @param {DSCallback=} callback to invoke when a fetch is complete. Fires                     only if server contact was required; see                      ListGrid.fetchData for details
* @param {DSRequest=} for databound components only - optional              additional properties to set on the DSRequest that will be issued
*/
filterData(criteria?:Criteria, callback?:DSCallback, requestProperties?:DSRequest): void;

/**
*  When Calendar.canGroupLanes is true, this method allows the grouping
*  in Calendar.timelineViews to be altered at runtime.
* 
* @param {string | Array<Partial<string>>} one or more field names to group by
*/
groupLanesBy(groupFieldName:string | Array<Partial<string>>): void;

/**
*  Sets the Calendar.lanes in the current calendar view. Only has an effect
*  in Calendar.timelineView, and in Calendar.dayView when 
*  Calendar.showDayLanes is true.
* 
* 
* @param {Array<Partial<Lane>>} array of lanes to display
* @param {boolean=} set to false to prevent events from being refreshed
*/
setLanes(lanes:Array<Partial<Lane>>, skipRefreshEvents?:boolean): void;

/**
*  Adds a new Lane to the calendar, for display in the 
*  Calendar.timelineView, and in the 
*  Calendar.dayView if Calendar.showDayLanes is true.
* 
* 
* @param {Lane} a new Lane object to add to the calendar
*/
addLane(lane:Lane): void;

/**
*  Removes a lane from the calendar in Calendar.timelineView, or in Calendar.dayView if 
*  Calendar.showDayLanes is true.
*  
*  Accepts either a Lane or a string that represents the 
*  Lane.name of a lane.
* 
* 
* @param {Lane | string} either the actual Lane object or the name of the lane to remove
*/
removeLane(lane:Lane | string): void;

/**
*  Changes the Calendar.showDayLanes of the day view at runtime - whether to show a
*  normal day column for the Calendar.chosenDate, or the specified set of 
*  Calendar.lanes.
*  
* 
* @param {boolean} whether or not to show lanes in the day view
*/
setShowDayLanes(showDayLanes:boolean): void;

/**
*  A notification method fired when the events in the current view have been refreshed.
* 
*/
eventsRendered(): void;

/**
*  Initialize the data object with the given array. Observes methods of the data object
*  so that when the data changes, the calendar will redraw automatically.
*  
* 
* @param {Array<Partial<CalendarEvent>>} data to show in the list
*/
setData(newData:Array<Partial<CalendarEvent>>): void;

/**
*  Returns the currently selected CalendarView.
* 
*/
getSelectedView(): CalendarView;

/**
*  Returns the CalendarView with the passed ViewName.
* 
* @param {ViewName} the name of the CalendarView to return
*/
getView(viewName:ViewName): CalendarView;

/**
*  Get the name of the visible view.  Returns one of 'day', 'week', 'month' or 'timeline'.
* 
* 
*/
getCurrentViewName(): ViewName;

/**
*  Sets the currently visible view.
* 
* 
* @param {ViewName} The name of the view that should be made visible.
*/
setCurrentViewName(viewName:ViewName): ViewName;

/**
*  Returns the title text for the passed event, which is displayed in the header area of an 
*  eventCanvas rendered in a vertical or horizontal view, or as a clickable link in a cell in a 
*  Calendar.showMonthView.
*  
*  The default implementation returns the event's
*  Calendar.nameField for timelines, and that same value pre-pended with
*  the event's Calendar.startDateField for day, week and month views.
*  
* 
* @param {CalendarEvent} the event to get the description text for
* @param {CalendarView=} the view in which the event is being rendered
*/
getEventHeaderHTML(event:CalendarEvent, view?:CalendarView): HTMLString;

/**
*  Returns the description text for the passed event, for display in the body area of an event 
*  canvas. The default implementation returns the event's
*  Calendar.descriptionField.
*  
* 
* @param {CalendarEvent} the event to get the description text for
* @param {CalendarView=} the view in which the event is being rendered
*/
getEventBodyHTML(event:CalendarEvent, view?:CalendarView): HTMLString;

/**
*  Returns the CalendarEvent.startDate of the passed event.
*  
* 
* @param {CalendarEvent} the event to get the start date of
*/
getEventStartDate(event:CalendarEvent): Date;

/**
*  Returns the Calendar.endDateField of the passed event. If the event is
*  Calendar.allowDurationEvents, the result is calculated from the 
*  CalendarEvent.startDate and the specified 
*  CalendarEvent.duration and CalendarEvent.durationUnit.
*  
* 
* @param {CalendarEvent} the event to get the start date of
*/
getEventEndDate(event:CalendarEvent): Date;

/**
*  Setter for updating Calendar.showWeekends at runtime.
* 
* 
* @param {boolean} whether or not to show weekends
*/
setShowWeekends(showWeekends:boolean): void;

/**
*  For Timelines, and for Calendar.dayView with 
*  Calendar.showDayLanes set, creates a new event and adds it to a 
*  particular Lane.
* 
* 
* @param {Lane} the Lane in which to add this event
* @param {Date | object} start date of event, or CalendarEvent Object
* @param {Date=} end date of event
* @param {string=} name of event
* @param {string=} description of event
* @param {any=} new values of additional fields to be updated
* @deprecated * in favor of Calendar.addCalendarEvent

*/
addLaneEvent(laneName:Lane, startDate:Date | object, endDate?:Date, name?:string, description?:string, otherFields?:any): void;

/**
*  Create a new event in this calendar instance.
* 
* 
* @param {Date | CalendarEvent} start date of event, or CalendarEvent Object
* @param {Date=} end date of event
* @param {string=} name of event
* @param {string=} description of event
* @param {object=} new values of additional fields to be updated
* @deprecated * in favor of Calendar.addCalendarEvent

*/
addEvent(startDate:Date | CalendarEvent, endDate?:Date, name?:string, description?:string, otherFields?:object): void;

/**
*  Create a new event in this calendar.
*  
*  In all cases, the CalendarEvent passed as the first parameter must have at 
*  least a Calendar.startDateField set. If the calendar is showing 
*  Calendar.lanes, the name of the CalendarEvent.lane and, if 
*  applicable, the CalendarEvent.sublane, must also be set.
* 
* 
* @param {CalendarEvent} the new calendar event to add
* @param {object=} additional, custom values to be saved with the event
*/
addCalendarEvent(event:CalendarEvent, customValues?:object): void;

/**
*  Remove an event from this calendar.
* 
* 
* @param {CalendarEvent} The event object to remove from the calendar
*/
removeEvent(event:CalendarEvent): void;

/**
*  Update an event in this calendar.
* 
* 
* @param {CalendarEvent} The event object to update
* @param {Date} start date of event
* @param {Date} end date of event
* @param {string} name of event
* @param {string} description of event
* @param {object} new values of additional fields to be updated
* @deprecated * in favor of Calendar.updateCalendarEvent

*/
updateEvent(event:CalendarEvent, startDate:Date, endDate:Date, name:string, description:string, otherFields:object): void;

/**
*  Update an event in this calendar.
* 
* 
* @param {CalendarEvent} The event object that will be updated
* @param {CalendarEvent} The new attributes for the event
* @param {object} new values of additional fields to be updated
*/
updateCalendarEvent(event:CalendarEvent, newEvent:CalendarEvent, otherFields:object): void;

/**
*  Refreshes the passed event in the current view.
* 
* 
* @param {CalendarEvent} The event object to refresh in the current view
*/
refreshEvent(event:CalendarEvent): void;

/**
*  Update the styleName for the passed event. Refreshes the event's canvas in the current view.
* 
* 
* @param {CalendarEvent} The event object to refresh in the current view
* @param {CSSStyleName} The new CSS style to apply to the canvases showing this event
*/
setEventStyle(event:CalendarEvent, styleName:CSSStyleName): void;

/**
*  Gets the hover HTML for an event being hovered over. Override here to return custom 
*  HTML based upon the parameter event object.
* 
* 
* @param {CalendarEvent} The event being hovered
* @param {EventCanvas} the event canvas being hovered over
* @param {CalendarView} the CalendarView in which the eventCanvas lives
* @param {string=} the default HTML to show when hovering over the passed event
*/
getEventHoverHTML(event:CalendarEvent, eventCanvas:EventCanvas, view:CalendarView, defaultValue?:string): HTMLString;

/**
*  Gets the hover HTML for a Calendar.zones being hovered over. Override here to 
*  return custom HTML based upon the parameter zone object.
* 
* 
* @param {CalendarEvent} The zone being hovered over
* @param {ZoneCanvas} the zone canvas being hovered over
* @param {CalendarView} the CalendarView in which the zoneCanvas is displayed
* @param {string} the default HTML to show when hovering over the passed Zone
*/
getZoneHoverHTML(zone:CalendarEvent, zoneCanvas:ZoneCanvas, view:CalendarView, defaultValue:string): HTMLString;

/**
*  Called whenever an IndicatorCanvas is clicked in the 
*  Calendar.timelineView. There is no default implementation.
* 
* 
* @param {CalendarEvent} indicator that was clicked on
* @param {ViewName} view where the event's canvas was clicked
*/
indicatorClick(indicatorEvent:CalendarEvent, viewName:ViewName): boolean;

/**
*  Gets the hover HTML for an Calendar.indicators being hovered over. 
*  Override here to return custom HTML based upon the parameter indicator object.
* 
* 
* @param {CalendarEvent} The indicator being hovered over
* @param {IndicatorCanvas} the indicator canvas being hovered over
* @param {CalendarView} the CalendarView in which the indicatorCanvas is displayed
* @param {string} the default HTML to show when hovering over the passed Indicator
*/
getIndicatorHoverHTML(indicator:CalendarEvent, indicatorCanvas:IndicatorCanvas, view:CalendarView, defaultValue:string): HTMLString;

/**
*  Returns the hover HTML for the cell at the passed co-ordinates in the passed view. By 
*  default, the hover text is the snap date closest to the mouse, if the cell being hovered is 
*  a normal date cell - otherwise, it is the title of the Calendar.laneFields
*  being hovered over.
*  
*  Override here to return custom HTML for the passed cell.
* 
* 
* @param {CalendarView} the CalendarView the mouse is hovered over
* @param {Record} The record containing the cell being hovered
* @param {Integer} The rowNum of the cell being hovered
* @param {Integer} the colNum of the cell being hovered
* @param {Date} the snap-date at the mouse, which may be different from the result of a           call to Calendar.getCellDate
* @param {string} the default hover text for the passed values
*/
getCellHoverHTML(view:CalendarView, record:Record, rowNum:Integer, colNum:Integer, date:Date, defaultValue:string): HTMLString;

/**
*  Returns the hover HTML to show in a hover when the mouse moves over the header area.
* 
* 
* @param {CalendarView} the CalendarView the mouse is hovered over
* @param {HeaderLevel} the header level hovered over
* @param {Date} the start date of the span being hovered over
* @param {Date} the end date of the span being hovered over
* @param {string} the default text for the passed header level and date range
*/
getHeaderHoverHTML(view:CalendarView, headerLevel:HeaderLevel, startDate:Date, endDate:Date, defaultValue:string): HTMLString;

/**
*  Switches the various levels of Calendar.showViewHovers on or off at runtime.
* 
* 
* @param {boolean} whether to allow CalendarViews to show hovers
*/
setShowViewHovers(showViewHovers:boolean): void;

/**
*  Returns the HTML to show in a hover when an existing event is dragged, or when a new event
*  is being created by dragging with the mouse.
* 
* 
* @param {CalendarView} the CalendarView the mouse is hovered over
* @param {CalendarEvent} the CalendarEvent attached to the EventCanvas being dragged
* @param {string} the default text for the passed values
*/
getDragHoverHTML(view:CalendarView, event:CalendarEvent, defaultValue:string): HTMLString;

/**
*  Should a component be applied to the passed EventCanvas in the 
*  EventCanvas.calendarView in which it appears? Return false from this method to 
*  override the global value of 
*  Calendar.showEventCanvasComponents for this canvas.
* 
* @param {EventCanvas} should this eventCanvas get a component component?
*/
showEventCanvasComponent(canvas:EventCanvas): boolean;

/**
*  Called from EventCanvas.setEvent when Calendar.showEventCanvasComponents is 
*  true and an eventCanvas needs a component.  The method is expected to create and return a 
*  single component to apply as a space-filling member of the passed EventCanvas 
*  in the EventCanvas.calendarView in which it appears.
*  
*  By default, this method returns a DetailViewer showing values from the 
*  associated event, according to the fields in the Calendar's 
*  Calendar.dataSource, or the default event-fields if no dataSource is 
*  present.
*  
*  However, the component can be any Canvas widget, including a Layout 
*  containing an arrangement of other widgets. When applied as a member of an 
*  eventCanvas, the component fills its container and limits its own content to that size,
*  showing scrollbars as appropriate.
*  
*  See Calendar.updateEventCanvasComponent for details on updating components that have
*  already been created and applied.
* 
* @param {EventCanvas} the eventCanvas to get the component for
*/
createEventCanvasComponent(canvas:EventCanvas): Canvas;

/**
*  Called from EventCanvas.setEvent when Calendar.showEventCanvasComponents is 
*  true and the eventCanvas already has a 
*  Calendar.createEventCanvasComponent. This method is expected to update 
*  the passed component as necessary, based on the 
*  EventCanvas.event.
*  
*  By default, if the passed component has methods called 
*  setEvent or setData, those methods are called automatically.
* 
* @param {EventCanvas} the eventCanvas to update the component for
* @param {Canvas} the component to be updated the canvas in question
*/
updateEventCanvasComponent(canvas:EventCanvas, component:Canvas): void;

/**
*  Returns the CSSStyleName to use for the passed 
*  CalendarEvent, in the passed CalendarView. By default, returns
*  the style Calendar.eventStyleNameField, if one is specified - otherwise, 
*  in Calendar.lanes views, it returns the style specified on the 
*  Lane.eventStyleName, or the style specified on the 
*  Calendar.eventStyleName.
* 
* @param {CalendarEvent} the event to get the CSS style for
* @param {CalendarView=} the CalendarView that contains the canvas being styled
*/
getEventCanvasStyle(event:CalendarEvent, view?:CalendarView): CSSStyleName;

/**
*  If this method returns a value, it is expected to return an array of 
*  MenuItem applicable to the passed canvas and its event. If an array 
*  with valid entries is returned, the rollover
*  Calendar.eventCanvasContextButton is shown for the passed canvas.
* 
* @param {EventCanvas} the canvas to get menu items for
*/
getEventCanvasMenuItems(canvas:EventCanvas): Array<Partial<MenuItem>>;

/**
*  Selects a single event in the current view, showing it in a selected style and deselecting 
*  any other selected events.
* 
* @param {CalendarEvent} the event to select
*/
selectSingleEvent(event:CalendarEvent): boolean;

/**
*  Adds an event to the list of selected events in the current view, showing it in a selected 
*  style.
* 
* @param {CalendarEvent} the event to add to the selection
*/
selectEvent(event:CalendarEvent): boolean;

/**
*  Adds one or more events to the list of selected events in the current view, showing them in 
*  a selected style.
* 
* @param {Array<Partial<CalendarEvent>>} the events to add to the selection
*/
selectEvents(events:Array<Partial<CalendarEvent>>): boolean;

/**
*  Removes an event from the list of selected events in the current view, clearing its selected 
*  style.
* 
* @param {CalendarEvent} the event to deselect
*/
deselectEvent(event:CalendarEvent): boolean;

/**
*  Removes one or more events from the list of selected events in the current view, clearing
*  their selected styles.
* 
* @param {Array<Partial<CalendarEvent>>} the events to deselect
*/
deselectEvents(events:Array<Partial<CalendarEvent>>): boolean;

/**
*  Returns the currently selected CalendarEvent, or the first one if more 
*  than one is selected.
* 
*/
getSelectedEvent(): CalendarEvent;

/**
*  Returns the currently selected list of CalendarEvent.
* 
*/
getSelectedEvents(): Array<Partial<CalendarEvent>>;

/**
*  Returns the Calendar.eventCanvasGripperIcon to use as the gripper for
*  the passed event canvas.
* 
* @param {EventCanvas} the canvas that will show the gripper
*/
getEventCanvasGripperIcon(canvas:EventCanvas): SCImgURL;

/**
*  Sets the Calendar.zones used to highlight areas of this calendar.
* 
* 
* @param {Array<Partial<CalendarEvent>>} array of zones to display
*/
setZones(zones:Array<Partial<CalendarEvent>>): void;

/**
*  Adds a new Calendar.zones to the calendar.
* 
* 
* @param {CalendarEvent} a new zone to add to the calendar
*/
addZone(zone:CalendarEvent): void;

/**
*  Removes a Calendar.zones from the calendar.
*  
*  Accepts either a CalendarEvent or a string that represents the 
*  CalendarEvent.name of a zone.
* 
* 
* @param {CalendarEvent | string} either the actual CalendarEvent representing the zone,         or the name of the zone to remove
*/
removeZone(zone:CalendarEvent | string): void;

/**
*  Called whenever a ZoneCanvas is clicked in the 
*  Calendar.timelineView. There is no default implementation.
* 
* 
* @param {CalendarEvent} zone that was clicked on
* @param {ViewName} view where the event's canvas was clicked
*/
zoneClick(zoneEvent:CalendarEvent, viewName:ViewName): boolean;

/**
*  Returns the CSSStyleName to use for the passed 
*  Calendar.zones, in the passed CalendarView. By default, 
*  returns the style Calendar.eventStyleNameField, if one is specified,
*  or the style specified on the Calendar.zoneStyleName otherwise.
* 
* @param {CalendarEvent} the zone to get the CSS style for
* @param {CalendarView=} the CalendarView that contains the canvas being styled
*/
getZoneCanvasStyle(zone:CalendarEvent, view?:CalendarView): CSSStyleName;

/**
*  Returns the CSSStyleName to use for the passed 
*  Calendar.indicators, in the passed CalendarView. By default, 
*  returns the style Calendar.eventStyleNameField, if one is specified,
*  or the style specified on the Calendar.indicatorStyleName otherwise.
* 
* @param {CalendarEvent} the indicator to get the CSS style for
* @param {CalendarView=} the CalendarView that contains the canvas being styled
*/
getIndicatorCanvasStyle(indicator:CalendarEvent, view?:CalendarView): CSSStyleName;

/**
*  Sets the Calendar.indicators used to highlight instants in time.
* 
* @param {Array<Partial<CalendarEvent>>} array of indicators to display
*/
setIndicators(indicators:Array<Partial<CalendarEvent>>): void;

/**
*  Adds a new Calendar.indicators to the calendar.
* 
* @param {CalendarEvent} a new indicator to add to the calendar
*/
addIndicator(indicator:CalendarEvent): void;

/**
*  Removes a Calendar.indicators from the calendar.
*  
*  Accepts either a CalendarEvent or a string that represents the 
*  CalendarEvent.name of anindicator.
* 
* @param {CalendarEvent | string} either the actual CalendarEvent representing the          indicator, or the name of the indicator to remove
*/
removeIndicator(indicator:CalendarEvent | string): void;

/**
*  Set the current date for which the calendar will display events.
* 
* 
* @param {Date} the new date to set as the current date
*/
setChosenDate(newDate:Date): void;

/**
*  Should the parameter date be considered a workday? By default this method tries to find the
*  parameter date day in Calendar.workdays, and returns true if found. Override this 
*  method to provide custom logic for determining workday, perhaps returning false on holidays.
*  
*  Note that, when showing Calendar.showDayLanes in the 
*  Calendar.dayView, this method is also passed the name of the associated lane.
* 
* 
* @param {Date} date to check for being a workday
* @param {string} the name of the lane if Calendar.showDayLanes is true, null otherwise
*/
dateIsWorkday(date:Date, laneName:string): boolean;

/**
*  Move to the next day, week, or month, depending on which tab is selected.
* 
* 
*/
next(): void;

/**
*  Move to the previous day, week, month, or timeline range depending on which tab is selected.
* 
* 
*/
previous(): void;

/**
*  For views that support Calendar.lanes, returns the padding to apply to events
*  rendered in lanes in the passed or current view. By default, returns 
*  Calendar.laneEventPadding.
* 
* 
* @param {CalendarView=} the view to get the lane padding for
*/
getLanePadding(view?:CalendarView): Integer;

/**
*  For views that support Calendar.lanes, returns the array of events in the
*  current dataset that apply to the passed lane in the passed or current view.
* 
* 
* @param {Lane | string} lane object or name to get the events for
* @param {CalendarView=} the view in which the passed lane lives - uses the selected                view if unset
*/
getLaneEvents(lane:Lane | string, view?:CalendarView): Array<Partial<CalendarEvent>>;

/**
*  For views that support Calendar.lanes and allow 
*  Calendar.useSublanes, returns the array of events in the
*  current dataset that apply to the passed lane and sublane in the passed or current view.
* 
* 
* @param {Lane | string} lane object or name to get the events for
* @param {Lane | string} sublane object or name to get the events for
* @param {CalendarView=} the view in which the passed sublane lives - uses the selected                view if unset
*/
getSublaneEvents(lane:Lane | string, sublane:Lane | string, view?:CalendarView): Array<Partial<CalendarEvent>>;

/**
*  Returns the text to display between the navigation buttons above the Calendar - indicates 
*  the visible date range.
* 
* @param {string} one of "day", "week", "month" or "timeline"
* @param {Date} the start of the visible date range
* @param {Date=} the optional end of the visible date range
*/
getDateLabelText(viewName:string, startDate:Date, endDate?:Date): string;

/**
*  Cancels the current edit-session, closing the builtin event 
*  Calendar.eventDialog or Calendar.eventEditor and clearing any 
*  visible edit-selection from the Calendar.getSelectedView.
*  
* 
*/
cancelEditing(): void;

/**
*  Open the Quick Event dialog showing minimal information about an existing 
*  CalendarEvent.
*  
*  The Calendar.startDateField field on the event is used to calculate the 
*  display location for the dialog.
*  
*  If this method is called when the Event Dialog is already showing another event, and if 
*  changes have been made, a confirmation dialog is displayed and editing of the new event 
*  is cancelled unless confirmed.
*  
*  You can override this method to prevent the default action, perhaps instead showing a custom 
*  interface that performs validations or gathers custom data before making a call to 
*  Calendar.addCalendarEvent or 
*  Calendar.updateCalendarEvent when the new data is available.
*  
* 
* @param {CalendarEvent=} the event to show in the Editor
* @param {boolean=} optional boolean indicating that this is a new event, event if                an event is passed - used to pass defaults for a new event
*/
showEventDialog(event?:CalendarEvent, isNewEvent?:boolean): void;

/**
*  Open the Quick Event dialog to begin editing a new CalendarEvent.
*  
*  If passed, the event parameter is used as defaults for the new event - in addition, the 
*  event's Calendar.startDateField, and its 
*  Calendar.laneNameField, for timeline events, are used to calculate the 
*  display location for the dialog.
*  
*  If this method is called when the Event Dialog is already showing another event, and if 
*  changes have been made, a confirmation dialog is displayed and editing of the new event 
*  is cancelled unless confirmed.
*  
*  You can override this method to prevent the default action, perhaps instead showing a custom 
*  interface that performs validations or gathers custom data before making a call to
*  Calendar.addCalendarEvent or 
*  Calendar.updateCalendarEvent when the new data is available.
*  
* 
* @param {CalendarEvent=} defaults for the new event
*/
showNewEventDialog(event?:CalendarEvent): void;

/**
*  Show an Event Editor for the passed event. Event Editor's fill the Calendar and allow 
*  for editing of the built-in Event fields, like Calendar.nameField and 
*  Calendar.descriptionField, as well as any 
*  custom fields supplied via Calendar.eventEditorFields.
*  
*  If isNewEvent is true, a new event is created - in this case, if an event is passed, it
*  represents default values to apply to the new event.
*  
*  You can override this method to prevent the default action, perhaps instead showing a custom 
*  interface that performs validations or gathers custom data before making a call to 
*  Calendar.addCalendarEvent or 
*  Calendar.updateCalendarEvent when the new data is available.
*  
* 
* @param {CalendarEvent=} an existing event to show in the Editor
* @param {boolean=} optional boolean indicating that this is a new event, even if                an event is passed - used to pass defaults for a new event
*/
showEventEditor(event?:CalendarEvent, isNewEvent?:boolean): void;

/**
*  Show an Event Editor for a new event. If an CalendarEvent is passed as the 
*  parameter, it is used as defaults for the new event.
*  
* 
* @param {CalendarEvent=} defaults for the new event to show in the Editor
*/
showNewEventEditor(event?:CalendarEvent): void;

/**
*  Return CSS text for styling the cell associated with the passed date and/or rowNum & colNum,
*  which will be applied in addition to the CSS class for the cell, as overrides.
*  
*  "CSS text" means semicolon-separated style settings, suitable for inclusion in a CSS
*  stylesheet or in a STYLE attribute of an HTML element.
* 
* 
* @param {Date} the date to return CSS text for
* @param {Integer} the row number containing the date to get the CSS for
* @param {Integer} the column number containing the date to get the CSS for
* @param {CalendarView} the relevant CalendarView
*/
getDateCSSText(date:Date, rowNum:Integer, colNum:Integer, view:CalendarView): string;

/**
*  Return the CSS styleName for the associated date-cell in the passed view.
* 
* 
* @param {Date} the date to return the CSS styleName for
* @param {Integer} the row number containing the date to get the CSS styleName for
* @param {Integer} the column number containing the date to get the CSS styleName for
* @param {CalendarView} the relevant CalendarView
*/
getDateStyle(date:Date, rowNum:Integer, colNum:Integer, view:CalendarView): CSSStyleName;

/**
*  Return the HTML to be displayed in the associated date-cell in the passed view.
* 
*  Note that the Calendar.monthView has default cell HTML, controlled via
*  Calendar.getDayBodyHTML.
* 
* 
* @param {Date} the date to get the HTML for
* @param {Integer} the row number containing the date to get the HTML for
* @param {Integer} the column number containing the date to get the HTML for
* @param {CalendarView} the relevant CalendarView
*/
getDateHTML(date:Date, rowNum:Integer, colNum:Integer, view:CalendarView): HTMLString;

/**
*  When Calendar.getDateHTML returns a value, this method returns the
*  horizontal alignment for that value in its cell, in the passed view.
*  
* 
* @param {Date} the date to get the cell-alignment for
* @param {Integer} the row number containing the date to get the cell-alignment for
* @param {Integer} the column number containing the date to get the cell-alignment for
* @param {CalendarView} the relevant CalendarView
*/
getDateCellAlign(date:Date, rowNum:Integer, colNum:Integer, view:CalendarView): HTMLString;

/**
*  When Calendar.getDateHTML returns a value, this method returns the
*  vertical alignment for that value in its cell, in the passed view.
*  
* 
* @param {Date} the date to get the cell-alignment for
* @param {Integer} the row number containing the date to get the cell-alignment for
* @param {Integer} the column number containing the date to get the cell-alignment for
* @param {CalendarView} the relevant CalendarView
*/
getDateCellVAlign(date:Date, rowNum:Integer, colNum:Integer, view:CalendarView): HTMLString;

/**
*  Return the title text to display in the header-button of the ListGridField showing the 
*  passed date, in the passed view.
* 
* 
* @param {Date} the date to return the header-title for - note that the           Calendar.monthView does not pass this parameter           because a single column represents multiple dates
* @param {Integer} the week-day number of the passed date, except for the              Calendar.monthView, where no date is passed,              because the week-day number represents multiple dates.
* @param {string} the default header-title for the passed date and view
* @param {CalendarView} the relevant CalendarView
*/
getDateHeaderTitle(date:Date, dayOfWeek:Integer, defaultValue:string, view:CalendarView): string;

/**
*  Return the Date instance associated with the passed co-ordinates in the passed or selected
*  view. If the cell at the passed co-ordinates is not a date-cell, returns null. If rowNum 
*  and colNum are both unset, returns the date from the cell under the mouse.
*  
*  To determine the date at a more specific point within a cell, see Calendar.getDateFromPoint.
* 
* 
* @param {Integer=} the row number to get the date for
* @param {Integer=} the column number to get the date for
* @param {CalendarView=} the view to use - uses the selected view if not passed
*/
getCellDate(rowNum?:Integer, colNum?:Integer, view?:CalendarView): Date;

/**
*  Returns a Date instance representing the point at the passed offsets into the body of the 
*  current view.
*  
*  If snapOffsets is passed as false, returns the date representing the 
*  exact position of the passed offsets. If unset or passed as true, returns the date at the 
*  nearest eventSnapGap to the left, for Timelines, or above for Calendar.dayView 
*  and Calendar.weekView views.
*  
*  If neither x nor y offsets are passed, assumes them from the last mouse event.
*  
*  If the cell at the eventual offsets is not a date-cell, returns null.
*  
*  Note that, for the Calendar.monthView, this method is functionally equivalent to 
*  Calendar.getCellDate, which determines the date associated with a cell, without the additional 
*  offset precision offered here.
* 
* 
* @param {Integer=} the x offset into the body of the selected view - non-functional for            the Calendar.dayView. If this param and "y" are both unset,            assumes both offsets from the last mouse event.
* @param {Integer=} the y offset into the body of the selected view - non-functional for the              Calendar.timelineView. If this param and "x" are               both unset, assumes both offsets from the last mouse event.
* @param {boolean=} whether to snap the offsets to the nearest eventSnapGap - if                 unset, the default is true
* @param {CalendarView=} the view to use - or the selected view if not passed
*/
getDateFromPoint(x?:Integer, y?:Integer, snapOffsets?:boolean, view?:CalendarView): Date;

/**
*  Returns the Lane with the passed name, in the passed view
* 
* @param {string} the name of the lane to return
* @param {CalendarView=} the view to get the lane object from
*/
getLane(lane:string, view?:CalendarView): Lane;

/**
*  Returns the Lane associated with the passed event, in the passed view
* 
* @param {CalendarEvent} the event to get the lane for
* @param {CalendarView=} the view to get the lane object from
*/
getEventLane(event:CalendarEvent, view?:CalendarView): Lane;

/**
*  Returns the Lane.sublanes with the passed name, from the Lane
*  with the passed name, in the passed view.
* 
* @param {string} the name of the lane containing the sublane to return
* @param {string} the name of the sublane to return
* @param {CalendarView=} the view to get the sublane object from
*/
getSublane(lane:string, sublane:string, view?:CalendarView): Lane;

/**
*  Returns the Lane.sublanes associated with the passed event, in the passed view
* 
* @param {CalendarEvent} the event to get the sublane for
* @param {CalendarView=} the view to get the sublane object from
*/
getEventSublane(event:CalendarEvent, view?:CalendarView): Lane;

/**
*  Returns the Lane at the passed co-ordinates. To get the lane under the mouse, pass
*  null for both x and y.
* 
* @param {Integer=} the x offset into the body of the selected view
* @param {Integer=} the y offset into the body of the selected view. If this param and "x" are               both unset, assumes both offsets from the last mouse event.
* @param {CalendarView=} the view to get the lane from - selected view if not passed
*/
getLaneFromPoint(x?:Integer, y?:Integer, view?:CalendarView): Lane;

/**
*  Returns the Lane.sublanes at the passed co-ordinates. To get the sublane under
*  the mouse, pass null for both x and y.
* 
* @param {Integer=} optional x offset into the body of the selected view
* @param {Integer=} optional y offset into the body of the selected view. If this param and "x" are               both unset, assumes both offsets from the last mouse event.
* @param {CalendarView=} the view to get the sublane from - selected view if not passed
*/
getSublaneFromPoint(x?:Integer, y?:Integer, view?:CalendarView): Lane;

/**
*  Notification that fires whenever the current view changes via the 
*  Calendar.mainView.
* 
* 
* @param {ViewName} the name of the current view after the change
*/
currentViewChanged(viewName:ViewName): void;

/**
*  Return the HTML to be shown in the body of a day in the month view.
*  
*  Default is to render a series of links that call Calendar.eventClick to provide details
*  and/or an editing interface for the events.
*  
*  getDayBodyHTML() is not called for days outside of the current month if
*  Calendar.showOtherDays is false.
* 
* 
* @param {Date} JavaScript Date object representing this day
* @param {Array<Partial<CalendarEvent>>} events that fall on this day
* @param {Calendar} the calendar itself
* @param {Integer} the row number to which the parameter date belongs
* @param {Integer} the column number to which the parameter date belongs
*/
getDayBodyHTML(date:Date, events:Array<Partial<CalendarEvent>>, calendar:Calendar, rowNum:Integer, colNum:Integer): HTMLString;

/**
*  This method returns the hover HTML to be displayed when the user hovers over a cell
*  displayed in the calendar month view tab.
*  
*  Default implementation will display a list of the events occurring on the date the user is
*  hovering over. Override for custom behavior. Note that returning null will suppress the
*  hover altogether.
* 
* 
* @param {Date} Date the user is hovering over
* @param {Array<Partial<CalendarEvent>>} array of events occurring on the current date. May be empty.
*/
getMonthViewHoverHTML(date:Date, events:Array<Partial<CalendarEvent>>): HTMLString;

/**
*  Called when the body area of a day in the month view is clicked on, outside of any links
*  to a particular event.
*  
*  By default, if the user can add events, shows a dialog for adding a new event for that
*  day. Return false to cancel this action.
*  
*  Not called if the day falls outside the current month and Calendar.showOtherDays is false.
* 
* 
* @param {Date} JavaScript Date object representing this day
* @param {Array<Partial<CalendarEvent>>} events that fall on this day
* @param {Calendar} the calendar itself
* @param {Integer} the row number to which the parameter date belongs
* @param {Integer} the column number to which the parameter date belongs
*/
dayBodyClick(date:Date, events:Array<Partial<CalendarEvent>>, calendar:Calendar, rowNum:Integer, colNum:Integer): boolean;

/**
*  Called when the header area of a day in the month view is clicked on.
*  
*  By default, moves to the day tab and shows the clicked days events.
*  Return false to cancel this action.
*  
*  Not called if the day falls outside the current month and Calendar.showOtherDays is false.
* 
* 
* @param {Date} JavaScript Date object representing this day
* @param {Array<Partial<CalendarEvent>>} events that fall on this day
* @param {Calendar} the calendar itself
* @param {Integer} the row number to which the parameter date belongs
* @param {Integer} the column number to which the parameter date belongs
*/
dayHeaderClick(date:Date, events:Array<Partial<CalendarEvent>>, calendar:Calendar, rowNum:Integer, colNum:Integer): boolean;

/**
*  Notification fired whenever a user changes an event, whether by dragging the event or by
*  editing it in a dialog.
*  
*  In a calendar with a DataSource, eventChanged() fires after the updated event has
*  been successfully saved to the server
* 
* 
* @param {CalendarEvent} the event that changed
*/
eventChanged(event:CalendarEvent): void;

/**
*  Notification fired whenever a user removes an event.
*  
*  In a calendar with a DataSource, eventRemoved() fires after the event has
*  been successfully removed from the server
* 
* 
* @param {CalendarEvent} the event that was removed
*/
eventRemoved(event:CalendarEvent): void;

/**
*  Notification fired whenever a user adds an event.
*  
*  In a calendar with a DataSource, eventAdded() fires after the event has
*  been successfully added at the server
* 
* 
* @param {CalendarEvent} the event that was added
*/
eventAdded(event:CalendarEvent): void;

/**
*  Called whenever an event is clicked on in the day, week or month views.
*  
*  By default, a dialog appears showing details for the event, and offering the ability to
*  edit events that can be edited. Return false to cancel the default action. This is a good
*  place to, for example, show a completely customized event dialog instead of the default one.
* 
* 
* @param {CalendarEvent} event that was clicked on
* @param {ViewName} view where the event's canvas was clicked
*/
eventClick(event:CalendarEvent, viewName:ViewName): boolean;

/**
*  Called whenever the close icon of an EventCanvas is clicked in the 
*  Calendar.dayView, Calendar.weekView and Calendar.timelineView views, or when
*  the Calendar.removeButton is pressed in the Calendar.eventEditor.
*  
*  Implement this method to intercept the automatic removal of data. You can return false to 
*  prevent the default action (calling Calendar.removeEvent) and instead 
*  take action of your own. For example, returning false from this method and then showing a 
*  custom confirmation dialog - if the user cancels, do nothing, otherwise
*  make a call to Calendar.removeEvent, passing the event.
* 
* 
* @param {CalendarEvent} event that was clicked on
* @param {string} view where the event was clicked on: "day", "week" or "month"
*/
eventRemoveClick(event:CalendarEvent, viewName:string): boolean;

/**
*  Called when an event is moved via dragging by a user. Return false to disallow the move.
* 
* @param {Date} new start date and time that the event is being moved to
* @param {CalendarEvent} the event as it will be after this movement
* @param {string} the name of the lane into which the event was moved
* @deprecated * in favor of Calendar.eventRepositionStop

*/
eventMoved(newDate:Date, event:CalendarEvent, newLane:string): boolean;

/**
*  Called when an event is resized with the mouse. The passed date value is the new
*  *end* date for the event, since resizing can only be performed on the bottom edge of an event
*  in normal calendar views.
* 
* @param {Date} new end date and time that event is being resized to
* @param {CalendarEvent} the event as it will be after this resize
* @deprecated * in favor of Calendar.eventResizeStop

*/
eventResized(newDate:Date, event:CalendarEvent): boolean;

/**
*  Called when a Timeline event is moved via dragging by a user. Return false to disallow the 
*  move.
* 
* @param {CalendarEvent} the event that was moved
* @param {Date} new start date of the passed event
* @param {Date} new end date of the passed event
* @param {Lane} the Lane in which this event has been dropped
* @deprecated * in favor of Calendar.eventRepositionStop

*/
timelineEventMoved(event:CalendarEvent, startDate:Date, endDate:Date, lane:Lane): boolean;

/**
*  Called when a Timeline event is resized via dragging by a user. Return false to disallow 
*  the resize.
* 
* @param {CalendarEvent} the event that was resized
* @param {Date} new start date of the passed event, after the resize
* @param {Date} new end date of the passed event, after the resize
* @deprecated * in favor of Calendar.eventResizeStop

*/
timelineEventResized(event:CalendarEvent, startDate:Date, endDate:Date): boolean;

/**
*  Selects the calendar view in the passed tab number.
* 
* 
* @param {number} the index of the tab to select
*/
selectTab(tabnum:number): void;

/**
*  Fires whenever the user changes the current date, including picking a specific date or
*  navigating to a new week or month.
* 
*/
dateChanged(): void;

/**
*  Gets the day of the week (0-6) that the mouse is currently over.
* 
* 
*/
getActiveDay(): Integer;

/**
*  Gets a date object representing the date over which the mouse is hovering for the current
*  selected view. For month view, the time will be set to midnight of the active day. For day
*  and week views, the time will be the rounded to the closest half hour relative to the mouse
*  position.
* 
*/
getActiveTime(): Date;

/**
*  Sets the range over which the timeline will display events.
*  
*  If the end parameter is not passed, the end date of the range will default to 
*  Calendar.defaultTimelineColumnSpan columns of the current 
*  Calendar.timelineGranularity following the start date.
*  
* 
* @param {Date} start of range
* @param {Date=} end of range
*/
setTimelineRange(start:Date, end?:Date): void;

/**
*  Reset the resolution, the header levels and scrollable range, of the timeline view.
*  
*  headerLevels specifies the array of HeaderLevel to show above 
*  the timeline, and the unit and unitCount parameters dictate the 
*  scrollable range (eg, passing "week" and 6 will create a timeline with a scrollable range of 
*  six weeks, irrespective of the number of columns that requires, according to the 
*  Calendar.timelineGranularity).
*  
*  If the optional granularityPerColumn parameter is passed, each column will span
*  that number of units of the granularity, which is determined from the unit of the innermost
*  of the passed headerLevels. For example, to show a span of 12 hours with inner columns that 
*  each span 15 minutes, you could pass "hour" and "minute" -based headerLevels, unit and 
*  unitCount values of "hour" and 12 respectively, and granularityPerColumn of 15.
*  
* 
* @param {Array<Partial<HeaderLevel>>} the header levels to show in the timeline
* @param {TimeUnit} the time unit to use when calculating the range of the timeline
* @param {Integer} the count of the passed unit that the timeline should span
* @param {Integer=} how many units of the granularity (the unit of the      innermost headerLevel) should each column span? The default is 1.
*/
setResolution(headerLevels:Array<Partial<HeaderLevel>>, unit:TimeUnit, unitCount:Integer, granularityPerColumn?:Integer): void;

/**
*  Returns the length of the passed CalendarEvent in the passed 
*  TimeUnit. If unit isn't passed, returns the length of the event 
*  in milliseconds.
*  
* 
* @param {CalendarEvent} the event to get the length of
* @param {TimeUnit=} the time unit to return the length in, milliseconds if not passed
*/
getEventLength(event:CalendarEvent, unit?:TimeUnit): void;

/**
*  Notification called whenever the drop position of an event being drag-moved changes.
*  
*  The newEvent parameter represents the event as it will be after the move, 
*  including CalendarEvent.startDate and CalendarEvent.endDate dates
*  and CalendarEvent.lane and CalendarEvent.sublane where 
*  applicable.
*  
*  Return false to prevent the default action, of positioning the drag canvas to the newEvent.
*  
* 
* @param {CalendarEvent} the event that's being moved
* @param {CalendarEvent} the event as it would be if dropped now
*/
eventRepositionMove(event:CalendarEvent, newEvent:CalendarEvent): boolean;

/**
*  Notification called when an event being drag-moved is dropped.
*  
*  The newEvent parameter represents the event as it will be after the move, 
*  including CalendarEvent.startDate and CalendarEvent.endDate dates
*  and CalendarEvent.lane and CalendarEvent.sublane where 
*  applicable.
*  
*  Return false to prevent the default action, of actually 
*  Calendar.updateCalendarEvent the event.
*  
* 
* @param {CalendarEvent} the event that's about to be moved
* @param {CalendarEvent} the event as it will be, unless this method returns false
* @param {object=} additional custom values associated with the event
*/
eventRepositionStop(event:CalendarEvent, newEvent:CalendarEvent, customValues?:object): boolean;

/**
*  Notification called on each resize during an event drag-resize operation.
*  
*  The newEvent parameter represents the event as it will be after the resize.
*  
*  Return false to prevent the default action, of resizing the drag canvas to the newEvent.
*  
* 
* @param {CalendarEvent} the event that's being drag-resized
* @param {CalendarEvent} the event as it would be if dropped now
*/
eventResizeMove(event:CalendarEvent, newEvent:CalendarEvent): boolean;

/**
*  Notification called when an event drag-resize operation completes.
*  
*  The newEvent parameter represents the event as it will be after the move.
*  
*  Return false to prevent the default action, of actually 
*  Calendar.updateCalendarEvent the event.
*  
* 
* @param {CalendarEvent} the event that's about to be resized
* @param {CalendarEvent} the event as it will be, unless this method returns false
* @param {object=} additional custom values associated with the event
*/
eventResizeStop(event:CalendarEvent, newEvent:CalendarEvent, customValues?:object): boolean;

/**
*  Callback fired when the mouse is clicked in a background-cell, ie, one without an 
*  event.
* 
* 
* @param {Date} start datetime of the selected slot
* @param {Date} end datetime of the selected slot
*/
backgroundClick(startDate:Date, endDate:Date): boolean;

/**
*  Callback fired when the mouse button is depressed over a background-cell, ie, one 
*  without an event. Return false to cancel the default behavior of allowing sweep
*  selection via dragging.
* 
* 
* @param {Date} start datetime of the selected slot
*/
backgroundMouseDown(startDate:Date): boolean;

/**
*  Notification method fired when the mouse button is released over a background-cell, ie, 
*  one without an event. Return false to cancel the default behavior of showing a dialog
*  to add a new event with the passed dates.
* 
* 
* @param {Date} the datetime of the slot where the mouse button was depressed
* @param {Date} the datetime of the slot where the mouse button was released
*/
backgroundMouseUp(startDate:Date, endDate:Date): boolean;



}
/**
* Creates a new Calendar
* 
* @param typeCheckedProperties {Partial<Calendar>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Calendar} The new instance
*/
class Calendar{
static create(typeCheckedProperties? : Partial<Calendar>, uncheckedProperties?: Object): Calendar;







}

/**
* CalendarView is a base class, extended by the various views available in a 
*  Calendar.
*/
interface CalendarView extends ListGrid {
/**
* The Calendar this view is in.
* @type {Calendar}
* @default null
*/
calendar?: Calendar;

/**
* The name of this view, used to identify it in the CalendarView.calendar.
* @type {string}
* @default null
*/
viewName?: string;

/**
* Should EventCanvas instances be reused when visible events change?
* @type {boolean}
* @default true
*/
useEventCanvasPool?: boolean;

/**
* If specified, overrides Calendar.eventStyleName and dictates the CSS style to 
*  use for events rendered in this view. Has no effect on events that already have a 
*  CalendarEvent.styleName.
* @type {CSSStyleName}
* @default null
*/
eventStyleName?: CSSStyleName;

/**
* CSS class applied to the CalendarView.eventDragCanvas.
* @type {CSSStyleName}
* @default "eventDragCanvas"
*/
eventDragCanvasStyleName?: CSSStyleName;

/**
* Canvas displayed while dragging or resizing an event in this view and styled
*  according to CalendarView.eventDragCanvasStyleName.
* @type {Canvas}
* @default null
*/
eventDragCanvas?: Canvas;



/**
*  Returns true if this view is the currently selected view in the parent calendar.
* 
*/
isSelectedView(): boolean;

/**
*  Returns true if this is the Calendar.timelineView, false otherwise.
* 
*/
isTimelineView(): boolean;

/**
*  Returns true if this is the Calendar.dayView, false otherwise.
* 
*/
isDayView(): boolean;

/**
*  Returns true if this is the Calendar.weekView, false otherwise.
* 
*/
isWeekView(): boolean;

/**
*  Returns true if this is the Calendar.monthView, false otherwise.
* 
*/
isMonthView(): boolean;

/**
*  Rebuild this CalendarView, including re-fetching its data as necessary. To avoid 
*  re-fetching the data, pass 'false' to this method, or call 
*  CalendarView.refreshEvents instead.
* 
* @param {boolean=} If false, prevents data from bing refreshed.
*/
rebuild(refreshData?:boolean): void;

/**
*  Clear, recalculate and redraw the events for the current range, without causing a fetch.
* 
*/
refreshEvents(): void;

/**
*  Move the viewport of this CalendarView to the start of its scrollable range.
* 
*/
scrollToStart(): void;

/**
*  Move the viewport of this CalendarView to the end of its scrollable range.
* 
*/
scrollToEnd(): void;



}
/**
* Creates a new CalendarView
* 
* @param typeCheckedProperties {Partial<CalendarView>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {CalendarView} The new instance
*/
class CalendarView{
static create(typeCheckedProperties? : Partial<CalendarView>, uncheckedProperties?: Object): CalendarView;







}

/**
* This object cannot be used; it exists for documentation purposes only as a place to put
*  documentation for callback methods, such as the callback for Callbacks.DSCallback.
*/
interface Callbacks extends Class {


/**
*  A notification Callback fired by the TabIndexManager to allow 
*  application code to react to the numeric tab-index of some registered target being
*  modified.
* 
* @param {string} the ID String passed to TabIndexManager.addTarget when          the callback was registered.
*/
TabIndexUpdatedCallback(ID:string): void;

/**
*  A Callback fired by the TabIndexManager when application code or user
*  action attempts to synthetically shift focus to some registered target. 
*  See TabIndexManager.shiftFocus.
*  
*  A typical implementation will shift focus to some native element associated
*  with the registered target, or if this is not currently possible, return false.
* 
* 
* @param {string} The ID String passed to TabIndexManager.addTarget when          the callback was registered.
*/
ShiftFocusCallback(ID:string): boolean;

/**
*  Callback executed when a PrintCanvas.setHTML is rendered into
*  a printCanvas.
* 
* 
* @param {PrintCanvas} The canvas being printed.
*/
PrintCanvasCallback(printPreview:PrintCanvas): void;

/**
*  Callback required for the property DataSourceField.getFieldValue.
* 
* 
* @param {object | XMLElement} record object selected from web service response data                   by OperationBinding.recordXPath
* @param {any} default value derived by the method described in          DataSourceField.valueXPath
* @param {DataSourceField} DataSourceField definition
* @param {FieldName} name of the DataSource field
*/
GetFieldValueCallback(record:object | XMLElement, value:any, field:DataSourceField, fieldName:FieldName): void;

/**
*  Callback fired when DataSource methods that send DSRequests complete (such as DataSource.fetchData).
* 
* 
* @param {DSResponse} a DSResponse instance with metadata about the returned data
* @param {any} data returned to satisfy the DataSource request. See the     dataSourceOperations topic for expected results     for each type of DataSource operation
* @param {DSRequest} the DSRequest that was sent. You can use    DSRequest.clientContext to track state during the server turnaround.
*/
DSCallback(dsResponse:DSResponse, data:any, dsRequest:DSRequest): void;

/**
*  A Callback fired when DataSource.hasFile completes.
* 
* 
* @param {DSResponse} A DSResponse instance with metadata about the                returned data.
* @param {boolean} Whether the file exists.
* @param {DSRequest} The DSRequest that was sent.
*/
HasFileCallback(dsResponse:DSResponse, data:boolean, dsRequest:DSRequest): void;

/**
*  A Callback fired when Sound.load completes.
*  
* 
*/
CanPlayCallback(): void;

/**
*  A Callback fired when Sound.play completes.
*  
* 
*/
PlaybackCompleteCallback(): void;

/**
*  A Callback fired when DataSource.hasFileVersion completes.
* 
* 
* @param {DSResponse} A DSResponse instance with metadata about the                returned data.
* @param {boolean} Whether the file version exists.
* @param {DSRequest} The DSRequest that was sent.
*/
HasFileVersionCallback(dsResponse:DSResponse, data:boolean, dsRequest:DSRequest): void;

/**
*  Callback fired when DataSource.getFile completes.
* 
* 
* @param {DSResponse} A DSResponse instance with metadata about the                returned data.
* @param {string} The file contents, or null if there was an error (such as file                not found).
* @param {DSRequest} The DSRequest that was sent.
*/
GetFileCallback(dsResponse:DSResponse, data:string, dsRequest:DSRequest): void;

/**
*  Callback fired when DataSource.getFileVersion completes.
* 
* 
* @param {DSResponse} A DSResponse instance with metadata about the                returned data.
* @param {string} The file contents, or null if there was an error (such as file                not found).
* @param {DSRequest} The DSRequest that was sent.
*/
GetFileVersionCallback(dsResponse:DSResponse, data:string, dsRequest:DSRequest): void;

/**
*  Generic callback interface.
* 
* 
*/
Function(): void;

/**
*  Generic callback interface.
* 
* 
* @param {DataSource} Client only Data Source.
*/
ClientOnlyDataSourceCallback(clientOnly:DataSource): void;

/**
*  A Callback to evaluate when a screen is loaded via RPCManager.loadScreen.
* 
* 
* @param {Canvas=} The last top-level component loaded
* @param {RPCResponse=} * @param {Map<any>=} A collection of suppressed globals.
*/
LoadScreenCallback(screen?:Canvas, rpcResponse?:RPCResponse, suppressedGlobals?:Map<any>): void;

/**
*  A Callback to evaluate when an RPCRequest completes.
*  
*  Parameters passed to this callback are:
*  
*  
* - rpcResponse: an RPCResponse encapsulating the server response to your
*    request
*  
* - data: just the "data" property from the RPCResponse, for convenience
*  
* - rpcRequest: the RPCRequest that was sent. You can use
*    RPCRequest.clientContext to track state during the server turnaround.
*  
*  For example, to take the data returned by the server and display it in a previously created
*  ListGrid with the ID "myGrid":
*  
*    isc.RPCManager.send("getData", "myGrid.setData(data)");
*  
*  Or
*  
*    isc.RPCManager.send("getData", function (rpcResponse, data, rpcRequest) { 
*                     myGrid.setData(data)
*    });
*  
* 
* 
* @param {RPCResponse} response a RPCResponse encapsulating the server response to your request
* @param {any} rawData The "data" property from the RPCResponse, for convenience. The data can also be obtained via {@link RPCResponse#getDataAsMap()}, {@link RPCResponse#getDataAsString()}, or {@link RPCResponse#getDataAsObject()}, depending on the type of data that is expected to be returned from the server.
* @param {RPCRequest} the RPCRequest that was sent.
*/
RPCCallback(response:RPCResponse, rawData:any, request:RPCRequest): void;

/**
*  Callback to fire when a queue of requests sent via 
*  {@link com.smartgwt.client.rpc.RPCManager#sendQueue(RPCQueueCallback)} returns.
*  Note that the Array of RPCResponses passed to this callback
*  will actually be DSResponse objects for any requests that were actually 
*  DSRequests. 
*  
* 
* 
* @param {Array<Partial<RPCResponse>>} array of responses returned from the sent queue of requests
*/
RPCQueueCallback(response:Array<Partial<RPCResponse>>): void;

/**
*  Callback for RPCManager.exportImage.
* 
* @param {string} image data from the server, in base64 format
*/
ExportImageCallback(imageData:string): void;

/**
*  Callback fired with the PaletteNode obtained asynchronously.
* 
* @param {Array<Partial<PaletteNode>>} an array of PaletteNodes
*/
PaletteNodeCallback(paletteNodes:Array<Partial<PaletteNode>>): void;

/**
*  Callback for DrawPane.getDataURL.
* 
* @param {string} the data URL
*/
DataURLCallback(dataURL:string): void;

/**
*  A Callback to evaluate when form validation completes.
*  
*  The available parameters are:
* 
* @param {Map<any>} null if validation succeeded for all fields, or an object mapping field names to the associated errors, for those fields that failed validation.
*/
ValidationStatusCallback(errorMap:Map<any>): void;

/**
*  Callback required for the property Validator.condition and
*  ValidatorDefinition.condition.
* 
* @param {DataSourceField | ListGridField | FormItem} FormItem, DataSourceField or                      ListGridField on which this validator was                      declared. NOTE: FormItem will not                      be available during a save performed without a                      form (eg programmatic save) or if the field                      is not available in the form.
* @param {Validator} Validator declaration from eg               DataSourceField.validators.
* @param {any} value to validate
* @param {object} Field values for record being validated.
* @param {object} Object containing extra context which may be useful to the                  condition function. Contains the following properties:           component: the DynamicForm or ListGrid being validated           rowNum: the row number of the cell being validated (only if component is a ListGrid)           colNum: the column number of the cell being validated (only if component is a ListGrid)
*/
ValidatorConditionCallback(item:DataSourceField | ListGridField | FormItem, validator:Validator, value:any, record:object, additionalContext:object): boolean;

/**
*  Callback required for the property ValidatorDefinition.action.
* 
* 
* @param {boolean} The result of the validator. The value will be null if               the validator was skipped because of conditional criteria.
* @param {DataSourceField | FormItem} FormItem or DataSourceField on which this                     validator was declared. NOTE: FormItem will not                      be available during a save performed without a                      form (eg programmatic save) or if the field                      is not available in the form.
* @param {Validator} Validator declaration from eg               DataSourceField.validators.
* @param {Record} Record that was validated
* @param {DataBoundComponent} The DataBoundComponent holding the item such                    DynamicForm or ListGrid.
*/
ValidatorActionCallback(result:boolean, item:DataSourceField | FormItem, validator:Validator, record:Record, component:DataBoundComponent): void;

/**
*  Callback for DateRangeDialog.askForRange.
* 
* @param {Criterion} criterion representing the selected range
*/
DateRangeCallback(criterion:Criterion): void;

/**
*  A Callback called when the move completes.
* 
* 
* @param {boolean} true if the animation was cut short. To quit an animation                early, simply call the non-animated version of the same                API, so for example call Canvas.hide to cut short                an animation from Canvas.animateHide already in                progress.
*/
AnimationCallback(earlyFinish:boolean): void;

/**
*  Callback to execute after the section has been shown.
* 
*/
ShowSectionCallback(): void;

/**
*  Callback to execute after the section has been expanded.
* 
*/
ExpandSectionCallback(): void;

/**
*  Callback to execute after the section has been hidden.
* 
*/
HideSectionCallback(): void;

/**
*  Callback to execute after the section has been collapsed.
* 
*/
CollapseSectionCallback(): void;

/**
*  Callback executed when a message is sent to a channel that you have
*  Messaging.subscribe to.
* 
* @param {any} data contained in the message
*/
MessagingCallback(data:any): void;

/**
*  A Callback to evaluate when an {Process.loadProcess} method completes.
*  
*  Loaded process passed as a parameter to this callback are:
* 
* 
* @param {DSResponse} a DSResponse instance with metadata about the returned data
* @param {Process} */
ProcessCallback(dsResponse:DSResponse, process:Process): void;



}
/**
* Creates a new Callbacks
* 
* @param typeCheckedProperties {Partial<Callbacks>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Callbacks} The new instance
*/
class Callbacks{
static create(typeCheckedProperties? : Partial<Callbacks>, uncheckedProperties?: Object): Callbacks;







}

/**
* Button that cancels any changes in the form, by calling DynamicForm.cancelEditing 
*  when clicked.
*  See DynamicForm.cancelEditing for details on what happens when a form editing is 
*  cancelled.
*/
interface CancelItem extends ButtonItem {
/**
* CancelItems show a title of "Cancel" by default. May be overridden.
* @type {string}
* @default "Cancel"
*/
title?: string;





}
/**
* Creates a new CancelItem
* 
* @param typeCheckedProperties {Partial<CancelItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {CancelItem} The new instance
*/
class CancelItem{
static create(typeCheckedProperties? : Partial<CancelItem>, uncheckedProperties?: Object): CancelItem;







}

/**
* Base class for all SmartClient visual components (except FormItem).
*  
*  Canvas provides:
*  
*  
* - basic visual lifecycle support - creation and destruction of HTML via
*    Canvas.draw and Canvas.clear, visibility via
*    Canvas.show and Canvas.hide, z-layering via
*    Canvas.bringToFront and Canvas.sendToBack.
*  
* - consistent cross-browser Canvas.moveTo,
*    Canvas.resizeTo and Canvas.getScrollHeight, with
*    automatic compensation for CSSStyleName.
*  
* - clipping, scrolling and overflow management of content via Canvas.overflow
*  
* - consistent cross-browser Canvas.keyPress and Canvas.mouseDown
*    events, including mobileDevelopment to mouse events
*  
* - built-in drag and drop capabilities including Canvas.canDragReposition, 
*    Canvas.canDragResize, Canvas.canDragScroll
*    and Canvas.snapToGrid behavior.
*  
* - the ability to either contain Canvas.contents or
*    Canvas.children, including 
*    Canvas.snapTo and 
*    Canvas.percentSource for children. For more advanced layout
*    capabilities, see Layout.
*  
* - various other details like Canvas.cursor,
*    Canvas.showClickMask, Canvas.animateMove,
*    Canvas.ariaRole, and
*    Canvas.locateChildrenBy for automatedTesting.
*  
*/
interface Canvas extends BaseWidget {
/**
* Configures where the Opacity filter is used for IE6-8.
*  
*  With the default of null, opacity filters are used unless
*  Canvas.neverUseFilters has been set. When set explicitly to true,
*  opacity filters are used even if neverUseFilters is true.
*  
*  See IEFilters for background.
* @type {boolean}
* @default null
*/
useOpacityFilter?: boolean;

/**
* Global identifier for referring to a widget in JavaScript. The ID property is optional if
*  you do not need to refer to the widget from JavaScript, or can refer to it indirectly
*  (for example, by storing the reference returned by Class.create).
*  
*  An internal, unique ID will automatically be created upon instantiation for any canvas
*  where one is not provided.
*  
*  The ID property should be unique in the global scope. If window[ID]
*  is already assigned to something else a warning will be logged using the developer console,
*  and the existing reference will be replaced, calling Canvas.destroy on the
*  previous object if it is a SmartClient Class instance.
*  
*  Automatically generated IDs will be unique as long as the canvases they refer to remain
*  active - once a canvas with an automatically generated ID has been destroyed, its ID may be
*  reused for the next canvas created with no explicitly specified ID.
* @type {GlobalId}
* @default null
*/
ID?: GlobalId;

/**
* Optional name for the canvas, which can later be used to reference it. Need not be
*  globally unique, but should be unique within the Canvas.parentCanvas to get
*  defined results for Layout.getMember and Layout.getMemberNumber.
* @type {Identifier}
* @default null
*/
name?: Identifier;

/**
* 
*  If true, this canvas will draw itself immediately after it is created.
*  
*  Note that you should turn this OFF for any canvases that are provided as children
*  of other canvases, or they will draw initially, then be clear()ed and drawn again when
*  added as children, causing a large performance penalty.
*  
*  For example, the following code is incorrect and will cause extra draw()s:
*  
*  
*    isc.Layout.create({
*      members : [
*        isc.ListGrid.create()
*      ]
*    });
*  
*  It should instead be:
*  
*    isc.Layout.create({
*      members : [
*        isc.ListGrid.create({ autoDraw: false })
*      ]
*    });
*  
*  In order to avoid unwanted autoDrawing systematically, it is recommend that you call
*  isc.setAutoDraw immediately after SmartClient is loaded
*  and before any components are created, then set autoDraw:true or call
*  draw() explicitly to draw components.
*  
*  Otherwise, if the global setting for autoDraw remains true, you must set
*  autoDraw:false, as shown above, on every component in your application that
*  should not immediately draw: all Canvas children, Layout members, Window items, Tab
*  panes, etc, however deeply nested. Forgetting to set autoDraw:false will result in one
*  more clear()s - these are reported on the Results tab of the
*  debugging, and can be tracked to individual components by
*  using the "clears" log category in the Developer Console.
*  
* @type {boolean}
* @default true
*/
autoDraw?: boolean;

/**
* This Canvas's immediate parent, if any.
*  
*  Can be initialized, but any subsequent manipulation should be via
*  Canvas.addChild and Canvas.removeChild calls
*  on the parent.
* @deprecated * As of Smartclient version 9.1, deprecated in favor of Canvas.parentCanvas       and Canvas.getParentCanvas

* @type {Canvas}
* @default null
*/
parentElement?: Canvas;

/**
* This Canvas's immediate parent, if any.
*  
*  Can be initialized, but any subsequent manipulation should be via
*  Canvas.addChild and Canvas.removeChild calls
*  on the parent. The parent Canvas should be fetched using
*  Canvas.getParentCanvas.
* @type {Canvas}
* @default null
*/
parentCanvas?: Canvas;

/**
* The top-most Canvas (i.e., not a child of any other Canvas), if any, in this widget's
*  containment hierarchy.
* @type {Canvas}
* @default null
*/
topElement?: Canvas;

/**
* This Canvas's "master" (the Canvas to which it was added as a peer), if any.
* @deprecated * In favor or Canvas.getMasterCanvas as of SmartClient release 9.1

* @type {Canvas}
* @default null
*/
masterElement?: Canvas;

/**
* Array of all Canvii that are immediate children of this Canvas.
*  
*  Use Canvas.addChild and Canvas.removeChild to add and remove children after a Canvas
*  has been created/drawn.
* @type {Array<Partial<Canvas>>}
* @default null
*/
children?: Array<Partial<Canvas>>;

/**
* Array of all Canvii that are peers of this Canvas.
*  
*  Use Canvas.addPeer and Canvas.removePeer to add and remove peers after a Canvas
*  has been created/drawn.
* @type {Array<Partial<Canvas>>}
* @default null
*/
peers?: Array<Partial<Canvas>>;

/**
* This initialization property allows developers to create a canvas using the
*  Class.addAutoChild method, and have it added as a child to some other
*  component. This property may be set to the childName of another
*  already-created auto-child, or "none" to cause the component to be 
*  created without being added as a child to any other widget.
*  If unset, the canvas will be added as a child to the component on which
*  addAutoChild(...) was called.
*  
*  See autoChildren for an overview of the autoChild subsystem.
* @type {string}
* @default null
*/
autoParent?: string;

/**
* If specified as a pointer to an HTML element present in the DOM, this canvas will be
*  rendered inside that element on Canvas.draw.
*  
*  NOTES:
*  This feature is intended for temporary integration with legacy page architectures only;
*  the native browser's reaction to DOM insertion is unspecified and unsupported. For
*  consistent cross-browser layout and positioning semantics, use Canvas parents
*  (especially Layouts) and use absolute positioning at top level.
*  
*  Persistence of htmlElement: If Canvas.htmlPosition is set to "replace"
*  the htmlElement will be removed from the DOM when the canvas is drawn - therefore the
*  htmlElement attribute will be cleared at this time.
*  Otherwise if a Canvas is clear()d and then draw()n again it will
*  be rendered inside the same htmlElement.
*  If a Canvas is added as a child to Canvas parent, its htmlElement will be dropped.
*  
*  Canvas.position should typically be set to "relative" if the widget
*  is to be rendered inline within a standard page.
* @type {DOMElement}
* @default null
*/
htmlElement?: DOMElement;

/**
* If Canvas.htmlElement is specified, this attribute specifies the position where
*  the canvas should be inserted relative to the htmlElement in the DOM.
* @type {DrawPosition}
* @default "afterBegin"
*/
htmlPosition?: DrawPosition;

/**
* If Canvas.htmlElement is specified, should this canvas initially be drawn
*  at the same dimensions as the htmlElement?
*  Note: setting this property will not force the canvas to resize if the element
*  subsequently resizes (for example due to page reflow).
* @type {boolean}
* @default null
*/
matchElement?: boolean;

/**
* Absolute or relative, corresponding to the "absolute" (with respect to parent) or
*  "relative" (with respect to document flow) values for the CSS position attribute.
*  
*  Setting position:"relative" enables SmartClient components to be embedded
*  directly into the native HTML flow of a page, causing the component to be rendered
*  within an existing DOM structure.
*  This attribute should only be set to "relative" on a top level component
*  (a component with no Canvas.getParentCanvas).
*  
*  There are 2 ways to embed relatively positioned canvases in the DOM - by default the
*  component will be written out inline when it gets Canvas.draw. For example
*  to embed a canvas in an HTML table you could use this code:
*  
*  &lt;table&gt;
*   &lt;tr&gt;
*    &lt;td&gt;
*     &lt;script&gt;
*      isc.Canvas.create({autoDraw:true, backgroundColor:"red", position:"relative"});
*     &lt;/script&gt;
*    &lt;td&gt;
*   &lt;/tr&gt;
*  &lt;/table&gt;
*  
*  Alternatively you can make use of the Canvas.htmlElement attribute.
*  
*  Relative positioning is intended as a short-term integration scenario while incrementally
*  upgrading existing applications.
*  Note that relative positioning is not used to manage layout within SmartClient components -
*  instead the Layout class would typically be used.
*  For best consistency and flexibility across browsers, all SmartClient layout managers
*  use absolute positioning.
*  
*  For canvases with a specified Canvas.htmlElement, this attribute defaults to
*  "relative". In all other cases the default value will be
*  "absolute". Note that if you plan to call Canvas.setHtmlElement after 
*  init, you will need to set this value to "relative" explicitly.
* @type {Positioning}
* @default null
*/
position?: Positioning;

/**
* Number of pixels the left side of the widget is offset to the right from its default
*  drawing context (either its parent's topleft corner, or the document flow, depending on
*  the value of the Canvas.position property).
*  
*  Can also be set as a percentage, specified as a String ending in '%', eg, "50%". In
*  this case the top coordinate is considered as a percentage of the specified width of
*  the Canvas.getParentCanvas.
* @type {number | string}
* @default "0"
*/
left?: number | string;

/**
* Number of pixels the top of the widget is offset down from its default drawing context
*  (either its parent's top-left corner, or the document flow, depending on the value of
*  the Canvas.position property).
*  
*  Can also be set as a percentage, specified as a String ending in '%', eg, "50%". In
*  this case the top coordinate is considered as a percentage of the specified height of
*  the Canvas.getParentCanvas.
* @type {number | string}
* @default "0"
*/
top?: number | string;

/**
* If set, overrides the global Canvas.defaultPageSpace.
* @type {Integer}
* @default null
*/
leavePageSpace?: Integer;

/**
* Size for this component's horizontal dimension.
*  
*  Can be a number of pixels, or a percentage like "50%". Percentage sizes are resolved to
*  pixel values as follows:
*  
*  
* - If a canvas has a specified Canvas.percentSource, sizing will be
*    a percentage of the size of that widget (see also Canvas.percentBox).
*  
* - Otherwise, if a canvas has a Canvas.getMasterCanvas, and
*    Canvas.snapTo is set for the widget, sizing will be a percentage of
*    the size of that widget (see also Canvas.percentBox).
*  
* - Otherwise if this is a child of some other canvas, percentages will be based on the
*    inner size of the Canvas.getParentCanvas's viewport.
*  
* - Otherwise, for top level widgets, sizing is calculated as a percentage of page size.
*  
*  Note that if a Canvas.maxWidth or Canvas.minWidth are specified
*  (or Canvas.maxHeight / Canvas.minHeight for heights), these
*  properties act as explicit pixel limits on the canvas' size. For example, a canvas
*  with Canvas.maxWidth set to 500, and width specified as 
*  "100%" will not render larger than 500 pixels in width even if there is more
*  space available in the parent canvas or percentSource.
*  
*  Layout may specially interpret percentage sizes on their children,
*  and also allow "*" as a size.
*  
*  Note that if Canvas.overflow is set to "visible", this size is a
*  minimum, and the component may overflow to show all content and/or children.
*  
*  If trying to establish a default width for a custom component, set
*  Canvas.defaultWidth instead.
* @type {number | string}
* @default null
*/
width?: number | string;

/**
* Size for this component's vertical dimension.
*  
*  Can be a number of pixels, or a percentage like "50%". See documentation for
*  Canvas.width for details on how percentage values are resolved actual size.
*  
*  Note that if Canvas.overflow is set to "visible", this size is a
*  minimum, and the component may overflow to show all content and/or children.
*  
*  If trying to establish a default height for a custom component, set
*  Canvas.defaultHeight instead.
* @type {number | string}
* @default null
*/
height?: number | string;

/**
* For custom components, establishes a default width for the component.
*  
*  For a component that should potentially be sized automatically by a Layout, set this
*  property rather than Canvas.width directly, because Layouts regard a width setting as
*  an explicit size that shouldn't be changed.
* @type {Integer}
* @default "100"
*/
defaultWidth?: Integer;

/**
* For custom components, establishes a default height for the component.
*  
*  For a component that should potentially be sized automatically by a Layout, set this
*  property rather than Canvas.height directly, because Layouts regard a height setting as
*  an explicit size that shouldn't be changed.
* @type {Integer}
* @default "100"
*/
defaultHeight?: Integer;

/**
* Minimum width available to this Canvas.
*  
*  The minWidth and Canvas.minHeight settings apply to:
*  
*  
* - For a canvas being managed as a member of a Layout, the minimum size the
*    layout should apply to the canvas.
*  
* - For a canvas with a width or height specified as a percent value, a minimum
*    numeric pixel value to limit how large the canvas is sized.
*  
* - determining size for a Canvas in a CanvasItem (minHeight only)
*  
* - end user Canvas.canDragResize
*  
* - minimum size when using Overflow "visible" outside of a Layout - minimum
*    size will be the greater of this setting or the minimum size needed to make all
*    content visible
*  
*  
*  Minimum sizes do not apply in various other circumstances where sizes are being
*  determined, such as ListGrid.showRecordComponents.
*  
*  See also Layout.minMemberLength as a way of establishing minimum sizes along
*  the length axis for all members of a Layout with a single setting.
* @type {number}
* @default null
*/
minWidth?: number;

/**
* Minimum height available to this Canvas. Minimum sizes do not apply to all
*  situations. See Canvas.minWidth for details.
* @type {number}
* @default null
*/
minHeight?: number;

/**
* Maximum width available to this Canvas. 
*  
*  The maxWidth and Canvas.maxHeight settings apply to:
*  
*  
* - For a canvas being managed as a member of a Layout, the maximum size the
*    layout should apply to the canvas.
*  
* - For a canvas with a width or height specified as a percent value, a maximum
*    numeric pixel value to limit how large the canvas is sized.
*  
* - determining size for a Canvas in a CanvasItem (maxHeight only)
*  
* - end user Canvas.canDragResize
*  
*  Maximum sizes do not apply in various other circumstances where sizes are being
*  determined, such as ListGrid.showRecordComponents.
* @type {number}
* @default "10000"
*/
maxWidth?: number;

/**
* Maximum height available to this Canvas. See Canvas.maxWidth for details of behavior.
* @type {number}
* @default "10000"
*/
maxHeight?: number;

/**
* Minimum width that this Canvas can be resized to by a user. Actual limit will be maximum
*  of dragMinWidth and Canvas.minWidth.
*  
*  Note that a Canvas with overflow:"visible" has an implicit minimize size based on it's
*  contents.
*  
*  Note that dragMinWidth affects only user-initiated drag resizes. To set
*  the minimum width of a Canvas embedded in a Layout, you can set +{minWidth}, or
*  Layout.minMemberLength to constrain the minimum size along the length axis of all
*  members of the Layout.
* @type {number}
* @default "10"
*/
dragMinWidth?: number;

/**
* Minimum height that this Canvas can be resized to by a user. See Canvas.dragMinWidth
*  for details of behavior.
* @type {number}
* @default "10"
*/
dragMinHeight?: number;

/**
* Maximum width that this Canvas can be resized to by a user. Actual limit will be minimum
*  of dragMaxWidth and Canvas.maxWidth.
* @type {number}
* @default "10000"
*/
dragMaxWidth?: number;

/**
* Sets maximum height that this Canvas can be resized to by a user. Actual limit will be
*  minimum of dragMaxHeight and Canvas.maxHeight.
* @type {number}
* @default "10000"
*/
dragMaxHeight?: number;

/**
* This is an advanced setting. If set to true, when a widget is
*  Canvas.hide, the widget's handle will be resized such that
*  it takes up no space, in addition to having its css visibility
*  property set to "hidden".
*  
*  In addition to preventing the size of this widget from impacting the
*  Canvas.getScrollWidth of any parent widget while hidden, this
*  setting works around a native bug observed in Internet Explorer 10, whereby
*  an &lt;IFRAME&gt; element with visibility set to hidden can
*  cause rendering problems, if the HTML loaded by the &lt;IFRAME&gt;
*  contains a &lt;frameset&gt;. In this case the browser may
*  refuse to draw other elements at the same coordinates with a lower z-index than
*  the hidden frame. Setting this property to true works around
*  this problem for cases where an &lt;IFRAME&gt; containing a
*  &lt;frameset will be rendered out, for example in an
*  HTMLFlow with contentsType set to "page".
* @type {boolean}
* @default false
*/
shrinkElementOnHide?: boolean;

/**
* If set to true, the widget's parent (if any) will automatically be shown whenever the
*    widget is shown.
* @type {boolean}
* @default false
*/
autoShowParent?: boolean;

/**
* Controls widget visibility when the widget is initialized. See Visibility type for
*  details.
*  
*  Specifying "visible" sets the CSS visiblity to "visible", forcing a child to be visible
*  even if the parent is hidden. Not supported for use with SmartClient layouts,
*  scrolling or auto-sizing but may be useful when working with third-party or legacy
*  DOM layout systems.
*  
*  Note that if Canvas.hideUsingDisplayNone is set for a hidden ancestor, setting
*  visibility will have no effect at all until that ancestor becomes visible.
* @type {Visibility}
* @default "inherit"
*/
visibility?: Visibility;

/**
* When this widget is hidden (see Canvas.visibility and Canvas.hide),
*  should display:none be applied to the Canvas.getOuterElement?
*  
*  This setting is not supported for general use, but in certain cases, it has been shown
*  that display:none is a work-around for browser bugs involving burn-through of iframes or
*  plugins, where the content of the iframe or plugin may still be visible despite the
*  containing widget being hidden.
* @type {boolean}
* @default false
*/
hideUsingDisplayNone?: boolean;

/**
* Whether native drag selection of contained text is allowed within this Canvas.
*  
*  Note that setting this property to false will not avoid text selection
*  which is initiated outside this Canvas from continuing into this Canvas, even if text
*  selection began in another Canvas.
* @type {boolean}
* @default false
*/
canSelectText?: boolean;

/**
* The CSS class applied to this widget as a whole.
* @deprecated * In favor or Canvas.styleName as of SmartClient release 5.5

* @type {CSSStyleName}
* @default "normal"
*/
className?: CSSStyleName;

/**
* The CSS class applied to this widget as a whole.
* @type {CSSStyleName}
* @default "normal"
*/
styleName?: CSSStyleName;

/**
* The CSS class to apply when printing this widget. If unset, falls back to the
*  Canvas.styleName.
* @type {CSSStyleName}
* @default null
*/
printStyleName?: CSSStyleName;

/**
* The contents of a canvas or label widget. Any HTML string is acceptable.
* @type {HTMLString}
* @default "&amp;nbsp;"
*/
contents?: HTMLString;

/**
* Dynamic contents allows the contents string to be treated as a simple but powerful
*  template. When this attribute is set to true, JavaScript expressions may be embedded 
*  within the contents string, using the format:
*  &#36;{[JavaScript to evaluate]}.
*  
*  For example, to include the current date in a templated message, 
*  canvas.contents could be set to:
*  "Today's date is &lt;b&gt;&#36;{new Date().toUSShortDate()}&lt;/b&gt;"
*  
*  Embedded expressions will be evaluated when the canvas is drawn or redrawn, and
*  the result of the evaluated expression will be displayed to the user. If the
*  expression does not evaluate to a String, the toString() representation
*  of the returned object will be displayed automatically
*  
*  Dynamic expressions are evaluated in the scope of the canvas displaying the content, 
*  so the this keyword may be used within your expression to refer to the 
*  canvas. Developers may also explicitly supply values for variables to be used within
*  the evaluation via the Canvas.dynamicContentsVars property.
*  
*  Notes:
*  
* - Calling markForRedraw() on the canvas will evaluate any embedded expressions.
*  
* - Multiple such expressions may be embedded within the contents string for a component.
*  
* - If an error occurs during evaluation, a warning is logged
*    to the debugging and the error string will be embedded in place 
*    of the expected value in the Canvas.
*  
*  
* @type {boolean}
* @default false
*/
dynamicContents?: boolean;

/**
* An optional map of name:value parameters that will be available within the scope of the
*  dynamicContents evaluation. For example - if you have e.g:
*  
*  Canvas.create({
*   dynamicContents: true,
*   dynamicContentsVars: {
*     name: "Bob"
*   },
*   contents: "hello &#36;{name}"
*  });
*  
*  The above will create a canvas with contents hello Bob. You can add, remove, and
*  change values in the dynamicContentsVars object literal, just call
*  markForRedraw() on the canvas to have the dynamicContents template re-evaluated.
*  
*  Note that this is always available inside a dynamic contents string and points to
*  the canvas instance containing the dynamic contents.
*  
*  Used only if Canvas.dynamicContents : true has been set.
* @type {ValueMap}
* @default null
*/
dynamicContentsVars?: ValueMap;

/**
* Set the CSS Margin, in pixels, for this component. Margin provides blank space outside
*  of the border.
*  
*  This property sets the same thickness of margin on every side. Differing per-side
*  margins can be set in a CSS style and applied via Canvas.styleName.
*  
*  Note that the specified size of the widget will be the size including the margin
*  thickness on each side.
* @type {number}
* @default null
*/
margin?: number;

/**
* Set the CSS padding of this component, in pixels. Padding provides space between the
*  border and the component's contents.
*  
*  This property sets the same thickness of padding on every side. Differing per-side
*  padding can be set in a CSS style and applied via Canvas.styleName.
*  
*  Note that CSS padding does not affect the placement of Canvas.children. To
*  provide a blank area around children, either use Canvas.margin or use
*  a Layout as the parent instead, and use properties such as
*  Layout.layoutMargin to create blank space.
* @type {number}
* @default null
*/
padding?: number;

/**
* Set the CSS border of this component, as a CSS string including border-width,
*  border-style, and/or color (eg "2px solid blue").
*  
*  This property applies the same border to all four sides of this component. Different
*  per-side borders can be set in a CSS style and applied via Canvas.styleName.
*  
*  If a Canvas.isGroup is being shown then border is derived from the
*  Canvas.groupBorderCSS attribute, not from the explicit border property.
* @type {string}
* @default null
*/
border?: string;

/**
* The background color for this widget. It corresponds to the CSS background-color
*  attribute. You can set this property to an RGB value (e.g. #22AAFF) or a named color
*  (e.g. red) from a list of browser supported color names.
* @type {CSSColor}
* @default null
*/
backgroundColor?: CSSColor;

/**
* URL for a background image for this widget (corresponding to the CSS "background-image"
*  attribute).
* @type {SCImgURL}
* @default null
*/
backgroundImage?: SCImgURL;

/**
* Specifies how the background image should be tiled if this widget
*  is larger than the image. It corresponds to the CSS background-repeat
*  attribute.
*  
*  The default of null means no background-repeat CSS will be
*  written out. See BackgroundRepeat type for details on other settings.
*  
*  NOTE: this setting directly sets the CSS property background-repeat but
*  does not attempt to work around various known bugs with this setting, or lack of support
*  in IE6. If you need to apply CSS-based workarounds for browser limitations with
*  this setting, it's best to do so via setting Canvas.styleName.
* @type {BackgroundRepeat}
* @default null
*/
backgroundRepeat?: BackgroundRepeat;

/**
* Specifies how the background image should be positioned on the widget.
*    It corresponds to the CSS background-position attribute. If unset,
*    no background-position attribute is specified if a background image is
*    specified.
* @type {string}
* @default null
*/
backgroundPosition?: string;

/**
* Default directory for app-specific images, relative to the Page-wide
*  Page.getAppImgDir.
* @type {URL}
* @default ""
*/
appImgDir?: URL;

/**
* Default directory for skin images (those defined by the class), relative to the
*  Page-wide Page.getSkinDir.
* @type {SCImgURL}
* @default "images/"
*/
skinImgDir?: SCImgURL;

/**
* Specifies the cursor image to display when the mouse pointer is
*    over this widget. It corresponds to the CSS cursor attribute. See Cursor type for
*    different cursors.
*    
*    See also Canvas.disabledCursor and Canvas.noDropCursor.
* @type {Cursor}
* @default "default"
*/
cursor?: Cursor;

/**
* Specifies the cursor image to display when the mouse pointer is
*    over this widget if this widget is disabled. It corresponds to the CSS cursor
*    attribute. See Cursor type for different cursors.
* @type {Cursor}
* @default "default"
*/
disabledCursor?: Cursor;

/**
* Specifies the cursor image to display when the user drags a droppable canvas
*    over this if it is not a valid drop target for the event and
*    EventHandler.showNoDropIndicator is true.
* @type {Cursor}
* @default "not-allowed"
*/
noDropCursor?: Cursor;

/**
* Renders the widget to be partly transparent. A widget's opacity property may
*    be set to any number between 0 (transparent) to 100 (opaque).
*    Null means don't specify opacity directly, 100 is fully opaque.
*    Note that heavy use of opacity may have a performance impact on some older
*    browsers.
*    
*    In older versions of Internet Explorer (Pre IE9 / HTML5), opacity is achieved
*    through proprietary filters. If
*    Canvas.neverUseFilters within this application
*    developers must set Canvas.useOpacityFilter to true for specific components
*    on which opacity support is required.
*    
*    Also note that opacity is incompatible
*    with Canvas.useBackMask.
* @type {number}
* @default null
*/
opacity?: number;

/**
* Controls what happens when the drawn size of the content of a Canvas is either
*      greater or smaller than the specified size of the Canvas. Similar to the CSS
*      property overflow, but consistent across browsers. See Overflow type for
*      details.
* @type {Overflow}
* @default "visible"
*/
overflow?: Overflow;

/**
* Indicates that this component is able to render itself at multiple possible sizes, in order
*  to fit into less space when very little space is available, or to display more information
*  or provide more immediate access to functionality when more space is available.
*  
*  Unlike simply indicating a flexible size via width:"*", setting canAdaptWidth
*  indicates that a component has 2 (or more) different ways of rendering itself with different
*  discrete sizes, but does not have the ability to use every additional available pixel.
*  
*  For example, a menu embedded in a toolbar might show as only a fixed-size icon which reveals
*  menu options when clicked, or if enough space is available, could show all of the menu
*  options directly in the toolbar. In this example, the menu should either be allocated just
*  enough space for an icon, or enough space for all options to appear inline; any other
*  amount of space being allocated is just a waste.
*  
*  When a component with canAdaptWidth is part of a Layout, Canvas.adaptWidthBy
*  will be called every time component sizes are being calculated, passing a positive or
*  negative pixel value: 
*  
*  
* - a positive pixel value indicates a surplus - all other component's minimum sizes
*    can be accommodated, including any Canvas.minWidth declared on the
*    canAdaptWidth component itself
*  
* - a negative pixel value indicates that the containing layout is going to be
*    forced to Overflow (introduce scrollbars, clip content, etc) unless some
*    component or set of components gives up at least this many pixels
*  
*  
*  If a non-zero value is returned from adaptWidthBy, this means that the
*  component is willing to shrink or expand by that many pixels. Returning 0 means that the
*  component is unable to reduce size further, or for a surplus, cannot make good use of the
*  surplus space.
*  
*  
* - A call to Canvas.adaptWidthBy may surrender as many pixels as desired
*  (as long as the widget won't drop below its minimum allowed width), no matter whether a
*  positive (surplus) or negative (overflow) pixel value is supplied, but
*  
* - A call to Canvas.adaptWidthBy may not increase its size by more than the number of
*  offered pixels - so if an overflow is present, it may not increase its size at all.
*  
*  
*  Note that when the initial width is specified as a stretch size (e.g. "*"), then after
*  Canvas.adaptWidthBy is called, the Framework will stretch (but not shrink) the
*  member like any other stretch-size Layout member, but the unadaptedWidth
*  argument will always reflect the unstretched width requested by the previous call to
*  Canvas.adaptWidthBy. This behavior may be disabled by specifying the initial
*  width as a number, or leaving it unspecified.
*  
*  Behavior is slightly different for overflow: "visible" members - in this case the
*  unadaptedWidth passed in will reflect the current visible width of the
*  member, rather than the last width requested by the previous call to
*  Canvas.adaptWidthBy or the specified width (on the first call). However, note that the
*  visible length will match your requested width unless the member is actually overflowed.
*  Stretch sizing is not supported for adaptive-width members with overflow: "visible".
*  
*  Caution: you must either determine the current size of the canvas by maintaining your own
*  internal state, or use the unadaptedWidth parameter passed to 
*  Canvas.adaptWidthBy. You must not call Canvas.getWidth or Canvas.getVisibleWidth 
*  on the canvas itself inside Canvas.adaptWidthBy as the size is in the processing of
*  being determined, but you may draw children or call Canvas.getVisibleWidth on them, as
*  we guarantee that the adaptive-width canvas is drawn before the first call to 
*  Canvas.adaptWidthBy. An example of drawing children in Canvas.adaptWidthBy to
*  compute overall width may be seen in the
*  ${isc.DocUtils.linkForExampleId('inlinedMenuMobileSample', 'Inlined Menu Mobile Sample')}.
*  
*  Note that reasonable settings for Canvas.minWidth should be applied to all other
*  flexible-sized members of a layout where a canAdaptWidth component appears,
*  because when too little space is available, a canAdaptWidth component will
*  absorb all available space until minimums for other components are hit (or the
*  canAdaptWidth component reaches its maximum size). If more than one
*  canAdaptWidth component is present, Canvas.adaptiveWidthPriority to give priority
*  to a particular component when allocating space.
*  
*  All of the above behaviors are exactly the same for height, using
*  Canvas.canAdaptHeight and Canvas.adaptHeightBy.
* @type {boolean}
* @default null
*/
canAdaptWidth?: boolean;

/**
* See Canvas.canAdaptWidth.
* @type {boolean}
* @default null
*/
canAdaptHeight?: boolean;

/**
* If multiple widgets in the same Layout have Canvas.canAdaptWidth,
*  adaptiveWidthPriority can be set to indicate which of the components should be 
*  given priority access to space.
*  
*  The widget with the highest priority setting will be offered surplus space first, and asked
*  to give up space last. Lack of a priority setting is treated as zero. Any adaptive widgets
*  with the same priority setting will be asked to give up or release space according to their
*  order in Layout.members.
* @type {Integer}
* @default null
*/
adaptiveWidthPriority?: Integer;

/**
* See Canvas.adaptiveWidthPriority.
* @type {Integer}
* @default null
*/
adaptiveHeightPriority?: Integer;

/**
* Whether to use the browser's native scrollbars or SmartClient-based scrollbars.
*  
*  SmartClient-based scrollbars are skinnable, giving you complete control over look and
*  feel. SmartClient-based scrollbars also enable some interactions not possible with
*  native scrollbars, such as ListGrid.fixedRecordHeights
*  in grids in combination with ListGrid.dataPageSize.
*  
*  Native browser scrollbars are slightly faster simply because there are less SmartClient
*  components that need to be created, drawn and updated. Each visible SmartClient-based
*  scrollbar on the screen has roughly the impact of two StretchImgButtons.
*  
*  SmartClient is always aware of the size of the scrollbar, regardless of whether native
*  or custom scrollbars are used, and regardless of what operating system and/or operating
*  system "theme" or "skin" is in use. This means SmartClient will correctly report the
*  Canvas.getViewportHeight, that is, the interior area of the
*  widget excluding space taken by scrollbars, which is key for exactly filling a component
*  with content without creating unnecessary scrolling.
*  
*  The showCustomScrollbars setting is typically overridden in load_skin.js in
*  order to change the default for all SmartClient components at once.
*  This may be achieved via the static Canvas.setShowCustomScrollbars 
*  method or via a simple addProperties block
*  , like so:
*  
*  
*    isc.Canvas.addProperties({ showCustomScrollbars:false });
*  
*  
*  
*  
*  On Browser.isTouch, custom scrollbars are disabled in favor of enabling
*  native touch scrolling if available. However, custom scrollbars and native touch
*  scrolling can be enabled for the component by setting Canvas.alwaysShowScrollbars
*  to true.
* @type {boolean}
* @default true
*/
showCustomScrollbars?: boolean;

/**
* Whether this canvas should receive eventHandling from its scrollbars,
*  which are Canvas.peers. Normally, a canvas only gets bubbled events from
*  its Canvas.children.
*  
*  Note that this property only has an impact if Canvas.showCustomScrollbars is true.
* @type {boolean}
* @default true
*/
receiveScrollbarEvents?: boolean;

/**
* On Browser.isTouch that support native touch scrolling, if
*  Canvas.showCustomScrollbars is true and touch
*  scrolling has not been disabled by the Canvas.useTouchScrolling and/or
*  Canvas.disableTouchScrollingForDrag settings, should custom scrollbars and
*  native touch scrolling be enabled for this component? If false or unset,
*  then only native touch scrolling will be enabled. If true, then both scrolling
*  mechanisms will be enabled.
*  
*  NOTE: Because native touch scrolling (also called momentum scrolling)
*  is computationally intensive, some mobile browsers implement an optimization where the state
*  of the DOM for the element being scrolled will be frozen or partially frozen during
*  the scroll animation. This results in a delay between when the scroll position reaches
*  a certain point in the animation and when the positions of the custom scrollbar thumbs
*  are updated to reflect that scroll position.
* @type {boolean}
* @default null
*/
alwaysShowScrollbars?: boolean;

/**
* On Browser.isTouch, if this Canvas can be scrolled,
*  should touch-dragging the content area result in scrolling? Set to false
*  if touch-dragging should not cause scrolling. Note that setting to false
*  enables the use of Canvas.showCustomScrollbars on touch devices.
*  
*  useTouchScrolling can default to false if
*  Canvas.disableTouchScrollingForDrag is true
*  and various built-in drag operations are enabled that normally interfere with touch scrolling
*  (e.g. ListGrid.canDragSelect and ListGrid.canReorderRecords).
*  
*  When touch scrolling is disabled, it can be difficult to interact with parts of the
*  custom scrollbars at their default size of 16 pixels. In touch browsers, any touch 8px
*  before the thumb of a Scrollbar will be mapped to the thumb, but
*  the other parts of the scrollbar do not have a similar tolerance applied. The width of
*  the custom scrollbars can be increased by setting the Canvas.scrollbarSize to a
*  larger value, but note that when skinning, changing the
*  scrollbarSize may cause tiling of certain images and backgrounds that make
*  up the custom scrollbar. This can be fixed for a component by creating it with
*  Canvas.scrollbarConstructor set to "Scrollbar"&mdash;a basic scrollbar class that
*  does not employ spriting.
* @type {boolean}
* @default null
*/
useTouchScrolling?: boolean;

/**
* Disables Canvas.useTouchScrolling whenever a built-in drag operation has been enabled which is
*  known to be non-functional if touch scrolling is enabled. Default behavior is to leave touch
*  scrolling enabled even if it makes other enabled drag operations non-functional, since any
*  accessibility application must provide an alternative way to perform
*  drag and drop operations anyway.
*  
*  disableTouchScrollingForDrag exists so that applications can change the default
*  setting on a per-component basis (via 
*  Class.changeDefaults),
*  in order to make a system-wide or
*  per-component-type decision about whether to favor touch scrolling vs retaining the ability to
*  drag and drop via finger drags, instead of having to set useTouchScrolling on each
*  individual instance.
*  
*  See the mobileDevelopment for more background
*  information.
* @type {boolean}
* @default null
*/
disableTouchScrollingForDrag?: boolean;

/**
* How thick should we make the scrollbars for this canvas. This only applies if
*  Canvas.showCustomScrollbars is true.
*  
*  NOTE: When skinning, changing the
*  scrollbarSize may cause tiling of certain images and backgrounds that make
*  up the custom scrollbar. This can be fixed for a component by creating it with
*  Canvas.scrollbarConstructor set to "Scrollbar"&mdash;a basic scrollbar class
*  that does not employ spriting.
* @type {number}
* @default "16"
*/
scrollbarSize?: number;

/**
* The class that will be used to create custom scrollbars for this component. Set this
*  attribute to a Scrollbar subclass with e.g. a different skinImgDir, to customize scrollbar
*  appearance for this component only.
*  
*  When skinning and supported by the skin, the default
*  scrollbarConstructor is changed to a different scrollbar class which handles
*  scrollbar spriting. Spriting of the scrollbars of an individual component can therefore
*  be disabled by creating the component with scrollbarConstructor set to the
*  "Scrollbar" class. "Scrollbar" is a basic scrollbar class that does not employ spriting.
* @type {string}
* @default "Scrollbar"
*/
scrollbarConstructor?: string;

/**
* If set to true, the widget will be disabled. A widget is only considered enabled
*  if it is individually enabled and all parents above it in the containment hierarchy
*  are enabled. This allows you to enable or disable all components of a complex
*  nested widget by enabling or disabling the top-level parent only.
* @type {boolean}
* @default false
*/
disabled?: boolean;

/**
* If set to true, this widget will be enabled, if set to false, or null, this
*  widget will be disabled.
* @deprecated * As of SmartClient version 5.5 this property has been deprecated. The  Canvas.disabled property will be used to govern enabled/disabled state instead
*  if this.enabled has not been changed from its default value.

* @type {boolean}
* @default "unset"
*/
enabled?: boolean;

/**
* Criteria to be evaluated to determine whether this Canvas should be visible. Re-evaluated
*  whenever data in the Canvas.ruleScope changes.
*  
*  A basic criteria uses textMatchStyle:"exact". When specified in
*  componentXML this property allows
*  xmlCriteriaShorthand for defining criteria.
* @type {AdvancedCriteria}
* @default null
*/
visibleWhen?: AdvancedCriteria;

/**
* Criteria to be evaluated to determine whether this Canvas should be enabled. Re-evaluated
*  whenever data in the Canvas.ruleScope changes.
*  
*  A basic criteria uses textMatchStyle:"exact". When specified in
*  componentXML this property allows
*  xmlCriteriaShorthand for defining criteria.
* @type {AdvancedCriteria}
* @default null
*/
enableWhen?: AdvancedCriteria;

/**
* Should this element be redrawn in response to a resize?
*  
*  Should be set to true for components whose Canvas.getInnerHTML will not
*  automatically reflow to fit the component's new size.
* @type {boolean}
* @default null
*/
redrawOnResize?: boolean;

/**
* Can this widget be allowed to become the target of keyboard events?
*  
*  If canFocus is unset (the default), only scrollable widgets with visible scrollbars are
*  focusable, to allow for keyboard scrolling.
*  
*  A widget normally receives focus by being clicked on or tabbed to.
* @type {boolean}
* @default null
*/
canFocus?: boolean;

/**
* If specified this governs the tabIndex of the widget in the page's tab order.
*  Setting this value to -1 will ensure the canvas does not show up
*  in the page's tab order, though if Canvas.canFocus is true, the user
*  may still give it keyboard focus by clicking on the widget directly.
*  
*  By default SmartClient auto-assigns tab-indices, ensuring focusable widgets
*  are reachable by tabbing in an intuitive order based on widget hierarchy and 
*  draw order. Specifying an explicit tab index means a widget will not participate 
*  in this automatic tab position allocation and is typically not recommended except
*  for very simple cases.
*  
*  For more information on automatic tab index assignment, including recommended
*  approaches for customizing tab order assignation, see the tabOrderOverview.
*  Developers may call Canvas.clearExplicitTabIndex to clear any explicitly
*  assigned tab index, and have the widget participate in automatic tab position allocation.
*  
*  canvas.tabIndex cannot be set to greater than
*  Canvas.TAB_INDEX_FLOOR - as we reserve the values above this range for
*  auto-assigned tab-indices.
* @type {number}
* @default null
*/
tabIndex?: number;

/**
* If specified this governs the HTML accessKey for the widget.
*  
*  This should be set to a character - when a user hits the html accessKey modifier for
*  the browser, plus this character, focus will be given to the widget in question.
*  The accessKey modifier can vary by browser and platform. 
*  
*  The following list of default behavior is for reference only, developers should also
*  consult browser documentation for additional information.
*  
*  
* - Internet Explorer (all platforms): Alt + accessKey
*  
* - Mozilla Firefox (Windows, Unix): Alt+Shift + accessKey
*  
* - Mozilla Firefox (Mac): Ctrl+Opt + accessKey
*  
* - Chrome and Safari (Windows, Unix): Alt + accessKey
*  
* - Chrome and Safari (Mac): Ctrl+Opt + accessKey
*  
* @type {string}
* @default null
*/
accessKey?: string;

/**
* Context menu to show for this object, an instance of the Menu widget.
*  
*  Note: if Canvas.destroy is called on a canvas, any specified context menu is
*  not automatically destroyed as well. This is in contrast to MenuButtons which
*  automatically destroy their specified MenuButton.menu by default. The behavior
*  is intentional as context menus are commonly reused across components.
* @type {Menu}
* @default null
*/
contextMenu?: Menu;

/**
* Default class used to construct menus created by this component, including context menus.
* @type {SCClassName}
* @default "Menu"
*/
menuConstructor?: SCClassName;

/**
* Prompt displayed in hover canvas if Canvas.showHover is true.
* @type {HTMLString}
* @default null
*/
prompt?: HTMLString;

/**
* If set, native HTML5 drag and drop will be used for all drags initiated on this widget
*  (on browsers where this is supported).
*  
*  When using native HTML5 drags, the same series of events fires as for a normal drag
*  (Canvas.dragStart, Canvas.dropMove, etc.), and the Canvas.dragType / Canvas.dropTypes
*  system works. Canvas.dragAppearance is not supported; however,
*  basic customization of the browser's tracker image is supported in certain browsers via
*  the EventHandler.setDragTrackerImage API.
*  
*  The primary difference with a native drag is that it can be cross-frame; that is, the
*  user can drag out of the current browser window and drop into a different window or tab.
*  
*  To provide information that will be available to a foreign frame, use
*  EventHandler.setNativeDragData. This API must be called when the Canvas.dragStart
*  event fires, and will not work if called at any other time.
*  
*  However, due to browser bugs and/or browser-imposed limitations, the information provided to
*  setNativeDragData cannot be accessed in the foreign frame until the actual drop
*  occurs (mouse button released). This means drop eligibility cannot be determined dynamically
*  based on the dragged data; instead, eligibility can only be determined based on the
*  Canvas.dragType / Canvas.dropTypes system. For this reason, a Canvas.dragType must
*  be set on the source of a drag.
*  
*  NOTE: Although Internet Explorer 10+ and Edge support a subset of the 
*  HTML5 drag and drop standard,
*  native drag and drop is disabled in IE and Edge because cross-window drags&mdash;the 
*  primary purpose of this API&mdash;are not possible.
* @type {boolean}
* @default null
*/
useNativeDrag?: boolean;

/**
* Indicates whether this widget can initiate custom drag-and-drop operations (other than
*  reposition or resize). Normally Canvas.canDragReposition or Canvas.canDragResize would
*  be used instead of this property.
*  
*  Note: this property may be manipulated by higher-level dragging semantics.
*  
*  If Canvas.useNativeDrag is true and this widget has been drawn, then this widget must be
*  Canvas.redraw in order for a change of the value of this attribute to take effect.
* @type {boolean}
* @default false
*/
canDrag?: boolean;

/**
* Number of pixels the cursor needs to move before the EventHandler starts a drag operation.
* @type {number}
* @default "5"
*/
dragStartDistance?: number;

/**
* If this Canvas is canAcceptDrop:true, when the user drags a droppable widget over
*    an edge of the widget, should we scroll to show the rest of the widget's content?
*    Returned from canvas.shouldDragScroll() if there are scrollbars.
* @type {boolean}
* @default true
*/
canDragScroll?: boolean;

/**
* If this widget supports drag-scrolling, This property specifies how many ms the
*    user must hover over the drag-scroll threshold before scrolling begins.
* @type {number}
* @default "100"
*/
dragScrollDelay?: number;

/**
* This indicates how the system will test for droppable targets: either by intersection
*  with the mouse or intersection with the rectangle of the dragMoveTarget.
* @type {DragIntersectStyle}
* @default "mouse"
*/
dragIntersectStyle?: DragIntersectStyle;

/**
* Indicates whether this widget can be moved by a user of your application by simply
*    dragging with the mouse.
* @type {boolean}
* @default false
*/
canDragReposition?: boolean;

/**
* Cursor to switch to if the mouse is over a widget that is drag repositionable.
* @type {Cursor}
* @default "move"
*/
dragRepositionCursor?: Cursor;

/**
* Indicates whether this widget can be resized by dragging on the edges and/or corners of
*    the widget with the mouse.
* @type {boolean}
* @default false
*/
canDragResize?: boolean;

/**
* If Canvas.canDragResize is true, this property specifies the conditions for when
*  proportional resizing is used. The default is "none"
*  
*  , which means that proportional resizing is disabled.
* @type {ProportionalResizeMode}
* @default "none"
*/
proportionalResizing?: ProportionalResizeMode;

/**
* If Canvas.proportionalResizing is set to "modifier"
*  
*  or "modifierOff"
*  
*  then proportional resizing of the widget is activated or deactivated, respectively,
*  whenever at least one key in this set of modifier keys is pressed.
*  
*  The keys allowed in this set are: "Alt", "Ctrl", and "Shift". If this set
*  of keys is empty then proportional resizing is always used if
*  proportionalResizing is "modifier"
*  and is never used if
*  proportionalResizing is "modifierOff"
*  .
* @type {Array<Partial<KeyName>>}
* @default "["Shift"]"
*/
proportionalResizeModifiers?: Array<Partial<KeyName>>;

/**
* When drag resizing is enabled via Canvas.canDragResize, restricts resizes to only
*  certain edges or corners.
*  
*  The default of null indicates the widget can be resized from any corner or edge (if
*  canDragResize is true).
*  
*  To restrict resizing to only certain corners, set resizeFrom to an Array of
*  EdgeNames.
* @type {Array<Partial<EdgeName>>}
* @default null
*/
resizeFrom?: Array<Partial<EdgeName>>;

/**
* This flag controls whether we register the component as a maskable item with the
*  EventHandler. If enabled, a backmask will be automatically created for the
*  dragMoveTarget on the fly to avoid burnthrough e.g. by plugins or frames.
*  
*  Note that this property will be defaulted to false unless the canvas contains an
*  IFrame, in which case it will be defaulted to true.
*  
*  The Canvas.dragMaskType property controls what kind of mask is used in case
*  useDragMask is enabled.
* @type {boolean}
* @default null
*/
useDragMask?: boolean;

/**
* This property controls what kind of mask is used in case Canvas.useDragMask is
*  enabled.
* @type {DragMaskType}
* @default "div"
*/
dragMaskType?: DragMaskType;

/**
* Will this Canvas fire hover events when the user hovers over it, or one of its children?
* @type {boolean}
* @default null
*/
canHover?: boolean;

/**
* If this.canHover is true, how long should the mouse be kept over this
*  widget before the hover event is fired
* @type {number}
* @default "300"
*/
hoverDelay?: number;

/**
* If this.canHover is true, should we show the global hover canvas by default
*  when the user hovers over this canvas?
* @type {boolean}
* @default true
*/
showHover?: boolean;

/**
* If Canvas.showHover is true, this property can be used to customize
*  the width of the hover canvas shown. See also Canvas.hoverAutoFitWidth and 
*  Canvas.hoverAutoFitMaxWidth.
* @type {Integer}
* @default "1"
*/
hoverWidth?: Integer;

/**
* if Canvas.showHover is true, this property will cause the specified 
*  Canvas.hoverWidth to be treated as a minimum width for the hover. If the
*  hover content string exceeds this, the hover will expand to accomodate it
*  up to Canvas.hoverAutoFitMaxWidth (without the text wrapping).
*  
*  Using this settings differs from simply disabling wrapping via 
*  Canvas.hoverWrap as the content will wrap if the
*  Canvas.hoverAutoFitMaxWidth is exceeded.
* @type {boolean}
* @default true
*/
hoverAutoFitWidth?: boolean;

/**
* Maximum auto-fit width for a hover if Canvas.hoverAutoFitWidth is 
*  enabled. May be specified as a pixel value, or a percentage of page width.
* @type {number | string}
* @default "200"
*/
hoverAutoFitMaxWidth?: number | string;

/**
* If this.showHover is true, this property can be used to customize the
*  height of the hover canvas shown.
* @type {Integer}
* @default null
*/
hoverHeight?: Integer;

/**
* If this.showHover is true, this property can be used to customize the
*  alignment of content in the hover canvas.
* @type {Alignment}
* @default null
*/
hoverAlign?: Alignment;

/**
* If this.showHover is true, this property can be used to customize the
*  vertical alignment of content in the hover canvas.
* @type {VerticalAlignment}
* @default null
*/
hoverVAlign?: VerticalAlignment;

/**
* If this.showHover is true, this property can be used to customize the
*  whether content in the hover canvas is displayed in a single line, or wraps.
*  
*  Note that if developers wish to have hovers expand horizontally to fit their
*  text without wrapping *up to some maximum*, and then wrap rather than exceeding
*  that maximum, the Canvas.hoverAutoFitWidth and Canvas.hoverAutoFitMaxWidth attributes
*  may be used instead of simply setting hoverWrap to false.
* @type {boolean}
* @default null
*/
hoverWrap?: boolean;

/**
* If this.showHover is true, this property can be used to specify the
*  css style to apply to the hover canvas.
* @type {CSSStyleName}
* @default null
*/
hoverStyle?: CSSStyleName;

/**
* If this.showHover is true, should the hover canvas be shown with opacity
*  other than 100?
* @type {number}
* @default null
*/
hoverOpacity?: number;

/**
* If this.showHover is true, should this widget's hover canvas be moved with
*  the mouse while visible?
* @type {boolean}
* @default null
*/
hoverMoveWithMouse?: boolean;

/**
* If this.showHover is true and Canvas.getHoverComponent is implemented, should
*  the hoverCanvas returned from it be automatically destroyed when it is hidden?
*  
*  The default of null indicates that the component will be automatically
*  destroyed. Set to false to prevent this.
* @type {boolean}
* @default null
*/
hoverAutoDestroy?: boolean;

/**
* How far into the edge of an object do we consider the "edge" for drag resize purposes?
* @type {number}
* @default "5"
*/
edgeMarginSize?: number;

/**
* If the widget has drag resize configured on one or more of it's edges, and the
*  edgeMarginSize is large enough that the remaining space is less than
*  minNonEdgeSize, the edgeMarginSize will be reduced such that the non-edge
*  part of the widget is at least 1/3 of the total space (with two draggable edges) or half
*  of it (with one draggable edge).
* @type {Integer}
* @default "0"
*/
minNonEdgeSize?: Integer;

/**
* Cursor to use when over each edge of a Canvas that is drag resizable.
*  
*  To disable drag resize cursors, set the edgeCursorMap property to null.
* @type {object}
* @default "{...}"
*/
edgeCursorMap?: object;

/**
* Constrains drag-resizing and drag-repositioning of this canvas to either the rect of its
*  parent (if set to true) or an arbitrary rect based on its parent (if set to a
*  [Left,Top,Width,Height] rect array). In the latter mode you may use negative offsets
*  for left/top and a width/height greater than the visible or scroll width of the parent
*  to allow positioning beyond the confines of the parent.
*  
*  If this canvas has no parent, constrains dragging to within the browser window.
*  
*  Affects target and outline dragAppearance, not tracker.
*  
*  Note: keepInParentRect affects only user drag interactions, not programmatic moves.
*  
*  Example use cases:
*  keepInParentRect: true - confine to parent
*  keepInParentRect: [0, 0, 500, 500] - confine to top left 500x500 region within parent
*  keepInParentRect: [0, 0, 10000, 10000] - in combination with
*  overflow: "auto", confine to parent, but allow moving off the right and bottom of the
*  parent to force scrolling (and hence enlarge the scrollWidth of the parent).
* @type {boolean | Array<Partial<Integer>>}
* @default null
*/
keepInParentRect?: boolean | Array<Partial<Integer>>;

/**
* Visual appearance to show when the object is being dragged. May be overridden for
*  dragResize or dragReposition events via Canvas.dragResizeAppearance
*  and Canvas.dragRepositionAppearance.
* @type {DragAppearance}
* @default "outline"
*/
dragAppearance?: DragAppearance;

/**
* If Canvas.canDragResize is true, this attribute specifies the visual appearance
*  to show during drag resize. If unset Canvas.dragAppearance will be used.
* @type {DragAppearance}
* @default null
*/
dragResizeAppearance?: DragAppearance;

/**
* If Canvas.canDragReposition is true, this attribute specifies the visual appearance
*  to show during drag reposition. If unset Canvas.dragAppearance will be used.
* @type {DragAppearance}
* @default null
*/
dragRepositionAppearance?: DragAppearance;

/**
* Sets a dragType for this widget used, to be compared to
*  Canvas.dropTypes on possible drop target widgets. See
*  Canvas.dropTypes for a full explanation.
* @type {string}
* @default null
*/
dragType?: string;

/**
* When a drag and drop interaction occurs, if a Canvas.dragType is
*  configured on the source widget, it is compared to the dropTypes configured
*  on the target widget, and a drop is only allowed if the dragType is listed
*  in the target widget's dropTypes array.
*  
*  The default setting means any dragType is eligible for dropping on this
*  widget, including no dragType setting.
*  
*  See also Canvas.willAcceptDrop for dynamic determination of drop eligibility.
* @type {Array<Partial<string>> | string}
* @default "**anything**"
*/
dropTypes?: Array<Partial<string>> | string;

/**
* A different widget that should be actually dragged when dragging initiates on this
*  widget. One example of this is to have a child widget that drags its parent, as with a
*  drag box. Because the parent automatically repositions its children, setting the drag
*  target of the child to the parent and then dragging the child will result in both
*  widgets being moved.
*  
*  Valid dragTarget values are:
*  
* - null (default) [this widget is its own drag target]
*  
* - another widget, or widget ID
*  
* - "parent" drag target is this widget's Canvas.getParentCanvas
*  
* - "top" drag target is this widget's Canvas.topElement
*  
*  Note that for dragging to work as intended, the Canvas.resizeFrom setting on the
*  dragTarget must be null or a superset of the Canvas.resizeFrom on this
*  canvas.
* @type {Canvas | string}
* @default null
*/
dragTarget?: Canvas | string;

/**
* When this widget is dragged, if its dragAppearance is "target", should
*  we show a shadow behind the canvas during the drag.
* @type {boolean}
* @default null
*/
showDragShadow?: boolean;

/**
* If this widget has dragAppearance "target", this value specifies the
*  opacity to render the target while it is being dragged. A null value implies we do
*  not modify the opacity.
* @type {Integer}
* @default null
*/
dragOpacity?: Integer;

/**
* Indicates that this object can be dropped on top of other widgets. Only valid if
*  canDrag or canDragReposition is true.
* @type {boolean}
* @default false
*/
canDrop?: boolean;

/**
* Indicates that this object can receive dropped widgets (i.e. other widgets can be
*  dropped on top of it).
* @type {boolean}
* @default false
*/
canAcceptDrop?: boolean;

/**
* When explicitly set to false, disallows drop before this member in the Layout.
* @type {boolean}
* @default null
*/
canDropBefore?: boolean;

/**
* Amount of time (in milliseconds) before mouseStillDown events start to be fired
*  repeatedly for this canvas. See Canvas.mouseStillDown for details.
* @type {Integer}
* @default "400"
*/
mouseStillDownInitialDelay?: Integer;

/**
* Amount of time (in milliseconds) between repeated 'mouseStillDown' events for this
*  canvas. See Canvas.mouseStillDown for details.
* @type {Integer}
* @default "100"
*/
mouseStillDownDelay?: Integer;

/**
* Amount of time (in milliseconds) between which two clicks are considered a single click
* @type {Integer}
* @default "250"
*/
doubleClickDelay?: Integer;

/**
* If true, this canvas will receive all mouse-clicks as single Canvas.click
*  events rather than as Canvas.doubleClick events.
* @type {boolean}
* @default null
*/
noDoubleClicks?: boolean;

/**
* In earlier versions of Internet Explorer (pre IE9), a native limitation
*  exists whereby if HTML elements are
*  overlapping on the page, certain elements can appear to "burn through"
*  elements in the same position with a higher z-index.
*  Specific cases in which this have been observed include Applets,
*  &lt;IFRAME&gt; elements, and for
*  older versions of IE, native &lt;SELECT&gt; items.
*  
*  The backMask is a workaround for this issue. If useBackMask is
*  set to true, the component will render an empty &lt;IFRAME&gt; element
*  behind the canvas, which prevents this effect in all known cases.
*  
*  Has no effect in other browsers.
* @type {boolean}
* @default false
*/
useBackMask?: boolean;

/**
* If this canvas has its size specified as a percentage, this property allows the user to
*  explicitly designate another canvas upon which sizing will be based.
*  
*  If unset percentage sizing is based on
*  - the Canvas.getMasterCanvas if there is one and
*   Canvas.snapTo is set,
*  - otherwise on the amount of space available in this widget's parent canvas, if this is
*   a child of some other widget
*  - otherwise the page size.
* @type {Canvas}
* @default null
*/
percentSource?: Canvas;

/**
* Governs the model to be used when sizing canvases with percentage width or height, or
*  positioning widgets with a specified Canvas.snapTo.
*  
*  Only affects widgets with a a specified Canvas.percentSource, or
*  widgets that have Canvas.snapTo set and are peers of some
*  Canvas.getMasterCanvas.
*  
*  Determines whether the coordinates used for sizing (for percentage sized widgets) and
*  positioning (if snapTo is set) should be relative to the visible size or the
*  viewport size of the percentSource or Canvas.getMasterCanvas widget.
* @type {PercentBoxModel}
* @default "visible"
*/
percentBox?: PercentBoxModel;

/**
* Position this widget such that it is aligned with ("snapped to") an edge of its
*  Canvas.getMasterCanvas (if specified), or its
*  Canvas.getParentCanvas.
*  
*  Note that this property also impacts the sizing of this widget. If this widgets size
*  is specified as a percent value, and has no explicit
*  Canvas.percentSource, sizing will be calculated based on the size of the
*  Canvas.getMasterCanvas when snapTo is set.
*  
*  Possible values: BR, BL, TR, TL, R, L, B, T, C where B=Bottom, T=Top, L=Left, R=right
*  and C=center
*  
*  Standard snapTo behavior will attach the outer edge of the widget to the parent or master
*  element - for example setting snapTo to "B" would align the
*  bottom edge of this component with the bottom edge of the master or parent element
*  (and center this component horizontally over its master or parent element).
*  Canvas.snapEdge can be specified to change this behavior allowing the developer
*  to, for example, align the top edge of this component with the bottom edge of its
*  Canvas.getMasterCanvas.
*  
*  Canvas.snapOffsetLeft and Canvas.snapOffsetTop may also be specified to
*  offset the element from exact snapTo alignment.
* @type {string}
* @default null
*/
snapTo?: string;

/**
* If Canvas.snapTo is defined to this widget, this property can be used to
*  define which edge of this widget should be snapped to an edge of the master or parent
*  element.
*  
*  If unspecified the, default snapTo behavior is set up to align the "snapTo" edge of this
*  widget with the snapTo edge of the master or parent.
* @type {string}
* @default null
*/
snapEdge?: string;

/**
* If Canvas.snapTo is defined for this widget, this property can be used to
*  specify an offset in px or percentage for the left coordinate of this widget.
*  
*  For example if snapTo is specified as "L" and
*  snapOffsetLeft is set to 6, this widget will be rendered 6px inside the left
*  edge of its parent or master element. Alternatively if snapTo was set
*  to "R", a snapOffsetLeft value of -6 would cause the
*  component to be rendered 6px inside the right edge of its parent or Canvas.getMasterCanvas.
* @type {Integer}
* @default null
*/
snapOffsetLeft?: Integer;

/**
* If Canvas.snapTo is defined for this widget, this property can be used to
*  specify an offset in px or percentage for the top coordinate of this widget.
*  
*  For example if snapTo is specified as "T" and
*  snapOffsetTop is set to 6, this widget will be rendered 6px below the top
*  edge of its parent or master element. Alternatively if snapTo was set
*  to "B", a snapOffsetTop value of -6 would cause the
*  component to be rendered 6px inside the bottom edge of its parent or Canvas.getMasterCanvas.
* @type {Integer}
* @default null
*/
snapOffsetTop?: Integer;

/**
* Causes this canvas to snap to its parent's grid when dragging.
* @type {boolean}
* @default null
*/
snapToGrid?: boolean;

/**
* Causes this canvas to snap to its parent's grid when resizing.
*  Note that this value defaults to the Canvas's
*  Canvas.snapToGrid value if undefined.
* @type {boolean}
* @default null
*/
snapResizeToGrid?: boolean;

/**
* If true, causes this canvas's children to snap to its grid when dragging.
*  This behavior can be overridden on a per-child basis by setting the
*  Canvas.snapToGrid value on the child.
* @type {boolean}
* @default null
*/
childrenSnapToGrid?: boolean;

/**
* If true, causes this canvas's children to snap to its grid when resizing.
*  This behavior can be overridden on a per-child basis by setting the
*  Canvas.snapToGrid or
*  Canvas.snapResizeToGrid value on the child.
* @type {boolean}
* @default null
*/
childrenSnapResizeToGrid?: boolean;

/**
* The horizontal grid size to use, in pixels, when snap-to-grid is enabled.
* @type {number}
* @default "20"
*/
snapHGap?: number;

/**
* The vertical grid size to use, in pixels, when snap-to-grid is enabled.
* @type {number}
* @default "20"
*/
snapVGap?: number;

/**
* The horizontal snap direction.
*  Set this value to "before" to snap to the nearest gridpoint to the left;
*  set it to "after" to snap to the nearest gridpoint to the right; and set
*  it to "nearest" to snap to the nearest gridpoint in either direction.
* @type {string}
* @default "after"
*/
snapHDirection?: string;

/**
* The vertical snap direction.
*  Set this value to "before" to snap to the nearest gridpoint above;
*  set it to "after" to snap to the nearest gridpoint below; and set
*  it to "nearest" to snap to the nearest gridpoint in either direction.
* @type {string}
* @default "after"
*/
snapVDirection?: string;

/**
* Describes which axes to apply snap-to-grid to.
*  Valid values are "horizontal", "vertical" or "both".
* @type {string}
* @default "both"
*/
snapAxis?: string;

/**
* When this canvas is dropped onto an object supporting snap-to-grid, should it snap
*  to the grid (true, the default) or just drop wherever the mouse is (false).
* @type {boolean}
* @default true
*/
snapOnDrop?: boolean;

/**
* Should this canvas print its children absolutely positioned when generating
*  Canvas.getPrintHTML.
*  
*  By default explicitly specified absolute positioning and sizing is ignored when generating
*  print HTML. This is done intentionally: there is no way for the framework to predict how
*  explicit sizes will translate to a the printed page and if HTML for printing includes
*  the same absolute positioning and sizing as is displayed within an application it is very
*  common to encounter undesirable effects, such as seeing tables get broken over several
*  pages horizontally when there is enough room to print them on a single page of paper.
*  
*  In some cases, however, a developer may wish to have explicit sizing and positioning
*  respected within the print-view.
*  Setting this attribute to true will cause this to occur.
* @type {boolean}
* @default false
*/
printChildrenAbsolutelyPositioned?: boolean;

/**
* This boolean flag will be set to true by framework logic while generating print HTML 
*  for this widget as a result to a call to Canvas.showPrintPreview 
*  (or just Canvas.getPrintHTML). Note that this flag is set recursively as
*  parent widgets generate print HTML for their children.
*  
*  This is a read-only property and should not be modified by application code.
* @type {boolean}
* @default false
*/
isPrinting?: boolean;

/**
* Whether this canvas should be included in a printable view.
*  
*  Default is to:
*  
*  
* - omit all peers (edges generated by showEdges:true, etc)
*  
* - omit anything considered a "control", such as a button or menu (see
*  PrintProperties.omitControls)
*  
* - include everything else not marked shouldPrint:false
*  
* @type {boolean}
* @default null
*/
shouldPrint?: boolean;

/**
* Is it possible that a call to Canvas.redraw on this widget will change its 
*  size?
*  
*  Used by framework layout code when determining whether a component which has been
*  Canvas.markForRedraw needs an immediate redraw to determine
*  its drawn size.
*  
*  If unset, default behavior assumes any component with overflow set to "visible" 
*  may change size on redraw, and any component with overflow set to "hidden", "scroll",
*  or "auto" will not. This property overrides that behavior, and may be used to indicate
*  that some component with non visible overflow can change size on redraw.
*  An example use case would be a custom component with an override to explicitly 
*  resize the component as part of the redraw() flow.
* @type {boolean}
* @default null
*/
sizeMayChangeOnRedraw?: boolean;

/**
* If this property is set to true, the Canvas.destroy method
*  has been called on this canvas. This implies the canvas is no longer valid. Its ID has been
*  removed from global scope, and calling standard canvas APIs on it is likely to result in
*  errors.
* @type {boolean}
* @default null
*/
destroyed?: boolean;

/**
* This property is set to true when the Canvas.destroy method is called on a widget.
*  If this property is true, but Canvas.destroyed is not, this indicates the canvas is
*  in the process of being destroyed.
* @type {boolean}
* @default null
*/
destroying?: boolean;

/**
* Automatically generated mask canvas displayed when Canvas.showComponentMask is called.
* @type {Canvas}
* @default null
*/
componentMask?: Canvas;

/**
* Defaults for the Canvas.componentMask autoChild.
*  Default properties include Canvas.backgroundColor being set to "black" and
*  Canvas.opacity being set to 20.
* @type {Partial<Canvas>}
* @default "{...}"
*/
componentMaskDefaults?: Partial<Canvas>;

/**
* The minimum speed in pixels per second that must be reached for momentum scrolling to kick in.
*  This setting only applies to touch-enabled devices.
* @type {Double}
* @default "200"
*/
momentumScrollMinSpeed?: Double;

/**
* Whether the Framework should correct for erroneous scrollHeight and scrollWidth values
*  reported by the browser when zoomed (via browser or OS-level zoom) by allowing 
*  Canvas.maxZoomOverflowError of overflow before enabling scrolling and displaying custom
*  scrollbars. Only relevant when Canvas.overflow is "auto".
*  
*  This property is defaulted to true in the Canvas prototype for those
*  browsers where the situation has been observed, except for Firefox, where a better solution
*  is applied that doesn't rely on Canvas.maxZoomOverflowError and never clips any content.
*  Setting this property false will disable the workaround for all browsers, including Firefox.
*  Without a workaround, scrollbars may oscillate rapidly when the browser or OS is zoomed.
* @type {boolean}
* @default "varies"
*/
correctZoomOverflow?: boolean;

/**
* When Canvas.correctZoomOverflow is true and browser or OS-level zoom is active, determines
*  how much overflow must occur before the Framework enables scrolling for this canvas.
*  
*  The larger the value, the more clipping that can occur before scrolling is enabled. So, this
*  property should never be set larger than the minimum amount needed to prevent scrollbar
*  oscillation when zoomed - which is the maximum scrollHeight or scrollWidth error.
* @type {Integer}
* @default "varies"
*/
maxZoomOverflowError?: Integer;

/**
* Should canvases with no Canvas.getParentCanvas be moved to
*  the end of the TabIndexManager tree on draw()?
*  
*  If set to false, the tab-position will not be modified on draw.
*  
*  This property is useful for cases where the tab position of a widget will be managed
*  by some explicit tabIndex management code.
* @type {boolean}
* @default true
*/
updateTabPositionOnDraw?: boolean;

/**
* Should canvases with a Canvas.getParentCanvas be added to
*  the TabIndexManager under the parent as described in Canvas.updateChildTabPositions and
*  Canvas.updateChildTabPosition?
*  
*  If set to false, the tab-position will not be modified on parent change.
*  
*  This property is useful for cases where the tab position of a widget will be managed
*  by something other than the parent canvas, for example for 
*  CanvasItem.canvas.
* @type {boolean}
* @default true
*/
updateTabPositionOnReparent?: boolean;

/**
* Should focus navigation for this canvas and its descendents be handled explicitly
*  by intercepting "Tab" key events and calling the TabIndexManager.shiftFocus
*  API?
*  
*  Setting this property to true will cause the registered TabIndexManager
*  entry for this canvas to be marked as 
*  TabIndexManager.setUseExplicitFocusNavigation,
*  and will cause standard event handling for the canvas and its descendents to 
*  intercept Tab keystrokes and explicitly call TabIndexManager.shiftFocus
*  rather than relying on native browser Tab navigation
* @type {boolean}
* @default null
*/
alwaysManageFocusNavigation?: boolean;

/**
* When set to true, shows a widget hovering at the mouse point instead of the builtin
*  hover label. Override Canvas.getHoverComponent to provide the
*  Canvas to show as the hoverComponent.
* @type {boolean}
* @default false
*/
showHoverComponents?: boolean;

/**
* Whether to show a snap grid for this Canvas. Note that the grid is only shown
*  when either Canvas.childrenSnapToGrid or Canvas.childrenSnapResizeToGrid
*  is true.
*  
*  Grid is based on Canvas.snapHGap and Canvas.snapVGap properties.
* @type {boolean}
* @default null
*/
showSnapGrid?: boolean;

/**
* Specifies indication style to use for snap points, either a grid of lines or an array of
*  crosses. The lines can be configured using the property Canvas.snapGridLineProperties.
* @type {SnapGridStyle}
* @default "crosses"
*/
snapGridStyle?: SnapGridStyle;

/**
* Specifies line styling to use when drawing the grid of lines for SnapGridStyle:
*  "lines".
* @type {Partial<DrawLine>}
* @default null
*/
snapGridLineProperties?: Partial<DrawLine>;

/**
* If enabled while Canvas.childrenSnapToGrid is enabled, children dragged within this Canvas
*  will also snap to positions where their edges or center lines would be aligned with the
*  edges or centers of other components, and lines will be shown to point out the possible
*  alignment (with appearance controlled by Canvas.snapAlignCenterLineStyle and
*  Canvas.snapAlignEdgeLineStyle respectively.
*  
*  By default, edge- or center-snapping is enabled for all components, but the set of eligible
*  components can be explicitly set via Canvas.snapAlignCandidates.
*  
*  See also Canvas.childrenSnapCenterAlign and Canvas.childrenSnapEdgeAlign for enabling or
*  disabling center alignment or edge alignment individually.
* @type {boolean}
* @default null
*/
childrenSnapAlign?: boolean;

/**
* Flag to disable snapping to alignment when children of this Canvas are resized
* @type {boolean}
* @default null
*/
childrenResizeSnapAlign?: boolean;

/**
* See Canvas.childrenSnapAlign. This setting enables or disables snapping on center
*  alignment only.
* @type {boolean}
* @default null
*/
childrenSnapCenterAlign?: boolean;

/**
* See Canvas.childrenSnapAlign. This setting enables or disables snapping on edge
*  alignment only.
* @type {boolean}
* @default null
*/
childrenSnapEdgeAlign?: boolean;

/**
* Flag to disable snapping to alignment when this Canvas is dragged when 
*  Canvas.childrenSnapAlign is enabled on this Canvas' parent.
*  
*  To control snapping to align for the children dragged within this Canvas, see
*  Canvas.childrenSnapAlign instead.
* @type {boolean}
* @default null
*/
snapToAlign?: boolean;

/**
* Flag to disable snapping to alignment when this Canvas is resized.
*  
*  To control snapping to align for the children resized within this Canvas, see
*  Canvas.childrenResizeSnapAlign instead.
* @type {boolean}
* @default null
*/
snapResizeToAlign?: boolean;

/**
* Flag to disable snapping to center alignment when this Canvas is dragged when 
*  Canvas.childrenSnapAlign is enabled on this Canvas' parent.
* @type {boolean}
* @default null
*/
snapToCenterAlign?: boolean;

/**
* Flag to disable snapping to edge alignment when this Canvas is dragged when 
*  Canvas.childrenSnapAlign is enabled on this Canvas' parent.
* @type {boolean}
* @default null
*/
snapToEdgeAlign?: boolean;

/**
* Flag to disable snapping to alignment against this Canvas when other Canvases
*  dragged into the same parent when Canvas.childrenSnapAlign is enabled on this
*  Canvas' parent.
* @type {boolean}
* @default null
*/
isSnapAlignCandidate?: boolean;

/**
* When Canvas.childrenSnapAlign is enabled, list of candidates to check for alignment.
*  
*  If a list of snapAlignCandidates is never provided, the default is to use all
*  children that are not explicitly excluded via Canvas.isSnapAlignCandidate, including
*  automatically adding newly added children as candidates, and ignoring children that have
*  been removed. Use Canvas.addSnapAlignCandidate and Canvas.removeSnapAlignCandidate to add
*  and remove special candidates while retaining all children as default candidates.
*  
*  Possible candidates which are not drawn or are hidden are automatically ignored.
* @type {Array<Partial<Canvas>>}
* @default null
*/
snapAlignCandidates?: Array<Partial<Canvas>>;

/**
* CSS border declaration used for the line shown to indicate snapping to a center line when
*  Canvas.childrenSnapAlign is set.
* @type {string}
* @default "1px solid blue"
*/
snapAlignCenterLineStyle?: string;

/**
* CSS border declaration used for the line shown to indicate snapping to a edge line when
*  Canvas.childrenSnapAlign is set.
* @type {string}
* @default "1px solid #555555"
*/
snapAlignEdgeLineStyle?: string;

/**
* If set, forces the main SVG image or icon in the canvas to be rendered in an image tag rather
*  than an object tag, the default. Typical use cases might be configuring the image of an
*  Img or ImgButton, or the icon of a Button.
*  
*  Rendering via object tag provides the maximum support for CSS in SVG, but may result in a
*  flicker at the browser level when changing images - either manually such as with
*  Canvas.setImage or via state change from rollover, mouseDown, etc. Using image tags to
*  inline the images breaks CSS support but may avoid flickering.
*  
*  If this property is not set, then you can also control whether an SVG image is
*  rendered in an object or image tag by setting the query param "tag" on the image URL - see
*  SCImgURL for details.
*  
*  Note that if multiple icons are potentially present in a canvas (e.g.
*  ListGrid.removeIcon in the cells of a grid body), then setting this
*  property on the widget instance may have no effect. In such case, the Canvas
*  prototype is consulted.
* @type {boolean}
* @default false
*/
useImageForSVG?: boolean;

/**
* If true, events sent to the SVG inside an object tag are forwarded to the object itself by
*  Framework handlers. If false, "pointer-events:none" CSS is set on the object so that events
*  are directly sent to the object by the browser, if supported.
*  
*  The advantage of forwarding events is that it allows hover CSS or any other event-driven
*  logic on the SVG to work as expected. If "pointer-events:none" is written out, no GUI
*  interaction will trigger events in the SVG, including hover CSS. The disadvantage is that
*  only a few critical events are forwarded, such as "mouseDown", "mouseMove", and "mouseUp" for
*  non-touch platforms, and "click" for touch platforms. Other events will be delivered to the
*  SVG, but not forwarded up to the parent document/object tag.
* @type {boolean}
* @default true
*/
forwardSVGeventsToObject?: boolean;

/**
* Whether an EdgedCanvas should be used to show image-based edges around this
*  component.
* @type {boolean}
* @default false
*/
showEdges?: boolean;

/**
* Size in pixels for corners and edges
* @type {number}
* @default "10"
*/
edgeSize?: number;

/**
* Amount the contained Canvas should be offset. Defaults to edgeSize; set to less than
*  edgeSize to allow the contained Canvas to overlap the edge and corner media.
* @type {number}
* @default null
*/
edgeOffset?: number;

/**
* Base name of images for edges. Extensions for each corner or edge piece will be added
*  to this image URL, before the extension. For example, with the default base name of
*  "edge.gif", the top-left corner image will be "edge_TL.gif".
*  
*  The full list of extensions is: "_TL", "_TR", "_BL", "_BR", "_T", "_L", "_B", "_R",
*  "_center".
* @type {SCImgURL}
* @default "[SKIN]edge.gif"
*/
edgeImage?: SCImgURL;

/**
* Array of side names ("T", "B", "L", "R") specifying which sides of the decorated
*  component should show edges. For example: 
*    customEdges : ["T", "B"]
*  
*  .. would show edges only on the top and bottom of a component.
*  
*  The default of null means edges will be shown on all sides.
* @type {Array<Partial<string>>}
* @default null
*/
customEdges?: Array<Partial<string>>;

/**
* Background color for the EdgedCanvas created to decorate this component. This can be used
*  to provide an underlying "tint" color for translucent edge media
* @type {CSSColor}
* @default null
*/
edgeBackgroundColor?: CSSColor;

/**
* Opacity of the edges. Defaults to matching this.opacity.
*  if Canvas.setOpacity is called on a Canvas where edgeOpacity is set,
*  edgeOpacity will be considered a percentage of the parent's opacity (so 50% opaque parent plus
*  edgeOpacity 50 means 25% opaque edges)
* @type {Integer}
* @default null
*/
edgeOpacity?: Integer;

/**
* Whether to show media in the center section, that is, behind the decorated Canvas.
* @type {boolean}
* @default false
*/
edgeShowCenter?: boolean;

/**
* Background color for the center section only. Can be used as a surrogate background
*  color for the decorated Canvas, if the Canvas is set to partially overlap the edges and
*  hence can't show a background color itself without occluding media.
* @type {CSSColor}
* @default null
*/
edgeCenterBackgroundColor?: CSSColor;

/**
* Whether to show a drop shadow for this Canvas. 
*  
*  Shadows may be rendered using Canvas.useCSSShadow or via images. 
*  The appearance of shadows can be customized via
*  Canvas.shadowColor (for css-based shadows) or Canvas.shadowImage 
*  (for image based shadows), Canvas.shadowDepth,
*  Canvas.shadowOffset and Canvas.shadowSoftness.
*  
*  When Canvas.useCSSShadow is false, developers should be aware that the drop shadow
*  is rendered as a Canvas.peers and is drawn outside the specified 
*  width and height of the widget meaning a widget with shadows 
*  takes up a little more space than it otherwise would. A full screen canvas with showShadow set 
*  to true as this would be likely to cause browser scrollbars to appear - developers can handle
*  this by either setting this property to false on full-screen widgets, or by setting
*  overflow to "hidden" on the &lt;body&gt; element if browser-level scrolling is
*  never intended to occur.
* @type {boolean}
* @default false
*/
showShadow?: boolean;

/**
* Depth of the shadow, or the virtual height above the page of the widget throwing the shadow.
*  
*  This is a single parameter that can be used to control both shadowSoftness and
*  shadowOffset.
* @type {number}
* @default "4"
*/
shadowDepth?: number;

/**
* Offset of the shadow. Defaults to half of shadowDepth if unset.
*  
*  Because of the blurred edges, a shadow is larger than the originating component by
*  2xsoftness. An shadowOffset of 0 means that the shadow will extend around the
*  originating component equally in all directions.
*  
*  If Canvas.useCSSShadow are being used, separate vertical and
*  horizontal offsets may be specified via Canvas.shadowHOffset and
*  Canvas.shadowVOffset.
* @type {number}
* @default null
*/
shadowOffset?: number;

/**
* Horizontal offset for the Canvas.showShadow. Takes precedence over 
*  Canvas.shadowOffset if set. Has no effect if Canvas.useCSSShadow
*  are not being used for this canvas.
* @type {number}
* @default null
*/
shadowHOffset?: number;

/**
* Vertical offset for the Canvas.showShadow. Takes precedence over 
*  Canvas.shadowOffset if set. Has no effect if Canvas.useCSSShadow
*  are not being used for this canvas.
* @type {number}
* @default null
*/
shadowVOffset?: number;

/**
* Softness, or degree of blurring, of the shadow.
*  
*  A shadow with softness:x is 2x pixels larger in each direction than the element
*  throwing the shadow, and the media for each edge should be x pixels wide/tall.
*  
*  Defaults to shadowDepth if unset.
* @type {number}
* @default null
*/
shadowSoftness?: number;

/**
* If Canvas.useCSSShadow is false, (or for browsers that do not support
*  css3), this property supplies the base name of the series of images for the sides, 
*  corners, and center of the shadow.
*  
*  The actual image names fetched for the dropShadow combine the segment name and the
*  shadowDepth setting. For example, given "ds.png" as the base name, a depth of
*  4, and the top-left segment of the shadow, we'd use "ds4_TL.png".
*  
*  The names for segments are the same as those given for controlling resizable edges; see
*  Canvas.resizeFrom. The center segment has the name "center". The center segment is
*  the only segment that doesn't include the depth in the URL, so the final image name for the
*  center given a baseName of "ds.png" would be just "ds_center.png".
* @type {SCImgURL}
* @default "[SKIN]ds.png"
*/
shadowImage?: SCImgURL;

/**
* If Canvas.showShadow is true, should we use the css box-shadow property
*  (where supported) to achieve the shadow?
*  
*  Set this property to false to switch to a media-based approach, achieved by
*  rendering the Canvas.shadowImage in an automatically generated peer.
*  This approach is also used regardless of this property value in older browsers 
*  where the css box-shadow property isn't supported.
*  
*  See also Canvas.shadowColor, Canvas.shadowDepth,
*  Canvas.shadowOffset, Canvas.shadowSoftness and Canvas.shadowSpread
* @type {boolean}
* @default true
*/
useCSSShadow?: boolean;

/**
* Color for the css-based drop shadow shown if Canvas.useCSSShadow is true 
*  and Canvas.showShadow is true.
*  
*  Has no effect if we are not using css-based shadows - in that case, use
*  Canvas.shadowImage instead.
* @type {CSSColor}
* @default "#AAAAAA"
*/
shadowColor?: CSSColor;

/**
* Explicit spread for the css-based drop shadow shown if Canvas.useCSSShadow is true 
*  and Canvas.showShadow is true. This property governs how much larger than the
*  widget the shadow will appear. A negative value (coupled with an explicit offset) will
*  result in a smaller shadow.
*  
*  Has no effect if we are not using css-based shadows - in that case, use
*  Canvas.shadowImage instead.
* @type {Integer}
* @default null
*/
shadowSpread?: Integer;

/**
* Should a grouping frame be shown around this canvas if a non-empty string has been specified
*  for Canvas.groupTitle.
* @type {boolean}
* @default true
*/
isGroup?: boolean;

/**
* Sets the style for the grouping frame around the canvas. Only necessary when showing a
*  Canvas.isGroup.
* @type {string}
* @default "2px solid black"
*/
groupBorderCSS?: string;

/**
* Sets the style for the grouping label. Only necessary when showing a Canvas.isGroup.
*  
*  Note that Canvas.groupLabelBackgroundColor overrides any background-color of this style.
* @type {CSSStyleName}
* @default "groupLabel"
*/
groupLabelStyleName?: CSSStyleName;

/**
* If set, the background color of the grouping label. Only applicable when showing a
*  Canvas.isGroup.
*  
*  This corresponds to the CSS background-color property on the grouping label. You can set this
*  property to an RGB value (e.g. #22AAFF) or a named color (e.g. red) from a list of browser
*  supported color names.
*  
* @type {CSSColor}
* @default null
*/
groupLabelBackgroundColor?: CSSColor;

/**
* The title/label for the grouping. Only applicable when Canvas.isGroup is set
*  to true. No Canvas.isGroup or title/label will be shown unless this property
*  is a non-empty string.
* @type {HTMLString}
* @default null
*/
groupTitle?: HTMLString;

/**
* Canvas.ID of the component that gathers the context for evaluation of criteria-based
*  rules specified by properties such as FormItem.visibleWhen, 
*  Class.dynamicProperties, and dynamicCriteria such as 
*  ListGrid.initialCriteria.
*  
*  If ruleScope is not specified, this component will search through its Canvas.parentCanvas 
*  chain until it either reaches the top or reaches a parent marked Canvas.isRuleScope. 
*  This means that typically, ruleScope does not have to be explicitly specified, since 
*  components that want to reference each other often have the same top-level parent (eg, they are 
*  part of the same screen). However, you would need to specify ruleScope in scenarios 
*  such as a modal Window that wants to reference values in the component it was launched from.
*  
*  Determination of the ruleScope happens when the component is first drawn.
*  
*  The component designated as the ruleScope manages a nested data structure called
*  the "rule context" which contains information from all DataBoundComponents that are
*  registered with the ruleScope. By specifying Criterion.fieldName as a
*  DataPath, AdvancedCriteria defined in properties such as
*  FormItem.visibleWhen can access any part of the rule context.
*  
*  By default, the rule context contains data as follows:
*  
*  
* - any DataBoundComponent that has a DataSource contributes the values of the
*   selected record or record being edited under the ID of the DataSource. For any collision
*   an editable display (such as a form or editable grid) wins over a static display (such as
*   a non-editable grid with a selection.) Hidden or cleared components have lowest priority
*   even if editable. For two editable components the first becomes the contributor.
*  
* - any ListGrid or other component that manages a selection and has been assigned an explicit
*   Canvas.ID will contribute the values of the selected record under
*   &lt;componentId&gt;.selectedRecord, and also contributes 3 flags for checking
*   for selection: anySelected, multiSelected,
*   numSelected.
*  
* - any DynamicForm or other component that edits values and has been assigned an explicit
*   Canvas.ID contributes its current values under
*   &lt;componentId&gt;.values, and contributes a flag hasChanges.
*  
* - any DynamicForm or ListGrid that has been assigned an explicit Canvas.ID
*   contributes a value &lt;componentId&gt;.focusField. When present the value
*   indicates the component has focus along with the name of the field that has focus. Its absense
*   indicates the component does not have focus at all.
*  
* - any ListGrid that has been assigned an explicit Canvas.ID contributes a flag
*   isGrouped under &lt;componentId&gt;.
*  
*  For example, given a screen where:
*  
*  
* - a ListGrid with ID "itemGrid" and DynamicForm with ID "itemForm" are both bound to the
*   supplyItem sample DataSource
*  
* - the ListGrid has a single selection, and the record selected in the ListGrid is being
*   edited in the form, and has been changed
*  
*  The default rule context available from Canvas.getRuleContext, expressed as JSON, would be:
*  
*  {
*  supplyItem : {
*    itemID : "654321",
*    itemName : "Sewing Machine",
*    price : 5.50, // note: user change
*    ..other properties..
*  },
*  itemForm.values : {
*    itemID : "654321",
*    itemName : "Sewing Machine",
*    price : 5.50, // note: user change
*    ..other properties..
*  },
*  itemForm.focusField : "itemName",
*  itemForm.hasChanges : true,
*  itemGrid.selectedRecord : {
*    itemID : "654321",
*    itemName : "Sewing Machine",
*    price : 3.50, // note: old price
*    ..other properties..
*  },
*  itemGrid.anySelected : true,
*  itemGrid.multiSelected : false,
*  itemGrid.numSelected : 1,
*  itemGrid.isGrouped : false
*  }
*  
*  In addition, an application can put custom data into the ruleScope via Canvas.provideRuleContext.
* @type {string}
* @default null
*/
ruleScope?: string;

/**
* Marks this Canvas as the Canvas.ruleScope that will be discovered by any contained
*  DataBoundComponents which do not specify an explicit ruleScope.
* @type {boolean}
* @default null
*/
isRuleScope?: boolean;

/**
* ValuesManager for managing values displayed in this component.
*  If specified at initialization time, this component will be added to the valuesManager via
*  ValuesManager.addMember.
*  
*  ValuesManagers allow different fields of a single object to be displayed or edited
*  across multiple UI components. Given a single values object, a valuesManager will handle
*  determining the appropriate field values for its member components and displaying them / 
*  responding to edits if the components support this.
*  
*  Data may be derived simply from the specified fieldNames within the member components, or for
*  complex nested data structures can be specified by both component and field-level
*  DataPath.
*  
*  Note that components may be automatically bound to an existing valuesManager attached to a 
*  parent component if dataPath is specified. See Canvas.dataPath for more information.
*  Also note that if a databound component has a specified dataSource and dataPath but no specified
*  valuesManager object one will be automatically generated as part of the databinding process
* @type {ValuesManager | GlobalId}
* @default null
*/
valuesManager?: ValuesManager | GlobalId;

/**
* A dataPath may be specified on any canvas. This provides a straightforward way to display or
*  edit complex nested data.
*  
*  For components which support displaying or editing data values, (such as DynamicForm or
*  ListGrid components), the dataPath may be set to specify how the components data is
*  accessed. In this case the dataPath essentially specifies a nested object to edit - typically
*  a path to a field value within a dataSource record. Note that a ValuesManager will be required
*  to handle connecting the dataBoundcomponent to the appropriate sub object. This may be explicitly
*  specified on the component, or a parent of the component, or automatically generated
*  if a DataSource is specified on either the component or a parent thereof.
*  
*  To provide a simple example - if a complex object existed with the following format:
*  
*  
*  { companyName:"Some Company",
*   address:{  street:"123 Main Street", city:"New York", state:"NY" }
*  }
*  
*  
*  
*  a developer could specify a DynamicForm instance with 'dataPath' set to "address" to edit
*  the nested address object:
*  
*  
*  isc.ValuesManager.create({
*    ID:'vm',
*    values: { companyName:"Some Company",
*        address:{  street:"123 Main Street", city:"New York", state:"NY" }
*    }
*  });
* 
*  isc.DynamicForm.create({
*    valuesManager:"vm",
*    dataPath:"address",
*    items:[{name:"street"}, {name:"city"}, {name:"state"}]
*  });
*  
*  
*  
*  If a component is specified with a dataPath attribute but does not have an
*  explicitly specified valuesManager, it will check its parent element chain for a specified
*  valuesManager and automatically bind to that. This simplifies binding multiple components used
*  to view or edit a nested data structure as the valuesManager needs only be defined once at a
*  reasonably high level component. Here's an example of this approach:
*  
*  
*  isc.ValuesManager.create({
*    ID:'vm',
*    values: { companyName:"Some Company",
*        address:{  street:"123 Main Street", city:"New York", state:"NY" }
*    }
*  });
* 
*  isc.Layout.create({
*    valuesManager:vm,
*    members:[
*      isc.DynamicForm.create({
*        dataPath:"/",
*        items:[{name:"companyName"}]
*      }),
*      isc.DynamicForm.create({
*        dataPath:"address",
*        items:[{name:"street"}, {name:"city"}, {name:"state"}]
*      })
*    ]
*  });
*  
*  
*  
*  Note that in this case the valuesManager is specified on a Layout, which has no 'values'
*  management behavior of its own, but contains items with a specified dataPath which do. In this
*  example you'd see 2 forms allowing editing of the nested data structure.
*  
*  dataPaths from multiple nested components may also be combined. For example:
*  
*  
*  isc.ValuesManager.create({
*    ID:'vm',
*    values: { companyName:"Some Company",
*        address:{  street:"123 Main Street", city:"New York", state:"NY" }
*        parentCompany:{
*          companyName:"Some Corporation",
*          address:{  street:"1 High Street", city:"New York", state:"NY" }
*        }
*    }
*  });
* 
*  isc.Layout.create({
*    valuesManager:vm,
*    members:[
*      isc.DynamicForm.create({
*        dataPath:"/",
*        items:[{name:"companyName"}]
*      }),
*      isc.DynamicForm.create({
*        dataPath:"address",
*        items:[{name:"street"}, {name:"city"}, {name:"state"}]
*      }),
*      isc.Layout.create({
*        dataPath:"parentCompany",
*        members:[
*          isc.DynamicForm.create({
*            dataPath:"",
*            items:[{name:"companyName", type:"staticText"}]
*          }),
*          isc.DetailViewer.create({
*            dataPath:"address",
*            fields:[{name:"street", name:"city", name:"state"}]
*          })
*        ]
*      })
*    ]
*  });
*  
*  
*  
*  In this example the detailViewer will display data from the parentCompany.address
*  object within the base record.
*  
*  Note that if a component has a specified dataSource and shows child components with a
*  specified dataPath, there is no need to explicitly declare a valuesManager at all. If a component
*  with a dataPath has a dataSource, or an ancestor with a dataSource specified, it will, a
*  valuesManager will automatically be generated on the higher level component (and be available as
*  component.valuesManager).
*  
*  Difference between "" and "/" - relative and absolute datapaths
*  In the above example, note how the form for entering the "main" company name is given a 
*  dataPath of "/", while the form for entering the parent company name is given a dataPath of
*  "". The difference here is exactly the same as you would find in a filesystem path: a 
*  dataPath starting with "/" is absolute, so "/" by itself means "root". A dataPath that 
*  does not start with "/" is relative, and the empty string indicates that dataPaths for 
*  items below this one in the hierarchy should apply the dataPath so far from the hierarchy 
*  above. 
*  
*  If that isn't clear, consider the form for entering the parent company name in the above 
*  example. The correct dataPath to the field is /parentCompany/companyName. We
*  have the "parentCompany" part of that path provided by the containing Layout, so we cannot 
*  reiterate it on the form itself. However, if we omit the dataPath property altogether, the
*  framework will not seek to apply dataPath at all. So, we specify the empty string, which
*  tells SmartClient to use dataPath and to retain the portion of the path derived so far from 
*  the containment hierarchy.
*  
*  One further clarification: relative paths are only different from absolute paths if they are
*  relative to something other than the root. So in the above example, although we specify 
*  "/" as the dataPath of the "main" company name form, we would get exactly the same behavior
*  by specifying it as "", because the correct dataPath for the companyName field is 
*  "/companyName" - ie, it is relative to the root.
* @type {DataPath}
* @default null
*/
dataPath?: DataPath;

/**
* Local name for referencing this canvas from an autoTest locator string. It will be
*  used instead of index if found. This name must by unique within the parent
*  component.
*  
*  By setting a static ID on certain top-level components and then using locatorName
*  in contained components, stable locators can be created for these components without
*  the need to pervasively assign IDs.
* @type {string}
* @default null
*/
locatorName?: string;

/**
* Strategy to use when locating children in this canvas from an autoTest locator string.
* @type {LocatorStrategy}
* @default null
*/
locateChildrenBy?: LocatorStrategy;

/**
* LocatorTypeStrategy to use when finding children within this canvas.
* @type {LocatorTypeStrategy}
* @default null
*/
locateChildrenType?: LocatorTypeStrategy;

/**
* Strategy to use when locating peers of this canvas from an autoTest locator string.
* @type {LocatorStrategy}
* @default null
*/
locatePeersBy?: LocatorStrategy;

/**
* LocatorTypeStrategy to use when finding peers of this canvas.
* @type {LocatorTypeStrategy}
* @default null
*/
locatePeersType?: LocatorTypeStrategy;

/**
* When nodes are added to an EditContext, should they be masked by setting
*  EditProxy.useEditMask true if not explicitly set?
* @deprecated * As of SmartClient version 10.0, deprecated in favor of EditProxy.autoMaskChildren

* @type {boolean}
* @default null
*/
autoMaskComponents?: boolean;

/**
* An EditProxy controls the behaviors of a component when it is placed into
*  devTools.
*  
*  The editProxy AutoChild is created when a component is first placed into
*  edit mode via Canvas.setEditMode.
*  
*  editProxy properties can be supplied on a PaletteNode or
*  EditNode as PaletteNode.editProxyProperties, but must
*  be provided before the component is first placed into edit mode.
*  
*  Most editable components use a custom EditProxy. See the documentation for
*  each class' Canvas.editProxyConstructor to determine
*  the class.
* @type {EditProxy}
* @default null
*/
editProxy?: EditProxy;

/**
* Default class used to construct the EditProxy for this component
*  when the component is Canvas.setEditMode.
* @type {SCClassName}
* @default "CanvasEditProxy"
*/
editProxyConstructor?: SCClassName;

/**
* The component's EditNode for a component that has been created by a
*  Palette from a PaletteNode.
* @type {EditNode}
* @default null
*/
editNode?: EditNode;

/**
* ARIA role of this component. Usually does not need to be manually set - see
*  accessibility.
* @type {string}
* @default null
*/
ariaRole?: string;

/**
* ARIA state mappings for this component. Usually this does not need to be manually
*  set - see accessibility.
*  
*  This attribute should be set to a mapping of aria state-names to values - for example
*  to have the "aria-haspopup" property be present with a value "true", you'd specify:
*  
*  { haspopup : true }
*  
* @type {object}
* @default null
*/
ariaState?: object;

/**
* If this canvas is being displayed in a CanvasItem, this property will be set
*  to point at the item. Otherwise this property will be null.
* @type {CanvasItem}
* @default null
*/
canvasItem?: CanvasItem;

/**
* Default total duration of animations. Can be overridden by setting animation times for
*  specific animations, or by passing a duration parameter into the appropriate
*  animate...() method.
* @type {number}
* @default "300"
*/
animateTime?: number;

/**
* Default acceleration effect to apply to all animations on this Canvas.
*  Can be overridden by setting animationAcceleration for specific animations or by passing
*  an acceleration function directly into the appropriate method.
* @type {AnimationAcceleration}
* @default "smoothEnd"
*/
animateAcceleration?: AnimationAcceleration;

/**
* Default animation effect to use if Canvas.animateShow is called without an
*  explicit effect parameter
* @type {AnimateShowEffectId | AnimateShowEffect}
* @default "wipe"
*/
animateShowEffect?: AnimateShowEffectId | AnimateShowEffect;

/**
* Default animation effect to use if Canvas.animateHide is called without an
*  explicit effect parameter
* @type {AnimateShowEffectId | AnimateShowEffect}
* @default "wipe"
*/
animateHideEffect?: AnimateShowEffectId | AnimateShowEffect;

/**
* Default time for performing an animated move. If unset, this.animateTime
*  will be used by default instead
* @type {number}
* @default null
*/
animateMoveTime?: number;

/**
* Default time for performing an animated resize. If unset, this.animateTime
*  will be used by default instead
* @type {number}
* @default null
*/
animateResizeTime?: number;

/**
* Default time for performing an animated setRect. If unset, this.animateTime
*  will be used by default instead
* @type {number}
* @default null
*/
animateRectTime?: number;

/**
* Default time for performing an animated fade. If unset, this.animateTime
*  will be used by default instead
* @type {number}
* @default null
*/
animateFadeTime?: number;

/**
* Default time for performing an animated scroll. If unset, this.animateTime
*  will be used by default instead
* @type {number}
* @default null
*/
animateScrollTime?: number;

/**
* Default time for performing an animated show. If unset, this.animateTime
*  will be used by default instead
* @type {number}
* @default null
*/
animateShowTime?: number;

/**
* Default time for performing an animated hide. If unset, this.animateTime
*  will be used by default instead
* @type {number}
* @default null
*/
animateHideTime?: number;

/**
* Default acceleration effect for performing an animated move. If unset, 
*  this.animateAcceleration will be used by default instead
* @type {AnimationAcceleration}
* @default null
*/
animateMoveAcceleration?: AnimationAcceleration;

/**
* Default acceleration function for performing an animated resize. If unset, 
*  this.animateAcceleration will be used by default instead
* @type {AnimationAcceleration}
* @default null
*/
animateResizeAcceleration?: AnimationAcceleration;

/**
* Default acceleration function for performing an animated move and resize. If unset, 
*  this.animateAcceleration will be used by default instead
* @type {AnimationAcceleration}
* @default null
*/
animateRectAcceleration?: AnimationAcceleration;

/**
* Default acceleration function for performing an animated scroll. If unset, 
*  this.animateAcceleration will be used by default instead
* @type {AnimationAcceleration}
* @default null
*/
animateScrollAcceleration?: AnimationAcceleration;

/**
* Default acceleration function for performing an animated show. If unset, 
*  this.animateAcceleration will be used by default instead
* @type {AnimationAcceleration}
* @default null
*/
animateShowAcceleration?: AnimationAcceleration;

/**
* Default acceleration function for performing an animated hide. If unset, 
*  this.animateAcceleration will be used by default instead
* @type {AnimationAcceleration}
* @default null
*/
animateHideAcceleration?: AnimationAcceleration;

/**
* When to update the Canvas.layoutChildren for a Canvas.animateResize. Updating the child layout more often may improve appearance, but risks
*  prohibitive overhead with more complicated widget hierarchies.
* @type {AnimationLayoutMode}
* @default "atEnd"
*/
animateResizeLayoutMode?: AnimationLayoutMode;

/**
* When this Canvas is included as a member in a Layout, layoutAlign controls alignment on the
*  breadth axis of the layout. Default is "left" for a VLayout, "top" for an HLayout.
* @type {Alignment | VerticalAlignment}
* @default null
*/
layoutAlign?: Alignment | VerticalAlignment;

/**
* When this Canvas is included as a member in a Layout, whether a resizeBar should be shown
*  after this member in the layout, to allow it to be resized.
*  
*  Whether a resizeBar is actually shown also depends on the 
*  Layout.defaultResizeBars attribute of the layout, and whether this
*  Canvas is the last layout member.
*  
*  By default the resize bar acts on the Canvas that it is declared on. If you want the resize
*  bar to instead act on the next member of the Layout (e.g. to collapse down or to the right),
*  set Canvas.resizeBarTarget as well.
* @type {boolean}
* @default false
*/
showResizeBar?: boolean;

/**
* When this Canvas is included as a member in a Layout, and Canvas.showResizeBar is set to
*  true so that a resizeBar is created, resizeBarTarget:"next" can be
*  set to indicate that the resizeBar should resize the next member of the layout rather than
*  this one. For resizeBars that support hiding their target member when clicked on, 
*  resizeBarTarget:"next" also means that the next member will be the one hidden.
*  
*  This is typically used to create a 3-way split pane, where left and right-hand sections can
*  be resized or hidden to allow a center section to expand.
*  
*  NOTE: as with any Layout, to ensure all available space is used, one or more members
*  must maintain a flexible size (eg 75%, or *). In a two pane Layout with a normal resize
*  bar, to fill all space after a user resizes, the member on the right should have
*  flexible size. With resizeBarTarget:"next", the member on the left should have
*  flexible size.
* @type {string}
* @default null
*/
resizeBarTarget?: string;

/**
* When this Canvas is included as a member in a Layout, extra blank space that should be left
*  after this member in a Layout.
* @type {PositiveInteger}
* @default "0"
*/
extraSpace?: PositiveInteger;

/**
* Whether this canvas should always start a new line when used as a tile in a 
*  FlowLayout. This property is not supported in a TileLayout with
*  TileLayout.layoutPolicy: "fit" or if databound (i.e. for a TileGrid).
* @type {boolean}
* @default false
*/
startLine?: boolean;

/**
* Whether this canvas should end the line it's in when used as a tile in a FlowLayout.
*  This property is not supported in a TileLayout with TileLayout.layoutPolicy:
*  "fit" or if databound (i.e.TileGrid).
* @type {boolean}
* @default false
*/
endLine?: boolean;



/**
*  Get the CSS class to apply when printing this widget. Returns the
*  Canvas.printStyleName, if specified, falling back to the 
*  Canvas.styleName otherwise.
* 
*/
getPrintStyleName(): CSSStyleName;

/**
*  See Canvas.canAdaptWidth.
* 
* @param {Integer} surplus (if positive) or overflow (if negative)
* @param {Integer} width of member currently assumed by parent layout
*/
adaptWidthBy(pixelDifference:Integer, unadaptedWidth:Integer): Integer;

/**
*  See Canvas.canAdaptWidth.
* 
* @param {Integer} surplus (if positive) or overflow (if negative)
* @param {Integer} height of member currently assumed by parent layout
*/
adaptHeightBy(pixelDifference:Integer, unadaptedHeight:Integer): Integer;

/**
* 
*  This method performs some basic initialization common to all UI components. To do custom UI
*  component initialization, you should generally override Canvas.initWidget. This
*  method does the following, in order:
*  
*  
* - Sets up a global reference to this instance as described in Canvas.ID.
*  
* - Ensures certain numeric properties have numeric values (e.g. width, height, padding,
*  margin)
*  
* - Ensures Canvas.children and Canvas.peers are Arrays.
*  
* - Calls Canvas.initWidget
*  
* - Creates Canvas.showEdges and Canvas.showShadow, if so configured.
*  
* - Calls Canvas.draw if Canvas.autoDraw is set on instance or globally.
*  
*  Unless you're in an advanced scenario where you need to inject code before the above
*  logic executes, place your initialization logic in initWidget() rather than init(). If you
*  do decided to override this method, you must call the superclass implementation like so:
*  
*   this.Super("init", arguments);
*  
* 
* 
* @param {any=} All arguments initially passed to Class.create
*/
init(...args:any[]): void;

/**
* 
*  For custom components, perform any initialization specific to your widget subclass.
*  
*  When creating a subclass of any Canvas-based component, you should generally override
*  this method rather than overriding Class.init. This is because Canvas has its own
*  Class.init override which performs some generally desirable initialization - see
*  Canvas.init for details.
*  
*  This method is called by Canvas.init when a component is create()d. When
*  overriding this method, You must call the superClass initWidget implementation, like
*  so:
*  
*   this.Super("initWidget", arguments);
*  
*  
*  In general, if you are going to call functionality supported by your superclass (eg calling
*  addTab() when your superclass is a TabSet), call Super() first. However, you can generally
*  assign properties to this before calling Super() as a way of mimicking the
*  effect of the property being passed to Class.create on normal instance
*  construction. For example, when subclassing a DynamicForm, you could set this.items to a
*  generated set of items before calling Super().
*  
*  NOTE: child creation: if you are creating a component that auto-creates certain children (eg
*  a Window which creates a Header child), typical practice is to create those children
*  immediately before drawing by overriding draw(). This postpones work until it is really
*  necessary and avoids having to update children if settings are changed between creation and
*  draw(). Alternatively, if you prefer callers to directly manipulate auto-created children,
*  it's best to create them earlier in initWidget(), in order to allow manipulation before
*  draw.
* 
* 
* @param {any=} All arguments initially passed to Class.create
*/
initWidget(...args:any[]): void;

/**
*  Retrieve a child of this Canvas by it's Canvas.getLocalId.
* 
* 
* @param {string} local ID of the Canvas
*/
getByLocalId(localId:string): Canvas;

/**
*  Retrieve the local ID of this canvas. If no local ID is assigned the normal
*  canvas ID is returned making this method a safe replacement for Canvas.getID.
*  
*  A "local ID" is name for a child widget which is unique only for this parent, and not globally
*  unique as is required for Canvas.ID. Widgets receive local IDs when loaded via
*  RPCManager.loadScreen or RPCManager.cacheScreens and
*  RPCManager.createScreen.
* 
* 
*/
getLocalId(): string;

/**
*    Returns the boolean true, if the widget has been completely drawn, and false otherwise.
* 
*/
isDrawn(): boolean;

/**
*    When a widget instance is created, it is assigned a unique global identifier that can be
*    used to access the instance by name. The getID method returns this ID for a particular
*    instance. Global IDs are essential when you need to embed a widget reference in a string,
*    usually a string that will be evaluated in the future and/or in another object, where you
*    may not have access to a variable or parameter holding the widget's reference.
* 
* 
*/
getID(): GlobalId;

/**
*  Return the inner HTML for this canvas. Called when the canvas is drawn or redrawn;
*  override to customize.
* 
* 
*/
getInnerHTML(): HTMLString;

/**
*  Setter for Canvas.snapOffsetLeft.
* 
* @param {Integer} new snapOffsetLeft value.
*/
setSnapOffsetLeft(snapOffsetLeft:Integer): void;

/**
*  Setter for Canvas.snapOffsetTop.
* 
* @param {Integer} new snapOffsetTop value.
*/
setSnapOffsetTop(snapOffsetTop:Integer): void;

/**
*    Draws the widget on the page.
* 
*/
draw(): Canvas;

/**
*  Retrieves printable HTML for this component and all printable subcomponents.
*  
*  By default any Canvas with children will simply collect the printable HTML of its children
*  by calling getPrintHTML() on each child that is considered Canvas.shouldPrint.
*  If a callback is provided, then null is always returned and the callback is fired asynchronously.
*  
*  If overriding this method for a custom component, you should either return a String
*  of printable HTML directly or return null and fire the provided callback using
*  Class.fireCallback.
*  
*  To return an empty print representation, return an empty string ("") rather than null.
*  
*  The printProperties argument, if passed, must be passed to any subcomponents on
*  which getPrintHTML() is called.
*  
*  Default implementation will set Canvas.isPrinting flag to true to indicate
*  printing is in progress, and clear this flag when the printing has completed (possibly via
*  an asynchronous callback).
*  
*  NOTE: Expecting a direct return value from the default implementation is deprecated usage.
*  This is because small changes to an application (such as adding a few more data points to a chart
*  or adding another button) or using certain browsers can make it necessary to generate the HTML
*  asynchronously. Thus, application code should not rely on the return value and always pass
*  a callback.
* 
* 
* @param {PrintProperties=} properties to configure printing behavior - may be null.
* @param {Callback=} optional callback. This is required to handle cases where HTML         generation is asynchronous - if a method generates HTML asynchronously, it should return          null, and fire the specified callback on completion of HTML generation. The          first parameter HTML should contain the generated print HTML.          The callback is only called if null is returned. Furthermore, the default          getPrintHTML() implementation always returns null and fires the callback          when a callback is provided.
*/
getPrintHTML(printProperties?:PrintProperties, callback?:Callback): HTMLString;

/**
*  Setter for the Canvas.htmlElement.
* 
* @param {DOMElement} New htmlElement for this canvas, or Null to clear the existing               htmlElement.  Note that you may need to set position:relative                explicitly following a call to this function.
*/
setHtmlElement(element:DOMElement): void;

/**
*  Setter for the Canvas.htmlPosition.
* 
* @param {DrawPosition} New htmlPosition for this canvas
*/
setHtmlPosition(element:DrawPosition): void;

/**
*  Returns whether a canvas is waiting to be redrawn. Will return true if
*  Canvas.markForRedraw has been called, but this canvas has not yet been
*  redrawn.
* 
*/
isDirty(): boolean;

/**
*  Marks the widget as "dirty" so that it will be added to a queue for redraw. Redraw of dirty
*  components is handled by a looping timer and will after a very short delay (typically less than
*  100ms). In most cases it is recommended that developers use markForRedraw()
*  instead of calling Canvas.redraw directly. Since this method queues the redraw, multiple
*  calls to markForRedraw() within a single thread of execution will only lead to a single DOM
*  manipulation which greatly improves application performance.
* 
* 
* @param {string=} ["no reason provided"] reason for performing the redraw
*/
markForRedraw(reason?:string): void;

/**
*  Redraws the widget immediately with its current property values.
* 
*  Generally, if you want a Canvas to redraw, call markForRedraw() - this will cause the Canvas to
*  be redrawn when current processing ends, so that a series of modifications made to a Canvas will
*  cause it to redraw only once.
* 
*  Only call redraw() directly if you need immediate responsiveness, for example you are redrawing
*  in response to continuous mouse motion.
* 
* 
* @param {string=} ["no reason provided"] reason for performing the redraw
*/
redraw(reason?:string): void;

/**
*  Remove all visual representation of a Canvas, including all child or member Canvases, or
*  managed top-level components such as the ListGrid drop location indicator.
*  
*  This is more expensive than hide(), because in order to become visible again, the Canvas must
*  be draw()n again. Generally, application code has no reason to call clear() unless it is
*  attempting to do advanced memory management. If you want to temporarily hide a Canvas, use
*  hide() and show(), and if you want to permanently destroy a Canvas, use Canvas.destroy.
*  
*  You would only use clear() if you were managing a very large pool of components and you
*  wanted to reclaim some of the memory used by components that had not been used in a while,
*  while still being able to just draw() them to make them active and visible again.
*  
*  Note: a clear() will happen as part of moving a Canvas to a different parent. See
*  Canvas.addChild.
* 
* 
*/
clear(): void;

/**
*  Permanently destroy a Canvas and all of it's children / members, recursively.
*  
*  Like Canvas.clear, calling destroy() removes all HTML for the component;
*  unlike clear(), a destroyed Canvas is permanently unusable: it cannot be draw()'n again,
*  cannot be referenced by its global ID, and is eligible for garbage collection (assuming
*  that application code is not holding a reference to the Canvas).
*  
*  Any attempt to call a method on a destroyed Canvas will generally result in an error. If your
*  application is forced to hold onto Canvas's that might be destroy()d without warning, you can
*  avoid errors by checking for the Canvas.destroyed property. If you override certain Canvas
*  methods, your code may be called while a Canvas is being destroy()d; in this case you can avoid
*  extra work (and possibly errors) by checking for the Canvas.destroying property.
*  
*  Note that destroy() should not be called directly in event handling code for this
*  canvas. For this reason, wherever possible we recommend using Canvas.markForDestroy
*  instead of calling this method directly.
*  
*  Also note that developers should not destroy() or markForDestroy() a component
*  while it is in the middle of an asynchronous operation. For example, if you need to submit
*  and then destroy a single-use dynamic form, the call to markForDestroy() should
*  be invoked from the callback to DynamicForm.saveData, rather than being invoked
*  synchronously after the the call to saveData().
* 
* 
*/
destroy(): void;

/**
*  Canvas.destroy this canvas on a timeout. This method should be used instead of
*  calling canvas.destroy() directly unless there's a reason a the canvas needs to be
*  destroyed synchronously. By using a timeout, this method ensures the destroy() will
*  occur after the current thread of execution completes. This allows you to easily mark canvases
*  for destruction while they're handling events, which must complete before the canvas can be
*  destroyed.
*  
*  Notes:
*  
*  
* - markForDestroy() performs some immediate cleanup and puts a component into a
*  "pending destroy" state. As far as application code is concerned, once a component has been
*  is in this state it should be considered invalid to invoke methods on the component.
*  
* - Developers should not destroy() or markForDestroy() a component
*  while it is in the middle of an asynchronous operation. For example, if you need to submit
*  and then destroy a single-use dynamic form, the call to markForDestroy() should
*  be invoked from the callback to DynamicForm.saveData, rather than being invoked
*  synchronously after the call to saveData()
*  
* 
* 
*/
markForDestroy(): void;

/**
*  Returns the DOM ID for the main element for this canvas which will have 
*  ARIA Canvas.ariaRole
*  and Canvas.ariaState applied to it. This can be useful when applying
*  Canvas.ariaState which need to refer to the DOM handle
*  of another canvas, such as 
*  aria-controls.
*  
*  Note that we do not recommend using this ID to identify the element in the DOM for integration
*  with automated testing tools. The AutoTest.getLocator and AutoTest.getElement
*  APIs should be used instead. These locators are much finer grained 
*  - they allow developers to identify sub-elements within the widget handle 
*  that map to specific logical SmartClient objects (for example FormItem Icons
*  within a DynamicForm). They are also more stable - AutoTest locators are able to reliably
*  identify components based on their position in the application hierarchy and other context, 
*  while the DOM element IDs are not guaranteed not to change across page reloads.
* 
* 
*/
getAriaHandleID(): string;

/**
*  Returns the outer DOM element of this Canvas. This method is provided for the 
*  very rare cases where a programmer needs to examine the DOM hierarchy created by
*  a drawn SmartClient component.
*  
*  Direct manipulation of the DOM elements created by SmartClient components
*  is not supported. SmartClient components should be rendered or cleared using
*  standard methods such as Canvas.draw, Canvas.clear. If direct
*  integration with existing DOM structures is required, this should be achieved via the
*  Canvas.htmlElement attribute, rather than by attempting to move the
*  component's outer element via native browser APIs.
*  The content of SmartClient components' DOM elements should also not be directly 
*  manipulated using native browser APIs - standard methods such as 
*  Canvas.setContents, Canvas.addChild, Canvas.removeChild,
*  Canvas.markForRedraw and Canvas.redraw should be used instead.
*  
*  In some cases, the element returned may match the element returned by 
*  Canvas.getContentElement, but this will not always be the case.
*  
*  If the widget is undrawn, this method will return null.
* 
* 
*/
getOuterElement(): DOMElement;

/**
*  Returns the DOM element for this Canvas which contains the Canvas.contents, or
*  for Canvas.getParentCanvas, the DOM elements for any drawn
*  children. This method is provided for the 
*  very rare cases where a programmer needs to examine the DOM hierarchy created by
*  a drawn SmartClient component.
*  
*  Direct manipulation of the DOM elements created by SmartClient components
*  is not supported. SmartClient components should be rendered or cleared using
*  standard methods such as Canvas.draw, Canvas.clear. If direct
*  integration with existing DOM structures is required, this should be achieved via the
*  Canvas.htmlElement attribute, rather than by attempting to move the
*  component's outer element via native browser APIs.
*  The content of SmartClient components' DOM elements should also not be directly 
*  manipulated using native browser APIs - standard methods such as 
*  Canvas.setContents, Canvas.addChild, Canvas.removeChild,
*  Canvas.markForRedraw and Canvas.redraw should be used instead.
*  
*  In some cases, the element returned may match the element returned by
*  Canvas.getOuterElement, but this will not always be the case.
*  
*  If the widget is undrawn, this method will return null.
* 
* 
*/
getContentElement(): DOMElement;

/**
*  Adds newChild as a child of this widget, set up a named object reference (i.e., this[name])
*  to the new widget if name argument is provided, and draw the child if this widget has been
*  drawn already.
*  
*  If newChild has a parent it will be removed from it. If it has a master, it will be detached
*  from it if the master is a child of a different parent. If newChild has peers, they'll be
*  added to this widget as children as well.
* 
* 
* @param {Canvas} new child canvas to add
* @param {string=} name to assign to child (eg: this[name] == child)
* @param {boolean=} if false, child will not automatically be drawn (only                     for advanced use)
*/
addChild(newChild:Canvas, name?:string, autoDraw?:boolean): Canvas;

/**
*  Remove a peer from this Canvas
* 
* @param {Canvas} Peer to be removed from this canvas
* @param {string=} If this peer was assigned a name when added via addPeer(), it should            be passed in here to ensure no reference is kept to the peer
*/
removePeer(peer:Canvas, name?:string): void;

/**
*  Make this Canvas no longer a peer of its master
* 
*/
depeer(): void;

/**
*  Remove this canvas from its parent if it has one.
* 
*/
deparent(): void;

/**
*  Remove a child from this parent.
* 
* @param {Canvas} Child canvas to remove from this parent.
* @param {string=} If the child canvas was assigned a name when added via addChild(), it            should be passed in here to ensure no reference is kept to the child
*/
removeChild(child:Canvas, name?:string): void;

/**
*  Adds newPeer as a peer of this widget (also making it a child of this widget's parent, if
*  any), set up a named object reference (i.e., this[name]) to the new widget if name is
*  provided, and draw the peer if this widget has been drawn already.
*  
*  The widget to be added as a peer will be removed from its old master and/or parent, if any,
*  and it will be added as a child to the parent of this canvas (if any)
* 
* 
* @param {Canvas} new peer widget to add
* @param {string=} name to assign to peer (eg: this[peer] == child)
* @param {boolean=} if true, peer will not automatically be drawn (only               for advanced use)
* @param {boolean=} if true, when draw is called on the master widget, the peer              will be drawn before the master
*/
addPeer(newPeer:Canvas, name?:string, autoDraw?:boolean, preDraw?:boolean): Canvas;

/**
*  Set the snapTo property of this canvas, and handle repositioning.
* 
* 
* @param {string} new snapTo value
*/
setSnapTo(snapTo:string): void;

/**
*  Return the snapTo value of this object
* 
* 
*/
getSnapTo(): string;

/**
*  Set the snapEdge property of this canvas, and handle repositioning.
* 
* 
* @param {string} new snapEdge value
*/
setSnapEdge(snapEdge:string): void;

/**
*  Return the snapEdge value of this object
* 
* 
*/
getSnapEdge(): string;

/**
*  Returns the parent of this canvas, if any.
* 
*/
getParentCanvas(): Canvas;

/**
*  Returns this canvas's "master" (the canvas to which it was added as a peer), if any.
* 
*/
getMasterCanvas(): Canvas;

/**
*  Returns an array of object references to all ancestors of this widget in the containment
*  hierarchy, starting with the direct parent and ending with the top element.
* 
*/
getParentElements(): Array<Partial<Canvas>>;

/**
*    Returns true if element is a descendant of this widget (i.e., exists below this widget in
*    the containment hierarchy); and false otherwise.
* 
* @param {Canvas} the canvas to be tested
* @param {boolean=} If passed this method will return true if the canvas                parameter is a pointer to this widget.
*/
contains(canvas:Canvas, testSelf?:boolean): boolean;

/**
*  Returns true if the current EventHandler.getTarget is this component
*  or a descendent of this component.
*  
*  Note that, unlike Canvas.containsEvent, this method is not based on reported event coordinates
*  and there are cases where containsEvent() and containsEventTarget() will
*  return different values, including when the mouse is within the bounds of a target component, but
*  another canvas is rendered in front of it in the page's z-order.
*  
* 
*/
containsEventTarget(): boolean;

/**
*  Show a clickMask over the entire screen that intercepts mouse clicks and fires some action.
*  The mask created will be associated with this canvas - calling this method multiple times
*  will not show multiple (stacked) clickMasks if the mask associated with this canvas is
*  already up.
*  
*  The clickMask useful for modal dialogs, menus and similar uses, where any click outside of
*  some Canvas should either be suppressed (as in a modal dialog) or just cause something (like
*  dismissing a menu).
* 
* 
* @param {Callback} action to fire when the user clicks on the mask
* @param {ClickMaskMode} whether to automatically hide the clickMask on mouseDown                   and suppress the mouseDown event from reaching                    the target under the mouse
* @param {any | Array<Partial<any>>} initially unmasked targets for this clickMask. Note that if this is a "hard"  mask, unmasked children of masked parents are not supported so any non-top-level widgets  passed in will have their parents unmasked. Children of unmasked parents can never be  masked, so you need only include the top widget of a hierarchy.
*/
showClickMask(clickAction:Callback, mode:ClickMaskMode, unmaskedTargets:any | Array<Partial<any>>): string;

/**
*  Hides the click mask associated with this canvas.
* 
* @param {string=} optional ID of specific clickMask to hide. If not passed,           defaults to hiding the click mask associated with this widget only.
*/
hideClickMask(ID?:string): void;

/**
*  Determines whether a clickmask is showing
* 
* @param {string=} optional ID of specific clickMask to check. If not passed,           checks for the click mask associated with this widget only.
*/
clickMaskUp(ID?:string): boolean;

/**
*  Temporariy block all user interaction with children of this widget, with the exception of those
*  passed in in the unmaskedChildren parameter. Children will remain blocked until
*  Canvas.hideComponentMask is called.
*  
*  This method will show the Canvas.componentMask canvas to block mouse interaction with
*  children, and temporarily remove masked children from the page's tab-order.
*  
*  This behavior differs from the standard Canvas.showClickMask in that the
*  modal mask shown by Canvas.showClickMask will cover the entire screen and typically only 
*  allow "unmasking" of top level components.
*  
*  Use Canvas.hideComponentMask to hide the component level mask.
* 
* 
* @param {Array<Partial<Canvas>>=} Children passed into this parameter will continue to    be interactive while other children are blocked. They will be moved above the componentMask in    the page's z-order and remain accessible via keyboard navigation. Note that this array should    contain direct children of this widget only.
*/
showComponentMask(unmaskedChildren?:Array<Partial<Canvas>>): void;

/**
*  Hide the Canvas.showComponentMask for this widget
* 
* @param {boolean=} By default when the component-level mask is hidden   it will attempt to reset focus to whatever had focus before the mask was shown.    Pass this parameter to suppress this behavior.
*/
hideComponentMask(suppressFocusReset?:boolean): void;

/**
*  Set all four coordinates, relative to the enclosing context, at once.
*  
*  Moves the widget so that its top-left corner is at the specified top-left coordinates,
*  then resizes it to the specified width and height.
* 
* 
* @param {number | Array<any> | object=} new left coordinate, Array of                         coordinates in parameter order, or Object                          with left, top, width, height properties.                          If an Array or Object is passed, the                          remaining parameters are ignored.
* @param {number=} new top coordinate
* @param {number=} new width
* @param {number=} new height
*/
setRect(left?:number | Array<any> | object, top?:number, width?:number, height?:number): boolean;

/**
*      Return the left coordinate of this object, relative to its enclosing context, in pixels.
* 
*/
getLeft(): number;

/**
*      Set the left coordinate of this object, relative to its enclosing context, in pixels.
*      NOTE: if you're setting multiple coordinates, use setRect(), moveTo() or resizeTo()
*      instead
* 
* @param {number | string} new left coordinate
*/
setLeft(left:number | string): void;

/**
*  Return the top coordinate of this object, relative to its enclosing context, in pixels.
* 
*/
getTop(): number;

/**
*  Set the top coordinate of this object, relative to its enclosing context, in pixels.
*  
*  NOTE: if you're setting multiple coordinates, use setRect() or moveTo() instead
* 
* 
* @param {number | string} new top coordinate
*/
setTop(top:number | string): void;

/**
*  Setter for Canvas.leavePageSpace.
* 
* 
* @param {Integer} new value for leavePageSpace.
*/
setLeavePageSpace(newPageSpace:Integer): void;

/**
*  Return the width of this object, in pixels.
* 
*/
getWidth(): number;

/**
*  Resizes the widget horizontally to the specified width (moves the right side of the
*  widget). The width parameter can be expressed as a percentage of viewport size or as
*  the number of pixels.
*  
*  NOTE: if you're setting multiple coordinates, use resizeTo() or setRect() instead
* 
* 
* @param {number | string} new width
*/
setWidth(width:number | string): void;

/**
*  Return the height of this object, in pixels.
* 
*/
getHeight(): number;

/**
*  Resizes the widget vertically to the specified height (moves the bottom side of the
*  widget). The height parameter can be expressed as a percentage of viewport size or as
*  the number of pixels.
*  
*  NOTE: if you're setting multiple coordinates, use resizeTo() or setRect() instead
* 
* 
* @param {number | string} new height
*/
setHeight(height:number | string): void;

/**
*  Resizes the widget horizontally if required to satisfy the specified Canvas.minWidth.
* 
* 
* @param {number} new minimum width
*/
setMinWidth(width:number): void;

/**
*  Resizes the widget vertically if required to satisfy the specified Canvas.minHeight.
* 
* 
* @param {number} new minimum height
*/
setMinHeight(height:number): void;

/**
*  Return the right coordinate of this object as rendered, relative to its enclosing context,
*  in pixels.
* 
* 
*/
getRight(): number;

/**
*  Resizes the widget horizontally to position its right side at the specified coordinate.
*  
*  NOTE: if you're setting multiple coordinates, use setRect(), moveTo() or resizeTo()
*  instead
* 
* 
* @param {number} new right coordinate
*/
setRight(right:number): void;

/**
*  Return the bottom coordinate of this object as rendered, relative to its enclosing context,
*  in pixels.
* 
* 
*/
getBottom(): number;

/**
*  Resizes the widget vertically to position its bottom edge at the specified coordinate.
*  
*  NOTE: if you're setting multiple coordinates, use setRect(), moveTo() or resizeTo()
*  instead
* 
* 
* @param {number} new bottom coordinate
*/
setBottom(bottom:number): void;

/**
*  Returns the scrollable width of the widget's contents, including children, ignoring
*  clipping.
* 
*/
getScrollWidth(): number;

/**
*      Returns the scrollable height of the widget's contents, including children, ignoring
*      clipping.
* 
*/
getScrollHeight(): number;

/**
*  Get the number of pixels this Canvas is scrolled from its left edge.
* 
*/
getScrollLeft(): number;

/**
*  Get the number of pixels this Canvas is scrolled from its top edge.
* 
*/
getScrollTop(): number;

/**
*  Set the page-relative left coordinate of this widget.
* 
* 
* @param {number} new left coordinate in pixels
*/
setPageLeft(left:number): void;

/**
*  Set the page-relative top coordinate of this widget.
* 
* 
* @param {number} new top coordinate in pixels
*/
setPageTop(top:number): void;

/**
*  Returns the page-relative left coordinate of the widget on the page, in pixels.
* 
*/
getPageLeft(): number;

/**
*  Returns the page-relative top coordinate of the widget on the page, in pixels
* 
*/
getPageTop(): number;

/**
*  Return the page-relative right coordinate of this object, in pixels.
* 
* 
*/
getPageRight(): number;

/**
*  Return the page-relative bottom coordinate of this object, in pixels.
* 
*/
getPageBottom(): number;

/**
*  Set the CSS Margin, in pixels, for this component. Margin provides blank space outside of
*  the border.
*  
*  This property sets the same thickness of margin on every side. Differing per-side
*  margins can be set in a CSS style and applied via Canvas.styleName.
*  
*  Note that the specified size of the widget will be the size including the margin
*  thickness on each side.
* 
* 
* @param {number} new margin in pixels
*/
setMargin(margin:number): void;

/**
*  Set the CSS padding of this component, in pixels. Padding provides space between the border
*  and the component's contents.
*  
*  This property sets the same thickness of padding on every side. Differing per-side
*  padding can be set in a CSS style and applied via Canvas.styleName.
*  
* 
* @param {number} new padding in pixels
*/
setPadding(newPadding:number): void;

/**
*  Return whether or not this object contains the specified global (x,y) coordinates.
*  
*  Will return false if any parent canvas does not contain the specified point, (EG:
*  you're hovering over an element's absolute location, but it is scrolled out of
*  view in a parent element)
* 
* 
* @param {Integer} GLOBAL x-coordinate
* @param {Integer} GLOBAL y-coordinate
* @param {boolean=} point lies specifically within our viewport                       (drawn area excluding margins and scrollbars if                        present)
*/
containsPoint(x:Integer, y:Integer, withinViewport?:boolean): boolean;

/**
*  Does this widget contain the specified global (x,y) coordinates, and have no other widgets
*  also at the specified position, obscuring this one? This is commonly used for (for example)
*  drag and drop interactions.
* 
* 
* @param {number} GLOBAL x-coordinate
* @param {number} GLOBAL y-coordinate
* @param {boolean=} point lies within our viewport rather than                       just our drawn area
* @param {Canvas=} If passed ignore widget(s), do not check whether                     those widgets occludes this one.
* @param {Canvas=} If passed, only check for siblings occluding the       component up as far as the specified parent widget.
*/
visibleAtPoint(x:number, y:number, withinViewport?:boolean, ignoreWidgets?:Canvas, upToParent?:Canvas): boolean;

/**
*      Returns true if the rectangles of this widget and the specified widget overlap.
* 
* @param {Canvas} other canvas to test for intersection
*/
intersects(other:Canvas): boolean;

/**
*      Returns true if the rectangle of this widget intersects with the rectangle coordinates
*      passed in, and false otherwise.
* 
* @param {number | Array<any>} left coord of rect (or rect array)
* @param {number} top coord of rect
* @param {number} width of rect
* @param {number} height of rect
*/
intersectsRect(left:number | Array<any>, top:number, width:number, height:number): boolean;

/**
*  Returns true if the rectangle of this widget encloses the rectangle of the specified widget.
* 
* @param {Canvas} other canvas to test for enclosure
*/
encloses(other:Canvas): boolean;

/**
*  Returns true if the rectangle of this widget encloses the rectangle coordinates
*  passed in, and false otherwise.
* 
* @param {number | Array<Partial<number>>} left coord of rect (or rect array)
* @param {number} top coord of rect
* @param {number} width of rect
* @param {number} height of rect
*/
enclosesRect(left:number | Array<Partial<number>>, top:number, width:number, height:number): boolean;

/**
*  Return true if the last event's mouse coordinates are within the bounds of this component.
*  
*  NOTE: Z-ordering is not considered for the purposes of this test. If the coordinate you're
*  testing is occluded by other component, but the X,Y coordinates are still within the bounds
*  that component, this method will return true.
*  
*  See the related Canvas.containsEventTarget method for checking whether a canvas 
*  contains the target canvas for the current mouse event.
* 
* 
*/
containsEvent(): boolean;

/**
*  Return the X-coordinate of the last event relative to the left edge of the content of this
*  Canvas.
* 
*  NOTE: To get a coordinate relative to the viewport of this Canvas, subtract
*  this.getScrollLeft()
* 
* 
*/
getOffsetX(): number;

/**
*  Return the Y-coordinate of the last event, relative to the top edge of the content of this
*  Canvas.
* 
*  NOTE: To get a coordinate relative to the viewport of this Canvas, subtract
*  this.getScrollTop()
* 
* 
*/
getOffsetY(): number;

/**
*  Returns the thickness of this widget's scrollbars.
*  For canvases showing custom scrollbars this is determined from this.scrollbarSize
* 
* 
*/
getScrollbarSize(): number;

/**
*  Returns the width of the viewport onto the scrollable content.
* 
* 
*/
getViewportWidth(): number;

/**
*  Returns the height of the viewport onto the scrollable content.
* 
* 
*/
getViewportHeight(): number;

/**
*  Returns the amount of space available for (an) absolutely positioned child widget(s) or
*  absolutely positioned HTML content, without introducing clipping, scrolling or overflow.
*  
*  This is the space within the viewport of the widget (including padding, but excluding
*  margins, borders or scrollbars) rendered at its specified size.
* 
* 
*/
getInnerHeight(): number;

/**
*  Returns the amount of space available for absolutely positioned child widget(s) or
*  absolutely positioned HTML content, without introducing clipping, scrolling or overflow.
*  
*  This is the space within the viewport of the widget (including padding, but excluding
*  margins, borders or scrollbars) rendered at its specified size.
* 
* 
*/
getInnerWidth(): number;

/**
*  Returns the amount of space available for interior content (or relatively positioned child
*  widget(s)) without introducing clipping, scrolling or overflow.
*  This is the space within the viewport of the widget (not including padding, and excluding
*  margins, borders or scrollbars) rendered at its specified size.
* 
* 
*/
getInnerContentHeight(): number;

/**
*  Returns the amount of space available for interior content (or relatively positioned child
*  widget(s)) without introducing clipping, scrolling or overflow.
*  This is the space within the viewport of the widget (not including padding, and excluding
*  margins, borders or scrollbars) rendered at its specified size.
* 
* 
*/
getInnerContentWidth(): number;

/**
*    Return the visible width of the Canvas.
* 
* 
*/
getVisibleWidth(): number;

/**
*    Return the visible height of the Canvas.
* 
* 
*/
getVisibleHeight(): number;

/**
*      Moves the widget deltaX pixels to the right and deltaY pixels down. Pass negative
*      numbers to move up and/or to the left.
* 
* @param {Integer} amount to move horizontally (may be negative)
* @param {Integer} amount to move vertically (may be negative)
*/
moveBy(deltaX:Integer, deltaY:Integer): boolean;

/**
*  Moves the widget so that its top-left corner is at the specified coordinates.
*  
*  This method will also accept a single parameter as an object array with left and top given
*  as properties.
* 
* 
* @param {number | object=} x-coordinate to move to in LOCAL coordinates                       or Object with left and top properties.
* @param {number=} y-coordinate to move to in LOCAL coordinates
*/
moveTo(left?:number | object, top?:number): boolean;

/**
*  Get an offset to be used when calculating snap positioning. Returns 0 by default.
* 
* 
* @param {Canvas=} the child that is being snapped
*/
getVSnapOrigin(snapChild?:Canvas): Integer;

/**
*  Get an offset to be used when calculating snap positioning. Returns 0 by default.
* 
* 
* @param {Canvas=} the child that is being snapped
*/
getHSnapOrigin(snapChild?:Canvas): Integer;

/**
*  Show this widget next to another widget on the page, positioned such that it will 
*  not extend beyond the browser viewport.
*  
*  Note that this method simply sets the coordinates of the widget and displays it (using
*  a Canvas.animateShow by default). It will
*  not change the Canvas.parentElement of either component.
*  
*  An example use case might be showing a menu next to a menu-button.
* 
* 
* @param {Canvas} Canvas to show next to
* @param {string=} Which side of the other canvas should we put. Options are                 "top", "bottom", "left", "right". (Defaults to "right")
* @param {boolean=} This argument controls whether this canvas can be positioned on top of the other  widget if there isn't room to put it next to the other widget extending out of the  browser viewport  If 'canOcclude' is true, simply shift this widget over the other widget, so that  it ends up onscreen. If 'canOcclude' is false, avoid extending offscreen  by positioning this widget on the other side of the other widget.
* @param {boolean=} If false do not use an animation to show the component.
*/
showNextTo(otherWidget:Canvas, side?:string, canOcclude?:boolean, skipAnimation?:boolean): void;

/**
*  Move this canvas to the specified point, or as close to the specified point as possible
*  without this widget extending beyond the edge of the browser viewport on any side.
* 
* @param {number=} Left coordinate (defaults to mouse position)
* @param {number=} Top coordinate (defaults to mouse position)
*/
placeNear(left?:number, top?:number): void;

/**
*      Resizes the widget, adding deltaX to its width and deltaY to its height (moves the right
*      and/or bottom sides of the widget).
* 
* @param {number=} amount to resize horizontally (may be negative)
* @param {number=} amount to resize vertically (may be negative)
*/
resizeBy(deltaX?:number, deltaY?:number): boolean;

/**
*  Observable method called 
*  whenever a Canvas changes size. Note that if this canvas is
*  Canvas.overflow, and is waiting for a queued redraw (see
*  Canvas.isDirty), the value for Canvas.getVisibleWidth and
*  Canvas.getVisibleHeight will be unreliable until redraw() fires.
* 
*/
resized(): void;

/**
*  Setter method for the Canvas.percentSource attribute.
* 
* @param {Canvas=} eterNew percent source (if omitted existing                   percentSource will just be cleared).
*/
setPercentSource(sourceWidget?:Canvas): void;

/**
* 
*  layoutChildren() is where a Canvas should implement a sizing policy for it's
*  Canvas children. Since layoutChildren calls parentResized() on its children,
*  Canvas.parentResized is a good place for a child to implement a layout policy that
*  can be used within any parent.
*  
*  Recommended practice for a Canvas that manages Canvas children is to create those children
*  without any initial coordinate or size settings and do all sizing when layoutChildren() is
*  called.
*  
*  layoutChildren() is always called at least once before children are drawn, and is called
*  automatically whenever the viewport size changes (which includes both resizing and
*  introduction/removal of scrolling). layoutChildren() can also be manually invoked in any
*  other component-specific situation which changes the layout.
*  
*  NOTE: layoutChildren() may be called before draw() if a widget is resized before draw(), so
*  be sure to avoid errors such as assuming that any children you automatically create have
*  already been created.
*  
*  NOTE: auto-sizing: layoutChildren() is also called once during the initial draw(), before
*  children are drawn, with a "reason" of "initial draw". During this invocation of
*  layoutChildren() it is legal to directly draw children (call child.draw()), which is
*  otherwise never allowed. This allows a Canvas to implement an auto-sizing layout policy by
*  drawing some children before deciding on the sizes of remaining children, which is far more
*  efficient than drawing all children and resizing some of them after they are drawn.
* 
* @param {string} reason why layoutChildren() is being called, passed when framework            code invokes layoutChildren()
*/
layoutChildren(reason:string): void;

/**
*      Resizes the widget to the specified width and height (moves the right and/ or bottom
*      sides of the widget). The width and height parameters can be expressed as a percentage
*      of viewport size or as the number of pixels.
* 
* @param {number=} new width for canvas.
* @param {number=} new height for canvas
*/
resizeTo(width?:number, height?:number): boolean;

/**
*  Fires when the interior size of the parent changes, including parent resize and
*  scrollbar introduction or removal.
*  
*  This method allows a child to implement a layout policy that can be used within any
*  parent, such as a Resizer component that always snaps to the parent's
*  bottom-right corner. The default implementation of this method applies a child's
*  percent sizes, if any, or implements layout based on the Canvas.snapTo property
* 
*/
parentResized(): void;

/**
*  If Canvas.dragAppearance is set to "tracker", this method will be called
*  (if defined), when the user starts to drag this widget. It is an opportunity to update the
*  drag tracker to display something relative to this canvas. Typical implementation will
*  be to call EventHandler.setDragTracker, passing in the desired custom tracker HTML
*  as a string
* 
*/
setDragTracker(): boolean;

/**
*  If this widget is showing scrollbars, and a user drags close to the edge of the viewport,
*  should we scroll the viewport in the appropriate direction?
*  Returns this.canDragScroll if there are scrollbars, else false.
* 
*/
shouldDragScroll(): void;

/**
*  Update the Canvas.overflow of a Canvas after it has been created.
* 
* @param {Overflow} New overflow value.
*/
setOverflow(newOverflow:Overflow): void;

/**
*  This method tells a component to adjust for changes in the size of its content that happen 
*  outside of SmartClient APIs. This may include:
*  
*  
* - size changes due to "replaced" HTML elements (elements that may change size after their 
*    content loads, such as &lt;img&gt; tags with no sizes). To avoid the need to call 
*    adjustForContent(), specify sizes on replaced elements wherever possible.
*  
*  
* - modification of HTML contained in a Canvas by direct manipulation of the DOM - see the 
*    domIntegration. Note that only contents 
*    supplied to a widget via Canvas.contents or via an override of 
*    Canvas.getInnerHTML should be manipulated directly. Contents automatically 
*    generated by SmartClient components (such as the basic structure of a Button) should 
*    never be manipulated: these structures are considered internal, differ by platform, 
*    and will change without notice.
*  
*  
* 
* @param {boolean} By default the adjustment will occur on a small delay for performance reasons.  Pass in this parameter to force immediate adjustment.
*/
adjustForContent(immediate:boolean): void;

/**
*  Returns the scrollTop required to scroll vertically to the end of this widget's content.
* 
*/
getScrollBottom(): Integer;

/**
*  Returns the scrollLeft required to scroll horizontally to the end of this widget's content.
* 
*/
getScrollRight(): Integer;

/**
*  Vertically scrolls the content of the widget to 0
* 
* 
*/
scrollToTop(): void;

/**
*  Vertically scrolls the content of the widget to the end of its content
* 
* 
*/
scrollToBottom(): void;

/**
*  Horizontally scrolls the content of the widget to 0
* 
* 
*/
scrollToLeft(): void;

/**
*  Horizontally scrolls the content of the widget to the end of its content
* 
* 
*/
scrollToRight(): void;

/**
*  Scroll this widget by some pixel increment in either (or both) direction(s).
*  
*  Note: If you attempt to call this API before the widget is drawn, the call will be ignored.
* 
* 
* @param {number} Number of pixels to scroll horizontally
* @param {number} Number of pixels to scroll vertically
*/
scrollBy(dX:number, dY:number): void;

/**
*  Scroll this widget by some percentage of scroll size in either (or both) direction(s).
* 
* 
* @param {number | string} Percentage to scroll horizontally. Will accept either                   a numeric percent value, or a string like "10%".
* @param {number | string} Percentage to scroll horizontally. Will accept either                   a numeric percent value, or a string like "10%".
*/
scrollByPercent(dX:number | string, dY:number | string): void;

/**
*  Scrolls the content of the widget so that the origin (top-left corner) of the content
*  is left pixels to the left and top pixels above the widget's top-left corner (but still
*  clipped by the widget's dimensions).
*  
*  This is guaranteed to be called whenever this Canvas is scrolled, whether scrolling is
*  initiated programmatically, by custom scrollbars, or a by a native scrollbar.
*  
*  Note: If you attempt to call this API before the widget is drawn, the call will be ignored.
* 
* 
* @param {Integer=} the left coordinate
* @param {Integer=} the top coordinate
*/
scrollTo(left?:Integer, top?:Integer): void;

/**
*  Scroll this widget to some position specified as a percentage of scroll size in either
*  (or both) direction(s).
* 
* 
* @param {number | string} Left Percentage position to scroll to                    Will accept either a numeric percent value, or a                     string like "10%".
* @param {number | string} Top Percentage position to scroll to                    Will accept either a numeric percent value, or a                    string like "10%".
*/
scrollToPercent(left:number | string, top:number | string): void;

/**
*  This method is the programmatic equivalent of the user pressing the "Page Up" key while 
*  this widget has the focus. It scrolls the widget's content upwards by the viewport 
*  height, if the content can be scrolled that far upwards
* 
*/
pageScrollUp(): void;

/**
*  This method is the programmatic equivalent of the user pressing the "Page Down" key while 
*  this widget has the focus. It scrolls the widget's content downwards by the viewport 
*  height, if the content can be scrolled that far downwards
* 
*/
pageScrollDown(): void;

/**
*  Sets this widget's visibility to "inherit", so that it becomes visible if all of its parents
*  are visible or it has no parents.
*  
*  If the widget has not yet been drawn (and doesn't have a parent or master), this method
*  calls the draw method as well.
* 
* 
*/
show(): void;

/**
*  Recursively show the canvas and all its parents so the canvas will be visible.
*  
*  If the widget has not yet been drawn, this method calls the draw method as well.
* 
* 
*/
showRecursively(): void;

/**
*  Returns this Canvas's "panel container". A panel container is a widget that manages a 
*  collection of panels, like a TabSet or SectionStack. If this
*  Canvas is not a child of such a panel container, this method returns null.
* 
* 
*/
getPanelContainer(): Canvas;

/**
*  Sets this Canvas's "panel container". A panel container is a widget that manages a 
*  collection of panels, like a TabSet or SectionStack. SmartClient
*  uses this method internally when child panels are added to panel container widgets; if you
*  need to create a panel container widget that does not extend one of the built-in ones (these
*  are TabSet, SectionStack and Window), your code 
*  should do the same thing.
* 
* 
* @param {Canvas} The container widget for this canvas
*/
setPanelContainer(container:Canvas): void;

/**
*  For "panel container" widgets like TabSet or SectionStack, this 
*  method reveals the child Canvas passed in by whatever means is applicable for the particular 
*  type of container. For example, when called on a TabSet, it selects the tab containing the
*  passed-in child.
*  
*  For other types of Canvas, this method simply shows the passed-in child if it is not 
*  currently visible.
*  
*  If the passed-in widget is not a child of this Canvas, this method has no effect
* 
* 
* @param {GlobalId | Canvas} the child Canvas to reveal, or its global ID
*/
revealChild(child:GlobalId | Canvas): void;

/**
*      Sets the widget's CSS visibility attribute to "hidden".
* 
*/
hide(): void;

/**
*  Returns true if the widget is visible, taking all parents into account, so that a widget
*  which is not hidden might still report itself as not visible if it is within a hidden
*  parent.
*  
*  NOTE: Undrawn widgets will report themselves as visible if they would be visible if drawn.
* 
* 
*/
isVisible(): boolean;

/**
*  set the enabled state of this object.
* 
* 
* @param {boolean} pass false to disable or anything else to enable
* @deprecated * As of Smartclient version 5.5, deprecated in favor of Canvas.setDisabled

*/
setEnabled(newState:boolean): void;

/**
*  set the disabled state of this object
* 
* @param {boolean} new disabled state of this object - pass true to disable the widget
*/
setDisabled(disabled:boolean): void;

/**
*  Enables this widget and any children / peers of this widget.
* 
*/
enable(): void;

/**
*  Disables this widget and any children and peers of this widget.
* 
*/
disable(): void;

/**
*  Is this canvas disabled? Note that the disabled state is inherited - this method will return
*  true if this widget, or any of its ancestors are marked disabled.
* 
*/
isDisabled(): boolean;

/**
*  Returns true if the widget and all widgets above it in the containment hierarchy are enabled.
*  Returns false otherwise.
* 
* @deprecated * As of SmartClient version 5.5 deprecated in favor of Canvas.isDisabled.

*/
isEnabled(): boolean;

/**
*  Change whether a widget can accept keyboard focus.
* 
* @param {boolean} whether the widget should now accept focus
*/
setCanFocus(canFocus:boolean): void;

/**
*  Returns true if this Canvas has the keyboard focus. Note that in Internet Explorer
*  focus notifications can be asynchronous (see EventHandler.synchronousFocusNotifications).
*  In this case, canvas.isFocused() method can correctly 
*  return false when, intuitively, you would expect it to return true:
*  
*    someCanvas.focus();
*    if (someCanvas.isFocused()) {
*      // In most browsers we would get here, but not in Internet Explorer with
*      // EventHandler.synchronousFocusNotifications disabled.
*    }
* 
*/
isFocused(): boolean;

/**
*  If this canvas can accept focus, give it keyboard focus. After this method, the canvas
*  will appear focused and will receive keyboard events.
* 
*/
focus(): void;

/**
*  If this canvas has keyboard focus, blur it. After this method, the canvas
*  will no longer appear focused and will stop receiving keyboard events.
* 
*/
blur(): void;

/**
*  Returns true if the keyboard focus is in this Canvas or any child of this Canvas.
* 
*/
containsFocus(): boolean;

/**
*  Set the accessKey for this canvas.
*  
*  The accessKey can be set to any alphanumeric character (symbols not supported)
*  Having set an accessKey, the canvas will be given focus when the user hits
*  Alt+[accessKey], or in Mozilla Firefox 2.0 and above, Shift+Alt+[accessKey].
* 
* 
* @param {string} Character to use as an accessKey for this widget. Case Insensitive.
*/
setAccessKey(accessKey:string): void;

/**
*  If an explicit Canvas.tabIndex was assigned to this widget, clear it.
*  This will enable automatic tab index managment behaviors via the TabIndexManager class
*  as described in the tabOrderOverview
* 
*/
clearExplicitTabIndex(): void;

/**
*  This method is executed on draw. Default implementation for top-level widgets ensures 
*  this widget is at the end of the tab-sequence.
*  
*  Has no effect if this canvas is embedded in a Canvas.getParentCanvas.
* 
*/
updateTabPositionForDraw(): void;

/**
*  Update the childrens' tab positions, slotting them under this widget in the
*  TabIndexManager, in the order defined by Canvas.getChildTabPosition.
*  This method will skip any children where Canvas.updateTabPositionOnReparent is
*  false.
*  
*  This method is called automatically on canvas draw(). It may be overridden by subclasses
*  for custom tab-order behavior.
* 
*/
updateChildTabPositions(): void;

/**
*  Ensure that a specific child is slotted correctly into the page's tab order. Default
*  implementation will, if Canvas.updateTabPositionOnReparent is true, ensure 
*  the child canvas shows up in the TabIndexManager tree
*  under the entry for this widget (the parent), in the position returned 
*  by Canvas.getChildTabPosition.
*  
*  This method is called automatically in cases where a single child's tab position may
*  need to be updated - such as if a child is added to a drawn parent.
*  
*  See also Canvas.updateChildTabPositions
* 
* 
* @param {Canvas} child to have tab position updated.
*/
updateChildTabPosition(child:Canvas): void;

/**
*  For a given child widget where Canvas.updateTabPositionOnReparent is true, 
*  return the expected tab position within this parent.
*  Default implementation will any explicit relative tab position specified by 
*  Canvas.setRelativeTabPosition, and otherwise put children in the 
*  same order as defined in the children array
*  
*  Overridden in the Layout class to return the position in the members array.
* 
* @param {Canvas} Child to get local tab position for
*/
getChildTabPosition(child:Canvas): Integer;

/**
*  Setter for the Canvas.updateTabPositionOnReparent attribute.
* 
* @param {boolean} new value for canvas.updateTabPositionOnReparent
*/
setUpdateTabPositionOnReparent(updateOnReparent:boolean): void;

/**
*  Assign a relative tab position for this canvas. The meaning of a "relative" tab position
*  varies depending on where the canvas is in the page.
*  
*  For canvases with no specified Canvas.getParentCanvas, (or where
*  Canvas.updateTabPositionOnReparent is false), this method will and move the
*  canvas to the appropriate tab-position among other top level canvases. It will also disable
*  Canvas.updateTabPositionOnDraw so if this method is called before draw, drawing
*  this canvas will not cause its tab position to change.
*  
*  For canvases embedded in a Canvas.getParentCanvas, this method will move the
*  canvas to the appropriate tab position among the other children of the parent.
*  Implementation note: This is achieved by setting an internal property to
*  indicate the new tab position which will be respected by the default 
*  Canvas.getChildTabPosition implementation, and calling 
*  Canvas.updateChildTabPositions to implement a reflow. Therefore if
*  Canvas.getChildTabPosition has been overridden, this method may have no effect.
*  
*  As with other APIs related to TabIndexManager, tab indices
*  are treated as a hierarchy by default. By setting the relative tab position of a
*  canvas which is not itself focusable but has focusable descendents, these descendents'
*  tab position will be updated.
*  
*  Note that after this method has been called, the tab position can be modified by
*  subsequent code to shift another sibling in front of this one, or reparent this canvas.
* 
* 
* @param {Integer} new relative tab position
*/
setRelativeTabPosition(position:Integer): void;

/**
*  Assign an explicit Canvas.tabIndex to this widget at runtime.
*  
*  Developers may also call Canvas.clearExplicitTabIndex to clear any explicitly
*  assigned tab index, and have the widget participate in automatic tab position allocation.
*  For more information see Canvas.tabIndex
* 
* 
* @param {number} New tabIndex for this widget. Must be less than             Canvas.TAB_INDEX_FLOOR to avoid interfering with auto-assigned              tab indices on the page.
*/
setTabIndex(tabIndex:number): void;

/**
*  Shifts focus to the next focusable element after this one. This programatically 
*  simulates the user experience of a Tab keypress, and is used automatically when 
*  a blocking Canvas.showClickMask is up to ensure focus does not
*  move to any masked elements in the UI.
*  
*  This method makes use of the TabIndexManager.shiftFocus method to request
*  focus be changed to the adjacent registered entry. By default standard focusable 
*  SmartClient UI elements, including Canvases, FormItems, FormItemIcons, etc are
*  registered with the TabIndexManager in the appropriate order, and will accept focus
*  if Canvas.canFocus, and not FormItem.disabled or
*  Canvas.showClickMask. 
*  
*  The TabIndexManager maintains a hierarchy of focusable targets - so if a parent
*  canvas contains focusable children, they will typically be nested under the parent
*  canvas in this hierarchy. If you want to shift focus to the next target outside this
*  hierarchy (IE: skip any children and put focus into the next widget on the page
*  outside this one), you can use use Canvas.focusAfterGroup.
*  
*  NOTE: Focusable elements created directly in the raw HTML bootstrap or 
*  by application code will not be passed focus by this method unless they have also been
*  explicitly registered with the TabIndexManager. See the tabOrderOverview
*  for more information.
* 
* 
*/
focusInNextTabElement(): void;

/**
*  Shifts focus to the next focusable element after this one, skipping any elements
*  nested inside the tabbing group for this canvas, such as focusable children.
*  
*  This method makes use of the TabIndexManager.shiftFocusAfterGroup method to request
*  focus be changed to the next registered entry. By default standard focusable 
*  SmartClient UI elements, including Canvases, FormItems, FormItemIcons, etc are
*  registered with the TabIndexManager in the appropriate order, and will accept focus
*  if Canvas.canFocus, and not FormItem.disabled or
*  Canvas.showClickMask. 
*  
*  This method differs from Canvas.focusInNextTabElement in that it will
*  skip any descendants of this widget in the TabIndexManager's hierarchy of potential
*  focus target. By default this means focus will be moved to the next target on the
*  page which is not a descendant of this widget.
*  
*  FormItems support a similar method: FormItem.focusAfterItem.
*  
*  NOTE: Focusable elements created directly in the raw HTML bootstrap or 
*  by application code will not be passed focus by this method unless they have also been
*  explicitly registered with the TabIndexManager. See the tabOrderOverview
*  for more information.
* 
* 
*/
focusAfterGroup(): void;

/**
*  Shifts focus to the previous focusable element before this one. This programatically 
*  simulates the user experience of a Shift+Tab keypress.
*  
*  This method makes use of the TabIndexManager.shiftFocus method to request
*  focus be changed to the adjacent registered entry. By default standard focusable 
*  SmartClient UI elements, including Canvases, FormItems, FormItemIcons, etc are
*  registered with the TabIndexManager in the appropriate order, and will accept focus
*  if Canvas.canFocus, and not FormItem.disabled or
*  Canvas.showClickMask. 
*  
*  NOTE: Focusable elements created directly in the raw HTML bootstrap or 
*  by application code will not be passed focus by this method unless they have also been
*  explicitly registered with the TabIndexManager. See the tabOrderOverview
*  for more information.
* 
* 
*/
focusInPreviousTabElement(): void;

/**
*  Shifts focus to the start or end of this canvas and its descendants.
*  
*  This method makes use of the TabIndexManager.shiftFocusWithinGroup API to
*  request focus be changed within the set of focusable targets registered under
*  this canvas.
*  
*  If the start parameter is true, if the canvas itself is focusable it will
*  be given focus, otherwise the first focusable descendant will be given focus.
*  If the start parameter is false, the last focusable descendant will be
*  given focus (or if the canvas itself is focusable but there are no focusable descendants, 
*  it will receive focus).
*  
*  A use case for this might be to programmatically shift focus to the first or last
*  button in a toolbar or similar.
* 
* @param {boolean} Should we focus at the start or the end of this widget and its  descendants.
*/
focusAtEnd(start:boolean): void;

/**
*  Get the z-Index of this canvas.
* 
* 
* @param {boolean} If passed true, for undrawn widgets, resolve "auto" to the next available zIndex.
*/
getZIndex(resolveToNumber:boolean): number;

/**
*  Puts this widget at the top of the stacking order, so it appears in front of all other
*  widgets in the same parent.
* 
*/
bringToFront(): void;

/**
*  Puts this widget at the bottom of the stacking order, so it appears behind all other
*  widgets in the same parent.
* 
*/
sendToBack(): void;

/**
*  Puts this widget just above the specified widget in the stacking order, so it appears in
*  front of the specified widget if both widgets have the same parent.
* 
* @param {Canvas} canvas to move above
*/
moveAbove(canvas:Canvas): void;

/**
*  Puts this widget just below the specified widget in the stacking order, so it appears
*  behind the specified widget if both widgets have the same parent.
* 
* @param {Canvas} canvas to move below
*/
moveBelow(canvas:Canvas): void;

/**
*    Returns the contents of a Canvas. The contents are an HTML string.
* 
*/
getContents(): HTMLString;

/**
*  Changes the contents of a widget to newContents, an HTML string.
*  
*  When Canvas.dynamicContents is set, setContents() can
*  also be called with no arguments to cause contents to be re-evaluated.
* 
* 
* @param {HTMLString=} an HTML string to be set as the contents of this widget
*/
setContents(newContents?:HTMLString): void;

/**
*  Sets the background color of this component to newColor.
* 
* @param {CSSColor} new background color, or null to remove the current background color.
*/
setBackgroundColor(newColor:CSSColor): void;

/**
*      Sets the background to an image file given by newImage. This URL should be given as a
*      string relative to the image directory for the page (./images by default).
* 
* @param {SCImgURL} new URL (local to Page image directory) for background image
*/
setBackgroundImage(newImage:SCImgURL): void;

/**
*  Set the CSS border of this component, as a CSS string including border-width,
*  border-style, and/or color (eg "2px solid blue").
*  
*  This property applies the same border to all four sides of this component. Different
*  per-side borders can be set in a CSS style and applied via Canvas.styleName.
* 
* 
* @param {string} new border to set to (eg: "2px solid black")
*/
setBorder(newBorder:string): void;

/**
*  Sets the opacity for the widget to the newOpacity value. This newOpacity
*  value must be within the range of 0 (transparent) to 100 (opaque).
*  Null means don't specify opacity directly.
*  Note that heavy use of opacity may have a performance impact on some older
*  browsers.
*  
*  In older versions of Internet Explorer (Pre IE9 / HTML5), opacity is achieved
*  through proprietary filters. If
*  Canvas.neverUseFilters within this application
*  developers must set Canvas.useOpacityFilter to true for specific components
*  on which opacity support is required.
*  
*  Also note that opacity is incompatible
*  with Canvas.useBackMask.
* 
* @param {number} new opacity level
*/
setOpacity(newOpacity:number): void;

/**
*      Sets the cursor for this widget to cursor. See the cursor property
*      for possible values.
* 
* @param {Cursor} new cursor
*/
setCursor(newCursor:Cursor): void;

/**
*  When Canvas.showHoverComponents is true, this method is called to get the component to show
*  as a hover for this Canvas. There is no default implementation of this method, so you need
*  to override it with an implementation that returns a Canvas that suits your needs.
*  
*  By default, components returned by getHoverComponent() will not be
*  automatically destroyed when the hover is hidden. To enforce this, set
*  Canvas.hoverAutoDestroy to true on the returned component.
* 
* 
*/
getHoverComponent(): Canvas | Partial<Canvas>;

/**
*  Handler fired on a delay when the user hovers the mouse over this hover-target.
*  Default implementation will fire this.hover() (if defined), and handle
*  showing the hover canvas if this.showHover is true.
* 
*/
handleHover(): void;

/**
*  If this canvas is currently showing a hover (see Canvas.handleHover), this method
*  can be called to update the HTML contents of the hover. Has no effect if this widget is 
*  showing Canvas.showHoverComponents, or if 
*  the hover canvas is not showing for this widget.
* 
* @param {string=} Option to specify new HTML for the hover. If not passed, the  result of Canvas.getHoverHTML will be used instead. Note that   if the hover HTML is empty the hover will be hidden.
*/
updateHover(hoverHTML?:string): void;

/**
*  If Canvas.showHover is true for this canvas, this notification method will be
*  fired whenever the hover shown in response to Canvas.handleHover is
*  hidden. This method may be observed or overridden.
* 
*/
hoverHidden(): void;

/**
*  If canHover is true for this widget, the hover string method will
*  be fired when the user hovers over this canvas. If this method returns false, it will
*  suppress the default behavior of showing a hover canvas if this.showHover
*  is true.
* 
*/
hover(): boolean;

/**
*  If this.showHover is true, when the user holds the mouse over this Canvas for
*  long enough to trigger a hover event, a hover canvas is shown by default. This method returns
*  the contents of that hover canvas. Default implementation returns this.prompt -
*  override for custom hover HTML. Note that returning null or an empty string will
*  suppress the hover canvas altogether.
* 
*/
getHoverHTML(): HTMLString;

/**
*  Sets the CSS class for this widget
* 
* @param {CSSStyleName} new CSS class name (must be defined previously)
* @deprecated * As of SmartClient version 5.5, use Canvas.setStyleName instead.

*/
setClassName(newClass:CSSStyleName): void;

/**
*  Sets the CSS class for this widget
* 
* @param {CSSStyleName} new CSS style name
*/
setStyleName(newStyle:CSSStyleName): void;

/**
*  Executed when the right mouse button is clicked. The default implementation of
*  this method auto-creates a Menu from the Canvas.contextMenu property on
*  this component and then calls Menu.showContextMenu on it to show it.
*  
*  If you want to show a standard context menu, you can simply define your Menu and set it as the
*  contextMenu property on your component - you do not need to override this method.
*  
*  If you want to do some other processing before showing a menu or do something else entirely, then
*  you should override this method. Note that the return value from this method controls whether or
*  not the native browser context menu is shown.
* 
* 
*/
showContextMenu(): boolean;

/**
* 
*  The default implementation of this method hides the contextMenu currently being shown for this
*  component (which occurs when the mouse button that toggles the context menu is released).
*  Override if you want some other behavior.
* 
* 
*/
hideContextMenu(): void;

/**
* 
*  Executed when mouse enters this widget. No default implementation.
* 
* 
*/
mouseOver(): boolean;

/**
*  Executed when the left mouse button is pressed on this widget. No default implementation.
* 
* 
*/
mouseDown(): boolean;

/**
*  Executed when the right mouse button is pressed on this widget. No default implementation.
* 
* 
*/
rightMouseDown(): boolean;

/**
* 
*  Repeating notification method for the user holding the left mouse button down over this
*  canvas.
*  
*  The mouseStillDown event is fired immediately when the mouse goes down.
*  If the user holds the mouse down, after a pause of
*  Canvas.mouseStillDownInitialDelay, it will begin to fire repeatedly every 
*  Canvas.mouseStillDownDelay milliseconds.
*  
*  This provides developers with a simple way to handle the common "repeated action" 
*  use case where a user can click a UI element to perform an action once, or click 
*  and hold to perform the action repeatedly.
*  Examples of this include standard scrollbar button behavior and buttons 
*  to increase or decrease the value in a spinner type input element.
*  
*  This event is not native to JavaScript, but is provided by the ISC system.
* 
* 
*/
mouseStillDown(): boolean;

/**
* 
*  Executed when the mouse moves within this widget. No default implementation.
* 
* 
*/
mouseMove(): boolean;

/**
* 
*  Executed when the mouse leaves this widget. No default implementation.
*  
*  Note that if the mouse goes over a child of this canvas, the mouseOut event will fire as it would
*  if the user rolled entirely off the canvas. Developers may determine whether the mouse is still over
*  a descendant of this component via Canvas.containsEventTarget.
* 
* 
*/
mouseOut(): boolean;

/**
* 
*  Executed when the left mouse is released on this widget. No default implementation.
* 
* 
*/
mouseUp(): boolean;

/**
* 
*  Executed when the left mouse is clicked (pressed and then released) on this widget. No default
*  implementation.
* 
* 
*/
click(): boolean;

/**
* 
*  Executed when the left mouse button is clicked twice in rapid succession (within
*  Canvas.doubleClickDelay by default) in this object.
* 
* 
*/
doubleClick(): boolean;

/**
* 
*  Executed when the mouse wheel is actuated.
* 
* 
*/
mouseWheel(): boolean;

/**
* 
*  Executed when dragging first starts. No default implementation. Create this handler to set
*  things up for the drag reposition.
* 
* 
*/
dragRepositionStart(): boolean;

/**
* 
*  Executed every time the mouse moves while drag-repositioning. If this method
*  does not return false, the Canvas.dragTarget (or outline if
*  Canvas.dragAppearance is set to "outline") will automatically be moved as appropriate
*  whenever the mouse moves.
* 
* 
*/
dragRepositionMove(): boolean;

/**
* 
*  Executed when the mouse button is released at the end of the drag. Your
*  widget can use this opportunity to fire custom code based upon where the
*  mouse button was released, etc.
*  
*  Returning true from this handler will cause the Canvas.dragTarget (or outline if
*  Canvas.dragAppearance is set to "outline") to be left in its current
*  location. Returning false from this handler will cause it to snap back to its
*  original location.
* 
* 
*/
dragRepositionStop(): boolean;

/**
* 
*  Executed when resize dragging first starts. No default implementation.
*  Create this handler to set things up for the drag resize.
* 
* 
*/
dragResizeStart(): boolean;

/**
* 
*  Executed every time the mouse moves while drag-resizing. If this method
*  does not return false, the Canvas.dragTarget (or outline if
*  Canvas.dragAppearance is set to "outline") will automatically be moved as appropriate
*  whenever the mouse moves.
* 
* 
*/
dragResizeMove(): boolean;

/**
* 
*  Executed when the mouse button is released at the end of the drag resize. Your
*  widget can use this opportunity to fire custom code based upon where the
*  mouse button was released, etc.
*  
*  Returning true from this handler will cause the Canvas.dragTarget (or outline if
*  Canvas.dragAppearance is set to "outline") to be left at its current size. Returning
*  false from this handler will cause it to snap back to its original location size
* 
* 
*/
dragResizeStop(): boolean;

/**
*  Executed when dragging first starts. Your widget can use this opportunity to set
*  things up for the drag, such as setting the drag tracker. Returning false from this
*  event handler will cancel the drag action entirely.
*  
*  A drag action is considered to be begin when the mouse has moved
*  Canvas.dragStartDistance with the left mouse down.
* 
* 
*/
dragStart(): boolean;

/**
*  Executed every time the mouse moves while dragging this canvas.
* 
* 
*/
dragMove(): boolean;

/**
*  Executed when the mouse button is released at the end of the drag. Your widget can
*  use this opportunity to fire code based on the last location of the drag or reset any
*  visual state that was sent.
* 
* 
*/
dragStop(): boolean;

/**
* 
*  Executed when the compatible dragged object is first moved over this drop target. Your
*  implementation can use this to show a custom visual indication that the object can be
*  dropped here.
* 
* 
*/
dropOver(): boolean;

/**
* 
*  Executed whenever the compatible dragged object is moved over this drop target. You
*  can use this to show a custom visual indication of where the drop would occur within the
*  widget, or to show the Canvas.noDropCursor to indicate that
*  this is not a valid drop target, typically if Canvas.willAcceptDrop returns false.
*  
*  For details on showing a 'no drop' cursor when the user drags over all invalid drop
*  targets, see EventHandler.showNoDropIndicator.
* 
* 
*/
dropMove(): boolean;

/**
* 
*  Executed when the dragged object is no longer over this drop target, including when the drag
*  interaction is ending with a drop on this drop target. If you have set a visual indication 
*  in dropOver or dropMove, you should reset it to its normal state in dropOut.
* 
* 
*/
dropOut(): boolean;

/**
* 
*  Executed when the mouse button is released over a compatible drop target at the end of
*  a drag sequence. Your widget should implement whatever it wants to do when receiving a
*  drop here. For example, in a file moving interface, a drop might mean that you should
*  move or copy the dragged file into the folder it was dropped on, or dropping something in
*  a trash can might mean to clear it from the screen.
* 
* 
*/
drop(): boolean;

/**
* 
*  Executed when a key is pressed on a focusable widget (Canvas.canFocus: true).
*  
*  Use EventHandler.getKey to find out the KeyName of the key that
*  was pressed, and use EventHandler.shiftKeyDown and related functions to determine
*  whether modifier keys were down.
* 
* 
*/
keyDown(): boolean;

/**
* 
*  Executed when a key is released on a focusable widget (Canvas.canFocus: true).
*  
*  Use EventHandler.getKey to find out the KeyName of the key that
*  was pressed, and use EventHandler.shiftKeyDown and related functions to determine
*  whether modifier keys were down.
* 
* 
*/
keyUp(): boolean;

/**
* 
*  Executed when a key is pressed and released on a focusable widget (Canvas.canFocus:
*  true).
*  
*  Use EventHandler.getKey to find out the KeyName of the key that
*  was pressed, and use EventHandler.shiftKeyDown and related functions to determine
*  whether modifier keys were down.
* 
* 
*/
keyPress(): boolean;

/**
* 
*  Returns true if the widget object being dragged can be dropped on this widget, and
*  false otherwise. The default implementation of this method simply compares the
*  Canvas.dragType of the dragTarget (the component being dragged from)
*  with the list of Canvas.dropTypes on this Canvas. If the Canvas.dropTypes
*  list contains the Canvas.dragType value, then this method returns true. Otherwise it
*  returns false.
*  No matter what you return, Canvas.dropOver and Canvas.dropMove
*  will still be called, and their return values will determine whether those events are bubbled
*  to parent elements.
*  However, what you return from willAcceptDrop() does determine whether
*  Canvas.drop will be called.
*  
*  
* - If you return true, then drop() will be called, and its return value
*  will determine whether the event is bubbled to parent elements
*  
* - If you return false, then drop() will not be called, and the event
*  will not be bubbled.
*  
* - If you return null, then drop() will not be called, but the event
*  will be bubbled to parent elements (giving them a chance to handle the drop).
*  
*  So, you should return false to definitively deny a drop, and return null if it could
*  make sense to allow a parent element, such as a Layout, to handle the drop. 
* 
* 
*/
willAcceptDrop(): boolean;

/**
* 
*  Override this method to provide a custom snap-to grid. Note that you do not need to do
*  this if your grid is regular (ie, grid points are every x pixels); regular grids should be
*  defined using Canvas.snapHGap and Canvas.snapVGap.
*  You should only override this method if you want to provide support for a grid of
*  irregularly-placed points
* 
* 
* @param {Integer} x-coordinate of the drag event relative to the inside of this widget
* @param {string=} "before" or "after" denoting whether the returned coordinate should  match the left or right edge of the current square. If unset Canvas.snapHDirection will  be used by default
*/
getHSnapPosition(coordinate:Integer, direction?:string): Integer;

/**
*  Override this method to provide a custom snap-to grid. Note that you do not need to do
*  this if your grid is regular (ie, grid points are every x pixels) - regular grids should be
*  defined using Canvas.snapHGap and Canvas.snapVGap.
*  You should only override this method if you want to provide support for a grid of
*  irregularly-placed points
* 
* 
* @param {Integer} y-coordinate of the drag event relative to the inside of this widget
* @param {string=} "before" or "after" denoting whether the returned coordinate should  match the top or bottom edge of the current square. If unset Canvas.snapHDirection will  be used by default
*/
getVSnapPosition(coordinate:Integer, direction?:string): Integer;

/**
*  Override this method to give programmatic control over whether or not the parameter
*  dragTarget should snap to this object's grid when dropped. Note that this only applies
*  if snap-to-grid is enabled on either dragTarget or this object. See
*  Canvas.snapToGrid and Canvas.childrenSnapToGrid.
*  
*  The default implementation simply returns true.
* 
* 
* @param {Canvas} The object about to be dropped
*/
shouldSnapOnDrop(dragTarget:Canvas): boolean;

/**
*  Set the showSnapGrid property.
* 
* @param {boolean} */
setShowSnapGrid(show:boolean): void;

/**
*  Add a candidate to Canvas.snapAlignCandidates. Duplicates are automatically avoided.
* 
* 
* @param {Canvas} */
addSnapAlignCandidate(newCandidate:Canvas): void;

/**
*  Remove a candidate from Canvas.snapAlignCandidates. If the passed widget was not actually
*  a candidate, nothing happens and no warning is logged.
* 
* 
* @param {Canvas} */
removeSnapAlignCandidate(candidate:Canvas): void;

/**
*  Return the full URL for an image to be drawn in this canvas.
*  
*  If the passed URL begins with the special prefix "[SKIN]", it will have the
*  widget.skinImgDir and Page.skinImgDir prepended. Otherwise the image is assumed to be
*  application-specific, and will have the widget.appImgDir and Page.appImgDir automatically
*  prepended.
*  
*  Note that if passed an absolute path (starting with "/" or "http://" for example), no extra
*  image directory information will be prepended to the generated URL.//
* 
* 
* @param {string} URL local to skin or application image directory
* @param {string=} optional image directory to override the default for this Canvas
*/
getImgURL(URL:string, imgDir?:string): string;

/**
*  Generates the HTML for an image unique to this Canvas.
*  
*  The full URL for the image will be formed according to the rules documented for
*  Canvas.getImgURL.
*  
*  The created image will have an identifier unique to this Canvas, and subsequent calls to
*  Canvas.getImage and
*  Canvas.setImage
*  with the name passed to this function will act on the image object produced by the HTML
*  returned from this call.
* 
* 
* @param {SCImgURL} URL local to the skin or application directory.   NOTE: instead of passing several parameters, you can pass an object as the 'src'    parameter with properties for all the various function parameters with, eg:    canvas.imgHTML( {src:"foo", width:10, height:10} );
* @param {number=} width of the image
* @param {number=} height of the image
* @param {string=} name for the image
* @param {string=} additional attributes to write in the tag
* @param {string=} image-specific image directory to override the default                 for this Canvas
*/
imgHTML(src:SCImgURL, width?:number, height?:number, name?:string, extraStuff?:string, imgDir?:string): string;

/**
*  Retrieve a native image or SVG object element by name.
*  
*  The element must have been created from HTML generated by calling
*  Canvas.imgHTML on this particular Canvas.
* 
* 
* @param {string} name of the image or SVG object to get, as originally passed to               imgHTML()
*/
getImage(identifier:string): DOMElement;

/**
*  Set the URL of an image or SVG object element by name.
*  
*  The element must have been created from HTML generated by calling
*  canvas.imgHTML() on this particular Canvas.
* 
* 
* @param {string} name of the image to change, as originally passed to               imgHTML
* @param {SCImgURL} URL for the image
* @param {string=} optional image directory, overrides the default for this Canvas
*/
setImage(identifier:string, URL:SCImgURL, imgDir?:string): void;

/**
*  Generates the HTML for a standard link (anchor) element.
* 
* 
* @param {string} URL for the link to point to
* @param {HTMLString=} HTML to display in the link element (defaults to the href)
* @param {string=} Target window for the link (defaults to opening in a new, unnamed window)
* @param {string=} optional ID for the link element to be written out
* @param {Integer=} optional tabIndex for the link
* @param {string=} optional accessKey for the link
*/
linkHTML(href:string, text?:HTMLString, target?:string, ID?:string, tabIndex?:Integer, accessKey?:string): HTMLString;

/**
*  Set the Canvas.edgeOpacity and mark the canvas for redraw
* 
* @param {Integer} new edge-opacity level
*/
setEdgeOpacity(newOpacity:Integer): void;

/**
*  Method to update Canvas.showShadow.
* 
* @param {boolean} true if the shadow should be visible false if not
*/
setShowShadow(showShadow:boolean): void;

/**
*  Setter for Canvas.groupLabelBackgroundColor.
* 
* @param {CSSColor} the new grouping label background color.
*/
setGroupLabelBackgroundColor(groupLabelBackgroundColor:CSSColor): void;

/**
*  Setter for Canvas.groupTitle.
* 
* @param {HTMLString} The new title for the grouping.
*/
setGroupTitle(newTitle:HTMLString): void;

/**
*  Get the current value of the rule context collected by the Canvas.ruleScope of this component
*  (which may be this component itself or whatever component is managing the
*  ruleScope for this component).
*  
*  If the databoundOnly parameter is passed as true, only data from components that
*  actually have a DataSource is included.
*  
*  Use Canvas.ruleContextChanged to get a notification of changes to the rule context.
* 
* 
* @param {boolean=} whether to include only data from components that have a                 DataSource
*/
getRuleContext(databoundOnly?:boolean): object;

/**
*  Notification that the rule context gathered by the Canvas.ruleScope has changed. 
*  
*  This notification fires only on the component designated as the Canvas.ruleScope; components
*  that are merely contributing data to the rule context do not fire
*  ruleContextChanged.
* 
* @param {object} the new rule context
*/
ruleContextChanged(data:object): void;

/**
*  Provide data to the Canvas.ruleScope component, to be made available in the rule context at the
*  specified path.
*  
*  path must be one or more valid identifiers with either dot (.) or slash (/) used
*  as a separator. 
*  
*  data can be any value, including both atomic values like a Boolean or String, or a
*  nested data structure. Pass data as null to remove data from the
*  context at the specified path. When a DataSource is passed to define the
*  provided data type (described below) a call to Canvas.removeRuleContext is the correct way
*  to remove the data along with the schema.
*  
*  dbc is the DataBoundComponent to be identified as the owner of the rule context
*  contribution. This component is used to handle any conflicts between multiple components
*  contributing to the same base path (i.e. first segment of path). For any collision an editable
*  display (such as a form or editable grid) wins over a static display (such as a non-editable
*  grid with a selection). Hidden or cleared components have lowest priority even if editable.
*  For two editable components the first becomes the owner.
*  
*  dbc can also be a DataSource to define the data type of the provided data.
*  For example, when a singular Date value is provided to the ruleContext the type is not
*  known when using the value in a ruleCriteria. To counter that, a data type can be specified
*  using a DataSource and optionally a fieldName. The path will then be associated
*  with the DataSource and fieldName so criteria processing knows to compare the value
*  correctly. When providing data as a Record or array of Records only the DataSource is needed.
*  
*  See ruleCriteria for details on matching values within criteria
*  when schema is not provided.
*  
*  Note that DataBoundComponents automatically contribute to the ruleContext as described
*  in Canvas.ruleScope.
* 
* 
* @param {string} path where data should be made available
* @param {any} data to contribute to rule context
* @param {DataBoundComponent | DataSource=} dataBoundComponent contributing to ruleContext   or DataSource defining the provided data type
* @param {string=} field name within DataSource defining the data type for a singular   value (i.e. not a Record and not an Array of Records)
*/
provideRuleContext(path:string, data:any, dbc?:DataBoundComponent | DataSource, fieldName?:string): void;

/**
*  Remove data in the rule context at the specified path along with any user-provided
*  schema.
*  
*  Normally data is removed from the ruleContext by passing null to
*  data in Canvas.provideRuleContext, however, this call will do the same but also
*  removes the schema as explicitly provided to provideRuleContext. 
*  
*  Data automatically provided to the ruleContext, as described by Canvas.ruleScope, along with
*  the associated schema is automatically removed when the contributing DataBoundComponent is
*  destroyed. Therefore there is no need to clean up those ruleContext paths manually.
* 
* 
* @param {string} path where data and schema should be removed
*/
removeRuleContext(path:string): void;

/**
*  Notification method fired when a canvas has been drawn into the page.
* 
*/
drawn(): void;

/**
*  Notification method fire when an ancestor of this component's position changes.
* 
* @param {Canvas} the ancestor that moved
* @param {Integer} horizontal difference between current and previous position
* @param {Integer} vertical difference between current and previous position
*/
parentMoved(parent:Canvas, deltaX:Integer, deltaY:Integer): void;

/**
*  Notification method fired when this component is explicitly moved.
*  Note that a component's position on the screen may also changed due to an ancestor being
*  moved. The Canvas.parentMoved method provides a notification entry point to catch
*  that case as well.
* 
* 
* @param {Integer} horizontal difference between current and previous position
* @param {Integer} vertical difference between current and previous position
*/
moved(deltaX:Integer, deltaY:Integer): void;

/**
*  Notification function fired when this widget receives or loses keyboard focus.
* 
* @param {boolean} If true this widget now has keyboard focus
*/
focusChanged(hasFocus:boolean): void;

/**
*  Notification that this component has just scrolled. Use with
*  Class.observe.
*  
*  Fires for both CSS and Scrollbar.
* 
* 
*/
scrolled(): void;

/**
*  Notification fired when this canvas becomes visible or hidden to the user.
*  Note - this method is fired when the Canvas.isVisible state of this
*  component changes. It may be fired in response an explicit call to Canvas.show
*  or Canvas.hide, or in response to a parent component
*  being shown or hidden when this widgets Canvas.visibility is set to "inherit".
*  
*  Note that a call to Canvas.show or Canvas.hide will not always fire this
*  notification. If this widget has a hidden parent, show or hide would change this
*  components Canvas.visibility property, and may update the CSS visibility attribute
*  of the drawn handle in the DOM, but would not actually hide or reveal the component to
*  the user and as such the notification would not fire.
*  
*  Note also that this notification will only be fired for components which have been
*  Canvas.draw.
* 
* @param {boolean} whether the canvas is visible to the user
*/
visibilityChanged(isVisible:boolean): void;

/**
*  Generate and show a PrintWindow containing a PrintCanvas
*  showing a printable view of this component.
* 
* 
* @param {PrintProperties=} PrintProperties object for customizing the   print HTML output
* @param {Partial<PrintWindow>=} Properties to apply to the   generated print window.
* @param {Callback=} callback to fire when the print preview canvas has   been populated with the printable HTML. This callback takes 2 parameters:   printPreview - a pointer to the generated print canvas shown in the   body of the print window.   printWindow - a pointer to the generated print window and
*/
print(printProperties?:PrintProperties, printWindowProperties?:Partial<PrintWindow>, callback?:Callback): void;

/**
*  Setter for the Canvas.valuesManager attribute. This method may be called directly at 
*  runtime to set the ValuesManager for a component; it has the same effect as calling 
*  ValuesManager.addMember, passing in this DataBoundComponent.
* 
* @param {ValuesManager} new dataPath
*/
setValuesManager(dataPath:ValuesManager): void;

/**
*  Setter for the Canvas.dataPath attribute. This method may be called directly at runtime
*  to set the dataPath on a component, and will also be re-run automatically whenever a canvas'
*  parent changes due to a call to addChild(). This method handles automatically binding
*  the component to the appropriate valuesManager if necessary.
* 
* @param {DataPath} new dataPath
*/
setDataPath(dataPath:DataPath): void;

/**
*  Returns a fully qualified DataPath for this canvas. This is calculated by combining
*  the canvas' specified DataPath with the dataPath of any parent 
*  canvases up to whichever canvas has a specified Canvas.valuesManager specified to actually
*  manage values from this component.
* 
*/
getFullDataPath(): DataPath;

/**
*  When using the devTools framework and asking an
*  EditContext to EditContext.serializeAllEditNodes,
*  updateEditNode is called during the serialization process on each
*  EditNode.liveObject.
*  
*  You can implement updateEditNode on your liveObject and make 
*  updates to EditNode.defaults to save state "lazily" - just as serialization is
*  occurring - instead of updating editNode.defaults as the end user makes
*  changes. This can be useful if constantly calculating changes to
*  editNode.defaults would slow down interactivity.
*  
*  Note: best practice is to use EditContext.setNodeProperties and 
*  EditContext.removeNodeProperties to change properties, rather than directly
*  modifying EditNode.defaults.
* 
* 
* @param {EditContext} the EditContext
* @param {EditNode} the EditNode
*/
updateEditNode(editContext:EditContext, editNode:EditNode): void;

/**
*  Enable or disable edit mode for this component. Components in editMode must be
*  associated with an EditNode within an EditContext.
*  
*  Components with editMode enabled support certain editing interactions which
*  vary depending on the componentType and settings on the 
*  Canvas.editProxy.
*  
*  To disable edit mode just pass editingOn as false. The other parameters are
*  not needed.
*  
*  To enable edit mode on this component all three parameters are required. The
*  editNode is the edit node for this component as it exists within the
*  editContext.
*  
*  An alternative method, EditContext.enableEditing, can be used when
*  only an editContext and editNode are available.
*  
*  Placing a component into editMode causes the component's
*  Canvas.editProxy to be created.
* 
* 
* @param {boolean} true to enable editMode; false to disable
* @param {EditContext=} the EditContext
* @param {EditNode=} the EditNode
*/
setEditMode(editingOn:boolean, editContext?:EditContext, editNode?:EditNode): void;

/**
*  Set a specific ARIA state for this component.
*  
* 
* @param {string} aria state to update
* @param {string | boolean | Integer | Float} value for the aria state
*/
setAriaState(stateName:string, stateValue:string | boolean | Integer | Float): void;

/**
*  Gets the Canvas.ariaState for this component.
*  
*  This is a list of attribute names to property values. For example 
*  
*  { haspopup:true, label:"Settings submenu"}
*  
*  Would result in ARIA html attributes like
*  aria-haspopup="true" aria-label="Settings submenu".
*  
*  This method will return a state mapping object which may include dynamically
*  calculated attributes based on the components state (for example "disabled") as well
*  as any explicitly specified Canvas.ariaState.
*  
*  This method will be invoked during on Canvas.draw and Canvas.redraw 
*  and the resulting state attributes will be appolied to the component handle.
* 
* 
*/
getAriaState(): object;

/**
*  Animate a reposition of this canvas from its current position to the specified position
* 
* @param {Integer} new left position (or null for unchanged)
* @param {Integer} new top position (or null for unchanged)
* @param {AnimationCallback=} When the move completes this callback will be            fired. Single 'earlyFinish' parameter will be passed if the            animation was cut short, for example by a call to the non-animated            APIs Canvas.moveTo or Canvas.moveBy.
* @param {Integer=} Duration in ms of the animated move
* @param {AnimationAcceleration=} Optional acceleration effect to bias the ratios
*/
animateMove(left:Integer, top:Integer, callback?:(earlyFinish:boolean) => void, duration?:Integer, acceleration?:AnimationAcceleration): void;

/**
*  Animate a resize of this canvas from its current size to the specified size
*  
*  Note that Canvas.animateResizeLayoutMode allows you to control whether child layout is
*  rerun during every step of the animation, or just at the end, since the former may incur
*  significant overhead depending on the widget hierarchy.
* 
* 
* @param {Integer} new width (or null for unchanged)
* @param {Integer} new height (or null for unchanged)
* @param {AnimationCallback=} When the resize completes this callback will be            fired. Single 'earlyFinish' parameter will be passed if the            animation was cut short, for example by a call to the non-animated            APIs Canvas.resizeTo or Canvas.resizeBy.
* @param {Integer=} Duration in ms of the animated resize
* @param {AnimationAcceleration=} Optional acceleration effect to apply to the resize
*/
animateResize(width:Integer, height:Integer, callback?:(earlyFinish:boolean) => void, duration?:Integer, acceleration?:AnimationAcceleration): void;

/**
*  Animate a reposition / resize of this canvas from its current size and position.
* 
* @param {Integer} new left position (or null for unchanged)
* @param {Integer} new top position (or null for unchanged)
* @param {Integer} new width (or null for unchanged)
* @param {Integer} new height (or null for unchanged)
* @param {AnimationCallback=} When the setRect completes this callback will be            fired. Single 'earlyFinish' parameter will be passed if the            animation was cut short, for example by a call to the non-animated            API Canvas.setRect.
* @param {Integer=} Duration in ms of the animated setRect
* @param {AnimationAcceleration=} Optional acceleration effect to apply to the animation
*/
animateRect(left:Integer, top:Integer, width:Integer, height:Integer, callback?:(earlyFinish:boolean) => void, duration?:Integer, acceleration?:AnimationAcceleration): void;

/**
*  Animate a change in opacity from the widget's current opacity to the specified opacity.
* 
* @param {Integer} desired final opacity
* @param {AnimationCallback=} When the fade completes this callback will be            fired. Single 'earlyFinish' parameter will be passed if the            animation was cut short, for example by a call to the non-animated            API Canvas.setOpacity.
* @param {Integer=} Duration in ms of the animated fade
* @param {AnimationAcceleration=} Optional animation acceleration to bias the ratios
*/
animateFade(opacity:Integer, callback?:(earlyFinish:boolean) => void, duration?:Integer, acceleration?:AnimationAcceleration): void;

/**
*  Animate a scroll from the current scroll position to the specified position.
* 
* @param {Integer} desired final left scroll position
* @param {Integer} desired final top scroll position
* @param {AnimationCallback=} When the scroll completes this callback will be            fired. Single 'earlyFinish' parameter will be passed if the            animation was cut short, for example by a call to the non-animated            APIs Canvas.scrollTo or Canvas.scrollBy.
* @param {Integer=} Duration in ms of the animated scroll
* @param {AnimationAcceleration=} Optional acceleration to bias the animation ratios
*/
animateScroll(scrollLeft:Integer, scrollTop:Integer, callback?:(earlyFinish:boolean) => void, duration?:Integer, acceleration?:AnimationAcceleration): void;

/**
*  Show a canvas by growing it vertically to its fully drawn height over a period of time.
*  This method will not fire if the widget is already drawn and visible, or has overflow
*  other than "visible" or "hidden".
* 
* @param {AnimateShowEffectId | AnimateShowEffect=} Animation effect to use   when revealing the widget. If ommitted, default behavior can be configured via    Canvas.animateShowEffect
* @param {AnimationCallback=} When the show completes this callback will be            fired. Single 'earlyFinish' parameter will be passed if the            animation was cut short, for example by a call to the non-animated            API Canvas.show.
* @param {Integer=} Duration in ms of the animated show. If unset, duration will be  picked up from Canvas.animateShowTime
* @param {AnimationAcceleration=} Optional acceleration effect function to  bias the animation ratios. If unset, acceleration will be picked up from  Canvas.animateShowAcceleration
*/
animateShow(effect?:AnimateShowEffectId | AnimateShowEffect, callback?:(earlyFinish:boolean) => void, duration?:Integer, acceleration?:AnimationAcceleration): void;

/**
*  Hide a canvas by shrinking it vertically to zero height over a period of time.
*  This method will not fire if the widget is already drawn and visible, or has overflow
*  other than "visible" or "hidden".
* 
* @param {AnimateShowEffectId | AnimateShowEffect=} How should the content of the window be hidden during the hide? If ommitted, default behavior can be configured via  Canvas.animateHideEffect
* @param {AnimationCallback=} When the hide completes this callback will be            fired. Single 'earlyFinish' parameter will be passed if the            animation was cut short, for example by a call to the non-animated            API Canvas.hide.
* @param {Integer=} Duration in ms of the animated hide. If unset, duration will be  picked up from Canvas.animateHideTime
* @param {AnimationAcceleration=} Optional acceleration effect function to bias  the animation ratios. If unset, acceleration will be picked up from  Canvas.animateShowTime
*/
animateHide(effect?:AnimateShowEffectId | AnimateShowEffect, callback?:(earlyFinish:boolean) => void, duration?:Integer, acceleration?:AnimationAcceleration): void;

/**
*  When this Canvas is included as a member in a Layout, dynamically updates whether a 
*  resizeBar should be shown after this member in the layout, to allow it to be resized.
*  
*  Whether a resizeBar is actually shown also depends on the 
*  Layout.defaultResizeBars attribute of the layout, and whether this
*  Canvas is the last layout member.
* 
* @param {boolean} setting for this.showResizeBar
*/
setShowResizeBar(show:boolean): void;



}
/**
* Creates a new Canvas
* 
* @param typeCheckedProperties {Partial<Canvas>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Canvas} The new instance
*/
class Canvas{
static create(typeCheckedProperties? : Partial<Canvas>, uncheckedProperties?: Object): Canvas;
/**
* A declared value of the enum type s 
*  Overflow and
*  Cursor.
* @type {Constant}
* @default "auto"
*/
static AUTO?: Constant;

/**
* A declared value of the enum type 
*  Positioning.
* @type {Constant}
* @default "absolute"
*/
static ABSOLUTE?: Constant;

/**
* A declared value of the enum type 
*  Positioning.
* @type {Constant}
* @default "relative"
*/
static RELATIVE?: Constant;

/**
* A declared value of the enum type 
*  Visibility.
* @type {Constant}
* @default "inherit"
*/
static INHERIT?: Constant;

/**
* A declared value of the enum type s 
*  Overflow and
*  Visibility.
* @type {Constant}
* @default "visible"
*/
static VISIBLE?: Constant;

/**
* A declared value of the enum type s 
*  Overflow and
*  Visibility.
* @type {Constant}
* @default "hidden"
*/
static HIDDEN?: Constant;

/**
* A declared value of the enum type 
*  Overflow.
* @type {Constant}
* @default "scroll"
*/
static SCROLL?: Constant;

/**
* A declared value of the enum type 
*  Overflow.
* @type {Constant}
* @default "clip-h"
*/
static CLIP_H?: Constant;

/**
* A declared value of the enum type 
*  Overflow.
* @type {Constant}
* @default "clip-v"
*/
static CLIP_V?: Constant;

/**
* A declared value of the enum type s 
*  ImageStyle,
*  VerticalAlignment and
*  Alignment.
* @type {Constant}
* @default "center"
*/
static CENTER?: Constant;

/**
* A declared value of the enum type s 
*  Side,
*  Alignment and
*  Direction.
* @type {Constant}
* @default "left"
*/
static LEFT?: Constant;

/**
* A declared value of the enum type s 
*  Side,
*  Alignment and
*  Direction.
* @type {Constant}
* @default "right"
*/
static RIGHT?: Constant;

/**
* A declared value of the enum type s 
*  VerticalAlignment and
*  Side.
* @type {Constant}
* @default "top"
*/
static TOP?: Constant;

/**
* A declared value of the enum type s 
*  VerticalAlignment and
*  Side.
* @type {Constant}
* @default "bottom"
*/
static BOTTOM?: Constant;

/**
* A declared value of the enum type 
*  Direction.
* @type {Constant}
* @default "up"
*/
static UP?: Constant;

/**
* A declared value of the enum type 
*  Direction.
* @type {Constant}
* @default "down"
*/
static DOWN?: Constant;

/**
* A declared value of the enum type 
*  Cursor.
* @type {Constant}
* @default "default"
*/
static DEFAULT?: Constant;

/**
* A declared value of the enum type 
*  Cursor.
* @type {Constant}
* @default "wait"
*/
static WAIT?: Constant;

/**
* A declared value of the enum type 
*  Cursor.
* @type {Constant}
* @default "hand"
*/
static HAND?: Constant;

/**
* A declared value of the enum type 
*  Cursor.
* @type {Constant}
* @default "pointer"
*/
static POINTER?: Constant;

/**
* A declared value of the enum type 
*  Cursor.
* @type {Constant}
* @default "help"
*/
static HELP?: Constant;

/**
* A declared value of the enum type 
*  Cursor.
* @type {Constant}
* @default "text"
*/
static TEXT?: Constant;

/**
* A declared value of the enum type 
*  ImageStyle.
* @type {Constant}
* @default "tile"
*/
static TILE?: Constant;

/**
* A declared value of the enum type 
*  ImageStyle.
* @type {Constant}
* @default "stretch"
*/
static STRETCH?: Constant;

/**
* A declared value of the enum type 
*  ImageStyle.
* @type {Constant}
* @default "normal"
*/
static NORMAL?: Constant;

/**
* A declared value of the enum type 
*  BackgroundRepeat.
* @type {Constant}
* @default "repeat"
*/
static REPEAT?: Constant;

/**
* A declared value of the enum type 
*  BackgroundRepeat.
* @type {Constant}
* @default "no-repeat"
*/
static NO_REPEAT?: Constant;

/**
* A declared value of the enum type 
*  BackgroundRepeat.
* @type {Constant}
* @default "repeat-x"
*/
static REPEAT_X?: Constant;

/**
* A declared value of the enum type 
*  BackgroundRepeat.
* @type {Constant}
* @default "repeat-y"
*/
static REPEAT_Y?: Constant;

/**
* Specifies the lower limit for automatically assigned tab indices for focusable canvii.
* @type {Integer}
* @default "1000"
*/
static TAB_INDEX_FLOOR?: Integer;

/**
* If set, the AlphaImageLoader IE filter will never be used.  Does not remove
*  AlphaImageLoader usage in already-drawn components.
*  
*  See IEFilters for background.
* @type {boolean}
* @default null
*/
static neverUsePNGWorkaround?: boolean;

/**
* Disables automatic use of filters in IE by default. Filters will only be used if
*  Canvas.useOpacityFilter is explicitly set to true on a component.
*  
*  Does not remove filters on already drawn components, or which are applied via CSS.
*  
*  See IEFilters for background.
* @type {boolean}
* @default null
*/
static neverUseFilters?: boolean;

/**
* If enabled, uses a moderately expensive workaround to allow the use of IE filters in CSS
*  to produce gradient effects for buttons, grid rows, and other elements, without the use
*  of image backgrounds.
*  
*  See IEFilters for background.
* @type {boolean}
* @default true
*/
static allowExternalFilters?: boolean;

/**
* Image URL to be displayed while data is being loaded (if enabled for the widget
*  waiting for data). Must be square; Canvas.loadingImageSize specifies the width
*  and height.
* @type {SCImgURL}
* @default "[SKINIMG]loadingSmall.gif"
*/
static loadingImageSrc?: SCImgURL;

/**
* Specifies the width and height of Canvas.loadingImageSrc.
* @type {Integer}
* @default "16"
*/
static loadingImageSize?: Integer;

/**
* A fixed number of pixels at the top of the page in which components will not be placed.
*  This is overridable per-instance via the Canvas.leavePageSpace attribute.
*  Essentially, the effect is that all top-level components are shifted down this number of
*  pixels, and the page height is treated as this number of pixels less than the
*  real page height.
*  
*  This attribute can be useful on certain mobile devices, when components should not be
*  placed in a top portion of the screen. For example, on iOS devices in certain configurations,
*  this can be set to 20 to avoid placing any component into the status bar area. Or, if
*  using iOS 7.1's 'minimal-ui' viewport parameter, this can be set to 20 to avoid placing
*  any component into the top 20px area of the screen, which if tapped on iPhone in landscape,
*  causes Mobile Safari's address bar and tab bar to be shown.
*  
*  This setting can be changed at runtime by calling Canvas.setDefaultPageSpace.
*  
*  Note: As documented by the mobileDevelopment
*  page, when the SmartClient application is running in Mobile Safari on iPhone running iOS 7.1 or
*  later, and neither the isc_useDefaultViewport nor the isc_useMinimalUI
*  global is set to false when the framework is loaded, then the framework
*  will automatically set the defaultPageSpace to 0 in portrait orientation,
*  and to 20 in landscape orientation.
* @type {Integer}
* @default "0"
*/
static defaultPageSpace?: Integer;

/**
* A declared value of the enum type 
*  DragDataAction.
* @type {Constant}
* @default "copy"
*/
static COPY?: Constant;

/**
* A declared value of the enum type s 
*  DragDataAction and
*  Cursor.
* @type {Constant}
* @default "move"
*/
static MOVE?: Constant;





/**
*  Changes the system-wide Canvas.neverUseFilters setting.
* 
* @param {boolean} new setting
*/
static setNeverUseFilters(neverUseFilters:boolean): void;

/**
*  Changes the system-wide Canvas.allowExternalFilters setting.
* 
* @param {boolean} new setting
*/
static setAllowExternalFilters(allExternalFilters:boolean): void;

/**
*  Changes the global Canvas.defaultPageSpace.
* 
* 
* @param {Integer} the new value for defaultPageSpace.
*/
static setDefaultPageSpace(newDefaultPageSpace:Integer): void;

/**
*  Check if an event is within an "edge" of this canvas.
* 
* 
* @param {Array<Partial<EdgeName>>=} Array of legal edges. Default is all the edges that                    allow resizing (see Canvas.resizeFrom)
*/
static getEventEdge(edgeMask?:Array<Partial<EdgeName>>): EdgeName;

/**
*  Retrieve a Canvas by it's global Canvas.ID.
* 
* @param {string} global ID of the Canvas
*/
static getById(ID:string): Canvas;

/**
*  Modify the size of fonts for some or all stylesheets defined in the page.
*  
*  This method can be used to dynamically increase or decrease font size for all of the fonts
*  in your application, or just fonts defined in your chosen SmartClient skin (the latter can
*  be achieved by passing styleSheets as "skin_styles.css" - the default name for
*  the CSS file used in each SmartClient skin).
*  
*  resizeFonts() must be called after the skin has been loaded, and before any
*  components have been created. Calling resizeFonts() at a later time is not
*  supported (you will notice that font sizes still increase, however, this approach is not
*  supported).
*  
*  Some browsers will disallow access or modification of styleSheets if they are loaded from a
*  domain that is different from the loading page. In this case resizeFonts()
*  cannot be used.
*  
*  This method has a small performance penalty which depends on the browser, number of
*  stylesheets being modified, and age of your machine. With modern browsers on modern
*  machines resizing just skin fonts, the impact is basically negligible (&lt;5ms).
*  
*  Certain controls such as icons are resized when fonts are resized (see 
*  Canvas.setAutoResizeIcons and Canvas.setAutoResizeAutoChildAttributes)
*  so you might want to set resizeRelatedControls to false where
*  you are just trying to make fonts in a dynamically loaded stylesheet match previously
*  loaded fonts, but controls such as icons should not be resized upwards again.
* 
* 
* @param {Integer} size to change fonts by. Can be negative to shrink fonts
* @param {string=} optional regular expression pattern for matching stylesheets
* @param {boolean=} resize icons and autoChild attributes? Set to false    to suppress default behavior.
*/
static resizeFonts(sizeChange:Integer, styleSheets?:string, resizeRelatedControls?:boolean): void;

/**
*  Modify the amount of padding for some CSS styles defined for the page. Only CSS styles
*  registered by Canvas.registerFontScaledPaddingStyles are modified.
*  
*  resizePadding() must be called after the skin has been loaded, and before any
*  components have been created. Calling resizePadding() at a later time is not
*  supported (you will notice that padding is modified, however, this approach is not supported).
*  
*  This method has similar browser security limitations as Canvas.resizeFonts.
*  
*  The intent is that the same font size change be passed to this method as is passed to
*  Canvas.resizeFonts, so that the targetSizeChange in the call to
*  Canvas.registerFontScaledPaddingStyles represents the right font size for the unadjusted
*  styles being registered.
* 
* @param {Integer} size change to apply to the padding of registered styles, so that               they aren't changed at all at the size change passed to                Canvas.registerFontScaledPaddingStyles, and the padding is               reduced to baseline style levels at a zero size change.
* @param {string=} optional regular expression pattern for matching stylesheets
*/
static resizePadding(fontSizeChange:Integer, styleSheets?:string): void;

/**
*  Registers one or more CSS classes to have their padding adjusted (independently on all edges)
*  according to the Canvas.resizePadding applied to the page. Each class
*  to be registered is provided along with a corresponding baseline class, and a single
*  targetSizeChange is specified for all the classes. The padding in each
*  registered class is adjusted downward towards the baseline as the padding size change
*  approaches 0 (no resizing), and upward as it increases, so that it exactly equals the
*  declared style's padding at a padding size change of targetSizeChange.
*  
*  Note that each call to this method replaces the registration of the previous call (if any),
*  and will have no effect until Canvas.resizePadding is called.
*  
*  For example:
*   isc.Canvas.registerFontScaledPaddingStyles(
*     [    "tabButtonTop",     "tabButtonBottom"], 
*     ["iconOnlyTabButtonTop", "iconOnlyTabButtonBottom"],
*     3
*   );
*  
*  In this case, the CSS style "tabButtonTop" will have its padding adjusted downward to the
*  padding from the baseline CSS style "iconOnlyTabButtonTop" style at a sizeChange
*  of 0, and be left unchanged at a sizeChange of 3.
* 
* @param {Array<Partial<CSSStyleName>>} styles whose padding should be adjusted
* @param {Array<Partial<CSSStyleName>>} corresponding baseline reference styles
* @param {Integer} sizeChange at which scaledStyles are unchanged
*/
static registerFontScaledPaddingStyles(scaledStyles:Array<Partial<CSSStyleName>>, baselineStyles:Array<Partial<CSSStyleName>>, targetSizeChange:Integer): void;

/**
*  Change the basic size of UI components in the current skin by "delta" pixels. Must be
*  called after the skin has been loaded, but before any components are created.
*  
*  In general, this method changes the height of various controls, except for certain 
*  controls that appear in alternate orientations (such as resizeBars and tabs), in which case 
*  thickness properties (resizeBarThickness, tabBarThickness) are adjusted.
*  
*  The height of a text input control implies the height of most other controls:
*  
*  
* - all other FormItems (eg selects) need to be the same height or mixed controls will look
*    odd. This includes Buttons
*  
* - anything that potentially contains a FormItem needs to be as tall or slightly taller:
*    this includes grid row (inline editing), Window.headerControls, 
*    TabBar and SectionHeaders
*  
*  
*  Because of this necessary uniformity, just specifying a single pixel value is enough for the
*  framework to resize all core controls, with several caveats:
*  
*  
* - skins that make extensive use of images (eg TreeFrog) will stretch those images, which
*    may result in ugly artifacts in some combinations of operating system and browser, for
*    which no workaround is possible. For this reason, resizeControls() is
*    only officially supported for Tahoe, Enterprise, EnterpriseBlue and Graphite skins, and only in
*    skinning
*  
* - even in Enterprise-series skins, TreeGrid.showConnectors 
*    vertically stretch, becoming obviously blurry and misshapen with an increase of 4-5px.
*    To avoid this, replace the tree connector media (see TreeGrid.connectorImage).
*  
* - FormItemIcons are not resized by default, because stretched icons generally look
*    worse than non-scaled icons that are a bit smaller than the input field
*  
* - images that use skinning will not be stretched because the sizes
*    for these controls are embedded in CSS. In most cases, this is desirable; for example,
*    the downward chevron shape used for SelectItem controls doesn't stretch, and looks
*    better that way.
*  
* 
* 
* @param {Integer} number of pixels to increase or decrease from current size
*/
static resizeControls(delta:Integer): void;

/**
*  Should icons be automatically resized with Canvas.resizeControls and
*  Canvas.resizeFonts?
*  
*  If true, icon attributes registered for resize with policy "controls" will
*  be resized when resizeControls() runs, and icons registered
*  with policy "fonts" will resize when resizeFonts() runs.
*  
*  To resize icons with other policies, developers should call
*  Canvas.resizeIcons directly
* 
* 
* 
* @param {boolean} true if icons should be auto-resized
*/
static setAutoResizeIcons(autoResizeIcons:boolean): void;

/**
*  Should registered autoChild attributes be automatically resized with 
*  Canvas.resizeControls and
*  Canvas.resizeFonts?
*  
*  If true, attributes registered for resize with policy "controls" will
*  be resized when resizeControls() runs, and icons registered
*  with policy "fonts" will resize when resizeFonts() runs.
*  
*  To resize autoChild attributes with other policies, developers should call
*  Canvas.resizeAutoChildAttributes directly
* 
* 
* 
* @param {boolean} true if attributes should be auto-resized
*/
static setAutoResizeAutoChildAttributes(autoResize:boolean): void;

/**
*  Change the basic size of icons in the current skin by "delta" pixels. This method may
*  be invoked automatically from Canvas.resizeControls.
*  
*  Must be called after the skin has been loaded, but before any components are created.
* 
* 
* @param {string} Which set of icons should be resized? This should correspond to the  iconSizingPolicy argument applied when   Canvas.registerIconSizingAttributes.
* @param {Integer} number of pixels to increase or decrease from current size
*/
static resizeIcons(policy:string, delta:Integer): void;

/**
*  Register numeric or measure type properties of some class(es) as
*  "icon sizing attributes". These are
*  properties that should be adjusted when Canvas.resizeIcons is called. 
*  Typically these will consist of icon width and height attributes on some SmartClient 
*  class, such as ListGrid.checkboxFieldImageHeight and
*  ListGrid.checkboxFieldImageWidth.
*  
*  The policy parameter allows icons to be grouped together into sets so particular types of
*  UI element can be resized in a targeted manner. For example, the width and height of
*  an icon which will appear aligned with a line of text would typically want to be resized
*  at the same time as font sizes are adjusted, whereas an icon that corresponds to a 
*  block of control UI (such as a picker icon that sits outside of an associated FormItem), 
*  should be resized when that control element is resized.
*  
*  A policy can be any string. To modify the sizes of attributes registered under some 
*  policy, that same policy string should be passed to the Canvas.resizeIcons method.
*  
*  The attributes parameter consists of a JavaScript object where each key specifies 
*  the name of the class on which the attributes, with its value set to an array, indicating
*  the attribute(s) to register within that class.
*  
*  Icon sizing attributes can be specified individually or in pairs. When a single attribute
*  passed in, a call to Canvas.resizeIcons will modify that attribute by the
*  delta parameter. If a pair of attributes is passed in, this is assumed to
*  be a height,width pair. When Canvas.resizeIcons is called in this case, 
*  both attributes will be modified such that they maintain the same scale. In other words,
*  the first registered attribute (typically the height) will be adjusted by the specified
*  delta (a simple numeric adjustment). The second attribute will be adjusted by 
*  a numeric delta calculated to have the same ratio to the original width as the 
*  provided delta had to the original height. This allows icons to be resized without
*  becoming distorted.
*  
*  For example, the following code would register FormItem attributes "valueIconSize",
*  "valueIconWidth" and "valueIconHeight" for resizing as part of the "fonts" policy,
*  such that both "valueIconSize" and "valueIconHeight" would be changed by the value passed
*  to resizeIcons(..), and "valueIconWidth" would be changed such that the
*  width/height ratio was retained.
*  
*  isc.Canvas.registerIconSizingAttributes(
*   "fonts",
*   {
*    FormItem:[
*      "valueIconSize",
*      ["valueIconHeight", "valueIconWidth"]
*    ]
*   }
*  );
*  
*  
*  The Canvas.setAutoResizeIcons feature will cause icon attributes
*  registered with policy "controls" to be resized when Canvas.resizeControls
*  is run and attributes registered as "fonts" to be resized when
*  Canvas.resizeFonts is run.
* 
* 
* @param {string} Icon sizing policy for this set of attributes
* @param {object} The attribute(s) to register for resizing.
*/
static registerIconSizingAttributes(policy:string, attributes:object): void;

/**
*  Register numeric or measure type properties of on autoChild properties blocks for
*  some class(es).
*  These are
*  properties that should be adjusted when Canvas.resizeAutoChildAttributes is called. 
*  Typically these will consist of width and height attributes on some autoChild defaults block
*  for some SmartClient class, such as Window.closeButtonDefaults
*  (to modify width and height of Window.closeButton).
*  
*  As with Canvas.registerIconSizingAttributes, the policy parameter allows
*  attributes to be grouped together into sets so particular types of
*  UI element can be resized in a targeted manner. For example, the width and height of
*  an icon which will appear aligned with a line of text would typically want to be resized
*  at the same time as font sizes are adjusted, whereas an icon that corresponds to a 
*  block of control UI (such as a picker icon that sits outside of an associated FormItem), 
*  should be resized when that control element is resized.
*  
*  A policy can be any string. To modify the sizes of attributes registered under some 
*  policy, that same policy string should be passed to the Canvas.resizeAutoChildAttributes 
*  method.
*  
*  The className should be the name of the class on which the auto child properties block 
*  exists.
*  
*  The attributes parameter consists of a JavaScript object where each key specifies 
*  the name of the autoChild properties block to modify, with its value set to an array, 
*  indicating the attribute(s) to register.
*  
*  Sizing attributes can be specified individually or in pairs. When a single attribute
*  passed in, a call to Canvas.resizeAutoChildAttributes will modify that attribute by the
*  delta parameter. If a pair of attributes is passed in, this is assumed to
*  be a height,width pair. When Canvas.resizeAutoChildAttributes is called in this case, 
*  both attributes will be modified such that they maintain the same scale. In other words,
*  the first registered attribute (typically the height) will be adjusted by the specified
*  delta (a simple numeric adjustment). The second attribute will be adjusted by 
*  a numeric delta calculated to have the same ratio to the original width as the 
*  provided delta had to the original height. This allows icons to be resized without
*  becoming distorted.
*  
*  For example, the following code would register the attributes "width" and "height" properties
*  on the "closeButtonDefaults" properties block, within the Window class - thus
*  customizing the default size of the Window.closeButton autoChild.
*  It registers them as part of the "controls" policy.
*  
*  isc.Canvas.registerAutoChildSizingAttributes(
*   "fonts",
*   "Window",
*   {
*    closeButtonDefaults:[
*      ["height", "width"]
*    ]
*   }
*  );
*  
*  
*  The Canvas.setAutoResizeAutoChildAttributes
*  feature will cause registered autoChild attributes to resize automatically.
*  Those registered with policy "controls" to be resized when Canvas.resizeControls
*  is run and attributes registered as "fonts" to be resized when
*  Canvas.resizeFonts is run.
* 
* 
* @param {string} Sizing policy for this set of attributes
* @param {string} Name of the class containing the autoChild configuration block to edit
* @param {object} The attribute(s) to register for resizing.
*/
static registerAutoChildSizingAttributes(policy:string, className:string, attributes:object): void;

/**
*  Change the value of attributes registered via Canvas.registerAutoChildSizingAttributes
*  by some number of pixels. This method may
*  be invoked automatically from Canvas.resizeControls or Canvas.resizeFonts
*  
*  Must be called after the skin has been loaded, but before any components are created.
* 
* 
* @param {string} Which set of attributes should be resized? This should correspond to the  sizing policy argument applied when   Canvas.registerAutoChildSizingAttributes.
* @param {Integer} number of pixels to increase or decrease from current size
*/
static resizeAutoChildAttributes(policy:string, delta:Integer): void;

/**
*  Whether to use the browser's native scrollbars or SmartClient-based scrollbars by default
*  for all canvases.
*  
*  This method changes the default value of Canvas.showCustomScrollbars.
* 
* @param {boolean} whether to show custom (SmartClient-based) scrollbars  rather than css-scrollbars by default.
*/
static setShowCustomScrollbars(showCustomScrollbars:boolean): void;

/**
*  Returns print-formatted HTML for the specified list of components.
* 
* 
* @param {Array<Partial<Canvas>>} Components to get the print HTML for. Strings of raw HTML may also be included in this array, and will be integrated into the final HTML at the appropriate  point.
* @param {PrintProperties} properties affecting print output
* @param {Callback=} Callback to fire when the method completes. The generated print HTML will be passed in as the first parameter HTML.
* @param {HTMLString=} Optional HTML separator to render between each component's printable HTML
*/
static getPrintHTML(components:Array<Partial<Canvas>>, printProperties:PrintProperties, callback?:Callback, separator?:HTMLString): HTMLString;

/**
*  Generates the HTML for an image. Also available at the 
*  Canvas.imgHTML.
* 
* 
* @param {SCImgURL} URL local to the skin or application directory.   NOTE: instead of passing several parameters, you can pass an object as the 'src'    parameter with properties for all the various function parameters with, eg:    canvas.imgHTML( {src:"foo", width:10, height:10} );
* @param {number=} width of the image
* @param {number=} height of the image
* @param {string=} name for the image
* @param {string=} additional attributes to write in the tag
* @param {string=} image-specific image directory
*/
static imgHTML(src:SCImgURL, width?:number, height?:number, name?:string, extraStuff?:string, imgDir?:string): string;

/**
*      Given a string and a character, hilite the first occurrence of the character in the
*      string (if it occurs), preferring uppercase to lowercase.
* 
* 
* @param {string} String to return with hilited character
* @param {string} Character to hilite
* @param {string=} Prefix to apply to hilighted character - defaults to                   "&lt;span style='text-decoration:underline;'&gt;"
* @param {string=} Suffix to apply to hilited character - defaults to                     "&lt;/span&gt;"
*/
static hiliteCharacter(string:string, character:string, hilitePrefix?:string, hiliteSuffix?:string): string;

/**
*  Return the position for snapper to be placed in order to "snap to" an edge or
*  corner of target, in the same sense as Canvas.snapTo.
*  
*  Default for snapEdge is the opposite edge or corner from
*  snapTo. For example, snapTo of "T" (top) means snapEdge
*  will default to "B" (bottom), so the returned coordinates would place snapper
*  centered along the top edge of target. A snapTo of "TL" (top left)
*  means snapEdge will default to "BR" (bottom right), so the returned coordinates
*  would place the bottom right of snapper at the top left corner of
*  target.
*  
*  target can be passed as either a Canvas or 4-element Array giving the top, left,
*  width and height of the target. 
*  snapper can be passed as either a Canvas or a
*  two-element Array of the width and height of the rectangle to be placed.
* 
* 
* @param {Canvas | Array<Partial<Integer>>} canvas to snap to
* @param {string} edge against which to snap
* @param {Canvas | Array<Partial<Integer>>} canvas being snapped
* @param {string=} optional edge to snapTo. Default is the opposite edge or corner from snapTo
*/
static getSnapPosition(target:Canvas | Array<Partial<Integer>>, snapTo:string, snapper:Canvas | Array<Partial<Integer>>, snapEdge?:string): Point;

/**
*  Generate printable HTML for the designated components and trigger the native print
*  dialog, without never showing the printable HTML to the user.
* 
* 
* @param {Array<Partial<Canvas>>} components to get the print HTML for. May also include    raw HTML strings which will be folded into the generated print output
* @param {PrintProperties=} object for customizing the print HTML output
*/
static printComponents(components:Array<Partial<Canvas>>, printProperties?:PrintProperties): void;

/**
*  Creates a printCanvas containing the full printHTML for a series of components, passing
*  it as an argument to the callback (if supplied) when it fires. Note that the generated
*  preview canvas will be drawn automatically by this method. Developers may also
*  explicitly create a PrintCanvas instance and populate it with HTML derived from the
*  Canvas.getPrintHTML for finer grained control over when the print canvas is
*  drawn.
* 
* @param {Array<Partial<Canvas>>} components to get the print HTML for. May also include  raw HTML strings which will be folded into the generated print output
* @param {PrintProperties=} PrintProperties object for customizing the print HTML output
* @param {Partial<Canvas>=} properties to apply to the generated printPreview Canvas.
* @param {PrintCanvasCallback=} callback to fire when the print preview canvas has been populated  with the printable HTML. The generated canvas will be passed to the callback as a single   printPreview parameter.
* @param {string=} optional string of HTML to render between each component
*/
static getPrintPreview(components:Array<Partial<Canvas>>, printProperties?:PrintProperties, previewProperties?:Partial<Canvas>, callback?:(printPreview:PrintCanvas) => void, separator?:string): void;

/**
*  Generate and show a PrintWindow containing a PrintCanvas
*  showing a printable view of the components passed in.
* 
* 
* @param {Array<Partial<Canvas>>} components to get the print HTML for. May also include raw HTML strings which will be folded into the generated print output
* @param {PrintProperties=} PrintProperties object for customizing the   print HTML output
* @param {Partial<PrintWindow>=} Properties to apply to the   generated print window.
* @param {Callback=} callback to fire when the print preview canvas has   been populated with the printable HTML. This callback takes 2 parameters:   printPreview - a pointer to the generated print canvas shown in the   body of the print window.   printWindow - a pointer to the generated print window and
* @param {string=} Optional HTML separator to render between each component's printable           HTML
*/
static showPrintPreview(components:Array<Partial<Canvas>>, printProperties?:PrintProperties, printWindowProperties?:Partial<PrintWindow>, callback?:Callback, separator?:string): void;




}

/**
* EditProxy that handles Canvas objects when editMode is enabled.
*/
interface CanvasEditProxy extends EditProxy {




}
/**
* Creates a new CanvasEditProxy
* 
* @param typeCheckedProperties {Partial<CanvasEditProxy>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {CanvasEditProxy} The new instance
*/
class CanvasEditProxy{
static create(typeCheckedProperties? : Partial<CanvasEditProxy>, uncheckedProperties?: Object): CanvasEditProxy;







}

/**
* FormItem which renders a Canvas inline in a DynamicForm instance.
*  
*  CanvasItem is CanvasItem.shouldSaveValue:false by default, meaning that
*  no value from the CanvasItem will be present in DynamicForm.getValues and no value will be
*  saved when DynamicForm.saveData is called. This is appropriate if the Canvas does
*  not participate in editing a value of the form and is embedded in the form for layout or UI
*  purposes only (e.g. ButtonItem, SectionItem). Note that some built-in CanvasItem
*  types override the shouldSaveValue default to true (e.g. MultiComboBoxItem, RichTextItem).
*  
*  If you set FormItem.shouldSaveValue:true, a 
*  CanvasItem.showValue event will be raised to provide a value that your
*  item should display. Handle this event by calling methods on the Canvas you've created
*  to cause the value to be displayed.
*  
*  The CanvasItem.showValue event will be triggered in various situations where
*  the form receives data, including a call to DynamicForm.setValues,
*  DynamicForm.editRecord, or if DynamicForm.fetchData is called and a Record
*  is returned. Bear in mind that the showValue event can be called when the form
*  and your item have not yet been drawn; in this case, store the value for later display.
*  
*  To provide a value to the form, call CanvasItem.storeValue whenever the user changes
*  the value in your Canvas. Generally, if storeValue() is called then
*  CanvasItem.shouldSaveValue should be overridden to true. Note that
*  the form will not call getValue() in order to discover your item's value, so there is
*  no purpose in overriding this method; instead, call storeValue() to proactively inform the
*  form about changes to the value. This approach is necessary in order to enable change events.
*  
*  If you cannot easily detect changes to values in your Canvas, a workaround is to call
*  storeValue right before the form saves.
*/
interface CanvasItem extends FormItem {
/**
* Height of the Canvas. Can be either a number indicating a fixed height in pixels, a
*  percentage indicating a percentage of the overall form's height, or "*" indicating take
*  whatever remaining space is available. See the formLayout overview for details.
*  
*  Height may also be explicitly specified on the CanvasItem.canvas. In this
*  any canvasItem.height will be ignored in favor of the value applied 
*  to the canvas directly. In either case, percentage values will be resolved using
*  standard formItem sizing rules as described in formLayout
* @type {Integer | string}
* @default null
*/
height?: Integer | string;

/**
* Whether this CanvasItem is intended to hold multiple values.
*  
* @type {boolean}
* @default null
*/
multiple?: boolean;

/**
* Should this item's value be saved in the form's values and hence returned from
*  DynamicForm.getValues?
*  
*  Note that by default, shouldSaveValue is false for CanvasItems,
*  meaning that no value from the CanvasItem will be present in DynamicForm.getValues
*  and no value for the CanvasItem will be saved when DynamicForm.saveData is called.
*  See the CanvasItem class overview for a discussion of values handling in
*  CanvasItems.
* @type {boolean}
* @default false
*/
shouldSaveValue?: boolean;

/**
* The canvas that will be displayed inside this item. You can pass an instance you've 
*  already created, or its global ID as a String. You can also implement 
*  CanvasItem.createCanvas to dynamically create the canvas when the FormItem
*  is initialized.
*  
*  If canvas and createCanvas() are unspecified, the 
*  canvas for this item will be auto-created using the overrideable defaults:
*  CanvasItem.canvasProperties and CanvasItem.canvasConstructor
*  
*  
*  Note that subclasses of CanvasItem may use a different AutoChild name than
*  just "canvas". For example, SliderItem uses "slider", and in that case, you need
*  to use the specific APIs provided by the subclass.
*  
*  Note that Canvas.canvasItem will be set on the canvas to point back to this
*  item.
* @type {Canvas}
* @default null
*/
canvas?: Canvas;

/**
* If CanvasItem.canvas is not specified as a canvas instance at init
*  time, a canvas will be created instead. This property denotes the class of that widget
*  (Should be set to the name of a subclass of Canvas).
* @type {string}
* @default "Canvas"
*/
canvasConstructor?: string;

/**
* Default properties for the canvas if this.canvas is not already a canvas instance.
* @type {object}
* @default "{}"
*/
canvasDefaults?: object;

/**
* Properties to apply to this canvas on creation if this.canvas is not already a canvas 
*  instance.
* @type {object}
* @default "{}"
*/
canvasProperties?: object;

/**
* Should this item's CanvasItem.canvas be automatically destroyed when the item 
*  is destroyed? Form items are destroyed automatically when a call to
*  DynamicForm.setItems removes them from their parent form, or if their
*  parent form is destroyed. This property governs whether, when this occurs, 
*  the item's canvas should also be Canvas.destroy.
*  
*  This property has no effect for canvases automatically created via the "autoChild" pattern, 
*  using CanvasItem.canvasProperties, CanvasItem.canvasDefaults etc. CanvasItems which create their
*  canvas in this way will always destroy the canvas when the item is destroyed or on an
*  explicit CanvasItem.setCanvas call, regardless of this property's value.
*  
*  Setting this property to true
*  is typically appropriate for cases where a custom CanvasItem automatically creates
*  its canvas as part of its initialization flow, and the canvas will not be re-used outside
*  the item.
*  Note that once a canvas has been destroyed it can not be re-used elsewhere within
*  an application.
* @type {boolean}
* @default false
*/
autoDestroy?: boolean;

/**
* CanvasItems support specifying overflow for the Canvas directly on the item.
* @type {Overflow}
* @default null
*/
overflow?: Overflow;

/**
* Minimum valid height for this CanvasItem in pixels. Used in calculating the row heights
*  of the containing DynamicForm if the item has a flexible CanvasItem.height.
* @type {Integer}
* @default null
*/
minHeight?: Integer;

/**
* Maximum valid height for this CanvasItem in pixels. Used in calculating the row heights
*  of the containing DynamicForm if the item has a flexible CanvasItem.height.
* @type {Integer}
* @default null
*/
maxHeight?: Integer;

/**
* Flag to disable the criteria editing overrides described in
*  CanvasItem.getCriterion
*  whereby
*  if this item contains a DynamicForm as its canvas, it will be used to edit nested
*  AdvancedCriteria automatically.
*  
*  This flag is required for cases where a canvasItem contains a DynamicForm, but the form
*  is not set up to show inner field values of nested objects, and therefore should not
*  attempt to apply nested advanced criteria directly to the form.
* @type {boolean}
* @default true
*/
editCriteriaInInnerForm?: boolean;

/**
* If FormItem.prompt is specified for this item, should the prompt be applied to the
*  CanvasItem.canvas for this item?
* @type {boolean}
* @default true
*/
applyPromptToCanvas?: boolean;

/**
* This text is shown as a tooltip prompt when the cursor hovers over this item.
*  
*  When item is FormItem.setCanEdit a different hover can be shown with
*  FormItem.readOnlyHover. Or, when item is FormItem.disabled or
*  read-only with FormItem.readOnlyDisplay a different hover can be shown
*  with FormItem.disabledHover.
*  
*  Note that when the form is Canvas.disabled this prompt will not
*  be shown.
* @type {HTMLString}
* @default null
*/
prompt?: HTMLString;



/**
*  Store (and optionally show) a value for this form item. 
*  
*  This method will fire standard FormItem.change and
*  DynamicForm.itemChanged handlers, and store the value passed in such that
*  subsequent calls to FormItem.getValue or DynamicForm.getValue will
*  return the new value for this item.
*  
*  This method is intended to provide a way for custom formItems - most commonly 
*  CanvasItem - to provide a new interface to the user, allowing them
*  to manipulate the item's value, for example in an embedded CanvasItem.canvas,
*  or a pop-up dialog launched from an FormItemIcon, etc. Developers
*  should call this method when the user interacts with this custom 
*  interface in order to store the changed value.
*  
*  CanvasItem.shouldSaveValue for CanvasItems is
*  false by default. Custom CanvasItems will need to override shouldSaveValue
*  to true if the values stored via this API should be included in the form's
*  DynamicForm.getValues and saved with the form when
*  DynamicForm.saveData is called.
*  
*  If you cannot easily detect interactions that should change the value as the 
*  user performs them, a workaround is to call
*  storeValue right before the form saves.
*  
*  Note that this method is not designed for customizing a value which is already being
*  saved by a standard user interaction. For example you should not call this method
*  from a FormItem.change. Other APIs such as 
*  FormItem.transformInput exist for this.
*  
* 
* @param {any} value to save for this item
* @param {boolean=} Should the formItem be updated to display the new value?
*/
storeValue(value:any, showValue?:boolean): void;

/**
*  This method will place an entry for the CanvasItem.canvas under this item in the 
*  TabIndexManager. This ensures the user can tab into the canvas (and its
*  descendants) in the expected position within this item's DynamicForm.
*  
*  See also Canvas.updateChildTabPositions.
*  
* 
*/
updateCanvasTabPosition(): void;

/**
*  Setter to update the CanvasItem.canvas at runtime
* 
* @param {Canvas} New canvas to display.
*/
setCanvas(canvas:Canvas): void;

/**
*  Notification method called when the FormItem.canEdit setting is modified.
*  Developers may make use of this to toggle between an editable and a read-only appearance
*  of the CanvasItem.canvas.
*  
*  The default behavior is:
*  
*  
* - If canvas is a DynamicForm, the form's DynamicForm.canEdit
*  setting is set to canEdit.
*  
* - CanvasItem.shouldDisableCanvas is called to determine if the canvas should
*  be disabled.
*  
*  
*  Standard CanvasItem-based form items may customize the default behavior.
*  For example, a MultiComboBoxItem will hide its MultiComboBoxItem.comboForm
*  if the FormItem.readOnlyDisplay is
*  "readOnly" or "static"
*  
*  and also disable the buttons when made read-only.
* 
* @param {boolean} New canEdit value
*/
canEditChanged(canEdit:boolean): boolean;

/**
*  Notification method called when the FormItem.readOnlyDisplay setting
*  is modified. Developers may make use of this to toggle between an editable and a read-only
*  appearance of the CanvasItem.canvas.
*  
*  The default behavior is: when the canvas is a DynamicForm, the form's
*  DynamicForm.readOnlyDisplay setting is set to appearance.
*  
*  Standard CanvasItem-based form items may customize the default behavior.
* 
* @param {ReadOnlyDisplayAppearance} new readOnlyDisplay value
*/
readOnlyDisplayChanged(appearance:ReadOnlyDisplayAppearance): boolean;

/**
*  Method called to determine whether the CanvasItem.canvas should be
*  Canvas.setDisabled when this CanvasItem is disabled
*  or its CanvasItem.canEditChanged.
*  By default, if the canvas is a DynamicForm, then it is disabled if
*  and only if this CanvasItem is disabled; otherwise, the canvas
*  is disabled if and only if this CanvasItem is disabled or
*  FormItem.getCanEdit.
*  
*  This method may be overridden to customize the default return value.
* 
*/
shouldDisableCanvas(): boolean;

/**
*  Does this CanvasItem have keyboard focus.
*  
*  This method will return true if this item's canvas, or any of its descendents, has
*  keyboard focus
* 
*/
isFocused(): boolean;

/**
*  Overridden to return true if CanvasItem.canvas is a dynamicForm.
*  See CanvasItem.getCriterion
*  
*  for details of editing advanced criteria using nested
*  dynamicForms.
* 
*/
hasAdvancedCriteria(): boolean;

/**
*  AdvancedCriteria objects may be edited via nested dynamicForms as described in
*  CanvasItem.getCriterion
*  
*  This method has been overridden to return true if this item's canvas is a DynamicForm,
*  where the DynamicForm.operator matches the operator of the criterion passed in
*  and dynamicForm contains items where FormItem.canEditCriterion returns true
*  for each sub-criterion in the object passed in.
* 
* @param {Criterion} criteria to test
*/
canEditCriterion(criterion:Criterion): boolean;

/**
*  The standard formItem criteria editing APIs have been overridden in the canvasItem class
*  to simplify the editing of complex AdvancedCriteria objects using nested 
*  DynamicForms.
*  
*  The following pattern is supported without need for further modification:
*  A complex Advanced criteria object may have nested sub criteria using the "and"
*  or "or" operators. For example:
*  
*  { _constructor:"AdvancedCriteria",
*   operator:"and",
*   criteria:[
*    {fieldName:"field1", value:"value1", operator:"iContains"},
*    {operator:"or", criteria:[
*      {fieldName:"innerField1", value:"value1", operator:"equals"},
*      {fieldName:"innerField2", value:"value2", operator:"iContains"}
*     ]
*    }
*   ]
*  }
*  
*  To create a form capable of editing the above criteria without providing custom overrides
*  to FormItem.getCriterion et al, you would create a form with 2 items.
*  The 'field1' criterion could be edited by a simple form item such as a TextItem.
*  The nested criteria ('innerField1' and 'innerField2') could be edited by a canvasItem
*  whose canvas property was set to a DynamicForm showing items capable of editing the 2
*  inner criteria, and whose operator was specified as "or".
*  For example:
*  
*  
*  isc.DynamicForm.create({
*    items:[
*      {name:"field1", type:"TextItem"},
*      {name:"nestedItem", shouldSaveValue:true, type:"CanvasItem",
*        canvas:isc.DynamicForm.create({
*          operator:"or",
*          items:[
*            {name:"innerField1", type:"TextItem", operator:"equals"},
*            {name:"innerField2", type:"TextItem"}
*          ]
*        })
*      }
*    ]
*  });
*  
*  
*  
*  This form would be able to edit the above advanced criteria object via
*  DynamicForm.setValuesAsCriteria. Edited values would be retrieved via
*  DynamicForm.getValuesAsCriteria.
*  
*  Note that the canvas item has shouldSaveValue set to true - this is required
*  to ensure the nested form is actually passed the values to edit. 
*  
*  The default implementation of this method checks for this.canvas being specified as a
*  dynamicForm, and in that case simply returns the result of 
*  DynamicForm.getValuesAsAdvancedCriteria on the inner form.
*  
*  Note that this functionality may be entirely bypassed by
*  setting CanvasItem.editCriteriaInInnerForm to false. This flag is useful when defining a
*  dynamicForm based canvasItem which is not intended for editing nested data -- for example
*  if a standard atomic field value is being displayed in some custom way using a
*  DynamicForm embedded in the item.
*  
* 
*/
getCriterion(): Criterion;

/**
*  Display a Criterion object in this item for editing. Overridden from 
*  FormItem.setCriterion in order to support editing nested criteria using 
*  nested dynamicForms as described in CanvasItem.getCriterion.
*  
*  Implementation checks for this.canvas being specified as a DynamicForm, and applies
*  criterion directly to the embedded form via setValuesAsCriteria()
* 
* @param {Criterion} criteria to edit
*/
setCriterion(criterion:Criterion): void;

/**
*  Set the FormItem.prompt for this item. Default implementation will also apply the
*  prompt to CanvasItem.canvas if CanvasItem.applyPromptToCanvas is true.
* 
* @param {HTMLString} new prompt for the item.
*/
setPrompt(prompt:HTMLString): void;

/**
*  This method allows dynamic creation of a CanvasItem's canvas, rather than
*  setting CanvasItem.canvas statically.
*  If specified this stringMethods will be called when the form item is 
*  initialized and should return the Canvas to display for this item.
* 
* @param {DynamicForm} the dynamicForm in which this item is contained
* @param {CanvasItem} the live form item instance
*/
createCanvas(form:DynamicForm, item:CanvasItem): Canvas;

/**
*  This method will be called whenever this FormItem's value is being set via a programmatic
*  call to e.g: DynamicForm.setValues or FormItem.setValue and may be
*  overridden by CanvasItems intended to support displaying data values to update the
*  embedded Canvas to reflect the value passed in. Note that the first parameter will be a 
*  formatted value - while the second parameter will contain the underlying data value for
*  the item.
* 
* @param {any} new display value for the item. This is the value after applying  any custom formatter or valueMap
* @param {any} underlying data value for the item
* @param {DynamicForm} the dynamicForm in which this item is contained
* @param {CanvasItem} the live form item instance
*/
showValue(displayValue:any, dataValue:any, form:DynamicForm, item:CanvasItem): void;



}
/**
* Creates a new CanvasItem
* 
* @param typeCheckedProperties {Partial<CanvasItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {CanvasItem} The new instance
*/
class CanvasItem{
static create(typeCheckedProperties? : Partial<CanvasItem>, uncheckedProperties?: Object): CanvasItem;







}

/**
* Maintains a representation of selection over a 2-dimensional grid of objects.
*  Automatically created to manage cell-selection on CubeGrid widgets.
*/
interface CellSelection extends Class {


/**
*  Return true if a particular item is selected
* 
* @param {Integer} row index of the cell to check
* @param {Integer} column index of the cell to check
*/
cellIsSelected(rowNum:Integer, colNum:Integer): boolean;

/**
*  Is anything in the list selected?
* 
*/
anySelected(): boolean;

/**
*  Returns an array of the currently selected cells. Each cell is returned as a 2 element
*  array in the form [rowNum, colNum].
* 
*/
getSelectedCells(): Array<any>;

/**
*  Returns the first record that has any cells selected.
* 
*/
getSelectedRecord(): ListGridRecord;

/**
*  Observable handler fired
*  whenever the cell selection is modified
* 
* 
*/
selectionChanged(): void;

/**
*  Select a particular cell
* 
* 
* @param {Integer} row index of cell to select
* @param {Integer} column index of cell to select
*/
selectCell(rowNum:Integer, colNum:Integer): boolean;

/**
*  Deselect a particular cell
* 
* 
* @param {Integer} row index of the cell to select
* @param {Integer} column index of the cell to select
*/
deselectCell(rowNum:Integer, colNum:Integer): boolean;

/**
*   select an array of cells
* 
* @param {Array<Partial<any>>} Array of cells to select. Each cell can be specified                    as a 2 element array [rowNum, colNum]
*/
selectCellList(list:Array<Partial<any>>): boolean;

/**
*   deselect an array of cells
* 
* 
* @param {Array<Partial<any>>} Array of cells to deselect. Each cell can be specified                    as a 2 element array [rowNum, colNum]
*/
deselectCellList(list:Array<Partial<any>>): boolean;

/**
*   select a single cell and deselect everything else
* 
* @param {Integer} row index of cell to select
* @param {Integer} column index of cell to select
*/
selectSingleCell(rowNum:Integer, colNum:Integer): boolean;



}
/**
* Creates a new CellSelection
* 
* @param typeCheckedProperties {Partial<CellSelection>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {CellSelection} The new instance
*/
class CellSelection{
static create(typeCheckedProperties? : Partial<CellSelection>, uncheckedProperties?: Object): CellSelection;







}

/**
* Checkbox form item, implemented with customizable checkbox images.
*/
interface CheckboxItem extends FormItem {
/**
* Default class used to construct the EditProxy for this component
*  when the component is Canvas.setEditMode.
* @type {SCClassName}
* @default "CheckboxItemEditProxy"
*/
editProxyConstructor?: SCClassName;

/**
* Base CSS class for this item's title cell.
*  
*  Note: This styling applies to the standard form item title cell for this item - it
*  does not apply to item's CheckboxItem.showLabel.
*  To modify the styling for that text, use CheckboxItem.textBoxStyle instead.
* @type {FormItemBaseStyle}
* @default null
*/
titleStyle?: FormItemBaseStyle;

/**
* Base CSS class for this item's title text
* @type {FormItemBaseStyle}
* @default "labelAnchor"
*/
textBoxStyle?: FormItemBaseStyle;

/**
* Width for the CheckboxItem, including both checkbox image and CheckboxItem.showLabel.
*  Note that if CheckboxItem.showLabel is false, this property will have no effect - the
*  item will render at the size required to contain the icon.
* @type {number}
* @default "200"
*/
width?: number;

/**
* CheckboxItem has special behavior for titles, see CheckboxItem.labelAsTitle.
* @type {boolean}
* @default true
*/
showTitle?: boolean;

/**
* By default a checkboxItem sets CheckboxItem.showTitle:true, and so takes up two cells with the
*  default TitleOrientation of "left" (see formLayout). However, the title cell is left blank by default, and the title specified
*  by FormItem.title is shown inside the formItem's cell instead, in an element
*  called the "label".
*  
*  To instead show the title in it's original location, set labelAsTitle:true.
*  You can also set CheckboxItem.showLabel:false to suppress the label and/or title altogether.
* @type {boolean}
* @default null
*/
labelAsTitle?: boolean;

/**
* Should we show the label text next to the checkbox item.
* @type {boolean}
* @default true
*/
showLabel?: boolean;

/**
* Should an "Over" state icon be shown when the user rolls over this checkbox
* @type {boolean}
* @default true
*/
showValueIconOver?: boolean;

/**
* Should a "Down" state icon be shown when the mouse goes down over this checkbox
* @type {boolean}
* @default true
*/
showValueIconDown?: boolean;

/**
* Should a "Disabled" state icon be shown when the item is disabled
* @type {boolean}
* @default true
*/
showValueIconDisabled?: boolean;

/**
* URL for the image to display when this checkbox is selected, or checked.
*  
*  This image is implemented using the FormItem.valueIcons,
*  and may be modified via the standard valueIcons properties such as
*  CheckboxItem.valueIconWidth
*  
*  Note that this is the base image name - if CheckboxItem.showValueIconOver et al are set, the
*  state ("Over", "Down" and "Disabled") will be
*  added to this name as the user interacts with the checkbox, as well as the image extension
*  
*  The special value "blank" means that no image will be shown.
*  
*  skinning can be used for this image, by setting this property to
*  a SCSpriteConfig formatted string. Alternatively developers can
*  omit this property and instead use CSS directly in the 
*  CheckboxItem.booleanBaseStyle property to provide a "checked" appearance.
* @type {SCImgURL}
* @default "[SKIN]/DynamicForm/checked.gif"
*/
checkedImage?: SCImgURL;

/**
* URL for the image to display when this checkbox is not selected, or unchecked.
*  
*  The special value "blank" means that no image will be shown.
*  
*  skinning can be used for this image, by setting this property to
*  a SCSpriteConfig formatted string. Alternatively developers can
*  omit this property and instead use CSS directly in the 
*  CheckboxItem.booleanBaseStyle property to provide an "unchecked" appearance.
* @type {SCImgURL}
* @default "[SKIN]/DynamicForm/unchecked.gif"
*/
uncheckedImage?: SCImgURL;

/**
* URL for the image to display when this checkbox is partially selected.
*  
*  The special value "blank" means that no image will be shown.
*  
*  skinning can be used for this image, by setting this property to
*  a SCSpriteConfig formatted string. Alternatively developers can
*  omit this property and instead use CSS directly in the 
*  CheckboxItem.booleanBaseStyle property to provide a "partially checked" appearance.
* @type {SCImgURL}
* @default "[SKIN]/DynamicForm/partialcheck.gif"
*/
partialSelectedImage?: SCImgURL;

/**
* If set, the CheckboxItem.checkedImage to use when printing.
* @type {SCImgURL}
* @default null
*/
printCheckedImage?: SCImgURL;

/**
* If set, the CheckboxItem.uncheckedImage to use when printing.
* @type {SCImgURL}
* @default null
*/
printUncheckedImage?: SCImgURL;

/**
* If set, the CheckboxItem.partialSelectedImage to use when printing.
* @type {SCImgURL}
* @default null
*/
printPartialSelectedImage?: SCImgURL;

/**
* Determines what image to display when the value for this checkbox is unset.
*  Set to true to display the CheckboxItem.unsetImage for null values, or false to
*  use the CheckboxItem.uncheckedImage for both null and explicitly unchecked values.
*  
*  If this attribute is not set, the CheckboxItem.unsetImage for null values if 
*  CheckboxItem.allowEmptyValue is true for this item, otherwise the unchecked
*  image will be used.
* @type {boolean}
* @default null
*/
showUnsetImage?: boolean;

/**
* URL for the image to display when this checkbox is unset. Note that if 
*  CheckboxItem.showUnsetImage is false or CheckboxItem.allowEmptyValue is false
*  the CheckboxItem.uncheckedImage will be used for null values rather than this
*  image.
*  
*  The special value "blank" means that no image will be shown.
*  
*  skinning can be used for this image, by setting this property to
*  a SCSpriteConfig formatted string. Alternatively developers can
*  omit this property and instead use CSS directly in the 
*  CheckboxItem.booleanBaseStyle property to provide an "unset" appearance.
* @type {SCImgURL}
* @default "[SKIN]/DynamicForm/unsetcheck.gif"
*/
unsetImage?: SCImgURL;

/**
* If set, the CheckboxItem.unsetImage to use when printing.
* @type {SCImgURL}
* @default null
*/
printUnsetImage?: SCImgURL;

/**
* The description shown in a hover when FormItem.showOldValueInHover is enabled
*  and a value represents the checked state.
* @type {HTMLString}
* @default "checked"
*/
checkedDescription?: HTMLString;

/**
* The description shown in a hover when FormItem.showOldValueInHover is enabled
*  and a value represents the unchecked state.
* @type {HTMLString}
* @default "unchecked"
*/
uncheckedDescription?: HTMLString;

/**
* The description shown in a hover when FormItem.showOldValueInHover is enabled
*  and a value represents the partial selected state.
* @type {HTMLString}
* @default "partially selected"
*/
partialSelectedDescription?: HTMLString;

/**
* The description shown in a hover when FormItem.showOldValueInHover is enabled
*  and a value represents the unset state.
* @type {HTMLString}
* @default "unset"
*/
unsetDescription?: HTMLString;

/**
* An optional base CSS style to apply to the checkbox image. If supplied, the base style is
*  suffixed with "True", "False", "Partial", or "Unset" if the checkbox is selected, unselected,
*  partially selected, or unset. The style is then suffixed with the state of the value icon
*  ("", "Over", "Down", "Disabled").
* @type {CSSStyleName}
* @default null
*/
booleanBaseStyle?: CSSStyleName;

/**
* If set, the CheckboxItem.booleanBaseStyle to use when printing.
* @type {CSSStyleName}
* @default null
*/
printBooleanBaseStyle?: CSSStyleName;

/**
* Width of the checkbox image.
* @type {number}
* @default "13"
*/
valueIconWidth?: number;

/**
* Height of the checkbox image.
* @type {number}
* @default "13"
*/
valueIconHeight?: number;

/**
* Object defining how the checkbox "checked" state will be mapped to the field value.
*  Checkboxes only support 2 states. By default a checked checkbox will have
*  value true, and an unchecked one will have value false
*  (or null if there is no default value and the value has never been set).
*  
*  A valueMap can modify this in 2 ways:
*  - If the desired checked/unchecked values can be resolved to true 
*   and false directly in JavaScript, the valueMap may be specified as
*   a 2-element array containing these values. Examples of this would include
*   &nbsp;&nbsp;[0,1]: 
*   In this case an unchecked checkbox would have value 0 and a checked box
*   would have value 1
*   &nbsp;&nbsp;[null,"Some String"]:
*   In this case an unchecked checkbox would have value null and a checked 
*   box would have value "Some String"
*  - More arbitrary data values can be resolved to checked / unchecked values via an
*   object mapping the arbitrary data values to display values of true and
*   false. For example:
*   &nbsp;&nbsp;{"A":false, "B":true}
*   In this case an unchecked checkbox would have value "A", and a checked box 
*   would have value "B"
*  
*  Note: ValueMaps in other formats will be ignored by the CheckboxItem class. To update
*  the valueMap at runtime, always use CheckboxItem.setValueMap
* @type {object | Array<any>}
* @default null
*/
valueMap?: object | Array<any>;

/**
* By default checkboxes allow the user to toggle between true and false values only.
*  Setting this property to true will allow the user to toggle between three values - 
*  true, false and unset.
* @type {boolean}
* @default false
*/
allowEmptyValue?: boolean;

/**
* Should a null value be shown as checked (i.e. true)?
* @type {boolean}
* @default false
*/
showNullAsTrue?: boolean;

/**
* If this checkbox item is CheckboxItem.showLabel, should
*  it ignore any specified FormItem.width and instead size to fit its 
*  CheckboxItem.checkedImage?
*  
*  When set to true (the default), the checkbox item ignores any specified width,
*  ensuring that it does not impact the the containing DynamicForm's table geometry
*  unnecessarily.
* @type {boolean}
* @default true
*/
sizeToCheckboxImage?: boolean;



/**
*  Setter method to apply a valueMap to a checkbox item.
*  Note that if this method is overridden, the override must call
*  this.Super("setValueMap", arguments); to maintain functionality in this
*  class.
* 
*/
setValueMap(): void;

/**
*  Return the value tracked by this form item as a Boolean. If the value is not
*  already a boolean, or is unset and CheckboxItem.allowEmptyValue is true,
*  then null will be returned.
* 
* 
*/
getValueAsBoolean(): boolean;

/**
*  Notification method called when FormItem.showPending is enabled and
*  this checkbox item should either clear or show its pending visual state.
*  
*  The default behavior is that the FormItem.cellStyle and checkbox label
*  style are updated to include/exclude the "Pending" suffix. Returning false
*  will cancel this default behavior.
* 
* @param {DynamicForm} the managing DynamicForm instance.
* @param {FormItem} the form item itself (also available as "this").
* @param {boolean} true if the item should show its pending visual state; false otherwise.
* @param {any} the current form item value.
* @param {any} the value that would be restored by a call to DynamicForm.resetValues.
*/
pendingStatusChanged(form:DynamicForm, item:FormItem, pendingStatus:boolean, newValue:any, value:any): boolean;



}
/**
* Creates a new CheckboxItem
* 
* @param typeCheckedProperties {Partial<CheckboxItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {CheckboxItem} The new instance
*/
class CheckboxItem{
static create(typeCheckedProperties? : Partial<CheckboxItem>, uncheckedProperties?: Object): CheckboxItem;







}

/**
* EditProxy that handles CheckboxItem when editMode is enabled.
*/
interface CheckboxItemEditProxy extends FormItemEditProxy {


/**
*  Returns the text based on the current component state to be edited inline.
*  Called by the EditProxy.inlineEditForm to obtain the starting edit value.
*  
*  Returns the component's value as [ ] or [x].
* 
* 
*/
getInlineEditText(): void;

/**
*  Save the new value into the component's state. Called by the
*  EditProxy.inlineEditForm to commit the change.
*  
*  Updates the component's value.
* 
* 
* @param {string} the new component value
*/
setInlineEditText(newValue:string): void;



}
/**
* Creates a new CheckboxItemEditProxy
* 
* @param typeCheckedProperties {Partial<CheckboxItemEditProxy>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {CheckboxItemEditProxy} The new instance
*/
class CheckboxItemEditProxy{
static create(typeCheckedProperties? : Partial<CheckboxItemEditProxy>, uncheckedProperties?: Object): CheckboxItemEditProxy;







}

/**
* The Class object is root of the Isomorphic SmartClient inheritance tree -- it includes
*  functionality for creating instances, adding methods and properties, getting prototypes,
*  etc.
* 
*  To add functionality to ALL classes, add them to Class.
* 
*  To create a Class, call ClassFactory.defineClass("MyClass", "MySuperClass")
*  
*  defineClass will return the created class, and make it available as
*  isc.MyClass, and as the global variable MyClass if not in
*  isc.
*  
*  You can then:
*  
*  
* - add class-level (static) properties and methods to the class:
*   MyClass.addClassProperties()	
*  	these methods and properties are accessed through the Class variable itself, eg:
*   MyClass.someStaticMethod() or MyClass.someStaticProperty
* 
*  
* - add default instance properties and methods to the class:
*   MyClass.addProperties()
*  	these methods and properties are accessed through a class instance, eg:
*   var myInstance = MyClass.create();
*   myInstance.someInstanceMethod()
* 
*  
* - create new instances of this class:
*   var myInstance = MyClass.create()
*  
*  NOTE: as a convention, all class names begin with a capital letter and all instances begin
*  with a lower case letter.
*/
interface Class {
/**
* Controls whether arguments passed to Class.create are assumed to be
*  Objects containing properties that should be added to the newly created instance. This
*  behavior is how create() works with almost all SmartClient widgets and
*  other components, allowing the convenient shorthand of setting a batch of properties via
*  an ObjectLiteral passed to create().
*  
*  The setting defaults to true if unset. To disable this behavior for a custom class,
*  such that create() works more like typical constructors found in Java and
*  other languages, use:
*  
*    isc.[i]ClassName[/i].addProperties({ addPropertiesOnCreate:false })
*  
*  
*  Note that it is not valid to disable this behavior for any subclass of Canvas
*  (Canvas relies on this property).
*  
*  Regardless of the setting for addPropertiesOnCreate, all arguments passed to
*  Class.create are still passed on to Class.init.
* @type {boolean}
* @default undefined
*/
addPropertiesOnCreate?: boolean;

/**
* For an autoChildren, a read-only reference to the component on
*  which Class.createAutoChild or Class.addAutoChild was called to create it.
*  Useful for authoring of event handlers (eg click:"this.creator.doSomething()")
* @type {Class}
* @default "varies"
*/
creator?: Class;

/**
* Specifies the component on which Class.createAutoChild should be called to
*  create autoChildren defined lazily on this component in the format
*  "autoChild:autoChildName". This property may be either specified as a live
*  component, or set to the childName of another already-created
*  AutoChild.
*  
*  If left unspecified, the Framework applies rules to determine which component to use as
*  the creator. If this component is itself an autochild, and properties or defaults for
*  the child are defined on its Class.creator but not on the component itself, then the
*  creator of this component is also used to create the new AutoChild. Otherwise, this
*  component is used.
* @type {Class | string}
* @default "varies"
*/
autoCreator?: Class | string;

/**
* Canvas.ID of the component that manages "rule context" for which
*  this class participates. A non-Canvas class can only use the ruleScope
*  for supporting Class.dynamicProperties. Unlike Canvas.ruleScope
*  ruleScope on a standalone class must be explicitly specified.
* @type {string}
* @default null
*/
ruleScope?: string;

/**
* Object mapping dynamic property names to the source - a DataPath, 
*  UserSummary or UserFormula. This is a declarative
*  alternative to calling Class.addDynamicProperty for each property.
*  
*  See Class.addDynamicProperty for details on using dynamic properties.
*  
*  
*  In JavaScript dynamicProperties can be declaratively initialized as follows:
*  
*  dynamicProperties: {
*    propName1 : "a/b/c",
*    propName2 : { formula: .. formula definition .. }
*    propName3 : { textFormula: .. summary definition .. }
*  }
*  
*  
*  
*  In ComponentXML dynamicProperties can be intialized as:
*  
*  &lt;dynamicProperties&gt;
*    &lt;property name="propName" dataPath="a/b/c"/&gt;
*    &lt;property name="propName2"&gt;
*      &lt;formula&gt;
*        &lt;UserFormula ... &gt;
*      &lt;/formula&gt;
*    &lt;/property&gt;
*    &lt;property name="propName3"&gt;
*      &lt;textFormula&gt;
*        &lt;UserSummary ... &gt;
*      &lt;/textFormula&gt;
*    &lt;/property&gt;
*  &lt;/dynamicProperties&gt;
*  
* @type {Map<any>}
* @default null
*/
dynamicProperties?: Map<any>;



/**
* 	
*  Initialize a new instance of this Class. This method is called automatically by
*  Class.create. 
*  
*  Override this method to provide initialization logic for your class. If your class is
*  a subclass of a UI component (i.e. descendant of Canvas), override
*  Canvas.initWidget instead. 
* 
* 
* @param {any=} All arguments initially passed to Class.create
*/
init(...args:any[]): void;

/**
*  Permanently destroy a class instance and any automatically created resources,
*  recursively.
* 
*/
destroy(): void;

/**
*  	Return the global identifier for this object.
* 
* 
*/
getID(): string;

/**
* 	
* 	Gets a pointer to the class object for this instance
* 
* 
*/
getClass(): Class;

/**
* 	
* 	Gets a pointer to the class object for this instance's superclass.
* 
* 
*/
getSuperClass(): Class;

/**
* 	
* 	Gets the name of this class as a string.
* 
* 
*/
getClassName(): string;

/**
* 	
*  Add properties or methods to this specific instance. 
* 	Properties with the same name as existing properties will override.
* 
* 
* @param {object=} Object containing name:value pairs to be added to this object
*/
addProperties(...args:any[]): object;

/**
* 
* 	Add properties to this instance.
* 
* 
* @param {Array<Partial<object>>} array of objects with properties to add
*/
addPropertyList(list:Array<Partial<object>>): object;

/**
*  Set a property on this object, calling the setter method if it exists.
*  
*  Whenever you set a property on an ISC component, you should call either the specific setter
*  for that property, or setProperty()/setProperties() if it doesn't have one.
*  This future-proofs your code against the later addition of required setters. 
* 
* 
* @param {string} name of the property to set
* @param {any} new value for the property
*/
setProperty(propertyName:string, newValue:any): void;

/**
*  Set multiple properties on an object, calling the appropriate setter methods if any are
*  found.
*  
*  Whenever you set a property on an ISC component, you should call either the specific setter
*  for that property, or setProperty()/setProperties() if it doesn't have one.
*  This future-proofs your code against the later addition of required setters. 
*  
*  With setProperties() in particular, some classes may be able to take shortcuts
*  and be more efficient when 2 or more related properties are set at the same time.
* 
* 
* @param {object=} objects with properties to add (think named parameters).     all the properties of each argument will be applied one     	after another so later properties will override
*/
setProperties(...args:any[]): void;

/**
* 
* 	Returns whether this object is of a particular class by class name, either as a direct
* 	instance of that class or as subclass of that class, or by implementing an interface
*  that has been mixed into the class.
* 
* 	NOTE: this only applies to ISC's class system, eg: myInstance.isA("Object") will be
* 	false.
* 
* 
* @param {string} Class name to test against
*/
isA(className:string): boolean;

/**
*  Set up a notification action to be invoked whenever some method is called on a target
*  object.
*  
*  For example, if you wanted to take an action every time some ListGrid on your page
*  had its ListGrid.selectionUpdated, instead of 
*  overriding the selectioUpdated() method on the grid, you could
*  observe it with code like this:
*  
*  myCanvas.observe(myListGrid, "selectionUpdated", "observer.gridSelectionUpdated()")
*  
*  In this example, every time selectionUpdated() fired on the grid "myListGrid", 
*  after that method completed, the specified action would be invoked. (In this case
*  the action would call a method called "gridSelectionUpdated" on the observer, "myCanvas").
*  
*  An unlimited number of observers can be set up to observe any method. The notification
*  actions will all be fired automatically in the order that the observations were set up.
*  
*  NOTES: 
*  
*  
* - The object to observe may be any JavaScript object with the specified method,
*    including simple JavaScript Objects or Arrays, or instances of SmartClient classes
*    like Canvas.
*  
* - For a given observer, observed object, and observed method combination, at most one
*    action can be registered. If you attempt to call observe() again with
*    the same combination, it will return false and the action will not be registered.
*  
* - A method could potentially trigger an observation of itself by another object, 
*    either through code within the method itself or within an observer's action.
*    In this case the observation will be set up, but the new observation action will
*    not fire as part of this thread. For subsequent calls to the method, the newly 
*    added observer will be fired.
*  
* - [Potential memory leak]: If the target object is a simple JavaScript object
*    (not an instance of a SmartClient class), developers should always call 
*    Class.ignore to stop observing the object when an observation 
*    is no longer necessary.
*    This ensures that if the observed object is subsequently allowed to go out of scope by 
*    application code, the observation system will not retain a reference
*    to it (so the browser can reclaim the allocated memory).
*    While cleaning up observations that are no longer required is always good practice, 
*    this memory leak concern is not an issue if the target object is an instance of
*    a SmartClient class. In that case the observation is automatically released when the 
*    target is Class.destroy.
*  
* 
* 
* @param {object} Object to observe. This may be any JavaScript object with the specified  target method, including native arrays, and instances of SmartClient classes such as   Canvas.
* @param {string} Name of the method to observe. Every time this method is invoked  on the target object the specified action will fire   (after the default implementation completes).
* @param {Function | string=} Optional action to take when the observed method is invoked on the target object.  If action is a string to execute, certain keywords are available for context:    
- observer is this object (the object on which the    observe(...) method was called).  
- observed is the target object being observed (on which the method was invoked).  
- returnVal is the return value from the observed method (if there is one)  
- For functions defined with explicit parameters, these will also be available as keywords within    the action string    If action is a function, this will be executed in the scope of the   observer (so this will be the object on which observe()   was invoked). The arguments for the original method will also be   passed to this action function as arguments. If developers need to access the target   object being observed from the action function they may use native javascript techniques  such as   javascript closure  to do so. The return value from the observed method is not available to the action function.  If the action parameter is omitted the default behavior will invoke the   same named method on the observer, passing in the same parameters.
*/
observe(object:object, methodName:string, action?:Function | string): boolean;

/**
*  Stop observing a method on some other object.
* 
* 
* @param {object} object to observe
* @param {string} name of the method to ignore
*/
ignore(object:object, methodName:string): boolean;

/**
*  Return true if this object is already observing a method of another object
* 
* 
* @param {object} object we may be observing
* @param {string} name of the method to observed
*/
isObserving(object:object, methodName:string): boolean;

/**
*  
*  Evaluate a string of script in the scope of this instance (so this
*  is available as a pointer to the instance).
* 
* 
* @param {string} the expression to be evaluated
* @param {object} Optional mapping of argument names to values - each key will   be available as a local variable when the script is executed.
*/
evaluate(expression:string, evalArgs:object): any;

/**
* 
* 	Method to fire a callback. Callback will be fired in the scope of the object on 
*  which this method is called.
*  Falls through to Class.fireCallback
* 
* 
* @param {Callback} Callback to fire
* @param {string=} comma separated string of variables
* @param {Array<any>=} array of arguments to pass to the method
*/
fireCallback(callback:Callback, argNames?:string, args?:Array<any>): any;

/**
*  This is a helper to delay a call to some method on this object by some specified
*  amount of time.
* 
* @param {string} name of the method to call
* @param {Array<any>=} array of arguments to pass to the method in question
* @param {number=} Number of ms to delay the call by - defaults to zero (so just pulls            execution of the method out of the current execution thread.
*/
delayCall(methodName:string, arrayArgs?:Array<any>, time?:number): string;

/**
*  Creates a component according to the "AutoChild" pattern, and adds it to this component.
*  
*  See the autoChildren to understand the general
*  purpose and usage of this method.
*  
*  addAutoChild() takes the following actions:
*  
*  
* - checks whether this.autoChildName is already populated, and returns it if so
*  
* - checks when there is a showAutoChildName with the value false, and if so
*  returns without creating a component 
*  
* - calls Class.createAutoChild to create the component
*  
* - sets this.autoChildName to the created component
*  
* - adds the component either to this component, or to some other parent, specified
*  by the "autoParent" property in the autoChild's defaults. The "autoParent" property may
*  be "none" indicating the autoChild should not be automatically added.
*  
*  
*  When adding an autoChild to a Layout subclass,
*  Layout.addMember will be called instead of the normal
*  Canvas.addChild. To prevent this behavior,
*  addAsChild:true can be set in the autoChild defaults. Similarly,
*  addAsPeer:true may be set to add an autoChild as a peer.
*  
*  Tip: because addAutoChild()
*  checks specifically for showAutoChildName:false, you do not have to add
*  showAutoChildName:true in order for an autoChild to be shown by default; leaving
*  the property undefined is sufficient.
*  
*  Note that by default the child created by this method will be destroyed when
*  Canvas.destroy is called on this instance. To disable this behavior,
*  set dontAutoDestroy to true on the auto child.
*  
* 
* @param {string} name of the autoChild
* @param {Properties} dynamic properties for the autoChild
*/
addAutoChild(childName:string, defaults:Properties): Class;

/**
*  Unconditionally creates and returns a component created according to the "AutoChild"
*  pattern.
*  
*  In addition to applying defaults and properties as described under the
*  autoChildUsage, the created autoChild:
*  
*  
* - is automatically autoDraw:false 
*  
* - has a creator property that points to this component, for easy
*  authoring of event handlers (eg click:"this.creator.doSomething()")
*  
*  
*  Unlike Class.addAutoChild, createAutoChild() does not create a
*  this.autoChildName reference to the component, check a showAutoChildName
*  flag, or automatically add the autoChild via Canvas.addChild. 
*  
*  General you use createAutoChild rather than addAutoChild when:
*  
*  
* - you are going to create several autoChildren with a common set of defaults (for
*  example the ColumnTree.column autoChild of the ColumnTree).
*  
* - children need to be created before their parents (eg, for layout/auto-sizing
*  reasons)
*  
*  
*  Note that by default the child created by this method will be destroyed when
*  Canvas.destroy is called on this instance. To disable this behavior,
*  set dontAutoDestroy to true on the auto child.
* 
* 
* @param {string} name of the autoChild
* @param {Properties} dynamic properties for the autoChild
*/
createAutoChild(childName:string, defaults:Properties): Class;

/**
* 
*  Call method on each item in argsList and return the Array of results.
* 
* 
* @param {string} Name of the method on this instance which should be called on each element of the Array
* @param {Array<any>} Array of items to call the method on
*/
map(methodName:string, items:Array<any>): Array<any>;

/**
* 
*  Call the SuperClass implementation of an instance method. For example:
*  
*   isc.defineClass("MyButton", "Button").addProperties({
*     // this override causes no change in behavior because it just 
*     // calls Super and returns whatever the superclass would return
*     getTitle : function () {
*       return this.Super("getTitle", arguments);
*     },
* 
*     // this override would add "foo" to the titles of all buttons
*     getTitle : function () {
*       // add code here to take actions before the superclass method is invoked
* 
*       var superResult = return this.Super("getTitle", arguments);
* 
*       // add code here to take action after the superclass method is invoked
* 
*       return superResult + "foo";
*     }
* 
*   })
*  
*  Note that Super() is always called with the name of the current method. You cannot call
*  the Super class implementation of another method directly.
*  
*  It is required to always pass the native 'arguments' object to Super(). Arguments
*  is a JavaScript builtin that is available within any JavaScript function - see any
*  JavaScript Reference for details.
*  
*  If you override a method in an instance, and then call Super(), the prototype
*  implementation will be called. This is similar to how anonymous classes in Java handle
*  super(). For example:
*  
*   isc.Button.create({
*     // this will set the title to indicate the runtime button class
*     initWidget : function () {
*       this.Super("initWidget", arguments);
*       this.title = "Parent Class: " + this.getSuperClass().getClassName();
*     },
*     width: 1,
*     overflow: "visible"
*   });
*  
*  See also ClassFactory.defineClass and
*  Class.addProperties for the basics of creating classes
*  and overriding methods.
* 
* 
* @param {string} name of the superclass method to call
* @param {any | Array<any>} native "arguments" object, or array of                     arguments to pass to the Super call
* @param {any=} native "arguments" object, required if an Array is                 passed for the "args" parameter in lieu of the native                  arguments object
*/
Super(methodName:string, args:any | Array<any>, nativeArgs?:any): any;

/**
*  Sets up the value of propertyName to be dynamically derived from the
*  Canvas.ruleScope, by either a simple DataPath into the ruleScope or via a textual
*  or numeric formula using the ruleScope as available formula inputs.
*  
*  The dataPath or formula is evaluated immediately when addDynamicProperty() is called, then re-evaluated
*  every time the ruleScope changes.
*  
*  It is invalid usage to use addDynamicProperty() on a property that is not runtime settable,
*  however, addDynamicProperty() will not throw an error or log a warning if this is done.
*  
*  If a property is already dynamic and addDynamicProperty() is called again, the new dynamic behavior
*  replaces the old. If a property should no longer be dynamic, call Class.clearDynamicProperty.
*  
*  Dynamic properties can also be declared together via Class.dynamicProperties.
* 
* 
* @param {Identifier} name of a settable property on this instance
* @param {DataPath | UserSummary | UserFormula} */
addDynamicProperty(propertyName:Identifier, source:DataPath | UserSummary | UserFormula): void;

/**
*  Clears a dynamic property previously established via Class.addDynamicProperty.
*  
*  If the property is not currently dynamic, nothing will be done (and no warning logged).
*  
*  The current value of the property will not be changed by this call.
* 
* 
* @param {Identifier} property name of the dynamic property to clear
*/
clearDynamicProperty(propertyName:Identifier): void;

/**
*  Returns true if the property is dynamic.
* 
* 
* @param {Identifier} name of a settable property on this instance
*/
hasDynamicProperty(propertyName:Identifier): boolean;

/**
*  Returns the last time the rule for the specified dynamic property fired, as a
*  Date.
* 
* 
* @param {Identifier} name of a settable property on this instance
*/
getDynamicPropertyRuleTime(propertyName:Identifier): Date;

/**
* 
*  The default toString() for instances reports that you have an instance of a class and prints
*  the instance ID if present.
* 
*/
toString(): void;

/**
*  Returns a one-line summary of the current method call, showing method name and passed
*  arguments. 
*  This function is available as a static on every ISC Class and as an instance
*  method on every instance of an ISC Class.
*  General best practice is to call the method as "this.getCallTrace(arguments)" whenever "this" is an
*  instance, or call the static classMethod on the Log class otherwise. 
*  
*  Note the arguments object is required in most cases for this method to function. In some
*  browsers, it can be derived automatically, but developers looking to debug on multiple platforms
*  should not rely on this.
* 
* 
* @param {any=} arguments object from the call to trace. If ommitted, where supported,  arguments will be derived from the calling function, or if this is not supported, the method  will not function.
*/
getCallTrace(args?:any): void;

/**
*  Returns a "stack trace" - one line per method in the current call stack, showing the method
*  name and any parameters passed.
*  This function is available as a static on every ISC Class and as an instance
*  method on every instance of an ISC Class.
*  General best practice is to call the method as "this.getStackTrace" whenever "this" is an
*  instance, or call the static classMethod on the Log class otherwise. 
*  
*  Platform Notes: In Mozilla Firefox, if Firebug is enabled, a stack trace will be logged
*  to the firebug console in addition to the standard stack trace string returned by
*  this method.
*  
*  In browsers other than Internet Explorer a complete stack trace may not be available - 
*  this occurs when a function is re-entrant (meaning it calls itself). In this case the
*  stack will terminate with text indicating where the recursive function call occurred.
*  
*  See debugging for further information information.
* 
* 
*/
getStackTrace(): string;

/**
*  Return a very short (generally less than 40 characters) string representation of any object,
*  suitable for viewing by a developer for debugging purposes.
*  This function is available as a static on every ISC Class and as an instance
*  method on every instance of an ISC Class.
*  General best practice is to call the method as "this.echoLeaf" whenever "this" is an
*  instance, or call the static classMethod on the Log class otherwise. 
* 
* 
* @param {any} object to echo
*/
echoLeaf(obj:any): string;

/**
*  Return a short string representation of any object, suitable for viewing by a developer for
*  debugging purposes.
*  
*  If passed an object containing other objects, echo will not recurse into subobjects,
*  summarizing them instead via echoLeaf().
*  
*  NOTE: echo() is used to generate the output shown in the Log window when evaluating an
*  expression.
*  
*  This function is available as a static on every ISC Class and as an instance
*  method on every instance of an ISC Class.
*  General best practice is to call the method as "this.echo()" whenever "this" is an
*  instance, or call the static classMethod on the Log class otherwise. 
*  
* 
* @param {any} object to echo
*/
echo(obj:any): string;

/**
*  Like echo(), except that if passed an Array, echoAll() will echo() every element of the
*  Array.
*  This function is available as a static on every ISC Class and as an instance
*  method on every instance of an ISC Class.
*  General best practice is to call the method as "this.echo()" whenever "this" is an
*  instance, or call the static classMethod on the Log class otherwise. 
* 
* 
* @param {any} object to echo
*/
echoAll(obj:any): string;

/**
*  Log a message at "debug" priority
*  
*  A method named logPriority exists for each priority level, on every ISC Class and
*  instance of an ISC Class. Messages logged on a Class or instance have a default
*  category of the classname. Messages logged on an instance will also automatically
*  incorporate the instance ID. General best practice is to call logDebug() et al as
*  "this.logDebug" whenever "this" is an instance, or as "Log.logDebug" otherwise.
* 
* 
* @param {string} message to log
* @param {string=} category to log in
*/
logDebug(message:string, category?:string): void;

/**
*  Log a message at "info" priority
*  
* 
* @param {string} message to log
* @param {string=} category to log in
*/
logInfo(message:string, category?:string): void;

/**
*  Log a message at "warn" priority
*  
* 
* @param {string} message to log
* @param {string=} category to log in
*/
logWarn(message:string, category?:string): void;

/**
*  Log a message at "error" priority
*  
* 
* @param {string} message to log
* @param {string=} category to log in
*/
logError(message:string, category?:string): void;

/**
*  Log a message at "fatal" priority
*  
* 
* @param {string} message to log
* @param {string=} category to log in
*/
logFatal(message:string, category?:string): void;

/**
*  Check whether a message logged at the given priority would be visible in the log.
*  
*  As with logDebug, category is defaulted to the current className. Use this method to avoid
*  putting together expensive log messages if they will never appear in the log.
* 
* 
* @param {LogPriority} priority level
* @param {string=} category to log in
*/
logIsEnabledFor(priority:LogPriority, category?:string): void;

/**
*  Check whether a message logged at "debug" priority would be visible in the log.
*  
*  As with logDebug, category is defaulted to the current className. Use this method to avoid
*  putting together expensive log messages if they will never appear in the log.
* 
* 
* @param {string=} category to log in
*/
logIsDebugEnabled(category?:string): void;

/**
*  Check whether a message logged at "info" priority would be visible in the log.
*  
*  As with logDebug, category is defaulted to the current className. Use this method to avoid
*  putting together expensive log messages if they will never appear in the log.
* 
* 
* @param {string=} category to log in
*/
logIsInfoEnabled(category?:string): void;

/**
*  Check whether a message logged at "warn" priority would be visible in the log.
*  
*  As with logDebug, category is defaulted to the current className. Use this method to avoid
*  putting together expensive log messages if they will never appear in the log.
* 
* 
* @param {string=} category to log in
*/
logIsWarnEnabled(category?:string): void;

/**
*  Check whether a message logged at "error" priority would be visible in the log.
*  
*  As with logDebug, category is defaulted to the current className. Use this method to avoid
*  putting together expensive log messages if they will never appear in the log.
* 
* 
* @param {string=} category to log in
*/
logIsErrorEnabled(category?:string): void;

/**
*  Set the priority of messages that will be visible for some log category, when logged on
*  this Class or Instance object.
*  If called with no category, this priority will be applied to every logged message on this
*  object
*  To set the visible log priority for some category across the entire page, use
*  isc.Log.setPriority() instead.
* 
* @param {string} Category for which the log priority will be updated. If not             all logs on this canvas will be logged at the priority passed in.
* @param {LogPriority} priority level
*/
setLogPriority(category:string, priority:LogPriority): void;

/**
*  Set the default priority of logging for messages logged on this Class or Instance object.
*  All categories for which there is no explicit, instance level logging priority set will
*  log at this level on this object. 
*  To set the default visible log priority across the entire page, use
*  isc.Log.setDefaultPriority() instead.
* 
* @param {string} Category for which the log priority will be updated. If not             all logs on this canvas will be logged at the priority passed in.
* @param {LogPriority} priority level
*/
setDefaultLogPriority(category:string, priority:LogPriority): void;

/**
*  Retrieves the default priority of messages for this class or instance.
* 
*/
getDefaultLogPriority(): LogPriority;

/**
*  Clear this object's priority setting for a particular category, so that the category's 
*  effective priority returns to the specified priority for this category at the Log level
*  (or Log.defaultPriority if not set).
*  To clear the Page-level priority setting for this log category use 
*  isc.Log.clearPriority() instead.
* 
* 
* @param {string} Category name. If not specified, all logging on this object               will revert to default priority settings.
*/
clearLogPriority(category:string): void;



}
/**
* Creates a new Class
* 
* @param typeCheckedProperties {Partial<Class>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Class} The new instance
*/
class Class{
static create(typeCheckedProperties? : Partial<Class>, uncheckedProperties?: Object): Class;
/**
* Is this a core SmartClient class (part of the SmartClient framework)?
*  This attribute may be used for debugging, and by the AutoTest subsystem to 
*  differentiate between SmartClient classes (part of the smartClient framework) and
*  subclasses created by specific applications
* @type {boolean}
* @default "varies"
*/
static isFrameworkClass?: boolean;





/**
*  Notifies the SmartClient Class system that any new classes created, or changes made 
*  to existing classes should be treated as part of the framework. This ensures that
*  Class.isFrameworkClass will be set to true on any classes defined after this 
*  method call, until Class.modifyFrameworkDone is called.
*  
*  Developers may call this method before applying changes which should be considered
*  part of the core framework, rather than application code, for example in load_skin.js 
*  files. When changes are complete, Class.modifyFrameworkDone should be called.
*  Note that this is an alternative approach to calling Class.markAsFrameworkClass 
*  directly on specific classes.
* 
* 
*/
static modifyFrameworkStart(): void;

/**
*  Notifies the SmartClient Class system that the developer is done making changes
*  to the SmartClient framework (as originally indicated by a call to 
*  Class.modifyFrameworkStart).
*  
*  New classes created or changes made to existing classes after this method call
*  will be considered application code. This ensures that 
*  Class.isFrameworkClass will not be set to true on Classes defined after
*  this method call.
* 
*/
static modifyFrameworkDone(): void;

/**
*  Replaces each of the methods named in methodNames with a new implementation
*  that simply logs a warning the first time the method is called, and nothing else. This can
*  be used to mark methods of derived classes which do not support certain parent class
*  methods as unsupported.
*  
*  The messageTemplate parameter is a template for the warning message logged
*  when the unsupported method is first called. The following variables in the template
*  are substituted as follows:
*  
*  
*   Variable
*   Substitution
*  
*  
*   $class
*   The Class.getClassName.
*  
*  
*   $method
*   The name of the method.
*  
*  
*  
*  If you want the literal string of a substitution variable to appear in the warning message,
*  you can escape it by prefixing with a dollar sign. For example, to include "$class" in the
*  warning message, use "$$class" in the template.
* 
* @param {string} template for the warning message logged when first called. If null, the default template string "$class does not support the $method() method." is used.
* @param {Array<Partial<Identifier>>} the method names to mark as unsupported.
*/
static markUnsupportedMethods(messageTemplate:string, methodNames:Array<Partial<Identifier>>): void;

/**
*  Returns true if the method is supported by this class, meaning that it is not null and
*  was not replaced by Class.markUnsupportedMethods.
* 
* @param {Identifier} the name of a method to test.
*/
static isMethodSupported(methodName:Identifier): boolean;

/**
* 
* 	Gets the name of this class as a string.
* 
* 
*/
static getClassName(): string;

/**
* 	
* 	Gets a pointer to the superClass' Class object.
* 
* 
*/
static getSuperClass(): Class;

/**
* 
*  Helper method for adding method definitions to all instances of this class.
* 
*  The added methods can be called as myInstance.method().
* 
*  Functionally equivalent to Class.addProperties, which works with both properties
*  and methods.
* 
* 
* @param {object=} objects with methods to add (think named parameters).                 all the methods of each argument will be applied                  as instance-level methods.
*/
static addMethods(...args:any[]): object;

/**
* 
* 	Register a method, or set of methods, that can be provided to instances of this class as
* 	Strings (containing a JavaScript expression) and will be automatically converted into
* 	functions.
*  
*  For example:
*  
*  isc.MyClass.registerStringMethods({
*    myStringMethod: "arg1, arg2"
*  });
*  
* 
* 
* @param {object} If this is a string, name of the property to register                 If this is an object, assume passing in a set of name/value                  pairs to register
* @param {string} named arguments for the property in a comma separated string                 (not used if methodName is an object)
*/
static registerStringMethods(methodName:object, argumentString:string): void;

/**
*  Evaluate a string of script and return the result.  
*  
*  This method is a wrapper around the native javascript method eval(). It
*  papers over some native issues to ensure evaluation of script behaves consistently across
*  browsers
* 
* 
* @param {string} the expression to be evaluated
* @param {object} Optional mapping of argument names to values - each key will   be available as a local variable when the script is executed.
*/
static evaluate(expression:string, evalArgs:object): any;

/**
* 
* 	Add static (Class-level) properties and methods to this object
* 
* 	These properties can then be accessed as MyClass.property, or for functions, called as
*  MyClass.methodName()
* 
* 
* @param {object=} objects with properties to add (think named parameters).     all the properties of each argument will be applied     	as class-level properties.
*/
static addClassProperties(...args:any[]): object;

/**
*  Mark this class as a framework class (member of the SmartClient framework).
*  Sets Class.isFrameworkClass. May be used in debugging and by the
*  AutoTest subsystem
* 
*/
static markAsFrameworkClass(): void;

/**
* 
* 	Add default properties and methods to all instances of this class.
* 
* 	These properties can then be accessed as myInstance.property, 
*  and methods can be called via myInstance.methodName()
* 
* 
* @param {object=} objects with properties to add (think named parameters).     all the properties of each argument will be applied
*/
static addProperties(...args:any[]): object;

/**
* 
* 	Add default properties to all instances of this class
* 
* 
* @param {Array<Partial<object>>} array of objects with properties to add
*/
static addPropertyList(list:Array<Partial<object>>): object;

/**
*  
*  Changes a set of defaults defined as a JavaScript Object. For these kind of properties,
*  simply calling Class.addProperties would replace the original Object
*  with yours, wiping out settings required for the basic functionality of the component.
*  This method instead applies your overrides over the existing properties, without
*  destroying non-overridden properties.
*  
*  For example let's say you have a component that's defined as follows
*  
*  isc.defineClass("MyComponent");
*  isc.MyComponent.addProperties({
*    simpleProperty: "some value",
*    propertyBlock : {
*     foo: "bar",
*     zoo: "moo"
*    }
*  }
*  
*  If you wanted to override simpleProperty, you can just call Class.addProperties
*  like this:
*  
*  isc.MyComponent.addProperties({
*    simpleProperty: "my override"
*  });
*  
*  If you want to override the value of propertyBlock.moo above,
*  but you don't want to clobber the value of propertyBlock.zoo. If you use
*  the above pattern like so:
*  
*  isc.MyComponent.addProperties({
*    propertyBlock: {
*      foo: "new value",
*      zoo: "moo"
*    }
*  });
*  
*  You need to re-specify the value of propertyBlock.zoo which you didn't want
*  to override. Failing to re-specify it would destroy the value.
*  
*  Instead of re-specifying the value, you can use this method to modify the value of
*  foo - like this:
*  
*  isc.MyComponent.changeDefaults("propertyBlock", {
*    foo: "new value"
*  });
*  
*  
*  See also the AutoChild system for information about standard sets of defaults
*  that are available for customization.
* 
* 
* @param {string} name of the property to change
* @param {object} overrides for defaults
*/
static changeDefaults(defaultsName:string, newDefaults:object): void;

/**
* 	Apply a set of properties to a class object, calling the appropriate setter class methods if
* 	any are found.
* 
* 
* @param {object=} objects with properties to add (think named parameters).     all the properties of each argument will be applied one after another     	so later properties will override
*/
static setProperties(...args:any[]): void;

/**
* 
*  Returns whether this class object is the provided class or is a subclass of the provided
*  class, or implements the provided interface.
* 
* 
* @param {string} Class name to test against
*/
static isA(className:string): boolean;

/**
* 
* 	Call the SuperClass implementation of a class method.
* 
* 
* @param {string} name of the superclass method to call
* @param {any | Array<any>} native "arguments" object, or array of                     arguments to pass to the Super call
* @param {any=} native "arguments" object, required if an Array is                 passed for the "args" parameter in lieu of the native                  arguments object
*/
static Super(methodName:string, args:any | Array<any>, nativeArgs?:any): any;

/**
* 
*  Call method on each item in argsList and return the Array of results.
* 
* 
* @param {string} Name of the method on this instance which should be called on each element of the Array
* @param {Array<any>} Array of items to call the method on
*/
static map(methodName:string, items:Array<any>): Array<any>;

/**
* 
* 	Gets a named property from the instance defaults for this object.
* 
* 
* @param {string} name of the property to return
*/
static getInstanceProperty(property:string): void;

/**
* 
* 	Sets a named property from the instance defaults for this object.
* 
* 
* @param {string} name of the property to return
* @param {any} value to set to
*/
static setInstanceProperty(property:string, value:any): void;

/**
* 
*  Fire some arbitrary action specified as a Callback.
*  Returns the value returned by the action.
*  
* 
* @param {Callback} Action to fire.
* @param {string=} Comma separated string of variable names. If the callback              passed in was a string of script, any arguments passed to the               callback will be available as local variables with these names.
* @param {Array<any>=} Array of arguments to pass to the method. Note that the number              of arguments should match the number of argNames.
* @param {object=} If specified the callback will be evaluated in the scope of this             object - the this keyword will be a pointer to this              target when the callback is fired.
*/
static fireCallback(callback:Callback, argNames?:string, args?:Array<any>, target?:object): any;

/**
*  This is a helper to delay a call to a method on some target by a specified
*  amount of time. Can be used to delay a call to a static method on this class by 
*  omitting the target parameter.
* 
* @param {string} name of the method to call
* @param {Array<any>=} array of arguments to pass to the method in question
* @param {number=} Number of ms to delay the call by - defaults to zero (so just pulls            execution of the method out of the current execution thread.
* @param {object=} Target to fire the method on - if unspecified assume this is             a call to a classMethod on this Class.
*/
static delayCall(methodName:string, arrayArgs?:Array<any>, time?:number, target?:object): string;

/**
* 
*  The default toString() for a ClassObject reports that you have a ClassObject and what class
*  it is.
* 
*/
static toString(): void;

/**
*  Returns a one-line summary of the current method call, showing method name and passed
*  arguments. 
*  This function is available as a static on every ISC Class and as an instance
*  method on every instance of an ISC Class.
*  General best practice is to call the method as "this.getCallTrace(arguments)" whenever "this" is an
*  instance, or call the static classMethod on the Log class otherwise. 
*  
*  Note the arguments object is required in most cases for this method to function. In some
*  browsers, it can be derived automatically, but developers looking to debug on multiple platforms
*  should not rely on this.
* 
* 
* @param {any=} arguments object from the call to trace. If ommitted, where supported,  arguments will be derived from the calling function, or if this is not supported, the method  will not function.
*/
static getCallTrace(args?:any): void;

/**
*  Returns a "stack trace" - one line per method in the current call stack, showing the method
*  name and any parameters passed.
*  This function is available as a static on every ISC Class and as an instance
*  method on every instance of an ISC Class.
*  General best practice is to call the method as "this.getStackTrace" whenever "this" is an
*  instance, or call the static classMethod on the Log class otherwise. 
*  
*  Platform Notes: In Mozilla Firefox, if Firebug is enabled, a stack trace will be logged
*  to the firebug console in addition to the standard stack trace string returned by
*  this method.
*  
*  In browsers other than Internet Explorer a complete stack trace may not be available - 
*  this occurs when a function is re-entrant (meaning it calls itself). In this case the
*  stack will terminate with text indicating where the recursive function call occurred.
*  
*  See debugging for further information information.
* 
* 
*/
static getStackTrace(): string;

/**
*  Return a very short (generally less than 40 characters) string representation of any object,
*  suitable for viewing by a developer for debugging purposes.
*  This function is available as a static on every ISC Class and as an instance
*  method on every instance of an ISC Class.
*  General best practice is to call the method as "this.echoLeaf" whenever "this" is an
*  instance, or call the static classMethod on the Log class otherwise. 
* 
* 
* @param {any} object to echo
*/
static echoLeaf(obj:any): string;

/**
*  Return a short string representation of any object, suitable for viewing by a developer for
*  debugging purposes.
*  
*  If passed an object containing other objects, echo will not recurse into subobjects,
*  summarizing them instead via echoLeaf().
*  
*  NOTE: echo() is used to generate the output shown in the Log window when evaluating an
*  expression.
*  
*  This function is available as a static on every ISC Class and as an instance
*  method on every instance of an ISC Class.
*  General best practice is to call the method as "this.echo()" whenever "this" is an
*  instance, or call the static classMethod on the Log class otherwise. 
*  
* 
* @param {any} object to echo
*/
static echo(obj:any): string;

/**
*  Like echo(), except that if passed an Array, echoAll() will echo() every element of the
*  Array.
*  This function is available as a static on every ISC Class and as an instance
*  method on every instance of an ISC Class.
*  General best practice is to call the method as "this.echo()" whenever "this" is an
*  instance, or call the static classMethod on the Log class otherwise. 
* 
* 
* @param {any} object to echo
*/
static echoAll(obj:any): string;

/**
*  Log a message at "debug" priority
*  
*  A method named logPriority exists for each priority level, on every ISC Class and
*  instance of an ISC Class. Messages logged on a Class or instance have a default
*  category of the classname. Messages logged on an instance will also automatically
*  incorporate the instance ID. General best practice is to call logDebug() et al as
*  "this.logDebug" whenever "this" is an instance, or as "Log.logDebug" otherwise.
* 
* 
* @param {string} message to log
* @param {string=} category to log in
*/
static logDebug(message:string, category?:string): void;

/**
*  Log a message at "info" priority
*  
* 
* @param {string} message to log
* @param {string=} category to log in
*/
static logInfo(message:string, category?:string): void;

/**
*  Log a message at "warn" priority
*  
* 
* @param {string} message to log
* @param {string=} category to log in
*/
static logWarn(message:string, category?:string): void;

/**
*  Log a message at "error" priority
*  
* 
* @param {string} message to log
* @param {string=} category to log in
*/
static logError(message:string, category?:string): void;

/**
*  Log a message at "fatal" priority
*  
* 
* @param {string} message to log
* @param {string=} category to log in
*/
static logFatal(message:string, category?:string): void;

/**
*  Check whether a message logged at the given priority would be visible in the log.
*  
*  As with logDebug, category is defaulted to the current className. Use this method to avoid
*  putting together expensive log messages if they will never appear in the log.
* 
* 
* @param {LogPriority} priority level
* @param {string=} category to log in
*/
static logIsEnabledFor(priority:LogPriority, category?:string): void;

/**
*  Check whether a message logged at "debug" priority would be visible in the log.
*  
*  As with logDebug, category is defaulted to the current className. Use this method to avoid
*  putting together expensive log messages if they will never appear in the log.
* 
* 
* @param {string=} category to log in
*/
static logIsDebugEnabled(category?:string): void;

/**
*  Check whether a message logged at "info" priority would be visible in the log.
*  
*  As with logDebug, category is defaulted to the current className. Use this method to avoid
*  putting together expensive log messages if they will never appear in the log.
* 
* 
* @param {string=} category to log in
*/
static logIsInfoEnabled(category?:string): void;

/**
*  Check whether a message logged at "warn" priority would be visible in the log.
*  
*  As with logDebug, category is defaulted to the current className. Use this method to avoid
*  putting together expensive log messages if they will never appear in the log.
* 
* 
* @param {string=} category to log in
*/
static logIsWarnEnabled(category?:string): void;

/**
*  Check whether a message logged at "error" priority would be visible in the log.
*  
*  As with logDebug, category is defaulted to the current className. Use this method to avoid
*  putting together expensive log messages if they will never appear in the log.
* 
* 
* @param {string=} category to log in
*/
static logIsErrorEnabled(category?:string): void;

/**
*  Set the priority of messages that will be visible for some log category, when logged on
*  this Class or Instance object.
*  If called with no category, this priority will be applied to every logged message on this
*  object
*  To set the visible log priority for some category across the entire page, use
*  isc.Log.setPriority() instead.
* 
* @param {string} Category for which the log priority will be updated. If not             all logs on this canvas will be logged at the priority passed in.
* @param {LogPriority} priority level
*/
static setLogPriority(category:string, priority:LogPriority): void;

/**
*  Set the default priority of logging for messages logged on this Class or Instance object.
*  All categories for which there is no explicit, instance level logging priority set will
*  log at this level on this object. 
*  To set the default visible log priority across the entire page, use
*  isc.Log.setDefaultPriority() instead.
* 
* @param {string} Category for which the log priority will be updated. If not             all logs on this canvas will be logged at the priority passed in.
* @param {LogPriority} priority level
*/
static setDefaultLogPriority(category:string, priority:LogPriority): void;

/**
*  Retrieves the default priority of messages for this class or instance.
* 
*/
static getDefaultLogPriority(): LogPriority;

/**
*  Clear this object's priority setting for a particular category, so that the category's 
*  effective priority returns to the specified priority for this category at the Log level
*  (or Log.defaultPriority if not set).
*  To clear the Page-level priority setting for this log category use 
*  isc.Log.clearPriority() instead.
* 
* 
* @param {string} Category name. If not specified, all logging on this object               will revert to default priority settings.
*/
static clearLogPriority(category:string): void;




}

/**
* Form item for selecting a color via a pop-up ColorPicker.
*/
interface ColorItem extends TextItem {
/**
* Should the pick button icon be shown for choosing colors from a ColorPicker
* @type {boolean}
* @default true
*/
showPickerIcon?: boolean;

/**
* If ColorItem.showPickerIcon is true for this item, this property governs the
*  size of the picker icon. If unset, the picker icon will be sized as a square to fit in the
*  available height for the icon.
*  
*  Note that if spriting is being used, and the image to be displayed is specified 
*  using css properties such as background-image, background-size,
*  changing this value may result in an unexpected appearance as the image will not
*  scale.
*  Scaleable spriting can be achieved using the SCSpriteConfig format.
*  See the section on spriting in the skinning for 
*  further information.
* @type {Integer}
* @default "18"
*/
pickerIconWidth?: Integer;

/**
* If ColorItem.showPickerIcon is true for this item, this property governs the
*  size of the picker icon. If unset, the picker icon will be sized as a square to fit in the
*  available height for the icon.
*  
*  Note that if spriting is being used, and the image to be displayed is specified 
*  using css properties such as background-image, background-size,
*  changing this value may result in an unexpected appearance as the image will not
*  scale.
*  Scaleable spriting can be achieved using the SCSpriteConfig format.
*  See the section on spriting in the skinning for 
*  further information.
* @type {Integer}
* @default "18"
*/
pickerIconHeight?: Integer;

/**
* If ColorItem.showPickerIcon is true for this item, this property governs the
*  FormItemIcon.src of the picker icon image to be displayed.
*  
*  skinning can be used for this image, by setting this property to
*  a SCSpriteConfig formatted string.
* @type {SCImgURL}
* @default "[SKIN]/DynamicForm/ColorPicker_icon.png"
*/
pickerIconSrc?: SCImgURL;

/**
* Prompt to show when the user hovers the mouse over the picker icon.
* @type {HTMLString}
* @default "Click to select a new color"
*/
pickerIconPrompt?: HTMLString;

/**
* The ColorPicker.defaultPickMode for the ColorPicker associated
*  with this ColorItem.
* @type {ColorPickerMode}
* @default "simple"
*/
defaultPickerMode?: ColorPickerMode;

/**
* Should "complex" mode be allowed for the ColorPicker window associated with 
*  this ColorItem?
*  If false, no "More" button is shown on the simple picker
* @type {boolean}
* @default true
*/
allowComplexMode?: boolean;

/**
* Determines whether the ColorPicker associated with this ColorItem allows the user 
*  to set transparency/opacity information whilst selecting a color. If false, no opacity
*  slider is shown and all colors are 100% opaque.
*  When an opacity value is selected, the HTML color string produced is 8 characters long
*  because the opacity setting is included.
*  You can also capture the the separate color and opacity information by overriding 
*  ColorItem.pickerColorSelected.
* @type {boolean}
* @default false
*/
supportsTransparency?: boolean;



/**
*  Store the color value selected by the user from the color picker. You will need to 
*  override this method if you wish to capture opacity information from the
*  ColorPicker.
* 
* @param {string} The selected color as a string.
* @param {Integer} The selected opacity, from 0 (transparent) to 100 (opaque),             or null if ColorItem.supportsTransparency is false or the              ColorPicker selected a color while in              ColorPickerMode.
*/
pickerColorSelected(color:string, opacity:Integer): void;



}
/**
* Creates a new ColorItem
* 
* @param typeCheckedProperties {Partial<ColorItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ColorItem} The new instance
*/
class ColorItem{
static create(typeCheckedProperties? : Partial<ColorItem>, uncheckedProperties?: Object): ColorItem;







}

/**
* The ColorPicker widget allows the user to select a color from anywhere in the 
*  color spectrum. It also supports selecting the alpha (opacity) value of the 
*  color. The picker supports a simple mode - which allows for one-click selection
*  from a standard palette of colors - and a complex mode which allow the user to
*  define any conceivable color. It is possible for the user to switch from simple
*  mode to complex by interacting with the widget. In general, the widget provides
*  very similar functionality to the color picker dialogs found in graphics packages
*  and other desktop software.
*/
interface ColorPicker extends Window {
/**
* "OK" button for the ColorPicker
* @type {IButton}
* @default null
*/
okButton?: IButton;

/**
* Should the OK button be visible. Set to false to hide the OK button.
* @type {boolean}
* @default true
*/
showOkButton?: boolean;

/**
* The class of the "OK" button. It is intended that you use either IButton or 
*    Button - other classes are unlikely to work correctly.
* @type {Class}
* @default "IButton"
*/
okButtonConstructor?: Class;

/**
* Cancel button for the ColorPicker
* @type {IButton}
* @default null
*/
cancelButton?: IButton;

/**
* Should the Cancel button be visible. Set to false to hide the Cancel button.
* @type {boolean}
* @default true
*/
showCancelButton?: boolean;

/**
* The class of the Cancel button. It is intended that you use either IButton or 
*    Button - other classes are unlikely to work correctly.
* @type {Class}
* @default "IButton"
*/
cancelButtonConstructor?: Class;

/**
* "More"/"Less" button for the ColorPicker
* @type {IButton}
* @default null
*/
modeToggleButton?: IButton;

/**
* Should the Mode Toggle button be visible. Set to false to hide the Mode Toggle button.
* @type {boolean}
* @default true
*/
showModeToggleButton?: boolean;

/**
* The class of the mode toggle button. It is intended that you use either IButton or 
*    Button - other classes are unlikely to work correctly.
* @type {Class}
* @default "IButton"
*/
modeToggleButtonConstructor?: Class;

/**
* The default color. This is the color that is selected when the picker first loads
* @type {string}
* @default "#808080"
*/
defaultColor?: string;

/**
* Width and height of the basic color boxes (they are always square, and they are
*  all the same size).
* @type {number}
* @default "20"
*/
colorButtonSize?: number;

/**
* Base CSS style applied to the basic color boxes
* @type {CSSStyleName}
* @default "ColorChooserCell"
*/
colorButtonBaseStyle?: CSSStyleName;

/**
* Array of 40 HTML color strings, used to render the basic color selection boxes.
* @type {Array<Partial<string>>}
* @default "[...]"
*/
colorArray?: Array<Partial<string>>;

/**
* Displayed width of the color swatch image. The default width is approximately
*  that used by the Windows&#174; XP color picking window
* @type {number}
* @default "170"
*/
swatchWidth?: number;

/**
* Displayed height of the color swatch image. The default height is approximately
*  that used by the Windows&#174; XP color picking window
* @type {number}
* @default "170"
*/
swatchHeight?: number;

/**
* Width of the Luminosity bar
* @type {number}
* @default "15"
*/
lumWidth?: number;

/**
* Determines whether to show the opacity slider. This allows the user to select colors
*  with an alpha element (ie, semi-transparent colors). If this attribute is set to false,
*  no opacity slider is shown, and all colors are completely opaque.
* @type {boolean}
* @default true
*/
supportsTransparency?: boolean;

/**
* The text to show underneath the selected color box, so that it can 
*  be seen through semi-transparent colors. If you do not want such text, set 
*  this value to blank. This value is irrelevant if 
*  ColorPicker.supportsTransparency is false.
* @type {HTMLString}
* @default "Lorem ipsum"
*/
opacityText?: HTMLString;

/**
* The location of the color swatch image file
* @type {SCImgURL}
* @default "[SKIN]ColorPicker/spectrum.png"
*/
swatchImageURL?: SCImgURL;

/**
* The location of the crosshair image file
* @type {SCImgURL}
* @default "[SKIN]ColorPicker/crosshair.png"
*/
crosshairImageURL?: SCImgURL;

/**
* The title for the button that switches to a less complex view.
* @type {HTMLString}
* @default "&lt;&lt; Less"
*/
lessButtonTitle?: HTMLString;

/**
* The title for the button that switches to a more complex view.
* @type {HTMLString}
* @default "More &gt;&gt;"
*/
moreButtonTitle?: HTMLString;

/**
* The label shown above the basic color blocks.
* @type {HTMLString}
* @default "Basic Colors:"
*/
basicColorLabel?: HTMLString;

/**
* The label shown next to the selected color box.
* @type {HTMLString}
* @default "Selected Color:"
*/
selectedColorLabel?: HTMLString;

/**
* The label shown next to the opacity slider. Ignored if 
*  ColorPicker.supportsTransparency is false.
* @type {HTMLString}
* @default "Opacity:"
*/
opacitySliderLabel?: HTMLString;

/**
* The initial opacity value for the component, as a percentage value between 0 and 100
* @type {number}
* @default "100"
*/
defaultOpacity?: number;

/**
* The title for the 'Red' field in the complex chooser.
* @type {HTMLString}
* @default "Red"
*/
redFieldTitle?: HTMLString;

/**
* The text to show when the mouse hovers over the 'Red' field in the complex chooser.
* @type {HTMLString}
* @default "The Red component of the selected color"
*/
redFieldPrompt?: HTMLString;

/**
* The title for the 'Green' field in the complex chooser.
* @type {HTMLString}
* @default "Green"
*/
greenFieldTitle?: HTMLString;

/**
* The text to show when the mouse hovers over the 'Green' field in the complex chooser.
* @type {HTMLString}
* @default "The Green component of the selected color"
*/
greenFieldPrompt?: HTMLString;

/**
* The title for the 'Blue' field in the complex chooser.
* @type {HTMLString}
* @default "Blue"
*/
blueFieldTitle?: HTMLString;

/**
* The text to show when the mouse hovers over the 'Blue' field in the complex chooser.
* @type {HTMLString}
* @default "The Blue component of the selected color"
*/
blueFieldPrompt?: HTMLString;

/**
* The title for the 'HTML' field in the complex chooser.
* @type {HTMLString}
* @default "HTML"
*/
htmlFieldTitle?: HTMLString;

/**
* The text to show when the mouse hovers over the 'HTML' field in the complex chooser.
* @type {HTMLString}
* @default "The selected color''s HTML coding"
*/
htmlFieldPrompt?: HTMLString;

/**
* The title for the 'Hue' field in the complex chooser.
* @type {HTMLString}
* @default "Hue"
*/
hueFieldTitle?: HTMLString;

/**
* The text to show when the mouse hovers over the 'Hue' field in the complex chooser.
* @type {HTMLString}
* @default "The Hue (base tone) of the selected color"
*/
hueFieldPrompt?: HTMLString;

/**
* The title for the 'Sat' field in the complex chooser.
* @type {HTMLString}
* @default "Sat"
*/
satFieldTitle?: HTMLString;

/**
* The text to show when the mouse hovers over the 'Saturation' field in the complex chooser.
* @type {HTMLString}
* @default "The Saturation (color purity) of the selected color"
*/
satFieldPrompt?: HTMLString;

/**
* The title for the 'Luminosity' field in the complex chooser.
* @type {HTMLString}
* @default "Lum"
*/
lumFieldTitle?: HTMLString;

/**
* The text to show when the mouse hovers over the 'Luminosity' field in the complex chooser.
* @type {HTMLString}
* @default "The Luminosity (brightness) of the selected color"
*/
lumFieldPrompt?: HTMLString;

/**
* The title for the 'OK' button.
* @type {HTMLString}
* @default "OK"
*/
okButtonTitle?: HTMLString;

/**
* The title for the 'Cancel' button.
* @type {HTMLString}
* @default "Cancel"
*/
cancelButtonTitle?: HTMLString;

/**
* If true, causes the ColorPicker to appear near where the mouse was last clicked.
*  If false, the ColorPicker is centered on first show; depending on the value of 
*  ColorPicker.autoCenterOnShow, it either reappears wherever it was last shown after hide/show(), 
*  or centered regardless of where it was last shown.
* @type {boolean}
* @default true
*/
autoPosition?: boolean;

/**
* If ColorPicker.autoPosition is false, this property controls whether to automatically center the
*  colorPicker every time it is redisplayed with the show() method.
* @type {boolean}
* @default true
*/
autoCenterOnShow?: boolean;

/**
* When this property is set to true, the ColorPicker will automatically hide when
*  a color has been selected using the swatch picker, even in "complex" mode. By default it will
*  only hide the ColorPicker in "simple" defaultPickMode.
*  
*  Set this property to false to disable the ColorPicker from automatically hiding,
*  this can be especially useful when for instance embedding this component inside another component.
* @type {boolean}
* @default null
*/
autoHide?: boolean;

/**
* The ColorPicker can operate in either a "simple" mode (where it displays just the
*  40 basic colors and allows the user to click one), or a "complex" mode (where the
*  user can specify a color from anywhere in the spectrum, with an optional alpha
*  element). The defaultPickMode attribute specifies which of these two modes is
*  in force when the picker first loads.
* @type {ColorPickerMode}
* @default "simple"
*/
defaultPickMode?: ColorPickerMode;

/**
* Should the "complex" mode be allowed for this ColorPicker?
*  If false, no "More" button is shown on the simple picker
* @type {boolean}
* @default true
*/
allowComplexMode?: boolean;



/**
*  Set the ColorPicker.supportsTransparency flag.
* 
* @param {boolean} Set to true to enable transparency/opacity
*/
setSupportsTransparency(transparencyFlag:boolean): void;

/**
*  Returns the Red element of the currently-selected color, as an integer from 0-255
* 
*/
getRed(): Integer;

/**
*  Returns the Green element of the currently-selected color, as an integer from 0-255
* 
*/
getGreen(): Integer;

/**
*  Returns the Blue element of the currently-selected color, as an integer from 0-255
* 
*/
getBlue(): Integer;

/**
*  Returns the Hue of the currently-selected color, as an integer from 0-239
* 
*/
getHue(): Integer;

/**
*  Returns the Saturation of the currently-selected color, as an integer from 0-240
* 
*/
getSaturation(): Integer;

/**
*  Returns the Luminosity (brightness) of the currently-selected color, as an 
*  integer from 0-240
* 
*/
getLuminosity(): Integer;

/**
*  Returns the currently-selected color, in HTML color representation form, as a string.
*  HTML color representation is a hash sign, followed by the red, green and blue elements
*  of the color in 2-digit hex form - for example "#F17F1D"
* 
*/
getHtmlColor(): string;

/**
*  Returns the opacity of the currently-selected color, as an integer from 0-100. If 
*  opacity is switched off, this is always 100.
* 
*/
getOpacity(): Integer;

/**
*  Sets the Red element of the selected color
* 
* @param {number} An integer between 0 and 255
*/
setRed(newValue:number): void;

/**
*  Sets the Green element of the selected color
* 
* @param {number} An integer between 0 and 255
*/
setGreen(newValue:number): void;

/**
*  Sets the Blue element of the selected color
* 
* @param {number} An integer between 0 and 255
*/
setBlue(newValue:number): void;

/**
*  Sets the Hue of the selected color
* 
* @param {number} An integer between 0 and 239
*/
setHue(newValue:number): void;

/**
*  Sets the Saturation of the selected color
* 
* @param {number} An integer between 0 and 240
*/
setSaturation(newValue:number): void;

/**
*  Sets the Luminosity (brightness) of the selected color
* 
* @param {number} An integer between 0 and 240
*/
setLuminosity(newValue:number): void;

/**
*  Changes the selected color to the one represented by the supplied HTML color 
*  string. Note that the method only accepts the parameter if it represents a 
*  valid color (otherwise it is simply ignored).
* 
* @param {string} A string in HTML color representation format (#RRGGBB)
*/
setHtmlColor(newValue:string): void;

/**
*  Sets the Opacity of the selected color. Ignored if opacity is switched off.
* 
* @param {number} An integer between 0 and 100
*/
setOpacity(newValue:number): void;

/**
*  Changes the pick mode of this ColorPicker to pickMode.
*  
*  Note: It is not allowed to set the pick mode to
*  "complex"
*  
*  if ColorPicker.allowComplexMode is false.
* 
* @param {ColorPickerMode} the new pick mode.
*/
setCurrentPickMode(pickMode:ColorPickerMode): void;

/**
*  Override this method to be kept informed when the ColorPicker changes in real-time 
*  (for example, if you need to update your own GUI accordingly). Then use the 
*  getXxxx() methods (for example, ColorPicker.getBlue or 
*  ColorPicker.getLuminosity)to obtain current state as required. 
* 
*/
colorChanged(): void;

/**
*  Override this method to be notified when the user selects a color
*  either by clicking a basic color box in simple mode, or by clicking 
*  the OK button in complex mode. It is not intended that client code 
*  call this method. The ColorPicker may automatically hide
*  itself after calling this method depending on ColorPicker.autoHide and
*  ColorPicker.defaultPickMode.
* 
* @param {string} The color selected, in HTML format.
* @param {Integer} The selected opacity, from 0 (transparent) to 100 (opaque),             or null if ColorPicker.supportsTransparency is false or the picker              selected a color while in ColorPickerMode.
*/
colorSelected(color:string, opacity:Integer): void;



}
/**
* Creates a new ColorPicker
* 
* @param typeCheckedProperties {Partial<ColorPicker>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ColorPicker} The new instance
*/
class ColorPicker{
static create(typeCheckedProperties? : Partial<ColorPicker>, uncheckedProperties?: Object): ColorPicker;




/**
*  Returns the shared global ColorPicker. 
*  Many applications will only need one ColorPicker instance; for such use 
*  cases, it is a good idea to use the shared object for performance reasons.
*  
*  The optional second parameter to this method indicates whether the shared picker
*  should retain the state (mode, color and opacity) it was in last time it was used, 
*  or revert to defaults.
*  Generally, you will want the picker to revert to default state; this gives the
*  same user experience as creating a new instance without incurring the overhead.
*  However, some use cases will benefit from the picker remembering what the user
*  did last time.
* 
* @param {object} Properties to apply to the global ColorPicker object
* @param {boolean=} Should we keep the current state?                  If false (or not provided), revert to default state
*/
static getSharedColorPicker(properties:object, keepCurrentState?:boolean): void;




}

/**
* Form item for selecting a color via a pop-up ColorPicker. This is an alias of
*  ColorItem.
*/
interface ColorPickerItem extends ColorItem {




}
/**
* Creates a new ColorPickerItem
* 
* @param typeCheckedProperties {Partial<ColorPickerItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ColorPickerItem} The new instance
*/
class ColorPickerItem{
static create(typeCheckedProperties? : Partial<ColorPickerItem>, uncheckedProperties?: Object): ColorPickerItem;







}

/**
* The SmartClient system supports hierarchical data (also referred to as tree data
*  due to its "branching" organization) with:
*  
*   
* - the Tree class, which manipulates hierarchical data sets
*   
* - the TreeGrid widget class, which extends the ListGrid class to visually
*     present tree data in an expandable/collapsible format.
*   
* - the ColumnTree widget class, which visually
*     presents tree data in a so-called 
*     "Miller Column" format.
*  
*  For information on DataBinding Trees, see treeDataBinding.
*  
*  A ColumnTree shows a single branch of the underlying Tree horizontally, from 
*  left to right. Thus, the leftmost column shows all the top-level nodes. When the user clicks
*  one of those nodes, a new column is shown immediately to the right of the top-level column,
*  showing the selected node's children. Additional columns are shown as required to present
*  lower-level children. The behavior of ColumnTree is similar to that of the Browser interface
*  in the Apple&trade; iTunes&trade; application.
*/
interface ColumnTree extends Omit<Layout, 'setDragTracker'>, Omit<DataBoundComponent, 'fields'>  {
/**
* The DataSource that this component should bind to for default fields and for performing
*  DSRequest.
*  
*  Can be specified as either a DataSource instance or the String ID of a DataSource.
* @type {DataSource | GlobalId}
* @default null
*/
dataSource?: DataSource | GlobalId;

/**
* A Tree object consisting of nested TreeNodes to 
*  display in this ColumnTree. 
*  The data property will typically not be explicitly specified for 
*  databound ColumnTrees, where the data is returned from the server via databound component
*  methods such as fetchData()
* @type {Tree}
* @default null
*/
data?: Tree;

/**
* For a ColumnTree that uses a DataSource, these properties will be passed to
*  the automatically-created ResultTree. This can be used for various customizations such as
*  modifying the automatically-chosen Tree.parentIdField.
* @type {Tree}
* @default null
*/
dataProperties?: Tree;

/**
* If true, when this component is first drawn, automatically call this.fetchData().
*  Criteria for this fetch may be picked up from ColumnTree.initialCriteria, and textMatchStyle may
*  be specified via ListGrid.autoFetchTextMatchStyle.
*  
*  NOTE: if autoFetchData is set, calling
*  ListGrid.fetchData before draw will cause two requests to be issued, one from the manual
*  call to fetchData() and one from the autoFetchData setting. The second request will use
*  only ColumnTree.initialCriteria and not any other criteria or settings from the first request.
*  Generally, turn off autoFetchData if you are going to manually call ListGrid.fetchData at any time.
* @type {boolean}
* @default false
*/
autoFetchData?: boolean;

/**
* If ColumnTree.autoFetchData is true, this attribute allows the developer to
*  specify a textMatchStyle for the initial ListGrid.fetchData call.
* @type {TextMatchStyle}
* @default null
*/
autoFetchTextMatchStyle?: TextMatchStyle;

/**
* Criteria to be used when ColumnTree.autoFetchData is set.
*  
*  This property supports dynamicCriteria - use Criterion.valuePath
*  to refer to values in the Canvas.ruleScope.
* @type {Criteria}
* @default null
*/
initialCriteria?: Criteria;

/**
* Criteria that are never shown to or edited by the user and are cumulative with any criteria
*  provided via DataBoundComponent.initialCriteria and related methods
* @type {Criteria}
* @default null
*/
implicitCriteria?: Criteria;

/**
* Whether to show fields marked detail:true when a DataBoundComponent is 
*  given a DataSource but no component.fields.
*  
*  The detail property is used on DataSource fields to mark fields that 
*  shouldn't appear by default in a view that tries to show many records in a small space.
* @type {boolean}
* @default true
*/
showDetailFields?: boolean;

/**
* How to fetch and manage records retrieve from the server. See FetchMode.
*  
*  This setting only applies to the ResultSet automatically created by calling
*  ListGrid.fetchData. If a pre-existing ResultSet is passed to setData() instead, it's
*  existing setting for ResultSet.fetchMode applies.
* @type {FetchMode}
* @default "paged"
*/
dataFetchMode?: FetchMode;

/**
* Enables fixed columns mode. All columns are created in advance instead of as navigation occurs.
* @type {Integer}
* @default null
*/
fixedColumns?: Integer;

/**
* With ColumnTree.fixedColumns enabled, defines the pixel or % width per column.
* @type {Array<Partial<string>>}
* @default null
*/
columnWidths?: Array<Partial<string>>;

/**
* With ColumnTree.fixedColumns enabled, defines the header title for each column.
* @type {Array<Partial<string>>}
* @default null
*/
columnTitles?: Array<Partial<string>>;

/**
* With ColumnTree.fixedColumns enabled, defines each column's ListGrid.emptyMessage.
* @type {Array<Partial<string>>}
* @default null
*/
emptyColumnMessages?: Array<Partial<string>>;

/**
* The URL of the base icon for all folder nodes in this columnTree. Note that this URL will
*  have ColumnTree.openIconSuffix or ColumnTree.closedIconSuffix appended to 
*  indicate state changes if appropriate - see documentation on ColumnTree.showOpenIcons
* @type {SCImgURL}
* @default "[SKIN]folder.png"
*/
folderIcon?: SCImgURL;

/**
* This property allows the developer to customize the icon displayed next to a node.
*  Set node[grid.customIconProperty] to the URL of the desired icon to display and
*  it will be shown instead of the standard ColumnTree.nodeIcon for this node.
*  Note that if ColumnTree.showCustomIconOpen 
*  is true for this grid, customized icons for folder nodes will be appended with the 
*  ColumnTree.openIconSuffix suffix on state change,
*  as with the standard ColumnTree.folderIcon. Also note that for
*  custom folder icons, the ColumnTree.closedIconSuffix will never be appended.
* @type {string}
* @default "icon"
*/
customIconProperty?: string;

/**
* The filename of the default icon for all leaf nodes in this grid. To specify a 
*  custom image for an individual node, set the ColumnTree.customIconProperty directly on
*  the node.
*  
*  See TreeGrid.showNodeIcons and TreeGrid.showFolderIcons for details on suppressing 
*  display of icons
* @type {SCImgURL}
* @default "[SKIN]file.png"
*/
nodeIcon?: SCImgURL;

/**
* If ColumnTree.showOpenIcons is true, this suffix will be appended to the
*  ColumnTree.folderIcon for open folders in this grid.
* @type {string}
* @default "open"
*/
openIconSuffix?: string;

/**
* This suffix will be appended to the ColumnTree.folderIcon for closed folders.
*  If ColumnTree.showOpenIcons is set to false this suffix will also be
*  appended to open folders' icons.
* @type {string}
* @default "closed"
*/
closedIconSuffix?: string;

/**
* If true, show a different icon for open folders than closed folders.
*  This is achieved by appending the ColumnTree.openIconSuffix onto the 
*  ColumnTree.folderIcon URL [for example "[SKIN]/folder.gif" might be 
*  replaced by "[SKIN]/folder_open.gif".
*  Note If this property is set to false the same icon is shown for
*  open folders as for closed folders, unless a custom folder icon was specified. This will be
*  determined by ColumnTree.folderIcon plus the ColumnTree.closedIconSuffix.
* @type {boolean}
* @default true
*/
showOpenIcons?: boolean;

/**
* Should folder nodes showing custom icons (set via the ColumnTree.customIconProperty),
*  show open state images when the folder is opened.
*  If true, the ColumnTree.openIconSuffix will be appended to the image URL
*  (so "customFolder.gif" might be replaced with 
*  "customFolder_open.gif").
*  Note that the ColumnTree.closedIconSuffix is never appended to custom folder icons.
*  Can be overridden at the node level via the default property TreeNode.showOpenIcon
*  and that property can be renamed via TreeGrid.customIconOpenProperty.
* @type {boolean}
* @default false
*/
showCustomIconOpen?: boolean;

/**
* This property allows the developer to rename the 
*  TreeNode.showOpenIcon property.
* @type {string}
* @default "showOpenIcon"
*/
customIconOpenProperty?: string;

/**
* When set to false, only displays a single column at a time, showing a slide animation 
*  when moving between columns.
* @type {boolean}
* @default null
*/
showMultipleColumns?: boolean;

/**
* For databound columnTree instances, should the entire tree of data be loaded on initial 
*  fetch, or should each column be loaded as needed. If unset the default 
*  ResultTree.loadDataOnDemand setting will be used.
* @type {boolean}
* @default null
*/
loadDataOnDemand?: boolean;

/**
* Instance of ListGrid used to display each column of the tree.
* @type {ListGrid}
* @default null
*/
column?: ListGrid;

/**
* Standard set of properties to apply to each generated ColumnTree.column
*  in this columnTree. Developers may also override ColumnTree.getColumnProperties
*  to return dynamic properties based on the node being displayed.
* @type {Partial<ListGrid>}
* @default null
*/
columnProperties?: Partial<ListGrid>;

/**
* An array of field objects, specifying the order, layout, dynamic calculation, and
*  sorting behavior of each field in each column in the columnTree object. In ColumnTrees, 
*  the fields array specifies sub-columns within each main column. 
*  Each field in the fields array is a ListGridField object.
*  
*  If ColumnTree.dataSource is also set, this value acts as a set of overrides as
*  explained in DataBoundComponent.fields.
* @type {Array<Partial<ListGridField>>}
* @default null
*/
fields?: Array<Partial<ListGridField>>;

/**
* If set, each column in the ColumnTree will show a header with the title of the selected
*  node from the column to the left.
* @type {boolean}
* @default false
*/
showHeaders?: boolean;

/**
* A title for the leftmost column if ColumnTree.showHeaders is set (the remaining columns
*  have their titles derived from the item selected in the column to the left). Ignored
*  if ColumnTree.showHeaders is not set.
*  Note: if you do not want a heading for the first column leave this attribute at its 
*  default value of " ". If you set it to null or the empty string, SmartClient will
*  fall back to displaying the field's name in the heading.
* @type {string}
* @default " "
*/
firstColumnTitle?: string;

/**
* If set, and ColumnTree.showHeaders is also set, each column's header will show 
*  a count of the number of nodes in that column
* @type {boolean}
* @default false
*/
showNodeCount?: boolean;

/**
* When using ColumnTree.showMultipleColumns, this i18n property
*  dictates the title for the ColumnTree.backButton docked to the top left 
*  which allows navigation back through the column tree.
* @type {string}
* @default "Back"
*/
backButtonTitle?: string;

/**
* When using ColumnTree.showMultipleColumns, this is the
*  "Back" button that you see hovering above the column UI and that allows backward
*  navigation.
* @type {IButton}
* @default null
*/
backButton?: IButton;

/**
* Indicates whether records can be dragged from this listGrid and dropped elsewhere.
*  
*  NOTE: If canDragRecordsOut is initially enabled or might be
*  dynamically enabled after the grid is created, it may be desirable to disable
*  Canvas.useTouchScrolling so that touch-dragging a record starts
*  a drag operation rather than a scroll, but see the discussion of 
*  ListGrid.showDragHandles. If 
*  Canvas.disableTouchScrollingForDrag is set to true, then touch
*  scrolling will be disabled automatically. However, for
*  accessibility reasons, it is recommended to leave touch
*  scrolling enabled and provide an alternative set of controls that can be used to perform
*  drag and drop of records out of the grid.
* @type {boolean}
* @default false
*/
canDragRecordsOut?: boolean;



/**
*  Only supported on ListGrid for now.
* 
* @param {Array<Partial<Hilite>>} Array of hilite objects
*/
setHilites(hilites:Array<Partial<Hilite>>): void;

/**
*  Uses a "fetch" operation on the current DataSource to retrieve data
*  that matches the provided criteria, and displays the matching data in this component as a
*  tree.
*  
*  This method will create a ResultTree to manage tree data, which will
*  subsequently be available as columnTree.data. DataSource records
*  returned by the "fetch" operation are linked into a tree structure according to
*  DataSourceField.primaryKey and
*  DataSourceField.foreignKey declarations on DataSource fields. See the
*  treeDataBinding topic for complete details.
*  
*  By default, the created ResultTree will use folder-by-folder load on demand, asking the
*  server for the children of each folder as the user opens it.
*  
*  The ResultTree created by fetchData() can be customized by setting
*  ColumnTree.dataProperties to an Object containing properties and methods to apply to
*  the created ResultTree. For example, the property that determines whether a node is a
*  folder (Tree.isFolderProperty) can be customized, or
*  level-by-level loading can be disabled via
*  ResultTree.loadDataOnDemand.
*  
*  The callback passed to fetchData will fire once, the first time that data is
*  loaded from the server. If using folder-by-folder load on demand, use the
*  ResultTree.dataArrived notification to be notified each time new nodes are loaded.
*  
*  Note that, if criteria are passed to fetchData(), they will be passed every
*  time a new "fetch" operation is sent to the server. This allows you to retrieve multiple
*  different tree structures from the same DataSource. However note that the server is expected
*  to always respond with an intact tree - returned nodes which do not have parents are dropped
*  from the dataset and not displayed.
* 
* 
* @param {Criteria=} Search criteria. If a DynamicForm is passed                     in as this argument instead of a raw criteria                      object, will be derived by calling                      DynamicForm.getValuesAsCriteria
* @param {DSCallback=} callback to invoke when a fetch is complete. Fires                     only if server contact was required
* @param {DSRequest=} additional properties to set on the DSRequest                     that will be issued
*/
fetchData(criteria?:Criteria, callback?:DSCallback, requestProperties?:DSRequest): void;

/**
*  Retrieves data that matches the provided criteria and displays the matching data in this
*  component.
*  
*  This method behaves exactly like ListGrid.fetchData except that
*  DSRequest.textMatchStyle is automatically set to "substring" so that String-valued
*  fields are matched by case-insensitive substring comparison.
* 
* 
* @param {Criteria=} Search criteria.            If a DynamicForm is passed in as this argument            instead of a raw criteria object, will be derived by calling            DynamicForm.getValuesAsCriteria
* @param {DSCallback=} callback to invoke when a fetch is complete. Fires                     only if server contact was required; see                      ListGrid.fetchData for details
* @param {DSRequest=} for databound components only - optional              additional properties to set on the DSRequest that will be issued
*/
filterData(criteria?:Criteria, callback?:DSCallback, requestProperties?:DSRequest): void;

/**
*  Get the selected record, that is, the parent of the nodes in the rightmost visible column.
*  
*  This is generally the most recently clicked node unless programmatic navigation has taken
*  place.
*  
*  If only the first column is showing, the root node is returned (which can be detected via
*  Tree.isRoot).
* 
* 
*/
getSelectedRecord(): Record;

/**
*  Navigate to the previous column.
* 
* 
*/
navigateBack(): void;

/**
*  Called when a node is selected in any column. Default behavior is to show the next level
*  of the tree in a column to the right of the current column.
*  
*  The new column will be created if it is not already showing. Any columns further to the
*  right, showing deeper levels of the tree, will be removed.
* 
* @param {ListGrid} the column where a node was selected
* @param {TreeNode} the node that was selected
*/
nodeSelected(column:ListGrid, node:TreeNode): boolean;

/**
*  Whether the indicated column should show a header. Returns this.showHeaders by default,
*  override for different behavior.
*  
* 
* @param {TreeNode} parent node for the nodes to be shown in the column
* @param {Integer} index of the column
*/
shouldShowHeader(node:TreeNode, colNum:Integer): void;

/**
*  Returns the title to show for the header of indicated column. Only called if
*  ColumnTree.shouldShowHeader returns true for this column.
*  
*  By default, returns ColumnTree.firstColumnTitle for the first column, and for subsequent
*  columns, the result of Tree.getTitle called on the
*  node passed to this function.
*  
* 
* @param {TreeNode} parent node for the nodes to be shown in the column
* @param {Integer} index of the column
*/
getColumnTitle(node:TreeNode, colNum:Integer): void;

/**
*  Retrieve a record by index. 
*  
*  If colNum is passed, returns the record found in that column at that index,
*  or null if the column doesn't exist or the index is too high.
*  
*  With no colNum parameter, a record's index is it's position counting from the
*  first record of the first column and including all records in each column. Note that both
*  index and colNum are zero-based - so the first column is column 0, not column 1.
*  
* 
* @param {Integer} index of record to return.
* @param {Integer=} optional index of the column
*/
getRecord(index:Integer, colNum?:Integer): TreeNode;

/**
* 
*  Returns the title to show for a node in the ColumnTree. The default implementation
*  returns the result of calling Tree.getTitle on the node.
*  
*  You can override this method to return a custom title for nodes in the tree.
* 
* 
* @param {TreeNode} The node for which the title is being requested.
* @param {number} The index of the node.
* @param {string} The field for which the title is being requested.
*/
getNodeTitle(node:TreeNode, recordNum:number, field:string): HTMLString;

/**
*  Returns the Tree object this ColumnTree is viewing
* 
* 
*/
getData(): void;

/**
*  Set the Tree object this ColumnTree will view
* 
* 
* @param {Tree} Tree to show
*/
setData(newData:Tree): void;

/**
*  Advanced API - get the ListGrid representing the indicated column.
* 
* @param {Integer | TreeNode} column number, or parent node of the nodes shown in the column
*/
getColumn(column:Integer | TreeNode): ListGrid;

/**
*  Additional properties to apply to the ListGrid that will show the indicated column.
*  Returns nothing by default. This method can be overridden to allow, for example, different 
*  styling, icons, or row heights per column.
* 
* @param {TreeNode} parent node for the nodes to be shown in the column
* @param {Integer} index of the column
*/
getColumnProperties(node:TreeNode, colNum:Integer): Partial<ListGrid>;

/**
* 
*  Process a drop of one or more nodes on a TreeGrid folder.
*  
*  
*  This method can be overridden to provide custom drop behaviors and is a more appropriate
*  override point than the lower level Canvas.drop handler.
*  
*  
*  
*  The default behavior is to simply delegate to the ColumnTree.transferNodes method; thus, the 
*  correct way to perform a programmatic folder drop, with all the built-in behaviors described
*  below, is to call transferNodes()
*  
*  If this is a self-drop, nodes are simply reordered. An "update" operation will
*  be submitted to update the Tree.parentIdField field of the moved node(s). 
*  
*  For a drop from another widget, TreeGrid.transferDragData is called which,
*  depending on the TreeGrid.dragDataAction specified on the source
*  widget, may either remove the source nodes from the original list (dragDataAction:"move")
*  or just provide a copy to this tree (dragDataAction:"copy").
*  
*  In either case the new row(s) appear in the folder at the index
*  specified by the arguments of the same name.
*  
*  If this grid is databound, the new nodes will be added to the dataset by calling
*  DataSource.addData. Further, if the new nodes were dragged from another
*  databound component, and DataBoundComponent.addDropValues
*  is true, DataBoundComponent.getDropValues will be called for every item
*  being dropped.
*  
*  As a special case, if the sourceWidget is also databound and a
*  DataSourceField.foreignKey relationship is declared from the
*  sourceWidget's DataSource to this TreeGrid's DataSource, the interaction will
*  be treated as a "drag recategorization" use case such as files being placed in folders,
*  employees being assigned to teams, etc. "update" DSRequests will be submitted that
*  change the foreignKey field in the dropped records to point to the tree folder that was the
*  target of the drop. In this case no change will be made to the Tree data as such, only to
*  the dropped records. 
*  
*  For multi-record drops, Queuing is automatically used to combine all DSRequests into a
*  single HTTP Request (see QuickStart Guide, Server Framework chapter). This allows the
*  server to persist all changes caused by the drop in a single transaction (and this is
*  automatically done when using the built-in server DataSources with Power Edition and
*  above).
*  
*  If these default persistence behaviors are undesirable, 
*  return false to cancel them
*  , then implement your own behavior, 
*  typically by using grid.updateData() or addData() to add new records.
*  NOTE: the records you receive in this event are the actual Records from the source
*  component. Use DataSource.copyRecords to create a copy before modifying the records
*  or using them with updateData() or addData().
* 
* 
* @param {Array<Partial<TreeNode>>} List of nodes being dropped
* @param {TreeNode} The folder being dropped on
* @param {Integer} Within the folder being dropped on, the index at which the drop is          occurring. Only passed if TreeGrid.canReorderRecords is true.
* @param {Canvas} The component that is the source of the nodes (where the nodes               were dragged from)
*/
folderDrop(nodes:Array<Partial<TreeNode>>, folder:TreeNode, index:Integer, sourceWidget:Canvas): void;

/**
* 
*  Transfer a list of TreeNodes within this treeGrid or from from some other component 
*  (does not have to be a databound component) into this component. This method is only 
*  applicable to list-type components, such as ListGrid, 
*  TreeGrid or TileGrid. Please see the paragraph below for
*  special rules concerning Tree.isMultiLinkTree.
*  
*  This method implements the automatic drag-copy and drag-move behavior and calling it is
*  equivalent to completing a drag and drop of the nodes (the default 
*  ColumnTree.folderDrop implementation simply calls transferNodes())
*  
*  Note that this method is asynchronous - it may need to perform server turnarounds to prevent
*  duplicates in the target component's data. If you wish to be notified when the transfer
*  process has completed, you can either pass the optional callback to this method or implement
*  the DataBoundComponent.dropComplete method on this component.
*  
*  For a TreeGrid, see also TreeGrid.transferSelectedData.
*  
*  Multi-link trees
*  If both the target treeGrid and the sourceWidget for this transfer are 
*  multi-link treeGrids, the nodes parameter must be an array of 
*  NodeLocators rather than TreeNodes. Likewise, if the target (this) component
*  is a multi-link treeGrid, the folder parameter must be a NodeLocator.
*  
*  You can obtain a NodeLocator for a visible node occurence in a multi-link TreeGrid by 
*  calling its data model's Tree.getNodeLocator method.
* 
* 
* @param {Array<Partial<TreeNode>> | Array<Partial<NodeLocator>>} Nodes to transfer to this component
* @param {TreeNode} The target folder (eg, of a drop interaction), for context
* @param {Integer} Insert point within the target folder data for the transferred nodes
* @param {Canvas} The databound or non-databound component from which the nodes               are to be transferred.
* @param {Callback=} optional callback to be fired when the transfer process has            completed. The callback will be passed a single parameter "records",            the list of nodes actually transferred to this component (it is called            "records" because this is logic shared with ListGrid)
*/
transferNodes(nodes:Array<Partial<TreeNode>> | Array<Partial<NodeLocator>>, folder:TreeNode, index:Integer, sourceWidget:Canvas, callback?:Callback): void;

/**
* 
*  Select/deselect a Record passed in explicitly, or by index.
* 
* 
* @param {Record | number} record (or row number) to select
* @param {boolean=} new selection state (if null, defaults to true)
* @param {number=} Column number
*/
selectRecord(record:Record | number, newState?:boolean, colNum?:number): void;

/**
* 
*  Deselect a Record passed in explicitly, or by index.
*  
*  Synonym for selectRecord(record, false)
* 
* 
* @param {Record | number} record (or row number) to deselect
* @param {number=} Column number
*/
deselectRecord(record:Record | number, colNum?:number): void;

/**
* 
*  Select/deselect a list of Records passed in explicitly, or by index.
* 
* 
* @param {Array<Partial<Record>> | number} records (or row numbers) to select
* @param {boolean=} new selection state (if null, defaults to true)
* @param {number=} Column number
*/
selectRecords(records:Array<Partial<Record>> | number, newState?:boolean, colNum?:number): void;

/**
* 
*  Deselect a list of Records passed in explicitly, or by index.
*  
*  Synonym for selectRecords(records, false)
* 
* 
* @param {Array<Partial<Record>> | number} records (or row numbers) to deselect
* @param {number=} Column number
*/
deselectRecords(records:Array<Partial<Record>> | number, colNum?:number): void;

/**
*  Select all records in the supplied column (the first column if none is passed)
* 
* 
* @param {number=} Column number
*/
selectAllRecords(colNum?:number): void;

/**
*  Deselect all records in the supplied column (the first column if none is passed)
* 
* 
* @param {number=} Column number
*/
deselectAllRecords(colNum?:number): void;

/**
*  Whether at least one item is selected in the supplied column (the first column if none is passed)
* 
* 
* @param {number=} Column number
*/
anySelected(colNum?:number): boolean;



}
/**
* Creates a new ColumnTree
* 
* @param typeCheckedProperties {Partial<ColumnTree>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ColumnTree} The new instance
*/
class ColumnTree{
static create(typeCheckedProperties? : Partial<ColumnTree>, uncheckedProperties?: Object): ColumnTree;







}

/**
* The Combobox is a text input field which can show a list of options via a drop-down
*  PickList.
*  
*  The set of options will be filtered based on the current value in the text field, so only
*  options that match what has been typed so far will be displayed.
*  The set of options can be derived from a ValueMap or dynamically retrieved from a
*  dataSource. See the PickList interface for further settings.
*  
*  The two most common use cases for ComboBoxItems are:
*  
* - With ComboBoxItem.addUnknownValues set to true, the ComboBoxItem acts as a
*  freeform text entry field with the picklist providing essentially a set of suggested completions
*  similar to a URL bar in a web browser.
*    
* - With ComboBoxItem.addUnknownValues set to false, the ComboBoxItem acts similarly
*  to a SelectItem where a fixed set of options is available to the user and the text entry field
*  is essentially used to filter which of these options are visible
*  
*  Other commonly used settings to configure ComboBoxItem behavior are:
*  - ComboBoxItem.defaultToFirstOption - this will select the first option from the pickList
*  as a default value for the item - and
*  - ComboBoxItem.completeOnTab which causes the
*  current selection in the pickList (if there is one) to be chosen when the user tabs out of the
*  field, allowing a user to type a few characters and hit tab to auto-complete to the first matched
*  option. completeOnTab is automatically set to true if ComboBoxItem.addUnknownValues is 
*  false.
*  
*  ComboBoxItem does not provide built-in support for multiple selection. For a Combobox
*  that does provide such a multiple-select feature use MultiComboBoxItem.
*/
interface ComboBoxItem extends TextItem, PickList {
/**
* Default class used to construct the EditProxy for this component
*  when the component is Canvas.setEditMode.
* @type {SCClassName}
* @default "SelectItemEditProxy"
*/
editProxyConstructor?: SCClassName;

/**
* Static default value for this ComboBoxItem. To default to the first option use
*  ComboBoxItem.defaultToFirstOption instead.
* @type {any}
* @default null
*/
defaultValue?: any;

/**
* Select the first option as the default value for this ComboBoxItem. If options are derived
*  from a dataSource, the first value returned by the server will be used, otherwise the first
*  value in the valueMap. If enabled, this setting overrides ComboBoxItem.defaultValue
*  and ComboBoxItem.defaultDynamicValue.
* @type {boolean}
* @default false
*/
defaultToFirstOption?: boolean;

/**
* If showing a hint for this form item, should it be shown within the field?
*  CSS style for the hint is SelectItem.textBoxStyle with the suffix
*  "Hint" appended to it.
* @type {boolean}
* @default null
*/
showHintInField?: boolean;

/**
* Not applicable to a ComboBoxItem.
* @type {string}
* @default null
*/
mask?: string;

/**
* Not applicable to a ComboBoxItem.
* @type {boolean}
* @default null
*/
maskSaveLiterals?: boolean;

/**
* Not applicable to a ComboBoxItem.
* @type {string}
* @default " "
*/
maskPadChar?: string;

/**
* Not applicable to a ComboBoxItem.
* @type {string}
* @default "_"
*/
maskPromptChar?: string;

/**
* Not applicable to a ComboBoxItem.
* @type {boolean}
* @default null
*/
maskOverwriteMode?: boolean;

/**
* Controls where the PickList is placed. 
*  Can be specified as a PanelPlacement
*  or a specific widget that should be filled (by specifying an actual Canvas or
*  Canvas.ID).
*  
*  Default behavior is to "fillPanel" if Browser.isHandset or
*  Browser.isTablet, to better accomodate the smaller screen real estate and 
*  less precise
*  pointing ability on such devices.
*  
*  When filling the whole screen, part of the screen or a specific panel, the expanded
*  interface is created as a FormItem.picker, and 
*  incorporates a ComboBoxItem.pickerNavigationBar and 
*  ComboBoxItem.pickerExitButton that hides the expanded interface, as well 
*  as a separate ComboBoxItem.pickerSearchField.
* @type {PanelPlacement | Canvas | string}
* @default null
*/
pickListPlacement?: PanelPlacement | Canvas | string;

/**
* For PickList items with PickListItemIconPlacement set such that the pickList does
*  not render near-origin, should specified FormItem.icons be rendered inline within
*  the formItem itself, or within the ComboBoxItem.pickerNavigationBar.
*  
*  May be overridden at the icon level via FormItemIcon.iconPlacement.
*  
*  For mobile browsing with limited available screen space, icons rendered in
*  the navigation bar may be easier for the user to interact with.
* @type {PickListItemIconPlacement}
* @default "both"
*/
iconPlacement?: PickListItemIconPlacement;

/**
* The pickerSearchField is a separate TextItem created for 
*  search string entry when ComboBoxItem.pickListPlacement indicates that the search 
*  interface takes over an entire panel or the entire screen.
*  
*  The following autoChildUsage apply:
*  
*  
* - ComboBoxItem.pickerSearchFieldHint for FormItem.hint
*  
* @type {TextItem}
* @default null
*/
pickerSearchField?: TextItem;

/**
* FormItem.hint for the ComboBoxItem.pickerSearchField.
* @type {HTMLString}
* @default "Search"
*/
pickerSearchFieldHint?: HTMLString;

/**
* FormItem.hint for the ComboBoxItem.pickerSearchField when the combobox is configured to 
*  ComboBoxItem.addUnknownValues
* @type {HTMLString}
* @default "Search or enter new value"
*/
pickerSearchOrNewValueFieldHint?: HTMLString;

/**
* Form that contains the ComboBoxItem.pickerSearchField.
* @type {DynamicForm}
* @default null
*/
pickerSearchForm?: DynamicForm;

/**
* NavigationButton to dismiss the picker interface, created when 
*  ComboBoxItem.pickListPlacement indicates that the search interface takes over the
*  entire panel or screen.
*  
*  The following autoChildUsage apply:
*  
*  
* - ComboBoxItem.pickerExitButtonTitle for Button.title
*  
* @type {NavigationButton}
* @default null
*/
pickerExitButton?: NavigationButton;

/**
* The title for the ComboBoxItem.pickerExitButton.
* @type {HTMLString}
* @default "Cancel"
*/
pickerExitButtonTitle?: HTMLString;

/**
* "Accept" button for ComboBoxItem.addUnknownValues ComboBoxItems
*  showing the mobile interface.
*  
*  The pickerSaveButton is an automatically created NavigationButton autoChild
*  to dismiss the picker interface and store out the
*  value entered in the ComboBoxItem.pickerSearchField, created when 
*  ComboBoxItem.pickListPlacement indicates that the search interface takes over the
*  entire panel or screen.
*  
*  This button will only be shown when ComboBoxItem.addUnknownValues is true.
*  Note that if a user has entered a partial known value, the pickList
*  will show a filtered list of possible matches. An "Enter" keypress (or native keyboard
*  "Done" button click on a mobile browser keyboard) will select the first match from the 
*  list. The pickerSaveButton provides a way for users to explicitly use
*  the value as entered instead.
*  
*  The following autoChildUsage apply:
*  
*  
* - ComboBoxItem.pickerSaveButtonTitle for Button.title
*  
* @type {NavigationButton}
* @default null
*/
pickerSaveButton?: NavigationButton;

/**
* The title for the ComboBoxItem.pickerSaveButton.
* @type {HTMLString}
* @default "Accept"
*/
pickerSaveButtonTitle?: HTMLString;

/**
* NavigationButton to clear the picker value, created when ComboBoxItem.pickListPlacement
*  indicates that the search interface takes over the entire panel or screen.
*  
*  This button will only be shown if ComboBoxItem.addUnknownValues or ComboBoxItem.allowEmptyValue
*  is true.
*  
*  The following autoChildUsage apply:
*  
*  
* - ComboBoxItem.pickerClearButtonTitle for Button.title
*  
* @type {NavigationButton}
* @default null
*/
pickerClearButton?: NavigationButton;

/**
* The title for the ComboBoxItem.pickerClearButton.
* @type {HTMLString}
* @default "Clear"
*/
pickerClearButtonTitle?: HTMLString;

/**
* Empty message to display in the comboboxItem if PickList.hideEmptyPickList
*  is false.
* @type {string}
* @default "No items to show"
*/
emptyPickListMessage?: string;

/**
* NavigationBar created when ComboBoxItem.pickListPlacement indicates that the search
*  interface takes over the entire panel or screen.
* @type {NavigationBar}
* @default null
*/
pickerNavigationBar?: NavigationBar;

/**
* Indicates whether or not this ComboBoxItem will load its list of options 
*  DataSource.progressiveLoading. This property is copied onto the
*  underlying PickList.
* @type {boolean}
* @default true
*/
progressiveLoading?: boolean;

/**
* Should we show a special 'picker' FormItemIcon for this form item? Picker
*  icons are customizable via FormItem.pickerIconProperties. By default
*  they will be rendered inside the form item's FormItem.controlStyle
*  area. By default clicking the pickerIcon will call FormItem.showPicker.
* @type {boolean}
* @default true
*/
showPickerIcon?: boolean;

/**
* If ComboBoxItem.showPickerIcon is true for this item, this property governs the
*  size of the picker icon. If unset, the picker icon will be sized as a square to fit in the
*  available height for the icon.
*  
*  Note that if spriting is being used, and the image to be displayed is specified 
*  using css properties such as background-image, background-size,
*  changing this value may result in an unexpected appearance as the image will not
*  scale.
*  Scaleable spriting can be achieved using the SCSpriteConfig format.
*  See the section on spriting in the skinning for 
*  further information.
* @type {Integer}
* @default "15"
*/
pickerIconWidth?: Integer;

/**
* Don't specify an explicit height for the picker icon - instead have it size to match the
*  height of the combo box item.
* @type {Integer}
* @default null
*/
pickerIconHeight?: Integer;

/**
* If ComboBoxItem.showPickerIcon is true for this item, this property governs the
*  FormItemIcon.src of the picker icon image to be displayed.
*  
*  When skinning is enabled, this property will not 
*  be used to locate an image, instead, the image is drawn via CSS based on the 
*  FormItem.pickerIconStyle property.
* @type {SCImgURL}
* @default "[SKIN]/DynamicForm/ComboBoxItem_PickButton_icon.gif"
*/
pickerIconSrc?: SCImgURL;

/**
* Should the list of options be displayed whenever the user types into the 
*  combo-box textArea, or only when the user clicks on the pick button or uses the 
*  explicit Alt+Arrow Down key combination?
* @type {boolean}
* @default true
*/
showPickListOnKeypress?: boolean;

/**
* ComboBox items will submit their containing form on enter keypress 
*  if DynamicForm.saveOnEnter is true. Setting this property to
*  false will disable this behavior.
*  
*  Note that if the drop down list of options (pickList) is visible an
*  Enter keypress is used to select a value from the available set of
*  options and will not automatically cause form submission.
* @type {boolean}
* @default true
*/
saveOnEnter?: boolean;

/**
* If true, when the pickList is showing, the user can select the current value by hitting
*  the Tab key.
*  
*  Note that completeOnTab is not compatible with ComboBoxItem.formatOnBlur
* @type {boolean}
* @default null
*/
completeOnTab?: boolean;

/**
* If true, when the pickList is showing, the user can select the current value by hitting
*  the Enter key.
*  
*  If not explicitly set, completeOnEnter will default to false for items embedded
*  in a SearchForm, true otherwise.
* @type {boolean}
* @default null
*/
completeOnEnter?: boolean;

/**
* With formatOnBlur enabled, this comboBoxItem will format its value
*  according to the rules described in FormItem.mapValueToDisplay as long as the 
*  item does not have focus. Once the user puts focus into the item
*  the formatter will be removed. This provides a simple way for developers to
*  show a nicely formatted display value in a freeform text field, without the need
*  for an explicit FormItem.formatEditorValue 
*  and FormItem.parseEditorValue pair.
*  
*  Note that this attribute is not compatible with ComboBoxItem.completeOnTab
* @type {boolean}
* @default false
*/
formatOnBlur?: boolean;

/**
* A set of "special" values such as "All", "None" or "Invalid" that do not appear in the normal
*  ValueMap or in the data returned by the ComboBoxItem.optionDataSource.
*  
*  Like other uses of ValueMap, either a list of values or a mapping from stored to
*  display value can be provided.
*  
*  These values can either be shown at the top of the list of values (in the order specified), or
*  can be shown in a separate, non-scrolling region - the setting
*  SelectItem.separateSpecialValues controls this. Note that data 
*  paging can only be used if separateSpecialValues is enabled.
*  
*  If specialValues are configured, SelectItem.allowEmptyValue is ignored - an empty
*  value, if desired, must be included in the specialValues. To provide a
*  specialValue which clears the value of the field, use the special constant
*  PickList.emptyStoredValue.
*  
*  specialValues can also be used to take a value that does appear in the
*  normal data and redundantly display it at the top of the list to make it more accessible. Note
*  that in this case it is expected that the special value appears both at the top of the
*  list and in it's normal position in the list, so this works best with
*  SelectItem.separateSpecialValues mode enabled.
*  
*  Also, if an ComboBoxItem.optionDataSource is used, ComboBoxItem.specialValues that appear in the normal
*  dataset will be updated by automatic cacheSync (if 
*  the ComboBoxItem.displayField is updated). However when using a distinct ComboBoxItem.valueField and
*  ComboBoxItem.displayField, you are required to provide ComboBoxItem.specialValues as a map (there is no
*  support for FormItem.fetchMissingValues automatically fetching appropriate display values).
*  
*  Note that specialValues are not supported in conjunction with MultiComboBoxItem. Whereas with
*  SelectItem.multiple, specialValues will never be normal values 
*  that may be selected. So, specialValues should have options such as "Select All", "Select None" and others.
* @type {ValueMap}
* @default null
*/
specialValues?: ValueMap;

/**
* If true, ComboBoxItem.specialValues special values such as the empty value will be shown in a
*  separate non-scrolling area, in the ComboBoxItem.separateValuesList. Aside from making these values
*  more easily accessible, showing them in a separate list allows data paging to be used, which is
*  disabled if the separateValues are shown in the normal drop-down list along with other values.
* @type {boolean}
* @default null
*/
separateSpecialValues?: boolean;

/**
* AutoChild used to show ComboBoxItem.specialValues.
* @type {ListGrid}
* @default null
*/
separateValuesList?: ListGrid;

/**
* This property controls whether the user can enter a value that is not present in the
*  set of options for this item.
*  
*  If set to false, the value the user enters in the text box is essentially used to filter the 
*  set of options displayed in the pickList.
*  
*  In this mode, when focus is taken from the field, if the entered value does not match
*  any entries in the ValueMap or ComboBoxItem.optionDataSource, it will be discarded. Note 
*  that in this mode, ComboBoxItem.completeOnTab behavior is automatically enabled
*  so if the user enters a valid partial value such that one or more options is displayed in 
*  the pickList, and hits the Tab key, the first matching option will be chosen automatically.
*  In this mode the user may also hit the "Escape" key to discard their edits.
*  
*  Note also that when addUnknownValues is set to false, the underlying value
*  returned by FormItem.getValue will not be updated until a value is explicitly chosen. This
*  means any change or changed handlers will not fire directly in response to the user typing
*  in the field - they will fire when the user actually selects a value, or takes focus from 
*  the field.
*  
*  If this property is set to true, the user is not limited to entering values present in
*  the set of options for the item. Instead the set of options essentially become a set of
*  suggestions that may be used, or the user can enter an entirely new value.
* @type {boolean}
* @default true
*/
addUnknownValues?: boolean;

/**
* If ComboBoxItem.addUnknownValues is false, this property 
*  determines whether
*  the user can clear the comboBoxItem value, or whether they are constrained to
*  choosing one of the available options (in which case clearing the text box will
*  simply revert to the last picked value when the user leaves the field).
*  
*  See also ComboBoxItem.specialValues as a way of providing several different special values in
*  addition to an empty value, such as "Invalid". Note that setting
*  specialValues disables the use of allowEmptyValue - see
*  details of how to have an empty value while using specialValues in 
*  in ComboBoxItem.specialValues.
* @type {boolean}
* @default true
*/
allowEmptyValue?: boolean;

/**
* Optional "pending" style for this item's text box.
*  
*  If ComboBoxItem.addUnknownValues is false, when the user modifies the
*  value displayed in the combobox item text box, the underlying data value (as returned
*  from item.getValue()) is not immediately updated - instead the value is used to filter
*  the set of results displayed in the comboBoxItem pickList.
*  
*  While the comboBoxItem is in this pending state (where the result of getEnteredValue() will
*  not necessarily match the display value for whatever is returned by getValue()), the 
*  pendingTextBoxStyle may be applied to the text box for the item.
*  
*  When the element value is updated to display the actual value for the item (typically due
*  to the user selecting a value from the pickList), the standard 
*  TextItem.textBoxStyle will be reapplied.
*  
*  May be left unset in which case the standard text box style is always applied.
*  Has no effect if ComboBoxItem.addUnknownValues is true.
* @type {CSSStyleName}
* @default null
*/
pendingTextBoxStyle?: CSSStyleName;

/**
* The Class to use when creating a picker of PickList.dataSetType for 
*  a FormItem. Must be a subclass of the builtin default, 
*  PickListMenu.
* @type {SCClassName}
* @default "PickListMenu"
*/
pickListConstructor?: SCClassName;

/**
* The Class to use when creating a picker of PickList.dataSetType for 
*  a FormItem. Must be a subclass of the builtin default, 
*  PickTreeMenu.
* @type {SCClassName}
* @default "PickTreeMenu"
*/
pickTreeConstructor?: SCClassName;

/**
* Whether to show the picker as a flat list, or a collapsible tree.
*  
*  The default value, "list", will use an instance of the
*  PickList.pickListConstructor as the picker - "tree" will 
*  show an instance of PickList.pickTreeConstructor.
* @type {string}
* @default "list"
*/
dataSetType?: string;

/**
* If set, this FormItem will derive data to show in the PickList by fetching records from
*  the specified optionDataSource. The fetched data will be used as a
*  FormItem.valueMap by extracting the
*  FormItem.valueField and FormItem.displayField in
*  the loaded records, to derive one valueMap entry per record loaded from the
*  optionDataSource. Multiple fields from the fetched data may be shown in the pickList by
*  setting ComboBoxItem.pickListFields.
*  
*  The data will be retrieved via a "fetch" operation on the DataSource, passing the 
*  PickList.pickListCriteria (if set) as criteria, and passing
*  ComboBoxItem.optionFilterContext (if set) as DSRequest Properties.
*  
*  The fetch will be triggered when the pickList is first shown, or, you can set
*  SelectItem.autoFetchData to fetch when the FormItem is
*  first drawn. 
*  
*  Note that providing an initial value when
*  FormItem.fetchMissingValues is enabled, or enabling
*  SelectItem.defaultToFirstOption, can also cause a fetch to
*  be initiated immediately upon form creation. You can also call PickList.fetchData
*  at any time to manually trigger a fetch.
*  
*  Data paging is automatically enabled if the optionDataSource supports it. As the
*  pickList is scrolled by the user, requests for additional data will be automatically
*  issued.
*  
*  For a pickList attached to a ComboBoxItem, new fetches are 
*  issued as the user types, with criteria set as described under 
*  ComboBoxItem.getPickListFilterCriteria.
*  If your dataSource is not capable of filtering results by search criteria (eg, the
*  dataSource is backed by an XML flat file), you can set ComboBoxItem.filterLocally to have the
*  entire dataset loaded up front and filtering performed in the browser. This disables
*  data paging.
*  
*  Note that if a normal, static FormItem.valueMap is also specified
*  for the field (either directly in the form item or as part of the field definition in
*  the dataSource), it will be preferred to the data derived from the optionDataSource for
*  whatever mappings are present.
* @type {DataSource | string}
* @default null
*/
optionDataSource?: DataSource | string;

/**
* This property allows the developer to specify which field[s] will be displayed in the 
*  drop down list of options.
*  
*  Only applies to databound pickLists (see PickList.optionDataSource, or pickLists
*  with custom data set up via the advanced PickList.getClientPickListData method.
*  
*  If this property is unset, we display the PickList.displayField, if specified, 
*  otherwise the PickList.valueField.
*  
*  If there are multiple fields, column headers will be shown for each field, the
*  height of which can be customized via the PickList.pickListHeaderHeight attribute.
*  
*  Each field to display should be specified as a ListGridField object. Note that
*  unlike in ListGrid, dataSource fields marked as 
*  DataSourceField.hidden will be hidden by default in pickLists. To
*  override this behavior, ensure that you specify an explicit value for 
*  ListGridField.showIf.
* @type {Array<Partial<ListGridField>>}
* @default null
*/
pickListFields?: Array<Partial<ListGridField>>;

/**
* For databound pickLists (see PickList.optionDataSource), by default SmartClient
*  will cache and re-use datasets shown by different pickLists displayed by different
*  SelectItems in an LRU (least recently used) caching pattern.
*  
*  Setting this flag to false avoids this caching for situations where it is too
*  aggressive.
*  
*  Note that this does not control re-use of data within a single pickList. To
*  control when client-side filtering is used in ComboBoxItem, see
*  ComboBoxItem.useClientFiltering and ComboBoxItem.filterLocally.
* @type {boolean}
* @default true
*/
cachePickListResults?: boolean;

/**
* If this item has a specified optionDataSource and this property is
*  true, the list of fields used by this pickList will be passed to
*  the datasource as DSRequest.outputs. If the datasource supports this feature
*  the returned fields will be limited to this list. A custom datasource will need to
*  add code to implement field limiting.
*  
*  This list of used fields consists of the values of FormItem.valueField,
*  FormItem.displayField and ComboBoxItem.pickListFields.
*  
*  NOTE: When enabled, FormItem.getSelectedRecord will only include the
*  fetched fields.
* @type {boolean}
* @default null
*/
fetchDisplayedFieldsOnly?: boolean;

/**
* If this item has a specified optionDataSource, and this property is not
*  null, this will be passed to the datasource as DSRequest properties when
*  performing the filter operation on the dataSource to obtain the set of options for the
*  list. This provides, among other capabilities, a way to trigger the server to return
*  summary records.
* @type {Partial<DSRequest>}
* @default null
*/
optionFilterContext?: Partial<DSRequest>;

/**
* If this item has a specified optionDataSource, this attribute may be set
*  to specify an explicit DSRequest.operationId when performing a fetch against the
*  option dataSource to pick up display value mapping.
* @type {string}
* @default null
*/
optionOperationId?: string;

/**
* If set, this item will display a value from another field to the user instead of
*  showing the underlying data value for the FormItem.name.
*  
*  This property is used in two ways:
*  
*  The item will display the displayField value from the 
*  DynamicForm.getValues if 
*  FormItem.useLocalDisplayFieldValue is true, (or if unset and the conditions
*  outlined in the documentation for that property are met).
*  
*  If this field has an FormItem.optionDataSource, this property is used by
*  default to identify which value to use as a display value in records from this
*  related dataSource. In this usage the specified displayField must be 
*  explicitly defined in the optionDataSource to be used - see 
*  ComboBoxItem.getDisplayFieldName for more on this behavior.
*  If not using FormItem.useLocalDisplayFieldValue, the display
*  value for this item will be derived by performing a fetch against the 
*  FormItem.getOptionDataSource 
*  to find a record where the FormItem.getValueFieldName matches 
*  this item's value, and use the displayField value from that record.
*  In addition to this, PickList-based form items that provide a list of possible
*  options such as the SelectItem or ComboBoxItem will show the 
*  displayField values to the user by default, allowing them to choose
*  a new data value (see FormItem.valueField) from a list of user-friendly
*  display values.
*  
*  This essentially allows the specified optionDataSource to be used as
*  a server based valueMap.
*  
*  If FormItem.useLocalDisplayFieldValue
*  are being used and FormItem.storeDisplayValues is true, selecting a new value
*  will update both the value for this field and the associated display-field value
*  on the record being edited.
*  
*  Note: Developers may specify the FormItem.foreignDisplayField property 
*  in addition to displayField. This is useful for cases where the
*  display field name in the local dataSource differs from the display field name in
*  the optionDataSource. See the documentation for
*  DataSourceField.foreignDisplayField for more on this.
*  If a foreignDisplayField is specified, as with just displayField, if 
*  FormItem.useLocalDisplayFieldValue
*  are being used and FormItem.storeDisplayValues is true, when the user 
*  chooses a value the associated display-field value 
*  on the record being edited will be updated. In this case it would be set to the 
*  foreignDisplayField value from the related record. This means foreignDisplayField 
*  is always expected to be set to the equivalent field in the related dataSources.
*  Developers looking to display some other arbitrary field(s) from the
*  related dataSource during editing should consider using custom
*  PickList.pickListFields instead of setting a foreignDisplayField. 
*  
*  Note that if optionDataSource is set and no valid display field is
*  specified,
*  FormItem.getDisplayFieldName will return the dataSource title 
*  field by default.
*  
*  If a displayField is specified for a freeform text based item (such as a 
*  ComboBoxItem), any user-entered value will be treated as a display value.
*  In this scenario, items will derive the data value for the item from the
*  first record where the displayField value matches the user-entered value.
*  To avoid ambiguity, developers may wish to avoid this usage if display values
*  are not unique.
* @type {string}
* @default null
*/
displayField?: string;

/**
* If this form item maps data values to display values by retrieving the 
*  FormItem.displayField values from an 
*  FormItem.optionDataSource, this property 
*  denotes the the field to use as the underlying data value in records from the 
*  optionDataSource.
*  If not explicitly supplied, the valueField name will be derived as
*  described in FormItem.getValueFieldName.
* @type {string}
* @default null
*/
valueField?: string;

/**
* If filterLocally is set for this item, and this item is showing options 
*  from a dataSource, fetch the entire set of options from the server, and use these values
*  to map the item value to the appropriate display value. Also use "local"
*  type filtering on drop down list of options.
*  
*  This means data will only be fetched once from the server, and then filtered on the
*  client.
*  
*  Note - when this property is set to false, filtering will still be 
*  performed on the client if a complete set of data for some criteria has been cached
*  by a fetch, and a subsequent fetch has more restrictive criteria. To explicitly
*  disable client-side filtering set the ComboBoxItem.useClientFiltering property to false.
* @type {boolean}
* @default false
*/
filterLocally?: boolean;

/**
* If this item has a databound pickList (for example PickList.optionDataSource is
*  set), this property can be used to provide static filter criteria when retrieving the data
*  for the pickList.
* @type {Criteria}
* @default null
*/
pickListCriteria?: Criteria;

/**
* If this item is part of a databound form, and has a specified valueMap,
*  by default we show the valueMap options in the pickList for the item.
*  Setting this property to true will ensure that the options displayed in our pickList
*  are derived from the form's dataSource.
* @type {boolean}
* @default null
*/
showOptionsFromDataSource?: boolean;

/**
* If specified this properties block will be applied to the PickListMenu
*  created for this FormItem.
*  
*  Note: Not every ListGrid property is supported when assigned to a pickList.
*  Where there is a dedicated API on the form item (such as 
*  PickList.pickListCellHeight), we recommend that be used in 
*  favor of setting the equivalent property on the pickListProperties block.
*  
*  PickLists and ListGrid.showFilterEditor:
*  ComboBoxItem do not support setting showFilterEditor 
*  to true on pickListProperties. This combination of settings leads to an ambiguous user
*  experience as the two sets of filter-criteria (those from the text-box and those from the 
*  pickList filter editor) interact with each other.
*  Setting ListGrid.showFilterEditor in 
*  SelectItem.pickListProperties is a valid way to create a filterable pickList, on 
*  a SelectItem, but this setting is not supported on a SelectItem with 
*  SelectItem.multiple set to true - this combination of settings can cause a 
*  selected value to be filtered out of view at which point further selection changes will 
*  discard that value.
*  In general we recommend the ComboBoxItem class (with ComboBoxItem.addUnknownValues 
*  set as appropriate) as a better interface for filtering pickList data.
* @type {Partial<ListGrid>}
* @default null
*/
pickListProperties?: Partial<ListGrid>;

/**
* Specifies one or more fields by which this item should be initially sorted. It can be 
*  a ListGridField.name, or an array of field names - but note that, if
*  multiple fields are supplied, then each will be sorted in the same 
*  ListGrid.sortDirection.
*  
*  For full sorting control, set PickList.initialSort to a list of
*  custom SortSpecifier.
*  
*  This attribute can also be set to the index of a field in the fields array, but note 
*  that it will be converted to a string (field name) after initialization.
* @type {string | Array<Partial<string>> | Integer}
* @default null
*/
sortField?: string | Array<Partial<string>> | Integer;

/**
* An array of SortSpecifier objects used to set up the initial sort configuration 
*  for this pickList. If specified, this will be used instead of any 
*  PickList.sortField specified.
* @type {Array<Partial<SortSpecifier>>}
* @default null
*/
initialSort?: Array<Partial<SortSpecifier>>;

/**
* When this item is showing a PickList.dataSetType, this is 
*  the SelectItem.valueField of the record to use as the 
*  Tree.rootValue node.
* @type {string | number}
* @default null
*/
rootNodeId?: string | number;

/**
* When this item is showing a PickList.dataSetType, which 
*  nodes should be opened automatically. Options are:
*  
*  
* - "none" - no nodes are opened automatically
*  
* - "root" - opens the PickList.rootNodeId - in databound 
*        tree-pickers, this node is always hidden
*  
* - "all" - when ResultTree.loadDataOnDemand, opens the
*        PickList.rootNodeId and all of it's direct
*        descendants - otherwise, opens all loaded nodes 
*  
* @type {string}
* @default "none"
*/
autoOpenTree?: string;

/**
* When this item is showing a PickList.dataSetType, the
*  default behavior is for folder open-state to be toggled by double-clicking. Set this 
*  attribute to true to toggle folders on a single-click instead.
*  
*  Note: when set to true, users can only choose leaf-nodes, since clicking folders would
*  simply toggle them.
* @type {boolean}
* @default false
*/
singleClickFolderToggle?: boolean;

/**
* If this combo box retrieves its options from a dataSource, should options
*  be fetched from the server when the item is first written out, or should this fetch be
*  delayed until the user opens the pickList.
* @type {boolean}
* @default false
*/
autoFetchData?: boolean;

/**
* Read-only property set by the ComboBoxItem to indicate whether we should use the 
*  current typed-in value as part of the filter criteria returned by 
*  ComboBoxItem.getPickListFilterCriteria. You can check this flag in order to
*  mimic the ComboBoxItem's default behavior if you provide a custom implementation of
*  getPickListFilterCriteria().
* @type {boolean}
* @default "varies"
*/
filterWithValue?: boolean;

/**
* As the user types into this item's textBox, a comboBoxItem will show the
*  pick-list of options, and filter the set of results displayed by the current value in the
*  text box. For a databound comboBoxItem, by default the entered value is filtered against
*  the ComboBoxItem.displayField if one is specified, otherwise the
*  ComboBoxItem.valueField.
*  
*  This attribute allows the developer to explicitly change which fields to filter against,
*  causing the user-entered text to be matched against any of the specified set of fields from 
*  the ComboBoxItem.optionDataSource.
*  
*  This essentially causes ComboBoxItem.getPickListFilterCriteria to return an AdvancedCriteria
*  object representing "field1 starts with value or field2 starts with value or ...". The
*  OperatorId used is controlled by TextMatchStyle as usual, that is,
*  "startsWith" implies the operator "iStartsWith, "substring" implies "iContains" and "exact"
*  implies "iEquals".
*  
*  The most common use case for this setting would be when a comboBoxItem is showing multiple
*  ComboBoxItem.pickListFields - if the same set of fields is specified as
*  ComboBoxItem.filterFields, the user can use the text-box to filter against
*  whichever fields are visible in the pickList.
*  
*  For finer grained control over comboBoxItem filtering, the 
*  ComboBoxItem.getPickListFilterCriteria method may be overridden.
*  
* @type {Array<Partial<string>>}
* @default null
*/
filterFields?: Array<Partial<string>>;

/**
* For ComboBoxItem.optionDataSource items, this property will be passed
*  to the generated ResultSet data object for the pickList as ResultSet.useClientFiltering.
*  Setting to false will disable filtering on the client and ensure criteria are
*  always passed to the DataSource directly.
* @type {boolean}
* @default null
*/
useClientFiltering?: boolean;

/**
* Minimum length in characters before a search is performed. If too few characters
*  are entered the pick list shows ComboBoxItem.searchStringTooShortMessage.
* @type {Integer}
* @default null
*/
minimumSearchLength?: Integer;

/**
* Message to display in pick list when ComboBoxItem.minimumSearchLength characters have not been entered.
* @type {string}
* @default "Enter a longer search string to search"
*/
searchStringTooShortMessage?: string;

/**
* When applying filter criteria to pickList data, what type of matching to use.
*  
*  For a databound pickList (ComboBoxItem.optionDataSource set), textMatchStyle is
*  sent to the server as DSRequest.textMatchStyle.
*  
*  For a non-databound pickList, textMatchStyle is applied by 
*  PickList.filterClientPickListData.
* @type {TextMatchStyle}
* @default "startsWith"
*/
textMatchStyle?: TextMatchStyle;

/**
* If true, even non-matching options will be shown, with configurable 
*  ComboBoxItem.separatorRows in between. Not valid for
*  ComboBoxItem.optionDataSource.
* @type {boolean}
* @default null
*/
showAllOptions?: boolean;

/**
* Array of records to show between matching and non-matching rows in the PickList.
*  
*  Not valid for ComboBoxItem.optionDataSource.
* @type {Array<Partial<ListGridRecord>>}
* @default "[{isSeparator:true}]"
*/
separatorRows?: Array<Partial<ListGridRecord>>;

/**
* When a comboBoxItem is used to generate search criteria in a SearchForm this property
*  governs whether, if the user explicitly chose an option from the pickList, we explicitly
*  generate criteria that will search for an exact match against the chosen value.
*  
*  In order to achieve this, when this property is set to true, this item will generate
*  AdvancedCriteria in its ComboBoxItem.getCriterion method
*  .
*  
*  See ComboBoxItem.shouldGenerateExactMatchCriteria for behavior when this flag is unset.
* @type {boolean}
* @default null
*/
generateExactMatchCriteria?: boolean;

/**
* The standard FormItem.allowExpressions behavior is always disabled for 
*  ComboBoxItem.
*  
*  The interface is not compatible with the allowExpressions feature.
*  A ComboBoxItem normally starts fetching matches as you type, and that mixes 
*  very strangely with the idea of entering expressions like "a..b"
*  - you will have the ComboBox seemingly switching back and forth between treating
*  the text as a normal search string vs as a special expression on a per-keystroke
*  basis.
*  
*  We recommend a normal TextItem as the correct UI element to supply for users to
*  enter filter expressions.
* @type {boolean}
* @default null
*/
allowExpressions?: boolean;



/**
*  Expression evaluated to determine the ComboBoxItem.defaultValue when no value is 
*  provided for this item. To default to the first option use
*  ComboBoxItem.defaultToFirstOption instead.
* 
*/
defaultDynamicValue(): void;

/**
*  Returns the raw text value that currently appears in the text field, which can differ from 
*  FormItem.getValue in various cases - for example:
*  
*  
* - for items that constrain the value range, such as a DateItem with
*  DateItem.enforceDate:true, or a ComboBoxItem with
*  ComboBoxItem.addUnknownValues:false
*  
* - for items with a defined valueMap or edit value formatter and parser functions
*  which converts display value to data value
*  
* - while the item has focus if TextItem.changeOnKeypress is false
*  
*  Note: if the pickList is being shown in any view other than the default 
*  ComboBoxItem.pickListPlacement, as is typically the case on a mobile device,
*  this method will return the value of the ComboBoxItem.pickerSearchField.
*  
* 
*/
getEnteredValue(): string;

/**
*  Only applies to databound items (see PickList.optionDataSource).
*  Performs a fetch type operation on this item's DataSource to retrieve the set of valid
*  options for the item, based on the current PickList.pickListCriteria.
* 
* @param {DSCallback=} Callback to fire when the fetch completes. Callback will        fire with 4 parameters:  
- item a pointer to the form item  
- dsResponse the DSResponse returned by the server  
- data the raw data returned by the server  
- dsRequest the DSRequest sent to the server  
* @param {Partial<DSRequest>=} properties to apply to the       dsRequest for this fetch.
*/
fetchData(callback?:DSCallback, requestProperties?:Partial<DSRequest>): void;

/**
*  Returns the displayField for this item.
*  
*  Behavior varies based on the configuration of this item, as follows:
*  
* - If this item has an ComboBoxItem.optionDataSource and an explicit 
*  FormItem.foreignDisplayField is specified, this will be returned.
*  
* - Otherwise if an explicit ComboBoxItem.displayField is specified it will be returned 
*  by default. If the displayField was specified on the underlying 
*  dataSource field, and no matching field is present in the ComboBoxItem.optionDataSource for
*  the item, we avoid returning the specified displayField value and instead return the
*  title field of the option DataSource. We do this to
*  avoid confusion for the case where the displayField is intended as a display-field
*  value for showing another field value within the same record in the underlying
*  dataSource only.
*  
* - If no explicit foreignDisplayField or displayField 
*  specification was found, and the FormItem.valueField for this item is hidden in the 
*  FormItem.optionDataSource, this method will return the title field for 
*  the optionDataSource.
* 
* 
*/
getDisplayFieldName(): FieldName;

/**
*  Getter method to retrieve the FormItem.valueField for this item. For
*  items with a specified FormItem.optionDataSource, this determines which
*  field in that dataSource corresponds to the value for this item.
*  
*  If unset, if a DataSourceField.foreignKey exists
*  between this field and the optionDataSource, this will be used,
*  otherwise default behavior will return the FormItem.name of this field.
* 
* 
*/
getValueFieldName(): string;

/**
*  Get the record returned from the ComboBoxItem.optionDataSource when FormItem.fetchMissingValues
*  is true, and the missing value is fetched.
*  
*  FormItem.fetchMissingValues kicks off the fetch when the form item is initialized
*  with a non null value or when setValue() is called on the item. Note that this method
*  will return null before the fetch completes, or if no record is found in the
*  optionDataSource matching the underlying value.
* 
*/
getSelectedRecord(): ListGridRecord;

/**
*  stringMethods to return filter criteria for options displayed for
*  this item. 
*  
*  See comboBoxFiltering for details on how pickList filter criteria
*  are calculated by default for a comboBoxItem.
*   
* 
*/
getPickListFilterCriteria(): Criteria;

/**
*  Returns the set of data to be displayed in this item's PickList.
*  
*  This method will be called for non-databound form items implementing the PickList
*  interface. The default implementation will derive data from the item's valueMap - 
*  can be overridden to allow a custom set of options to be displayed.
*  
*  Note that for PickLists that filter data based on user input
*  (ComboBoxItem), this method should return the data before
*  filtering. To customize the data returned after filtering, override
*  ComboBoxItem.filterClientPickListData instead.
*  
*  As an example, for a formItem with ComboBoxItem.valueField set to "valueFieldName", the
*  default implementation would take a valueMap like the following:
*  
*    valueMap: { value1: "display 1", value2: "display 2" }
*  
*  .. and returning the following set of records: 
*  
*    [
*      { valueFieldName : "value1" },
*      { valueFieldName : "value2" }
*    ]
*  
*  Due to the valueMap, these records will appear as a two row pickList displayed as
*  "display 1" and "display 2".
* 
* 
*/
getClientPickListData(): Array<Partial<ListGridRecord>>;

/**
*  Returns the data to display in the pick list.
*  
*  The default implementation applies the criteria returned by 
*  PickList.getPickListFilterCriteria to the data returned by
*  PickList.getClientPickListData. A record passes the filter if it has a
*  matching value for all fields in the criteria object. Matching is performed according
*  to TextMatchStyle.
*  
*  If PickList.showAllOptions is set, all values are shown, with matching values
*  shown below a PickList.separatorRows.
* 
* 
*/
filterClientPickListData(): Array<Partial<ListGridRecord>>;

/**
*  If this item is showing a dataBound pickList, this notification method will be fired 
*  when new data arrives from the server.
* 
* @param {Integer} index of first row returned by the server
* @param {Integer} index of last row returned by the server
* @param {ResultSet} pointer to this pickList's data
*/
dataArrived(startRow:Integer, endRow:Integer, data:ResultSet): void;

/**
*  When a comboBoxItem is used to generate search criteria in a SearchForm, 
*  if the user explicitly chose an option from the pickList, should the criterion generated
*  by ComboBoxItem.getCriterion
*  
*  enforce a search for an exact match against the chosen value?
*  
*  In order to achieve this, when this property is set to true, this item will generate
*  AdvancedCriteria in its ComboBoxItem.getCriterion
*  
*  method.
*  
*  Default implementation will return ComboBoxItem.generateExactMatchCriteria if specified, otherwise
*  true if the DataSource for this item 
*  DataSource.supportsAdvancedCriteria, false if it
*  does not.
* 
*/
shouldGenerateExactMatchCriteria(): boolean;

/**
*  Will this item return advancedCriteria if DynamicForm.getValuesAsCriteria is 
*  called on this item's form?
*  Overridden for ComboBoxItem to return true if ComboBoxItem.generateExactMatchCriteria is true
*  - in this case if an exact value is chosen from our set of options (always the
*  case if addUnkownValues is false), we will use advancedCriteria to ensure
*  the generated search criteria exactly matches the chosen value for this item.
*  
*  Note that AdvancedCriteria are not supported by all dataSources. When a form is bound
*  to a dataSource, we therefore default generateExactMatchCriteria to false unless
*  the dataSource is known to support AdvancedCriteria.
*  
*  As with formItem.hasAdvancedCriteria() this will also return true if a Operator
*  was explicitly specified for this item
* 
*/
hasAdvancedCriteria(): boolean;

/**
*  Returns criterion derived from the current value of this item.
*  
*  If ComboBoxItem.addUnknownValues is true for this item, we implement the following behavior.
*  If the user explicitly selected an item from the pickList, we treat this as an attempt
*  to explicitly match the data value. In this case returned criteria will match the
*  selected (data) value against this item's fieldName.
*  If the user typed a value into the text field, we treat this as an attempt to do a substring
*  type filter. In this case returned criteria will match the entered text value against the
*  displayField for this item if one is specified.
*  
*  If ComboBoxItem.addUnknownValues is false we always match the chosen data value against the item's 
*  fieldName.
*  
*  Note that ComboBoxItem.shouldGenerateExactMatchCriteria will be called in the case when a
*  value was explicitly picked from the set of options. If that method returns true, we will
*  return AdvancedCriteria with an operator specified to ensure an exact match occurs.
*  
* 
*/
getCriterion(): Criterion;

/**
*  This method is overridden in comboBoxItem. When addUnknownValues is true, 
*  comboBoxItems allow the user to edit substring match type criteria applied to the
*  display field (if one is specified).
*  
*  The user can also edit criteria attempting to match exactly against the item's field name.
* 
* 
* @param {Criterion} sub-criterion from an AdvancedCriteria object
*/
canEditCriterion(criterion:Criterion): boolean;

/**
*  Overridden to support editing criterion against the display field or value field when
*  ComboBoxItem.addUnknownValues is true.
* 
*/
setCriterion(): void;



}
/**
* Creates a new ComboBoxItem
* 
* @param typeCheckedProperties {Partial<ComboBoxItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ComboBoxItem} The new instance
*/
class ComboBoxItem{
static create(typeCheckedProperties? : Partial<ComboBoxItem>, uncheckedProperties?: Object): ComboBoxItem;







}

/**
* Provides background communication with an HTTP server
*/
interface Comm extends Class {




}
/**
* Creates a new Comm
* 
* @param typeCheckedProperties {Partial<Comm>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {Comm} The new instance
*/
class Comm{
static create(typeCheckedProperties? : Partial<Comm>, uncheckedProperties?: Object): Comm;
/**
* If true, SmartClient will use a 
*  web worker to send 
*  RPCRequests and DSRequests concurrent with the main
*  Javascript thread, if:
*  
* - The RPCRequest.transport is "xmlHttpRequest"
*  
* - The browser is Internet Explorer 10 or greater
*  
*  We do this because Internet Explorer sometimes queues the sending of data with other,
*  timer-delayed tasks on the single Javascript thread. With a busy application, this can
*  lead to an xmlHttpRequest seeming to block; the HTTP connection is made to the server, and
*  the server then goes into a wait state while the browser completes other timer-delayed 
*  tasks. If there are expensive timer-delayed tasks in the way, such as a reflow of the
*  UI, completing the send to the server can block for a significant length of time,
*  leading in turn to server turnaround times that are significantly longer than they need
*  to be.
*  
*  Internet Explorer is the only browser that behaves like this, so this option does not 
*  apply to other browsers.
* @type {boolean}
* @default false
*/
static concurrentXHRsInIE?: boolean;

/**
* The message to show the user if Comm.concurrentXHRsInIE is 
*  in force and an error occurs on the concurrent worker thread. If this attribute is
*  left at its default null value, the user is shown the error message reported by the 
*  browser, along with the URL and line number where the error occurred (this information
*  is also logged to the developer console, regardless of the message shown to the user)
* @type {string}
* @default null
*/
static concurrentXHRErrorMessage?: string;








}

/**
* Base class for tasks that target SmartClient
*  UI-specific operations.
*  
*  Note: This task is not for direct use - use one of the subclasses instead.
*/
interface ComponentTask extends ProcessElement {
/**
* ID of component.
* @type {GlobalId}
* @default null
*/
componentId?: GlobalId;

/**
* Base class of components that this task targets.
* @type {string | Array<Partial<string>>}
* @default "DataBoundComponent"
*/
componentBaseClass?: string | Array<Partial<string>>;

/**
* Must target components of this task have a DataSource?
* @type {boolean}
* @default null
*/
componentRequiresDataSource?: boolean;





}
/**
* Creates a new ComponentTask
* 
* @param typeCheckedProperties {Partial<ComponentTask>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {ComponentTask} The new instance
*/
class ComponentTask{
static create(typeCheckedProperties? : Partial<ComponentTask>, uncheckedProperties?: Object): ComponentTask;







}

/**
* The CubeGrid is an interactive grid component that presents very large, multi-dimensional
*  data sets (also known as data cubes) for reporting or analytic applications.
*  
*  CubeGrids are often called crosstabs, for their cross-tabular display of data dimensions in
*  stacked/nested rows and columns, or pivot tables, for their ability to "pivot" dimensions
*  between rows and columns to view a data cube from different perspectives. They are typically
*  used in the querying and reporting front-ends of data warehousing, decision support, OLAP,
*  and business intelligence systems.
*  
*  For example, CubeGrids can be connected to Pentaho Mondrian, Jasper Reports, Microsoft
*  Analysis Services and any other OLAP technology that supports the XMLA standard - the
*  Isomorphic public wiki has 
*  examples.
*  of such integration.
*  
*  NOTE: you must load the Analytics 
*  loadingOptionalModules before you can use CubeGrid.
*  
*  Multi-Dimensional Data Terminology
*  
*  The CubeGrid refers to the dimensions of a data cube as facets, to the possible values in
*  each facet as facet values, and to the values within the data cube as data values or cell
*  values. Equivalent terms that are commonly used in data warehousing or business intelligence
*  systems include:
*  facet: dimension, attribute, feature
*  facet value: dimension member, attribute value, feature value
*  cell value: data value, metric value, measure
*  
*  Visual Structure
*  
*  Like the ListGrid and TreeGrid components, the CubeGrid displays data values in a tabular
*  "body" with adjacent "headers". While the ListGrid and TreeGrid display rows of records
*  with field values, the CubeGrid displays a body of individual cell values, each associated
*  with a combination of facet values. The facet values for a cell are displayed in the column
*  headers above the cell and row headers to the left of the cell. CubeGrids can display an
*  arbitrary number of facets, by stacking multiple levels of row and/or column headers.
*  
*  Except for the innermost column facet, each facet in a CubeGrid has a facet label adjacent
*  to its row or column headers. The facet labels serve two main purposes: they display the
*  titles of the facets, and they provide drag-and-drop reordering or pivoting of facets within
*  the CubeGrid. The row facet labels also provide interactive selection, resizing, and other
*  operations on the columns of row facet values.
*  
*  The innermost column headers provide special behaviors and controls for manipulating the
*  columns of data in a CubeGrid. End users may select, resize, reorder, minimize, maximize, or
*  auto-fit the columns of data via mouse interactions with these headers. Customizable
*  indicators and controls may be included at the top of each innermost column header.
*  
*  If a CubeGrid is not large enough to display all of its cell values, horizontal and/or
*  vertical scrollbars will appear below and to the right of the body. The body of the CubeGrid
*  may be scrolled on either axis. The headers are "frozen" from scrolling on one axis - row
*  headers only scroll vertically, while column headers only scroll horizontally - so the facet
*  values for the visible cells are always displayed.
*  
*  Data Loading
*  
*  Data can be provided to the Cube via CubeGrid.data as an Array of
*  CellRecord, each representing the data for one cell.
*  
*  For large datasets, CubeGrid.dataSource with one field per
*  facetId, and the CubeGrid will load data on demand to fill the visible area, including
*  lazily loading data for expanding/collapsing tree facets and when facetValues are made
*  visible programmatically or via menus.
*  
*  Picking Facets
*  
*  By "facet" we mean an aspect of the data which is orthogonal to other aspects of the data,
*  that is, combining values from any two "facets" should make sense.
*  
*  For example, in sales data, two facets might be "quarter" and "region" - it makes sense to
*  combine any quarter and region, although for some combinations, there may not be data
*  available.
*  
*  An example of two aspects that would not be independent facets are "state" and "city"
*  - it's senseless to combine arbitrary states with arbitrary cities - most combinations are
*  invalid. Consider instead a Facet.isTree that combines "city" and "state"
*  values. 
*  
*  Note that if "city" and "state" are represented as facets, they may look correct if they are
*  both on the same axis of the grid and
*  CubeGrid.hideEmptyFacetValues is used to trim nonsense
*  combinations, but if the data is CubeGrid.canMoveFacets such that "state" and
*  "city" are on opposing axes, there will be a roughly diagonal "stripe" of data for
*  combinations of "state" and "city" that make sense, and all other space will be blank. This
*  is a strong indication that two facets should be represented as a single tree facet instead.
*/
interface CubeGrid extends Omit<ListGrid, 'exportClientData' | 'getEditValues' | 'setEditValues' | 'clearEditValue' | 'discardAllEdits' | 'saveAllEdits' | 'getCellRecord'>  {
/**
* CSS class for the CubeGrid as a whole
* @type {CSSStyleName}
* @default "normal"
*/
styleName?: CSSStyleName;

/**
* CSS class for the CubeGrid body
* @type {CSSStyleName}
* @default "cubeGridBody"
*/
bodyStyleName?: CSSStyleName;

/**
* GridRenderer.baseStyle for this listGrid.
*  If this property is unset, base style may be derived from ListGrid.normalBaseStyle
*  or ListGrid.tallBaseStyle as described in
*  ListGrid.getBaseStyle.
*  
*  See cellStyleSuffixes for details on how stateful suffixes are combined
*  with the base style to generate stateful cell styles.
* @type {CSSStyleName}
* @default "cubeCell"
*/
baseStyle?: CSSStyleName;

/**
* Whether alternating rows should be drawn in alternating styles, in order to create a "ledger"
*  effect for easier reading. If enabled, the cell style for alternate rows will have "Dark"
*  appended to it.
* @type {boolean}
* @default true
*/
alternateRecordStyles?: boolean;

/**
* Default directory for skin images (those defined by the class), relative to the
*  Page-wide Page.getSkinDir.
* @type {SCImgURL}
* @default "images/CubeGrid/"
*/
skinImgDir?: SCImgURL;

/**
* Button.baseStyle for the buttons in the innermost column header for 
*  this cubeGrid.
* @type {CSSStyleName}
* @default "innerHeader"
*/
innerHeaderBaseStyle?: CSSStyleName;

/**
* Button.baseStyle for the buttons in this grid's column headers.
*  
*  Exception: The innermost column
*  header will always be styled using CubeGrid.innerHeaderBaseStyle.
* @type {CSSStyleName}
* @default "colHeader"
*/
colHeaderBaseStyle?: CSSStyleName;

/**
* Button.baseStyle for the facet-label buttons above this grid's 
*  column headers.
* @type {CSSStyleName}
* @default "colHeaderLabel"
*/
colHeaderLabelBaseStyle?: CSSStyleName;

/**
* Button.baseStyle for the buttons in this grid's row headers.
* @type {CSSStyleName}
* @default "rowHeader"
*/
rowHeaderBaseStyle?: CSSStyleName;

/**
* Button.baseStyle for the facet-label buttons above the grid's 
*  row headers.
* @type {CSSStyleName}
* @default "rowHeaderLabel"
*/
rowHeaderLabelBaseStyle?: CSSStyleName;

/**
* This property is not supported for CubeGrid.
* @type {boolean}
* @default null
*/
rotateHeaderTitles?: boolean;

/**
* If specified and this.showHover is true, this is the default width to apply
*  to hover tips shown for facetValues. If unset, the hover canvas will be sized to 
*  this.hoverWidth if specified instead.
* @type {number | string}
* @default null
*/
facetValueHoverWidth?: number | string;

/**
* If specified and this.showHover is true, this is the default width to apply
*  to hover tips shown for facetLabels. If unset, the hover canvas will be sized to 
*  this.hoverWidth if specified instead.
* @type {number | string}
* @default null
*/
facetLabelHoverWidth?: number | string;

/**
* If specified and this.showHover is true, this is the default height to apply
*  to hover tips shown for facetValues. If unset, the hover canvas will be sized to 
*  this.hoverHeight if specified instead.
* @type {number | string}
* @default null
*/
facetValueHoverHeight?: number | string;

/**
* If specified and this.showHover is true, this is the default height to apply
*  to hover tips shown for facetLabels. If unset, the hover canvas will be sized to 
*  this.hoverHeight if specified instead.
* @type {number | string}
* @default null
*/
facetLabelHoverHeight?: number | string;

/**
* Allows the developer to override the horizontal text alignment of hover tips shown for
*  facet values. If unspecified the hover canvas content alignment will be set by
*  this.hoverAlign if specified.
* @type {Alignment}
* @default null
*/
facetValueHoverAlign?: Alignment;

/**
* Allows the developer to override the horizontal text alignment of hover tips shown for
*  facetLabels. If unspecified the hover canvas content alignment will be set by
*  this.hoverAlign if specified.
* @type {Alignment}
* @default null
*/
facetLabelHoverAlign?: Alignment;

/**
* Allows the developer to override the vertical text alignment of hover tips shown for
*  facet values. If unspecified the hover canvas content alignment will be set by
*  this.hoverVAlign if specified.
* @type {VerticalAlignment}
* @default null
*/
facetValueHoverVAlign?: VerticalAlignment;

/**
* Allows the developer to override the vertical text alignment of hover tips shown for
*  facetLabels. If unspecified the hover canvas content alignment will be set by
*  this.hoverVAlign if specified.
* @type {VerticalAlignment}
* @default null
*/
facetLabelHoverVAlign?: VerticalAlignment;

/**
* Allows the developer to override the css class applied to hover tips shown for
*  facet values. If unspecified, and this.hoverStyle is not null, that 
*  css class will be applied to facet value hovers instead.
* @type {CSSStyleName}
* @default null
*/
facetValueHoverStyle?: CSSStyleName;

/**
* Allows the developer to override the css class applied to hover tips shown for
*  facet labels. If unspecified, and this.hoverStyle is not null, that 
*  css class will be applied to facet label hovers instead.
* @type {CSSStyleName}
* @default null
*/
facetLabelHoverStyle?: CSSStyleName;

/**
* A FacetValueMap describing the set of facet values that should be regarded as
*  "fixed" in this cubeGrid. These are used as fixed criteria for load on demand, and also
*  allow using a dataset with more facets in it than are currently shown in the grid.
* @type {FacetValueMap}
* @default null
*/
fixedFacetValues?: FacetValueMap;

/**
* Hilites to be applied to the data for this component. See hiliting.
* @type {Array<Partial<Hilite>>}
* @default null
*/
hilites?: Array<Partial<Hilite>>;

/**
* DataSource to use to fetch CubeGrid data.
*  
*  The DataSource should have a field named after each facetId. The CubeGrid will submit
*  requests for data as DataSource "fetch" operations that request 
*  CellRecord only for currently visible area
*  (including ListGrid.drawAheadRatio). The Criteria passed
*  in each fetch operation will be a FacetValueMap that corresponds to a rectangular
*  swath of cells the CubeGrid needs data for, for example:
*    { region:"west", product:["chair", "table"], timePeriod:"Q1 2004" }
*  
*  Note that in the criteria above, the Array value for the "product" facet indicates the
*  CubeGrid needs cellRecords for both the "chair" and "table" facetValues.
*  
*  CubeGrid.fixedFacetValues, if any, are always sent in each "fetch" operation.
*  
*  Scrolling, expanding/collapsing tree facets, pivoting, and changing the currently
*  visible facets may all trigger data requests.
*  
*  The CubeGrid will generally attempt to fetch a minimal set of data to fill the viewport,
*  sending multiple fetch operations in a batch (via RPCManager.startQueue)
*  which minimally describe newly revealed areas without specifying each individual cell.
*  The CubeGrid will automatically handle being provided more data than was asked
*  for, so server-side fetch-ahead policies can be implemented without any client-side
*  customization.
*  
*  Note that the DataSource.serverType shipped with the SmartClient
*  SDK is capable of responding to the CubeGrid's data requests without writing any custom
*  server code.
* @type {DataSource}
* @default null
*/
dataSource?: DataSource;

/**
* If true, when this component is first drawn, automatically call this.fetchData().
*  Criteria for this fetch may be picked up from ListGrid.initialCriteria, and textMatchStyle may
*  be specified via ListGrid.autoFetchTextMatchStyle.
*  
*  NOTE: if autoFetchData is set, calling
*  ListGrid.fetchData before draw will cause two requests to be issued, one from the manual
*  call to fetchData() and one from the autoFetchData setting. The second request will use
*  only ListGrid.initialCriteria and not any other criteria or settings from the first request.
*  Generally, turn off autoFetchData if you are going to manually call ListGrid.fetchData at any time.
* @type {boolean}
* @default true
*/
autoFetchData?: boolean;

/**
* If CubeGrid.autoFetchData is true, this attribute allows the developer to
*  specify a textMatchStyle for the initial ListGrid.fetchData call.
* @type {TextMatchStyle}
* @default "exact"
*/
autoFetchTextMatchStyle?: TextMatchStyle;

/**
* Facet definitions for this CubeGrid. Facets, also called "dimensions", are orthogonal
*  aspects of the data model.
*  
*  For example, you can look at profit by the facets "plant and product" or by "product and
*  plant" and it's the same number, because the facets - plant and product - are the same.
*  What would change the profit numbers would be to remove a facet, called "summarizing",
*  or add a new facet, called "drilling down". For example, showing profit by plant and
*  product, you could "drill down" by adding the region facet, which would divide profit
*  among each region. Or you could remove the "plant" facet, showing total profit for each
*  "product", summed across all plants.
*  
*  This property need not be set and will automatically be constructed during widget
*  initialization if data is provided up front and CubeGrid.rowFacets and 
*  CubeGrid.columnFacets have been set. If CubeGrid.facets is not set and
*  there is no initial data but a DataSource is present, drawing the grid will
*  automatically issue a fetch to allow CubeGrid.facets to be resolved.
* @type {Array<Partial<Facet>>}
* @default null
*/
facets?: Array<Partial<Facet>>;

/**
* If true, show facet label context menus with some built-in operations.
*  Otherwise, use generic context menu handling.
* @type {boolean}
* @default false
*/
showFacetContextMenus?: boolean;

/**
* If true, show facet value context menus with some built-in operations.
*  Otherwise, use generic context menu handling.
*  Use this in place of ListGrid.showHeaderContextMenu and
*  ListGrid.showHeaderMenuButton for CubeGrids.
* @type {boolean}
* @default true
*/
showFacetValueContextMenus?: boolean;

/**
* Determines whether cell values in the body can be selected.
* @type {boolean}
* @default true
*/
canSelectValues?: boolean;

/**
* Determines whether row or column facetValue headers can be selected.
* @type {boolean}
* @default true
*/
canSelectHeaders?: boolean;

/**
* If true, when multiple facets appear on one side in a nested headers presentation, the
*  selection state of parent/child headers are automatically kept in sync.
* @type {boolean}
* @default true
*/
autoSelectHeaders?: boolean;

/**
* Whether to select cells in the body when row or column headers are selected.
* @type {AutoSelectionModel}
* @default "both"
*/
autoSelectValues?: AutoSelectionModel;

/**
* If true, clicking on the existing selection causes it to be entirely deselected.
* @type {boolean}
* @default false
*/
simpleDeselect?: boolean;

/**
* The list of Facet.id for facets that will appear on top of the body.
* @type {Array<Partial<string>>}
* @default null
*/
columnFacets?: Array<Partial<string>>;

/**
* The list of Facet.id for facets that will appear to the left of the body.
* @type {Array<Partial<string>>}
* @default null
*/
rowFacets?: Array<Partial<string>>;

/**
* This causes the headers for any combination of facetValues for which there are no 
*  cellRecords to be suppressed.
*  
*  To use this feature, either:
*  
*  
* - all must be provided via ListGrid.setData before the CubeGrid is first drawn, OR
*  
* - all data must be returned by the first DataSource fetch, OR
*  
* - CubeGrid.hideEmptyAxis must be set to either "row" or "column" so that empty values
*  are only automatically hidden for one axis
*  
*  This last point is required because there is no way to determine whether a row is empty
*  unless data for all columns of the row has been loaded (and vice-versa). For this
*  reason if you set hideEmptyFacetValues but do not set hideEmptyAxis, the default
*  behavior of DataSource is automatically disabled and
*  only CubeGrid.fixedFacetValues will be sent as criteria.
* @type {boolean}
* @default null
*/
hideEmptyFacetValues?: boolean;

/**
* With CubeGrid.hideEmptyFacetValues, controls on which axis hiding of empty values is applied,
*  "row" (only empty rows are hidden), "column" (only empty columns are hidden) or both
*  (the default).
* @type {Axis}
* @default null
*/
hideEmptyAxis?: Axis;

/**
* An array of "cellRecords", each of which represents data for one cell of the body area.
* @type {Array<Partial<CellRecord>>}
* @default null
*/
data?: Array<Partial<CellRecord>>;

/**
* Name of the property in a cell record that holds the cell value.
* @type {Identifier}
* @default "_value"
*/
valueProperty?: Identifier;

/**
* Name of the property in a cell record that holds it's unique ID. Note cell record IDs are
*  optional.
* @type {Identifier}
* @default "ID"
*/
cellIdProperty?: Identifier;

/**
* Whether cells can be edited in this grid. Can be overridden on a per-facetValue basis.
* @type {boolean}
* @default false
*/
canEdit?: boolean;

/**
* CubeGrids only support editing by cell.
* @type {boolean}
* @default true
*/
editByCell?: boolean;

/**
* CubeGrids only support editing by cell.
* @type {boolean}
* @default true
*/
saveByCell?: boolean;

/**
* For Browser.isTouch, canDragSelect defaults to false
*  so that touch scrolling can be used to navigate scrollable CubeGrids. In all other
*  browsers it defaults to true.
*  
*  NOTE: If canDragSelect is enabled, it may be desirable to disable
*  Canvas.useTouchScrolling so that touch-dragging cells of the CubeGrid
*  selects them rather than starting a scroll. If Canvas.disableTouchScrollingForDrag
*  is set to true, then touch scrolling will be disabled automatically.
*  However, for accessibility reasons, it is recommended to
*  leave touch scrolling enabled and provide an alternative set of controls that can be
*  used to perform drag-selection.
* @type {boolean}
* @default null
*/
canDragSelect?: boolean;

/**
* If true, body columns can be reordered via the innermost column headers.
* @type {boolean}
* @default null
*/
canReorderColumns?: boolean;

/**
* If true, body columns can be resized via the innermost column headers.
* @type {boolean}
* @default null
*/
canResizeColumns?: boolean;

/**
* If true, hierarchical facets will show expand/collapse controls to allow the user to
*  expand and collapse the tree of facetValues for that facet.
* @type {boolean}
* @default false
*/
canCollapseFacets?: boolean;

/**
* If true, when multiple facets are shown on a side, all facetValues in the second level
*  of headers or higher will show controls to "minimize" the values of the next facet.
*  Minimizing means showing only one, or very few, of the next facet's values.
*  
*  Set FacetValue.isMinimizeValue to indicate which facetValues should be
*  shown when a facet is minimized.
* @type {boolean}
* @default false
*/
canMinimizeFacets?: boolean;

/**
* Turns on indenting of any Facet.isTree column facets. This can be
*  overridden at the facet level via Facet.indentVTree. Setting this property also
*  ensures that the header is sized tall enough to accommodate the fully expanded facet. 
*  
*  The amount of indenting per level can be set with CubeGrid.vTreeFacetIndent, and the
*  direction of the indenting specified with CubeGrid.vTreeFacetIndentDirection.
*  
*  Note that if you specify an explicit height for such a fscet, such as by setting
*  Facet.height or Facet.labelHeight, then the greater of that or the space
*  required to accommodate the fully expanded facet will determine the actual height used.
* @type {boolean}
* @default false
*/
indentVTreeFacets?: boolean;

/**
* Determines how many pixels to move for each level when Facet.isTree
*  column facets are being CubeGrid.indentVTreeFacets.
* @type {Integer}
* @default "8"
*/
vTreeFacetIndent?: Integer;

/**
* Determines layout of facet value titles in each column facet being 
*  CubeGrid.indentVTreeFacets.
* @type {FacetIndentDirection}
* @default "descending"
*/
vTreeFacetIndentDirection?: FacetIndentDirection;

/**
* FacetValueMap of facet values representing a set of facetValues by which the
*  cubeGrid data is sorted.
* @type {FacetValueMap}
* @default null
*/
sortedFacetValues?: FacetValueMap;

/**
* Direction of sorting if sortedFacet or sortedFacetValues is specified.
* @type {SortDirection}
* @default "ascending"
*/
sortDirection?: SortDirection;

/**
* If true, sort controls will be shown on FacetHeaders.
*  
*  When clicked, sort controls call CubeGrid.sortByFacetId.
* @type {boolean}
* @default null
*/
canSortFacets?: boolean;

/**
* If true, sort controls will be shown on facet values.
*  
*  When clicked, sort controls call CubeGrid.sortByFacetValues.
* @type {boolean}
* @default null
*/
canSortData?: boolean;

/**
* If true, allow columns in the grid body to be minimized (reduced to the width of the
*  minimize control) by clicking on a minimize control in the innermost column headers.
* @type {boolean}
* @default null
*/
canMinimizeColumns?: boolean;

/**
* Whether row and column facets can be rearranged by the user, by dragging and dropping the
*  facet labels.
* @type {boolean}
* @default false
*/
canMoveFacets?: boolean;

/**
* If enabled row headers for this cubeGrid will be rendered using a GridRenderer
*  component. This improves performance for very large cubeGrids.
*  
*  Note that this attribute must be set for hierarchical row facets to be indented properly.
* @type {boolean}
* @default false
*/
rowHeaderGridMode?: boolean;

/**
* Automatically size row headers to fit wrapped text.
* @type {boolean}
* @default false
*/
autoSizeHeaders?: boolean;

/**
* Default alignment for facet labels.
* @type {Alignment}
* @default "center"
*/
facetTitleAlign?: Alignment;

/**
* Default alignment for facet values (in headers).
* @type {Alignment}
* @default "center"
*/
facetValueAlign?: Alignment;

/**
* Default align for cell values (in body).
* @type {Alignment}
* @default "center"
*/
cellAlign?: Alignment;

/**
* Whether to allow text wrapping on facet titles.
* @type {boolean}
* @default false
*/
wrapFacetTitles?: boolean;

/**
* Whether to allow text wrapping on facet value titles.
*  
*  Note that this property is incompatible with CubeGrid.indentVTreeFacets column
*  facets.
* @type {boolean}
* @default false
*/
wrapFacetValueTitles?: boolean;

/**
* Whether to pad titles so they aren't flush with header borders.
* @type {boolean}
* @default true
*/
padTitles?: boolean;

/**
* Default width of inner column headers.
* @type {number}
* @default "100"
*/
defaultFacetWidth?: number;

/**
* facetValueId of the default rollupValue for each facet. Can be overridden per facet via
*  facet.rollupValue.
* @type {string}
* @default "sum"
*/
rollupValue?: string;

/**
* In a CubeGrid that displays values of different types (eg "Revenue" and "Income"), the
*  different types of values on display are enumerated as the facet values of the "metric
*  facet". 
*  
*  The metric facet is treated identically to any other facet by the CubeGrid: it can be
*  represented as row or column headers, can be innermost or have other facets under it,
*  can be moved around, etc. However when a metric facet is used, CubeGrid.metricFacetId
*  must be set to allow the CubeGrid to generate meaningful descriptions of values shown in
*  cells for use in hovers and other situations; see CubeGrid.valueTitle for a full
*  explanation.
* @type {string}
* @default "metric"
*/
metricFacetId?: string;

/**
* A label for the data values shown in cells, such as "Sales in Thousands", typically used
*  when the CubeGrid must generate a description for a cell value or set of cell values.
*  
*  For example, in a CubeGrid showing "Revenue" by region and product, a cell with a
*  CellRecord like:
*  
*  {product:"chairs", region:"northwest", _value:"$5k"}
*  
*  Should be described as "Revenue for Chairs for Northwest Region", not "Chairs for
*  Revenue for Northwest Region".
*  
*  For CubeGrids that show multiple types of values at once (eg both "Revenue" and
*  "Income") see CubeGrid.metricFacetId.
* @type {string}
* @default null
*/
valueTitle?: string;

/**
* Minimum width for the body of this cubeGrid.
* @type {number}
* @default null
*/
bodyMinWidth?: number;

/**
* Minimum height for the body of this cubeGrid.
* @type {number}
* @default null
*/
bodyMinHeight?: number;

/**
* This property is not supported for CubeGrid.
*  
*  Consider setting explicit widths via FacetValue.width or CubeGrid.defaultFacetWidth.
* @type {boolean}
* @default null
*/
autoFitFieldWidths?: boolean;

/**
* If set to true, context menu items will be included on the cells and headers providing the
*  user with an option to create a chart of the cubeGrid's data set.
*  See CubeGrid.makeChart for more information.
* @type {boolean}
* @default false
*/
enableCharting?: boolean;

/**
* Name of the SmartClient Class to be used when creating charts. Must support the
*  Chart interface.
* @type {string}
* @default "FacetChart"
*/
chartConstructor?: string;

/**
* Default type of chart to plot.
* @type {ChartType}
* @default "Column"
*/
chartType?: ChartType;

/**
* If CubeGrid.makeChart is called with a chart specification that will show more than
*  chartConfirmThreshold data elements, the user will be presented with a
*  isc.confirm.
*  
*  Set to 0 to disable this confirmation.
* @type {number}
* @default "2000"
*/
chartConfirmThreshold?: number;

/**
* Default separator string used by CubeGrid.exportClientData to separate column
*  and row facet value titles.
* @type {string}
* @default " - "
*/
exportFacetSeparatorString?: string;

/**
* Array of MenuItem to replace the default menu. 
*  Call CubeGrid.getDefaultFacetValueContextItems to get a default set of
*  items to start with.
* @type {Array<Partial<MenuItem>>}
* @default null
*/
facetValueContextItems?: Array<Partial<MenuItem>>;

/**
* Sets the background color for the row and column headers of the cube,
*  if not otherwise set by a more specific property.
*  (see CubeGrid.exportRowFacetBGColor and
*  CubeGrid.exportColumnFacetBGColor).
*  See also exportBGColor.
* @type {CSSColor}
* @default null
*/
exportFacetBGColor?: CSSColor;

/**
* Sets the background color for the row headers of the cube.
*  See also exportBGColor.
* @type {CSSColor}
* @default null
*/
exportRowFacetBGColor?: CSSColor;

/**
* Sets the background color for the column headers of the cube.
*  See also exportBGColor.
* @type {CSSColor}
* @default null
*/
exportColumnFacetBGColor?: CSSColor;

/**
* Sets the text color for the row and column headers of the cube, 
*  if not otherwise set by a more specific property.
*  (see CubeGrid.exportRowFacetTextColor and
*  CubeGrid.exportColumnFacetTextColor).
* @type {CSSColor}
* @default null
*/
exportFacetTextColor?: CSSColor;

/**
* Sets the text color for the row headers of the cube.
* @type {CSSColor}
* @default null
*/
exportRowFacetTextColor?: CSSColor;

/**
* Sets the text color for the column headers of the cube.
* @type {CSSColor}
* @default null
*/
exportColumnFacetTextColor?: CSSColor;

/**
* FormatString for numeric or date formatting. See DataSourceField.format.
* @type {FormatString}
* @default null
*/
valueFormat?: FormatString;

/**
* FormatString used during exports for numeric or date formatting. See
*  DataSourceField.exportFormat.
* @type {FormatString}
* @default null
*/
valueExportFormat?: FormatString;



/**
*  Returns the row number of the most recent mouse event.
* 
* @param {Integer=} optional y-coordinate to obtain row number, in lieu of the y            coordinate of the last mouse event
*/
getEventRow(y?:Integer): Integer;

/**
*  Returns the column number of the most recent mouse event.
* 
* @param {Integer=} optional x-coordinate to obtain column number for, in lieu of the x            coordinate of the last mouse event
*/
getEventColumn(x?:Integer): Integer;

/**
*  Return the CSS class for a cell. By default this method has the following implementation:
*  - return any custom style for the record (GridRenderer.recordCustomStyleProperty) 
*   if defined.
*  - create a style name based on the result of GridRenderer.getBaseStyle and the 
*   state of the record using the rules described in cellStyleSuffixes.
*  
*  Cell Styles are customizable by:
*  
*  
* - attaching a custom style to a record by setting 
*   record[this.recordCustomStyleProperty] to some valid CSS style name. 
*  
* - modifying the base style returned by getBaseStyle() [see that method for further 
*    documentation on this]
*  
* - overriding this function
*  
*  
*  In addition to this, GridRenderer.getCellCSSText may be overriden to
*  provide custom cssText to apply on top of the styling attributes derived from the named
*  style.
*  
* 
* 
* @param {ListGridRecord} record object for this row and column
* @param {number} number of the row
* @param {number} number of the column
*/
getCellStyle(record:ListGridRecord, rowNum:number, colNum:number): CSSStyleName;

/**
*  Return the base stylename for this cell. Default implementation just returns this.baseStyle.
*  See ListGrid.getCellStyle for a general discussion of how to style cells.
* 
* 
* @param {ListGridRecord} cell record as returned by getCellRecord
* @param {number} row number for the cell
* @param {number} column number of the cell
*/
getBaseStyle(record:ListGridRecord, rowNum:number, colNum:number): CSSStyleName;

/**
*  Note: This is a ListGrid feature which is inapplicable on this class.
* 
* @param {ListGridViewState} Object describing the desired view state for the grid
*/
setViewState(viewState:ListGridViewState): void;

/**
*  Note: This is a ListGrid feature which is inapplicable on this class.
* 
*/
getViewState(): ListGridViewState;

/**
*  Set the temporary edit values for some cell in the cubeGrid.
*  Note that only the CubeGrid.valueProperty of the object passed in
*  will be displayed in the cell.
* 
* @param {Array<Partial<number>>} 2 element array of the form          [rowNum,colNum] indicating the record being edited
* @param {object} New values for the record
*/
setEditValues(cell:Array<Partial<number>>, values:object): void;

/**
* 
*  Set the edit value for some cell in the cube grid.
*  Note that cubeGrids display one record per cell - the value passed in should be the 
*  desired edit value for the CubeGrid.valueProperty of the record.
* 
* 
* @param {number} Row number
* @param {number} Column number
* @param {any} New value for the record
*/
setEditValue(rowNum:number, colNum:number, value:any): void;

/**
* 
*  Returns the current temporary locally stored edit value for a cell being edited.
*  Note this is the CubeGrid.valueProperty that will be saved for the
*  cell in question.
* 
* 
* @param {number} index of the row for which the editValue should be returned
* @param {number} index of the column for which value should be returned
*/
getEditValue(rowNum:number, colNum:number): any;

/**
*  Only supported on ListGrid for now.
* 
* @param {Array<Partial<Hilite>>} Array of hilite objects
*/
setHilites(hilites:Array<Partial<Hilite>>): void;

/**
*  Returns the combination of unsaved edits (if any) and original values (if any) for a given
*  cell being edited.
*  
*  The returned value is never null, and can be freely modified.
* 
* 
* @param {number | object} rowNum of the record being edited, or an Object                   containing values for all the record's primary keys
* @param {number=} colNum of the record being edited. Only required if the records             rowNum is passed in as the first parameter
*/
getEditedRecord(valuesID:number | object, colNum?:number): object;

/**
*  Returns the current value of a cell. If the cell has an outstanding edit value, this will
*  be returned, otherwise the underlying value of the record will be returned.
* 
* 
* @param {number | object} rowNum of the record being edited, or an Object                   containing values for all the record's primary keys
* @param {number} colNum of the cell. Only required if the first parameter is               a rowNum
*/
getEditedCell(record:number | object, colNum:number): any;

/**
*  Returns the current set of unsaved edits for a given row being edited.
* 
* 
* @param {number | object} rowNum of the record being edited, or an Object                   containing values for all the record's primary keys
* @param {number} colNum of the record being edited. Only required if valuesID             is passed in as a rowNum.
*/
getEditValues(valuesID:number | object, colNum:number): object;

/**
* 
*  Clear a field value being tracked as an unsaved user edit for some cell.
*  The saved record value will be displayed in the the appropriate cell instead.
* 
* 
* @param {number | object} Row number, primary keys object for the record,                       or editValues object
* @param {number=} Column number for the cell in question. Only required if the               first parameter is a row number.
*/
clearEditValue(editValuesID:number | object, colNum?:number): void;

/**
* 
*  Validates and saves edits for some cell. If rowNum and colNum are not passed in, the current
*  edit cell will be saved.
*  
*  The 'callback' parameter provides a notification when the save attempt completes.
*  Cases under which the callback will fire are:
*  
* - Save completed successfully
*  
* - No changes to the edited cell, so save not required
*  
* - Validation failure occurred on the client or on the server
*  Note that if no rowNum/colNum were passed in and the editor is not showing for the cell,
*  the callback will NOT fire - in this case, the method is a no-op.
*  
*  Other, standard callbacks such as ListGrid.editComplete,
*  ListGrid.editFailed and ListGrid.cellChanged will
*  fire normally.
*  
*  Note this method does not hide the inline editors if they are showing - to explicitly save 
*  and end editing, use the method 'endEditing()'
* 
* 
* @param {EditCompletionEvent=} Event used to complete cell editing.     Optional, and defaults to "programmatic". Can be used by the      callback method to perform custom actions such as navigation when the      save completes.
* @param {Callback=} Callback to fire on completion of the saving process.                  If no edits were made or client-side validation fails                   the callback will be fired synchronously at the end of this                   method.                   Takes the following parameters:                   - rowNum (Number) edited row number                   - colNum (Number) edited column number                   - editCompletionEvent (EditCompletionEvent) event                    passed in (defaults to "programmatic")                   - success (boolean) false if the save was unable to                    complete due to a validation failure or server-side                    error.
* @param {number=} Which row should be saved. If unspecified the current edit row              is saved by default. Note that if there is no current edit cell               this method will no op.
* @param {number=} Which row should be saved. If unspecified the current edit column              is saved by default. Note that if there is no current edit cell               this method will no op.
*/
saveEdits(editCompletionEvent?:EditCompletionEvent, callback?:Callback, rowNum?:number, colNum?:number): void;

/**
*  Method to determine which records currently have pending (unsubmitted) edits.
*  Returns an array of 2 element arrays indicating the [rowNum,colNum] of the
*  cells in question.
* 
*/
getAllEditCells(): Array<any>;

/**
* 
* 
* @param {Array<Partial<any>>=} allows you to specify which cell(s) to drop          edits for. Each record should be identified as an array containing          [rowNum,colNum]
* @param {boolean=} By default this method will hide the editor if               it is currently showing for any row in the grid. Passing in                this parameter will leave the editor visible (and just reset                the edit values underneath the editor).
*/
discardAllEdits(records?:Array<Partial<any>>, dontHideEditor?:boolean): void;

/**
*  If this cubeGrid can be edited, this method will return true if the record passed in has been
*  edited, but the edits have not yet been saved to the CubeGrid's data object.
* 
* @param {number} row index of record to check for changes
* @param {number} column index of the record to check for changes
*/
recordHasChanges(rowNum:number, colNum:number): boolean;

/**
*  Determines whether any cells in this cubeGrid have been edited but not yet saved to the
*  underlying data set.
* 
*/
hasChanges(): boolean;

/**
*  Save a number of outstanding edits for this CubeGrid. If no cells are specified, all 
*  outstanding edits will be saved
* 
* @param {Array<Partial<any>>=} optionally specify which cells to save. Each cell should be specified as a 2 element array in the format [rowNum,colNum].
* @param {Callback=} If specified this callback will be fired on a successful save of the specified rows. Note that if there are no pending edits to be saved this callback will  not fire - you can check for this condition using CubeGrid.hasChanges or  CubeGrid.recordHasChanges.
*/
saveAllEdits(cells?:Array<Partial<any>>, saveCallback?:Callback): boolean;

/**
*  Get a facet definition by facetId. Constant time.
* 
* 
* @param {string} the id of the facet to retrieve
*/
getFacet(facetId:string): Facet;

/**
*  Get a facet value definition by facetId and facetValueId. Constant time.
* 
* 
* @param {string} the id of the facet to retrieve
* @param {string} the id of the facet value to retrieve
*/
getFacetValue(facetId:string, facetValueId:string): FacetValue;

/**
*  Get the facetValue definition for the facetValue to show when this facet is "rolled up" under
*  another facet, during a breakout.
* 
*  A facet is not required to have a rollup value, and if it does not have one, then rollups will
*  simply be blank rows. The facetValueId of the rollup value can be declared as
*  cubeGrid.rollupValue or facet.rollupValue. 
* 
* 
*/
getRollupValue(): string;

/**
*  Given a cell coordinate within this CubeGrid return a FacetValueMap
*  indicating the facet values for the cell.
* 
* @param {number} row index of the cell
* @param {number} column index of the cell
*/
getCellFacetValues(rowNum:number, colNum:number): FacetValueMap;

/**
*  Return a FacetValueMap of the facet values for the row field at the specified level
*  containing the requested row number. Note that outer row fields may span several grid rows.
* 
* 
* @param {Integer} 0-based index into the grid rows (and inner row header fields)
* @param {Integer} target header level; 0 represents the outer row header
*/
getRowHeaderFacetValues(rowNum:Integer, level:Integer): FacetValueMap;

/**
*  Return a FacetValueMap of the facet values for the column field at the specified
*  level containing the requested column number. Note that outer column fields may span several
*  grid columns.
* 
* 
* @param {Integer} 0-based index into the grid columns (and inner column header fields)
* @param {Integer} target header level; 0 represents the outer column header
*/
getColumnHeaderFacetValues(colNum:Integer, level:Integer): FacetValueMap;

/**
*  Get the index of the first column in the grid that matches the specified FacetValueMap.
*  
*  The facetValues passed in should contain values for at least one column facet. It may
*  contain properties other than column facets, which will be ignored. If values are 
*  sparse (values not specified for every column facet), the first column matching the specified
*  facet values will be returned.
* 
* 
* @param {FacetValueMap} facet values to find
*/
getFacetValuesColumn(facetValues:FacetValueMap): Integer;

/**
*  Get the index of the first row in the grid that matches the specified FacetValueMap.
*  
*  The facetValues passed in should contain values for at least one row facet. It may contain
*  properties other than row facets, which will be ignored. If values are 
*  sparse (values not specified for every row facet), the first row matching the specified
*  facet values will be returned.
* 
* 
* @param {FacetValueMap} facet values to find
*/
getFacetValuesRow(facetValues:FacetValueMap): Integer;

/**
*  Return a FacetValueMap indicating the facet values for a specific 
*  row in the grid.
* 
* @param {number} index of the row
*/
getRowFacetValues(rowNum:number): FacetValueMap;

/**
*  Return a FacetValueMap indicating the facet values for a specific 
*  column in the grid.
* 
* @param {number} index of the column
*/
getColumnFacetValues(colNum:number): FacetValueMap;

/**
*  Return the pointer to a particular record by record and column number.
* 
* @param {number} row index of record to return.
* @param {number} column index of record to return.
*/
getCellRecord(rowNum:number, colNum:number): ListGridRecord;

/**
*  Given a record in this grid, this method returns the rowNum on which the record is displayed.
* 
* @param {CellRecord} record to find coordinates for
*/
getCellRow(cellRecord:CellRecord): Integer;

/**
*  Given a record in this grid, this method returns the colNum in which the record is displayed.
* 
* @param {CellRecord} record to find coordinates for
*/
getCellColumn(cellRecord:CellRecord): Integer;

/**
*  Given a record in this grid, this method returns the coordinates of the cell in which the 
*  record is displayed as a 2 element array.
* 
* @param {CellRecord} record to find coordinates for
*/
getCellCoordinates(record:CellRecord): Array<any>;

/**
*  Notification method fired when new data arrives from the server to be displayed in this
*  CubeGrid. For example in response to the user openng a collapsed facet, or as a result of
*  an initial fetch request for all data from a CubeGrid where CubeGrid.facets is not
*  set and there is no initial data. Only applies to databound CubeGrids.
* 
*/
dataArrived(): void;

/**
*  This method is not currently supported for this grid-type. See 
*  ListGrid.loadAllRecords for more information.
* 
* 
* @param {Integer=} optional maximum record count - if passed, no fetch takes place                if maxRecords is below the known length of the data
* @param {DSCallback=} callback to fire if a fetch is issued - if all data was                 already loaded, the callback is fired with no parameters
*/
loadAllRecords(maxRecords?:Integer, callback?:DSCallback): boolean;

/**
*  Method handler fired when the user clicks on a facet label.
* 
* 
*/
facetLabelClick(): void;

/**
*  Get the HTML for the facet label hover. Default implementation returns null.
*  methodType callback
* 
* @param {string} ID for the facet
*/
facetLabelHoverHTML(facetId:string): void;

/**
*  Get the HTML for the facetValue button hover. Default implementation returns null.
*  
* 
* @param {FacetValueMap} facetValues object for the button
*/
facetValueHoverHTML(facetValues:FacetValueMap): void;

/**
*  Toggles the open state of the specified field. No-ops if it's not showing.
* 
* @param {FacetValueMap} field specified as a facetValueMap
*/
toggleFieldOpenState(facetValueMap:FacetValueMap): boolean;

/**
*  Return whether the specified CubeGrid field is open, taking into account both
*  Facet.canCollapse and Facet.canMinimize.
*  
*  Note that if you don't already have a FacetValueMap to the field in question, you can
*  get one by calling CubeGrid.getRowHeaderFacetValues or CubeGrid.getColumnHeaderFacetValues,
*  
*  You can also construct a FacetValueMap on your own by using the Facet.ids from
*  CubeGrid.rowFacets or CubeGrid.columnFacets together with the FacetValue.ids of the
*  Facet.values for the row or column that you want to query. Given a Facet.id,
*  you can use CubeGrid.getFacet to obtain the correponding Facet.
* 
* 
* @param {FacetValueMap} field specified as a facetValueMap
*/
isFieldOpen(facetValueMap:FacetValueMap): boolean;

/**
*  Expands the specified field. No-ops if it's not showing, or if it's already expanded.
* 
* @param {FacetValueMap} field specified as a facetValueMap
*/
expandField(facetValueMap:FacetValueMap): boolean;

/**
*  Collapses the specified field. No-ops if it's not showing, or it it's already collapsed.
* 
* @param {FacetValueMap} field specified as a facetValueMap
*/
collapseField(facetValueMap:FacetValueMap): boolean;

/**
*  auto-size the header facet horizontally
* 
* @param {string} ID of facet to resize.
*/
autoSizeFacet(facetId:string): void;

/**
* 
*  Handler fired when facet is closed
* 
*    methodType handler
* 
* @param {string} ID of facet that was closed
*/
closeFacet(facetId:string): void;

/**
* 
*  Handler fired when column is closed
* 
*    methodType handler
* 
* @param {FacetValue} FacetValues for the appropriate col.
*/
closeColumn(headerFacetValues:FacetValue): void;

/**
*  Hides the specified field if it is currently visible. No-ops if it's already hidden.
* 
* @param {FacetValueMap} field specified as a facetValueMap
*/
hideFacetValues(facetValueMap:FacetValueMap): boolean;

/**
*  Shows the specified field if it was previsouly hidden. No-ops if it's already showing.
* 
* @param {FacetValueMap} field specified as a facetValueMap
*/
showFacetValues(facetValueMap:FacetValueMap): boolean;

/**
*  Handler/Notification function for facetValue selection change (no default implementation).
* 
* 
* @param {FacetValueMap} facetValues with new selection state
* @param {boolean} new selection state
*/
facetValueSelectionChanged(facetValues:FacetValueMap, newState:boolean): void;

/**
*  Returns an array of the selected cell records.
* 
*  methodType getter
* 
*/
getSelectedCells(): Array<Partial<CellRecord>>;

/**
*  Returns an array of the IDs of all selected cell records.
* 
*  methodType getter
* 
*/
getSelectedCellIds(): Array<Partial<string>>;

/**
*  Determine whether any cells are selected in this cubeGrid.
* 
*  methodType tester
* 
*/
anyCellSelected(): boolean;

/**
*  Determine whether the cell passed in is selected in this cubeGrid.
* 
*  methodType tester
* 
* @param {CellRecord} cell to test
*/
cellIsSelected(cell:CellRecord): boolean;

/**
*  Select / deselect a single cell - accepts cell ID or CellRecord.
* 
* 
* @param {CellRecord | GlobalId} cell to select / deselect
* @param {boolean=} new selection state (if null, defaults to true)
*/
selectCell(cell:CellRecord | GlobalId, newState?:boolean): void;

/**
*  Deselect a single cell - accepts cell ID or cell record.
* 
* 
* @param {CellRecord | GlobalId} cell to deselect
*/
deselectCell(cell:CellRecord | GlobalId): void;

/**
*  Select/deselect cells that match a FacetValueMap.
*  Also supports an explicit list of CellRecords or cell IDs.
* 
* 
* @param {Array<Partial<CellRecord>> | FacetValueMap | Array<Partial<GlobalId>>} cells to select
* @param {boolean=} new selection state (if null, defaults to true)
*/
selectCells(cellList:Array<Partial<CellRecord>> | FacetValueMap | Array<Partial<GlobalId>>, newState?:boolean): void;

/**
*  Deselect cells that match a FacetValueMap.
*  Also supports an explicit list of CellRecords or cell IDs.
* 
* 
* @param {Array<Partial<CellRecord>> | FacetValueMap | Array<Partial<GlobalId>>} cells to deselect
*/
deselectCells(cellList:Array<Partial<CellRecord>> | FacetValueMap | Array<Partial<GlobalId>>): void;

/**
*  Select all cells.
* 
* 
*/
selectAllCells(): void;

/**
*  Deselect all cells.
* 
* 
*/
deselectAllCells(): void;

/**
*  Returns an array of facetValues objects indicating the headers that are selected in the
*  headerBar for this facet. If facetId is not passed, returns selection for all
*  facets.
* 
* 
* @param {string=} id for facet for which we are getting selected facetValues
*/
getSelectedFacetValues(facetId?:string): Array<Partial<FacetValueMap>>;

/**
*  Return whether the header indicated by the set of facetValues is selected.
* 
* 
* @param {FacetValueMap} facetValues to test
*/
facetValuesSelected(facetValues:FacetValueMap): boolean;

/**
*  Return whether any facet value for this facet is selected in headers. If no facetId passed,
*  return whether any facet has a selection.
* 
*  methodType tester
* 
* @param {string} Id for facet to test
*/
facetHasSelection(facetId:string): boolean;

/**
*  Return the list of facets that have any selection in their headers.
* 
*  methodType getter
* 
*/
getFacetsHavingSelection(): Array<Partial<string>>;

/**
*  Select/deselect the header showing a given set of facet values.
* 
* 
* @param {FacetValueMap} facetValues to select the header of
* @param {boolean=} new selection state - if null defaults to true
*/
selectFacetValues(facetValues:FacetValueMap, newState?:boolean): void;

/**
*  Deselect the header showing a given set of facet values.
* 
*  methodType action
* 
* @param {FacetValueMap} facetValues to deselect
*/
deselectFacetValues(facetValues:FacetValueMap): void;

/**
*  Select/deselect the header for a given facet value.
* 
*  methodType action
* 
* @param {string} ID of facet
* @param {string} ID of facetValue to select
* @param {boolean=} new selection state - if null defaults to true
*/
selectFacetValue(facetId:string, facetValueId:string, newState?:boolean): void;

/**
*  Deselect the header for a given facet value.
* 
*  methodType action
* 
* @param {string} ID of facet
* @param {string} ID of facetValue to select
*/
deselectFacetValue(facetId:string, facetValueId:string): void;

/**
*  Select/deselect all headers in a headerBar (specified by facetId) or
*  all headerBars (if no facetId).
* 
*  methodType action
* 
* @param {string=} ID of facet - if null, selects all headerbars' headers
* @param {boolean=} new selection state - if null defaults to true
*/
selectAllFacetValues(facetId?:string, newState?:boolean): void;

/**
*  Deselect all headers in a headerBar (specified by facetId) or
*  all headerBars (if no facetId).
* 
*  methodType action
* 
* @param {string=} ID of facet - if null, selects all headerbars' headers
*/
deselectAllFacetValues(facetId?:string): void;

/**
*  Deselect all cells and facetValues.
* 
*    methodType action
* 
*/
deselectAll(): void;

/**
*  Resizes all columns for the provided facetValueId, which must be a facetValueId from the
*  innermost column facet.
* 
* 
* @param {Identifier} facetValueId of columns to be resized
* @param {number} column's new width
*/
resizeFacetValue(facetValueId:Identifier, newWidth:number): void;

/**
*    Apply a hilite to a specific cell.
*    Note: can be called either as 
*      hiliteCell(cellObject, hiliteID) or 
*      hiliteCell(row, column, hiliteID)
* 
*    methodType action
* 
* @param {any | any} cell to hilite / row of cell to hilite
* @param {string | any} hilite ID / column of cell to hilite
* @param {string=} optional third parameter - hilite ID.
*/
hiliteCell(cellObj:any | any, hiliteID:string | any, alternateHiliteID?:string): boolean;

/**
*    Apply a hilite to an array of cells.
* 
*    methodType action
* 
* @param {Array<Partial<any>>} cells to hilite
* @param {string} ID of hilite to apply to cells
*/
hiliteCellList(cellObjList:Array<Partial<any>>, hiliteID:string): boolean;

/**
*    Apply a hilite to all cells corresponding to a facetValue.
* 
*    methodType action
* 
* @param {string} facet ID
* @param {string} facet value ID
* @param {string} hilite ID
*/
hiliteFacetValue(facetID:string, facetValueID:string, hiliteID:string): boolean;

/**
*  Set the title of a facet (appears in facet label).
* 
* 
* @param {Identifier} facet to update
* @param {string} title for the facet
*/
setFacetTitle(facetId:Identifier, newTitle:string): void;

/**
*  Set the align of a facet title (appears in facet label).
* 
* 
* @param {Identifier} facet to update
* @param {Alignment} new alignment for facet title
*/
setFacetTitleAlign(facetId:Identifier, align:Alignment): void;

/**
*  Set the title for a facet value.
* 
* 
* @param {Identifier} facet to update
* @param {Identifier} facetValue to update
* @param {string} title for the facet
*/
setFacetValueTitle(facetId:Identifier, facetValueId:Identifier, newTitle:string): void;

/**
*  Set the align for the title for a facet value.
* 
* 
* @param {Identifier} facet to update
* @param {Identifier} facetValue to update
* @param {Alignment} new alignment for facet value title
*/
setFacetValueTitleAlign(facetId:Identifier, facetValueId:Identifier, align:Alignment): void;

/**
*  Get the current visual order and width for the facet values of a facet or facetValueGroup as
*  an object of the form:
*  
*  [ {id:facetValueId, width:currentWidth }, ... ]
*  
* 
* 
* @param {string | string} Which facet do we want details for?
*/
getFacetValueLayout(id:string | string): Array<any>;

/**
*  Get the current widths of the row facets, as an object of the form:
*  
*  [ {facetId:facetId, width:currentWidth }, ... ]
*  
* 
* 
*/
getRowFacetLayout(): Array<any>;

/**
*  Get the current heights of the column facets, as an object of the form:
*  
*  [ {facetId:facetId, height:currentHeight}, ... ]
*  
* 
* 
*/
getColumnFacetLayout(): Array<any>;

/**
*  Setter for the CubeGrid.enableCharting attribute.
* 
* @param {boolean} */
setEnableCharting(enableCharting:boolean): void;

/**
*  Chart the portion of the dataset indicated by fixedFacetValues, for all 
*  values of the variableFacets. Note that the current CubeGrid.fixedFacetValues of
*  the CubeGrid will be implicitly added to whatever you pass as fixedFacetValues,
*  since the idea is that this chart should be taken from data available in the CubeGrid.
*  
*  One, two or more variableFacets may be passed. Two variable facets for a column chart will
*  result in Chart.stacked or clustering. Three facets or more may be
*  supported by some CubeGrid.chartType or
*  CubeGrid.chartConstructor.
*  
* 
* @param {FacetValueMap} set of facet values to hold constant. Pass null to                     chart the entire dataset.
* @param {Array<Partial<string>>} set of facets to be charted
* @param {Partial<Chart>} properties to pass through to the created            Chart
*/
makeChart(fixedFacetValues:FacetValueMap, variableFacets:Array<Partial<string>>, chartProperties:Partial<Chart>): Chart;

/**
*  Returns a default set of items, which can be updated/modified, and then assigned to
*  CubeGrid.facetValueContextItems to be used in the context menu of the appropriate 
*  header button.
* 
* 
* @param {FacetValueMap} FacetValueMap for the appropriate header button
*/
getDefaultFacetValueContextItems(facetValues:FacetValueMap): Array<Partial<MenuItem>>;

/**
*  Add a row facet to the view at index "index". Handles the facet already being in the view
*  (does a pivot).
* 
*  The facet being added should currently have a fixed facet value (unless it's already part of the
*  view), which will be removed from cubeGrid.fixedFacetValues.
* 
*  methodType action
* 
* @param {Identifier} facetId to add. Definition must have been provided at init time.
* @param {Integer=} index to add the facet at. 0 = outermost (default innermost)
*/
addRowFacet(facetId:Identifier, index?:Integer): void;

/**
*  Add a column facet to the view at index "index". Handles the facet already being in the view
*  (does a pivot).
* 
*  The facet being added should currently have a fixed facet value (unless it's already part of the
*  view), which will be removed from cubeGrid.fixedFacetValues.
* 
*  methodType action
* 
* @param {Identifier} facetId to add. Definition must have been provided at init time.
* @param {Integer=} index to add the facet at. 0 = outermost (default innermost)
*/
addColumnFacet(facetId:Identifier, index?:Integer): void;

/**
* 
*  Add a facet to the view, into the row or column facets (intoRows true or false), at index
*  "index". Handles the facet already being in the view (does a pivot).
*  
*  The facet being added should currently have a fixed facet value (unless it's already part of the
*  view), which will be removed from cubeGrid.fixedFacetValues.
* 
* 
* @param {Identifier} facetId to add. Definition must have been provided at init time.
* @param {boolean=} [true] whether to add facet as a row facet
* @param {Integer=} index to add the facet at. 0 = outermost (default innermost)
*/
addFacet(facetId:Identifier, intoRows?:boolean, index?:Integer): void;

/**
* 
*  Remove a facet from the current view, using a fixed value from that facet. For example, remove
*  the "months" facet from the view, collapsing to just January, or total for all months.
* 
* 
* @param {Identifier} facetId to remove
* @param {Identifier=} New fixed value for the facet, to be added to             CubeGrid.fixedFacetValues. Default is the rollup value for the facet.
*/
removeFacet(facetId:Identifier, fixedFacetValueId?:Identifier): void;

/**
* 
*  Modify fixedFacetValues for this cubeGrid.
* 
* 
* @param {Identifier} facetId
* @param {Identifier} New fixed value for the facet, to be added to             CubeGrid.fixedFacetValues. Default is the rollup value for the facet.
*/
setFixedFacetValue(facetId:Identifier, fixedFacetValueId:Identifier): void;

/**
*  Note that CubeGrid does not support a WYSIWYG print view by default(also used when
*  RPCManager.exportContent). Instead we recommend 
*  DataSource.exportClientData.
* 
* 
* @param {PrintProperties=} properties to configure printing behavior - may be null.
* @param {Callback=} optional callback. This is required to handle cases where HTML         generation is asynchronous - if a method generates HTML asynchronously, it should return          null, and fire the specified callback on completion of HTML generation. The          first parameter HTML should contain the generated print HTML.          The callback is only called if null is returned. Furthermore, the default          getPrintHTML() implementation always returns null and fires the callback          when a callback is provided.
*/
getPrintHTML(printProperties?:PrintProperties, callback?:Callback): HTMLString;

/**
*  Exports this component's data with client-side formatters applied, so is suitable for direct
*  display to users. This feature requires the SmartClient server.
*  
*  The export format will combine the column facet value titles, generating a single row of
*  column headers at the top with titles such as "All Years - Budget" if Time and Scenario were
*  column facets. The row facet value titles for separate facets won't be combined, so that
*  each row facet will have a separate column, with the facet titles at the top in the 
*  "column header" row, and the row facet value titles below their corresponding facet title.
*  Data values each get their own row and column position.
* 
* 
* @param {object} contains configuration settings for the export, including:  
- facetSeparatorString (String) - if specified, the separator to use in favor   of CubeGrid.exportFacetSeparatorString when combining titles from multiple   facet values. 
* @param {Partial<DSRequest>=} Request properties for the export.
* @param {RPCCallback=} Optional callback. If you specify DSRequest.exportToClient: false in the request  properties, this callback will fire after export completes. Otherwise the callback will  fire right before the download request is made to the server.
*/
exportClientData(settings:object, requestProperties?:Partial<DSRequest>, callback?:(response:RPCResponse, rawData:any, request:RPCRequest) => void): void;

/**
*  StringMethod handler fired when mouseover occurs over a facetValues header button
*  
* 
* @param {FacetValueMap} FacetValueMap for the appropriate header button
*/
facetValueOver(facetValues:FacetValueMap): void;

/**
*  StringMethod handler fired when mouseout occurs for a facetValues header button
* 
* @param {FacetValueMap} FacetValueMap for the appropriate header button
*/
facetValueOut(facetValues:FacetValueMap): void;

/**
*  StringMethod handler fired when mouse hovers over a facetValue button in a header.
* 
* @param {FacetValueMap} FacetValueMap for the appropriate header button
*/
facetValueHover(facetValues:FacetValueMap): void;

/**
*  StringMethod handler fired when context click occurs over a facetValue button in a header.
* 
* 
* @param {FacetValueMap} FacetValueMap describing the appropriate header button
*/
facetValueContextClick(facetValues:FacetValueMap): void;

/**
*  StringMethod handler fired when the user right clicks on a facet label.
* 
* 
* @param {string} ID of facet
*/
facetContextClick(facetId:string): void;

/**
*  
*  Notification fired when a facet or facetValueGroup is reordered
* 
* 
* @param {string} facetValueGroupId or facetId
*/
facetValueReordered(groupId:string): void;

/**
*    Notification fired when a new facet is added.
*  
* 
* @param {string} facetId that was added
*/
facetAdded(facetId:string): void;

/**
*  Notification fired when a facet is removed.
*  
* 
* @param {string} facetId that was removed
*/
facetRemoved(facetId:string): void;

/**
*  Notification fired when a facet is moved.
*  
* 
* @param {string} facetId which moved
*/
facetMoved(facetId:string): void;

/**
*  Notification fired when a fixed facet value is set for some facet.
* 
* @param {string} facetId
* @param {string} new fixed facet value
*/
fixedFacetValueChanged(facetId:string, facetValueId:string): void;

/**
*  Called when a sort control is clicked on a FacetValueHeader. Does nothing by default.
*  
* 
* @param {FacetValueMap} facetValues to sort
* @param {boolean} true for ascending
*/
sortByFacetValues(facetValues:FacetValueMap, sortDirection:boolean): void;

/**
*  Called when a sort control is clicked on a FacetHeader. Does nothing by default.
*  
* 
* @param {string} ID of facet to sort
* @param {boolean} true for ascending
*/
sortByFacetId(facetId:string, sortDirection:boolean): void;

/**
*  StringMethod handler fired when mouseover occurs over a facet label
* 
* 
* 
* @param {string} ID of the appropriate facet
*/
facetLabelOver(facetId:string): void;

/**
*  StringMethod handler fired when mouseout occurs over a facet label
* 
* 
* @param {string} ID of the appropriate facet
*/
facetLabelOut(facetId:string): void;

/**
*  StringMethod handler fired from hover over a facet label
* 
* 
* @param {string} ID of the appropriate facet
*/
facetLabelHover(facetId:string): void;

/**
* 
*    Handler/Notification function for cell selection change
*    May update header button styles.
* 
*    methodType handler
* 
* @param {Array<Partial<CellRecord>>} array of cells with new selection state
*/
cellSelectionChanged(cellList:Array<Partial<CellRecord>>): boolean;



}
/**
* Creates a new CubeGrid
* 
* @param typeCheckedProperties {Partial<CubeGrid>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {CubeGrid} The new instance
*/
class CubeGrid{
static create(typeCheckedProperties? : Partial<CubeGrid>, uncheckedProperties?: Object): CubeGrid;
/**
* A declared value of the enum type 
*  FacetIndentDirection.
* @type {Constant}
* @default "descending"
*/
static DESCENDING?: Constant;

/**
* A declared value of the enum type 
*  FacetIndentDirection.
* @type {Constant}
* @default "ascending"
*/
static ASCENDING?: Constant;








}

/**
* Static singleton class with APIs for dmiOverview of
*  server side methods when running the SmartClient java server.
*/
interface DMI extends Class {




}
/**
* Creates a new DMI
* 
* @param typeCheckedProperties {Partial<DMI>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DMI} The new instance
*/
class DMI{
static create(typeCheckedProperties? : Partial<DMI>, uncheckedProperties?: Object): DMI;




/**
* 
*  Calls a server-side DMI method. At a minimum, you need to specify the appID 
*  (applicationDeclaration), ServerObject.className
*  or ServerObject.ID and methodName to call.
*  Arguments and callback are optional. There are two ways to invoke this method:
*  
*  DMI.call(appID, className, methodName, 
*      arg1, arg2 ...argN, callback);
*  
*  or:
*  
*  DMI.call({
*    appID: appID,
*    className: className,
*    methodName: methodName,
*    arguments: [arg1, arg2, ...argN], //optional
*    callback: callback, //optional
*    requestParams: requestProps // optional
*  });
*  
*  If you use the first signature, you must either specify a callback or if you don't want a
*  callback, pass a null as the last argument. The second signature allows you to
*  specify requestParams that are applied to the RPCRequest generated by this DMI call.
*  This allows you to override some defaults - for example to suppress the "Contacting Server"
*  prompt, change it's text; change the timeout or set any other property settable on
*  RPCRequest.
*  
*  Arguments to be passed to the server method may be specified.
*  If present, each argument will be serialized into JSON, 
*  sent to the server and translated to an equivalent Java object to be passed into the
*  method. The translation follows the same rules as for data passed to the server as 
*  part of a standard RPCRequest.data. 
*  For example if a server side method has the signature
*  
*  someMethod(String title, Map overrides);
*  
*  a DMI call to that method would could pass in a standard String and JavaScript object -
*  for example:
*  
*  isc.DMI.call({
*    appID: "someApp",
*    className: "com.smartclient.demo.MyClass",
*    methodName: "someMethod",
*    arguments: ["Title String", {field1:"Value 1", field2:"Value 2"}]
*  });
*  
*  
*  Note that you can use the loadDMIStubsTag tag to bind all methods of
*  ServerObjects defined in a given .app.xml file and call methods on them
*  directly.
* 
* 
* @param {string | object} the appID (.app.xml file to look in) or comprehensive request             object as documented above.
* @param {string} ServerObject.className or ServerObject.ID
* @param {string} the name of the method to call on the serverObject
* @param {any} The next N-1 params specify arguments to the server-side method.
* @param {RPCCallback} The callback of the response. If you do not want a callback, you             must specify a null value for this parameter when              using the first signature (documented above).
*/
static call(appID:string | object, className:string, methodName:string, args:any, callback:(response:RPCResponse, rawData:any, request:RPCRequest) => void): RPCRequest;

/**
* 
*  Returns a URL to a server-side DMI method. At a minimum, you need to specify the appID 
*  (applicationDeclaration), ServerObject.className
*  or ServerObject.ID and methodName to call.
*  Arguments are optional. There are two ways to invoke this method:
*  
*  DMI.getURL(appID, className, methodName, 
*      arg1, arg2 ...argN);
*  
*  or:
*  
*  DMI.getURL({
*    appID: appID,
*    className: className,
*    methodName: methodName,
*    arguments: [arg1, arg2, ...argN], //optional
*    requestParams: requestProps // optional
*  });
*  
*  The second signature allows you to specify requestParams that are applied to the
*  RPCRequest generated by this DMI call.
*  
*  Note that because the entirety of the request is encoded in the URL, there is an
*  inherent limitation on the amount of data that you can send viat he criteria argument to
*  the server. The actual length depends on your server configuration and other factors
*  such as the size of cookies (if any) being sent to the server and other HTTP headers in
*  use. Conservatively, assume that you have about 2 kilobytes to work with.
* 
* 
* @param {string | object} the appID (.app.xml file to look in) or comprehensive                     request object as documented above.
* @param {string} ServerObject.className or ServerObject.ID
* @param {string} the name of the method to call on the serverObject
* @param {any=} The next N-1 params specify arguments to the server-side method.
*/
static getURL(appID:string | object, className:string, methodName:string, args?:any): string;




}

/**
* Provides a tree view of any DOM-compliant structure, such as an XML or HTML document.
*/
interface DOMGrid extends TreeGrid {
/**
* Root element (or document) to view in the tree.
* @type {DOMElement}
* @default null
*/
rootElement?: DOMElement;



/**
*  Set the root element (or document) to view in the tree.
* 
* @param {DOMElement} new root element
*/
setRootElement(rootElement:DOMElement): void;



}
/**
* Creates a new DOMGrid
* 
* @param typeCheckedProperties {Partial<DOMGrid>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DOMGrid} The new instance
*/
class DOMGrid{
static create(typeCheckedProperties? : Partial<DOMGrid>, uncheckedProperties?: Object): DOMGrid;







}

/**
* A ServiceTask configured to perform an add.
*/
interface DSAddTask extends ServiceTask {




}
/**
* Creates a new DSAddTask
* 
* @param typeCheckedProperties {Partial<DSAddTask>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DSAddTask} The new instance
*/
class DSAddTask{
static create(typeCheckedProperties? : Partial<DSAddTask>, uncheckedProperties?: Object): DSAddTask;







}

/**
* A ServiceTask configured to perform a fetch.
*/
interface DSFetchTask extends ServiceTask {




}
/**
* Creates a new DSFetchTask
* 
* @param typeCheckedProperties {Partial<DSFetchTask>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DSFetchTask} The new instance
*/
class DSFetchTask{
static create(typeCheckedProperties? : Partial<DSFetchTask>, uncheckedProperties?: Object): DSFetchTask;







}

/**
* A ServiceTask configured to perform a remove.
*/
interface DSRemoveTask extends ServiceTask {




}
/**
* Creates a new DSRemoveTask
* 
* @param typeCheckedProperties {Partial<DSRemoveTask>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DSRemoveTask} The new instance
*/
class DSRemoveTask{
static create(typeCheckedProperties? : Partial<DSRemoveTask>, uncheckedProperties?: Object): DSRemoveTask;







}

/**
* Response sent by the server in response to a DSRequest. Contains
*  all the properties available on the basic RPCResponse, in addition to the
*  properties listed here.
*/
interface DSResponse extends RPCResponse {
/**
* The DSRequest.clientContext object as set on the DSRequest.
* @type {object}
* @default null
*/
clientContext?: object;

/**
* The DataSource of this DSResponse.
* @type {string}
* @default null
*/
dataSource?: string;

/**
* The operation type of the request corresponding to this DSResponse.
* @type {DSOperationType}
* @default null
*/
operationType?: DSOperationType;

/**
* The operation ID of the request corresponding to this DSResponse.
* @type {string}
* @default null
*/
operationId?: string;

/**
* For "fetch" operations, this is the array of Records fetched. For "update", "add", and
*  "remove" operations, this is typically an array containing a single Record representing
*  the record that was updated, added, or removed.
* @type {Array<Partial<Record>> | any}
* @default null
*/
data?: Array<Partial<Record>> | any;

/**
* Same meaning as RPCResponse.status, except DSResponses have additional error codes,
*  such as RPCResponse.STATUS_VALIDATION_ERROR.
* @type {Integer}
* @default "see below"
*/
status?: Integer;

/**
* An extra property of each DSResponse to a queued request that indicates whether the queue
*  as a whole succeeded. A queueStatus of RPCResponse.STATUS_SUCCESS, or 0, indicates
*  that the queue succeeded whereas a queueStatus of RPCResponse.STATUS_FAILURE, or -1,
*  indicates that the queue failed.
* 
*  For example, if two "update" requests are sent in a queue and the first succeeded, but
*  the second failed validation, then both DSResponses' queueStatus would be -1, but the
*  DSResponse.status of the first would be RPCResponse.STATUS_SUCCESS and
*  the status of the second would be an error code such as RPCResponse.STATUS_VALIDATION_ERROR.
* @type {Integer}
* @default "see below"
*/
queueStatus?: Integer;

/**
* Server-side validation errors for an attempted "update" or "add" operation, as a 
*  JS Object where each property name is a field name from the record and each property
*  value contains error information.
*  
*  To extract just the simple error strings for each field we recommend passing this
*  object to DataSource.getSimpleErrors
*  
*  The Java API DSResponse.addError(fieldName, errorMessage) is used to send server-side
*  errors to the client. See the Java Server Reference for details.
* @type {object}
* @default null
*/
errors?: object;

/**
* Starting row of returned server results, when using paged result fetching
*  
*  Note that startRow and endRow are zero-based, inclusive at the beginning and exclusive at
*  the end (like substring), so startRow: 0, endRow: 2 is a response containing two records.
* @type {number}
* @default null
*/
startRow?: number;

/**
* End row of returned server results, when using paged result fetching
*  
*  Note that startRow and endRow are zero-based, inclusive at the beginning and exclusive at
*  the end (like substring), so startRow: 0, endRow: 2 is a response containing two records.
* @type {number}
* @default null
*/
endRow?: number;

/**
* Total number of rows available from the server that match the current filter criteria,
*  when using paged result fetching.
* @type {number}
* @default null
*/
totalRows?: number;

/**
* Optional flag that can be set by the server to force ResultSets to drop any caches of
*  records from the DataSource that was the target of the operation.
* @type {boolean}
* @default null
*/
invalidateCache?: boolean;

/**
* Timestamp (millisecond value) to indicate when this dsResponse was cached in 
*  Offline. Not applicable if the response has never been 
*  stored offline.
* @type {number}
* @default null
*/
offlineTimestamp?: number;

/**
* If set, indicates that this response came from the offline cache, not the server. This 
*  flag is the only reliable way for application code to determine the source of a response.
* @type {boolean}
* @default null
*/
fromOfflineCache?: boolean;

/**
* HTTP headers returned by the server as a map from header name to header value.
*  
*  Headers are available only when the default RPCTransport "xmlHttpRequest" is in use,
*  and browsers may limit access to headers for cross-domain requests or in other
*  security-sensitive scenarios.
* @type {object}
* @default null
*/
httpHeaders?: object;





}
/**
* Creates a new DSResponse
* 
* @param typeCheckedProperties {Partial<DSResponse>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DSResponse} The new instance
*/
class DSResponse{
static create(typeCheckedProperties? : Partial<DSResponse>, uncheckedProperties?: Object): DSResponse;







}

/**
* A ServiceTask configured to perform a update.
*/
interface DSUpdateTask extends ServiceTask {




}
/**
* Creates a new DSUpdateTask
* 
* @param typeCheckedProperties {Partial<DSUpdateTask>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DSUpdateTask} The new instance
*/
class DSUpdateTask{
static create(typeCheckedProperties? : Partial<DSUpdateTask>, uncheckedProperties?: Object): DSUpdateTask;







}

/**
* TextItem subclass for managing a DataPath
*/
interface DataPathItem extends TextItem {




}
/**
* Creates a new DataPathItem
* 
* @param typeCheckedProperties {Partial<DataPathItem>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DataPathItem} The new instance
*/
class DataPathItem{
static create(typeCheckedProperties? : Partial<DataPathItem>, uncheckedProperties?: Object): DataPathItem;







}

/**
* A DataSource is data-provider-independent description of a set of objects
*  that will be loaded, edited and saved within the user interface of your application.
*  
*  Each DataSource consists of a list of DataSource.fields that make up a
*  DataSource record, along with DataSourceField.type, 
*  DataSourceField.validators, 
*  DataSourceField.foreignKey to other DataSources, and other
*  metadata.
*  
*  The abstract object description provided by a DataSource is easily mapped to a variety of
*  backend object models and storage schemes. The following table shows analogous terminology
*  across systems.
*  
*  
*   Isomorphic SmartClient
*   Relational Database
*   Enterprise Java Beans (EJB)
*   Entity/Relationship Modeling
*   OO/UML
*   XML Schema/WSDL
*   LDAP
*  
*   DataSource
*   Table
*   EJB class
*   Entity
*   Class
*   Element Schema (ComplexType)
*   Objectclass
*  
*   Record
*   Row
*   EJB instance
*   Entity instance
*   Class instance/Object
*   Element instance (ComplexType)
*   Entry
*  
*   Field
*   Column
*   Property
*   Attribute
*   Property/Attribute
*   Attribute or Element (SimpleType)
*   Attribute
*  
*  
*  DataSources can be dataSourceDeclaration in either JavaScript or XML
*  format, and can also be metadataImport from existing metadata formats,
*  including XML Schema.
*  
*  Data Binding is the process by which
*  DataBoundComponent can automatically configure
*  themselves for viewing, editing and saving data described by DataSources. DataBinding is
*  covered in the ${isc.DocUtils.linkForDocNode('QuickStartGuide', 'QuickStart Guide')}, Chapter 6, Data
*  Binding.
*  
*  clientServerIntegration is the process by which a DataSource
*  can be connected to server systems such as SQL DataBases, Java Object models, WSDL web
*  services and other data providers. Data Integration comes in two variants: client-side and
*  server-side. serverDataIntegration uses the
*  SmartClient Java-based server to connect to data represented by Java Objects or
*  JDBC-accessible databases. clientDataIntegration
*  connects SmartClient DataSources to XML, JSON or other formats accessible via HTTP.
*  
*  DataSources have a concept of dataSourceOperations ("fetch",
*  "add", "update" and "remove") that can be performed on the set of objects represented by a
*  DataSource. Once a DataSource has been integrated with your data store, databinding-capable 
*  UI components can leverage the 4 core DataSource operations to provide many complete user
*  interactions without the need to configure how each individual component loads and saves
*  data.
*  
*  These interactions include ListGrid, TreeGrid, 
*  DetailViewer, DynamicForm-based 
*  DynamicForm.editRecord and DynamicForm.saveData,
*  grid-based ListGrid.canEdit and ListGrid.saveByCell,
*  and custom interactions provided by ${isc.DocUtils.linkForExampleId('patternReuse')} custom
*  databinding-capable components.
*/
interface DataSource extends Class {
/**
* Unique identifier for this DataSource. Required for all DataSources.
* 
*  DataSources will make themselves available as JavaScript globals under
*  the same name as their ID only if DataSource.addGlobalId is set.
* @type {string}
* @default null
*/
ID?: string;

/**
* Whether to make this DataSource available as a global variable for convenience.
* @type {boolean}
* @default true
*/
addGlobalId?: boolean;

/**
* Indicates the format to be used for HTTP requests and responses when
*  fulfilling DSRequests (eg, when DataSource.fetchData is called).
* @type {DSDataFormat}
* @default "iscServer"
*/
dataFormat?: DSDataFormat;

/**
* Should HTTP responses to requests by this dataSource be formatted using the
*  strict JSON subset of the javascript language?
*  If set to true, responses returned by the server should match the format described
*  here.
*  
*  Only applies to dataSources which send requests to a server and have
*  DataSource.dataFormat set to "json" or "iscServer".
*  
*  Note: using strict JSON avoids a known issue in Internet Explorer 9 where
*  datasource transactions can leak memory due to a browser behavior where the native
*  eval() method fails to clean up references when the objects
*  go out of scope. See RPCManager.allowIE9Leak for more on this.
* @type {boolean}
* @default null
*/
useStrictJSON?: boolean;

/**
* Controls the format in which inputs are sent to the dataURL when fulfilling DSRequests.
*  May be overridden for individual request types using
*  OperationBinding.dataProtocol.
* @type {DSProtocol}
* @default null
*/
dataProtocol?: DSProtocol;

/**
* Like OperationBinding.useHttpProxy, but serves as a default for this DataSource
*  that may be overridden by individual operationBindings.
* @type {boolean}
* @default null
*/
useHttpProxy?: boolean;

/**
* Applies only to dataFormat: "json" and DataSource.dataTransport:"scriptInclude". Specifies
*  the name of the query parameter that tells your JSON service what function to call as
*  part of the response.
* @type {string}
* @default "callback"
*/
callbackParam?: string;

/**
* Additional properties to pass through to the DSRequests made by this
*  DataSource. This must be set before any DSRequests are issued and before
*  any component is bound to the DataSource.
*  
*  These properties are applied before DataSource.transformRequest is called.
* @type {Partial<DSRequest>}
* @default null
*/
requestProperties?: Partial<DSRequest>;

/**
* Transport to use for all operations on this DataSource. Defaults to
*  RPCManager.defaultTransport. This would typically only be set to
*  enable "scriptInclude" transport for contacting DataSource.dataFormat web
*  services hosted on servers other than the origin server.
*  
*  When using the "scriptInclude" transport, be sure to set DataSource.callbackParam or
*  OperationBinding.callbackParam to match the name of the query parameter name expected
*  by your JSON service provider.
* @type {RPCTransport}
* @default "xmlHttpRequest"
*/
dataTransport?: RPCTransport;

/**
* If set, the DataSource will ensure that it never uses a cached HTTP response, even if the
*  server marks the response as cacheable.
*  
*  Note that this does not disable caching at higher levels in the framework, for example, the
*  caching performed by ResultSet.
* @type {boolean}
* @default true
*/
preventHTTPCaching?: boolean;

/**
* For a DataSource stored in .xml format on the SmartClient server, indicates what server-side
*  connector to use to execute requests, that is, what happens if you call
*  dsRequest.execute() in server code.
* @type {DSServerType}
* @default "generic"
*/
serverType?: DSServerType;

/**
* For DataSources using the sqlDataSource for
*  persistence, what database table name to use. The default is to use the DataSource ID
*  as the table name.
* @type {string}
* @default null
*/
tableName?: string;

/**
* This property only applies to the built-in SQL DataSource provided in Pro and better 
*  editions of SmartClient
*  
*  Defines the name of the schema we use to qualify the DataSource.tableName in generated
*  SQL. If you do not provide this property, table names will not be qualified in generated 
*  SQL, and thus the default schema will be used. Support for multiple schemas (or schemata)
*  varies quite significantly across the supported databases, as does the meaning of the 
*  phrase "default schema". In addition, some databases allow you to override the default 
*  schema in the JDBC connection URL, which is a preferable approach if all your tables are 
*  in the same (non-default) schema.
*  
*  The following table provides information by product:
*  
*  
*  Product Notes
*  DB2
*  Arbitrarily named schemas are supported. The default schema is named after the connecting
*  user, though this can be overridden by specifying the "currentSchema" property on the JDBC
*  connection URL
*  DB2 for iSeries
*  Arbitrarily named schemas are supported. "Schema" is synonymous with "library". The 
*  default schema depends on the setting of the "naming" connection property. When this is 
*  set to "sql", behavior is similar to other DB2 editions: the default schema is named after
*  the connecting user, unless overridden by specifying a library name in the JDBC connection 
*  URL. When "naming" is set to "system", the schema of an unqualified table is resolved 
*  using a traditional search of the library list; the library list can be provided in the 
*  "libraries" property
*  Firebird
*  Firebird does not support the concept of schema at all - all "schema objects" like tables 
*  and indexes belong directly to the database. In addition, Firebird actively rejects 
*  qualified table names in queries as syntax errors; therefore, you should not set
*  the schema property for a DataSource that will be backed by a Firebird database
*  
*  HSQLDB
*  Arbitrarily named schemas are supported. The default schema is auto-created when the
*  database is created; by default it is called "PUBLIC", but can be renamed. It is not 
*  possible to set the default schema in the JDBC connection URL
*  Informix
*  Informix databases can be flagged as "ANSI mode" at creation time. ANSI-mode databases 
*  behave similarly to DB2 for schema support: arbitrarily named schemas are supported, and
*  the default schema is the one named after the connected user. Non-ANSI databases have no
*  real schema support at all. It is not possible to set the default schema in the JDBC 
*  connection URL with either type of database
*  Microsoft SQL Server
*  Prior to SQL Server 2005, schema support is similar to Oracle: "schema" is synonymous with 
*  "owner". As of SQL Server 2005, schema is supported as a separate concept, and a user's 
*  default schema can be configured (though it still defaults to a schema with the same name
*  as the user). It is not possible to set the default schema in the JDBC connection URL
*  MySQL
*  MySQL does not have a separate concept of "schema"; it treats the terms "schema" and 
*  "database" interchangeably. In fact MySQL databases actually behave more like schemas, in 
*  that a connection to database X can refer to a table in database Y simply by qualifying 
*  the name in the query. Also, because schema and database are the same concept in MySQL, 
*  overriding the "default schema" is done implicitly when you specify which database to 
*  connect to in your JDBC connection URL
*  Oracle
*  Arbitrarily named schemas are not supported; in Oracle, "schema" is synonymous with "user",
*  so each valid user in the database is associated implicitly with a schema of the same name,
*  and there are no other schemas possible. It is possible to refer to tables in another
*  user's schema (assuming you have the privileges to do so) by simply qualifying the table name.
*  The default schema is always implied by the connecting user and cannot be overridden.
*  Postgres
*  Arbitrarily named schemas are supported.
*  Rather than the concept of a "default schema", Postgres supports the idea of a search path 
*  of schemas, whereby unqualified table references cause a search of the list of schemas in 
*  order, and the first schema in the path is the "current" one for creation purposes. 
*  Unfortunately, there is no way to specify this search path on the JDBC connection URL, so
*  the default schema comes from the user definition, ultimately defaulting to the default 
*  "public" schema
*  
* @type {string}
* @default null
*/
schema?: string;

/**
* For SQL DataSources, tells the framework whether to surround the associated 
*  DataSource.tableName with quotation marks whenever it appears in
*  generated queries. This is only required if you have to connect to a table with a name
*  that is in breach of your database product's naming conventions. For example, some 
*  products (eg, Oracle) internally convert all unquoted references to upper case, so if
*  you create a table called myTest, the database actually calls it 
*  MYTEST unless you quoted the name in the create command, like this:
*  
*  &nbsp;&nbsp;CREATE TABLE "myTest"
*  
*  If you do quote the name like this, or if you have to connect to a legacy table
*  that has been named in this way, then you must set this property to tell the SQL engine 
*  that it must quote any references to this table name (this requirement depends on the 
*  database in use - as noted below, some are not affected by this problem). If you do 
*  not, you will see exceptions along the lines of "Table or view 'myTest' does not exist".
*  
*  Note, other database products (eg, Postgres) convert unquoted names to lower case, which
*  leads to the same issues. Still others (eg, SQL Server) are not case sensitive and are 
*  not affected by this issue.
*  
*  Generally, we recommend that you avoid using this property unless you have a specific 
*  reason to do so. It is preferable to avoid the issue altogether by simply not quoting 
*  table names at creation time, if you are able to do so.
* @type {boolean}
* @default null
*/
quoteTableName?: boolean;

/**
* For DataSources using the sqlDataSource for
*  persistence, which database configuration to use. Database configurations can be
*  created using the adminConsole. If unset, the default
*  database configuration is used (which is also settable using the "Databases" tab).
* @type {string}
* @default null
*/
dbName?: string;

/**
* For DataSources of DataSource.serverType "hibernate", the name of a Spring 
*  bean to query to obtain Hibernate Configuration for this particular DataSource. Note
*  that this is intended for DataSource-specific configuration overrides for unusual 
*  circumstances, such as a DataSource whose physical data store is a completely 
*  different database to that used by other DataSources. See the 
*  hibernateIntegration article for more 
*  information
* @type {string}
* @default null
*/
configBean?: string;

/**
* If set, tells the SQL engine to quote column names in all generated DML and DDL 
*  statements for this dataSource. This will ensure that queries generated against
*  tables that do not follow the database product's natural column-naming conventions 
*  will still work.
*  
*  In general we recommend that you allow the database to use its natural naming scheme
*  when creating tables (put more simply, just do not quote column names in the 
*  CREATE TABLE statement); if you do this, you will not need to worry about 
*  quoting column names when querying. However, if you are dealing with pre-existing 
*  tables, or do not have control over the database naming conventions used, this property
*  may become necessary. This property may also be necessary if you are using field/column
*  names that clash with reserved words in the underlying database (these vary by database,
*  but a field called "date" or "timestamp" would have problems with most database products)
*  
*  Note: Only applicable to dataSources of DataSource.serverType "sql".
* @type {boolean}
* @default null
*/
quoteColumnNames?: boolean;

/**
* Allows you to specify an arbitrary prefix string to apply to all json format responses 
*  sent from the server to this application.
*  
*  The inclusion of such a prefix ensures your code is not directly executable outside of
*  your application, as a preventative measure against
*  javascript hijacking.
*  
*  Only applies to responses formatted as json objects. Does not apply to responses returned
*  via scriptInclude type transport.
*  Note: If the prefix / suffix served by your backend is not a constant, you can use 
*  OperationBinding.dataFormat instead and explicitly parse
*  the prefix out as part of DataSource.transformResponse.
* @type {string}
* @default null
*/
jsonPrefix?: string;

/**
* Allows you to specify an arbitrary suffix string to apply to all json format responses 
*  sent from the server to this application.
*  
*  The inclusion of such a suffix ensures your code is not directly executable outside of
*  your application, as a preventative measure against
*  javascript hijacking.
*  
*  Only applies to responses formatted as json objects. Does not apply to responses returned
*  via scriptInclude type transport.
* @type {string}
* @default null
*/
jsonSuffix?: string;

/**
* The list of fields that compose records from this DataSource.
*  
*  Each DataSource field can have type, user-visible title, validators, and other metadata
*  attached.
*  
*  
*  After a DataSource has been Class.create, access the list of fields via
*  DataSource.getFieldNames and access individual fields via
*  DataSource.getField.
*  
* @type {Array<Partial<DataSourceField>>}
* @default null
*/
fields?: Array<Partial<DataSourceField>>;

/**
* ID of another DataSource this DataSource inherits its DataSource.fields from.
*  
*  Local fields (fields defined in this DataSource) are added to inherited fields 
*  to form the full set of fields. Fields with the same name are merged in the same way
*  that DataBoundComponent.fields are merged with
*  DataSource fields.
*  
*  The default order of the combined fields is new local fields first (including any fields
*  present in the parent DataSource which the local DataSource re-declares), then parent
*  fields. You can set DataSource.useParentFieldOrder to instead use the parent's field
*  order, with new local fields appearing last. You can set DataSource.showLocalFieldsOnly to
*  have all non-local fields hidden.
*  
*  Note that only fields are inherited - other properties such as dataURL and
*  dataFormat are not. You can use ordinary inheritance, that is, creating a subclass of
*  DataSource, in order to share properties such as dataURL across a series of DataSources
*  that also inherit fields from each other via inheritsFrom.
*  
*  This feature can be used for:
*  
*  
* - creating a customized view (eg, only certain fields shown) which will be used by
*  multiple DataBoundComponent.
*  
* - adding presentation-specific attributes to metadata that has been automatically
*  derived from XMLTools.loadXMLSchema or other metadata formats
*  
* - modeling object subclassing and extension in server-side code and storage systems
*  
* - modeling relational database joins, and the equivalents in other systems
*  
* - creating hooks for others to customize your application in a maintainable way. For
*  example, if you have a dataSource "employee", you can create a dataSource
*  "customizedEmployee" which inherits from "employee" but does not initially define any
*  fields, and bind all DataBoundComponent to
*  "customizedEmployee". Customizations of fields (including appearance changes, field
*  order, new fields, hiding of fields, and custom validation rules) can be added to
*  "customizedEmployee", so that they are kept separately from the original field data and
*  have the best possible chance of working with future versions of the "employee"
*  dataSource.
*  
* @type {string}
* @default null
*/
inheritsFrom?: string;

/**
* For audit DataSources, this required property specifies the ID of the
*  DataSource.audit DataSource. Automatically populated for
*  DataSource.generateAuditDS audit DataSources.
* @type {string}
* @default "varies"
*/
auditedDataSourceID?: string;

/**
* For an DataSource.audit DataSource, controls whether the Framework will
*  attempt to auto-generate the audit DataSource. Note that this property is independent of
*  DataSource.auditDataSourceID so that, by default, even when the audit DataSource is given
*  a non-default ID, the Framework will still attempt to auto-generate it.
* @type {boolean}
* @default true
*/
generateAuditDS?: boolean;

/**
* For dataSources of DataSource.serverType "sql" and "hibernate", specifies the inheritance 
*  mode to use. This property has no effect for any other type of DataSource.
* @type {DSInheritanceMode}
* @default "full"
*/
inheritanceMode?: DSInheritanceMode;

/**
* For fields of DataSourceField.type "sequence" in a dataSource of 
*  DataSource.serverType "sql", indicates the 
*  SequenceMode to use. This property has no effect for fields or dataSources of 
*  other types.
*  
*  You can set a default sequenceMode for all DataSources of a given database type by setting
*  property "sql.{database_type}.default.sequence.mode" in server.properties. 
*  You set a global default sequenceMode that applies to all database types by setting property
*  "sql.default.sequence.mode". For example:
*   sql.mysql.default.sequence.mode: jdbcDriver
*  
*  
* @type {SequenceMode}
* @default "native"
*/
sequenceMode?: SequenceMode;

/**
* Like DataBoundComponent.useFlatFields, but applies to all DataBound components
*  that bind to this DataSource.
* @type {boolean}
* @default null
*/
useFlatFields?: boolean;

/**
* For a DataSource that inherits DataSource.fields from another DataSource 
*  (via DataSource.inheritsFrom), indicates that only the fields listed in this DataSource
*  should be shown. All other inherited parent fields will be marked "hidden:true".
* @type {boolean}
* @default null
*/
showLocalFieldsOnly?: boolean;

/**
* For a DataSource that inherits DataSource.fields from another DataSource 
*  (via DataSource.inheritsFrom), indicates that the parent's field order should be used
*  instead of the order of the fields as declared in this DataSource. New fields, if any,
*  are placed at the end.
* @type {boolean}
* @default null
*/
useParentFieldOrder?: boolean;

/**
* Indicates that for server responses, for any data being interpreted as DataSource records, 
*  only data that
*  corresponds to declared fields should be retained; any extra fields should be discarded.
*  
*  For DataSource.dataFormat data, this means extra properties in selected
*  objects are dropped.
*  
*  By default, for DMI DSResponses, DSResponse.data is filtered on the server to just
*  the set of fields defined on the DataSource. This type of filtering can also be enabled
*  for non-DMI DSResponses (see the overview in dmiOverview). Setting this property to
*  false disables this filtering for this DataSource only. This setting
*  overrides the configuration in server_properties. This setting can
*  be overridden by ServerObject.dropExtraFields.
* @type {boolean}
* @default null
*/
dropExtraFields?: boolean;

/**
* Analogous to DataSource.dropExtraFields, for data sent to the server. Setting this
*  attribute to false ensures that for any records in the data object, only fields that
*  correspond to declared dataSource fields will be present on the dsRequest data object passed
*  to DataSource.transformRequest and ultimately sent to the server.
* @type {boolean}
* @default true
*/
sendExtraFields?: boolean;

/**
* This property allows you to specify that your DataSource's schema (field definitions) should
*  be automatically derived from some kind of metadata. This causes SmartClient to create 
*  a special super-DataSource, which is used purely as a source of default schema for this 
*  DataSource; this is arranged by causing the autoDerived DataSource to automatically 
*  DataSource.inheritsFrom the special super-DataSource. This allows you to 
*  auto-derive schema from existing metadata, whilst still being able to override any or all
*  of it as required.
*  This property has a different implementation depending upon the 
*  DataSource.serverType of the DataSource:
*  
* - For a DataSource with serverType: "sql", automatically derive the dataSource's schema 
*  from the Spring bean or Java class specified in DataSource.schemaBean. If 
*  schemaBean is not specified, derive the schema from the columns in the SQL 
*  table specified in DataSource.tableName. More information on SQL DataSources 
*  is sqlDataSource 
*  
* - For a DataSource with serverType: "hibernate", automatically derive the dataSource's
*  schema from the Spring bean, Hibernate mapping or Java class named in the 
*  DataSource.schemaBean property. If no such thing exists, derive the schema from
*  the Hibernate mapping or Java class specified in the DataSource.beanClassName
*  property (this allows you to specify schema and mapping separately, in the unusual 
*  circumstance that you have a need to do so). Note that the "mappings" referred to here
*  can mean either .hbm.xml files or annotated classes; both are supported.
*  If neither of these is successful, derive the schema from the underlying SQL table 
*  specified in DataSource.tableName. More information on Hibernate DataSources is
*  hibernateIntegration 
*  
* - For a DataSource with serverType: "jpa1" or "jpa", automatically derive the 
*  dataSource's schema from the annotated JPA class named in the DataSource.schemaBean 
*  property. If the schemaBean property is not defined, derive the schema from the 
*  annotated JPA class named in the DataSource.beanClassName property (as with 
*  Hibernate, this allows you to specify schema and mapping separately if you need to do 
*  so). JPA DataSource generation relies on annotations (the orm.xml mapping file is not 
*  supported). More information on JPA DataSources is jpaIntegration 
*  
* - For other DataSource types, attempt to find a Spring bean with the name specified in
*  the DataSource.schemaBean property. If no such bean is found (or Spring is not 
*  present), attempt to instantiate an object whose fully-qualified class name is the value
*  in the schemaBean property. If one of these approaches succeeds, we derive
*  the schema from the discovered object (by treating it as a Java Bean and assuming that 
*  each one of its getters corresponds to a like-named field in the DataSource). More 
*  information on custom DataSource implementations is writeCustomDataSource.
*  
*  Note that when dataSource schema is derived by introspecting the Java class or Spring bean the
*  order of derived fields is non-deterministic, so you may find that it changes if you switch
*  JVMs, Java compilers or other parts of your Java toolchain. This means that you need to include
*  the order of all fields that are ultimately client-visible. Otherwise, if you bind a ListGrid
*  or DynamicForm to this DataSource without specifying the order of fields in that grid or form,
*  you could get fields in a different order on each application restart.
*  Field types
*  The following table shows how SQL types are derived into 
*  FieldType when we use an SQL table as a source of metadata
*  for a SQL or Hibernate DataSource:
*  
*  SQL typeDataSourceField.type
*  CHAR, VARCHAR, LONGVARCHAR, TEXT, CLOBtext
*  BIT, TINYINT, SMALLINT, INTEGER, BIGINT, DECIMAL*, NUMBER**integer
*  REAL, FLOAT, DOUBLE, DECIMAL*, NUMBER**float
*  DATEdate
*  TIMEtime
*  TIMESTAMPdatetime
*  BLOB, BINARY, VARBINARY, LONGVARBINARYbinary
*  
*  *For DECIMAL types, we inspect the "DECIMAL_DIGITS" attribute of the JDBC 
*  metadata and designate the field type "integer" if that attribute is 0, or "float" if
*  it is some other value.
*  **NUMBER is an Oracle-only type that appears in the JDBC metadata as DECIMAL
*  and is handled exactly the same way as DECIMAL
*  
*  The following table shows how Java types are derived into DataSource types when we use
*  an unannotated Java class (Spring bean, Hibernate mapping or POJO) as a source of 
*  metadata for a SQL, Hibernate or custom DataSource:
*  
*  Java typeDataSourceField.type
*  boolean, Booleanboolean
*  char, Character, String, Readertext
*  byte, short, int, long, Byte, Short, Integer, Long, BigIntegerinteger
*  float, double, Float, Double, BigDecimalfloat
*  Enumenum (see discussion below)
*  InputStreambinary
*  java.sql.Date, java.util.Date, java.util.Calendardate
*  java.sql.Timetime
*  java.sql.Timestampdatetime
*  
*  
*  Finally, this table shows how Java types are derived into DataSource types when we use an 
*  annotated class as a source of metadata. Note annotated classes are necessary for JPA
*  DataSources, but you can choose to use them for other types of DataSource as well. 
*  For Hibernate DataSources, this is very worthwhile because Hibernate will also make use
*  of the annotations as config, meaning you don't need to specify .hbm.xml 
*  files. For SQL and custom DataSources, there is no benefit at the persistence level, 
*  but it may still be worthwhile because the use of an annotated class gives us better 
*  metadata and allows us to generate a better, closer-fitting autoDerive DataSource than 
*  we can from examination of SQL schema or plain Java Beans:
*  
*  Java typeDataSourceField.type
*  boolean, Booleanboolean
*  char, Character, String, Readertext
*  byte, short, int, long, Byte, Short, Integer, Long, BigIntegerinteger
*  float, double, Float, Double, BigDecimalfloat
*  InputStreambinary
*  java.util.Date (with Temporal set to DATE), java.sql.Datedate
*  java.util.Date (with Temporal set to TIME), java.sql.Timetime
*  java.util.Date (with Temporal set to TIMESTAMP), java.util.Calendar, java.sql.Timestampdatetime
*  Enum (with Enumerated set to STRING)enum (see discussion below)
*  Enum (with Enumerated set to ORDINAL)intEnum (see discussion below)
*  Field with Lob annotationbinary
*  Field with GeneratedValue annotationsequence, if the field is an integer type (see discussion below)
*  
*  
*  Primary keys, sequences and identity columns
*  We attempt to derive information about primary keys from the metadata we have.
*  
*  If the metadata source is an SQL table:
*  
*  
* - If the table does not have a native primary key constraint, no attempt is made to 
*    identify primary key fields. Otherwise:
*  
* - The column or columns that make up the table's native primary key constraint are 
*    identified using the JDBC DatabaseMetaData.getPrimaryKeys() API
*  
* - Each DataSource field that corresponds to one of these native primary key columns 
*    is marked primaryKey: true
*  
* - For each of these columns, the metadata returned by 
*    DatabaseMetaData.getColumns() is inspected. If the metadata includes
*    IS_AUTOINCREMENT=YES, we mark the corresponding field as 
*    type="sequence". This information should be reliably provided by 
*    databases that implement "auto-increment" or "identity" column types, such as MySQL
*    or Microsoft SQL Server
*  
* - If the previous step does not identify a column as a sequence, we inspect the 
*    ResultSetMetaData obtained by running a dummy query on the table. If 
*    the isAutoIncrement() API returns true for that column, we mark the 
*    corresponding field as type="sequence"
*  
* - If the previous steps have not identified the column as a sequence, we check the 
*    TYPE_NAME in the column metadata. If it is "serial", this means the 
*    column is a PostgreSQL "serial" or "serial8" type column. Postgres does not 
*    transparently implement auto-increment columns, but it does provide this serial 
*    type, which causes the column to be implicitly bound to an underlying sequence. So
*    this type causes us to mark the field type="sequence", and we also set
*    DataSourceField.implicitSequence true
*  
* - If the previous steps have not identified the column as a sequence, we check the 
*    COLUMN_DEF in the column metadata. If this contains the token "$$ISEQ"
*    and ends with "NEXTVAL", this means the column is an Oracle "GENERATED AS IDENTITY" 
*    column. This type of column was introduced in Oracle 12c and is conceptually 
*    exactly the same thing as the Postgres "serial" column described above. We treat 
*    it the same way: mark it type="sequence" and 
*    implicitSequence="true"
*  
* - If the previous steps have not identified the column as a sequence, then you may
*    be using a pure sequence database, such as an Oracle version earlier than 12c, or 
*    you may be using a database where both sequences and identity columns are available
*    (Oracle, Postgres, DB2), and a sequence is being used either by design or because 
*    the table was created before the database product supported identity columns. In 
*    this case, we cannot determine if the column should be a sequence or not. However,
*    in many applications, the fact that the column is an integer and is a primary key
*    would imply that it is also a sequence. Therefore, if the column is an integer and
*    the server.properties flag auto.derive.integer.pk.always.sequence
*    is true, we mark the field as type="sequence"
*  
* - If, after all this, SmartClient ends up incorrectly marking a primary key field as 
*    a sequence (or vice versa), you can always override it on a per-field basis by 
*    simply redeclaring the field with the correct type in your .ds.xml 
*    file:
*  &lt;DataSource serverType="sql" tableName="myTable" autoDeriveSchema="true"&gt;
*   &lt;fields&gt;
*    &lt;!-- This field was incorrectly marked as a sequence --&gt;
*    &lt;field name="notASeq" type="integer" /&gt;
*    &lt;!-- This field was incorrectly marked as an integer when it should be a sequence --&gt;
*    &lt;field name="isASeq" type="sequence" /&gt;
*   &lt;/fields&gt;
*  &lt;/DataSource&gt;
*  
*   
*  If the metadata source is Hibernate mappings described in a .hbm.xml file:
*  
*  
* - The first field we encounter that is described in the mapping with an &lt;id&gt; tag
*    is marked as a primaryKey
*  
* - If that field is marked as being generated, we set its type to "sequence"
*  
*   
*  If the metadata source is an annotated object (whether JPA, Hibernate or just an 
*  annotated POJO):
*  
*  
* - Any field with an @Id annotation is is marked as a primaryKey (this 
*    differs from the Hibernate .hbm.xml file case because that is specific 
*    to Hibernate, which does support composite keys, but not by specifying multiple 
*    &lt;id&gt; tags. Annotations are supported, via annotated POJOs, for any kind of 
*    persistence strategy, so multiple @Id fields are perfectly valid)
*  
* - Any field with a @GeneratedValue annotation is either marked as 
*    type="sequence" (if it is an integer type) or as 
*    DataSourceField.autoGenerated="true" (for other 
*    field types)
*  
*  Finally, if the metadata is a plain, unannotated Java object, no attempt is made to
*  derive primary key fields.
*  enums and valueMaps
*  When we come across Java Enum properties in plain or annotated classes, 
*  as well as setting the field type as noted in the above tables, we also generate a 
*  valueMap for the field, based on the Enum members. 
*  
*  For cases where we generate a field of SmartClient type "enum" (see the above tables), 
*  the keys of the valueMap are the result of calling name() on each member
*  of the underlying Java Enum (in other words, its value exactly as declared in its 
*  enum declaration). For cases where we generate a field of SmartClient type "intEnum",
*  the keys of the valueMap are strings representing the ordinal number of each member 
*  in the Java Enum - "0", "1", etc. Note that this behavior will be overriden by 
*  DataSource.enumTranslateStrategy if both are set.
*  
*  In both of these case, the display values generated for the valueMap are the result 
*  of calling toString() on each Enum member. If that gives the same 
*  value as calling name(), the value is passed through 
*  DataTools.deriveTitleFromName(), which applies the same processing rules
*  as DataSource.getAutoTitle to derive a more user-friendly display value.
*  Further notes
*  schemaBean implies autoDeriveSchema, because it has no other 
*  purpose than to name the bean to use for auto-derived schema. Thus, if you specify
*  schemaBean you do not need to specify autoDeriveSchema as well
*  (though it does no harm to do so). However, tableName and 
*  beanClassName can be validly specified without implying 
*  autoDeriveSchema, so in those cases you must explicitly specify 
*  autoDeriveSchema.
*  
*  The underlying super-DataSource is cached in server memory, so that it does 
*  not have to be derived and created each time you need it. However, the cache manager 
*  will automatically refresh the cached copy if it detects that the deriving DataSource 
*  has changed. Thus, if you change the metadata your DataSource is deriving (if, for 
*  example, you add a column to a table), all you need to do is touch the 
*  .ds.xml file (ie, update its last changed timestamp - you don't actually
*  have to change it) and the cached copy will be refreshed next time it is needed.
*  
*  When autoDeriveSchema is set, SQLDataSource will automatically discover foreignKeys and
*  deliver table and column name information to the client in hashed form so that two
*  DataSources that are linked by native SQL foreign keys will automatically discover each
*  other if loaded into the same application, and set 
*  DataSourceField.foreignKey automatically. Because the table and column 
*  names are delivered as cryptohashes, there is no information leakage, but regardless,
*  the feature can be disabled via setting datasource.autoLinkFKs to false in
*  server_properties. This hashed linkage information is
*  delivered to the client in properties DataSource.tableCode and
*  DataSourceField.fkTableCode/DataSourceField.fkColumnCode
* @type {boolean}
* @default null
*/
autoDeriveSchema?: boolean;

/**
* For DataSources that specify DataSource.autoDeriveSchema, this property indicates the name
*  of the Spring bean, Hibernate mapping or fully-qualified Java class to use as parent 
*  schema.
* @type {string}
* @default null
*/
schemaBean?: string;

/**
* For JPA and Hibernate DataSources this property indicates, that data source has composite primary key and
*  specifies fully-qualified Java class:
*  
* - with @EmbeddedId you have to specify class name of declared id
*  
* - with @IdClass you have to specify class specified in annotation declaration
* @type {string}
* @default null
*/
idClassName?: string;

/**
* When using the patternOperators OperatorId, 
*  character that matches any single character.
*  
*  Pass multiple strings to provide multiple alternative wildcards.
* @type {string | Array<Partial<string>>}
* @default "["?","%"]"
*/
patternSingleWildcard?: string | Array<Partial<string>>;

/**
* When using the patternOperators OperatorId, 
*  character that matches a series of one or more characters.
*  
*  Pass multiple strings to provide multiple alternative wildcards.
* @type {string | Array<Partial<string>>}
* @default "*"
*/
patternMultiWildcard?: string | Array<Partial<string>>;

/**
* When using the patternOperators OperatorId, 
*  character that escapes the DataSource.patternSingleWildcard or DataSource.patternMultiWildcard if placed before 
*  it, so that it is treated as a literal character.
* @type {string}
* @default "\"
*/
patternEscapeChar?: string;

/**
* OperatorId like "matchesPattern" use patterns like
*  "foo*txt" to match text values. The patterns are similar to the patterns you use to
*  match names of files in a command-line interface, or to the pattern allowed for the SQL
*  "LIKE" operator.
*  
*  translatePatternOperators controls whether these pattern operators should
*  be translated to a nested series of "startsWith"/"endsWidth"/"contains" operators before
*  being sent to the server. This allows a server that only implements simple operators
*  like "startsWith" to support pattern operators such as "matchesPattern" and
*  "containsPattern", but with caveats:
*   
*     
* - single-character wildcards are not supported
*     
* - multiple wildcards are not truly order-dependent, for example *1*2*3* will
*        match 1,2,3 as interior characters in any order. 
*     
* - may be less efficient than a direct server-side implementation that is able to
*        translate the pattern directly to the underlying storage engine.
*   
*  
*  Note that since "containsPattern" is essentially equivalent to "matchesPattern" but with
*  "*" wildcards at the beginning and end of every pattern, the second limitation (pattern
*  not really order dependent) may be fairly obvious to users when using this feature. For
*  example, "m*t" will match "we meet" and "we teem".
*  
*  The following are examples of how patterns are translated to simpler operators. Note
*  that the case sensitive version of the operator is referred to below, but of course
*  "iMatchesPattern" and "iContainsPattern" will be translated to case-insensitive versions
*  of these operators, such as "iStartsWith".
*  
*  *foo (endsWith)
*  foo* (startsWith)
*  *foo* (contains)
*  *foo*bar (contains and endsWith)
*  foo*bar* (startsWith and contains)
*  foo*bar (startsWith and endsWith)
*  *foo*bar* (multiple contains)
*  
*  Also supported: one startsWith, multiple contains, one endsWith.
* @type {boolean}
* @default false
*/
translatePatternOperators?: boolean;

/**
* Sets the XML namespace prefixes available for XPaths on a DataSource-wide basied. See
*  OperationBinding.xmlNamespaces for details.
* @type {object}
* @default "See below"
*/
xmlNamespaces?: object;

/**
* For an XML DataSource, URN of the WebService to use to invoke operations. This URN
*  comes from the "targetNamespace" attribute of the &lt;wsdl:definitions&gt; element in a
*  WSDL (Web Service Description Language) document, and serves as the unique identifier of
*  the service.
*  
*  Having loaded a WebService using XMLTools.loadWSDL, setting
*  serviceNamespace combined with specifying 
*  OperationBinding
*  that set OperationBinding.wsOperation will cause a DataSource to invoke web
*  service operations to fulfill DataSource requests (DSRequest).
*  
*  Setting serviceNamespace also defaults
*  DataSource.dataURL to the service's location,
*  DataSource.dataFormat to "xml" and
*  OperationBinding.dataProtocol to "soap".
* @type {string}
* @default null
*/
serviceNamespace?: string;

/**
* For a DataSource derived from WSDL or XML schema, the XML namespace this schema belongs
*  to. This is a read-only attribute automatically present on DataSources returned from
*  SchemaSet.getSchema and WebService.getSchema.
* @type {string}
* @default null
*/
schemaNamespace?: string;

/**
* See OperationBinding.recordXPath. recordXPath can be specified
*  directly on the DataSource for a simple read-only DataSource only capable of "fetch"
*  operations.
* @type {XPathExpression}
* @default null
*/
recordXPath?: XPathExpression;

/**
* Default URL to contact to fulfill all DSRequests. Can also be set on a
*  per-operationType basis via OperationBinding.dataURL.
*  
*  NOTE: Best practice is to use the same dataURL for all DataSources which
*  fulfill DSRequests via the server-side RPCManager API. Otherwise, cross-DataSource
*  RPCManager.startQueue will not be possible.
* @type {URL}
* @default null
*/
dataURL?: URL;

/**
* Tag name to use when serializing to XML. If unspecified, the dataSource.ID
*  will be used.
* @type {string}
* @default null
*/
tagName?: string;

/**
* Whether we store server responses for this DataSource into 
*  Offline, and then use those stored responses
*  at a later time if we are offline (ie, the application cannot connect to the server). 
*  Note that by default we do NOT use offline storage for a dataSource.
* @type {boolean}
* @default null
*/
useOfflineStorage?: boolean;

/**
* A clientOnly DataSource simulates the behavior of a remote data store by manipulating a
*  static dataset in memory as DSRequest are executed on it. Any changes
*  are lost when the user reloads the page or navigates away.
*  
*  A clientOnly DataSource will return responses asynchronously, just as a DataSource
*  accessing remote data does. This allows a clientOnly DataSource to be used as a
*  temporary placeholder while a real DataSource is being implemented - if a clientOnly
*  DataSource is replaced by a DataSource that accesses a remote data store, UI code for
*  components that used the clientOnly DataSource will not need be changed.
*  
*  A clientOnly DataSource can also be used as a shared cache of modifiable data across
*  multiple UI components when immediate saving is not desirable. In this case, several
*  components may interact with a clientOnly DataSource and get the benefit of
*  ResultSet behaviors such as automatic cache sync and in-browser data filtering
*  and sorting. When it's finally time to save, DataSource.cacheData can be
*  inspected for changes and data can be saved to the original DataSource via
*  DataSource.addData, DataSource.updateData and DataSource.removeData, possibly in a
*  RPCManager.startQueue. Note that
*  DataSource.getClientOnlyDataSource lets you easily obtain a
*  clientOnly DataSource representing a subset of the data available from a
*  normal DataSource. 
*  
*  See also DataSource.cacheAllData - a cacheAllData behaves like a write-through
*  cache, performing fetch and filter operations locally while still performing remote save
*  operations immediately.
*  
*  ClientOnly DataSources can be populated programmatically via DataSource.cacheData - see
*  clientOnlyDataSources for other ways to populate a
*  client-only DataSource with data.
* @type {boolean}
* @default false
*/
clientOnly?: boolean;

/**
* If set, causes this DataSource to use a read-only "mock" or "test" dataset, if specified, or if no test
*  data is available, then to load data normally but then operate similarly to a DataSource.clientOnly
*  DataSource, never writing changes back to the server.
*  
*  mockMode has no effect on MockDataSource or a DataSource.clientOnly DataSource. 
*  
*  For other DataSources, a one-time fetch will be performed to retrieve sample data, similar to a
*  DataSource.cacheAllData DataSource, except that changes will never be saved back to the server. Only a
*  subset of data will be retrieved, based on DataSource.mockDataRows. DataSource.mockDataCriteria can optionally
*  be set to retrieve specific data.
*  
*  Alternatively, mock data can be provided with DataSource.cacheData.
*  
*  DataSources can be loaded in mockMode via passing settings to DataSource.load, or
*  if loaded with a screen or project, by passing settings to RPCManager.loadScreen or
*  the server-side Project.load() API.
* @type {boolean}
* @default null
*/
mockMode?: boolean;

/**
* When DataSource.mockMode is enabled, number of rows of data to retrieve via an initial "fetch" DSRequest, for
*  use as sample data. Set to null to retrieve all available rows.
* @type {Integer}
* @default "75"
*/
mockDataRows?: Integer;

/**
* When DataSource.mockMode is enabled, criteria to use in an initial "fetch" DSRequest to retrieve sample data.
* @type {Criteria}
* @default null
*/
mockDataCriteria?: Criteria;

/**
* Decides under what conditions the ResultSet cache should be dropped when the
*  ResultSet.criteria changes.
* @type {CriteriaPolicy}
* @default "dropOnShortening"
*/
criteriaPolicy?: CriteriaPolicy;

/**
* The default textMatchStyle to use for DSRequests that do not explicitly state 
*  a DSRequest.textMatchStyle.  Note, however, that DSRequests 
*  issued by ListGrids and other DataBoundComponent will 
*  generally have a setting for textMatchStyle on the component itself (see
*  ListGrid.autoFetchTextMatchStyle, for example).
* @type {TextMatchStyle}
* @default "exact"
*/
defaultTextMatchStyle?: TextMatchStyle;

/**
* For fields on this dataSource that specify 
*  DataSourceField.ignoreTextMatchStyle true, the prevailing
*  textMatchStyle is ignored and SmartClient matches exact values. This property dictates
*  whether that match is case-sensitive like the "exactCase" textMatchStyle, or 
*  case-insensitive like the "exact" textMatchStyle (the default). Please see the 
*  TextMatchStyle for a discussion of the nuances
*  of case-sensitive matching.
* @type {boolean}
* @default false
*/
ignoreTextMatchStyleCaseSensitive?: boolean;

/**
* Class for ResultTrees used by this datasource. If null, defaults to using
*  ResultTree.
*  
*  This can be set to a custom subclass of ResultTree that, for example, hangs on to extra
*  information necessary for integration with web services.
* @type {string}
* @default null
*/
resultTreeClass?: string;

/**
* Class for ResultSets used by this datasource. If null, defaults to using
*  ResultSet.
*  
*  This can be set to a custom subclass of ResultSet that, for example, hangs onto to extra
*  information necessary for integration with web services.
* @type {string}
* @default null
*/
resultSetClass?: string;

/**
* Whether to attempt validation on the client at all for this DataSource. If unset (the
*  default), client-side validation is enabled.
*  
*  Disabling client-side validation entirely is a good way to test server-side validation.
* @type {boolean}
* @default null
*/
useLocalValidators?: boolean;

/**
* The required message when a field that has been marked as
*  DataSourceField.required is not filled in by the user.
*  
*  Note that DataSourceField.requiredMessage wins over this setting if both are set.
* @type {HTMLString}
* @default null
*/
requiredMessage?: HTMLString;

/**
* Whether RPCRequests sent by this DataSource should enable
*  RPCRequest.showPrompt in order to block user interactions until the
*  request completes. 
*  
*  DataSource requests default to blocking UI interaction because, very often, if the user
*  continues to interact with an application that is waiting for a server response, some
*  kind of invalid or ambiguous situation can arise.
*  
*  Examples include pressing a "Save" button a second time before the first save completes,
*  making further edits while edits are still being saved, or trying to initiate editing on
*  a grid that hasn't loaded data.
*  
*  Defaulting to blocking the UI prevents these bad interactions, or alternatively, avoids
*  the developer having to write repetitive code to block invalid interactions on every
*  screen.
*  
*  If an operation should ever be non-blocking, methods that initiate DataSource requests
*  (such as DataSource.fetchData) will generally have a
*  requestProperties argument allowing showPrompt to be set to
*  false for a specific request.
* @type {boolean}
* @default true
*/
showPrompt?: boolean;

/**
* Sets the strategy this DataSource uses to translate Java enumerated types (objects of type
*  enum) to and from Javascript.
*  
*  This property is only applicable if you are using the SmartClient server
* @type {EnumTranslateStrategy}
* @default null
*/
enumTranslateStrategy?: EnumTranslateStrategy;

/**
* The name of the property this DataSource uses for constant name when translating
*  Java enumerated types to and from Javascript, if the EnumTranslateStrategy is set
*  to "bean". Defaults to "_constant" if not set.
*  
*  This property is only applicable if you are using the SmartClient server
* @type {string}
* @default null
*/
enumConstantProperty?: string;

/**
* The name of the property this DataSource uses for ordinal number when translating
*  Java enumerated types to and from Javascript, if the EnumTranslateStrategy is set
*  to "bean". Defaults to "_ordinal" if not set.
*  
*  This property is only applicable if you are using the SmartClient server
* @type {string}
* @default null
*/
enumOrdinalProperty?: string;

/**
* If set, titles are automatically derived from DataSourceField.name for any 
*  field that does not have a DataSourceField.title and is not marked
*  DataSourceField.hidden:true, by calling the method DataSource.getAutoTitle.
* @type {boolean}
* @default true
*/
autoDeriveTitles?: boolean;

/**
* When true, indicates that this DataSource supports multi-level sorting.
* @type {boolean}
* @default true
*/
canMultiSort?: boolean;

/**
* When true, indicates that any updates for this DataSource include only those fields 
*  that have actually changed (and primaryKey fields); when false (the default), all 
*  field values are included in updates, whether they have changed or not
* @type {boolean}
* @default false
*/
sparseUpdates?: boolean;

/**
* When true, indicates that fields in this DataSource will never be positively updated
*  to the null value; they may arrive at null values by being omitted, but we will 
*  not send actual null values in update requests. When false (the default), null is 
*  not treated in any special way.
*  
*  Setting this value causes null-assigned fields to be replaced with the field's
*  DataSourceField.nullReplacementValue, if one is declared.
*  If no nullReplacementValue is declared for the field, the null assignment
*  is replaced with the DataSource's DataSource.nullStringValue, 
*  DataSource.nullIntegerValue, DataSource.nullFloatValue 
*  or DataSource.nullDateValue, depending on the field type.
*  
*  For "add" operations, setting DataSource.omitNullDefaultsOnAdd causes
*  null-valued fields to be removed from the request entirely, rather than replaced with
*  default values as described above.
* @type {boolean}
* @default false
*/
noNullUpdates?: boolean;

/**
* When true, and DataSource.noNullUpdates is also true, indicates that "add" 
*  requests on this DataSource will have null-valued fields removed from the request 
*  entirely before it is sent to the server, as opposed to the default behavior of 
*  replacing such null values with defaults.
* @type {boolean}
* @default false
*/
omitNullDefaultsOnAdd?: boolean;

/**
* If DataSource.noNullUpdates is set, the value to use for any text field that has a null
*  value assigned on an update operation, and does not specify an explicit
*  DataSourceField.nullReplacementValue.
* @type {string}
* @default ""
*/
nullStringValue?: string;

/**
* If DataSource.noNullUpdates is set, the value to use for any integer field that has a null
*  value assigned on an update operation, and does not specify an explicit
*  DataSourceField.nullReplacementValue.
* @type {Integer}
* @default "0"
*/
nullIntegerValue?: Integer;

/**
* If DataSource.noNullUpdates is set, the value to use for any float field that has a null
*  value assigned on an update operation, and does not specify an explicit
*  DataSourceField.nullReplacementValue.
* @type {Float}
* @default "0.0"
*/
nullFloatValue?: Float;

/**
* If DataSource.noNullUpdates is set, the value to use for any boolean field that has a null
*  value assigned on an update operation, and does not specify an explicit
*  DataSourceField.nullReplacementValue.
* @type {boolean}
* @default false
*/
nullBooleanValue?: boolean;

/**
* If DataSource.noNullUpdates is set, the value to use for any date or time field that has a
*  null value assigned on an update operation, and does not specify an explicit
*  DataSourceField.nullReplacementValue.
*  
*  Unlike strings and numbers, there is no "natural" choice for a null replacement value 
*  for dates. The default value we have chosen is midnight on January 1st 1970, simply 
*  because this is "the epoch" - the value that is returned by calling "new Date(0)"
* @type {Date}
* @default "See below"
*/
nullDateValue?: Date;

/**
* Whether a user must be authenticated in order to access this DataSource. This establishes a
*  default for the DataSource as a whole; individual DataSource.operationBindings within the
*  DataSource may still override this setting by explicitly setting
*  OperationBinding.requiresAuthentication.
*  
*  Whether the user is authenticated is determined by calling
*  httpServletRequest.getRemoteUser(), hence works with both simple J2EE security
*  (realms and form-based authentication) and JAAS (Java Authentication & Authorization
*  Service).
*  
*  If you wish to use an authentication scheme that does not make use of the servlet API's 
*  standards, SmartClient Server also implements the setAuthenticated method
*  on RPCManager. You can use this API to tell SmartClient that all the 
*  requests in the queue currently being processed are associated with an authenticated
*  user; in this case, SmartClient will not attempt to authenticate the user via 
*  httpServletRequest.getRemoteUser()
*  
*  You can set the default value for this property via setting "authentication.defaultRequired"
*  in server_properties. This allows you to, for example,
*  cause all DataSources to require
*  authentication for all operations by default.
*  
*  Note that setting this property does not automatically cause an authentication mechanism to
*  appear - you still need to separately configure an authentication system. Likewise, setting
*  requiresAuthentication="false" does not automatically allow users to bypass your authentication
*  mechanism - you need to set up a URL that will accept DSRequests and process them similar to
*  the default "IDACall" servlet, and which is not protected by the authentication system. See
*  servletDetails for details on the IDACall servlet.
* @type {boolean}
* @default null
*/
requiresAuthentication?: boolean;

/**
* Similar to OperationBinding.requiresRole, but controls access to the DataSource as a
*  whole.
* @type {string}
* @default null
*/
requiresRole?: string;

/**
* Indicates that the specified VelocityExpression must evaluate to true for a user 
*  to access this DataSource.
*  
*  See also OperationBinding.requires.
* @type {VelocityExpression}
* @default null
*/
requires?: VelocityExpression;

/**
* Indicates that declarative security rules are waived for rows that were created by the 
*  current user. Practically, this means that when a security check fails, instead of a 
*  security exception being thrown, we alter the criteria to ensure that the request can 
*  only return or affect rows that were created by the current authenticated user. This 
*  allows you to create security regimes whereby users can see and edit data they created, 
*  but have access to other users' data forbidden or limited.
*  
*  In order for this to work, we require two things:
*  
* - The DataSource must specify a field of type "creator" - this field type is described 
*    on FieldType
*  
* - The authentication regime in use must include the idea of a "current user". The 
*    authentication provided by the Servlet API is an example of such a regime.
*  
*  This setting can be overridden at operationBinding and field level, allowing extremely
*  fine-grained control.
* @type {boolean}
* @default null
*/
creatorOverrides?: boolean;

/**
* Requires that the currently authenticated user match the contents of
*  this field, for client-initiated requests (i.e., where
*  DSRequest.isClientRequest() returns true on the server).
* 
*  When a new row is added by a client-initiated DSRequest, the
*  ownerIdField will be automatically populated with the currently
*  authenticated user (clobbering any value supplied by the client).
*  Client-initiated attempts to update the ownerIdField will also be
*  prevented.
* 
*  If you wish to set the ownerIdField to a different value via an "add"
*  or "update" operation, you can do so in server-side DMI code (possibly
*  consulting DSRequest.getClientSuppliedValues() to get the
*  value that was clobbered).
* 
*  For client-initiated "fetch", "update" or "remove" operations, the
*  server will modify client-supplied criteria so that only rows whose
*  ownerIdField matches the currently authenticated user can be read,
*  updated or deleted. 
* 
*  The ownerIdField setting can be overridden at the
*  OperationBinding.ownerIdField level.
* 
*  If ownerIdField is specified,
*  DataSource.requiresAuthentication will
*  default to true. If requiresAuthentication is
*  explicitly set to false, then unauthenticated users will be
*  able to see all records. To avoid this, you can use
*  DataSource.guestUserId to specify a default user to
*  apply when no one has authenticated.
* @type {string}
* @default null
*/
ownerIdField?: string;

/**
* Value to use for the DataSource.ownerIdField if no one 
*  has authenticated.
* 
*  This setting can be overridden at the operationBinding level.
* @type {string}
* @default null
*/
guestUserId?: string;

/**
* Setting a DataSource to be serverOnly="true" ensures that it will not be visible
*  to the client. Any request through IDACall to this DataSource will return a failure response.
*  Only requests which have been initiated on the server-side will be executed against this DataSource.
* @type {string}
* @default null
*/
serverOnly?: string;

/**
* If true, causes all operations on this DataSource to automatically start or join a 
*  transaction associated with the current HttpServletRequest. This means that multiple
*  operations sent to the server in a RPCManager.startQueue will be
*  committed in a single transaction. 
*  
*  Note that this includes fetch operations - setting this property to true has the same
*  effect as a transaction policy of ALL for just this DataSource's operations - see the 
*  server-side RPCManager.setTransactionPolicy() for details of the different 
*  TransactionPolicy settings.
*  
*  If this property is explicitly false, this causes all operations on this DataSource 
*  to be committed individually - the same as a transaction policy of NONE, just for this
*  DataSource's operations.
*  
*  In either case, you can override the setting for individual operations - see 
*  OperationBinding.autoJoinTransactions.
*  
*  If this property if null or not defined, we fall back to the default setting for this
*  type of DataSource. These are defined in server_properties
*  as follows:
*  
*  
* - Hibernate: hibernate.autoJoinTransactions
*  
* - JPA/JPA2: jpa.autoJoinTransactions
*  
* - SQL: There is one setting per configured database connection (DataSource.dbName).
*  For example, the setting for the default MySQL connection is 
*  sql.Mysql.autoJoinTransactions
*  
*  If the setting is not defined at the DataSource-type level, we use the system global 
*  default, which is defined in server.properties as 
*  autoJoinTransactions.
*  
*  At the dbName and global system levels, you can set the autoJoinTransactions attribute 
*  to a valid Transaction Policy, rather than a simple true or false (although these 
*  values work too - true is the same as ALL, false is the same as NONE). For valid 
*  TransactionPolicy values and their meanings, see the server-side Javadoc for 
*  RPCManager.setTransactionPolicy()
*  
*  Note that the configuration settings discussed here can be overridden for a particular 
*  queue of requests by setting the server-side RPCManager's transaction policy. Look in
*  the server-side Javadoc for RPCManager.getTransactionPolicy().
*  
*  Transactions can also be initiated manually, separate from the
*  RPCManager/HttpServletRequest lifecycle, useful for both multi-threaded web
*  applications, and standalone applications that don't use a servlet container - see
*  standaloneDataSourceUsage.
*  
*  NOTE: Setting this property to true does not cause a transactional persistence 
*  mechanism to automatically appear - you have to ensure that your DataSource supports 
*  transactions. The SmartClient built-in SQL, Hibernate and JPA DataSources support transactions, 
*  but note that they do so at the provider level. This means that you can combine 
*  updates to, say, an Oracle database and a MySQL database in the same queue, but they
*  will be committed in two transactions - one per database. The SmartClient 
*  server will commit or rollback these two transactions as if they were one, so a 
*  failure in some Oracle update would cause all the updates to both databases to be 
*  rolled back. However, this is not a true atomic transaction; it is possible for one
*  transaction to be committed whilst the other is not - in the case of hardware failure,
*  for example.
*  
*  NOTE: Not all the supported SQL databases are supported for transactions. Databases 
*  supported in this release are:
*  
*  
* - DB2
*  
* - HSQLDB
*  
* - Firebird
*  
* - Informix
*  
* - Microsoft SQL Server
*  
* - MySQL (you must use InnoDB tables; the default MyISAM storage engine does not
*  support transactions)
*  
* - MariaDB
*  
* - Oracle
*  
* - PostgreSQL
*  
* @type {boolean}
* @default null
*/
autoJoinTransactions?: boolean;

/**
* This flag is part of the Automatic Transactions feature; it is only applicable in 
*  Power Edition and above.
*  
*  If true, causes all transactional operations on this DataSource to use the current 
*  Spring-managed transaction, if one exists. If there is no current Spring transaction 
*  to use at the time of execution, a server-side Exception is thrown. Note, a 
*  "transactional operation" is one that would have joined the SmartClient shared 
*  transaction in the absence of Spring integration - see 
*  DataSource.autoJoinTransactions.
*  
*  This feature is primarily intended for situations where you have 
*  dmiOverview that make use of both Spring DAO operations and 
*  SmartClient DSRequest operations, and you would like all of them to share the same 
*  transaction. An example of the primary intended use case:
*   &#x0040;Transactional(isolation=Isolation.READ_COMMITTED, propagation=Propagation.REQUIRED)
*   public class WorldService {
* 
*    public DSResponse doSomeStuff(DSRequest dsReq, HttpServletRequest servletReq) 
*    throws Exception 
*    {
*     ApplicationContext ac = (ApplicationContext)servletReq.getSession().getServletContext().getAttribute("applicationContext");
*     WorldDao dao = (WorldDao)ac.getBean("worldDao");
*     dao.insert(req.getValues());
*     DSRequest other = new DSRequest("MyOtherDataSource", "add");
*     // Set up the 'other' dsRequest with critiera, values, etc
*     // ...
* 
*     // This dsRequest execution will use the same transaction that the DAO operation
*     // above used; if it fails, the DAO operation will be rolled back
*     other.execute();
* 
*     return new DSResponse();
*    }
*   }
*  Note: if you want to rollback an integrated Spring-managed transaction, you can use 
*  any of the normal Spring methods for transaction rollback:
*  
* - Programmatically mark the transaction for rollback with the 
*  setRollbackOnly() API
*  
* - Throw a RuntimeException, or
*  
* - Throw an ordinary checked Exception. but configure Spring to rollback
*    on that Exception. This can be done in the @Transactional annotation:
*    &#x0040;Transactional(isolation=Isolation.READ_COMMITTED, propagation=Propagation.REQUIRED, rollbackFor=MyRollbackException.class)
*  
*  Spring's exception-handling model is different from SmartClient's, so care must be
*  taken to get the correct error processing. If a transactional DSRequest fails, 
*  SmartClient code will throw an ordinary checked Exception; but Spring will
*  ignore that Exception. So you can either:
*  
* - Wrap every DSRequest.execute() in a try/catch block. Catch 
*  Exception and throw a RuntimeException instead
*  
* - Just use the "rollbackFor" annotation to make your transactional method rollback 
*  for all instances of Exception
*  
*  
*  Note: Spring transaction integration is conceptually different from SmartClient's 
*  DataSource.autoJoinTransactions, because SmartClient
*  transactions apply to a queue of DSRequests, whereas Spring transactions are scoped to 
*  a single method invocation. If you want to make a whole SmartClient queue share a 
*  single Spring-managed transaction, you can wrap the processing of an entire queue in a 
*  call to a transactional Spring method. See the Using Spring Transactions with 
*  SmartClient DMI section at the bottom of the 
*  springIntegration for more details.
*  
*  You can set useSpringTransaction as the default setting for all dataSources 
*  for a given database provider by adding the property 
*  {dbName}.useSpringTransaction to your server.properties file.
*  For example, Mysql.useSpringTransaction: true or 
*  hibernate.useSpringTransaction: true. You can set it as the default for 
*  all providers with a server.properties setting like this: 
*  useSpringTransaction: true. When useSpringTransaction is 
*  the default, you can switch it off for a specific dataSource by explicitly setting the
*  flag to false for that DataSource.
*  
*  Finally, this setting can be overridden at the operationBinding level - see 
*  OperationBinding.useSpringTransaction
*  Configuration
*  When using Spring transactions, SmartClient needs a way to lookup the JNDI connection 
*  being used by Spring, and this needs to be configured. First, register a bean like 
*  this in your applicationContext.xml file:
*   &lt;bean id="dataSource" class="org.springframework.jndi.JndiObjectFactoryBean"&gt;
*     &lt;!-- Set this to the JNDI name Spring is using --&gt;
*     &lt;property name="jndiName" value="isomorphic/jdbc/defaultDatabase"/&gt;
*   &lt;/bean&gt;
*  
*  and then add a line like this to your server.properties:
*   # Set this property to match the "id" of the JndiObjectFactoryBean registered in Spring
*   sql.spring.jdbcDataSourceBean: dataSource
*  
* @type {boolean}
* @default null
*/
useSpringTransaction?: boolean;

/**
* The paging strategy to use for this DataSource. If this property is not set, the 
*  default paging strategy, specified with the server_properties
*  setting sql.defaultPaging, is used. 
*  
*  This setting can be overridden with the OperationBinding.sqlPaging property.
*  
*  NOTE: Operations that involve a OperationBinding.customSQL 
*  clause ignore this property, because customSQL operations usually need to be treated
*  as special cases. For these operations, the paging strategy comes from the 
*  server_properties setting
*  sql.defaultCustomSQLPaging or 
*  sql.defaultCustomSQLProgressivePaging, depending on whether or not 
*  DataSource.progressiveLoading is in force. Note that these
*  can always be overridden by a sqlPaging setting on the OperationBinding.
* @type {SQLPagingStrategy}
* @default null
*/
sqlPaging?: SQLPagingStrategy;

/**
* If explicitly set true or left null, causes the server to use a "hint" in the SQL we 
*  generate for paged queries. If explicitly set false, forces off the use of hints. This
*  property can be overridden per operationBinding - see 
*  OperationBinding.sqlUsePagingHint.
*  
*  Note this property is only applicable to DataSource.serverType DataSources, 
*  only when a DataSource.sqlPaging of "sqlLimit" is in force,
*  and it only has an effect for those specific database products where we employ a native
*  hint in the generated SQL in an attempt to improve performance.
* @type {boolean}
* @default null
*/
sqlUsePagingHint?: boolean;

/**
* This property is only applicable to DataSource.serverType DataSources, and 
*  only for OperationBinding that express a 
*  OperationBinding.customSQL clause. In these circumstances, we 
*  generally switch off paging because we are unable to generate a "row count" query that
*  tells the framework the size of the complete, unpaged resultset.
*  
*  The useSubselectForRowCount flag causes the framework to derive a rowcount
*  query by simply wrapping the entire customSQL clause in a subselect, like so:
*  &nbsp;&nbsp;&nbsp;&nbsp;SELECT COUNT(*) FROM ({customSQL clause here})
*  
*  However, this is not guaranteed to give good results. Because the customSQL clause can
*  contain anything that you can write in SQL, running it inside a subselect in order to 
*  count the rows might not work, might have unintended side-effects (if, for example, it 
*  is a stored procedure call that performs updates as part of its function), or it might 
*  just be a bad idea - for example, if the customSQL query is slow-running, you'll make 
*  it twice as slow with this flag, simply because you'll be running it twice. We 
*  recommend using this flag with care.
*  
*  NOTE: This setting can be overridden per-operation - see 
*  OperationBinding.useSubselectForRowCount. You can also set a global default for
*  this setting, so you don't have to specify it in every dataSource - define 
*  useSubselectForRowCount as true in your 
*  server_properties file.
* @type {boolean}
* @default null
*/
useSubselectForRowCount?: boolean;

/**
* Only applicable to the built-in SQL DataSource
*  
*  tableCode and the related properties DataSourceField.columnCode,
*  DataSourceField.fkTableCode and DataSourceField.fkColumnCode are read-only
*  attributes that are secure and unique cryptographic hashes of table and column names
*  used by this DataSource.
*  
*  These properties are used automatically by client-side framework code to 
*  link dataSources together by DataSourceField.foreignKey when a
*  foreignKey is not explicitly declared, but is found in the SQL schema via
*  the DataSource.autoDeriveSchema feature.
*  
*  A secure hash is used rather than the actual SQL table or column name for security
*  reasons - sending the actual SQL table or column name to the client could aid in
*  attempted SQL injection attacks.
*  
*  This feature can be disabled system-wide via setting datasource.autoLinkFKs
*  to false in server_properties.
* @type {string}
* @default null
*/
tableCode?: string;

/**
* If true, causes SmartClient Server to use the "progressive loading" pattern for 
*  fetches on this dataSource, as described in the Paging and total dataset length
*  section of the ResultSet. Essentially, this means that
*  we avoid issuing a row count query and instead advertise total rows as being slightly 
*  more than the number of rows we have already read (see DataSource.endGap). This 
*  allows users to load more of
*  a dataset by scrolling past the end of the currently-loaded rows, but it prevents them
*  from scrolling directly to the end of the dataset.
*  
*  Generally, progressive loading is appropriate when you have to deal with very large 
*  datasets. Note that by default, a dataSource will switch into progressive loading mode 
*  automatically when it detects that it is dealing with a dataset beyond a certain size - 
*  see DataSource.progressiveLoadingThreshold.
*  
*  This setting can be overridden for individual fetch operations with the 
*  OperationBinding.progressiveLoading property, and also at the level of the
*  individual DSRequest.progressiveLoading. You can also specify 
*  progressiveLoading on 
*  DataBoundComponent.progressiveLoading and certain types of 
*  FormItem - SelectItem.progressiveLoading and 
*  ComboBoxItem.progressiveLoading.
*  
*  Currently, this property only applies to users of the built-in SQLDataSource, but you 
*  could use it in custom DataSource implementations to trigger the server behavior 
*  described in the ResultSet documentation linked to above.
* @type {boolean}
* @default null
*/
progressiveLoading?: boolean;

/**
* Indicates the dataset size that will cause SmartClient Server to automatically
*  switch into DataSource.progressiveLoading for this DataSource.
*  To prevent automatic switching to progressive loading, set this property to -1. This
*  may also be prevented on a per-request basis by setting DSRequest.progressiveLoading
*  to false.
* @type {Integer}
* @default "200000"
*/
progressiveLoadingThreshold?: Integer;

/**
* If we are DataSource.progressiveLoading, indicates the number of 
*  extra records SmartClient Server will read beyond the end record requested by the client,
*  in order to establish if there are more records to view. This property has no effect 
*  if we are not progressive-loading.
*  
*  This property can be tweaked in conjunction with DataSource.endGap to change behavior
*  at the end of a dataset. For example, with the default values of lookAhead: 1
*  and endGap: 20, we can end up with the viewport shrinking if we get a case
*  where there really was only one more record (because the client was initially told there
*  were 20 more). This is not a problem per se, but it may be surprising to the user.
*  You could prevent this happening (at the cost of larger reads) by setting 
*  lookAhead to be endGap+1.
* @type {Integer}
* @default "1"
*/
lookAhead?: Integer;

/**
* For DataSource.serverType DataSources, sets the
*  default DataSourceField.sqlStorageStrategy to use for boolean fields
*  where no sqlStorageStrategy has been declared on the field.
*  
*  Can also be set system-wide via the server_properties setting
*  sql.defaultBooleanStorageStrategy, or for a particular database configuration by
*  setting sql.dbName.defaultBooleanStorageStrategy (see
*  adminConsole for more information on SQL configuration).
*  
*  Note that when this property is unset, the default DataSourceField.sqlStorageStrategy
*  strategy is effectively "string".
* @type {string}
* @default null
*/
defaultBooleanStorageStrategy?: string;

/**
* If we are DataSource.progressiveLoading, indicates the number of 
*  extra records SmartClient Server will advertise as being available, if it detects that 
*  there are more records to view (see DataSource.lookAhead). This property has no 
*  effect if we are not progressive-loading.
* @type {Integer}
* @default "20"
*/
endGap?: Integer;

/**
* Set this property to true to have a DataSource fetch all of its data client-side on the 
*  first fetch request. However, unlike a DataSource.clientOnly DataSource, this DataSource 
*  will still save changes normally, sending remote requests.
*  
*  You can manually set this attribute after initialization by calling 
*  DataSource.setCacheAllData; setting DataSource.autoCacheAllData:true causes a DataSource to 
*  automatically switch to cacheAllData:true when a fetch results in the
*  entire dataset being brought client-side.
*  
*  To cause automatic cache updates, you can set DataSource.cacheMaxAge to a number of seconds
*  and once data has been client-side for that length of time, the next fetch causes the
*  cache to be dropped and a new cache retrieved.
*  
*  Note that multiple DataSource.operationBindings of type "fetch" which return distinct
*  results will not work with cacheAllData: only one cache is created and is
*  used for all fetch operations, regardless of whether DSRequest.operationId has
*  been set. However, "fetch" operationBindings used as a
*  OperationBinding.cacheSyncOperation will work normally, so long as they return
*  all data fields that are returned by the default "fetch" operation, so that the cache
*  can be updated.
*  
*  To specify which operationId to use for fetching all data, use
*  DataSource.cacheAllOperationId.
*  
*  To use the cache only for requests that have the cacheAllOperationId,
*  allowing any other operationId (or absence of an operationId) to contact the server as
*  normal, set DataSource.cacheAcrossOperationIds.
* @type {boolean}
* @default null
*/
cacheAllData?: boolean;

/**
* DSRequest.operationId to use for fetching data in case DataSource.cacheAllData is
*  true. By default a standard fetch operation is used (with no operationId
*  specified).
* @type {string}
* @default null
*/
cacheAllOperationId?: string;

/**
* When DataSource.cacheAllData mode is enabled and a
*  DataSource.cacheAllOperationId has been set, this flag affects whether cached
*  results are used for all "fetch" requests regardless of their
*  DSRequest.operationId, or are used only for "fetch" requests that use the
*  cacheAllOperationId, allowing other requests to go to server normally.
*  
*  Default of true means that the cacheAllOperationId will be
*  used to fetch all rows, but the resulting cache will be used for all "fetch" operations
*  regardless of the operationId specified on the request.
*  
*  Switching to "false" effectively creates caching just for one specific
*  operationId - the cacheAllOperationId - while allowing all
*  other requests to go to the server normally.
* @type {boolean}
* @default true
*/
cacheAcrossOperationIds?: boolean;

/**
* The maximum time, in seconds, to maintain the client-side cache. If a fetch occurs after
*  the cacheMaxAge has expired, the current cache will be dropped and another complete
*  cache fetched.
* @type {number}
* @default "60"
*/
cacheMaxAge?: number;

/**
* For a DataSource.cacheAllData or client-only DataSource, a set of records to use as a dataset, 
*  specified as an Array of JavaScript Objects representing records.
*  
*  See clientOnlyDataSources for ways to populate a
*  client-only DataSource with cache data.
*  
*  Additionally, when a DataSource is loaded in DataSource.mockMode, cacheData,
*  if provided, is used as the mock data.
*  
*  For any other DataSource, cacheData is dropped when loaded.
* @type {Array<Partial<Record>>}
* @default null
*/
cacheData?: Array<Partial<Record>>;

/**
* For a client-only DataSource, a set of records to use as a dataset, specified as an
*  Array of JavaScript Objects. 
*  
*  See clientOnlyDataSources for ways to populate a
*  client-only DataSource with test data.
* @deprecated * In favor of DataSource.cacheData.

* @type {Array<Partial<Record>>}
* @default null
*/
testData?: Array<Partial<Record>>;

/**
* Set this attribute if you need to send the dsRequest.parentNode to the server-side.
* @type {boolean}
* @default false
*/
sendParentNode?: boolean;

/**
* When a DataSource is not DataSource.cacheAllData:true and a fetch results in the
*  entire dataset being retrieved, this attribute being set to true causes the DataSource
*  to automatically switch to cacheAllData:true and prevent further server-trips for fetch 
*  requests.
*  
*  DataSource.cacheAllData is automatically enabled in either of these conditions:
*  
*   
* - 
*    The request has no criteria and no startRow/endRow request properties. The latter can
*    be accomplished by disabling paging with a DataBoundComponent.dataFetchMode
*    setting of "basic" or "local" or by an explicit fetchData request with those request properties
*    excluded.
*  
*  
* - 
*    The request has no criteria but has startRow/endRow specified and the response received has
*    all data available (startRow:0 and endRow:totalRows).
*  
*  
* @type {boolean}
* @default false
*/
autoCacheAllData?: boolean;

/**
* When set, causes a DataSource.clientOnly or DataSource.cacheAllData DataSource to 
*  create a second DataSource to perform it's one-time fetch. By default, this attribute
*  will be considered true when clientOnly is true, cacheAllData is false or unset and
*  a dataURL or testFileName is specified on the DataSource.
* @type {boolean}
* @default null
*/
useTestDataFetch?: boolean;

/**
* Whether to convert relative date values to concrete date values before sending to the 
*  server. Default value is true, which means that the server does not need to understand 
*  how to filter using relative dates - it receives all date values as absolute dates.
* @type {boolean}
* @default true
*/
autoConvertRelativeDates?: boolean;

/**
* Causes Tree.discoverTree to be called on dsResponse.data in order to automatically
*  discover tree structures in the response data.
*  
*  If autoDiscoverTree is set, discoverTree() is called after the default dsResponse.data
*  has been derived (OperationBinding.recordXPath and
*  DataSourceField.valueXPath have been applied) and after
*  DataSource.transformResponse has been called.
*  
*  If a DataSourceField is declared with
*  DataSourceField.childrenProperty, discoverTree() will be invoked with
*  DiscoverTreeSettings.newChildrenProperty set to the name of
*  the field marked as the childrenField. Similarly, if the DataSource has a
*  DataSource.titleField it will be used as the
*  DiscoverTreeSettings.nameProperty.
* @type {boolean}
* @default false
*/
autoDiscoverTree?: boolean;

/**
* Settings to use when discoverTree() is automatcially called because
*  DataSource.autoDiscoverTree is set to true for this DataSource
* @type {DiscoverTreeSettings}
* @default null
*/
discoverTreeSettings?: DiscoverTreeSettings;

/**
* For DataSources using the sqlDataSource for
*  persistence, whether to use ANSI-style joins (ie, joins implemented with JOIN directives
*  in the table clause, as opposed to additional join expressions in the where clause).
*  The default value of null has the same meaning as setting this flag to false.
*  
*  Note, outer joins (see DataSourceField.joinType) only work with certain
*  database products if you choose not to use ANSI joins. Other than that, the join 
*  strategies are equivalent.
*  
*  If you wish to switch on ANSI-style joins for every DataSource, without the need to 
*  manually set this property on all of them, set
*  server_properties flag 
*  sql.useAnsiJoins to true.
* @type {boolean}
* @default null
*/
useAnsiJoins?: boolean;

/**
* For this dataSource, should the millisecond portion of time and datetime values be 
*  trimmed off before before being sent from client to server or vice versa. By default, 
*  millisecond information is preserved (ie, it is not trimmed). You only need to consider
*  trimming millisecond values if their presence causes a problem - for example, a custom
*  server that is not expecting that extra information and so fails parsing.
*  
*  Note that there is no inherent support for millisecond precision in SmartClient widgets;
*  if you need millisecond-precise visibility and editability of values in your client, 
*  you must write custom formatters and editors (or sponsor the addition of such things to
*  the framework). Server-side, millisecond-precise values are delivered to and obtained
*  from DataSources, so DataSource implementations that are capable of persisting and 
*  reading millisecond values should work transparently. Of the built-in DataSource types,
*  the JPA and Hibernate DataSources will transparently handle millisecond-precise values
*  as long as the underlying database supports millisecond precision, and the underlying 
*  column is of an appropriate type. The SQLDataSource provides accuracy to the nearest
*  second by default; you can switch on millisecond precision per-field with the 
*  DataSourceField.storeMilliseconds attribute.
* @type {boolean}
* @default null
*/
trimMilliseconds?: boolean;

/**
* If set to "false", transformation of values for
*  DataSourceField.multiple fields, normally controlled by
*  DataSourceField.multipleStorage, is instead disabled for this entire DataSource.
* @type {boolean}
* @default null
*/
transformMultipleFields?: boolean;

/**
* Enables saving of a log of changes to this DataSource in a second DataSource with the same
*  fields, called the "audit DataSource". NOTE: this feature applies to Enterprise
*  Edition only; for more information on edition-specific features, see
*  http://smartclient.com/product.
*  
*  When auditing is enabled, every time a DSRequest modifies this DataSource, a Record is added
*  to the audit DataSource that shows the record as it existed after the change was made (or
*  for a "remove", the values of the record at the time of deletion). In addition, the audit
*  DataSource has the following additional metadata fields:
*  
*  
* - DataSource.auditTypeFieldName: type of the change ("update", "add" or "remove")
*  
* - DataSource.auditUserFieldName: username of the user that made the change. The username is
*    determined in the same way that the 
*    OperationBinding.requiresRole subsystem determines the 
*    current user.
*  
* - DataSource.auditTimeStampFieldName: a field of type "datetime" recording
*    the timestamp of the change
*  
* - DataSource.auditRevisionFieldName: a field of type "sequence" recording a
*    simple increasing integer value
*  
* - DataSource.auditChangedFieldsFieldName: a
*    DataSourceField.multiple field of type "string". For "update"
*    operations, records which fields have changed; otherwise, null
*  
*  
*  If any of the field names above collide with field names of the DataSource being audited,
*  an integer suffix will also be added, starting with 2 (for example, "audit_modifier2", then
*  "audit_modifier3", etc).
*  
*  To omit a data field from the automatically generated audit DataSource, just set
*  DataSourceField.audit to false. Audit can be disabled for a given DSRequest via
*  the server-side API DSRequest.setSkipAudit(), or for a specific operation 
*  via the OperationBinding.skipAudit setting.
*  
*  Note: The DataSource audit feature works only with single row operations; operations with 
*  OperationBinding.allowMultiUpdate enabled are not supported.
*  
*  Auto-generated Audit DataSources
*  
*  The audit DataSource is normally automatically generated, and unless otherwise specified
*  with DataSource.auditDataSourceID, the ID of the audit DataSource will be
*  audit_[OriginalDSID]. 
*  
*  By default, the automatically generated audit DataSource will be of the same type as the
*  DataSource being audited, however, if the DataSource being audited is not already a
*  SQLDataSource, we recommend using DataSource.auditDSConstructor to use
*  a SQLDataSource as the audit DataSource. This is because a SQLDataSource used an audit
*  DataSource will automatically generate a SQL table for storing audit data the first time
*  changes are made. JPA would require manual creation of a Java Bean, and Hibernate requires 
*  hbm2ddl.auto=update to be set,
*  which is widely considered unsafe for production use.
*  
*  Automatically created audit DataSources can be loaded and queried just like other
*  DataSources, using the DataSourceLoader, and using the server-side API
*  DataSource.getAuditDataSource(). However, you must load the DataSource
*  being audited before loading its automatically created audit DataSource.
*  
*  Note, that automatic SQL tables creation can be disabled. See 
*  DataSource.autoCreateAuditTable for details.
*  
*  Manually created Audit DataSources
*  
*  The audit DataSource can also be manually created. In this case, you can 
*  can either follow the naming conventions described above for the ID of the audit DataSource
*  and the names of metadata fields, or use the linked properties to assign custom names. If
*  you omit any data fields from the tracked DataSource in your audit DataSource, those fields
*  will be ignored for auditing purposes, exactly as though DataSourceField.audit had
*  been set to false for an automatically-generated audit DataSource.
*  
*  Also, note that in case of manually defined audit DataSource, if this DataSource
*  is defined so it inherits the audited DataSource, all the audited DataSource's fields
*  will be inherited, this including the primary keys. Since for the audit
*  DataSource the primary key should be the revision field, in order to prevent the
*  audit DataSource having two primary keys, the inherited DataSource's primary key 
*  will have to be declared in audit DataSource, but with the primaryKey attribute 
*  omitted (as well not being of type "sequence") in the audit DataSource.
* @type {boolean}
* @default false
*/
audit?: boolean;

/**
* For DataSources with DataSource.audit, optionally specifies the ID of the audit
*  DataSource. If this property is not specified, the ID of the audit DataSource will
*  be audit_[OriginalDSID] 
* @type {string}
* @default null
*/
auditDataSourceID?: string;

/**
* For DataSources with DataSource.audit, specifies the field name used to store
*  the user who performed the operation. If empty string is specified as the field name, the
*  audit DataSource will not store this field.
* @type {string}
* @default "audit_modifier"
*/
auditUserFieldName?: string;

/**
* For DataSources with DataSource.audit, specifies the field name used to store
*  the timestamp when the operation was performed (in a field of type "datetime"). If empty
*  string is specified as the field name, the audit DataSource will not store this field.
* @type {string}
* @default "audit_changeTime"
*/
auditTimeStampFieldName?: string;

/**
* For DataSources with DataSource.audit, specifies the field name used to store
*  the revision number for the change (in a field of type "sequence"). If empty
*  string is specified as the field name, the audit DataSource will not store this field.
* @type {string}
* @default "audit_revision"
*/
auditRevisionFieldName?: string;

/**
* For DataSources with DataSource.audit, specifies the field name used to store
*  the names of the fields which were updated. If empty string is specified as the field name,
*  the audit DataSource will not store this field.
*  
*  Note that this field will only be set for DSOperationType operations.
* @type {string}
* @default "audit_changedFields"
*/
auditChangedFieldsFieldName?: string;

/**
* For DataSources with DataSource.audit, specifies the length of the field used 
*  to store the names of the fields which were updated. See also 
*  DataSource.auditChangedFieldsFieldName
*  
*  To set the changedFields field length for all DataSources that do not override the default,
*  set audit.auditChangedFieldsFieldLength in your server.properties
*  file. For example 
*     audit.auditChangedFieldsFieldLength: 512
*  
* @type {Integer}
* @default "255"
*/
auditChangedFieldsFieldLength?: Integer;

/**
* For DataSources with DataSource.audit, specifies the field name used to store
*  the DSOperationType (in a field of type "text"). If empty
*  string is specified as the field name, the audit DataSource will not store this field.
* @type {string}
* @default "audit_operationType"
*/
auditTypeFieldName?: string;

/**
* For DataSources with DataSource.audit, optionally specifies the
*  DataSource.serverConstructor for the automatically generated audit DataSource. The
*  default is to use the same serverConstructor as the DataSource where
*  audit="true" was declared.
*  
*  This property is primarily intended to allow the use of SQLDataSource
*  (DataSource.serverType) as an audit DataSource for a DataSource that
*  might be of another type. For example, you might have a DataSource that implements all CRUD
*  operations via Java logic in dmiOverview methods, and so doesn't provide generic
*  storage; by using SQLDataSource as the type of your audit DataSource, you don't need to
*  implement your own scheme for storing and querying audit data, and the necessary audit
*  tables will be automatically generated in the database.
*  
*  Similarly, even if you do use a reusable DataSource type such as the built-in JPADataSource,
*  using SQLDataSource for audit DataSources means there's no need to write a JPA bean just to
*  achieve storage of an audit trail.
*  
*  To simplify this intended usage, the string "sql" is allowed for
*  auditDSConstructor as a means of specifying that the built-in SQLDataSource class
*  should be used. For any other type, use the fully qualified Java classname, as is normal
*  for serverConstructor.
* @type {string}
* @default null
*/
auditDSConstructor?: string;

/**
* Setting autoCreateAuditTable to true indicates that audit DataSource 
*  will automatically create SQL table when DataSource.audit is enabled.
*  
*  Note, that autoCreateAuditTable attribute takes effect only if framework setting 
*  audit.autoCreateTables in server.properties is set to false,
*  which by default is set to true.
* @type {boolean}
* @default true
*/
autoCreateAuditTable?: boolean;

/**
* Only applicable to binaryFields on DataSource.audit
*  DataSources.
*  
*  When determining if a binary field has changed for auditing purposes, should we compare the
*  metadata values (ie, the associated _filename and _filesize 
*  fields) or the actual binary content? If you set this flag to false, this will cause 
*  SmartClient to fetch the existing content of any binary field from the database before any
*  update, and then compare it byte by byte to the new content. You should consider if this 
*  will have performance implications for your application, particularly if you store large 
*  binary values.
*  
*  Note that value comparison of any kind is only performed if the field's 
*  DataSourceField.audit setting is "change"
* @type {boolean}
* @default true
*/
compareMetadataForAuditChangeStatus?: boolean;

/**
* User-visible name for this DataSource.
*  
*  For example, for the supplyItem DataSource, "Supply Item".
*  
*  If is unset, getAutoTitle() method will be used with dataSource.ID.
*  value in order to derive a default value for the title.
*  
*  For example "employee" ID will be derived to "Employee", "team_member" ID will be 
*  derived to "Team Member".
* @type {string}
* @default "dataSource.ID"
*/
title?: string;

/**
* User-visible plural name for this DataSource.
*  
*  For example, for the supplyItem DataSource, "Supply Items".
*  
*  Defaults to dataSource.title + "s".
* @type {string}
* @default "dataSource.ID"
*/
pluralTitle?: string;

/**
* Best field to use for a user-visible title for an individual record from this dataSource.
*  
*  For example, for a DataSource of employees, a "full name" field would probably most clearly
*  label an employee record.
*  
*  If not explicitly set, titleField looks for fields named "title", "label", "name", and "id"
*  in that order. If a field exists with one of those names, it becomes the titleField. If not,
*  then the first field is designated as the titleField.
* @type {string}
* @default "see below"
*/
titleField?: string;

/**
* Designates a field of FieldType:"image" as the field to use when rendering a
*  record as an image, for example, in a TileGrid.
*  
*  For example, for a DataSource of employees, a "photo" field of type "image" should be
*  designated as the iconField.
*  
*  If not explicitly set, iconField looks for fields named "picture", "thumbnail", "icon",
*  "image" and "img", in that order, and will use any of these fields as the iconField if it
*  exists and has type "image". 
*  
*  To avoid any field being used as the iconField, set iconField to null.
* @type {string}
* @default "see below"
*/
iconField?: string;

/**
* Name of the field that has the second most pertinent piece of textual information in the
*  record, for use when a DataBoundComponent needs to show a short summary of a record.
*  
*  For example, for a DataSource of employees, a "job title" field would probably be the second
*  most pertinent text field aside from the employee's "full name".
*  
*  Unlike DataSource.titleField, infoField is not automatically determined in the absence of an
*  explicit setting.
* @type {string}
* @default null
*/
infoField?: string;

/**
* Name of the field that has the most pertinent numeric, date, or enum value, for use when a
*  DataBoundComponent needs to show a short summary of a record.
*  
*  For example, for a DataSource of employees, good choices might be the "salary" field, "hire
*  date" or "status" (active, vacation, on leave, etc).
*  
*  Unlike DataSource.titleField, dataField is not automatically determined in the absence of an
*  explicit setting.
* @type {string}
* @default null
*/
dataField?: string;

/**
* Name of the field that has a long description of the record, or has the primary text data
*  value for a record that represents an email message, SMS, log or similar.
*  
*  For example, for a DataSource representing employees, a field containing the employee's
*  "bio" might be a good choice, or for an email message, the message body.
*  
*  If descriptionField is unset, it defaults to any field named "description" or "desc" in the
*  record, or the first long text field (greater than 255 characters) in the record, or null if
*  no such field exists.
* @type {string}
* @default null
*/
descriptionField?: string;

/**
* fieldName for a field in the dataSource expected to contain an explicit array of child nodes.
*  Enables loading a databound tree as a hierarchical data structure, rather than a flat list of
*  nodes linked by foreignKey.
*  Note this is an alternative to setting DataSourceField.childrenProperty directly on
*  the childrenField object.
*  By default the children field will be assumed to be DataSourceField.multiple,
*  for XML databinding. This implies that child data should be delivered in the format:
*  
*    &lt;childrenFieldName&gt;
*      &lt;item name="firstChild" ...&gt;
*      &lt;item name="secondChild" ...&gt;
*    &lt;/childrenFieldName&gt;
*  
*  However data may also be delivered as a direct list of childrenFieldName 
*  elements:
*  
*    &lt;childrenFieldName name="firstChild" ...&gt;
*    &lt;childrenFieldName name="secondChild" ...&gt;
*  
*  If you want to return your data in this format, you will need to explicitly set 
*  multiple to false in the appropriate dataSource field definition.
* @type {string}
* @default null
*/
childrenField?: string;

/**
* This property allows you to write and use custom DataSource subclasses on the server, by 
*  specifying either 
*  
* - the fully-qualified name of the DataSource subclass that should be instantiated
*  server-side for this dataSource, or
*  
* - the token "spring:" followed by a valid Spring bean ID, if you wish to instantiate 
*  your custom dataSource object using Spring dependency injection. For example, 
*  "spring:MyDataSourceBean". See also serverInit for special
*  concerns with framework initialization when using Spring. It is also particularly important
*  that you read the discussion of caching and thread-safety linked to below, as there are 
*  special considerations in this area when using Spring.
*  
* - the token "cdi:" followed by a valid CDI bean name, if you wish to instantiate 
*  your custom dataSource object using CDI dependency injection. For example, 
*  "cdi:MyDataSourceBean".
*  
*  
*  One reason you might wish to do this would be to override the validate() method to provide 
*  some arbitrary custom validation (such as complex database lookups, validation embedded in 
*  legacy applications, etc). It is also possible - though obviously a more substantial task -
*  to override the execute() method in your custom DataSource. This is one way of creating 
*  a completely customized DataSource implementation.
*  
*  Note: If you use this property, you are responsible for making sure that it refers to
*  a valid server-side class that extends com.isomorphic.datasource.BasicDataSource,
*  or to a Spring bean of the same description.
*  If your implementation relies on methods or state only present in certain specialized 
*  subclasses of DataSource (for example, you want the normal behavior and features of a 
*  HibernateDataSource, but with a specialized validate() method), then you should extend the 
*  subclass rather than the base class.
*  
*  NOTE: Please take note of the points made in 
*  serverDataSourceImplementation of caching and thread-safety
*  issues in server-side DataSources.
* @type {string}
* @default null
*/
serverConstructor?: string;

/**
* For dataSources of DataSource.serverType "sql", determines whether we qualify column
*  names with table names in any SQL we generate. This property can be overridden on specific
*  operationBindings.
* @type {boolean}
* @default true
*/
qualifyColumnNames?: boolean;

/**
* If true, indicates that the SmartClient Server should automatically apply a 
*  ValidatorType of "hasRelatedRecord" to every field 
*  on this dataSource that has a DataSourceField.foreignKey defined.
* @type {boolean}
* @default null
*/
validateRelatedRecords?: boolean;

/**
* This property has different meanings depending on the DataSource.serverType:
*  
*  For SQL DataSources (DataSources with serverType "sql")
*  If set, results from the database will be used to create one instance of the indicated Java
*  bean per database row. Otherwise a Map is used to represent each row retrieved from SQL.
*  
*  With this feature active, a DSResponse from this DataSource will contain a Collection of
*  instances of the indicated beanClassName, available via DSResponse.getData().
*  This creates a couple of possibilities:
*  
*  
*  Add business logic for derived properties, such as computed formulas
*  For example, declare a DataSourceField named "revenueProjection". By default this
*  field will call getRevenueProjection() on your bean to retrieve the value to send to the
*  client. Your implementation of getRevenueProjection() could apply some kind of formula to
*  other values loaded from the database.
*  Call business logic on retrieved beans via DMI
*  By adding a dmiOverview method that calls DSRequest.execute() to retrieve a DSResponse,
*  you have an opportunity to call business logic methods on the beans representing each
*  row affected by the DSRequest. For example, notify a related BPEL process of changes to
*  certain fields.
*  
*  
*  By using beanClassName on a specific OperationBinding, you can:
*  
*  
* - Use a bean to represent your data only when it matters; for example, avoid the overhead
*    of using a bean for "fetch" operations, but do use a bean for "update" operations so
*    that you can execute relevant business logic after the update completes.
*  
* - Skip the use of beans for complex reporting queries that produce results unrelated to
*    your persistent object model. Set beanClassName to blank ("") on a specific
*    operationBinding to override DataSource.beanClassName for that specific operation.
*  
* - For SQL joins that produce additional data fields, use a special, operation-specific
*    bean that represents a join of multiple entities and contains business logic specific
*    to that joined dataset
*  
*  
*  Note that beanClassName affects what numeric field types will be used for 
*  inbound DSRequest data. For fields with numeric types, the DSRequest.data 
*  in DSRequests will automatically be converted to the type of the target field, before the 
*  request is received in a DMI. For details, see dsRequestBeanTypes.
*  
*  Note that dmiOverview also has a built-in facility for populating a bean 
*  with the inbound DSRequest.data - just declare the bean as a method argument.
*  
*  For generic DataSources (DataSources with serverType "generic")
*  visualBuilder sets this property when it creates a generic 
*  DataSource using the Javabean Wizard. It has no built-in server-side effects.
*  
*  For Hibernate DataSources (DataSources with serverType "hibernate")
*  The name of the Java bean or POJO class that is mapped in Hibernate. This will typically 
*  be the fully-qualified class name - eg com.foo.MyClass - but it may be the 
*  simple class name - just MyClass - or it may be some other value. It all 
*  depends on how your classes have been mapped in Hibernate.
*  
*  The declared Java bean will affect how its properties will be mapped to built-in numeric 
*  types, see hibernateIntegration for details.
*  
*  Note: If you are intending to use Hibernate as a data-access layer only, 
*  you do not need to create Hibernate mappings or Java objects: SmartClient will generate
*  everything it needs on the fly.
*  
*  For JPA DataSources (DataSources with serverType "jpa" or "jpa1")
*  The fully qualified class name of the JPA annotated entity.
*  
*  NOTE for Hibernate and JPA users: When you use JPA, or use Hibernate as a full ORM
*  system (ie, not just allowing SmartClient Server to drive Hibernate as a data access layer),
*  the beans returned on the server-side are live. This means that if you make any 
*  changes to them, the ORM system will persist those changes. This is true even if the beans
*  were created as part of a fetch operation.
*  
*  This causes a problem in the common case where you want to use a DMI or custom DataSource 
*  implementation to apply some post-processing to the beans fetched from the persistent 
*  store. If you change the values in the beans directly, those changes will be persisted.
*  
*  If you want to alter the data returned from a JPA or Hibernate persistent store as part of
*  a fetch request just so you can alter what gets sent to the client, you can use the 
*  server-side DSResponse's getRecords() method. This will return
*  your bean data in "record" format - ie, as a List of Maps. You can alter these records 
*  without affecting your persistent store, and then call setData() on the 
*  DSResponse), passing the altered list of records. See the server-side Javadocs
*  for DSResponse for details of these two methods.
* @type {string}
* @default null
*/
beanClassName?: string;

/**
* Criteria that are implicitly applied to all fetches made through this DataSource. These
*  criteria are never shown to or edited by the user and are cumulative with any other 
*  criteria provided on the DSRequest.
*  
*  For example, a DataSource might *always* implicitly limit its fetch results to records 
*  owned by the current user's department. Components and ResultSets requesting data 
*  from the DataSource can then apply further implicitCriteria of their own, separately 
*  from their regular, user-editable criteria.
*  
*  For instance, a grid bound to this dataSource might be further limited to 
*  implicitly show only the subset of records created by the current user. See 
*  DataBoundComponent.implicitCriteria and ResultSet.implicitCriteria for 
*  more on these localized options.
*  
*  Note that, while implicitCriteria can be declared in a server DataSource 
*  file using componentXML, it is an entirely client-side 
*  feature, added by client-side components. So it does not affect server-side requests, 
*  and will not be added to client-side requests that don't come from a SmartClient UI 
*  (eg RestHandler).
* @type {Criteria}
* @default null
*/
implicitCriteria?: Criteria;

/**
* Sets what level of JSON validation will apply for this DataSource.
*  
*  Note that the point of "partial" validation mode is that if the JSON ihat's delivered is
*  correct, we'll still need to validate to get "date" and such in the correct time, but
*  shouldn't need to for the rest.
* @type {SkipJSONValidation}
* @default "none"
*/
skipJSONValidation?: SkipJSONValidation;

/**
* Namespaces definitions to add to the root element of outbound XML messages sent to a web
*  service, as a mapping from namespace prefix to namespace URI.
*  
*  The default value is:
*  
*   globalNamespaces : {
*    xsi: "http://www.w3.org/2001/XMLSchema-instance",
*    xsd: "http://www.w3.org/2001/XMLSchema"
*   },
*  
*  This default value allows the use of the xsi:type and xsi:nil attributes without further
*  declarations.
*  
*  Note that some web services will only accept specific revisions of the XML Schema URI.
*  If xsi-namespaced attributes seem to be ignored by an older webservice, try the URI
*  "http://www.w3.org/1999/XMLSchema-instance" instead.
* @type {object}
* @default "..."
*/
globalNamespaces?: object;

/**
* Very advanced: for servers that do not support paging, and must return large numbers of XML
*  records in one HTTP response, SmartClient breaks up the processing of the response
*  in order to avoid the "script running slowly" dialog appearing for an end user.
*  
*  If you have a relatively small number of records with a great deal of properties or
*  subobjects on each record, and you have not set DataSource.dropExtraFields to eliminate unused
*  data, and you see the "script running slowly" dialog, you may need to set this number
*  lower.
* @type {Integer}
* @default "150"
*/
resultBatchSize?: Integer;

/**
* The native field name used by this DataSource on the server to represent the
*  fileName for fileSource operations. Any
*  extensions to the fileName to indicate type or format (e.g. ".ds.xml") are stored in the
*  DataSource.fileTypeField and DataSource.fileFormatField, if specified
*  for this DataSource.
* 
*  If not specified for a DataSource, the fileNameField will be inferred
*  on the server as follows:
* 
*  
*  -->
*  
* - If there is a field named "fileName", "name", or "title",
*    then that field is used.
*  
* - Otherwise, if there is a single primary key, and it has the type "text",
*    then that field is used.
*  
* - Otherwise, an error is logged
*  
* @type {string}
* @default null
*/
fileNameField?: string;

/**
* The native field name used by this DataSource on the server to represent the
*  fileType for fileSource. 
* 
*  If the fileTypeField is not configured, then a field named "fileType"
*  will be used, if it exists. Otherwise, the DataSource will not track
*  fileTypes -- this may be acceptable if, for instance, you use a separate
*  DataSource for each fileType.
* 
*  The fileType is specified according to the extension that would have been used in the
*  filesystem -- for instance, the fileType for employees.ds.xml would be "ds".
* @type {string}
* @default null
*/
fileTypeField?: string;

/**
* The native field name used by this DataSource on the server to represent
*  the fileFormat for fileSource. 
* 
*  If the fileFormatField is not configured, then a field named
*  "fileFormat" will be used, if it exists. Otherwise, the DataSource will not
*  track fileFormats -- this may be acceptable if, for instance, the
*  fileFormat is always the same.
* 
*  The fileFormat is specified according to the extension that would have been used in the
*  filesystem -- for instance, the fileFormat for employees.ds.xml would be "xml".
* @type {string}
* @default null
*/
fileFormatField?: string;

/**
* The native field name used by this DataSource on the server to represent
*  the fileContents for fileSource.
* 
*  If the fileContentsField is not configured, then a field named
*  "fileContents" or "contents" will be used, if it exists. If not found, the longest text field which is not the
*  DataSource.fileNameField, DataSource.fileTypeField or
*  DataSource.fileFormatField will be used.
* 
*  Note that the only method which will actually return the fileContents
*  is DataSource.getFile -- the other fileSource
*  methods omit the fileContents for the sake of efficiency.
* @type {string}
* @default null
*/
fileContentsField?: string;

/**
* The native field name used by this DataSource on the server to represent
*  fileLastModified for fileSource.
* 
*  If the fileLastModifiedField is not configured, then a field named
*  "fileLastModified" will be used, if it exists. 
*  Otherwise, the server will look for a field with a "modifierTimestamp" type.
*  If that is not found, the DataSource will not keep track of the last
*  modified date.
* @type {string}
* @default null
*/
fileLastModifiedField?: string;

/**
* The native field name used by this DataSource on the server to represent
*  fileVersion for fileSource.
*  
*  Automatic file versioning is configured by the presence of this property: if you want
*  automatic versioning for a FileSource DataSource, it is sufficient to simply define a 
*  fileVersionField. When automatic versioning is on:
*  
* - Calls to DataSource.saveFile will save a new version of the file, retaining previous 
*  versions up to the maximum configured by DataSource.maxFileVersions; when that
*  maximum is reached, the oldest version is overwritten
*  
* - The DataSource.getFile API always returns the most recent version
*  
* - The DataSource.listFiles API only includes the most recent version of any file
*  
* - You can view and retrieve earlier versions of a file with the 
*  DataSource.listFileVersions and DataSource.getFileVersion APIs. Note that retrieving a
*  previous version of a file and then calling saveFile() goes through the
*  normal process of saving a new version
*  
*  
*  The fileVersion field is expected to be of type "datetime", and automatic 
*  versioning will not work otherwise. Note, to minimize the possibility of version 
*  timestamp collisions, we recommend that fileVersion fields specify 
*  DataSourceField.storeMilliseconds: true.
*  
*  If the fileVersionField is not configured, no automatic file versioning will be done.
* @type {string}
* @default null
*/
fileVersionField?: string;

/**
* If DataSource.fileVersionField is enabled for a FileSource
*  DataSource, this property configures the maximum number of versions to retain.
* @type {Integer}
* @default "20"
*/
maxFileVersions?: Integer;

/**
* For DataSources with type DSServerType, looks
*  up the locations to use as DataSource.projectFileLocations from the project's configuration (i.e.
*  project.properties, server_properties
*  etc.).
* 
*  For instance, to look up the value of project.datasources and use it
*  for projectFileLocations, use "datasources" as the 
*  projectFileKey.
* 
*  If you specify both projectFileKey and 
*  projectFileLocations, then both with be used, with the
*  projectFileLocations applied last.
* @type {string}
* @default null
*/
projectFileKey?: string;

/**
* For DataSources with type DSServerType,
*  specifies locations for the project files. In XML, each location is
*  expressed with a &lt;location&gt; tag, e.g.:
* 
*  
*    &lt;projectFileLocations&gt;
*      &lt;location&gt;[WEBROOT]/shared/ds&lt;/location&gt;
*      &lt;location&gt;ds://datasources&lt;/location&gt;
*    &lt;/projectFileLocations&gt;
*  
* 
*  Directories should be specified as absolute paths on the server. If you
*  want to construct a webroot-relative path, then prefix the path with
*  [WEBROOT] (unlike in
*  server_properties, where you would use
*  $webRoot as the prefix).
* 
*  To specify another DataSource to be used via 
*  fileSource, use ds://dsName
*  (where "dsName" is the name of the other DataSource).
* 
*  A projectFile DataSource uses the standard
*  fileSource field names: fileName,
*  fileType, fileFormat,
*  fileContents, fileSize and
*  fileLastModified. When defining a projectFile
*  DataSource, you can use DataSource.inheritsFrom with
*  a value of "ProjectFile" to inherit definitions for these fields -- e.g.:
* 
*  
*    &lt;DataSource ID="MyDataSources" type="projectFile" inheritsFrom="ProjectFile"&gt;
*      &lt;projectFileLocations&gt;
*        &lt;location&gt;[WEBROOT]/shared/ds&lt;/location&gt;
*        &lt;location&gt;ds://datasources&lt;/location&gt;
*      &lt;/projectFileLocations&gt;
*    &lt;/DataSource&gt; 
*  
* 
*  For directory locations, the
*  fileName is relative to the directory specified. Note that
*  the fileName does not include any extension for type or
*  format. For instance, for "employees.ds.xml", the fileName
*  would be "employees", the fileType would be "ds" and the
*  fileFormat would be "xml".
* 
*  A projectFile DataSource executes the various 
*  fileSource in the following manner.
*  The general rule is that fileName, fileType,
*  and fileFormat are treated as primary keys. If files with the
*  same combination of those attributes exist in more than one of the
*  configured locations, the locations are considered in reverse
*  order, with priority given to the location listed last. When modifying
*  an existing file, the last location which contains the file will be
*  used. When creating a new file, the file will be created in the last
*  configured location.
* 
*  
*  DataSource.listFiles 
*  Returns a combined list of files from
*    all configured locations. Note that listFiles does not 
*    recurse into subdirectories. If the same combination of 
*    fileName / fileType / fileFormat exists in more than
*    one configured location, then the data for fileSize and
*    fileLastModified will be taken from the last configured
*    location which contains the file.
* 
*  DataSource.hasFile
*  Indicates whether the file exists in any of the configured locations.
* 
*  DataSource.getFile
*  Returns file data by searching the locations in reverse order.
* 
*  DataSource.saveFile
*  If the file exists, it will be saved in the last location in which
*    it exists. If it is a new file, it will be saved in the last
*    configured location.
* 
*  DataSource.renameFile
*  The file will be renamed in the last location in which it exists.
*    Note that if the file exists in more than one location, the rename
*    will not affect other locations. Thus, a subsequent listFiles
*    operation will return the file from the other location (as well
*    as the renamed file).
* 
*  DataSource.removeFile
*  The file will be removed from the last location in which it exists.
*    Note that if the file exists in more than one location, the removal
*    will not affect other locations. Thus, a subsequent listFiles
*    operation will return the file from the other location.
* 
*  
* 
*  For convenience, a projectFile DataSource also responds to the standard
*  DataSource operations, in the following manner:
* 
*  
*  add
*  Executes a saveFile operation, either adding the file or updating
*    an existing file.
* 
*  fetch
*  Executes a listFiles operation. Note that the results will not
*    include the fileContents. In order to obtain the
*    fileContents, you must use a
*    DataSource.getFile.
* 
*  update
*  Executes a renameFile operation. Note that this will not update
*    the fileContents -- for that, you need to use "add", or
*    a DataSource.saveFile. 
* 
*  remove
*  Executes a removeFile operation.
* 
*  
* 
*  If you specify both DataSource.projectFileKey and 
*  projectFileLocations, then both with be used, with the
*  projectFileLocations applied last.
* @type {Array<Partial<string>>}
* @default null
*/
projectFileLocations?: Array<Partial<string>>;

/**
* Before we start editing values in DataBoundComponents bound to this 
*  dataSource, should we perform a deep clone of the underlying values (a "deep clone" is 
*  one created by traversing the original values object recursively, cloning the contents 
*  of nested objects and arrays). If this flag is explicitly set to false, we perform a 
*  shallow clone of the underlying values before edit (a "shallow clone" is a copy created 
*  by simply copying the top-level attributes of an object). Note, this setting only 
*  affects deep-cloning of attributes that are bound to a field; for other, non-field 
*  values, see DataSource.deepCloneNonFieldValuesOnEdit.
*  
*  If this flag is not explicitly set, it defaults to the value of the same-named static 
*  property, DataSource.deepCloneOnEdit. This flag can also be overridden 
*  per-component and per-field - see DataBoundComponent.deepCloneOnEdit and 
*  DataSourceField.deepCloneOnEdit.
*  
*  Note, this flag only has an effect if you are editing a values object that contains nested 
*  objects or arrays, using Canvas.dataPaths. When editing "flat" data - the
*  mainstream case - there is no difference between a deep clone and a shallow clone.
* @type {boolean}
* @default null
*/
deepCloneOnEdit?: boolean;

/**
* When editing values in DataBoundComponents bound to this dataSource, should we
*  perform a deep clone of values that are not associated with a field (ie, attributes on
*  the record that do not map to a component field either directly by name, or by 
*  FormItem.dataPath). If this flag is not explicitly set, it defaults to
*  the value of the same-named static property, 
*  DataSource.deepCloneNonFieldValuesOnEdit. This flag can also be 
*  overridden per-component - see DataBoundComponent.deepCloneNonFieldValuesOnEdit.
*  
*  Note, a "deep clone" is one created by traversing the original values object 
*  recursively, cloning the contents of nested objects and arrays; a "shallow clone" is a
*  copy created by simply copying the top-level attributes of an object; if you have 
*  nested objects that are copied like this, the "copies" are actual references to the 
*  original objects.
*  
*  Like the other deepCloneOnEdit settings, this flag only has an effect if you are 
*  editing a values object that contains nested objects or arrays.
* @type {boolean}
* @default null
*/
deepCloneNonFieldValuesOnEdit?: boolean;

/**
* Default scriptlet to be executed on the server for each operation. If OperationBinding.script is
*  specified, it will be executed for the operation binding in question instead of running this scriptlet.
*  
*  Scriptlets are used similarly to DMIs configured via DataSource.serverObject or
*  OperationBinding.serverObject -
*  they can add business logic by modifying the DSRequest before it's executed, modifying the
*  default DSResponse, or taking other, unrelated actions. 
*  
*  For example:
*  
*   &lt;DataSource&gt;
*     &lt;script language="groovy"&gt;
*      ... Groovy code ...
*     &lt;/script&gt;
*    ... other DataSource properties
*   &lt;/DataSource&gt;
*  
*  
*  Scriptlets can be written in any language supported by the "JSR 223" standard, including Java
*  itself. See the dmiOverview for rules on how to return data,
*  add additional imports, and other settings.
*  
*  The following variables are available for DMI scriptlets:
*  
*  
* - requestContext: RequestContext (from com.isomorphic.servlet)
*  
* - dataSource: the current DataSource (same as DSRequest.getDataSource())
*  
* - dsRequest: the current DSRequest
*  
* - criteria: shortcut to DSRequest.getCriteria() (a Map)
*  
* - values: shortcut to DSRequest.getValues() (a Map)
*  
* - oldValues: shortcut to DSRequest.getOldValues() (a Map)
*  
* - sqlConnection: SQLDataSource only: the current SQLConnection object. If
*  using DataSource.autoJoinTransactions are enabled, this
*  SQLConnection is in the context of the current transaction.
*  
* - rpcManager: the current RPCManager
*  
* - applicationContext: the Spring ApplicationContext (when applicable)
*  
* - beanFactory: the Spring BeanFactory (when applicable)
*  
*  
*  Scriptlets also have access to a set of contextual variables related to the Servlets
*  API, as follows:
*  
*  
* - servletRequest: the current ServletRequest
*  
* - session: the current HttpSession
*  
* - servletResponse: the current ServletResponse (advanced use only)
*  
* - servletContext: the current ServletContext(advanced use only)
*  
*  As with DMI in general, be aware that if you write scriptlets that depend upon these variables,
*  you preclude your DataSource from being used in the widest possible variety of
*  circumstances. For example, adding a scriptlet that relies on the HttpSession
*  prevents your DataSource from being used in a command-line process.
*  
*  Note that if a dataSource configuration has both a &lt;script&gt; block and
*  a specified OperationBinding.serverObject for some operation, the
*  script block will be executed, and the serverObject ignored.
* @type {string}
* @default null
*/
script?: string;

/**
* For Direct Method Invocation (DMI) binding, declares the ServerObject to use as the default
*  target for all DataSource.operationBindings. Specifying this
*  attribute in an XML DataSource stored on the server enables DMI for this DataSource.
*  
*  Note that if a dataSource configuration has both a OperationBinding.script block and
*  a specified serverObject for some operation, the
*  script block will be executed, and the serverObject ignored.
* @type {ServerObject}
* @default null
*/
serverObject?: ServerObject;

/**
* Optional array of OperationBindings, which provide instructions to the DataSource about how each
*  DSOperation is to be performed.
*  
*  When using the SmartClient Server, OperationBindings are specified in your DataSource
*  descriptor (.ds.xml file) and control server-side behavior such as what Java object to route
*  DSRequest to (OperationBinding.serverObject) or customizations to SQL, JQL and HQL queries
*  (OperationBinding.customSQL, OperationBinding.customJQL and OperationBinding.customHQL).
*  See the ${isc.DocUtils.linkForExampleId('javaDataIntegration', 'Java Integration samples')}.
*  
*  For DataSources bound to WSDL-described web services using
*  DataSource.serviceNamespace, OperationBindings are used to bind each DataSource
*  OperationBinding.operationType to an
*  OperationBinding.wsOperation of a WSDL-described 
*  WebService, so that a DataSource can both fetch and save data to a web
*  service.
*  
*  For example, this code accomplishes part of the binding to the 
*  SalesForce partner web services
*  
*  
*  isc.DataSource.create({
*   serviceNamespace : "urn:partner.soap.sforce.com",
*   operationBindings : [
*     { operationType:"fetch", wsOperation:"query", recordName: "sObject" },
*     { operationType:"update", wsOperation:"update", recordName: "SaveResult" },
*     { operationType:"add", wsOperation:"create", recordName: "SaveResult" },
*     { operationType:"remove", wsOperation:"delete", recordName: "DeleteResult" }
*   ],
*   ...
*  }); 
*  
*  
*  
*  NOTE: additional code is required to handle authentication and other details, see the
*  complete code in smartclientSDK/examples/databinding/SalesForce.
*  
*  For DataSources that contact non-WSDL-described XML or JSON services, OperationBindings can
*  be used to separately configure the URL, HTTP method, input and output processing for each
*  operationType. This makes it possible to fetch JSON data from one URL for the "fetch"
*  operationType and save to a web service for the "update" operationType, while appearing as a
*  single integrated DataSource to a DataBoundComponent such as an
*  ListGrid.canEdit.
*  
*  If no operationBinding is defined for a given DataSource operation, all of the properties
*  which are valid on the operationBinding are checked for on the DataSource itself. 
*  
*  This also means that for a read-only DataSource, that is, a DataSource only capable of fetch
*  operations, operationBindings need not be specified, and instead all operationBinding
*  properties can be set on the DataSource itself. In the ${isc.DocUtils.linkForExampleId('rssFeed', 'RSS Feed')}
*  sample, you can see an example of using OperationBinding properties directly on the 
*  DataSource in order to read an RSS feed.
* @type {Array<Partial<OperationBinding>>}
* @default null
*/
operationBindings?: Array<Partial<OperationBinding>>;

/**
* If a DSRequest arrives from the client that requests
*  serverSummaries, should it be allowed?
*  
*  Note this setting only affects dsRequests that come from the browser (or
*  another client). This setting has no effect on server summaries declared in .ds.xml files or
*  summaries configured in DSRequests created programmatically on the server side, which are
*  always allowed.
*  
*  Default value of null means this DataSource will use the system-wide default, which is set via
*  datasources.allowClientRequestedSummaries in
*  server_properties, and defaults to allowing client-requested
*  summaries.
*  
*  If client-requested summarization is allowed, but the server-side &lt;operationBinding&gt;
*  provides specific summarization settings, the client-requested summarization is ignored.
* @type {boolean}
* @default null
*/
allowClientRequestedSummaries?: boolean;

/**
* Provides a default value for OperationBinding.recordName.
* @type {string}
* @default null
*/
recordName?: string;

/**
* Controls when primary keys are required for "update" and "remove" server operations, when allowMultiUpdate 
*  has not been explicitly configured on either the OperationBinding.allowMultiUpdate or via the server-side API DSRequest.setAllowMultiUpdate().
*  
*  Default value of null means this DataSource will use the system-wide default, which is set via
*  datasources.defaultMultiUpdatePolicy in
*  server_properties, and defaults to not allowing multi updates for 
*  requests associated with an RPCManager, see MultiUpdatePolicy for details.
* @type {MultiUpdatePolicy}
* @default null
*/
defaultMultiUpdatePolicy?: MultiUpdatePolicy;

/**
* If the criteria applied to a fetch type operation contain fields that are not present
*  in the dataSource, should they be ignored when performing filtering on the client.
*  This property is useful for cases where you custom server logic makes use of criteria 
*  values to determine what set of records to return to the client, but the data
*  does not actually have record values for these fields and as such the client-side filtering
*  logic should ignore them.
* @type {boolean}
* @default true
*/
dropUnknownCriteria?: boolean;

/**
* If set to true, both client and server-side advanced filtering used by SmartClient will follow
*  SQL99 behavior for dealing with NULL values, which is often counter-intuitive to users.
*  Specifically, when a field has NULL value, all of the following expressions are false:
*  
*   field == "someValue" (normally false)
*   field != "someValue" (normally true)
*   not (field == "someValue")  (normally true)
*   not (field != "someValue")  (normally false)
*  
*  This property can be overridden per-query by specifying strictSQLFiltering
*  directly as a property on the AdvancedCriteria.
*  
*  NOTE: On the server side, this property is only applicable if you are using the 
*  SQL DataSource; the other built-in types (Hibernate and JPA/JPA2) do not offer this mode.
* @type {boolean}
* @default false
*/
strictSQLFiltering?: boolean;

/**
* By default, all DataSources are assumed to be capable of handling 
*  AdvancedCriteria on fetch or filter type operations. This property may be
*  set to false to indicate that this dataSource does not support 
*  advancedCriteria. See DataSource.supportsAdvancedCriteria for further information on this.
*  
*  NOTE: If you specify this property in a DataSource descriptor 
*  (.ds.xml file), it is enforced on the server. This means that if you run 
*  a request containing AdvancedCriteria against a DataSource that advertises itself as 
*  allowAdvancedCriteria:false, it will be rejected.
* @type {boolean}
* @default null
*/
allowAdvancedCriteria?: boolean;



/**
*  Call this method to switch cacheAllData on or off after initialization. Passing a 
*  shouldCache value of false clears any existing client-side cache,
*  cancels any outstanding requests for a full cache and issues any other pending requests
*  normally.
*  
* 
* @param {boolean} New value for DataSource.cacheAllData
*/
setCacheAllData(shouldCache:boolean): void;

/**
*  Call this method to set the data in the client-side cache after initialization.
* 
* @param {Array<Partial<Record>>} Array of records to apply as the client-side cache
*/
setCacheData(data:Array<Partial<Record>>): void;

/**
*  Call this method to set the data in the client-side test-data after initialization. 
* 
* @param {Array<Partial<Record>>} Array of records to apply as the client-side test-data
* @deprecated * In favor of DataSource.setCacheData.

*/
setTestData(data:Array<Partial<Record>>): void;

/**
*  Invalidate the cache when DataSource.cacheAllData or DataSource.clientOnly are true.
* 
*/
invalidateCache(): void;

/**
*  Switch into clientOnly mode, taking the cache from the cacheAllData ResultSet if it 
*  exists.
* 
*/
setClientOnly(): void;

/**
*  When DataSource.cacheAllData is true, has all the data been retrieved to the client?
* 
*/
hasAllData(): boolean;

/**
*  Takes all relative date values found anywhere within a Criteria / AdvancedCriteria object
*  and converts them to concrete date values, returning the new criteria object.
* 
* @param {Criteria} criteria to convert
* @param {string=} optional timezone offset. Defaults to the current timezone
* @param {Integer=} first day of the week (zero is Sunday). Defaults to                DateChooser.firstDayOfWeek
* @param {Date=} base value for relative conversion - defaults to now
*/
convertRelativeDates(criteria:Criteria, timezoneOffset?:string, firstDayOfWeek?:Integer, baseDate?:Date): Criteria;

/**
*  Notification method fired when a DataSource operation such as an 
*  DataSource.addData, DataSource.removeData or 
*  DataSource.updateData modifies the underlying data for a DataSource.
*  
*  This method is used by ResultSets to keep the user-visible data up to date as
*  changes are made.
*  
*  
*  Note: rather than overriding this method, we recommend using 
*  Class.observe to be notified when it is fired.
*  
*  
* 
* @param {DSResponse} response from the operation that modified the underlying  data set
* @param {DSRequest} request that initiated the data change
*/
dataChanged(dsResponse:DSResponse, dsRequest:DSRequest): void;

/**
*  Causes any components using this DataSource to be notified of changes that have been made to
*  the remote dataset accessed via this DataSource, as though the provided DSResponse had just
*  successfully completed. This will cause cache managers such as ResultSet or 
*  ResultTree to automatically update their caches, and components using such cache
*  managers to visually update to show modified data.
*  
*  This API should be used when you have found out about changes made by other users or by
*  automatic processes. For example, using the SmartClient Messaging system to receive
*  real-time updates via HTTP streaming, you may get updates that should affect a ListGrid
*  which is using a ResultSet to view a portion of a large dataset. 
*  
*  The provided DSResponse should have
*  DSResponse.operationType "update", "add" or "remove" - there is no
*  way for a "fetch" response to meaningfully update arbitrary caches. However, if you
*  have a list of updated records (possibly retrieved via DataSource.fetchData) you can still call
*  updateCaches()with DSResponses of type "update". Typically DataSource operations
*  that manipulate data operate on a single record at a time, but if you explicitly set the
*  response.data attribute to an array of records, framework code will handle this
*  as it would multiple updates.
*  
*  Example usage: if you had a ListGrid bound to the supplyItem sample DataSource, 
*  and that ListGrid was showing a Record with itemId 23, and you wanted to update 
*  the unitCost field to a new value, you would use the following code:
*  
*   
*   &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;// updatedRecord is the record we want to update
*   &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;var record = supplyItemDS.copyRecord(updatedRecord);
*   &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;record.unitCost = 500;
*   &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;var dsResponse = {
*   &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;data: [record],
*   &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;operationType: "update"
*   &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;};
*   &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;supplyItemDS.updateCaches(dsResponse);
*   
*  
*  
*  
*  To cause all components that have cache managers to drop their caches, provide a
*  DSResponse with DSResponse.invalidateCache set.
*  
*  As an alternative to calling updateCaches() directly, if updates to other
*  DataSources occur as a result of server-side logic, you can use the server-side API
*  DSResponse.addRelatedUpdate(DSResponse) (Pro Edition and above), which ultimately calls
*  updateCaches() for you - see that method's documentation for details.
*  
*  NOTE:: if updateCaches is called for a
*  DataSource.clientOnly DataSource, it will update
*  DataSource.cacheData synchronously in addition to notifying all cache managers as
*  normal.
*  
*  If a DataSource has DataSource.cacheAllData set and a full cache has been obtained, calling
*  updateCaches will automatically update the cache.
*  
*  Note that the DSResponse provided to this method will not go through
*  DataSource.transformResponse or other processing that would normally occur for a
*  DSResponse resulting from a DSRequest sent by the application in this page.
* 
* 
* @param {DSResponse} the provided DSResponse must minimally have                 dataSource, operationType, and data set
* @param {DSRequest=} optional dsRequest. If not specified, a DSRequest will be                automatically created based on the DataSource and operationType of                 the DSResponse
*/
updateCaches(dsResponse:DSResponse, dsRequest?:DSRequest): void;

/**
*  For a DataSource that describes a DOM structure, the list of legal child elements that can
*  be contained by the element described by this DataSource.
*  
*  For a DataSource described by XML schema, this is the list of legal subelements of
*  complexType (elements of simpleType become DataSourceFields with atomic type).
*  
*  Note that currently, if an XML schema file contains ordering constraints, DataSources
*  derived from XML Schema do not capture these constraints.
* 
* 
*/
getLegalChildTags(): void;

/**
*  Copies all DataSource field values of a Record (including a TreeNode) to a new 
*  Record, omitting component-specific metadata such as selected state from grids, 
*  or parent folders for TreeNodes.
* 
* @param {Record} The record to be copied.
*/
copyRecord(record:Record): Record;

/**
*  Copies all DataSource field values of an (Array) of Records (including a TreeNode) 
*  to a new array of Records, omitting component-specific metadata such as selected state 
*  from grids, or parent folders for TreeNodes. This method calls 
*  DataSource.copyRecord for each item in the array.
* 
* @param {Array<Partial<Record>>} The array of Record objects to be copied.
*/
copyRecords(records:Array<Partial<Record>>): Array<Partial<Record>>;

/**
*  Creates a shallow copy of the given DSRequest. The request's DSRequest.data,
*  if any, is shallow copied in the cloned request.
*  
*  The DSRequest.callback property of the given request is not copied into
*  the cloned request.
* 
* @param {DSRequest} the DSRequest to clone.
*/
cloneDSRequest(dsRequest:DSRequest): DSRequest;

/**
*  Creates a shallow copy of the given DSResponse. All properties that would affect
*  the processing of the response are copied into the resulting DSResponse so that the cloned
*  response could substitute for the original response. The response's DSResponse.data,
*  if any, is shallow copied in the cloned response.
* 
* @param {DSResponse} the DSResponse to clone.
*/
cloneDSResponse(dsResponse:DSResponse): DSResponse;

/**
*  For a dataSource using clientDataIntegration,
*  return the data that should be sent to the DataSource.dataURL.
*  
*  By default, HTTP requests sent to non-SmartClient servers do not include DSRequest
*  metadata such as DSRequest.startRow, DSRequest.endRow,
*  and DSRequest.oldValues. Only the core
*  dataSourceOperations is sent, such as the criteria
*  passed to ListGrid.fetchData or the updated values submitted by
*  DynamicForm.saveData. 
*  
*  transformRequest() allows you to transform dsRequest metadata into a
*  format understood by your server and include it in the HTTP request, so that you can
*  integrate DataSource features such as data paging with servers that support such
*  features.
*  
*  How the data is actually sent to the URL is controlled by 
*  OperationBinding.dataProtocol. If using the "getParams" or
*  "postParams" protocol, data is expected to be a JavaScript Object where each property
*  will become a GET or POST'd parameter. If using dataProtocol:"soap" or "postXML", data
*  will be serialized as an XML message by DataSource.xmlSerialize.
*  
*  As an example, if you have a dataURL that can return paged data given URL parameters
*  "start" and "end", you could implement transformRequest like so:
*  
*  
*   isc.DataSource.create({
*    ... 
*    transformRequest : function (dsRequest) {
*      if (dsRequest.operationType == "fetch") {
*        var params = {
*         start : dsRequest.startRow,
*         end : dsRequest.endRow
*        };
*        // combine paging parameters with criteria
*        return isc.addProperties({}, dsRequest.data, params);
*      }
*    }
*   });
*  
*  
*  
*  Other reasons to implement transformRequest():
*  
*  
* - transform a Criteria object into the custom query language of a web
*    service
*  
* - add a session id to requests that require authentication
*  
* - detect colliding updates by sending both updated values and the values the user
*    originally retrieved before editing began (available as DSRequest.oldValues)
*  
*  Special case: If the dataProtocol is "clientCustom"
*  the SmartClient system will not attempt to send data to the server in any way. Instead 
*  transformRequest should be implemented such that it accesses or updates the underlying
*  data-set and calls DataSource.processResponse when the operation is complete. This
*  setting allows straightforward integration with non SmartClient comm mechanisms that 
*  directly send requests to the server (such as GWT-RPC), or handle data manipulation without
*  sending HTTP at all (such as Google Gears).
*  A transformRequest override may also be used to set the DSRequest.dataProtocol
*  to clientCustom at runtime, giving developers a way to intercept normal handling for
*  some particular request, and provide entirely custom handling written on the client.
*  
*  Note: The RestDataSource class overrides transformRequest() to handle xml-serializing
*  the request (including meta data) into a standard format.
* 
* 
* @param {DSRequest} the DSRequest being processed
*/
transformRequest(dsRequest:DSRequest): any;

/**
*  Helper method to retrieve the updated data from a successful dataSource update or add 
*  operation.
* 
* @param {DSRequest} Update request object passed to the server. Note that this request may have been modified by DataSource.transformRequest
* @param {DSResponse} Response returned by the server
* @param {boolean} If DSResponse.data is empty, should data be derived from the submitted request.
*/
getUpdatedData(dsRequest:DSRequest, dsResponse:DSResponse, useDataFromRequest:boolean): any | Array<Partial<any>>;

/**
*  Returns the appropriate OperationBinding.dataProtocol for a DSRequest
* 
* @param {DSRequest} DataSource Request object
*/
getDataProtocol(dsRequest:DSRequest): DSProtocol;

/**
*  Process a dsResponse for a request initiated by a DataSource with
*  OperationBinding.dataProtocol.
*  requestId parameter should be dsRequest.requestId as found on the dsRequest
*  passed to DataSource.transformRequest.
*  
*  You must provide a response for both error and non-error cases. For an error case, a
*  sufficient response is:
*  
*  { status : -1 }
*  
* 
* @param {string} requestId attribute from the associated dataSource request object
* @param {Partial<DSResponse>} Configuration for the dsResponse
*/
processResponse(requestId:string, dsResponse:Partial<DSResponse>): void;

/**
*  Get the XML to be posted to the dataURL based on the passed DSRequest.
*  
*  This API is intended to be overridden in order to integrate with web services that
*  expect XML messages rather than simple HTTP parameters, but lack a
*  WSDL description. For WSDL-described web services, having loaded the service
*  description, SmartClient knows the correct XML message structure, so customization is
*  best done by modifying the JavaScript data that is used to form the message.
* 
* 
* @param {DSRequest} the request to encode as an XML message.
*/
getXMLRequestBody(dsRequest:DSRequest): string;

/**
*  Serialize a JavaScript object as XML.
*  
*  The JavaScript Object passed to DataSource.xmlSerialize becomes an XML
*  element named after the DataSource.tagName (or DataSource.ID if
*  tagName is unset). Each property of the object becomes a subElement. For example,
*  using a DataSource to serialize like this:
*  
*    var inputObject = {
*     startRow : 5,
*     endRow : 50,
*     data : [
*       { field1 : "value1", field2: new Date() },
*       { field1 : "value3", field2: null }
*     ]
*    };
*    var myDS = isc.DataSource.create({ tagName:"DSRequest" });
*    myDS.xmlSerialize(inputObject);
*  
*  .. produces the following XML:
*  
*    &lt;DSRequest&gt;
*      &lt;startRow&gt;5&lt;/startRow&gt;
*      &lt;endRow&gt;50&lt;/endRow&gt;
*      &lt;data&gt;
*        &lt;field1&gt;value1&lt;/field1&gt;
*        &lt;field2&gt;2005-10-14T18:01:16&lt;/field2&gt;
*      &lt;/data&gt;
*      &lt;data&gt;
*        &lt;field1&gt;value3&lt;/field1&gt;
*        &lt;field2&gt;&lt;/field2&gt;
*      &lt;/data&gt;
*    &lt;/DSRequest&gt;
*  
*  
*  Various properties on the DataSource and DataSourceField can affect how serialization is
*  performed, see for example DataSource.tagName,
*  DataSource.schemaNamespace, 
*  DataSourceField.xmlAttribute, 
*  DataSourceField.multiple and DataSourceField.childTagName.
*  By setting the DataSourceField.type to the ID of another
*  DataSource which has further XML serialization settings, you can control serialization of
*  nested structures.
*  
*  If you are working with a WSDL-described web service, XML serialization is performed
*  automatically by APIs like WebService.callOperation - you only need to
*  know about serialization in order to understand how to put together JavaScript data that
*  will fill in an XML message properly, and for simple messages, setting
*  DSRequest.useFlatFields makes that unnecessary as well.  
*  
*  Note: when trying to send data to a web service, it is best to avoid putting
*  together any XML yourself, instead modify the JavaScript data being fed to SmartClient's
*  SOAP engine. This is because the WSDL and SOAP rules for correctly namespacing and
*  encoding Web Service messages are very complex and are subject to change with new
*  versions of the web service you are contacting, whereas the data itself is easy to
*  manipulate and less likely to change.
*  
*  To troubleshoot message formation, you can set the log category "xmlSerialize" to 
*  INFO or DEBUG level in order to see diagnostics about XML
*  message formation, and you can use the RPC tab in the Developer Console to see the
*  actual messages being passed to web services.
* 
* 
* @param {any} data to be serialized
* @param {SerializationContext} options for the serialization engine
*/
xmlSerialize(data:any, flags:SerializationContext): string;

/**
*  Transform a list of XML elements to DataSource records.
*  
*  recordsFromXML() will create a List of DataSource records in the form of
*  JavaScript objects. The value for each field is extracted from the XML according to the
*  rules described under DataSourceField.valueXPath. 
*  
*  Derived JavaScript values for each field will be the appropriate JavaScript type, eg,
*  for a field of "date" type, the JS value will be a JavaScript Date object.
*  
*  Note that if the set of nodes exceeds DataSource.resultBatchSize, this method
*  will break the XML processing logic into multiple threads and therefore complete
*  asynchronously. In this case the return value will be null, and the callback parameter
*  must be used to process the extracted records.
*  
* 
* @param {Array<Partial<XMLElement>>} XML elements to transform, eg, the result of a                    call to XMLTools.selectNodes
* @param {Callback} Callback to fire when the transform completes. Takes a single  parameter records - the array of DataSource records derived from the   XML elements.
*/
recordsFromXML(elements:Array<Partial<XMLElement>>, callback:Callback): List;

/**
*  Modify the DSResponse object derived from the response returned from the 
*  DataSource.dataURL.
*  
*  This is an override point that makes it possible to use DataSource features such as
*  paging with web services that support such features, by allowing you to fill in metadata
*  fields in the DSResponse object (such as DSResponse.startRow) based on
*  service-specific metadata fields contained in the service's response.
*  
*  The DSResponse passed to this method already has DSResponse.data, which is
*  derived differently depending on the DataSource.dataFormat setting:
*  
*  
* - dataFormat:"xml" : either the
*  OperationBinding.recordXPath or 
*  OperationBinding.recordName is used to select the XML elements
*  that represent DataSource records. The selected XML elements are passed to
*  DataSource.recordsFromXML, which transforms the XML elements to typed
*  JavaScript data using the DataSource as a schema.
*  
* - dataFormat:"json" : the
*  OperationBinding.recordXPath, if specified, is used to select
*  records from the returned JSON data via XMLTools.selectObjects. 
*  DataSourceField.valueXPath is used to derive correctly typed field values.
*  
* - dataFormat:"custom" : dsResponse.data is the raw response
*  in String form. It must be parsed into an Array of Objects for subsequent processing to
*  work.
*  
*  
*  In addition to dsResponse.data, DSResponse.status is defaulted
*  to 0 (indicating no error), and DSResponse.startRow is assumed to be zero,
*  with DSResponse.endRow and DSResponse.totalRows
*  both set to dsResponse.data.length - 1, that is, the returned data is
*  assumed to be all records that matched the filter criteria.
*  
*  Examples of using this API include:
*  
*  
* - setting DSResponse.startRow, 
*    DSResponse.endRow and DSResponse.totalRows
*    to allow paging for a service that supports it. For example, if an XML service
*    returns a "resultRow" tag that contained the row number of the first row of the
*    returned results:
*   
*    dsResponse.startRow = isc.XMLTools.selectNumber(xmlData, "//resultRow");
*   
*   
*  
* - setting DSResponse.status to recognized ISC error values based on
*    service-specific errors, in order to trigger standard ISC error handling. For
*    example, setting dsResponse.status to
*    RPCResponse.STATUS_VALIDATION_ERROR and filling in
*    DSResponse.errors in order to cause validation errors to be shown in
*    forms and grids.
*  
* - for services that either do not return cache update data, or return partial data,
*    using DSRequest.oldValues to create cache update data (whether this is
*    appropriate is application-specific), or setting
*    DSResponse.invalidateCache.
*  
*  
*  NOTE: this method is NOT an appropriate time to call
*  methods on visual components such as grids, initiate new DSRequests or RPCRequests, or
*  in general do anything other than fill in fields on the DSResponse based on data that is
*  already available. Any actions that need to be taken as a result of the web
*  service response should be implemented exactly as for a DataSource where
*  transformResponse() has not been overridden, that is, use the callback
*  passed to high-level methods such as 
*  dataBoundComponentMethods, and do error
*  handling via either DataSource.handleError or by setting
*  RPCRequest.willHandleError.
* 
* 
* @param {DSResponse} default DSResponse derived from the response data
* @param {DSRequest} DSRequest object that initiated this request
* @param {XMLDocument | JSON} XML document or JSON objects returned by the web                  service
*/
transformResponse(dsResponse:DSResponse, dsRequest:DSRequest, data:XMLDocument | JSON): DSResponse;

/**
*  Fetch a single record from the DataSource by DataSourceField.primaryKey.
*  This simply calls DataSource.fetchData after creating Criteria that contain the primary
*  key field and value.
*  
*  If you call this method on a DataSource with a composite primary key - ie, one with 
*  multiple primaryKey fields - this method returns the first record where the first 
*  defined primary field matches the supplied pkValue; this may or may not be meaningful, 
*  depending on your use case. Generally, for DataSources with composite keys, it makes
*  more sense to use fetchData() directly, rather than this convenience 
*  method.
* 
* 
* @param {any} value for the field marked           DataSourceField.primaryKey:true in this DataSource            (or the first field so marked if there is more than one)
* @param {DSCallback=} callback to invoke on completion
* @param {Partial<DSRequest>=} additional properties to set on the                          DSRequest that will be issued
*/
fetchRecord(pkValue:any, callback?:DSCallback, requestProperties?:Partial<DSRequest>): void;

/**
*  Perform a "fetch" DataSource operation against this DataSource, sending search criteria
*  and retrieving matching records.
*  
*  NOTE: do not attempt to override this method to create a custom DataSource. For
*  a server-side custom DataSource, use the DataSource.serverConstructor attribute,
*  and the ${isc.DocUtils.linkForExampleId('customDataSource', 'Custom DataSource samples')}. For a
*  client-side custom DataSource, see DataSource.dataProtocol.
*  
*  In contrast to ListGrid.fetchData, which creates a ResultSet to manage
*  the returned data, calling dataSource.fetchData() provides the returned
*  data in the callback as a 
*  simple JavaScript Array of JavaScript Objects.
*   Calling
*  dataSource.fetchData() does not automatically update any visual components or
*  caches: code in the callback passed to fetchData() decides what to do with
*  the returned data. 
*  
*  For example, given a ListGrid "myGrid" and a DataSource "employees", the following code
*  would populate "myGrid" with data fetched from the DataSource:
*  
*   isc.DataSource.get("employees").fetchData(null, "myGrid.setData(data)");
*   
*  
*  Unlike calling myGrid.fetchData(), which creates a ResultSet, the
*  data provided to the grid is "disconnected" data, unmanaged by SmartClient's databinding
*  facilities and safe to directly modify. This is useful when, for example, a ListGrid is
*  being used as a more sophisticated version of HTML's multi-select component.
*  
*  Disconnected datasets may be used to populate various visual components. For example,
*  while an individual FormItem can be configured to fetch
*  FormItem.valueMap options from a DataSource via the
*  FormItem.optionDataSource property, the following code shows
*  storing a dataset to derive valueMaps from later:
*  
*   
*   isc.DataSource.get("countries").fetchData(null, "window.countries = data");
* 
*   ... later, a form is created dynamically ...
* 
*   function showForm() {
*     isc.DynamicForm.create({
*       items : [
*        { name:"country", title:"Pick Country",
*         valueMap: window.countries.getValueMap("countryId", "countryName")
*        },
*     ...
*   
*   
*  
*  
*  You can also create a ResultSet from the data retrieved from fetchData(),
*  like so:
*  
*   
*   isc.DataSource.get("countries").fetchData(null,
*     function (dsResponse, data) {
*       isc.ResultSet.create({
*         dataSource:"countries",
*         allRows:data
*       })
*     }
*   )
*   
*   
*  
*  
*  This gives you a dataset that supports client-side filtering (via
*  ResultSet.setCriteria), can provide 
*  ResultSet.getValueMap, will 
*  ResultSet.disableCacheSync to the DataSource made via
*  other components, and can be re-used with multiple visual components.
*  
*  See also DataSource.getClientOnlyDataSource and DataSource.cacheAllData for
*  similar capabilities for dealing with smaller datasets entirely within the browser, or working
*  with modifiable caches representing subsets of the data available from a DataSource.
*  
*  See also the server-side com.isomorphic.js.JSTranslater class in the
*  ${isc.DocUtils.linkForDocNode('JavaServerReference', 'Java Server Reference')}
*  for other, similar approaches
*  involving dumping data into the page during initial page load. Note: care should
*  be taken when using this approach. Large datasets degrade the basic performance of some
*  browsers, so use PickList.optionDataSource and similar
*  facilities to manage datasets that may become very large.
*  
*  Data-Driven Visual Component Creation
*  
*  
*  DataSource.fetchData() can also be used to create SmartClient components in
*  a data-driven way. Many properties on SmartClient visual components are configured via
*  an Array of Objects - the same data format that dataSource.fetchData()
*  returns. These include ListGrid.fields, TabSet.tabs,
*  DynamicForm.items, Facet.values and even DataSource.fields.
*  
*  For example, if you had a DataSource "myFormFields" whose fields included the
*  basic properties of FormItem (name, title, type, etc), this example
*  code would create a form based on stored field definitions, loaded from the
*  "myFormFields" DataSource on the fly:
*  
*   isc.DataSource.get("myFormFields").fetchData(null, 
*     "isc.DynamicForm.create({ items:data })"
*   )
*  
*  
*  
*  This capability to dynamically create visual components from dynamically fetched data
*  provides a foundation for creating interfaces that can be customized by end users.
*  See also the server-side API com.isomorphic.datasource.DataSource.addDynamicDSGenerator() for
*  dynamically creating DataSources supporting all server-side DataSource features, and
*  DataSource.inheritsFrom for sharing field definitions across multiple DataSources.
* 
* 
* @param {Criteria=} search criteria
* @param {DSCallback=} callback to invoke on completion
* @param {Partial<DSRequest>=} additional properties to set on                            the DSRequest that will be issued
*/
fetchData(criteria?:Criteria, callback?:DSCallback, requestProperties?:Partial<DSRequest>): void;

/**
*  Perform a "fetch" DataSource operation against this DataSource, sending search criteria
*  and retrieving matching records.
*  
*  This is identical to DataSource.fetchData except that
*  DSRequest.textMatchStyle is set to "substring" to cause case insensitive
*  substring matching (if the server respects this setting).
* 
* 
* @param {Criteria=} search criteria
* @param {DSCallback=} callback to invoke on completion
* @param {Partial<DSRequest>=} additional properties to set on                            the DSRequest that will be issued
*/
filterData(criteria?:Criteria, callback?:DSCallback, requestProperties?:Partial<DSRequest>): void;

/**
*  Exports arbitrary client-side data, with client-side formatters applied, so is suitable 
*  for direct display to users. This method can be used to export data formatted outside
*  of any kind of visual component.
*  
*  If you do not specify an OperationBinding.operationId in the 
*  requestProperties you pass to this method, it behaves exactly the same as
*  the DataSource.exportClientData of the same name.
*  If you do specify an operationId, the framework expects your DataSource to
*  configure an OperationBinding of OperationBinding.operationType
*  "clientExport" with the same operationId. The framework will then send the
*  exportClientData request via the ordinary DSRequest mechanism,
*  which allows you to use normal framework features in the client data export. 
*  
*  For example, you could add a dmiOverview to your
*  operationBinding, which would allow you to write server-side code that
*  intervenes in the export process - for instance, by calling the
*  getExportObject() API to do something special with the export document,
*  like saving it to a database table or sending it to an email list.
*  
*  When you use the specific operationId version of this API, both the 
*  SmartClient Server and server-side DataSources are required.
*  
*  To export unformatted data, see DataSource.exportData which does
*  not include client-side formatters, but requires both the SmartClient server and the 
*  presence of server-side DataSources.
*  
*  Note that field DataSourceField.displayFormat is honored for 
*  "date" and "datetime" fields when exporting direct to Excel; see the displayFormat
*  docs for details.
* 
* 
* @param {Array<Partial<Record>>} Array of Records to export
* @param {Partial<DSRequest>=} Request properties for the export
* @param {DSCallback=} Optional callback. Note that this is only applicable if you also specify DSRequest.exportToClient: false in the  request properties
*/
exportClientData(data:Array<Partial<Record>>, requestProperties?:Partial<DSRequest>, callback?:DSCallback): void;

/**
*  Perform a "fetch" DataSource operation against this DataSource, sending search criteria,
*  retrieving matching records and exporting the results. See 
*  OperationBinding.exportResults or DSRequest.exportResults and for more
*  information.
* 
* 
* @param {Criteria=} search criteria
* @param {Partial<DSRequest>=} additional properties to set on                            the DSRequest that will be issued
* @param {DSCallback=} callback to invoke on completion. Note that this parameter                 only applies where DSRequest.exportToClient is                  explicitly set to false, because file downloads do not                  provide ordinary SmartClient callbacks
*/
exportData(criteria?:Criteria, requestProperties?:Partial<DSRequest>, callback?:DSCallback): void;

/**
*  Produces a clientOnly "copy" of a particular subset of data from a normal 
*  DataSource, via calling fetchData() to fetch matching rows, and constructing 
*  a clientOnly DataSource that DataSource.inheritsFrom the original DataSource.
*  
*  This clientOnly "copy" can be useful in situations where you want to allow 
*  a series of local changes without immediately committing to the server. 
*  See also ListGrid.autoSaveEdits for more fine-grained tracking of 
*  edits (eg, special styling for uncommitted changes).
*  
*  The new DataSource is returned via the "callback" argument. If DataSource.cacheAllData is
*  enabled and DataSource.hasAllData returns true, the new DataSource is synchronously
*  returned as the result of the method. In this case, if a callback was passed, it also
*  is executed synchronously.
* 
* 
* @param {Criteria} The criteria for the clientOnly DS
* @param {ClientOnlyDataSourceCallback} The callback to fire passing the clientOnly DS
* @param {Partial<DSRequest>=} optional properties to pass through to the DSRequest
* @param {Partial<DataSource>=} optional properties to pass                            through to the clientOnly DS
*/
getClientOnlyDataSource(criteria:Criteria, callback:(clientOnly:DataSource) => void, requestProperties?:Partial<DSRequest>, dataSourceProperties?:Partial<DataSource>): void;

/**
*  Perform an "add" DataSource operation against this DataSource, to create a new DataSource
*  record.
*  
*  NOTE: do not use repeated calls to addData() to provide the initial
*  dataset for a DataSource.clientOnly DataSource, instead, provide
*  initial data via DataSource.cacheData. Using addData() for subsequent,
*  incremental updates from sources like user editing is fine.
* 
* 
* @param {Record} new record
* @param {DSCallback=} callback to invoke on completion
* @param {Partial<DSRequest>=} additional properties to set on                            the DSRequest that will be issued
*/
addData(newRecord:Record, callback?:DSCallback, requestProperties?:Partial<DSRequest>): void;

/**
*  Perform an "update" DataSource operation against this DataSource, to update values in an
*  existing DataSource record.
* 
* 
* @param {Partial<Record>} updated record
* @param {DSCallback=} callback to invoke on completion
* @param {Partial<DSRequest>=} additional properties to set on                            the DSRequest that will be issued
*/
updateData(updatedRecord:Partial<Record>, callback?:DSCallback, requestProperties?:Partial<DSRequest>): void;

/**
*  Perform a "remove" DataSource operation against this DataSource, to delete an existing
*  DataSource record.
* 
* 
* @param {Record | any} primary key values of record to delete, (or complete record)
* @param {DSCallback=} callback to invoke on completion
* @param {Partial<DSRequest>=} additional properties to set on                            the DSRequest that will be issued
*/
removeData(data:Record | any, callback?:DSCallback, requestProperties?:Partial<DSRequest>): void;

/**
*  Contacts the server to run server-side validation on a DSRequest and either returns
*  DSResponse.errors validation errors or a DSResponse.status code of 0.
*  
*  A "validate" dsRequest is effectively always RPCRequest.willHandleError:true.
*  It is a normal condition for a "validate" DSResponse to have validation errors and
*  the response will never go to system-wide handling for unexpected errors
*  (RPCManager.handleError).
* 
* 
* @param {Record} record values to validate
* @param {DSCallback=} callback to invoke on completion
* @param {Partial<DSRequest>=} additional properties to set on                          the DSRequest that will be issued
*/
validateData(values:Record, callback?:DSCallback, requestProperties?:Partial<DSRequest>): void;

/**
*  Invoke an operation declared with OperationBinding.operationType "custom".
*  
*  This is a rarely used API. If the operation you are performing can be thought of
*  as one of the standard "CRUD" DSOperationType, declare it
*  with a CRUD operationType. For example, if your operation updates a record, declare it
*  with operationType "update" and invoke it via DataSource.updateData - this will cause
*  ResultSet to work correctly.
*  
*  In particular:
*  
*  
* - do not use this API just because you need to add additional server-side logic to a
*  CRUD operation (DMI allows this)
*  
* - do not use this API to implement variants of core CRUD operations
*  (DSRequest.operationId is the correct way to do this)
*  
* - do not use this API just because an operation affects more than one record. Most
*  kinds of multi-record operations should use RPCManager.startQueue. 
*  However, a custom operation is appropriate for genuine "batch" updates, as 
*  opposed to just a number of ordinary updates by primaryKey - see 
*  OperationBinding.allowMultiUpdate
*  
* - do not use this API just because you are calling a stored procedure in SQL - if the
*  stored procedure performs some kind of CRUD operation on the records of this DataSource,
*  use a standard CRUD operationType
*  
*  
*  Instead, the specific purpose of this API is to bypass all checks and side effects that
*  normally occur for CRUD operations, for example, that a "fetch" requires valid Criteria
*  or that an "update" or "remove" operation contains a valid primary key, or that an "add"
*  operation returns the newly added record. performCustomOperation allows
*  you to pass an arbitrary Record to the server, act on it with custom code, and return
*  arbitray results or even no results.
*  
*  The "data" parameter becomes DSRequest.data. With the SmartClient
*  Server Framework, the data is accessible server-side via DSRequest.getValues() and in
*  velocitySupport (such as &lt;customSQL&gt;) as $values.
*  
*  Note that with SQLDataSource, performCustomOperation must be used if you
*  plan to have a &lt;customSQL&gt; tag in your operationBinding that will execute SQL
*  operations other than SELECT, UPDATE, INSERT, DELETE (such as creating a new table).
*  By declaring OperationBinding.operationType "custom" in your .ds.xml file, all
*  checks related to normal CRUD operations will be skipped and your &lt;customSQL&gt;
*  can do arbitrary things.
* 
* 
* @param {string} the operation ID
* @param {Partial<Record>=} data to pass to the server.
* @param {DSCallback=} callback to invoke on completion
* @param {Partial<DSRequest>=} additional properties to set on                            the DSRequest that will be issued
*/
performCustomOperation(operationId:string, data?:Partial<Record>, callback?:DSCallback, requestProperties?:Partial<DSRequest>): void;

/**
*  Download a file stored in a field of type:"binary" in a DataSource record.
*  
*  This will trigger the browser's "Save As" dialog and allow the user to save the file
*  associated with some record.
*  
*  Note that if this method is called for a record with no associated file, the
*  download URL may not be functional. By default when dataSources encounter a 
*  FieldType, an additional field,
*  &lt;fieldName&gt;_filename, is generated to
*  store the filename for the binary field value. If this field is present in the
*  data source but has no value for this record, developers can assume they're working
*  with a record with no stored file. If this field is not present in some custom
*  dataSource configuration, or the record is not loaded on the client, an additional
*  server transaction may be required to determine whether the record has an associated
*  file before calling this method to download a file.
*  
*  See the overview of binaryFields for more details.
* 
* 
* @param {Record} Record to download. Only required to have a value for the            primary key field.
* @param {FieldName=} Optional name of the binary field containing the file.                  If not provided, the first binary field is used.
* @param {Partial<DSRequest>=} Additional properties to set on the                           DSRequest that will be issued.
*/
downloadFile(data:Record, fieldName?:FieldName, requestProperties?:Partial<DSRequest>): void;

/**
*  Display a file stored in a field of type:"binary" in a new browser window.
*  
*  This will open a new browser window to show the file. Depending on the file type,
*  the user's installed plugins and applications, and the user's browser settings, this may
*  cause the file to be actually displayed in the new browser window, or may prompt the
*  user to either launch an external application to view the file or save the file to disk.
*  
*  Note that if this method is called for a record with no associated file, the
*  target window's new URL may not be functional. By default when dataSources encounter a 
*  FieldType, an additional field,
*  &lt;fieldName&gt;_filename, is generated to
*  store the filename for the binary field value. If this field is present in the
*  data source but has no value for this record, developers can assume they're working
*  with a record with no stored file. If this field is not present in some custom
*  dataSource configuration, or the record is not loaded on the client, an additional
*  server transaction may be required to determine whether the record has an associated
*  file before calling this method to view a file.
*  
*  See the overview of binaryFields for details.
* 
* 
* @param {Record} Record to download. Only required to have a value for the            primary key field.
* @param {FieldName=} Optional name of the binary field containing the file.                  If not provided, the first binary field is used.
* @param {Partial<DSRequest>=} Additional properties to set on the                           DSRequest that will be issued.
*/
viewFile(data:Record, fieldName?:FieldName, requestProperties?:Partial<DSRequest>): void;

/**
*  Returns a direct URL to access a file stored in a field of type:"binary".
*  
*  This URL can be used as the "src" attribute of an Img widget or &lt;img&gt; tag (if the file is
*  an image), or can be used in an ordinary HTML link (&lt;a&gt; tag) to download the file.
*  However, for the latter use case, see also DataSource.downloadFile and DataSource.viewFile.
*  
*  The URL returned is not to a static file on disk, rather, the returned URL essentially
*  encodes a DSRequest as URL parameters, in a format understood by the IDACall servlet
*  that comes with the Server Framework. 
*  
*  Hence, this URL will dynamically retrieve whatever file is currently stored in the
*  binary field via executing a normal DSRequest server side. The request will run through
*  normal security checks, so if your application requires authentication, the user must
*  have a valid session and be authorized to access the binary field.
*  
*  Note that if this method is called for a record with no associated file, the
*  returned URL may not be functional. By default when dataSources encounter a 
*  FieldType, an additional field,
*  &lt;fieldName&gt;_filename, is generated to
*  store the filename for the binary field value. If this field is present in the
*  data source but has no value for this record, developers can assume they're working
*  with a record with no stored file. If this field is not present in some custom
*  dataSource configuration, or the record is not loaded on the client, an additional
*  server transaction may be required to determine whether the record has an associated
*  file before calling this method to retrieve a download URL.
* 
* 
* @param {Record | any} Record containing at least the primary key field,                  or just the value of the primary key field.
* @param {FieldName=} Optional name of the binary field containing the file. If                  not provided, the first binary field is used.
* @param {Partial<DSRequest>=} Additional properties to set on the                           DSRequest that will be issued.
*/
getFileURL(data:Record | any, fieldName?:FieldName, requestProperties?:Partial<DSRequest>): string;

/**
*  Returns a URL to DataSource fetch operation. This API is intended to return
*  media such as images or videos to the browser. 
*  
*  Note that because the entirety of the request is encoded in the URL, there is an
*  inherent limitation on the amount of data that you can send viat he criteria argument to
*  the server. The actual length depends on your server configuration and other factors
*  such as the size of cookies (if any) being sent to the server and other HTTP headers in
*  use. Conservatively, assume that you have about 2 kilobytes to work with.
* 
* 
* @param {Criteria} Criteria to be sent to server.
* @param {Partial<DSRequest>=} additional properties to set on                            the DSRequest that will be issued
*/
getFetchDataURL(criteria:Criteria, requestProperties?:Partial<DSRequest>): string;

/**
* 
*  Gets the contents of a file stored in this DataSource.
* 
* 
* @param {FileSpec | string} Either a FileSpec, or a String which         will be parsed to determine the fileName, fileType and fileFormat.         For instance, "employees.ds.xml" would be parsed as         {fileName: "employees", fileType: "ds", fileFormat: "xml"}.         If fileType and fileFormat are not provided, will return         the first file with the specified fileName.
* @param {GetFileCallback} Callbacks.GetFileCallback executed with the results. The         data parameter is either a String with the         contents of the file, or null to indicate error (such as         file not found).          You can examine DSResponse.status         and DSResponse.data         for additional information about any error.
*/
getFile(fileSpec:FileSpec | string, callback:(dsResponse:DSResponse, data:string, dsRequest:DSRequest) => void): void;

/**
* 
*  Indicates whether a file exists in this DataSource. 
* 
* 
* @param {FileSpec | string} Either a FileSpec, or a String which         will be parsed to determine the fileName, fileType and fileFormat.         For instance, "employees.ds.xml" would be parsed as         {fileName: "employees", fileType: "ds", fileFormat: "xml"}.         If fileType or fileFormat are not provided, will indicate whether         any file with the provided fileName exists.
* @param {HasFileCallback} Callbacks.HasFileCallback executed with the results.         The data parameter is a boolean indicating         whether the file is present.         You can examine DSResponse.status         and DSResponse.data         for additional information about any error.
*/
hasFile(fileSpec:FileSpec | string, callback:(dsResponse:DSResponse, data:boolean, dsRequest:DSRequest) => void): void;

/**
* 
*  Get a list of files from the DataSource. Note, if 
*  DataSource.fileVersionField is switched on for the dataSource,
*  the resulting list contains only the most recent version of each file.
* 
* 
* @param {Criteria} Criteria to apply. References to fileName,         fileType and fileFormat fields will         be translated to the native field names configured for         this DataSource.
* @param {DSCallback} Callback executed with the results.         The data parameter is either an array of records,         or null to indicate an error.          The records will have the DataSource.fileNameField,         DataSource.fileTypeField,         DataSource.fileFormatField,         DataSource.fileLastModifiedField, and          DataSource.fileVersionField fields populated, but          not the DataSource.fileContentsField field. (You          can use DataSource.getFile to get the fileContents).         You can examine DSResponse.status         and DSResponse.data         for additional information about any error.
*/
listFiles(criteria:Criteria, callback:DSCallback): void;

/**
* 
*  Save a file to the DataSource.
* 
* 
* @param {FileSpec | string} Either a FileSpec, or a String which         will be parsed to determine the fileName, fileType and fileFormat.         For instance, "employees.ds.xml" would be parsed as         {fileName: "employees", fileType: "ds", fileFormat: "xml"}.         Depending on the configuration of the DataSource, the fileType         and fileFormat may be optional.
* @param {string} The contents of the file
* @param {DSCallback=} Callback executed with the results.         The data parameter is either a record represening         the new file, or null to indicate an error. The record will have its         fileName, fileType and fileFormat         field populated, but not the fileContents         field.         You can examine DSResponse.status         and DSResponse.data         for additional information about any error.
*/
saveFile(fileSpec:FileSpec | string, contents:string, callback?:DSCallback): void;

/**
* 
*  Rename a file stored in this DataSource. Note, if 
*  DataSource.fileVersionField is switched on for the dataSource,
*  all versions of the file are renamed.
* 
* 
* @param {FileSpec | string} Either a FileSpec, or a String which         will be parsed to determine the fileName, fileType and fileFormat         of the file to rename.         For instance, "employees.ds.xml" would be parsed as         {fileName: "employees", fileType: "ds", fileFormat: "xml"}.         Depending on the configuration of the DataSource, the fileType and fileFormat         may be optional.
* @param {FileSpec | string} Either a FileSpec, or a String which         will be parsed to determine the fileName, fileType and fileFormat         to rename the file to.         For instance, "employees.ds.xml" would be parsed as         {fileName: "employees", fileType: "ds", fileFormat: "xml"}.         If the fileType or fileFormat are not provided, then they will not         be changed.
* @param {DSCallback=} Callback executed with the results.         The data parameter is either an array of         records represening the renamed file(s), or null to         indicate an error.         The records will have their fileName fields and         fileType fields populated, but not the         fileContents field.         You can examine DSResponse.status         and DSResponse.data         for additional information about any error.
*/
renameFile(oldFileSpec:FileSpec | string, newFileSpec:FileSpec | string, callback?:DSCallback): void;

/**
* 
*  Remove a file stored in this DataSource. Note, if 
*  DataSource.fileVersionField is switched on for the dataSource,
*  all versions of the file are removed (to remove an individual file version, use the 
*  DataSource.removeFileVersion API).
* 
* 
* @param {FileSpec | string} Either a FileSpec, or a String which         will be parsed to determine the fileName, fileType and fileFormat.         For instance, "employees.ds.xml" would be parsed as         {fileName: "employees", fileType: "ds", fileFormat: "xml"}.         Depending the configuration of the DataSource, the fileType         and fileFormat may be optional.
* @param {DSCallback=} Callback executed with the results.         The data parameter is either an array of         records represening the removed file(s), or null to         indicate an error.         The records will have their fileName fields and         fileType fields populated, but not the         fileContents field.         You can examine DSResponse.status         and DSResponse.data         for additional information about any error.
*/
removeFile(fileSpec:FileSpec | string, callback?:DSCallback): void;

/**
* 
*  Get the list of a given file's versions from the dataSource, sorted in version order 
*  (most recent version first). If the dataSource does not specify a 
*  DataSource.fileVersionField, this API will return an error
* 
* 
* @param {FileSpec | string} Either a FileSpec, or a String which         will be parsed to determine the fileName, fileType and fileFormat.         For instance, "employees.ds.xml" would be parsed as         {fileName: "employees", fileType: "ds", fileFormat: "xml"}.         If fileType and fileFormat are not provided, will return         the first file with the specified fileName.
* @param {DSCallback} Callback executed with the results.         The data parameter is either an array of records,         or null to indicate an error.          The records will have the DataSource.fileNameField,         DataSource.fileTypeField,         DataSource.fileFormatField,         DataSource.fileLastModifiedField and          DataSource.fileVersionField fields populated, but          not the DataSource.fileContentsField field. (You         can use DataSource.getFileVersion to get the          fileContents).         You can examine DSResponse.status         and DSResponse.data         for additional information about any error.
*/
listFileVersions(fileSpec:FileSpec | string, callback:DSCallback): void;

/**
* 
*  Gets the contents of a particular file version stored in this DataSource.
* 
* 
* @param {FileSpec | string} Either a FileSpec, or a String which         will be parsed to determine the fileName, fileType and fileFormat.         For instance, "employees.ds.xml" would be parsed as         {fileName: "employees", fileType: "ds", fileFormat: "xml"}.         If fileType and fileFormat are not provided, will return         the first file with the specified fileName.
* @param {Date} A version timestamp. This must exactly match the version         timestamp recorded in the DataSource. You can obtain the list of          versions for a given file with the DataSource.listFileVersions API.
* @param {GetFileVersionCallback} Callbacks.GetFileVersionCallback executed with the results. The         data parameter is either a String with the         contents of the file, or null to indicate error (such as         file not found).          You can examine DSResponse.status         and DSResponse.data         for additional information about any error.
*/
getFileVersion(fileSpec:FileSpec | string, version:Date, callback:(dsResponse:DSResponse, data:string, dsRequest:DSRequest) => void): void;

/**
* 
*  Indicates whether a particular file version exists in this DataSource. 
* 
* 
* @param {FileSpec | string} Either a FileSpec, or a String which         will be parsed to determine the fileName, fileType and fileFormat.         For instance, "employees.ds.xml" would be parsed as         {fileName: "employees", fileType: "ds", fileFormat: "xml"}.         If fileType or fileFormat are not provided, will indicate whether         any file with the provided fileName exists.
* @param {Date} A version timestamp. This must exactly match the version         timestamp recorded in the DataSource for hasFileVersion to         return true. Note, you can obtain the list of versions for a given          file with the DataSource.listFileVersions API.
* @param {HasFileCallback} Callbacks.HasFileVersionCallback         executed with the results. The data parameter is a boolean         indicating whether the file version is present.         You can examine DSResponse.status         and DSResponse.data         for additional information about any error.
*/
hasFileVersion(fileSpec:FileSpec | string, version:Date, callback:(dsResponse:DSResponse, data:boolean, dsRequest:DSRequest) => void): void;

/**
* 
*  Remove a particular file version stored in this DataSource. Any other versions of the
*  file are left untouched.
* 
* 
* @param {FileSpec | string} Either a FileSpec, or a String which         will be parsed to determine the fileName, fileType and fileFormat.         For instance, "employees.ds.xml" would be parsed as         {fileName: "employees", fileType: "ds", fileFormat: "xml"}.         If fileType and fileFormat are not provided, will return         the first file with the specified fileName.
* @param {Date} A version timestamp. This must exactly match the version         timestamp recorded in the DataSource. You can obtain the list of          versions for a given file with the DataSource.listFileVersions API.
* @param {DSCallback=} Callback executed with the results.         The data parameter is either a record representing the         removed file version, or null to indicate an error.         The record will have its DataSource.fileNameField,         DataSource.fileTypeField,         DataSource.fileFormatField,         DataSource.fileLastModifiedField, and          DataSource.fileVersionField fields populated, but          not the DataSource.fileContentsField field.         You can examine DSResponse.status         and DSResponse.data         for additional information about any error.
*/
removeFileVersion(fileSpec:FileSpec | string, version:Date, callback?:DSCallback): void;

/**
*  Executes the given DSRequest on this DataSource.
*  
*  This method is typically used by a DataSource whose DataSource.dataProtocol
*  is set to "clientCustom".
*  Execution of a DSRequest can be delayed, either after a timeout or until some condition
*  is met, by saving the DSRequest object passed to the DataSource.transformRequest
*  override and calling execute() on the DSRequest at a later time.
* 
* @param {DSRequest} the DSRequest to execute.
*/
execute(dsRequest:DSRequest): void;

/**
*  Override point to allow application code to suppress use of a particular offline
*  response. For example, application code may wish to examine the response's 
*  DSResponse.offlineTimestamp to make a decision about whether
*  the response is too stale to be useful.
*  
*  This is an application override point only; there is no default implementation.
* 
* 
* @param {DSRequest} The dsRequest object
* @param {DSResponse} The corresponding dsResponse object returned from                 offline cache
*/
useOfflineResponse(dsRequest:DSRequest, dsResponse:DSResponse): boolean;

/**
*  If you define this method on a DataSource, it will be called whenever the server returns
*  a DSResponse with a status other than RPCResponse.STATUS_SUCCESS. You can use
*  this hook to do DataSource-specific error handling. 
*  Unless you return false from this method, 
*  
*  RPCManager.handleError will be called by SmartClient right after this method completes.
*  
* 
* @param {DSResponse} the DSResponse or DSResponse object returned from the server
* @param {DSRequest} the DSRequest or DSRequest that was sent to the server
*/
handleError(response:DSResponse, request:DSRequest): boolean;

/**
*  Return the field definition object.
* 
* @param {FieldName} Name of the field to retrieve
*/
getField(fieldName:FieldName): DataSourceField;

/**
*  Return the field definition object corresponding to the supplied dataPath
* 
* @param {string} dataPath of the field to retrieve
*/
getFieldForDataPath(dataPath:string): DataSourceField;

/**
*  Given a fieldName and a dataValue, apply any DataSourceField.valueMap for the
*  field and return the display value for the field
* 
* @param {FieldName} name of the field to retrieve a value for
* @param {any} data value for the field
*/
getDisplayValue(fieldName:FieldName, value:any): any;

/**
*  Retrieves the list of fields declared on this DataSource.
* 
* 
* @param {boolean} If true, returns only those fields that are                 not marked as hidden
*/
getFieldNames(excludeHidden:boolean): Array<Partial<FieldName>>;

/**
*  Returns this DataSource's DataSourceField.primaryKey fields as a map
*  of fieldName to field.
* 
*/
getPrimaryKeyFields(): Record;

/**
*  Returns a list of the names of this DataSource's 
*  DataSourceField.primaryKey fields.
* 
*/
getPrimaryKeyFieldNames(): Array<Partial<string>>;

/**
*  Returns a pointer to the primaryKey field for this DataSource. If this dataSource has
*  a composite primary key (ie, multiple primaryKey fields), returns just the first 
*  primaryKey field.
* 
*/
getPrimaryKeyField(): DataSourceField;

/**
*  Returns the primary key fieldName for this DataSource.  If this dataSource has
*  a composite primary key (ie, multiple primaryKey fields), returns just the first 
*  primaryKey field name.
* 
*/
getPrimaryKeyFieldName(): string;

/**
*  Return a reasonable user-visible title given a fieldName. Called when
*  DataSource.autoDeriveTitles is true and by default, calls the class method
*  DataSource.getAutoTitle. Override to provide a different policy
*  for auto-deriving titles for a particular DataSource or subclass of DataSource.
* 
* 
* @param {string} identifier for which a title is desired.
*/
getAutoTitle(identifier:string): string;

/**
*  Formats the supplied value using the SimpleType derived from the field definition.
*  
*  Note that if DataSourceField.format is defined for a date, time or numeric-based
*  value, or DataSourceField.dateFormatter or DataSourceField.timeFormatter is
*  defined for a date or time-based value, that format is given priority and used to format
*  the value rather than the SimpleType.
* 
* 
* @param {DataSourceField | FieldName} name of the field to use to format value
* @param {any} raw value to be formatted
*/
formatFieldValue(field:DataSourceField | FieldName, value:any): string;

/**
*  Returns the complete set of data cached by this dataSource. Note that this may 
*  have been supplied via DataSource.cacheData, or may have been fetched
*  from the server for dataSources with DataSource.cacheAllData set to true.
* 
*/
getCacheData(): Array<Partial<Record>>;

/**
*  Return a "spoofed" response for a DataSource.clientOnly or DataSource.cacheAllData DataSource.
*  
*  The default implementation will use DataSource.cacheData to provide an appropriate
*  response, by using DataSource.applyFilter for a "fetch" request, and
*  by modifying the cacheData for other requests.
*  
*  Override this method to provide simulations of other server-side behavior, such as
*  modifying other records, or to implement synchronous client-side data providers
*  (such as Google Gears). For asynchronous third-party data providers, such as
*  GWT-RPC, HTML5 sockets, or bridges to plug-in based protocols (Java, Flash,
*  Silverlight..), use DSProtocol instead. 
*  
*  Overriding this method is also a means of detecting that a normal DataSource (not
*  clientOnly) would be contacting the server.
* 
* 
* @param {DSRequest} DataSource request to respond to
* @param {Array<Partial<Record>>} for cacheAllData DataSources, the data from the local cache
*/
getClientOnlyResponse(request:DSRequest, serverData:Array<Partial<Record>>): DSResponse;

/**
*  Returns records in the passed Array that match the provided filter
*  Criteria. Handles simple or AdvancedCriteria criteria.
*  
*  By default:
*  
*  
* - any criteria that do not correspond to a DataSource field are ignored
*  
* - for simple criteria, any null or empty string criteria are ignored and
*    all other criteria are passed to DataSource.fieldMatchesFilter
*  
* - for advanced criteria, each criterion is evaluated via DataSource.evaluateCriterion
*  
*  This method is called by ResultSet.applyFilter to provide filtering when a
*  ResultSet has a complete cache and filtering can be performed client-side. You may want
*  to override this method in order to mimic the filtering behavior that your server performs.
* 
* 
* @param {Array<Partial<Record>>} the list of rows
* @param {Criteria} the filter criteria
* @param {Partial<DSRequest>=} optional dataSource request properties
* @param {Integer=} starting index - earlier rows are excluded
* @param {Integer=} ending index (exclusive) - this row and beyond are excluded
*/
applyFilter(data:Array<Partial<Record>>, criteria:Criteria, requestProperties?:Partial<DSRequest>, startPos?:Integer, endPos?:Integer): Array<any>;

/**
*  Compares a criteria value to a field value and returns whether they match, as follows:
*  
*  
* - any non-String filter value is directly compared (==) to the field value
*  
* - any String filter value is compared according to
*    DSRequest.textMatchStyle in the passed requestProperties,
*    regardless of the actual field type
*  
* - if the filter value is an Array, the comparison is a logical OR. If textMatchStyle
*    is "exact", it matches if fieldValue (or any of it's entries, if it's also an array) 
*    is contained in the filterValue Array. If textMatchStyle if substring, it matches 
*    if any of the entries in filterValue appear as a case-insensitive substring of any
*    of the entries in fieldValue.
*  
* - Dates are compared as logical dates if either the field value or the filter value is a logical date.
*    Only if none of them is a logical date they will be compared as standard Dates
*  
* 
* @param {any} field value to be compared
* @param {any} filter value to be compared
* @param {Partial<DSRequest>=} optional dataSource request properties
*/
fieldMatchesFilter(fieldValue:any, filterValue:any, requestProperties?:Partial<DSRequest>): boolean;

/**
*  Does this dataSource support the specified "textMatchStyle" when performing a filter
*  operation against a text field.
* 
* @param {TextMatchStyle} textMatchStyle to check. If passed a null value,   assume an exact match is being requested.
*/
supportsTextMatchStyle(textMatchStyle:TextMatchStyle): void;

/**
*  Given two sets of criteria, determine whether they are equivalent, the new criteria is
*  guaranteed more restrictive, or the new criteria is not guaranteed more restrictive,
*  returning 0, 1 or -1 respectively.
*  
*  Comparisons between AdvancedCriteria are made via recursively calling
*  Operator.compareCriteria for all criteria involved.
*  
*  For simple Criteria, by default (CriteriaPolicy:"dropOnShortening"), returns:
*  
*  
* - -1 if the new criteria has fewer properties than the old criteria (indicating that it
*    isn't more restrictive)
*  
* - -1 if the value for any property in the old criteria is an array and 1) the value for
*    the same property in the new criteria isn't an array, or 2) is an array but
*    of different length, or 3) the arrays do not contain the exact same set of objects
*    (order can be different)
*  
* - -1 if the value for any given property in the old criteria is not an array, and the
*    the value for the same property property in the new criteria is different
*  
* - -1 if both values for a given property are strings and the new criteria value doesn't
*    contain the old criteria value
*  
* - 1 if none of the above are true and, for at least one of the properties, the
*    respective criteria values are both strings, and the old criteria value is a substring
*    of, and is shorter than, the new criteria value
*  
* - 0 otherwise (indicating the sets of criteria are equivalent)
*  
*  
*  For (CriteriaPolicy:"dropOnChange"), returns:
*  
*  
* - -1 if the two sets of criteria have a different number of properties
*  
* - -1 if the value for any property in the old criteria is an array and 1) the value for
*    the same property in the new criteria isn't an array, or 2) is an array but
*    of different length, or 3) the arrays do not contain the exact same set of objects
*    (order can be different)
*  
* - -1 if the value for any given property in the old criteria is not an array, and the
*    the value for the same property in the new criteria is different
*  
* - 0 otherwise (indicating the sets of criteria are equivalent)
*  
*  This method is called by ResultSet.compareCriteria to determine whether a change
*  in criteria should cause the cache to be invalidated. You may want
*  to override this method in order to mimic the filtering behavior that your server performs.
* 
* 
* @param {Criteria} new filter criteria
* @param {Criteria} previous filter criteria
* @param {Partial<DSRequest>=} dataSource request properties
* @param {string=} overrides CriteriaPolicy
*/
compareCriteria(newCriteria:Criteria, oldCriteria:Criteria, requestProperties?:Partial<DSRequest>, policy?:string): number;

/**
*  Returns the depth-first match of a criterion matching the given fieldName.
* 
* 
* @param {Criteria} the criteria to search
* @param {FieldName} the fieldName to find criteria for
*/
getFieldCriterion(criterion:Criteria, fieldName:FieldName): Criteria;

/**
*  Convenience method to compare two Date objects appropriately, depending on whether the
*  passed-in fieldName refers to a field of FieldType "datetime" or 
*  "date". In the former case, the dates are compared using DateUtil.compareDates;
*  in the latter case, or if the supplied fieldName is null or unknown to this DataSource,
*  the dates are compared using DateUtil.compareLogicalDates.
* 
* @param {Date} First date in comparison
* @param {Date} Second date in comparison
* @param {FieldName} The name of the field for which the comparison is being run
*/
compareDates(date1:Date, date2:Date, fieldName:FieldName): number;

/**
*  Convenience method to test if two records are equal. Testing is done only for the
*  fields defined in the DataSource, anything else is ignored.
* 
* 
* @param {any} record to be compared against.
* @param {any} record to be compared.
*/
recordsAreEqual(record1:any, record2:any): boolean;

/**
*  Converts criteria expressed in SmartClient's simple criteria format to an 
*  AdvancedCriteria object. This instance method differs from the class method
*  DataSource.convertCriteria in that it makes use of the dataSource 
*  as schema to help in the conversion. For example, this method is able to honor 
*  DataSourceField.ignoreTextMatchStyle and use the dataSource's 
*  DataSource.defaultTextMatchStyle rather than assuming "substring"
* 
* 
* @param {Criteria} simple criteria
* @param {TextMatchStyle=} default style of matching text. Defaults to                     the dataSource's defaultTextMatchStyle
*/
convertDataSourceCriteria(criteria:Criteria, textMatchStyle?:TextMatchStyle): AdvancedCriteria;

/**
*  Do fetch and filter operations on this dataSource support being passed 
*  AdvancedCriteria?
*  
*  For a DataSource to support being passed AdvancedCriteria, it must be 
*  DataSource.clientOnly or DataSource.cacheAllData, or have 
*  server side logic which can process AdvancedCriteria objects passed from the client.
*  
*  AdvancedCriteria are supported on the server for standard 
*  sqlDataSource, hibernateIntegration and 
*  jpaIntegration DataSources in SmartClient Enterprise or Power editions
*  (not supported in SmartClient Pro).
*  
*  The framework assumes that custom dataSources support AdvancedCriteria; if you have a 
*  a custom DataSOurce implementation that does not support AdvancedCriteria, you can set
*  the DataSource.allowAdvancedCriteria property to false.
*  
* 
*/
supportsAdvancedCriteria(): boolean;

/**
*  Add a new search operator, only to this DataSource. 
*  
*  If an existing Operator is passed, restricts the set of FieldTypes to which that
*  operator can be applied in this DataSource.
*  
*  See also DataSource.addSearchOperator for adding operators to all
*  DataSources.
* 
* 
* @param {Operator} definition of the operator to add
* @param {Array<Partial<FieldType>>=} types to which this operator applies
*/
addSearchOperator(operator:Operator, types?:Array<Partial<FieldType>>): void;

/**
*  Get the Operator definition for an OperatorId.
* 
* 
* @param {OperatorId} the id of the operator
*/
getSearchOperator(operatorId:OperatorId): Operator;

/**
*  Get the list of OperatorIds available on this DataSource for the given FieldType.
*  
*  If DataSource.setTypeOperators has been called for this DataSource and FieldType, returns that
*  list, otherwise, returns the set of valid operators for the FieldType as specified by 
*  SimpleType.validOperators, otherwise, the system-wide set of valid operators for the
*  type as registered via DataSource.addSearchOperator.
*  
* 
* @param {FieldType | string=} Defaults to "text" if not passed.
*/
getTypeOperators(typeName?:FieldType | string): Array<Partial<OperatorId>>;

/**
*  Set the list of OperatorIds valid for a given FieldType.
*  
* 
* @param {FieldType | string} * @param {Array<Partial<OperatorId>>} available Operators
*/
setTypeOperators(typeName:FieldType | string, operators:Array<Partial<OperatorId>>): void;

/**
*  Get the list of OperatorIds available for this field.
*  
*  By default, if DataSourceField.validOperators is set,
*  returns that list, otherwise returns the result of DataSource.getTypeOperators.
* 
* @param {string | DataSourceField} Field (or field name) to obtain operators for
*/
getFieldOperators(field:string | DataSourceField): Array<Partial<OperatorId>>;

/**
*  Get the default OperatorId for this field.
*  
*  By default, if DataSourceField.defaultOperator is set,
*  returns that value, otherwise returns the data-type default from 
*  SimpleType.defaultOperator.
* 
* @param {string | DataSourceField} Field (or field name) to obtain the default                     operator for
*/
getFieldDefaultOperator(field:string | DataSourceField): Array<Partial<OperatorId>>;

/**
*  Get the list of Operators available for this field, as a ValueMap from
*  OperatorId to the Operator.title specified for the Operator, or
*  the corresponding property in Operators if Operator.titleProperty is set.
*  
*  This valueMap is suitable for use in a UI for building queries, similar to the
*  FilterBuilder, and optionally omits operators marked Operator.hidden:true.
*  
*  It is also possible to have this function return only operators of a given 
*  OperatorValueType, or everything except operators of that type. This is useful, 
*  for example, if you want to return all the logical operators (like "and"), or everything 
*  except the logical operators.
* 
* 
* @param {string | DataSourceField} Field (or field name) to obtain operator map for.
* @param {boolean=} whether to include Operators marked hidden:true
* @param {OperatorValueType=} If passed, returns only operators of this OperatorValueType
* @param {boolean=} If set, reverses the meaning of the valueType     parameter, so operators of that OperatorValueType are the only ones omitted
*/
getFieldOperatorMap(field:string | DataSourceField, includeHidden?:boolean, valueType?:OperatorValueType, omitValueType?:boolean): ValueMap;

/**
*  Get the list of Operators available for this FieldType, as a ValueMap from
*  OperatorId to the Operator.title specified for the Operator, or
*  the corresponding property in Operators if Operator.titleProperty is set.
*  
*  This valueMap is suitable for use in a UI for building queries, similar to the
*  FilterBuilder, and optionally omits operators marked Operator.hidden : true.
*  
*  It is also possible to have this function return only operators of a given 
*  OperatorValueType, or everything except operators of that type. This is useful, 
*  for example, if you want to return all the logical operators (like "and"), or everything 
*  except the logical operators.
* 
* 
* @param {FieldType=} Type to obtain operator map for. Defaults to "text" if not passed.
* @param {boolean=} whether to include Operators marked hidden:true
* @param {OperatorValueType=} If passed, returns only operators of this OperatorValueType
* @param {boolean=} If set, reverses the meaning of the valueType     parameter, so operators of that OperatorValueType are the only ones omitted
*/
getTypeOperatorMap(type?:FieldType, includeHidden?:boolean, valueType?:OperatorValueType, omitValueType?:boolean): ValueMap;

/**
*  Converts a list of Records to simple text formats with a Record per line and values separated
*  by a configurable separator, including both tab-separated-values and comma-separated-values
*  (aka CSV).
*  
*  In addition to the settings parameter for this method,
*  DataSourceField.exportForceText can be set. 
*  
*  If two or more different text exports are needed for the same DataSource creating a conflict
*  for any DataSourceField setting, DataSource.inheritsFrom can be used to create a
*  child DataSource where these settings can be changed without recapitulating all field
*  definitions.
* 
* 
* @param {Array<Partial<Record>>} records to convert
* @param {Partial<TextExportSettings>=} settings for the export
*/
recordsAsText(records:Array<Partial<Record>>, settings?:Partial<TextExportSettings>): string;

/**
*  Derive a list of Records from Microsoft Excel-compatible tab-separated-values format, using
*  the current DataSource field order, or an explicitly specified list of fields.
*  
*  If a specified field does not exist in the DataSource, it's assumed the values for that
*  field should end up as Strings.
*  
* 
* @param {string} records as CSV/TSV (separator can be specified)
* @param {Partial<TextImportSettings>=} optional settings for the import
*/
recordsFromText(text:string, settings?:Partial<TextImportSettings>): Array<Partial<Record>>;

/**
*  Evaluate the given criterion with respect to the passed record.
*  
*  Typically called by the Operator.condition function of a custom
*  Operator to evaluate Criterion.criteria.
* 
* 
* @param {Record} record to evaluate
* @param {Criterion} criterion to use
*/
evaluateCriterion(record:Record, criterion:Criterion): boolean;

/**
*  Split a criteria apart based on fields. A new simple criteria
*  is returned with any criteria applicable to the specified fields. The passed
*  criteria is then modified to remove these fields resulting in
*  two distinct criteria.
*  
*  Incoming criteria can be a simple or advanced criteria. For an AdvancedCriteria
*  only a single level of criteria with a top-level operator of "and" is supported.
*  
*  To avoid modifying an original criteria, use DataSource.copyCriteria to
*  make a copy to be passed in.
* 
* 
* @param {Criteria} criteria to be split. May be modified if criteria is              extracted.
* @param {Array<Partial<string>>} list of fields to extract from criteria
*/
splitCriteria(criteria:Criteria, fields:Array<Partial<string>>): Criteria;



}
/**
* Creates a new DataSource
* 
* @param typeCheckedProperties {Partial<DataSource>=} Object literal containing init properties that will be checked against the class field types.
* @param uncheckedProperties {Object} Object literal containing init properties that will not be checked against the class field types.
* @return {DataSource} The new instance
*/
class DataSource{
static create(typeCheckedProperties? : Partial<DataSource>, uncheckedProperties?: Object): DataSource;
/**
* The URL where the DataSourceLoader servlet has been installed. Defaults to the
*  Page.setIsomorphicDir plus "/DataSourceLoader". Change by calling
*  calling DataSource.setLoaderURL
* @type {URL}
* @default "[ISOMORPHIC]/DataSourceLoader"
*/
static loaderURL?: URL;

/**
* Default value if DataSourceField.stringInBrowser is not defined. See
*  stringInBrowser docs for details.
*  Note this setting should be used only if you are *not* using the SmartClient Server.
* @type {boolean}
* @default null
*/
static defaultStringInBrowser?: boolean;

/**
* Specifies how time field values should be serialized when being sent to the server for
*  dataSources with dataFormat "xml" or "json".
*  If false the time field will be serialized as a logical time object in
*  UTC, using the standard truncated XML Schema format: "HH:MM:SS".
*  If true the time field will be serialized as a complete dateTime object
*  matching the value of the underlying JavaScript date object representing this time on the
*  client.
* @type {boolean}
* @default false
*/
static serializeTimeAsDatetime?: boolean;

/**
* A message returned by a DataSource when it is returning an empty dataset for a fetch
*  because the browser is currently offline and there is no suitable cached offline response.
* @type {string}
* @default "This data not available while offline"
*/
static offlineMessage?: string;

/**
* A message returned by a DataSource when an uploaded file's size exceeded
*  DataSourceField.maxFileSize.
*  
*  If this is not set, then Validator.maxFileSizeExceeded value will be used.
* @type {HTMLString}
* @default "Size of '${uploadedFileName}' (${uploadedFileSize}) exceeded maximum allowed file size of ${maxFileSize}"
*/
static maxFileSizeExceededMessage?: HTMLString;

/**
* A message returned by a DataSource when an uploaded file was empty, but the
*  field is DataSourceField.required.
* @type {HTMLString}
* @default "'${uploadedFileName}' was empty, but empty files are not allowed."
*/
static requiredFileMessage?: HTMLString;

/**
* A message returned by a DataSource when an operation requires criteria, but none was provided.
* @type {HTMLString}
* @default "Operation requires criteria for the following field(s): [${requiredCriterion}]"
*/
static requiredCriterionMessage?: HTMLString;

/**
* Provides the global default setting for deep or shallow cloning of objects prior to
*  edit. See DataSource.deepCloneOnEdit for details of what this means.
*  
*  The default setting of null is the same as false; with this default in place, 
*  deepCloneOnEdit must be set explicitly set at dataSource, component or 
*  field level if you require deep cloning.
* @type {boolean}
* @default null
*/
static deepCloneOnEdit?: boolean;

/**
* Provides the global default setting for deep or shallow cloning of non-field attributes
*  of value objects prior to edit. See 
*  DataSource.deepCloneNonFieldValuesOnEdit for details of what this means.
*  
*  The default setting of null is the same as false; with this default in place, 
*  deepCloneNonFieldValuesOnEdit must be set explicitly set at dataSource or
*  component level if you require deep cloning.
* @type {boolean}
* @default null
*/
static deepCloneNonFieldValuesOnEdit?: boolean;





/**
*  Lookup a DataSource by ID.
* 
* 
* @param {GlobalId} DataSource ID
*/
static getDataSource(ID:GlobalId): DataSource;

/**
*  Synonym of DataSource.getDataSource: Lookup a DataSource by ID.
* 
* 
* @param {GlobalId} DataSource ID
*/
static get(ID:GlobalId): DataSource;

/**
*  Helper method to return the value of the supplied field from within the supplied record,
*  looking up the value from the supplied dataPath. This method is only of real
*  use when you are dealing with complex nested data via Canvas.dataPath; 
*  it is obviously a trivial matter to obtain a field value from a flat record directly.
*  
*  If the dataPath is null, this method will follow any ListGridField.dataPath 
*  specified on the component field instead. In either case, it will also extract 
*  SimpleType.getAtomicValue from custom SimpleType fields
*  where this is required.
* 
* 
* @param {DataSourceField | ListGridField | DetailViewerField | FormItem} Field definition from a dataSource or dataBoundComponent.
* @param {Record} The valueset in which to look up the field value
* @param {string} Optionally, a string expressing the dataPath to use for value lookup.              If null, the dataPath from the field will be used
* @param {Canvas} Optionally, a component to prvide extra context for the dataPath               search. This is typically required if the dataPath traverses a list
* @param {string} An optional reason for the get request, to be passed into                  any SimpleType.getAtomicValue method - see that                  API for details
*/
static getFieldValue(field:DataSourceField | ListGridField | DetailViewerField | FormItem, record:Record, dataPath:string, component:Canvas, reason:string): any;

/**
*  Helper method to save the argument value inside the argument values record, storing the 
*  value at the supplied dataPath, or at the field's declared dataPath if the argument 
*  dataPath is null. This method is only of real use when you are dealing with complex 
*  nested data via Canvas.dataPath; it is obviously a trivial matter 
*  to store a 