#!/usr/bin/env node
/**
 * Selenese Playwright Runner
 *
 * Executes Selenese HTML test files (.rctest.html) using Playwright instead of Selenium WebDriver.
 * Provides a migration path from Selenium-based testing to Playwright while maintaining existing
 * Selenese test suites.
 *
 * Requirements:
 *   npm install playwright
 *
 * Usage:
 *   node selenese-runner.js [options] <baseUrl> <testFile|suiteFile>...
 *
 * Options:
 *   -ch, --chrome      Use Chrome/Chromium browser (default)
 *   -ff, --firefox     Use Firefox browser
 *   -d, --debug        Enable debug mode (slow down, show actions)
 *   -q, --quiet        Suppress output
 *   -j <file>          Write JUnit XML report to file
 *   --headed           Run with visible browser window
 *   --screenshot       Take screenshot on test failure
 *
 * Examples:
 *   node selenese-runner.js http://localhost:8080/app tests/suite.html
 *   node selenese-runner.js -d --headed http://localhost:8080 tests/mytest.rctest.html
 *   node selenese-runner.js -j results.xml http://localhost:8080 tests/suite.html
 */

const fs = require('fs');
const path = require('path');
const { chromium, firefox } = require('playwright');

// Import SmartClient Playwright extensions
const { extendPage, scGoto, detectJSLoops } = require('./commands.js');

// ============================================================================
// Key mappings from Selenese ${KEY_XXX} to Playwright keyboard names
// ============================================================================
const KEY_MAP = {
    'KEY_ENTER': 'Enter', 'KEY_RETURN': 'Enter', 'KEY_TAB': 'Tab',
    'KEY_ESC': 'Escape', 'KEY_ESCAPE': 'Escape',
    'KEY_BACKSPACE': 'Backspace', 'KEY_BACK_SPACE': 'Backspace',
    'KEY_DELETE': 'Delete', 'KEY_DEL': 'Delete', 'KEY_INSERT': 'Insert',
    'KEY_HOME': 'Home', 'KEY_END': 'End',
    'KEY_PAGE_UP': 'PageUp', 'KEY_PAGEUP': 'PageUp',
    'KEY_PAGE_DOWN': 'PageDown', 'KEY_PAGEDOWN': 'PageDown',
    'KEY_UP': 'ArrowUp', 'KEY_DOWN': 'ArrowDown',
    'KEY_LEFT': 'ArrowLeft', 'KEY_RIGHT': 'ArrowRight',
    'KEY_CTRL': 'Control', 'KEY_CONTROL': 'Control',
    'KEY_ALT': 'Alt', 'KEY_SHIFT': 'Shift', 'KEY_META': 'Meta', 'KEY_SPACE': ' ',
    'KEY_F1': 'F1', 'KEY_F2': 'F2', 'KEY_F3': 'F3', 'KEY_F4': 'F4',
    'KEY_F5': 'F5', 'KEY_F6': 'F6', 'KEY_F7': 'F7', 'KEY_F8': 'F8',
    'KEY_F9': 'F9', 'KEY_F10': 'F10', 'KEY_F11': 'F11', 'KEY_F12': 'F12'
};

// ============================================================================
// HTML Parser - Parse Selenese .rctest.html files
// ============================================================================
function decodeHtmlEntities(str) {
    if (!str) return str;
    return str.replace(/&amp;/g, '&').replace(/&lt;/g, '<').replace(/&gt;/g, '>')
        .replace(/&quot;/g, '"').replace(/&#39;/g, "'").replace(/&apos;/g, "'")
        .replace(/&#(\d+);/g, (m, dec) => String.fromCharCode(dec))
        .replace(/&#x([0-9A-Fa-f]+);/g, (m, hex) => String.fromCharCode(parseInt(hex, 16)));
}

function stripTags(str) { return str ? str.replace(/<[^>]*>/g, '') : ''; }

function isHeaderRow(cmd) {
    const lc = cmd.toLowerCase();
    return lc.includes('colspan') || lc.includes('rowspan') || lc.endsWith('.rctest') ||
           lc.endsWith('.test') || lc === 'command' || lc === 'target' || lc === 'value';
}

async function parseSeleneseFile(filePath) {
    let content = fs.readFileSync(filePath, 'utf-8');
    const commands = [];
    // Extract only tbody content to avoid header rows
    const tbodyMatch = content.match(/<tbody[^>]*>([\s\S]*?)<\/tbody>/i);
    if (tbodyMatch) content = tbodyMatch[1];

    const rowRegex = /<tr[^>]*>\s*<td[^>]*>([^<]*(?:<(?!\/td>)[^<]*)*)<\/td>\s*<td[^>]*>([^<]*(?:<(?!\/td>)[^<]*)*)<\/td>\s*<td[^>]*>([^<]*(?:<(?!\/td>)[^<]*)*)<\/td>\s*<\/tr>/gi;
    let match;
    while ((match = rowRegex.exec(content)) !== null) {
        const command = stripTags(match[1]).trim();
        const target = decodeHtmlEntities(stripTags(match[2]).trim());
        const value = decodeHtmlEntities(stripTags(match[3]).trim());
        if (command && !isHeaderRow(command)) commands.push({ command, target, value });
    }
    return commands;
}

async function parseSuiteFile(filePath) {
    const content = fs.readFileSync(filePath, 'utf-8');
    const suite = { name: path.basename(filePath, '.html'), tests: [] };
    const titleMatch = content.match(/<td[^>]*><b>([^<]+)<\/b><\/td>/i);
    if (titleMatch) suite.name = titleMatch[1].trim();

    const testRegex = /<tr[^>]*>\s*<td[^>]*>\s*<a\s+href="([^"]+)"[^>]*>([^<]*)<\/a>\s*<\/td>\s*<\/tr>/gi;
    let match;
    while ((match = testRegex.exec(content)) !== null) {
        const file = match[1].trim();
        const name = match[2].trim() || path.basename(file, '.rctest.html');
        if (file.includes('.rctest.html') || file.includes('.test.html')) suite.tests.push({ name, file });
    }
    return suite;
}

// ============================================================================
// Locator parsing - supports both scLocator and standard Selenium locators
// ============================================================================
function parseLocator(locator) {
    if (!locator) return { type: 'empty', value: '' };
    const lc = locator.toLowerCase();
    if (lc.startsWith('sclocator=')) return { type: 'scLocator', value: locator.substring(10) };
    // SmartClient AutoTest locators start with // but don't have [@
    if (locator.startsWith('//') && !locator.includes('[@')) return { type: 'scLocator', value: locator };
    if (lc.startsWith('id=')) return { type: 'id', value: locator.substring(3) };
    if (lc.startsWith('name=')) return { type: 'name', value: locator.substring(5) };
    if (lc.startsWith('xpath=')) return { type: 'xpath', value: locator.substring(6) };
    if (lc.startsWith('css=')) return { type: 'css', value: locator.substring(4) };
    if (lc.startsWith('link=')) return { type: 'link', value: locator.substring(5) };
    if (lc.startsWith('identifier=')) return { type: 'identifier', value: locator.substring(11) };
    return { type: 'css', value: locator };
}

async function getPlaywrightLocator(page, locInfo) {
    switch (locInfo.type) {
        case 'scLocator': return await page.getSC(locInfo.value);
        case 'id': return page.locator(`#${locInfo.value}`);
        case 'name': return page.locator(`[name="${locInfo.value}"]`);
        case 'xpath': return page.locator(`xpath=${locInfo.value}`);
        case 'css': return page.locator(locInfo.value);
        case 'link': return page.locator(`text=${locInfo.value}`);
        case 'identifier': return page.locator(`#${locInfo.value}, [name="${locInfo.value}"]`).first();
        default: throw new Error(`Unknown locator type: ${locInfo.type}`);
    }
}

// ============================================================================
// Variable expansion and key sequence handling
// ============================================================================
function expandVariables(str, storedVars) {
    if (!str || typeof str !== 'string') return str;
    return str.replace(/\$\{([^}]+)\}/g, (match, varName) => {
        if (varName in storedVars) return String(storedVars[varName]);
        if (varName in KEY_MAP) return `\${${varName}}`;
        return match;
    });
}

async function sendKeys(page, element, keySequence) {
    const parts = [];
    const keyPattern = /\$\{(KEY_[A-Z_0-9]+)\}/g;
    let lastIndex = 0, match;
    while ((match = keyPattern.exec(keySequence)) !== null) {
        if (match.index > lastIndex) parts.push({ type: 'text', value: keySequence.substring(lastIndex, match.index) });
        parts.push({ type: 'key', value: KEY_MAP[match[1]] || match[1] });
        lastIndex = match.index + match[0].length;
    }
    if (lastIndex < keySequence.length) parts.push({ type: 'text', value: keySequence.substring(lastIndex) });

    const modifiers = [];
    for (const part of parts) {
        if (part.type === 'key') {
            const key = part.value;
            if (['Control', 'Alt', 'Shift', 'Meta'].includes(key)) {
                const idx = modifiers.indexOf(key);
                if (idx >= 0) { await page.keyboard.up(key); modifiers.splice(idx, 1); }
                else { await page.keyboard.down(key); modifiers.push(key); }
            } else { await page.keyboard.press(key); }
        } else { await page.keyboard.type(part.value); }
    }
    for (const mod of modifiers) await page.keyboard.up(mod);
}

// ============================================================================
// Command Executor - Maps Selenese commands to Playwright/SmartClient actions
// Uses page.clickSC(), page.typeSC(), page.waitForSCDone() from commands.js
// ============================================================================
async function executeCommand(page, cmd, options = {}) {
    const { baseUrl = '', storedVars = {}, timeout = 30000 } = options;
    const target = expandVariables(cmd.target, storedVars);
    const value = expandVariables(cmd.value, storedVars);
    const command = cmd.command;
    const andWait = command.endsWith('AndWait');
    const baseCommand = andWait ? command.slice(0, -7) : command;

    switch (baseCommand) {
        case 'open': {
            let url = target;
            const isHashNav = url.startsWith('#');
            if (isHashNav) {
                url = (baseUrl.endsWith('/') ? baseUrl.slice(0, -1) : baseUrl) + url;
            } else if (!url.startsWith('http://') && !url.startsWith('https://')) {
                url = baseUrl + (url.startsWith('/') ? '' : '/') + url;
            }
            // Use scGoto for proper SmartClient page load handling
            await scGoto(page, url);
            // Extra wait for hash navigation (showcase-style apps)
            if (isHashNav) {
                await page.waitForTimeout(200);
                await page.waitForSCDone();
            }
            break;
        }
        case 'click': case 'clickAt': {
            const locInfo = parseLocator(target);
            if (locInfo.type === 'scLocator') {
                await page.clickSC(locInfo.value);
            } else {
                const loc = await getPlaywrightLocator(page, locInfo);
                await loc.click({ timeout });
                await page.waitForSCDone();
            }
            break;
        }
        case 'doubleClick': case 'doubleClickAt': {
            const locInfo = parseLocator(target);
            const el = await getPlaywrightLocator(page, locInfo);
            await el.dblclick();
            await page.waitForSCDone();
            break;
        }
        case 'contextMenu': case 'contextClick': {
            const locInfo = parseLocator(target);
            const el = await getPlaywrightLocator(page, locInfo);
            await el.click({ button: 'right' });
            await page.waitForSCDone();
            break;
        }
        case 'mouseMove': case 'hover': case 'mouseOver': {
            const locInfo = parseLocator(target);
            if (locInfo.type === 'scLocator') {
                await page.hoverSC(locInfo.value);
            } else {
                const loc = await getPlaywrightLocator(page, locInfo);
                await loc.hover({ timeout });
                await page.waitForSCDone();
            }
            break;
        }
        case 'mouseDown': {
            const locInfo = parseLocator(target);
            const el = await getPlaywrightLocator(page, locInfo);
            const box = await el.boundingBox();
            if (box) {
                await page.mouse.move(box.x + box.width / 2, box.y + box.height / 2);
                await page.mouse.down();
            }
            break;
        }
        case 'mouseUp': {
            await page.mouse.up();
            await page.waitForSCDone();
            break;
        }
        case 'type': {
            const locInfo = parseLocator(target);
            if (locInfo.type === 'scLocator') {
                await page.typeSC(locInfo.value, value);
            } else {
                const loc = await getPlaywrightLocator(page, locInfo);
                await loc.fill(value, { timeout });
                await page.waitForSCDone();
            }
            break;
        }
        case 'sendKeys': {
            const locInfo = parseLocator(target);
            const element = await getPlaywrightLocator(page, locInfo);
            await element.focus();
            await sendKeys(page, element, value);
            await page.waitForSCDone();
            break;
        }
        case 'keyPress': case 'keyDown': {
            const keyName = KEY_MAP[(value || target).replace(/\$\{|\}/g, '')] || value || target;
            await page.keyboard.down(keyName);
            break;
        }
        case 'keyUp': {
            const keyName = KEY_MAP[(value || target).replace(/\$\{|\}/g, '')] || value || target;
            await page.keyboard.up(keyName);
            break;
        }
        case 'dragAndDrop': case 'dragAndDropToObject': {
            const srcInfo = parseLocator(target), tgtInfo = parseLocator(value);
            if (srcInfo.type === 'scLocator' && tgtInfo.type === 'scLocator') {
                await page.dragAndDropSC(srcInfo.value, tgtInfo.value);
            } else {
                const srcEl = await getPlaywrightLocator(page, srcInfo);
                const tgtEl = await getPlaywrightLocator(page, tgtInfo);
                const srcBox = await srcEl.boundingBox(), tgtBox = await tgtEl.boundingBox();
                if (srcBox && tgtBox) {
                    await page.mouse.move(srcBox.x + srcBox.width / 2, srcBox.y + srcBox.height / 2);
                    await page.mouse.down();
                    await page.mouse.move(tgtBox.x + tgtBox.width / 2, tgtBox.y + tgtBox.height / 2, { steps: 10 });
                    await page.mouse.up();
                }
                await page.waitForSCDone();
            }
            break;
        }
        case 'waitForElementClickable': case 'waitForElementPresent': case 'waitForElementVisible': {
            const locInfo = parseLocator(target);
            if (locInfo.type === 'scLocator') {
                // Uses isc.AutoTest.waitForElement via getSC
                await page.getSC(locInfo.value, { timeout });
            } else {
                const loc = await getPlaywrightLocator(page, locInfo);
                await loc.waitFor({ state: 'visible', timeout });
            }
            break;
        }
        case 'waitForElementNotPresent': case 'waitForElementNotVisible': case 'waitForElementNotClickable': {
            const locInfo = parseLocator(target);
            if (locInfo.type === 'scLocator') {
                await page.evaluate(({ locator, timeout }) => new Promise((resolve) => {
                    const isc = globalThis.isc, startTime = Date.now();
                    const check = () => {
                        if (!isc?.AutoTest?.getElement?.(locator)) return resolve();
                        if (Date.now() - startTime > timeout) return resolve();
                        setTimeout(check, 100);
                    };
                    check();
                }), { locator: locInfo.value, timeout });
            } else {
                const loc = await getPlaywrightLocator(page, locInfo);
                await loc.waitFor({ state: 'hidden', timeout });
            }
            break;
        }
        case 'pause': {
            await page.waitForTimeout(parseInt(target) || parseInt(value) || 1000);
            break;
        }
        case 'verifyText': case 'assertText': {
            const locInfo = parseLocator(target);
            const actualText = await page.scGetLocatorText(locInfo.value);
            if (actualText.replace(/\s+/g, ' ').trim() !== value.replace(/\s+/g, ' ').trim()) {
                throw new Error(`Text verification failed. Expected: "${value}", Actual: "${actualText}"`);
            }
            break;
        }
        case 'verifyTextPresent': case 'assertTextPresent': {
            const bodyText = await page.evaluate(() => document.body.innerText || '');
            if (!bodyText.includes(target)) throw new Error(`Text "${target}" not found`);
            break;
        }
        case 'verifyElementPresent': case 'assertElementPresent': {
            const locInfo = parseLocator(target);
            const exists = locInfo.type === 'scLocator'
                ? await page.existsSCElement(locInfo.value)
                : (await (await getPlaywrightLocator(page, locInfo)).count()) > 0;
            if (!exists) throw new Error(`Element not present: ${target}`);
            break;
        }
        case 'verifyElementNotPresent': case 'assertElementNotPresent': {
            const locInfo = parseLocator(target);
            const exists = locInfo.type === 'scLocator'
                ? await page.existsSCElement(locInfo.value)
                : (await (await getPlaywrightLocator(page, locInfo)).count()) > 0;
            if (exists) throw new Error(`Element should not be present: ${target}`);
            break;
        }
        case 'verifyValue': case 'assertValue': {
            const locInfo = parseLocator(target);
            const actualValue = locInfo.type === 'scLocator'
                ? await page.evaluate((loc) => globalThis.isc?.AutoTest?.getElement?.(loc)?.value, locInfo.value)
                : await (await getPlaywrightLocator(page, locInfo)).inputValue();
            if (actualValue !== value) throw new Error(`Value verification failed. Expected: "${value}", Actual: "${actualValue}"`);
            break;
        }
        case 'store': { storedVars[value] = target; break; }
        case 'storeText': {
            const locInfo = parseLocator(target);
            storedVars[value] = await page.scGetLocatorText(locInfo.value);
            break;
        }
        case 'storeValue': {
            const locInfo = parseLocator(target);
            storedVars[value] = locInfo.type === 'scLocator'
                ? await page.evaluate((loc) => globalThis.isc?.AutoTest?.getElement?.(loc)?.value || '', locInfo.value)
                : await (await getPlaywrightLocator(page, locInfo)).inputValue();
            break;
        }
        case 'storeEval': {
            storedVars[value] = await page.evaluate((script) => eval(script), target);
            break;
        }
        case 'select': {
            const loc = await getPlaywrightLocator(page, parseLocator(target));
            if (value.startsWith('label=')) await loc.selectOption({ label: value.substring(6) });
            else if (value.startsWith('value=')) await loc.selectOption({ value: value.substring(6) });
            else if (value.startsWith('index=')) await loc.selectOption({ index: parseInt(value.substring(6)) });
            else await loc.selectOption({ label: value });
            await page.waitForSCDone();
            break;
        }
        case 'focus': {
            const locInfo = parseLocator(target);
            const el = await getPlaywrightLocator(page, locInfo);
            await el.focus();
            break;
        }
        case 'runScript': case 'getEval': {
            await page.evaluate((script) => eval(script), target);
            break;
        }
        case 'controlKeyDown': { await page.keyboard.down('Control'); break; }
        case 'controlKeyUp': { await page.keyboard.up('Control'); break; }
        case 'shiftKeyDown': { await page.keyboard.down('Shift'); break; }
        case 'shiftKeyUp': { await page.keyboard.up('Shift'); break; }
        case 'altKeyDown': { await page.keyboard.down('Alt'); break; }
        case 'altKeyUp': { await page.keyboard.up('Alt'); break; }
        case 'echo': { console.log(`[echo] ${target}`); break; }
        case 'refresh': {
            await page.reload();
            await page.waitForFunction(() => window.isc?.Page?.isLoaded?.() === true, { timeout });
            await page.waitForSCDone();
            break;
        }
        case 'goBack': { await page.goBack(); await page.waitForSCDone(); break; }
        case 'goForward': { await page.goForward(); await page.waitForSCDone(); break; }
        case 'windowFocus': { await page.bringToFront(); break; }
        case 'setSpeed': case 'windowMaximize': break; // No-ops
        default:
            console.warn(`Unknown Selenese command: ${command}`);
            break;
    }
    // Always wait for system done after every command, matching Java Selenese runner behavior
    await page.waitForSCDone();
}

// ============================================================================
// Main Runner Class
// ============================================================================
const DEFAULT_CONFIG = {
    browser: 'chromium',
    headed: false,
    debug: false,
    quiet: false,
    screenshotOnFailure: true,
    timeout: 30000,
    baseUrl: 'http://localhost:8080'
};

class SeleneseRunner {
    constructor(config = {}) {
        this.config = { ...DEFAULT_CONFIG, ...config };
        this.browser = null;
        this.page = null;
        this.storedVars = {};
    }

    async initialize() {
        const browserType = (this.config.browser === 'chromium' || this.config.browser === 'chrome') ? chromium : firefox;
        this.browser = await browserType.launch({
            headless: !this.config.headed,
            slowMo: this.config.debug ? 500 : 0
        });
        const context = await this.browser.newContext({
            viewport: { width: 1280, height: 1024 }
        });
        this.page = await context.newPage();

        // Extend page with SmartClient commands from commands.js
        extendPage(this.page);
        // Suppress verbose logging from commands.js
        this.page.setLogLevel('silent');
    }

    async close() {
        if (this.browser) {
            await this.browser.close();
            this.browser = null;
        }
    }

    async runTest(testFile) {
        const testName = path.basename(testFile, '.rctest.html');
        const result = {
            name: testName,
            file: testFile,
            passed: true,
            commands: [],
            startTime: Date.now(),
            error: null
        };

        // Set up infinite loop detection for the test
        const stopMonitor = await detectJSLoops(this.page, 30);

        try {
            if (!this.config.quiet) console.log(`\n*** Running test: ${testName}`);
            const commands = await parseSeleneseFile(testFile);
            if (!this.config.quiet) console.log(`    Found ${commands.length} commands`);

            for (let i = 0; i < commands.length; i++) {
                const cmd = commands[i];
                if (this.config.debug && !this.config.quiet) {
                    console.log(`    [${i + 1}/${commands.length}] ${cmd.command}: ${cmd.target} ${cmd.value || ''}`);
                }
                try {
                    await executeCommand(this.page, cmd, {
                        baseUrl: this.config.baseUrl,
                        storedVars: this.storedVars,
                        timeout: this.config.timeout
                    });
                } catch (err) {
                    result.passed = false;
                    result.error = `Command ${i + 1} (${cmd.command}) failed: ${err.message}`;
                    if (!this.config.quiet) {
                        console.error(`    ✗ FAILED at command ${i + 1}: ${cmd.command}`);
                        console.error(`      ${err.message}`);
                    }
                    if (this.config.screenshotOnFailure) {
                        try {
                            const ss = path.join(path.dirname(testFile), `${testName}-failure-${i}.png`);
                            await this.page.screenshot({ path: ss, fullPage: true });
                            console.log(`      Screenshot saved: ${ss}`);
                        } catch (e) { /* ignore screenshot errors */ }
                    }
                    break;
                }
            }
        } catch (err) {
            result.passed = false;
            result.error = err.message;
        } finally {
            await stopMonitor();
        }

        result.duration = Date.now() - result.startTime;
        if (!this.config.quiet) {
            console.log(`*** ${testName}: ${result.passed ? '✓ PASSED' : '✗ FAILED'} (${result.duration}ms)`);
        }
        return result;
    }

    async runSuite(suiteFile) {
        const suiteDir = path.dirname(suiteFile);
        const suite = await parseSuiteFile(suiteFile);
        if (!this.config.quiet) {
            console.log(`\n=== Running suite: ${suite.name} ===`);
            console.log(`    ${suite.tests.length} tests\n`);
        }

        const suiteResult = {
            name: suite.name,
            tests: [],
            passed: 0,
            failed: 0,
            startTime: Date.now()
        };

        for (const testRef of suite.tests) {
            const result = await this.runTest(path.resolve(suiteDir, testRef.file));
            suiteResult.tests.push(result);
            result.passed ? suiteResult.passed++ : suiteResult.failed++;
        }

        suiteResult.duration = Date.now() - suiteResult.startTime;
        if (!this.config.quiet) {
            console.log(`\n=== Suite complete: ${suiteResult.passed}/${suite.tests.length} passed ===`);
        }
        return suiteResult;
    }

    generateJUnitXML(suiteResult) {
        const esc = (s) => s ? s.replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;') : '';
        let xml = `<?xml version="1.0" encoding="UTF-8"?>\n`;
        xml += `<testsuite name="${esc(suiteResult.name)}" tests="${suiteResult.tests.length}" failures="${suiteResult.failed}" errors="0" time="${(suiteResult.duration / 1000).toFixed(3)}">\n`;
        for (const test of suiteResult.tests) {
            xml += `  <testcase name="${esc(test.name)}" classname="selenese" time="${(test.duration / 1000).toFixed(3)}"`;
            if (test.passed) {
                xml += '/>\n';
            } else {
                xml += `>\n    <failure message="${esc(test.error)}">${esc(test.error)}</failure>\n  </testcase>\n`;
            }
        }
        xml += '</testsuite>\n';
        return xml;
    }
}

// ============================================================================
// CLI Entry Point
// ============================================================================
async function main() {
    const args = process.argv.slice(2);
    const config = { ...DEFAULT_CONFIG };
    const testFiles = [];
    let junitFile = null;

    for (let i = 0; i < args.length; i++) {
        const arg = args[i];
        if (arg === '-ch' || arg === '--chrome') config.browser = 'chrome';
        else if (arg === '-ff' || arg === '--firefox') config.browser = 'firefox';
        else if (arg === '-d' || arg === '--debug') config.debug = true;
        else if (arg === '-q' || arg === '--quiet') config.quiet = true;
        else if (arg === '--headed') config.headed = true;
        else if (arg === '--screenshot') config.screenshotOnFailure = true;
        else if (arg === '-j' && i + 1 < args.length) junitFile = args[++i];
        else if (arg.startsWith('http://') || arg.startsWith('https://')) config.baseUrl = arg;
        else if (!arg.startsWith('-')) testFiles.push(arg);
    }

    if (testFiles.length === 0) {
        console.log(`Selenese Playwright Runner

Usage: node selenese-runner.js [options] <baseUrl> <testFile|suiteFile>...

Options:
  -ch, --chrome    Use Chrome/Chromium browser (default)
  -ff, --firefox   Use Firefox browser
  -d, --debug      Enable debug mode (slow down, show actions)
  -q, --quiet      Suppress output
  -j <file>        Write JUnit XML report to file
  --headed         Run with visible browser window
  --screenshot     Take screenshot on test failure

Examples:
  node selenese-runner.js http://localhost:8080/app tests/suite.html
  node selenese-runner.js -d --headed http://localhost:8080 tests/mytest.rctest.html
  node selenese-runner.js -j results.xml http://localhost:8080 tests/suite.html`);
        process.exit(1);
    }

    const runner = new SeleneseRunner(config);
    let allPassed = true;
    let suiteResult = null;

    try {
        await runner.initialize();
        for (const file of testFiles) {
            const fullPath = path.resolve(file);
            if (file.endsWith('.html') && !file.includes('.rctest.')) {
                suiteResult = await runner.runSuite(fullPath);
                if (suiteResult.failed > 0) allPassed = false;
            } else {
                if (!(await runner.runTest(fullPath)).passed) allPassed = false;
            }
        }
        if (junitFile && suiteResult) {
            fs.writeFileSync(junitFile, runner.generateJUnitXML(suiteResult));
            if (!config.quiet) console.log(`\nJUnit XML report written to: ${junitFile}`);
        }
    } catch (err) {
        console.error(`Fatal error: ${err.message}`);
        allPassed = false;
    } finally {
        await runner.close();
    }
    process.exit(allPassed ? 0 : 1);
}

module.exports = { SeleneseRunner };
if (require.main === module) main().catch(err => { console.error(err); process.exit(1); });
