# Internationalization and Localization

[← Back to API Index](../reference.md)

---

## KB Topic: Internationalization and Localization

### Description
**Internationalizing SmartClient Applications**

Internationalizing a SmartClient application involves creating locale-specific versions of all strings, images, and possibly colors. In some cases, component layout may be affected as well (for example date field order differs between some locales).

**Character Encodings**

In order to deliver content in different languages to the browser and have it be displayed correctly, it is important that the browser and server agree on the character encoding being used to send the page.

Generally speaking, you can use whatever character encoding you prefer, so long as you're consistent about serving your files to the browser with exactly the same encoding as was used to save the file to disk. Keep in mind though that the character encoding you choose must be supported by all browsers you intend to support. Isomorphic recommends that, where possible, you use the UTF-8 encoding. Regardless of which character encoding you choose, keep in mind that for the browser to correctly display non-ASCII characters, you **must** explicitly set the character encoding of your page to match the encoding you used to save the file. Browsers have built-in heuristics to guess the character encoding, but they can't be relied upon to work 100% of the time.

There are two ways to explicitly tell the browser what character encoding is being used for a given page. One is to set the "Content-Type" HTTP header, for example:

```
 Content-Type: text/html; charset=UTF-8
 
```
If you're using JSP on the back-end, you can set this header as follows:
```
 <%@ page contentType="text/html; charset=UTF-8"%>
 
```
Or using Java Servlet APIs in a Servlet as follows:
```
 response.setContentType("text/html; charset=UTF-8");
 
```
Note that the latter needs to be done before any content is written to the response output stream.

The other approach to setting the content encoding is to use an HTML META tag to embed it in the page itself (note that this applies to HTML documents only). The META tag must go inside the `<HEAD>` HTML element - e.g. as follows:

```
 <HEAD>
 <META HTTP-EQUIV="Content-Type" CONTENT="text/html; charset=UTF-8">
 </HEAD>
 
```
In addition, for a .html or other static (non-.jsp) file, you must also ensure that the file itself is saved in UTF-8 encoding. Advanced text editors (including the Eclipse IDE) can show you the encoding of a text file and allow you to change it. In Eclipse this is in the Preferences area (search for "Encoding") and UTF-8 is **not the default** for .html files, and should be changed.

Isomorphic recommends that you follow **all** of the above approaches for maximum compatibility. For example, if you omit the META tag approach, pages saved by the user using the browser save function may not render correctly when subsequently read from disk because HTTP headers are not available in that context. Conversely, not providing the HTTP header or not saving the file in UTF-8 can result in application servers delivering the file in the encoding in which it is saved, or in their own default and incorrect content encoding.

If you're using a given character encoding pervasively in your pages, you can also configure your web server or application server to use that character encoding as the default for all pages of a given mime type or some other criteria (depending on the capability of your server) - for example on a per-directory basis.

For more information on character encodings, character sets, and working with HTML, please see W3C's tutorial here: [http://www.w3.org/International/tutorials/tutorial-char-enc/](http://www.w3.org/International/tutorials/tutorial-char-enc/)

**NOTE: Default Encoding**

As mentioned above, Isomorphic recommends the use of UTF-8 encoding. However, it is not possible to configure some servlet engines to do this by default. Both Tomcat and Jetty default to the ISO-8859-1 character set (commonly known as Latin-1), even if the Java VM specifies a different encoding (the Tomcat documentation claims that this behavior is a requirement of the Servlet specification).

To work around this, we change charset encoding to UTF-8 on every request/response that goes through two core Isomorphic servlets: `IDACall` and `DataSourceLoader`. If you need to change this, you can do so by adding `init-param` settings to your `web.xml` file. Please see the Javadocs for those two servlets for examples of how to do this.

**Framework Message Localization**

SmartClient components include standard prompts and error messages in various cases, and all such messages support localization.

Language packs are included in the directory smartclientSDK/isomorphic/locales. To load a language pack, simply include the desired locale via a standard HTML `<script src=>` tag, for example:

```
    <SCRIPT SRC="isomorphic/locales/frameworkMessages_pt.properties"></SCRIPT>
 
```
.. to load the Portuguese locale.

Alternatively can use the `locale` attribute on the `loadISC` or `loadModules` jsp tags:

```
   <isomorphic:loadISC skin="Enterprise" locale="pt"/> 
 
```
Or, if you have the [Network Performance](networkPerformance.md#kb-topic-network-performance), you can use the [FileLoader.loadLocale](../classes/FileLoader.md#classmethod-fileloaderloadlocale)/[FileLoader.cacheLocale](../classes/FileLoader.md#classmethod-fileloadercachelocale) APIs:
```
   isc.FileLoader.loadLocale("pt");
 
```

To see various locales, have a look at the *Localization example*.

If you find that the language pack you are using has any incorrect or missing translations, or you want to add a new language pack, please visit [this forums thread](http://forums.smartclient.com/showthread.php?t=9987) for instructions on how to contribute translations so that they will be added to future SmartClient builds.

You can alternatively maintain your own private additions or overrides to the default language packs. A complete list of standard messages is presented [here](i18nMessages.md#kb-topic-i18n-messages).  
To customize these messages, use [Class.addProperties](../classes/Class.md#method-classaddproperties) or [Class.addClassProperties](../classes/Class.md#classmethod-classaddclassproperties) to override default values at the instance or class level respectively. For example, the class property [RPCManager.timeoutErrorMessage](../classes/RPCManager.md#classattr-rpcmanagertimeouterrormessage) is displayed to the user in a warning dialog when a SmartClient operation fails to complete. This can be modified using the [Class.addClassProperties](../classes/Class.md#classmethod-classaddclassproperties) method, as follows:

```
 isc.RPCManager.addClassProperties({timeoutErrorMessage:"Custom Timeout Message"});
 
```

**Application Message Localization**

Text such as button titles typically appears in the middle of JavaScript code:

```
     isc.Button.create({ title: "Click me!"});
 
```
If you are on the Java platform and your JavaScript code appears in a .jsp, you can use the JSTL Core package (Java Server Pages Standard Tag Library) to replace that static "Click me!" text with a localized reference. This can be done as follows:
```
     <%@ taglib prefix="fmt" uri="/WEB-INF/fmt.tld" %>
     <SCRIPT>
     isc.Button.create({ title: <fmt:message key="buttonTitle"> });
     </SCRIPT>
 
```
Assuming you've set the current Locale and provided a resource bundle with the relevant translated text, the above would create a button with a title that comes from the locale-specific bundle you provided. For more information, see Sun's i18n tutorial: [http://java.sun.com/docs/books/tutorial/i18n/](http://java.sun.com/docs/books/tutorial/i18n/).

Note that this approach does not imply that your JavaScript must appear in the midst of an HTML page. A .jsp file can be used to return a pure JavaScript response via just removing the `<SCRIPT>` tags in the example above, and targetting the .jsp with a `<SCRIPT SRC=>` tag from another .jsp or .html file.

A second common technique is to load internationalized messages as a JavaScript Object mapping from a message name to the internationalized message. For example, if your messages are stored in Java .properties format, you could load your .properties file as a standard Java [Properties object](http://docs.oracle.com/javase/6/docs/api/java/util/Properties.html), and use the server-side com.isomorphic.js.JSTranslater.toJSVariable() API to output the properties as a JavaScript variable.

If the variable name chosen was 'i18nMessages', code to make use of the .properties data would look like:

```
     isc.Button.create({ title: i18nMessages["buttonTitle"] });
 
```

Approaches for non-Java platforms are similar, and generally any approach that works for HTML or JavaScript files works with SmartClient. See the end of this section for pointers to additional resources.

**Localizing Server-side Error Messages**

If you are providing error messages from server-side code, use standard Java techniques to determine the locale (`servletRequest.getLocale()`) and load ResourceBundles. Note that if you are using [DMI validation](../classes/Validator.md#attr-validatorserverobject), the HttpServletRequest is available via the standard [DMI](dmiOverview.md#kb-topic-direct-method-invocation) approach: just declare it as an additional parameter, and it is provided.

**Support for Right-to-Left (RTL) languages**

SmartClient includes support for RTL languages. To enable, simply set `dir="rtl"` on the HTML element:

```
    <HTML dir="rtl">
 
```
ListGrid columns, horizontal layouts, scrolling directions, etc will reverse order automatically.

Because RTL in some cases requires additional media files, RTL is only supported for the Enterprise, EnterpriseBlue and Graphite skins.

**DataSource and Component XML localization**

Please see the separate article on [DataSource and Component XML Localization](dataSourceLocalization.md#kb-topic-datasource-and-component-xml-localization)

**Localized Number and Currency Formatting**

Please see the separate article on [Localized Number Formatting](../reference.md#kb-topic-localized-number-formatting)

**Image, CSS localization**

Most SmartClient components use a mixture of text, CSS and images to render. If you wish to provide locale-specific images or use locale-specific CSS, you can create localized copies of a SmartClient skin named after the locale and modify images as you see fit. A skin is specified at load time using either the `skin` attribute of the `<isomorphic:loadISC>` tag or by loading the skin directly using a script tag like so: `<SCRIPT SRC=/isomorphic/skins/\[skin\]/load\_skin.js>`. If you're using the `loadISC` tag, you can specify a locale-specific skin like so:

```
 <isomorphic:loadISC skin="[localeName]"/>
 
```
Or you can use a script tag like so:
```
 <SCRIPT SRC=/isomorphic/skins/[localeName]/load_skin.js></SCRIPT>
 
```
Where the `[localeName]` is the name of the current locale - you'll have this in the current page context as you'll need it to use the JSTL <fmt:setLocale> tag.

Resources:

**Java**

*   Sun's i18n tutorial: [http://java.sun.com/docs/books/tutorial/i18n/](http://java.sun.com/docs/books/tutorial/i18n/)
*   JSTL documentation home page: [https://www.oracle.com/java/technologies/jstl-documentation.html](https://www.oracle.com/java/technologies/jstl-documentation.html)
*   Apache taglibs: [https://tomcat.apache.org/taglibs/index.html](https://tomcat.apache.org/taglibs/index.html)

**.NET**

*   MSDN Developer Resource: "Developing World-Ready Applications": [https://docs.microsoft.com/en-us/dotnet/core/extensions/best-practices-for-developing-world-ready-apps](https://docs.microsoft.com/en-us/dotnet/core/extensions/best-practices-for-developing-world-ready-apps)

**PHP**

*   PHP Gettext manual: [http://us2.php.net/gettext](http://us2.php.net/gettext)

### Related

- [Comm.concurrentXHRErrorMessage](../classes/Comm.md#classattr-commconcurrentxhrerrormessage)

---
