// --- TEST vs REAL ------------------------------------------------------------
let TEST_MODE = true;
let MCP_SESSION_ID = null;   // captured from initialize
let MCP_BEARER_TOKEN = null; // set from UI (GitHub token, etc.)

const TEST_FILES = {
  "initialize": "[ISOMORPHIC]/system/reference/inlineExamples/serverExamples/mcp/initialize_fetch_rest.js",
  "tools/list": "[ISOMORPHIC]/system/reference/inlineExamples/serverExamples/mcp/tools_list_fetch_rest.js",
  "tools/call": "[ISOMORPHIC]/system/reference/inlineExamples/serverExamples/mcp/tools_call_fetch_rest.js",
};

// Build proxy URL once
function getProxyURL() {
  // canonical path in SDKPackage
  return "http://localhost:47011/isomorphic/HttpProxy";
}

// Case-insensitive header getter from SmartClient resp
function getHeader(resp, name) {
  const headers = resp.httpHeaders || {};
  const target = String(name).toLowerCase();
  for (var k in headers) if (headers.hasOwnProperty(k)) {
    if (String(k).toLowerCase() === target) return headers[k];
  }
  return null;
}

function buildRealHeaders() {
  const headers = { "Accept": "application/json" };
  if (MCP_BEARER_TOKEN) headers["Authorization"] = "Bearer " + MCP_BEARER_TOKEN;
  if (MCP_SESSION_ID)   headers["Mcp-Session-Id"] = MCP_SESSION_ID;
  return headers;
}

// IMPORTANT: send everything as STRINGS to the proxy to avoid NPEs
function proxyFormPayload(targetURL, httpMethod, bodyObj) {
  const headersJSON = JSON.stringify(buildRealHeaders());  // string
  const dataJSON    = bodyObj ? JSON.stringify(bodyObj) : ""; // string (may be empty)

  // These names match what HttpProxyServlet expects
  // (actionURL, httpMethod, httpHeaders, data) as form fields
  return {
    actionURL:  targetURL,
    httpMethod: httpMethod,
    httpHeaders: headersJSON,
    data: dataJSON
  };
}

function encodeForm(form) {
  const parts = [];
  for (const [k, v] of Object.entries(form)) {
    parts.push(
      encodeURIComponent(k) + "=" + encodeURIComponent(v == null ? "" : String(v))
    );
  }
  return parts.join("&");
}

/** JSON-RPC:
 *  TEST_MODE -> GET the local static json
 *  REAL_MODE -> POST directly to the external URL (SmartClient auto-proxies via HttpProxy)
 */
function rpcJson(mcpUrl, method, params = {}) {
  return new Promise((resolve) => {
    const isTest = TEST_MODE === true;

    if (isTest) {
      const actionURL = TEST_FILES[method] || TEST_FILES["initialize"];
      isc.RPCManager.sendRequest({
        actionURL,
        httpMethod: "GET",
        callback(resp) {
          let data = resp.data;
          if (typeof data === "string") { try { data = JSON.parse(data); } catch {} }
          resolve({ ok: resp.status === 0, data, code: resp.httpResponseCode || 200 });
        }
      });
      return;
    }

    const isInit = method === "initialize";
    const payload = {
      jsonrpc: "2.0",
      id: Date.now(),
      method,
      params: isInit || !MCP_SESSION_ID ? params : { ...params, session: { id: MCP_SESSION_ID } }
    };

    isc.RPCManager.sendRequest({
      actionURL: mcpUrl,                 // call the external URL directly
      httpMethod: "POST",
      contentType: "application/json",   // send JSON body
      data: JSON.stringify(payload),
      httpHeaders: buildRealHeaders(),   // custom headers (Authorization, Mcp-Session-Id)
      useHttpProxy: true,                // let SmartClient route cross-origin via HttpProxy
      willHandleError: true,
      callback(resp) {
        // Try to capture session id from headers on initialize
        if (isInit) {
          const sid = getHeader(resp, "mcp-session-id");
          if (sid) MCP_SESSION_ID = sid;
        }

        let data = resp.data;
        if (typeof data === "string") { try { data = JSON.parse(data); } catch {} }

        // Fallback: some servers echo session in body
        if (isInit && !MCP_SESSION_ID) {
          const r = data?.result || data;
          const inBody = r?.sessionId || r?.session?.id;
          if (inBody) MCP_SESSION_ID = inBody;
        }

        const ok = resp.status === 0 &&
                   (resp.httpResponseCode == null ||
                    (resp.httpResponseCode >= 200 && resp.httpResponseCode < 300));

        if (!ok) {
          const code = resp.httpResponseCode || 500;
          const msg = (typeof data === "string" && data) ||
                      data?.error?.message ||
                      "HTTP error";
          resolve({ ok: false, code, data: { error: msg } });
          return;
        }
        resolve({ ok: true, code: resp.httpResponseCode || 200, data });
      }
    });
  });
}

// High-level helpers ----------------------------------------------------------

async function loadTools(mcpUrl) {
  const init = await rpcJson(mcpUrl, "initialize", {
    protocol: "2024-10-07",
    client: { name: "mcp-http-smartclient", version: "0.1.0" }
  });
  if (!init.ok) throw new Error("Initialize failed: " + (init.data?.error || init.code));

  const res = await rpcJson(mcpUrl, "tools/list");
  if (!res.ok) throw new Error("Failed listing tools: " + (res.data?.error || res.code));
  return (res.data?.result?.tools) || res.data?.tools || [];
}

async function callTool(mcpUrl, name, args = {}) {
  const res = await rpcJson(mcpUrl, "tools/call", { name, arguments: args });
  if (!res.ok) throw new Error("Tool call failed: " + (res.data?.error || res.code));
  return res.data;
}

// Schema → SmartClient fields
function makeFieldsFromSchema(tool) {
  const props = tool?.inputSchema?.properties || {};
  return Object.entries(props).map(([name, prop]) => {
    let type = "text";
    if (prop.type === "integer" || prop.type === "number") type = "integer";
    if (prop.type === "boolean") type = "boolean";
    return { name, type, title: prop.title || name };
  });
}

// UI builders (unchanged except we set token + mode) --------------------------

function makeServiceTree(mcpUrl, tools) {
  const nodes = [];
  tools.forEach((t, i) => {
    const toolId = t.name || `tool_${i}`;
    const schemaNode = `${toolId}.schema`;

    nodes.push({ id: toolId, parentId: "root", name: toolId, isFolder: true });
    nodes.push({ id: schemaNode, parentId: toolId, name: "schema", isFolder: true });

    const fields = makeFieldsFromSchema(t);
    fields.forEach(f => {
      nodes.push({ id: `${schemaNode}.${f.name}`, parentId: schemaNode, name: `${f.name}: ${f.type}`, isFolder: false });
    });

    nodes.push({ id: `${toolId}.call`, parentId: toolId, name: "Call", isFolder: false, _toolName: toolId, _fields: fields });
  });

  const tree = isc.Tree.create({ modelType: "parent", idField: "id", parentIdField: "parentId", rootValue: "root", data: nodes });

  return isc.TreeGrid.create({
    ID: "mcpServiceTree",
    width: "100%",
    height: 240,
    data: tree,
    showHeader: true,
    fields: [{ name: "name", title: "Services" }],
    nodeClick(_viewer, node) { if (node.name === "Call") openCallDialog(mcpUrl, node._toolName, node._fields); }
  });
}

function openCallDialog(mcpUrl, toolName, fields) {
  const df = isc.DynamicForm.create({
    width: "100%", numCols: 2, colWidths: [240, "*"], padding: 10,
    fields: fields.length ? fields : [{ name: "arg", title: "arg", type: "text" }]
  });

  const runBtn = isc.IButton.create({
    ID: "runBtn", title: "Run", layoutAlign: "center", autoFit: true, minWidth: 200,
    click: async () => {
      try {
        const args = df.getValues();
        const result = await callTool(mcpUrl, toolName, args);
        context_answer_textarea.setValue("context_http_answer", JSON.stringify(result, null, 2));
        isc.say("Tool executed!");
        win.destroy();
      } catch (e) { isc.warn("Call failed: " + e.message); }
    }
  });

  const contentWindow = isc.VLayout.create({ ID: "contentWindow", width: "100%", padding: 10, layoutAlign: "center", members: [df, runBtn] });

  const win = isc.Window.create({
    title: `Run ${toolName}`, autoCenter: true, autoSize: true, maxWidth: 420, maxHeight: 600,
    showMinimizeButton: false, showMaximizeButton: false, items: [contentWindow]
  });
}

// Top bar with URL / token / TEST_MODE
isc.DynamicForm.create({
  ID: "findForm",
  width: "80%",
  numCols: 6,
  colWidths: [120, "*", 120, "*", 120, "*"],
  fields: [
    { name: "url", title: "MCP URL", type: "text", colSpan: 3, value: "https://api.githubcopilot.com/mcp/", width: "100%" },
    { name: "token", title: "Bearer Token", type: "text", value:"", colSpan: 3, width: "100%" },
    { name: "testMode", title: "Test Mode", type: "checkbox", value: true, colSpan: 1 }
  ],
  async loadServices() {
    const vals = this.getValues();
    TEST_MODE = !!vals.testMode;
    MCP_BEARER_TOKEN = vals.token || null;
    MCP_SESSION_ID = null; // reset when reloading

    try {
      const tools = await loadTools(vals.url);
      const tree = makeServiceTree(vals.url, tools);
      servicePane.setMembers([tree]);
      context_answer_textarea.setValue("context_http_answer", JSON.stringify({ tools }, null, 2));
      isc.say("Services loaded!");
    } catch (e) {
      isc.warn("Failed to load services: " + e.message);
    }
  }
});

isc.IButton.create({ ID: "loadBtn", title: "Load Services", width: 140, click: "findForm.loadServices();" });

const findBar = isc.HLayout.create({ width: "100%", height: "10%", membersMargin: 8, members: [findForm, loadBtn] });
const servicePane = isc.VLayout.create({ width: "100%", height: "50%" });

isc.DynamicForm.create({
  ID: "context_answer_textarea",
  width: "100%", height: "40%", titleWidth: 100,
  fields: [{ name: "context_http_answer", title: "Response", type: "textArea", width: "100%", height: "100%" }]
});

isc.VLayout.create({
  ID: "mainLayout",
  width: "100%", height: "100%", membersMargin: 10,
  members: [findBar, servicePane, context_answer_textarea],
});
