const iso = isc.Page.getIsomorphicDir();

/* ---------- tiny utils ---------- */
const esc = (s) => (s ?? "").toString()
  .replace(/&/g,"&amp;").replace(/</g,"&lt;").replace(/>/g,"&gt;");
const escJS = (s) => (s ?? "").toString()
  .replace(/\\/g,"\\\\").replace(/'/g,"\\'");

function refDocsPath(testMode, { relative = false } = {}) {
  const base = relative ? "isomorphic/system/reference/" : iso + "system/reference/";
  return base + (testMode ? "referenceDocs/referenceDocs.simplified.js" : "referenceDocs.js");
}


/* ---------- loader helpers ---------- */
function loadOnce(url, checkFn, next) {
  try { if (checkFn && checkFn()) return next && next(); } catch (e) {}
  isc.FileLoader.loadJSFiles([url], () => next && next());
}

function series(tasks, done) {
  const t = tasks.slice();
  (function step() {
    const fn = t.shift();
    if (!fn) return done && done();
    fn(step);
  })();
}

function ensureReferenceLibs(then) {
  series([
    (cb) => loadOnce(iso + "system/reference/JSDoc.js", () => !!isc.jsdoc, cb)
  ], then);
}

function clearJsDocState() {
  try {
    if (isc.jsdoc) {
      if (isc.jsdoc.clearData) isc.jsdoc.clearData();
      if (isc.jsdoc._data !== undefined) isc.jsdoc._data = null;
    }
  } catch (e) {}
  isc.docItems = null;
}


function initJsDocIfNeeded() {
  if (!isc.jsdoc) return;
  const has = (typeof isc.jsdoc.hasData === "function") ? isc.jsdoc.hasData() : false;
  if (!has && isc.docItems) {
    isc.jsdoc.init(isc.docItems);
    return;
  }
  // force re-init path
  if (has && isc.jsdoc.clearData && isc.docItems) {
    isc.jsdoc.clearData();
    isc.jsdoc.init(isc.docItems);
  }
}

function loadReferenceDocs(testMode, then) {
  const path = refDocsPath(testMode, { relative: false });
  clearJsDocState();
  isc.FileLoader.loadJSFiles([path], () => { initJsDocIfNeeded(); then && then(); });
}

function withDocsLoaded(testMode, cb) {
  ensureReferenceLibs(() => loadReferenceDocs(testMode, () => cb && cb()));
}

/* ---------- orchestrator ---------- */
function reloadDocsAndGrid(testMode) {
  withDocsLoaded(testMode, () => buildOrWireGrid(testMode));
}

/* ---------- initial draw + layout ---------- */

isc.HLayout.create({
  ID: "smartClientContent",
  width: "100%", height: "100%", membersMargin: 10, autoDraw: true
});

reloadDocsAndGrid(false);

isc.HTMLFlow.create({
  ID: "htmlContent",
  width: "50%", height: "100%", overflow: "auto", padding: 10,
  contents: "<i>Select an item to view documentation…</i>"
});

isc.DynamicForm.create({
  ID: "controlsForm",
  fields: [{
    name: "testMode", type: "checkbox", title: "Test Mode", width:"100%",
    changed(form, item, value) {
      reloadDocsAndGrid(!!value);
    }
  }]
});

// factory to (re)create the grid wired to SCReference
function buildOrWireGrid(testMode) {
  const setup = function(ds) {
    if (!window.refGrid) {
      isc.ListGrid.create({
        ID: "refGrid",
        width: "100%", height: "100%",
        dataSource: ds,
        autoFetchData: false,
        showFilterEditor: true,
        filterOnKeypress: true,
        fields: [
          { name: "name", title: "Name", width: 250 },
          { name: "kind", title: "Kind", width: 120 },
          { name: "ref",  title: "Ref",  width: "*" }
        ],
        selectionChanged(rec, state){ if (state) showDocForRecord(rec); }
      });

      smartClientContent.addMembers([
        isc.VLayout.create({
          width: "50%", height: "100%", membersMargin: 6,
          members: [controlsForm, refGrid]
        }),
        htmlContent
      ]);
    } else {
      refGrid.setData([]);
    }

    const docsPath = refDocsPath(testMode,{relative:true});
    refGrid.fetchData(
      null,                          // criteria
      null,                          // callback
      { params: { docsPath } }       // requestProperties -> dsRequest.params
    );
  };

  const dsMaybe = isc.DataSource.get("SCReference");
  if (dsMaybe && dsMaybe.fetchData) return setup(dsMaybe);

  isc.DataSource.load("SCReference", function(dsLoaded) {
    const ds = dsLoaded && dsLoaded.fetchData ? dsLoaded : isc.DataSource.get("SCReference");
    setup(ds);
  });
}

function showDocForRecord(rec){
  if (!rec){ htmlContent.setContents("<i>No selection</i>"); return; }
  if (!isc.jsdoc || !isc.jsdoc.hasData()){
    htmlContent.setContents("<b>JSDoc not available</b>"); return;
  }
  const ref = rec.ref || computeRef(rec);
  if (!ref){
    htmlContent.setContents("<b>No ref</b><br><pre style='white-space:pre-wrap'>" + isc.echoAll(rec) + "</pre>");
    return;
  }
  const item = isc.jsdoc.getDocItem(ref);
  const data = item ? isc.jsdoc.toJS(item) : null;
  const html = data ? simpleFormatDoc(data) : null;
  htmlContent.setContents(
    html ? `<div style="margin-bottom:8px;opacity:.8">${ref}</div>${html}`
        : `<b>Docs not found for:</b> ${ref}`
  );
}

function computeRef(rec){
  const {kind, name} = rec || {};
  if (!kind || !name) return null;
  function ownerFrom(r){
    let s = (r.id || r.parentId || r.fqName || "");
    const h = s.indexOf("#"); if (h>0) s = s.substring(0,h);
    const sl = s.indexOf("/"); if (sl>0) s = s.substring(0,sl);
    const c = s.indexOf(":");  if (c>0) return { ownerKind:s.substring(0,c), ownerName:s.substring(c+1) };
    return {};
  }
  if (["attr","method","classMethod","staticMethod","classAttr"].includes(kind)){
    const {ownerName} = ownerFrom(rec);
    if (!ownerName) return null;
    const docType = (kind==="classMethod"||kind==="staticMethod") ? "classMethod"
                  : (kind==="classAttr") ? "classAttr" : kind;
    return `${docType}:${ownerName}.${name}`;
  }
  if (["class","object","interface","type","group"].includes(kind)){
    return `${kind}:${name}`;
  }
  return null;
}

/* ---------- inline links / doc formatting ---------- */
function labelForRef(ref) {
  if (!ref) return "";
  const [, rest] = String(ref).split(":"); // after the first colon
  return rest || ref;
}

const extractRef = (s) => {
  if (!s) return null;
  const parts = String(s).split(":");
  return parts.length >= 2 ? (parts[1] || null) : null;
};

function renderInlineDocButton(ref, labelOverride) {
  const safeRef = String(ref).replace(/\\/g,"\\\\").replace(/'/g,"\\'");
  const label = labelOverride != null && labelOverride !== "" ? labelOverride : labelForRef(ref);
  return `
    <button type="button" class="doclink"
      title="Open docs for ${label}"
      onclick="showDocForRecord({ ref: '${safeRef}' })">${label}</button>`;
}

function expandInlineDocLinks(str) {
  if (!str) return str;
  return String(str).replace(
    /\$\{isc\.DocUtils\.linkForRef\(\s*(['"])([^'"]+)\1\s*(?:,\s*(['"])([^'"]+)\3\s*)?\)\}/g,
    (_m, _q1, ref, _q2, label) => renderInlineDocButton(ref, label)
  );
}

const renderRefSection = (label, items) => {
  if (!Array.isArray(items) || items.length === 0) return "";
  const lis = items.map(item => {
    const ref = extractRef(item);
    if (ref) {
      const safeRefForJS = escJS(ref);
      const safeText = esc(item);
      return `
        <li>
          <button
            type="button"
            class="doclink"
            title="Open docs for ${esc(ref)}"
            onclick="showDocForRecord({ ref: '${safeRefForJS}' })"
          >${safeText}</button>
        </li>`;
    }
    return `<li>${esc(item)}</li>`;
  }).join("");

  return `
    <div style="margin-top:10px">
      <div style="font-weight:600">${esc(label)}</div>
      <ul style="margin:6px 0 0 18px">${lis}</ul>
    </div>`;
};

function simpleFormatDoc(d) {

  const definingClassLink = d.definingClass ? (() => {
    const ref = String(d.definingClass);
    const safeRefForJS = escJS(ref);
    const safeLabel = esc(ref);
    return `
      <div style="margin-bottom:6px">
        <button type="button" class="doclink"
          title="Open docs for ${safeLabel}"
          onclick="showDocForRecord({ ref: '${safeRefForJS}' })">${safeLabel}</button>
      </div>`;
  })() : "";

  const header = (() => {
    if (d.type === "method" || d.type === "classMethod" || d.type === "staticMethod") {
      const scope = (d.type === "classMethod" || d.type === "staticMethod") ? "static " : "";
      const cls = (d.definingClass || "").replace(/^class:/,"");
      const params = (d.params || []).map(p => p.optional ? `[${p.name}]` : p.name).join(", ");
      return `<div style="font-weight:600">${esc(scope)}function ${esc(cls)}.${esc(d.name)}(${esc(params)})</div>`;
    }
    if (d.type === "attr" || d.type === "classAttr") {
      const scope = d.type === "classAttr" ? "static " : "";
      const cls = (d.definingClass || "").replace(/^class:/,"");
      const vt  = d.valueType ? `: <code>${esc(d.valueType)}</code>` : "";
      return `<div style="font-weight:600">${esc(scope)}${esc(cls)}.${esc(d.name)}${vt}</div>`;
    }
    return `<div style="font-weight:700">${esc(d.type)}: ${esc(d.name || d.ref || "")}</div>`;
  })();

  const rawDesc = d.description
    ? (isc.DocUtils ? isc.DocUtils.stripHTML(d.description) : String(d.description))
    : "";

  const desc = rawDesc ? `<div style="margin-top:6px">${expandInlineDocLinks(rawDesc)}</div>` : "";

  const params = Array.isArray(d.params) && d.params.length
    ? `<div style="margin-top:10px">
         <div style="font-weight:600">Parameters</div>
         <ul style="margin:6px 0 0 18px">
           ${d.params.map(p => `
             <li><code>${esc(p.name)}</code>${p.optional ? " <i>(optional)</i>" : ""}${p.type ? `: <code>${esc(p.type)}</code>` : ""}${p.description ? " — " + esc(p.description) : ""}</li>
           `).join("")}
         </ul>
       </div>`
    : "";

  const classAttrs     = renderRefSection("Class Attributes", d.classAttrs);
  const classMethods   = renderRefSection("Class Methods",    d.classMethods);
  const methods        = renderRefSection("Methods",          d.methods);
  const attrs          = renderRefSection("Attributes",       d.attrs);
  const staticMethods  = renderRefSection("Static Methods",   d.staticMethods);

  const returns = d.returns
    ? `<div style="margin-top:10px">
         <div style="font-weight:600">Returns</div>
         <div>${d.returns.type ? `<code>${esc(d.returns.type)}</code>` : ""}${d.returns.description ? " — " + esc(d.returns.description) : ""}</div>
       </div>`
    : "";

  const flags = d.flags
    ? `<div style="margin-top:10px;opacity:.8">Flags: <code>${esc(d.flags)}</code></div>`
    : "";

  const localCSS = `
    <style>
      .doclink {
        background: none; border: none; padding: 0;
        font: inherit; color: #06c; cursor: pointer; text-decoration: none;
      }
      .doclink:hover { text-decoration: underline; }
      .doclink:focus { outline: 1px dotted; outline-offset: 2px; }
    </style>`;

  return `<div style="font:13px/1.5 -apple-system,BlinkMacSystemFont,Segoe UI,Roboto,Arial">
    ${localCSS}
    ${definingClassLink}
    ${header}
    ${desc}
    ${params}
    ${classAttrs}
    ${classMethods}
    ${methods}
    ${attrs}
    ${returns}
    ${flags}
  </div>`;
}

window.showDocForRecord = showDocForRecord;
window.computeRef = computeRef;

