// Demonstrates precise geometric hit detection via programmatic line intersection.
// Click anywhere to draw a line from the center - shapes the line passes through
// are highlighted. This uses isPointInPath() at points along the line, NOT mouse events,
// so it requires precise geometric hit detection rather than bounding-box detection.

var mainPane = isc.DrawPane.create({
    autoDraw: false,
    ID: "mainPane",
    width: 600,
    height: 500,
    showEdges: true,
    drawingType: "svg",
    cursor: "crosshair",
    canHover: true,
    prompt: "Click somewhere to see intersections"
});

// Center point coordinates
var centerX = 300,
    centerY = 250;

// Shapes that benefit from precise hit detection - these have significant
// empty space within their bounding boxes where bounding-box detection
// would incorrectly report hits.

// Large arc flipped - opens downward with empty space above
var arc1 = isc.DrawCurve.create({
    autoDraw: true,
    drawPane: mainPane,
    shapeName: "Blue Arc",
    startPoint: [80, 350],
    endPoint: [200, 350],
    controlPoint1: [80, 480],
    controlPoint2: [200, 480],
    lineColor: "#3498DB",
    lineWidth: 6,
    lineCap: "round"
});

// Arc on the right side flipped - curves left instead of right
var arc2 = isc.DrawCurve.create({
    autoDraw: true, 
    drawPane: mainPane,
    shapeName: "Green Arc",
    startPoint: [550, 100],
    endPoint: [450, 250],
    controlPoint1: [450, 100],
    controlPoint2: [450, 175],
    lineColor: "#27AE60",
    lineWidth: 6,
    lineCap: "round"
});

// Curve rotated so tail points toward bottom center of drawpane
var sCurve = isc.DrawCurve.create({
    autoDraw: true,
    drawPane: mainPane,
    shapeName: "Red Curve",
    startPoint: [241, 30],
    endPoint: [238, 252],
    controlPoint1: [136, 55],
    controlPoint2: [234, 194],
    lineColor: "#E74C3C",
    lineWidth: 6,
    lineCap: "round"
});

// Oval - corners of bounding box are empty
var oval1 = isc.DrawOval.create({
    autoDraw: true,
    drawPane: mainPane,
    shapeName: "Purple Oval",
    left: 420,
    top: 320,
    width: 140,
    height: 80,
    lineColor: "#8E44AD",
    lineWidth: 4,
    fillColor: null  // No fill - only the outline should be detected
});

// Triangle in middle-left empty space
var triangle1 = isc.DrawTriangle.create({
    autoDraw: true,
    drawPane: mainPane,
    shapeName: "Orange Triangle",
    points: [[50, 280], [150, 280], [100, 180]],
    lineColor: "#E67E22",
    lineWidth: 4,
    fillColor: null  // No fill - only the outline should be detected
});

// Another curve at the bottom
var arc3 = isc.DrawCurve.create({
    autoDraw: true,
    drawPane: mainPane,
    shapeName: "Cyan Curve",
    startPoint: [350, 450],
    endPoint: [550, 450],
    controlPoint1: [400, 350],
    controlPoint2: [500, 350],
    lineColor: "#00BFFF",
    lineWidth: 6,
    lineCap: "round"
});

var allShapes = [arc1, arc2, sCurve, oval1, triangle1, arc3];
var originalStyles = {};
allShapes.forEach(function(shape) {
    originalStyles[shape.shapeName] = {
        lineColor: shape.lineColor,
        lineWidth: shape.lineWidth
    };
});

// Toggle for precise vs bounding-box detection
var usePreciseDetection = true;

// Bounding-box implementation of isPointInPath - the workaround before precise
// detection existed. Same method name as DrawItem.isPointInPath, but uses
// bounding-box instead of precise geometry. Works, but produces false positives.
function isPointInPath(shape, x, y) {
    return shape.isInBounds(x, y);
}

// Draw center point marker - black with white ring for visibility
var centerMarker = isc.DrawOval.create({
    autoDraw: true,
    drawPane: mainPane,
    left: centerX - 10,
    top: centerY - 10,
    width: 20,
    height: 20,
    fillColor: "#000000",
    lineColor: "#FFFFFF",
    lineWidth: 3
});

// Label for center
isc.DrawLabel.create({
    autoDraw: true,
    drawPane: mainPane,
    left: centerX + 18,
    top: centerY - 8,
    contents: "Center",
    fontSize: 12,
    fontWeight: "bold",
    lineColor: "#000000"
});

// The line from center to click point - bright magenta for visibility
var scanLine = isc.DrawLine.create({
    autoDraw: true,
    drawPane: mainPane,
    startPoint: [centerX, centerY],
    endPoint: [centerX, centerY],
    lineColor: "#FF00FF",
    lineWidth: 3,
    linePattern: "dash"
});

// Reset all shapes to original styles
function resetShapes() {
    allShapes.forEach(function(shape) {
        var style = originalStyles[shape.shapeName];
        shape.setLineColor(style.lineColor);
        shape.setLineWidth(style.lineWidth);
    });
}

// Check what shapes a line from center to (endX, endY) passes through
// endX/endY are in local drawing coordinates
function detectIntersections(endX, endY) {
    resetShapes();

    // Temporarily hide the scanLine so it doesn't block hit detection on shapes
    scanLine.hide();

    var hitShapes = {};
    var numSamples = 100;  // Sample 100 points along the line

    for (var i = 0; i <= numSamples; i++) {
        var t = i / numSamples;
        // Calculate local coordinates along the line
        var localX = centerX + t * (endX - centerX);
        var localY = centerY + t * (endY - centerY);

        // Check each shape for a hit
        for (var j = 0; j < allShapes.length; j++) {
            var shape = allShapes[j];
            if (hitShapes[shape.shapeName]) continue;  // Already found this one

            var hit;
            if (usePreciseDetection) {
                // Use shape's built-in isPointInPath - precise geometric detection
                hit = shape.isPointInPath(localX, localY);
            } else {
                // Use sample's isPointInPath - bounding-box fallback
                hit = isPointInPath(shape, localX, localY);
            }

            if (hit) {
                hitShapes[shape.shapeName] = shape;
            }
        }
    }

    // Show the scanLine again
    scanLine.show();

    // Highlight hit shapes with bright yellow and thicker lines
    var hitNames = [];
    for (var name in hitShapes) {
        var shape = hitShapes[name];
        shape.setLineColor("#FFFF00");  // Bright yellow highlight
        shape.setLineWidth(10);
        hitNames.push(name);
    }

    // Update status
    if (hitNames.length > 0) {
        statusLabel.setContents("<b>Line intersects:</b> " + hitNames.join(", "));
    } else {
        statusLabel.setContents("<b>Line intersects:</b> (no shapes)");
    }
}

// Handle clicks on the pane
mainPane.click = function() {
    var x = mainPane.getOffsetX(),
        y = mainPane.getOffsetY();

    // Disable the one-time hover hint after first click
    if (this.prompt) this.setPrompt(null);

    // Update the scan line and detect intersections
    scanLine.setEndPoint([x, y]);
    detectIntersections(x, y);
};

// Status label
var statusLabel = isc.Label.create({
    autoDraw: false,
    width: 600,
    height: 30,
    contents: "<b>Line intersects:</b> (click anywhere to draw a line from center)",
    padding: 5
});

// Toggle for precise hit detection - allows comparison with bounding-box fallback
var toggleForm = isc.DynamicForm.create({
    autoDraw: false,
    ID: "hitDetectionToggleForm",
    width: 400,
    items: [{
        name: "preciseHitDetection",
        editorType: "ToggleItem",
        title: "Use precise geometric hit detection",
        wrapTitle: false,
        defaultValue: true,
        changed : function (form, item, value) {
            // Toggle between precise detection and bounding-box fallback
            usePreciseDetection = value;
            // Re-run detection with existing line position
            var endPoint = scanLine.endPoint;
            if (endPoint[0] != centerX || endPoint[1] != centerY) {
                // Line has been drawn - replot with new detection mode
                detectIntersections(endPoint[0], endPoint[1]);
            } else {
                resetShapes();
            }
        }
    }]
});

isc.VStack.create({
    width: "100%",
    membersMargin: 10,
    members: [statusLabel, mainPane, toggleForm]
});
