// --- Simple demo: defaults to test mode and minimal logic ---
let TEST_MODE = true;

/** Test endpoints for SmartClient samples (adjust paths if needed) */
const TEST_FILES = {
  "initialize": "[ISOMORPHIC]/system/reference/inlineExamples/serverExamples/mcp/initialize_fetch_rest.js",
  "tools/list": "[ISOMORPHIC]/system/reference/inlineExamples/serverExamples/mcp/tools_list_fetch_rest.js",
  "tools/call": "[ISOMORPHIC]/system/reference/inlineExamples/serverExamples/mcp/tools_call_fetch_rest.js",
};

/** Very small JSON-RPC over SmartClient RPCManager.
 *  In test mode, fetches static JSON files above via GET.
 *  In real mode, sends POST JSON to your MCP server.
 */
function rpcJson(mcpUrl, method, params = {}) {
  return new Promise((resolve) => {
    const isTest = TEST_MODE === true;
    const actionURL = isTest ? (TEST_FILES[method] || TEST_FILES["initialize"]) : mcpUrl;
    const httpMethod = isTest ? "GET" : "POST";

    isc.RPCManager.sendRequest({
      actionURL,
      httpMethod,
      ...(isTest ? {} : {
        data: JSON.stringify({ jsonrpc: "2.0", id: Date.now(), method, params }),
        httpHeaders: {
          "Accept": "application/json",
          "Content-Type": "application/json",
        },
        useHttpProxy: true
      }),
      callback(resp) {
        let data = resp.data;
        if (typeof data === "string") {
          try { data = JSON.parse(data); } catch { /* leave as string */ }
        }
        resolve({ ok: resp.status === 0, data, code: resp.httpResponseCode || 200 });
      }
    });
  });
}

/** get tools list (does a quick initialize first to keep it simple) */
async function loadTools(mcpUrl) {
  await rpcJson(mcpUrl, "initialize");
  const res = await rpcJson(mcpUrl, "tools/list");
  if (!res.ok) throw new Error("Failed listing tools");
  // try both shapes: {result:{tools:[...]}} or {tools:[...]}
  return (res.data?.result?.tools) || res.data?.tools || [];
}

/** Call a tool with arguments */
async function callTool(mcpUrl, name, args = {}) {
  const res = await rpcJson(mcpUrl, "tools/call", { name, arguments: args });
  if (!res.ok) throw new Error("Tool call failed");
  return res.data;
}

/** Build a tiny DS from JSON Schema (name, type, title only). */
function makeFieldsFromSchema(tool) {
  const props = tool?.inputSchema?.properties || {};
  return Object.entries(props).map(([name, prop]) => {
    let type = "text";
    if (prop.type === "integer" || prop.type === "number") type = "integer";
    if (prop.type === "boolean") type = "boolean";
    return { name, type, title: prop.title || name };
  });
}

/** Render a tree of tools with a “Call” action under each tool. */
function makeServiceTree(mcpUrl, tools) {
  const nodes = [];
  tools.forEach((t, i) => {
    const toolId = t.name || `tool_${i}`;
    const schemaNode = `${toolId}.schema`;

    // tool node
    nodes.push({ id: toolId, parentId: "root", name: toolId, isFolder: true });

    // schema node
    nodes.push({ id: schemaNode, parentId: toolId, name: "schema", isFolder: true });

    // fields under schema
    const fields = makeFieldsFromSchema(t);
    fields.forEach(f => {
      nodes.push({
        id: `${schemaNode}.${f.name}`,
        parentId: schemaNode,
        name: `${f.name}: ${f.type}`,
        isFolder: false
      });
    });

    // call node
    nodes.push({
      id: `${toolId}.call`,
      parentId: toolId,
      name: "Call",
      isFolder: false,
      _toolName: toolId,
      _fields: fields
    });
  });

  const tree = isc.Tree.create({
    modelType: "parent",
    idField: "id",
    parentIdField: "parentId",
    rootValue: "root",
    data: nodes
  });

  return isc.TreeGrid.create({
    ID: "mcpServiceTree",
    width: "100%",
    height: 240,
    data: tree,
    showHeader: true,
    fields: [{ name: "name", title: "Services" }],
    nodeClick(_viewer, node) {
      if (node.name === "Call") {
        openCallDialog(mcpUrl, node._toolName, node._fields);
      }
    }
  });
}

/** Simple dialog to collect arguments and call the tool. */
function openCallDialog(mcpUrl, toolName, fields) {
  const df = isc.DynamicForm.create({
    width: "100%",
    numCols: 2,
    colWidths: [240, "*"],
    padding: 10,
    fields: fields.length ? fields : [{ name: "arg", title: "arg", type: "text" }]
  });


  isc.IButton.create({
    ID: "runBtn",
    title: "Run",
    layoutAlign: "center",
    autoFit: true,
    minWidth: 200,
    click: async () => {
      try {
        const args = df.getValues();
        const result = await callTool(mcpUrl, toolName, args);
        context_answer_textarea.setValue(
          "context_http_answer",
          JSON.stringify(result, null, 2)
        );
        isc.say("Tool executed!");
        win.destroy();
      } catch (e) {
        isc.warn("Call failed: " + e.message);
      }
    }
  }),

    // Add some edge padding inside the window
    isc.VLayout.create({
      ID: "contentWindow",
      width: "100%",
      padding: 10,
      layoutAlign: "center",
      members: [df, runBtn]
    });

  const win = isc.Window.create({
    title: `Run ${toolName}`,
    autoCenter: true,
    autoSize: true,
    maxWidth: 420,
    maxHeight: 600,
    showMinimizeButton: false,
    showMaximizeButton: false,
    items: [contentWindow]
  });
}

isc.DynamicForm.create({
  ID: "findForm",
  width: "70%",
  numCols: 4,
  fields: [
    { name: "url", title: "MCP URL", type: "text", colSpan: 4, value: "https://api.githubcopilot.com/mcp/", width: "100%" },
    { name: "testMode", title: "Test Mode", type: "checkbox", value: true, colSpan: 4 }
  ],
  async loadServices() {
    const vals = this.getValues();
    TEST_MODE = !!vals.testMode;
    try {
      const tools = await loadTools(vals.url);
      const tree = makeServiceTree(vals.url, tools);
      servicePane.setMembers([tree]);
      context_answer_textarea.setValue("context_http_answer", JSON.stringify({ tools }, null, 2));
      isc.say("Services loaded!");
    } catch (e) {
      isc.warn("Failed to load services: " + e.message);
    }
  }
});

isc.IButton.create({
  ID: "loadBtn",
  title: "Load Services",
  width: 140,
  click: "findForm.loadServices();",
});

const findBar = isc.HLayout.create({
  width: "100%",
  height: "10%",
  membersMargin: 8,
  members: [findForm, loadBtn]
});

const servicePane = isc.VLayout.create({
  width: "100%",
  height: "50%",
});

isc.DynamicForm.create({
  ID: "context_answer_textarea",
  width: "100%",
  height: "40%",
  titleWidth: 100,
  fields: [{ name: "context_http_answer", title: "Response", type: "textArea", width: "100%", height: "100%" }]
});

isc.VLayout.create({
  ID: "mainLayout",
  width: "100%",
  height: "100%",
  membersMargin: 10,
  members: [findBar, servicePane, context_answer_textarea],
});
