/*

  SmartClient Ajax RIA system
  Version v14.1p_2026-01-27/LGPL Deployment (2026-01-27)

  Copyright 2000 and beyond Isomorphic Software, Inc. All rights reserved.
  "SmartClient" is a trademark of Isomorphic Software, Inc.

  LICENSE NOTICE
     INSTALLATION OR USE OF THIS SOFTWARE INDICATES YOUR ACCEPTANCE OF
     ISOMORPHIC SOFTWARE LICENSE TERMS. If you have received this file
     without an accompanying Isomorphic Software license file, please
     contact licensing@isomorphic.com for details. Unauthorized copying and
     use of this software is a violation of international copyright law.

  DEVELOPMENT ONLY - DO NOT DEPLOY
     This software is provided for evaluation, training, and development
     purposes only. It may include supplementary components that are not
     licensed for deployment. The separate DEPLOY package for this release
     contains SmartClient components that are licensed for deployment.

  PROPRIETARY & PROTECTED MATERIAL
     This software contains proprietary materials that are protected by
     contract and intellectual property law. You are expressly prohibited
     from attempting to reverse engineer this software or modify this
     software for human readability.

  CONTACT ISOMORPHIC
     For more information regarding license rights and restrictions, or to
     report possible license violations, please contact Isomorphic Software
     by email (licensing@isomorphic.com) or web (www.isomorphic.com).

*/
//> @type AsyncOperationResultType
// The type of result of an asynchronous operation.

// @value "success" The asynchronous operation completed successfully.
// @value "error" The asynchronous operation encountered an error.
// @value "canceled" The asynchronous operation was canceled.
// @value "disabled" The asynchronous operation is disabled.
// @value "deferred" The asynchronous operation was deferred.
// @see AsyncOperationResult
// @treeLocation Client Reference/System
// @visibility external
//<

//> @object AsyncOperationResult
// An object containing information about the result of an asynchronous operation.
// <smartclient>
// <p>
// Note: +link{isc.defaultAsyncOperationCatchCallback()} can be used to provide default error
// handling for a +link{Promise}-based asynchronous operation.
// </smartclient>
// @treeLocation Client Reference/System
// @visibility external
//<
//> @attr AsyncOperationResult.type (AsyncOperationResultType : *initialized by the implementation* : IR)
// The result type.
// <p>
// <code>null</code> is interpreted as a non-successful result type.
// @visibility external
//<
//> @attr AsyncOperationResult.errorMessage (HTMLString : null : IR)
// If this is an <smartclient>"error"</smartclient><smartgwt>ERROR</smartgwt>-+link{type} result,
// an optional statement to be displayed to the user of the error that occurred.
// @visibility external
//<
//> @attr AsyncOperationResult.cancellationReason (HTMLString : null : IR)
// If this is a <smartclient>"canceled"</smartclient><smartgwt>CANCELED</smartgwt>-+link{type} result,
// an optional statement to be displayed to the user of the reason for cancellation.
// @see CancellationController.cancel()
// @visibility external
//<

//> @attr AsyncOperationResult.disabledMessage (HTMLString : null : IR)
// If this is a <smartclient>"disabled"</smartclient><smartgwt>DISABLED</smartgwt>-+link{type} result,
// an optional statement to be displayed to the user that the operation was disabled. This may
// include information about the reason why the operation was disabled.
// <p>
// It is recommended to use the past tense, because this message may continue to be displayed
// to the user when the conditions for the operation being disabled are no longer the case.

// @visibility external
//<

isc._genericAsyncOperationResultCallbackArgNames = "result";

//> @object AsyncSingleValueGenerationResult
// The result of an asynchronous operation to generate a value.
// @inheritsFrom AsyncOperationResult
// @treeLocation Client Reference/System/AsyncOperationResult
// @group fieldGeneration
// @visibility external
//<
//> @attr AsyncSingleValueGenerationResult.generatedValue (Any : *initialized by the implementation* : IR)
// If successful, the generated value.
// @visibility external
//<

//> @method Callbacks.AsyncSingleValueGenerationResultCallback()
// Callback fired with the result of an asynchronous operation to generate a value.
//
// @param result (AsyncSingleValueGenerationResult) The result.
// @visibility external
//<
isc._asyncSingleValueGenerationResultCallbackArgNames = isc._genericAsyncOperationResultCallbackArgNames;

//> @object AsyncMultipleValuesGenerationResult
// The result of an asynchronous operation to generate multiple values.
// @inheritsFrom AsyncOperationResult
// @treeLocation Client Reference/System/AsyncOperationResult
// @group fieldGeneration
// @visibility external
//<
//> @attr AsyncMultipleValuesGenerationResult.generatedValues (Array of Any : *initialized by the implementation* : IR)
// If successful, the generated values.
// @visibility external
//<

//> @method Callbacks.AsyncMultipleValuesGenerationResultCallback()
// Callback fired with the result of an asynchronous operation to generate multiple values.
//
// @param result (AsyncMultipleValuesGenerationResult) The result.
// @visibility external
//<
isc._asyncMultipleValuesGenerationResultCallbackArgNames = isc._genericAsyncOperationResultCallbackArgNames;


//> @staticMethod isc.createErrorResult()
// Creates an <smartclient>"error"</smartclient><smartgwt>ERROR</smartgwt>-+link{AsyncOperationResult.type, type}
// +link{AsyncOperationResult}.
// @param [errorMessage] (HTMLString) a message describing the error, if available.
// @param [additionalProperties] (AsyncOperationResult Properties) additional properties to add
// to the error result. Note that the <code>type</code> and/or <code>errorMessage</code> properties,
// if set on <code>additionalProperties</code>, will not be preserved.
// @visibility external
//<
isc.createErrorResult = function (errorMessage, additionalProperties) {
    var errorResult;
    if (!additionalProperties) {
        errorResult = {
            type: "error",
            errorMessage: errorMessage
        };
    } else {
        errorResult = isc.addPropertiesWithAssign({}, additionalProperties);
        // If `additionalProperties` had 'type' and/or 'errorMessage' properties, replace them.
        errorResult.type = "error";
        errorResult.errorMessage = errorMessage;
    }
    return errorResult;
};

isc.createDisabledResult = function (disabledMessage, additionalProperties) {
    var disabledResult;
    if (!additionalProperties) {
        disabledResult = {
            type: "disabled",
            disabledMessage: disabledMessage
        };
    } else {
        disabledResult = isc.addPropertiesWithAssign({}, additionalProperties);
        // If `additionalProperties` had 'type' and/or 'disabledMessage' properties, replace them.
        disabledResult.type = "disabled";
        disabledResult.disabledMessage = disabledMessage;
    }
    return disabledResult;
};

//> @staticMethod isc.rejectWithError()
// Creates a +link{Promise} that rejects with an
// <smartclient>"error"</smartclient><smartgwt>ERROR</smartgwt>-+link{AsyncOperationResult.type, type}
// +link{AsyncOperationResult}.
// @param [errorMessage] (HTMLString) a message describing the error, if available.
// @param [additionalProperties] (AsyncOperationResult Properties) additional properties to add
// to the reject result. Note that the <code>type</code> and/or <code>errorMessage</code> properties,
// if set on <code>additionalProperties</code>, will not be preserved.
// @return (Promise) the <code>Promise</code>.
// @visibility external
//<
isc.rejectWithError = function (errorMessage, additionalProperties) {
    return Promise.reject(isc.createErrorResult(errorMessage, additionalProperties));
};

isc.rejectAsDisabled = function (disabledMessage, additionalProperties) {
    return Promise.reject(isc.createDisabledResult(disabledMessage, additionalProperties));
};

//> @staticMethod isc.defaultAsyncOperationCatchCallback()
// A catch callback that can be used on a Promise for an +link{AsyncOperationResult} to
// provide default error handling.
// @param [reason] (Any) The reason (result of a rejected Promise).
// @return (AsyncOperationResult) an <code>AsyncOperationResult</code> from the provided reason.
// @visibility external
//<
isc.defaultAsyncOperationCatchCallback = function (reason) {
    var result;
    if (isc.isAn.Object(reason)) {
        // If `reason` already looks like a non-successful AsyncOperationResult, then we'll
        // keep it as-is.
        if (reason.type == "error" || reason.type == "canceled" || reason.type == "disabled" ||
            reason.type == "deferred" || "errorMessage" in reason)
        {
            result = reason;
        } else {
            
            var message = String(reason.message != null ? reason.message : reason).asHTML();
            
            if (reason.name == "AbortError") {
                
                result = {
                    type: "canceled",
                    cancellationReason: message
                };
            } else {
                result = {
                    type: "error",
                    errorMessage: message
                };
            }
            result.originalReason = reason;
        }
    } else {
        var message = reason == null ? null : String(reason).asHTML();
        result = {
            type: "error",
            errorMessage: message,
            originalReason: reason
        };
    }
    return result;
};


//> @class AsyncUtil
// Contains utilities and constants to support asynchronous code.
// @treeLocation Client Reference/System
// @visibility external
//<
isc.defineClass("AsyncUtil").addClassProperties({
    //> @classAttr AsyncUtil.missingRequiredParameterErrorMessage (HTMLString : "<code>${paramName}</code> is required." : RW)
    // An error message displayed when a required parameter to an API routine was not specified.
    // @group i18nMessages
    // @visibility external
    //<
    missingRequiredParameterErrorMessage: "`${paramName}` is required.",

    //> @classAttr AsyncUtil.dataBeingFetchedMessage (HTMLString : "Data were being fetched." : RW)
    // A +link{AsyncOperationResult.disabledMessage, disabledMessage} to use when an operation
    // was disabled because data were being fetched.
    // @group i18nMessages
    // @visibility external
    //<
    dataBeingFetchedMessage: "Data were being fetched.",

    //> @classAttr AsyncUtil.asyncErrorMessageGeneric (HTMLString : "An error occurred during an asynchronous operation." : RW)
    // A message string to use to when displaying an +link{AsyncOperationResult} of
    // +link{AsyncOperationResult.type, type} <smartclient>"error"</smartclient><smartgwt>ERROR</smartgwt>
    // to the user, and no +link{AsyncOperationResult.errorMessage} is available.
    // @group i18nMessages
    // @see AsyncUtil.getAsyncMessage()
    // @visibility external
    //<
    asyncErrorMessageGeneric: "An error occurred during an asynchronous operation.",

    //> @classAttr AsyncUtil.asyncCanceledMessageGeneric (HTMLString : "An asynchronous operation was canceled." : RW)
    // A message string to use when displaying an +link{AsyncOperationResult} of
    // +link{AsyncOperationResult.type, type} <smartclient>"canceled"</smartclient><smartgwt>CANCELED</smartgwt>
    // to the user, and no +link{AsyncOperationResult.cancellationReason} is available.
    // <p>
    // +link{AsyncUtil.asyncCanceledMessage} is used when a cancellationReason is available.
    // @group i18nMessages
    // @see AsyncUtil.getAsyncMessage()
    // @visibility external
    //<
    asyncCanceledMessageGeneric: "An asynchronous operation was canceled.",

    //> @classAttr AsyncUtil.asyncCanceledMessage (HTMLString : "An asynchronous operation was canceled with reason: ${cancellationReason}" : RW)
    // A message string to use when displaying an +link{AsyncOperationResult} of
    // +link{AsyncOperationResult.type, type} <smartclient>"canceled"</smartclient><smartgwt>CANCELED</smartgwt>
    // to the user, and a +link{AsyncOperationResult.cancellationReason} is available.
    // <p>
    // +link{AsyncUtil.asyncCanceledMessageGeneric} is used when no cancellationReason is available.
    // @group i18nMessages
    // @see AsyncUtil.getAsyncMessage()
    // @visibility external
    //<
    asyncCanceledMessage: "An asynchronous operation was canceled with reason: ${cancellationReason}",

    //> @classAttr AsyncUtil.asyncDisabledMessageGeneric (HTMLString : "An asynchronous operation was disabled." : RW)
    // A message string to use when displaying an +link{AsyncOperationResult} of
    // +link{AsyncOperationResult.type, type} <smartclient>"disabled"</smartclient><smartgwt>DISABLED</smartgwt>
    // to the user, and no +link{AsyncOperationResult.disabledMessage} is available.
    // @group i18nMessages
    // @see AsyncUtil.getAsyncMessage()
    // @visibility external
    //<
    asyncDisabledMessageGeneric: "An asynchronous operation was disabled.",

    //> @classAttr AsyncUtil.asyncNonSuccessMessageGeneric (HTMLString : "An asynchronous operation resulted in an unspecified type of result." : RW)
    // A message string to use when displaying an +link{AsyncOperationResult} to the user, when
    // the +link{AsyncOperationResult.type, type} is not provided.
    // @group i18nMessages
    // @see AsyncUtil.getAsyncMessage()
    // @see AsyncUtil.asyncNonSuccessMessage
    // @visibility external
    //<
    asyncNonSuccessMessageGeneric: "An asynchronous operation resulted in an unspecified type of result.",

    //> @classAttr AsyncUtil.asyncNonSuccessMessage (HTMLString : "An asynchronous operation resulted in type '${type}'." : RW)
    // A message string to use when displaying an +link{AsyncOperationResult} to the user, when
    // the provided +link{AsyncOperationResult.type, type} is neither
    // <smartclient>"error" nor "canceled"</smartclient><smartgwt>ERROR nor CANCELED</smartgwt>.
    // @group i18nMessages
    // @see AsyncUtil.getAsyncMessage()
    // @see AsyncUtil.asyncNonSuccessMessageGeneric
    // @visibility external
    //<
    asyncNonSuccessMessage: "An asynchronous operation resulted in type '${type}'.",

    //> @classMethod AsyncUtil.getAsyncMessage()
    // Returns a user-displayable message for the given +link{AsyncOperationResult} if its
    // +link{AsyncOperationResult.type, type} is not
    // <smartclient>"success"</smartclient><smartgwt>SUCCESS</smartgwt>.
    // <p>
    // +link{isc.getAsyncMessage(), isc.getAsyncMessage()} and
    // +link{AsyncUtil.getAsyncMessage(), AsyncUtil.getAsyncMessage()} are equivalent.
    //
    // @param result (AsyncOperationResult) the <code>AsyncOperationResult</code>.
    // @return (HTMLString) If the +link{AsyncOperationResult.type, type} is
    // <smartclient>"success"</smartclient><smartgwt>SUCCESS</smartgwt>, then <code>null</code>;
    // otherwise, a user-displayable message describing the non-successful result.
    // @see AsyncUtil.asyncErrorMessageGeneric
    // @see AsyncUtil.asyncCanceledMessage
    // @see AsyncUtil.asyncCanceledMessageGeneric
    // @see AsyncUtil.asyncDisabledMessageGeneric
    // @see AsyncUtil.asyncNonSuccessMessage
    // @see AsyncUtil.asyncNonSuccessMessageGeneric
    // @visibility external
    //<
    getAsyncMessage : function (result) {
        var type = result.type;
        if (type != "success") {
            var dynamicString;
            if (type == "error") {
                if (isc.isA.nonemptyString(result.errorMessage)) return result.errorMessage;
                dynamicString = this.asyncErrorMessageGeneric;
            } else if (type == "canceled") {
                dynamicString = isc.isA.nonemptyString(result.cancellationReason) ? this.asyncCanceledMessage : this.asyncCanceledMessageGeneric;
            } else if (type == "disabled") {
                if (isc.isA.nonemptyString(result.disabledMessage)) return result.disabledMessage;
                dynamicString = this.asyncDisabledMessageGeneric;
            } else {
                dynamicString = isc.isA.nonemptyString(type) ? this.asyncNonSuccessMessage : this.asyncNonSuccessMessageGeneric;
            }
            return dynamicString.evalDynamicString(this, result);
        }
    },

    //> @classAttr AsyncUtil.unspecifiedTypeRawMessage (HTMLString : "[Unspecified non-successful result type]" : IRW)
    // @group i18nMessages
    //<
    unspecifiedTypeRawMessage: "[Unspecified non-successful result type]",

    //> @classAttr AsyncUtil.unhandledTypeRawMessage (HTMLString : "[Unhandled non-successful result type '${resultType}']" : IRW)
    // @group i18nMessages
    //<
    unhandledTypeRawMessage: "[Unhandled result type '${resultType}']",

    getRawMessage : function (result) {
        var type = result.type;
        if (type != "success") {
            if (type == null) {
                return this.unspecifiedTypeRawMessage;
            } else if (type == "error") {
                return result.errorMessage;
            } else if (type == "canceled") {
                return result.cancellationReason;
            } else if (type == "disabled") {
                return result.disabledMessage;
            } else {
                return this.unhandledTypeRawMessage.evalDynamicString(this, {resultType: type});
            }
        }
    },


    //> @object AsyncOperationParams
    // Parameters to an asynchronous operation.
    // @see AsyncOperationContext
    // @treeLocation Client Reference/System
    // @visibility external
    //<
    //> @attr AsyncOperationParams.cancellationController (CancellationController : null : IR)
    // A +link{CancellationController} that the asynchronous operation should look to for whether
    // the operation is canceled.
    // <p>
    // This may be <code>null</code> if the invoker of the asynchronous operation does not need the
    // ability to cancel. However, note that the +link{AsyncOperationResult.type, result type} may
    // still be <smartclient>"canceled"</smartclient><smartgwt>CANCELED</smartgwt>; for example,
    // the asynchronous operation may be self-canceling after a timeout.
    // <p>
    // Implementations of asynchronous operations must treat this <code>CancellationController</code>
    // as observe-only, and they cannot +link{CancellationController.cancel(), cancel} it.
    // @visibility external
    //<

    //> @object AsyncOperationContext
    // Context for an asynchronous operation.
    // @see AsyncOperationParams
    // @treeLocation Client Reference/System
    // @visibility external
    //<
    //> @attr AsyncOperationContext.cancellationController
    // @include AsyncOperationParams.cancellationController
    // @visibility external
    //<

    //> @method Callbacks.AsyncOperationImpl()
    // Callback fired to perform the implementation of an asynchronous operation, given the
    // parameters to the asynchronous operation and the asynchronous operation context.
    //
    // @param opContext (AsyncOperationContext) The asynchronous operation context.
    // @param params (AsyncOperationParams) The original parameters.
    // @return (Promise) A +link{Promise} for an +link{AsyncOperationResult}.
    //<
    _asyncOperationImplArgNames: "opContext,params",

    // @param params (AsyncOperationParams) The parameters of the asynchronous operation.
    // @param impl (AsyncOperationImpl) The implementation of the asynchronous operation. It is
    // fired synchronously (before asyncOperation() returns).
    // @return (Promise) A +link{Promise} for an +link{AsyncOperationResult}, or <code>null</code>
    // if <code>impl</code> returned <code>null</code>, representing a synchronous operation that
    // is already complete.
    asyncOperation : function (params, impl) {
        var givenCC = params.cancellationController;
        if (givenCC && givenCC.canceled) {
            return Promise.resolve(givenCC.asCanceledResult());
        }

        var opContext = {
            cancellationController: givenCC
        };

        var promise = this.fireCallback(impl, this._asyncOperationImplArgNames, [opContext, params]);
        
        if (!promise) return;

        if (!givenCC) {
            return promise
                ._catch(function (rejectReason) {
                    return Promise.reject(isc.defaultAsyncOperationCatchCallback(rejectReason));
                });
        }

        var resolversObj = Promise.withResolvers(),
            cancelObservationId = isc.observe(givenCC, "cancel", "isc.AsyncUtil._handleCancelAsyncOperation(returnVal,observed,observer._resolversObj)");
        isc.addPropertiesWithAssign(isc.Page._staticObservers[cancelObservationId].observer, {
            _resolversObj: resolversObj
        });

        promise
            .then(
                resolversObj.resolve,
                function (rejectReason) {
                    resolversObj.reject(isc.defaultAsyncOperationCatchCallback(rejectReason));
                }
            );

        return resolversObj.promise
            ._finally(function () {
                isc.ignore(cancelObservationId);
            });
    },

    _handleCancelAsyncOperation : function (wasFirstCancel, givenCC, resolversObj) {
        if (!wasFirstCancel) return;
        resolversObj.reject(givenCC.asCanceledResult());
    },


    //> @object AsyncDataBoundOperationParams
    // Parameters to an asynchronous data-bound operation.
    // @see AsyncDataBoundOperationContext
    // @inheritsFrom AsyncOperationParams
    // @treeLocation Client Reference/System/AsyncOperationParams
    // @visibility external
    //<
    //> @attr AsyncDataBoundOperationParams.component (DataBoundComponent : null : IR)
    // A +link{DataBoundComponent} for context.
    // <p>
    // If +link{AsyncDataBoundOperationParams.dataSource, dataSource} is also specified, then the component
    // should be bound to the same <code>DataSource</code> or a <code>DataSource</code> inheriting
    // from it.
    // @visibility external
    //<
    //> @attr AsyncDataBoundOperationParams.dataSource (DataSource : null : IR)
    // A +link{DataSource} for context.
    // <p>
    // If +link{AsyncDataBoundOperationParams.component, component} is also specified, then the component
    // should be bound to the same <code>DataSource</code> or a <code>DataSource</code> inheriting
    // from it.
    // @visibility external
    //<

    //> @object AsyncDataBoundOperationContext
    // Context for an asynchronous data-bound operation.
    // @see AsyncDataBoundOperationParams
    // @inheritsFrom AsyncOperationContext
    // @treeLocation Client Reference/System
    // @visibility external
    //<
    //> @attr AsyncDataBoundOperationContext.component (DataBoundComponent : *initialized by the implementation* : IR)
    // @visibility external
    //<
    //> @attr AsyncDataBoundOperationContext.dataSource (DataSource : *initialized by the implementation* : IR)
    // @visibility external
    //<
    //> @attr AsyncDataBoundOperationContext.availableFieldNames (Array of FieldName : null : IR)
    // If set, the names of the fields of the +link{AsyncDataBoundOperationContext.component}
    // that are the only fields that may be used or referenced by this asynchronous data-bound
    // operation.
    // <p>
    // <code>null</code> means all fields.
    //<

    //> @method Callbacks.AsyncDataBoundOperationImpl()
    // Callback fired to perform the implementation of an asynchronous data-bound operation,
    // given the parameters to the asynchronous operation and the asynchronous operation context.
    //
    // @param dabOpContext (AsyncDataBoundOperationContext) The asynchronous data-bound operation
    // context.
    // @param params (AsyncDataBoundOperationParams) The original parameters.
    // @return (Promise) A +link{Promise} for an +link{AsyncOperationResult}.
    //<
    _asyncDataBoundOperationImplArgNames: "dabOpContext,params",

    // @param params (AsyncDataBoundOperationParams) The parameters of the asynchronous data-bound operation.
    // @param impl (AsyncOperationImpl) The implementation of the asynchronous data-bound operation. It is
    // fired synchronously (before asyncDataBoundOperation() returns).
    asyncDataBoundOperation : function (params, impl) {
        return this.asyncOperation(params, function (opContext, params) {
            var givenCC = opContext.cancellationController;

            var component = params.component,
                dataSource = params.dataSource,
                ownCC,
                setDataSourceObservationId,
                updateFieldDependenciesObservationId;
            if (component) {
                if (dataSource) {
                    var componentDataSource = component.getDataSource();
                    if (!componentDataSource) {
                        isc.AsyncUtil.logWarn(component.getID() + " is not bound to DataSource " + dataSource.ID + ". Ignoring the component...");
                        component = null;
                    } else if (!componentDataSource.inheritsSchema(dataSource)) {
                        isc.AsyncUtil.logWarn(component.getID() + " is not bound to a DataSource inheriting from " + dataSource.ID + ". Ignoring the component...");
                        component = null;
                    } else {
                        dataSource = componentDataSource;
                    }
                } else {
                    dataSource = component.getDataSource();
                }
            }

            var ownCC;
            if (component) ownCC = !givenCC ? isc.CancellationController.create() : givenCC.createSubController();

            var dabOpContext = {
                cancellationController: ownCC || givenCC,
                component: component,
                dataSource: dataSource
            };

            var promise = isc.AsyncUtil.fireCallback(impl, isc.AsyncUtil._asyncDataBoundOperationImplArgNames, [dabOpContext, params]);
            
            if (!promise) {
                if (ownCC) ownCC.destroy();
                return;
            }

            if (!component) {
                isc.AsyncUtil._assert(!ownCC);
                return promise;
            }

            isc.AsyncUtil._assert(ownCC);
            var resolversObj = Promise.withResolvers();

            var setDataSourceObservationId = isc.observe(component, "setDataSource", function () {
                if (dataSource === component.getDataSource()) return;

                if (!ownCC.canceled) {
                    ownCC.cancel(isc.CancellationController.componentDataSourceWasChangedReason.evalDynamicString(isc.AsyncUtil, {
                        componentID: component.getID()
                    }), "application");
                }
                resolversObj.reject(ownCC.asCanceledResult());
            });

            var updateFieldDependenciesObservationId = isc.observe(component, "updateFieldDependencies", function () {
                if (!ownCC.canceled) {
                    ownCC.cancel(isc.CancellationController.componentFieldsWereChangedReason.evalDynamicString(isc.AsyncUtil, {
                        componentID: component.getID()
                    }), "application");
                }
                resolversObj.reject(ownCC.asCanceledResult());
            });

            promise
                .then(
                    resolversObj.resolve,
                    resolversObj.reject
                );

            return resolversObj.promise
                ._finally(function () {
                    isc.ignore(updateFieldDependenciesObservationId);
                    isc.ignore(setDataSourceObservationId);
                    ownCC.destroy();
                });
        });
    }
});


//> @staticMethod isc.getAsyncMessage()
// @include AsyncUtil.getAsyncMessage()
// @visibility external
//<
isc.getAsyncMessage = function (result) {
    return isc.AsyncUtil.getAsyncMessage(result);
};








//> @object AsyncFilterContext
// Provides context for an asynchronous filter operation.
// @inheritsFrom AsyncOperationContext
// @treeLocation Client Reference/System/AsyncOperationContext

//<
//> @attr AsyncFilterContext.dataSource (DataSource : *initialized by the caller* : IR)
// The +link{DataSource}.

//<
//> @attr AsyncFilterContext.strictSQLFiltering (Boolean : false : IR)
// When set to <code>true</code>, this is a hint that handling of <code>NULL</code> values
// should follow SQL99 behavior.

//<
//> @attr AsyncFilterContext.recordsWereUpdated (Boolean : false : IR)
// If set to <code>true</code>, asynchronous filtering is being requested on records that have
// been updated.

//<


//> @object AsyncFilterResult
// @inheritsFrom AsyncOperationResult
// @treeLocation Client Reference/System/AsyncOperationResult

//<
//> @attr AsyncFilterResult.criterion (AdvancedCriteria : *initialized by the implementation* : IR)
// The criterion that was asynchronously evaluated, for reference.

//<
//> @attr AsyncFilterResult.matchingRecords (Array of Record : null : IR)
// The records that match the criterion.

//<


//> @class PausableAsyncOperation
// Represents a pausable, asynchronous operation.
// <p>
// Note that a <code>PausableAsyncOperation</code> instance is an +link{AsyncOperationContext}.
// @treeLocation Client Reference/System

//<

isc.defineClass("PausableAsyncOperation");
isc.PausableAsyncOperation.addProperties({
    //> @attr PausableAsyncOperation.params (AsyncOperationParams : null : IR)
    // The initial parameters to this asynchronous operation.
    
    //<

    //> @attr PausableAsyncOperation.cancellationController (CancellationController : *initialized by the implementation* : R)
    // A <code>CancellationController</code> created for the asynchronous operation's own use.
    
    //<

    //> @attr PausableAsyncOperation.initialCanceledResult (AsyncOperationResult : *initialized by the implementation* : R)
    // If this asynchronous operation is initially canceled (because
    // +link{AsyncOperationParams.cancellationController, this.params.cancellationController}
    // is already canceled), then this is a reference to the type
    // <smartclient>"canceled"</smartclient><smartgwt>CANCELED</smartgwt> +link{AsyncOperationResult}
    // that the <code>Promise</code> returned by +link{PausableAsyncOperation.asyncGetResult()}
    // will reject with.
    // <p>
    // If not initially canceled, then this property will be <code>null</code>.
    
    //<
    //initialCanceledResult: null,

    //> @attr PausableAsyncOperation.paused (boolean : false : IRW)
    // <code>true</code> if the asynchronous operation is paused and not canceled;
    // <code>false</code> otherwise.
    // <p>
    // This attribute is writable indirectly by calling the +link{PausableAsyncOperation.pause()}
    // method.
    // @setter pause()
    
    //<
    paused: false,

    

    

    init : function () {
        this.Super("init", arguments);

        var givenCC = this.params && this.params.cancellationController,
            ownCC = this.cancellationController = !givenCC
                                                  ? isc.CancellationController.create()
                                                  : givenCC.createSubController();
        if (!ownCC.canceled) {
            this.observe(ownCC, "cancel", "observer.handleCancel(returnVal,observed)");
        }

        this.__resolversObj = Promise.withResolvers();
        if (ownCC.canceled) this.__resolversObj.reject(this.initialCanceledResult = ownCC.asCanceledResult());

        this.paused = (this.paused == true);
        if (this.paused) {
            this.__doPause(this.pauseInitiator || "application");
        } else {
            delete this.pauseInitiator;
            if (ownCC.canceled) this.unpauseInitiator = ownCC.cancellationInitiator;
        }
    },

    destroy : function () {
        if (!this.isCanceled()) this.cancel(this.getID() + " was destroyed.", "application");
        this.ns.ClassFactory.dereferenceGlobalID(this);
        this.Super("destroy", arguments);
        this.destroyed = true;
    },

    //> @method PausableAsyncOperation.asyncGetResult()
    // Returns a <code>Promise</code> for the result of this asynchronous operation. The first
    // of the following events to occur determines whether this <code>Promise</code> fulfills
    // or rejects, and the +link{AsyncOperationResult.type, type} of result:
    // <ul>
    // <li>The asynchronous operation is initially canceled: rejects with type <smartclient>"canceled"</smartclient><smartgwt>CANCELED</smartgwt>,
    //     and +link{PausableAsyncOperation.initialCanceledResult} is a reference to this result.
    // <li>The asynchronous operation is canceled: rejects with type <smartclient>"canceled"</smartclient><smartgwt>CANCELED</smartgwt>
    // <li>+link{postResult()} is called with a successful result: fulfills with type <smartclient>"success"</smartclient><smartgwt>SUCCESS</smartgwt>
    // <li>postResult() is called with a non-successful result: rejects with a non-successful result type
    // </ul>
    // <p>
    // Note that the settling of the returned <code>Promise</code> is independent of this
    // asynchronous operation being paused; even if paused, if postResult() is called to post
    // the result, then the result <code>Promise</code> will settle.
    //
    // @return (Promise) A <code>Promise</code> for an +link{AsyncOperationResult}.
    
    //<
    asyncGetResult : function () {
        return this.__resolversObj.promise;
    },

    //> @method PausableAsyncOperation.postResult()
    // (protected) Method that the implementation uses to post its result.
    // <p>
    // Assuming that this asynchronous operation was not initially canceled, only the first
    // call to this method will have an effect. See +link{PausableAsyncOperation.asyncGetResult()}
    // for more information.
    //
    // @param result (AsyncOperationResult) The result of this asynchronous operation.
    // @see PausableAsyncOperation.asyncGetResult()
    
    //<
    postResult : function (result) {
        if (!isc.isAn.Object(result) || result.type != "success") {
            var nonSuccessfulResult = isc.defaultAsyncOperationCatchCallback(result);
            this.__resolversObj.reject(nonSuccessfulResult);
        } else {
            this.__resolversObj.resolve(result);
        }
        if (!this.isCanceled()) {
            this.cancel("A result of the asynchronous operation was posted.", "system");
        }
        this.__destroyOwnCC();
    },

    __destroyOwnCC : function () {
        var ownCC = this.cancellationController;
        if (!ownCC.destroyed) {
            if (this.isObserving(ownCC, "cancel")) this.ignore(ownCC, "cancel");
            ownCC.destroy();
        }
        
    },

    handleCancel : function (wasFirstCancel, ownCC) {
        
        if (!wasFirstCancel) return;
        if (this.paused) this.__doUnpause(ownCC.cancellationInitiator);
        this.__resolversObj.reject(this.initialCanceledResult || ownCC.asCanceledResult());
        this.__destroyOwnCC();
    },

    //> @method PausableAsyncOperation.cancel()
    // Cancels the asynchronous operation.
    //
    // @include CancellationController.cancel()
    
    //<
    cancel : function () {
        var ownCC = this.cancellationController;
        return ownCC.cancel.apply(ownCC, arguments);
    },

    //> @method PausableAsyncOperation.isCanceled()
    //
    // @return (boolean) <code>true</code> if this asynchronous operation is canceled;
    // <code>false</code> otherwise.
    
    //<
    isCanceled : function () {
        return this.cancellationController.canceled;
    },

    //> @method PausableAsyncOperation.pause()
    // Pauses this asynchronous operation if not already paused.
    //
    // @return (Promise) If not +link{isCanceled(), canceled}, a <code>Promise</code> that
    // resolves when the asynchronous operation is unpaused, or rejects if the asynchronous
    // operation is canceled before being unpaused.
    
    //<
    pause : function (_initiator) {
        
        if (!this.paused) this.handlePaused(_initiator);
        return this.__unpauseResolversObj && this.__unpauseResolversObj.promise;
    },

    getUnpausePromise : function () {
        return this.__unpauseResolversObj && this.__unpauseResolversObj.promise;
    },

    handlePaused : function (initiator) {
        this.__doPause(initiator);
    },

    __doPause : function (initiator) {
        delete this.pauseInitiator;
        delete this.unpauseInitiator;
        if (this.isCanceled()) {
            this.paused = false;
            this.unpauseInitiator = this.cancellationController.cancellationInitiator;
            return;
        }
        this.paused = true;
        this.pauseInitiator = initiator;
        this.__unpauseResolversObj = Promise.withResolvers();
    },

    //> @method PausableAsyncOperation.unpause()
    // Unpauses this asynchronous operation if paused.
    //
    // @return (boolean) <code>true</code> if this operation was paused (and so now is unpaused);
    // <code>false</code> otherwise.
    
    //<
    unpause : function (_initiator) {
        
        if (!this.paused) return false;
        this.handleUnpaused(_initiator);
        return true;
    },

    handleUnpaused : function (initiator) {
        this.__doUnpause(initiator);
    },

    __doUnpause : function (initiator) {
        this.paused = false;
        delete this.pauseInitiator;
        this.unpauseInitiator = initiator;
        var ownCC = this.cancellationController,
            resolvers = this.__unpauseResolversObj;
        
        delete this.__unpauseResolversObj;
        if (ownCC.canceled) resolvers.reject(this.initialCanceledResult || ownCC.asCanceledResult());
        else resolvers.resolve(initiator);
    },

    
    _whenUnpaused : function (onUnpaused, onUnpauseRejected, thisArg, EH) {
        var promise,
            ownCC = this.cancellationController;
        if (ownCC.canceled) {
            promise = Promise.reject(this.initialCanceledResult || ownCC.asCanceledResult());
        } else if (!this.paused) {
            // Even when not currently paused, `onUnpaused` is not called immediately (i.e. synchronously)
            // so as to avoid releasing Zalgo: https://blog.izs.me/2013/08/designing-apis-for-asynchrony/
            promise = Promise.resolve(this.unpauseInitiator);
        } else {
            promise = this.__unpauseResolversObj.promise;
        }

        if (!EH) EH = this.ns.EH;

        if (isc.isA.Function(onUnpaused)) {
            onUnpaused = EH._wrapCallback("Unpaused", onUnpaused, thisArg, EH);
        }
        if (isc.isA.Function(onUnpauseRejected)) {
            onUnpauseRejected = EH._wrapCallback("UnpauseRejected", onUnpauseRejected, thisArg, EH);
        }

        return promise.then(onUnpaused, onUnpauseRejected);
    }
});
